 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>             Data Management Attribute Code
   >>>>
   >>>>   Static:
   >>>>                 _kdms_match_association()
   >>>>                 _kdms_vargs_to_attribute_data()
   >>>>                 _kdms_attribute_data_to_vargs()
   >>>>                 _kdms_copy_attribute_data()
   >>>>                 
   >>>>                 _kdms_get_generic()
   >>>>                 _kdms_set_generic()
   >>>>                 _kdms_match_generic()
   >>>>                 _kdms_copy_generic()
   >>>>                 _kdms_query_generic()
   >>>>                 _kdms_print_generic()
   >>>>                 _kdms_construct_generic()
   >>>>                 
   >>>>                 _kdms_get_undefined()
   >>>>                 _kdms_set_undefined()
   >>>>                 _kdms_match_undefined()
   >>>>                 _kdms_copy_undefined()
   >>>>                 _kdms_query_undefined()
   >>>>                 _kdms_print_undefined()
   >>>>                 _kdms_construct_quasi()
   >>>>                 
   >>>>                 _kdms_destruct_attribute()
   >>>>                 
   >>>>                 _kdms_append_attribute()
   >>>>                 _kdms_locate_attribute()
   >>>>                 _kdms_delete_attribute()
   >>>>                 
   >>>>                 _kdms_construct_generic_defin()
   >>>>                 _kdms_construct_quasi_defin()
   >>>>                 _kdms_destruct_attribute_defin()
   >>>>                 
   >>>>                 _kdms_hash_attribute_defin()
   >>>>                 _kdms_locate_attribute_defin()
   >>>>                 _kdms_delete_attribute_defin()
   >>>>                 
   >>>>                 _kdms_locate_attribute_list()
   >>>>                 _kdms_retrieve_attribute()
   >>>>                 _kdms_copy_attribute()
   >>>>                 _kdms_copy_attribute_list()
   >>>>
   >>>>  Private:
   >>>>                 kdms_free_attribute_defins()
   >>>>                 kdms_establish_attribute()
   >>>>                 kdms_get_Vattribute()
   >>>>                 kdms_set_Vattribute()
   >>>>                 kdms_free_pres_attributes()
   >>>>                 kdms_free_object_attributes()
   >>>>                 
   >>>>   Public:
   >>>>                 kdms_define_quasi_attribute()
   >>>>                 kdms_define_attribute()
   >>>>                 kdms_undefine_attribute()
   >>>>                 kdms_create_attribute()
   >>>>                 kdms_destroy_attribute()
   >>>>
   >>>>                 kdms_vset_attribute()
   >>>>                 kdms_vset_attributes()
   >>>>                 kdms_set_attribute()
   >>>>                 kdms_set_attributes()
   >>>>
   >>>>                 kdms_vget_attribute()
   >>>>                 kdms_vget_attributes()
   >>>>                 kdms_get_attribute()
   >>>>                 kdms_get_attributes()
   >>>>
   >>>>                 kdms_match_attribute()
   >>>>                 kdms_vmatch_attributes()
   >>>>                 kdms_match_attributes()
   >>>>
   >>>>                 kdms_copy_attribute()
   >>>>                 kdms_vcopy_attributes()
   >>>>                 kdms_copy_attributes()
   >>>>
   >>>>                 kdms_query_attribute()
   >>>>                 kdms_print_attribute()
   >>>>
   >>>>                 kdms_get_attribute_names()
   >>>>                 kdms_copy_segment_attributes()
   >>>>                                  
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"

extern int _kdms_compiled;

/* 
 *    ================================================================== 
 *    Attribute Definition Table
 *    ==================================================================
 */
#define TABLESIZE 1024

static int hash_next = 1;      /* -- hash_next must be initially > 0 --*/

static klist         **hash_table = NULL;

/*-----------------------------------------------------------
|
|  Routine Name: (static) _kdms_match_association 
|
|       Purpose: This routine will compare two given associations
|                and determine if they match.  The first association 
|	         passed in is considered to be the master association, 
| 		 against which the second association is compared.  The 
|	         master association may be the general catch-all 
|      		 KDMS_ALL_SEGMENTS, in which case it will match against 
|                any thing except NULL.
|
|         Input: association1 - the master association
|                association2 - the association to compare with association1
|
|        Output: none
|
|       Returns: TRUE (1) if associations match, FALSE (0) otherwise
|
|    Written By: Jeremy Worley and Steve Kubica
|          Date: Mar 17, 1994 10:47
| Modifications:
|
------------------------------------------------------------*/
static int
_kdms_match_association(
   char *association1,
   char *association2)
{
   static int all_segments = 0;

   if (all_segments == 0)
      all_segments = kstring_to_token(KDMS_ALL_SEGMENTS);

   /* 
    * first check : if first association is NULL, then second
    * 		    association must be NULL, or else no match
    */
   if (association1 == NULL)
   {
      if (association2 == NULL)
	 return TRUE;
      else
	 return FALSE;
   }

   /*
    *  second check :  we now know that association1 != NULL,
    *                  so if association2 == NULL, then it 
    *                  won't match association1.
    */
   if (association2 == NULL)
      return FALSE;

   /*
    *  third check : we now know that neither association1 or
    *                association2 is NULL, so if they are
    *                identical, then they match!
    */
   if (kstrcmp(association1, association2) == 0)
      return TRUE;

   /*
    *  last check : they are not the same, so our only hope
    *               is for association1 to be KDMS_ALL_SEGMENTS.
    *               This will match any association2 except for NULL
    *               and we know from before that association2 != NULL!
    */
   if (kstrcmp(association1, KDMS_ALL_SEGMENTS) == 0)
      return TRUE;

   /* -- oh well, nothing matched -- */
   return FALSE;

}

/*-----------------------------------------------------------
|
|  Routine Name: _kdms_find_defin - find a defin for a given attribute
|
|       Purpose: This routine locates the data services attribute
|	         definition structure for a specified atttribute and
|	         association.
|
|         Input: attribute  - the name of the attribute to find
|                assocation - the name of the assocation to match
|
|        Output: indx   - the hash index into the hash_table
|
|       Returns: the definition or NULL if it doesn't exist
|
|    Written By: Mark Young and Steve Kubica
|          Date: Dec 15, 1994
| Modifications:
|
------------------------------------------------------------*/
static kdms_attribute_defin *_kdms_find_defin(
   char *attribute,
   char *association,
   int  *indx)
{
   int       temp;
   klist    *list;
   kdms_attribute_defin *defin;
   

   /* -- sanity check to make sure that string is not NULL -- */
   if (!attribute) return NULL;

   /* -- check to make sure that hash table has been initialized -- */
   if (!hash_table)
      hash_table = (klist **) kcalloc(TABLESIZE, sizeof(klist *));

   /* -- compute the index into the hash table -- */
   temp = khash(attribute, -1);
   temp = kabs(temp) % TABLESIZE;
   if (indx) *indx = temp;

   for (list = hash_table[temp]; list != NULL; list = klist_next(list))
   {
      defin = (kdms_attribute_defin *) list->client_data;
      
      /* -- return if definition matches 
	 --   - this implies that the attribute name is identical,
	 --     and the associations match */
      if (strcmp(attribute, defin->attribute) == 0 &&
	  _kdms_match_association(defin->association, association))
	     return defin;
   }
   return NULL;
}

/* 
 *    ================================================================== 
 *    Attribute Utility Routines
 *    ==================================================================
 */

/*-----------------------------------------------------------
|
|  Routine Name: (static) _kdms_vargs_to_attribute_data 
|
|       Purpose: This routine reads a variable argument list of 
|                data and combines that data into a single block 
|                of data.  The variable argument list will consist
|                of a given number of arguments, where each argument
|                consists of a given argument size of elements where
|                each element is of a given data type.  The data
|                is pulled off of the variable argument list and
|                combined into a single data block.
|
|                Note that if the argument size is equal to one, then
|                the data is simply assigned.  If the argument size
|                is greater than one, then the data is copied.
|
|                This data block can be decomposed and returned via 
|                variable argument pointers using the call
|                _kdms_attribute_data_to_vargs().
|
|		 The data type of KSTRUCT is assumed to refer to
|		 a generic data pointer (kaddr), since there is
|		 no pointer data type.
|
|         Input: data_type - data type of attribute data
|                arg_size  - argument size of each attribute arguyment
|                num_args  - number of arguments in attribute data 
|
|                list      - a variable argument list containing 
|                            data in the form :
|
|                               value1 [, value2, ...]
|
|                            The list should consist of "num_arg" arguments,
|                            where each argument is of "arg_size" size of
|                            "data_type" data type.
|
|        Output: dest      - pointer to pointer of where to return
|                            block of attribute data
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley and Steve Kubica
|          Date: Mar 17, 1994 10:47
| Modifications:
|
------------------------------------------------------------*/
static int
_kdms_vargs_to_attribute_data(
   kaddr     *dest,
   int        data_type,
   int        arg_size,
   int        num_args,
   kva_list  *list)
{
   int i,
      status = TRUE;

   if (dest == NULL)
      return FALSE;

   /* sanity check: if the size or number of arguments is zero, then return */
   if (arg_size == 0 || num_args == 0)
   {
      kinfo(KDEBUG, "_kdms_vargs_to_attribute_data : erroneous input\n"
            "   arg_size = %d\n   num_args = %d\n",
            arg_size, num_args);
      return FALSE;
   }

   /* 
    * if the destination data pointer is null, then we must allocate 
    * space for this thing.
    */
   if (*dest == NULL)
      if ((*dest =
       (kaddr) kcalloc((unsigned)(num_args * arg_size), 
		       (unsigned)kdata_size(data_type))) == NULL)
      {
         _kdms_set_error(KMEMORY_ALLOCATION);
         return FALSE;
      }

   /*
    * do it in a generic way if the the arg_size is > 1... OOGY kstring stuff
    */
   if (arg_size > 1)
   {
      if (data_type != KSTRING)
      {
         char *dbyte = (char *)*dest;
         for (i = 0; i < num_args; i++)
            kmemcpy(dbyte + i * arg_size * kdata_size(data_type),
                    kva_arg(*list, char *),
                    (unsigned)(arg_size * kdata_size(data_type)));
         return TRUE;
      }
      else
      {
         int j;
	 int k;
         kstring *stmp;
	 kstring *tmp = (kstring *) * dest;
         for (i = 0, k = 0; i < num_args; i++)
         {
            stmp = kva_arg(*list, kstring *);
            for (j = 0; j < arg_size; j++, k++)
            {
               kfree(tmp[k]);
               tmp[k] = kstring_copy(stmp[j], NULL);
            }
         }
         return TRUE;
      }
   }

   /* -- for when arg_size == 1 -- */
   switch (data_type)
   {
      case KBYTE:
         {
            signed char *tmp = (signed char *)*dest;
            for (i = 0; i < num_args; i++)
               tmp[i] = (char)kva_arg(*list, int);
         }
         break;

      case KUBYTE:
         {
            unsigned char *tmp = (unsigned char *)*dest;
            for (i = 0; i < num_args; i++)
               tmp[i] = (unsigned char)kva_arg(*list, unsigned int);
         }
         break;

      case KSHORT:
         {
            short *tmp = (short *)*dest;
            for (i = 0; i < num_args; i++)
               tmp[i] = (short)kva_arg(*list, int);
         }
         break;

      case KUSHORT:
         {
            unsigned short *tmp = (unsigned short *)*dest;
            for (i = 0; i < num_args; i++)
               tmp[i] = (unsigned short)kva_arg(*list, unsigned int);
         }
         break;

      case KINT:
         {
            int *tmp = (int *)*dest;
            for (i = 0; i < num_args; i++)
               tmp[i] = kva_arg(*list, int);
         }
         break;

      case KUINT:
         {
            unsigned int *tmp = (unsigned int *)*dest;
            for (i = 0; i < num_args; i++)
               tmp[i] = kva_arg(*list, unsigned int);
         }
         break;

      case KLONG:
         {
            long *tmp = (long *)*dest;
            for (i = 0; i < num_args; i++)
               tmp[i] = kva_arg(*list, long);
         }
         break;

      case KULONG:
         {
            unsigned long *tmp = (unsigned long *)*dest;
            for (i = 0; i < num_args; i++)
               tmp[i] = kva_arg(*list, unsigned long);
         }
         break;

      case KFLOAT:
         {
            float *tmp = (float *)*dest;
            for (i = 0; i < num_args; i++)
               tmp[i] = (float)kva_arg(*list, double);
         }
         break;

      case KDOUBLE:
         {
            double *tmp = (double *)*dest;
            for (i = 0; i < num_args; i++)
               tmp[i] = kva_arg(*list, double);
         }
         break;

      case KCOMPLEX:
         {
            kcomplex *tmp = (kcomplex *) * dest;
            for (i = 0; i < num_args; i++)
               tmp[i] = kva_arg(*list, kcomplex);
         }
         break;

      case KDCOMPLEX:
         {
            kdcomplex *tmp = (kdcomplex *) * dest;
            for (i = 0; i < num_args; i++)
               tmp[i] = kva_arg(*list, kdcomplex);
         }
         break;

      case KSTRING:
         {
            kstring *tmp = (kstring *) * dest;
            for (i = 0; i < num_args; i++)
               tmp[i] = kstring_copy(kva_arg(*list, kstring), NULL);
         }
         break;

      case KSTRUCT:
         {
            kaddr *tmp = (kaddr *) * dest;
            for (i = 0; i < num_args; i++)
               tmp[i] = kva_arg(*list, kaddr);
         }
         break;

      default:
         _kdms_set_error(KDMS_EATTR_INVALID);
         kinfo(KDEBUG, "_kdms_vargs_to_attribute_data :"
               "data type %d is invalid", data_type);
         status = FALSE;
   }
   return status;
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _kdms_attribute_data_to_vargs
|
|       Purpose: This routine decomposes a block of attribute data
|                into its consituent parts and returns them via a
|                variable argument list of pointers to data.
|                The variable argument list will consist of a given
|                number of arguments, where each argument is a pointer
|                to data of a given argument size of elements where
|                each element is of a given data type.  
|
|                Note that if the argument size is equal to one, then
|                the data is simply returned via the given data pointer.  
|                If the argument size is greater than one, then a pointer 
|                to the internal copy of the data is returned.  The user
|                must be careful not to change or free this data.
|
|                This data can be composed into a data block from
|                the variable argument pointers using the call
|                _kdms_vargs_to_attribute_data().
|
|		 The data type of KSTRUCT is assumed to refer to
|		 a generic data pointer (kaddr), since there is
|		 no pointer data type.
|
|         Input: src       - pointer to attribute data 
|                data_type - data type of attribute data
|                arg_size  - argument size of each attribute arguyment
|                num_args  - number of arguments in attribute data 
|
|        Output: list      - a variable argument list containing 
|                            data in the form :
|
|                               &value1 [, &value2, ...]
|
|                            The list should consist of "num_arg" arguments,
|                            where each argument is a pointer to "arg_size" 
|                            size of "data_type" data type.
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley and Steve Kubica
|          Date: Mar 17, 1994 10:47
| Modifications:
|
------------------------------------------------------------*/
static int
_kdms_attribute_data_to_vargs(
   kaddr     src,
   int       data_type,
   int       arg_size,
   int       num_args,
   kva_list *list)
{
   kaddr *entity;
   int i;

   /* sanity check: if the size or number of arguments is zero, then return */
   if (arg_size == 0 || num_args == 0)
   {
      kinfo(KDEBUG, "_kdms_attribute_data_to_vargs : erroneous input\n"
            "   arg_size = %d\n   num_args = %d\n",
            arg_size, num_args);
      return FALSE;
   }

   /* -- if the data is null, then the attribute has not yet been set -- */
   if (src == NULL)
   {
      _kdms_set_error(KDMS_EATTR_UNSET);
      return FALSE;
   }

   /* -- do it in a generic way if the the arg_size is > 1 -- */
   if (arg_size > 1)
   {
      char *sbyte = (char *)src;
      for (i = 0; i < num_args; i++)
      {
         entity = kva_arg(*list, kaddr *);
         if (entity != NULL)
            *entity = (kaddr) (sbyte + i * arg_size * kdata_size(data_type));
      }
      return TRUE;
   }

   /* -- case for when arg_size == 1 -- */
   switch (data_type)
   {
      case KBYTE:
         {
	    signed char *tmp;
	    for (i = 0; i < num_args; i++)
	    {
	       tmp = (kva_arg(*list, signed char *));
	       if (tmp)
		  *tmp = (signed char)((signed char *)src)[i];
	    }
	 }
	 break;

      case KUBYTE:
	 {
	    unsigned char *tmp;
	    for (i = 0; i < num_args; i++)
	    {
	       tmp = (kva_arg(*list, unsigned char *));
	       if (tmp)
		  *tmp = (unsigned char)((unsigned char *)src)[i];
	    }
	 }
	 break;

      case KSHORT:
	 {
	    short *tmp;
	    for (i = 0; i < num_args; i++)
	    {
	       tmp = (kva_arg(*list, short *));
	       if (tmp)
		  *tmp = (short)((short *)src)[i];
	    }
	 }
	 break;

      case KUSHORT:
	 {
	    unsigned short *tmp;
	    for (i = 0; i < num_args; i++)
	    {
	       tmp = (kva_arg(*list, unsigned short *));
	       if (tmp)
		  *tmp = (unsigned short)((unsigned short *)src)[i];
	    }
	 }
	 break;

      case KINT:
	 {
	    int *tmp;
	    for (i = 0; i < num_args; i++)
	    {
	       tmp = (kva_arg(*list, int *));
	       if (tmp)
		  *tmp = (int)((int *)src)[i];
	    }
	 }
	 break;

      case KUINT:
	 {
	    unsigned int *tmp;
	    for (i = 0; i < num_args; i++)
	    {
	       tmp = (kva_arg(*list, unsigned int *));
	       if (tmp)
		  *tmp = (unsigned int)((unsigned int *)src)[i];
	    }
	 }
	 break;

      case KLONG:
	 {
	    long *tmp;
	    for (i = 0; i < num_args; i++)
	    {
	       tmp = (kva_arg(*list, long *));
	       if (tmp)
		  *tmp = (long)((long *)src)[i];
	    }
	 }
	 break;

      case KULONG:
	 {
	    unsigned long *tmp;
	    for (i = 0; i < num_args; i++)
	    {
	       tmp = (kva_arg(*list, unsigned long *));
	       if (tmp)
		  *tmp = (unsigned long)((unsigned long *)src)[i];
	    }
	 }
	 break;

      case KFLOAT:
	 {
	    float *tmp;
	    for (i = 0; i < num_args; i++)
	    {
	       tmp = (kva_arg(*list, float *));
	       if (tmp)
		  *tmp = (float)((float *)src)[i];
	    }
	 }
	 break;

      case KDOUBLE:
	 {
	    double *tmp;
	    for (i = 0; i < num_args; i++)
	    {
	       tmp = (kva_arg(*list, double *));
	       if (tmp)
		  *tmp = (double)((double *)src)[i];
	    }
	 }
	 break;

      case KCOMPLEX:
	 {
	    kcomplex *tmp = (kcomplex *) src;
	    kcomplex *tmp2;
	    for (i = 0; i < num_args; i++)
	    {
	       tmp2 = (kva_arg(*list, kcomplex *));
	       if (tmp2)
		  *tmp2 = tmp[i];
	    }
	 }
	 break;

      case KDCOMPLEX:
	 {
	    kdcomplex *tmp = (kdcomplex *) src;
	    kdcomplex *tmp2;
	    for (i = 0; i < num_args; i++)
	    {
	       tmp2 = (kva_arg(*list, kdcomplex *));
	       if (tmp2)
		  *tmp2 = tmp[i];
	    }
	 }
	 break;

      case KSTRING:
	 {
	    for (i = 0; i < num_args; i++)
	       *(kva_arg(*list, kstring *)) = ((kstring *) src)[i];
	 }
	 break;

      case KSTRUCT:
	 {
	    kaddr *tmp;
	    for (i = 0; i < num_args; i++)
	    {
	       tmp = (kva_arg(*list, kaddr *));
	       if (tmp)
		  *tmp = (kaddr)((kaddr *)src)[i];
	    }
	 }
	 break;

      default:
	 _kdms_set_error(KDMS_EATTR_INVALID);
	 kinfo(KDEBUG, "_kdms_attribute_data_to_vargs :"
	       "data type %d is invalid", data_type);
	 return FALSE;
   }
   return TRUE;
}


/*-----------------------------------------------------------
|
|  Routine Name: (static) _kdms_copy_attribute_data
|
|       Purpose: This routine copies a block of attribute data from
|                one pointer to another.  If the destination pointer
|                points to NULL, then space is allocated for the
|                resulting copy.  The space is otherwise assumed to 
|                be of a sufficient size to hold the copy.  This size
|                is determined by multiplying the given data type by
|                the given argument size by the given number of 
|                arguments.
|
|         Input: src       - pointer to src attribute data
|                data_type - data type of attribute data to copy
|                arg_size  - argument size of attribute data to copy
|                num_args  - number of arguments in attribute data to copy
|
|        Output: dest      - pointer to pointer of where to put copied data
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley and Steve Kubica
|          Date: Mar 17, 1994 10:47
| Modifications:
|
------------------------------------------------------------*/
static int
_kdms_copy_attribute_data(
   kaddr  src,
   kaddr *dest,
   int    data_type,
   int    arg_size,
   int    num_args)
{
   int i;

   if (src == NULL || dest == NULL)
      return FALSE;

   /* sanity check: if the size or number of arguments is zero, then return */
   if (arg_size == 0 || num_args == 0)
   {
      kinfo(KDEBUG, "_kdms_copy_attribute_data : erroneous input\n"
	    "   arg_size = %d\n   num_args = %d\n",
	    arg_size, num_args);
      return FALSE;
   }

   /* 
    *  if the destination data pointer is null, then we must allocate 
    *  space for this thing.  otherwise assume space there is enough
    *  (it is necessary to use calloc because we've gotta realloc if its
    *  a string and the krealloc is smart enough to detect the NULL)
    */
   if (*dest == NULL)
      if ((*dest = (kaddr) kcalloc((unsigned)(num_args * arg_size *
				   kdata_size(data_type)), 1)) == NULL)
      {
	 _kdms_set_error(KMEMORY_ALLOCATION);
	 return FALSE;
      }


   /* do the copy */
   if (data_type == KSTRING)
   {
      kstring *sbyte = (kstring *) src;		/* indexable version of src */
      kstring **dbyte = (kstring **) dest;	/* indexable version of dest */

      for (i = 0; i < num_args * arg_size; i++)
      {
	 kfree((*dbyte)[i]);
	 (*dbyte)[i] = kstring_copy(sbyte[i], NULL);
      }
   }
   else
      kmemcpy((*dest), src, (unsigned)(arg_size * num_args * 
				       kdata_size(data_type)));

   return TRUE;
}

/* 
 *    ================================================================== 
 *    Generic Attribute Methods
 *    ==================================================================
 */

/*-----------------------------------------------------------
|  Routine Name: (static) _kdms_get_generic
|       Purpose: perform a generic attribute retrieval
|    Written By: Jeremy Worley
|          Date: Sep 23, 1993 13:12
------------------------------------------------------------*/
static int
_kdms_get_generic(
   kdms_generic_attribute *atr,
   kva_list               *list)
{
   int status;

   /* sanity check */
   if (atr == NULL)
   {
      kinfo(KDEBUG, "_kdms_get_generic : null attribute provided\n");
      return FALSE;
   }

   status = _kdms_attribute_data_to_vargs(atr->value, atr->data_type,
					atr->arg_size, atr->num_args, list);
   return status;
}


/*-----------------------------------------------------------
|  Routine Name: (static) _kdms_set_generic
|       Purpose: perform a generic attribute set
|    Written By: Jeremy Worley
|          Date: Sep 23, 1993 13:12
------------------------------------------------------------*/
static int
_kdms_set_generic(
   kdms_generic_attribute *atr,
   kva_list *list)
{
   int status;

   /* sanity check */
   if (atr == NULL)
   {
      kinfo(KDEBUG, "_kdms_set_generic : null attribute provided\n");
      return FALSE;
   }

   status = _kdms_vargs_to_attribute_data(&(atr->value), atr->data_type,
					atr->arg_size, atr->num_args, list);
   return status;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _kdms_match_generic
|       Purpose: perform a generic attribute match
|    Written By: Jeremy Worley
|          Date: Sep 23, 1993 13:12
------------------------------------------------------------*/
static int
_kdms_match_generic(
   kdms_generic_attribute * atr1,
   kdms_generic_attribute * atr2)
{
   int i;

   /* sanity check */
   if (atr1 == NULL || atr2 == NULL)
   {
      kinfo(KDEBUG, "_kdms_match_generic : null attribute provided\n");
      return FALSE;
   }

   /* do the attributes have the same characteristics? */
   if ((atr1->data_type != atr2->data_type) ||
       (atr1->arg_size != atr2->arg_size) ||
       (atr1->num_args != atr2->num_args))
      return FALSE;		/* not an error */

   if (atr1->data_type == KSTRING)
   {
      kstring *c1data = (kstring *) atr1->value;
      kstring *c2data = (kstring *) atr2->value;
      for (i = 0; i < atr1->num_args; i++)
	 if (kstrcmp(c1data[i], c2data[i]) != 0)
	    return FALSE;	/* not an error */
   }
   else
   {
      if (kmemcmp(atr1->value, atr2->value, 
		  (unsigned)(atr1->arg_size * atr1->num_args *
			     kdata_size(atr1->data_type))) != 0)
	 return FALSE;	/* not an error */
   }

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _kdms_copy_generic
|       Purpose: perform a generic attribute copy
|    Written By: Jeremy Worley
|          Date: Sep 23, 1993 13:01
------------------------------------------------------------*/
static int
_kdms_copy_generic(
   kdms_generic_attribute * atr1,
   kdms_generic_attribute * atr2)
{
   int status;

   /* sanity check */
   if (atr1 == NULL || atr2 == NULL)
   {
      kinfo(KDEBUG, "_kdms_copy_generic : null attribute provided\n");
      return FALSE;
   }

   /* only copy if both attributes have the same characteristics */
   if ((atr1->data_type != atr2->data_type) ||
       (atr1->arg_size != atr2->arg_size) ||
       (atr1->num_args != atr2->num_args))
   {
      _kdms_set_error(KDMS_EATTR_COPY_FAILED);
      kinfo(KDEBUG, "_kdms_copy_generic : "
	    "attribute 1 : "
	    "data type = %s, argument size = %d, number of arguments = %d\n"
	    "attribute 2 : "
	    "data type = %s, argument size = %d, number of arguments = %d\n",
       kdefine_to_datatype(atr1->data_type), atr1->arg_size, atr1->num_args,
      kdefine_to_datatype(atr2->data_type), atr2->arg_size, atr2->num_args);
      return FALSE;
   }

   status = _kdms_copy_attribute_data(atr1->value, &atr2->value,
				      atr1->data_type, atr1->arg_size,
				      atr1->num_args);
   return status;
}


/*-----------------------------------------------------------
|  Routine Name: (static) _kdms_query_generic
|       Purpose: perform a generic attribute query
|    Written By: Jeremy Worley and Steve Kubica
|          Date: Sep 23, 1993 13:09
------------------------------------------------------------*/
static int
_kdms_query_generic(
   kdms_generic_attribute *atr,
   int                    *num_args,
   int                    *arg_size,
   int                    *data_type,
   int                    *permanent)
{
   /* sanity check */
   if (atr == NULL)
   {
      kinfo(KDEBUG, "_kdms_query_generic : null attribute provided\n");
      return FALSE;
   }

   if (num_args)
      *num_args = atr->num_args;

   if (arg_size)
      *arg_size = atr->arg_size;

   if (data_type)
      *data_type = atr->data_type;

   if (permanent)
      *permanent = atr->permanent;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _kdms_print_generic
|       Purpose: perform a generic attribute print
|    Written By: Steve Kubica
|          Date: Mar 21, 1994 15:28
------------------------------------------------------------*/
static int
_kdms_print_generic(
   kdms_generic_attribute *atr,
   kfile *outfile)
{
   int i;
   int j;
   int status = TRUE;

   if (atr->value == NULL)
   {
      kinfo(KDEBUG, "_kdms_print_generic : null attribute provided\n");
      return FALSE;
   }

   /* sanity check: if the size or number of arguments is zero, then return */
   if (atr->arg_size == 0 || atr->num_args == 0)
   {
      kinfo(KDEBUG, "_kdms_attribute_data_to_vargs : erroneous input\n"
	    "   arg_size = %d\n   num_args = %d\n",
	    atr->arg_size, atr->num_args);
      return FALSE;
   }

   switch (atr->data_type)
   {
      case KBYTE:
	 {
	    signed char *tmp = (signed char *)atr->value;
	    for (i = 0; i < atr->num_args; i++)
	    {
	       for (j = 0; j < atr->arg_size; j++)
		  kfprintf(outfile, " %d ", tmp[j + i * atr->arg_size]);
	       if (i < atr->num_args - 1)
		  kfprintf(outfile, ",");
	    }
	 }
	 break;

      case KUBYTE:
	 {
	    unsigned char *tmp = (unsigned char *)atr->value;
	    for (i = 0; i < atr->num_args; i++)
	    {
	       for (j = 0; j < atr->arg_size; j++)
		  kfprintf(outfile, " %d ", tmp[j + i * atr->arg_size]);
	       if (i < atr->num_args - 1)
		  kfprintf(outfile, ",");
	    }
	 }
	 break;

      case KSHORT:
	 {
	    short *tmp = (short *)atr->value;
	    for (i = 0; i < atr->num_args; i++)
	    {
	       for (j = 0; j < atr->arg_size; j++)
		  kfprintf(outfile, " %d ", tmp[j + i * atr->arg_size]);
	       if (i < atr->num_args - 1)
		  kfprintf(outfile, ",");
	    }
	 }
	 break;

      case KUSHORT:
	 {
	    unsigned short *tmp = (unsigned short *)atr->value;
	    for (i = 0; i < atr->num_args; i++)
	    {
	       for (j = 0; j < atr->arg_size; j++)
		  kfprintf(outfile, " %d ", tmp[j + i * atr->arg_size]);
	       if (i < atr->num_args - 1)
		  kfprintf(outfile, ",");
	    }
	 }
	 break;

      case KINT:
	 {
	    int *tmp = (int *)atr->value;
	    for (i = 0; i < atr->num_args; i++)
	    {
	       for (j = 0; j < atr->arg_size; j++)
		  kfprintf(outfile, " %d ", tmp[j + i * atr->arg_size]);
	       if (i < atr->num_args - 1)
		  kfprintf(outfile, ",");
	    }
	 }
	 break;

      case KUINT:
	 {
	    unsigned int *tmp = (unsigned int *)atr->value;
	    for (i = 0; i < atr->num_args; i++)
	    {
	       for (j = 0; j < atr->arg_size; j++)
		  kfprintf(outfile, " %d ", tmp[j + i * atr->arg_size]);
	       if (i < atr->num_args - 1)
		  kfprintf(outfile, ",");
	    }
	 }
	 break;

      case KLONG:
	 {
	    long *tmp = (long *)atr->value;
	    for (i = 0; i < atr->num_args; i++)
	    {
	       for (j = 0; j < atr->arg_size; j++)
		  kfprintf(outfile, " %d ", tmp[j + i * atr->arg_size]);
	       if (i < atr->num_args - 1)
		  kfprintf(outfile, ",");
	    }
	 }
	 break;

      case KULONG:
	 {
	    unsigned long *tmp = (unsigned long *)atr->value;
	    for (i = 0; i < atr->num_args; i++)
	    {
	       for (j = 0; j < atr->arg_size; j++)
		  kfprintf(outfile, " %d ", tmp[j + i * atr->arg_size]);
	       if (i < atr->num_args - 1)
		  kfprintf(outfile, ",");
	    }
	 }
	 break;

      case KFLOAT:
	 {
	    float *tmp = (float *)atr->value;
	    for (i = 0; i < atr->num_args; i++)
	    {
	       for (j = 0; j < atr->arg_size; j++)
		  kfprintf(outfile, " %g ", tmp[j + i * atr->arg_size]);
	       if (i < atr->num_args - 1)
		  kfprintf(outfile, ",");
	    }
	 }
	 break;

      case KDOUBLE:
	 {
	    double *tmp = (double *)atr->value;
	    for (i = 0; i < atr->num_args; i++)
	    {
	       for (j = 0; j < atr->arg_size; j++)
		  kfprintf(outfile, " %g ", tmp[j + i * atr->arg_size]);
	       if (i < atr->num_args - 1)
		  kfprintf(outfile, ",");
	    }
	 }
	 break;

      case KCOMPLEX:
	 {
	    kcomplex *tmp = (kcomplex *) atr->value;
	    for (i = 0; i < atr->num_args; i++)
	    {
	       for (j = 0; j < atr->arg_size; j++)
		  kfprintf(outfile, " ( %g, %g ) ",
			   (tmp[j + i * atr->arg_size]).r,
			   (tmp[j + i * atr->arg_size]).i);
	       if (i < atr->num_args - 1)
		  kfprintf(outfile, ",");
	    }
	 }
	 break;

      case KDCOMPLEX:
	 {
	    kdcomplex *tmp = (kdcomplex *) atr->value;
	    for (i = 0; i < atr->num_args; i++)
	    {
	       for (j = 0; j < atr->arg_size; j++)
		  kfprintf(outfile, " ( %g, %g ) ",
			   (tmp[j + i * atr->arg_size]).r,
			   (tmp[j + i * atr->arg_size]).i);
	       if (i < atr->num_args - 1)
		  kfprintf(outfile, ",");
	    }
	 }
	 break;

      case KSTRING:
	 {
	    kstring *tmp = (kstring *) atr->value;
	    for (i = 0; i < atr->num_args; i++)
	    {
	       for (j = 0; j < atr->arg_size; j++)
		  if (tmp[j + i * atr->arg_size] != NULL)
		     kfprintf(outfile, " %s ", tmp[j + i * atr->arg_size]);
		  else 
		     kfprintf(outfile, " (null) ");
		  
	       if (i < atr->num_args - 1)
		  kfprintf(outfile, ",");
	    }
	 }
	 break;

      default:
	 _kdms_set_error(KDMS_EATTR_INVALID);
	 kinfo(KDEBUG, "_kdms_print_generic : data type %d is invalid",
	       atr->data_type);
	 status = FALSE;
   }

   return status;
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _kdms_construct_generic
|
|       Purpose: This routine will construct a generic-type
|                attribute according to the given arguments.
|
|		 The attribute is distinguished by the attribute
|		 identifier, which is simply a tokenized
|		 version of the attribute name, such as "size".
|
|		 A default data block may be provided for
|		 initializing the value of the attribute. If
|		 a default value is provided, a copy of that
|		 value will be made and assigned to the attribute.
|		 If no default is provided (NULL), then the
|		 attribute value will be initialized to be NULL.
|
|                Note that the constructors for each attribute
|                type are implemented as separate functions 
|                because of their wildly different function parameters.
|
|         Input: attribute  - attribute identifier 
|		 data_type  - data type of the attribute
|  		 arg_size   - size of each attribute argument
|		 num_args   - number of arguments in attribute
| 		 permanent  - is this attribute permanent
|		 def        - default values to use when
|                             initializing the attribute value.
|        Output: none
|
|       Returns: a pointer to the constructed attribute on success,
|		 NULL otherwise
|
|    Written By: Steve Kubica
|          Date: Mar 17, 1994 10:47
| Modifications:
|
------------------------------------------------------------*/
static kdms_attribute *
_kdms_construct_generic(
   int attribute,
   int data_type,
   int arg_size,
   int num_args,
   int permanent,
   kaddr def)
{
   kdms_attribute *atr = NULL;

   /* -- allocate a new attribute structure -- */
   if ((atr = (kdms_attribute *) kcalloc(1, sizeof(kdms_attribute))) == NULL)
   {
      _kdms_set_error(KMEMORY_ALLOCATION);
      return NULL;
   }

   /* -- assign the base information -- */
   atr->generic.type = KDMS_GENERIC_ATTRIBUTE_TYPE;
   atr->generic.attribute = attribute;

   /* -- fill out the generic attribute information -- */
   atr->generic.data_type = data_type;
   atr->generic.arg_size = arg_size;
   atr->generic.num_args = num_args;
   atr->generic.permanent = permanent;
   atr->generic.value = NULL;

   /* -- assign the default data (if present) to the attributes value -- */
   if (def)
      _kdms_copy_attribute_data(def, &(atr->generic.value),
				data_type, arg_size, num_args);
   else
      atr->generic.value = kcalloc((unsigned)(arg_size * num_args), 
				   (unsigned)kdata_size(data_type));

   return atr;
}

/*
 *    ================================================================== 
 *    Quasi Attribute Methods
 *    ==================================================================
 */

/*
 * Centerline's CLCC doesn't honor the ARGSUSED *standard* so
 * this pragma turns off warnings about unused arguments, which
 * are numerous in the next few routines
 */
#ifdef __CLCC__
#pragma Warning_level (3)
#endif

/*-----------------------------------------------------------
|  Routine Name: (static) _kdms_get_undefined
|       Purpose: used if a quasi-attribute get method is undefined
|    Written By: Steve Kubica
|          Date: Mar 21, 1994 16:21
------------------------------------------------------------*/
/* ARGSUSED */
static int
_kdms_get_undefined(kobject obj, int assoc, int attrib, kaddr clientData,
		    kva_list * list)
{
   /* -- try to fall through to a generic attribute -- */
   if (!kdms_vget_attribute(obj, ktoken_to_string(assoc), 
			    ktoken_to_string(attrib), list))
   {
      kinfo(KDEBUG, "kdms_get_undefined : get method for '%s' is not defined",
	    ktoken_to_string(attrib));
      return FALSE;
   }
   
   return TRUE;
}


/*-----------------------------------------------------------
|  Routine Name: (static) _kdms_set_undefined
|       Purpose: used if a quasi-attribute put method is undefined
|    Written By: Steve Kubica
|          Date: Mar 21, 1994 16:21
------------------------------------------------------------*/
/* ARGSUSED */
static int
_kdms_set_undefined(kobject obj, int assoc, int attrib, kaddr clientData,
		    kva_list * list)
{
   /* -- try to fall through to a generic attribute -- */
   if (!kdms_vset_attribute(obj, ktoken_to_string(assoc), 
			    ktoken_to_string(attrib), list))
   {
      kerror ("kdataman", "kdms_set_attribute", "The %s attribute "
	      "is a read-only attribute.  Any attempt to set this "
	      "attribute will fail.", ktoken_to_string(attrib));
      return FALSE;
   }
   
   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _kdms_match_undefined
|       Purpose: used if a quasi-attribute match method is undefined
|    Written By: Steve Kubica
|          Date: Mar 21, 1994 16:21
------------------------------------------------------------*/
/* ARGSUSED */
static int
_kdms_match_undefined(kobject obj1, kobject obj2, int assoc, int attrib,
		      kaddr clientData1, kaddr clientData2)
{
   /* -- try to fall through to a generic attribute -- */
   if (!kdms_match_attribute(obj1, obj2, ktoken_to_string(assoc), 
			    ktoken_to_string(attrib)))
   {
      kinfo(KDEBUG, "kdms_match_undefined : match method for '%s' is not "
	            "defined", ktoken_to_string(attrib));
      return FALSE;
   }
   
   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _kdms_copy_undefined
|       Purpose: used if a quasi-attribute copy method is undefined
|    Written By: Steve Kubica
|          Date: Mar 21, 1994 16:21
| Modifications:
------------------------------------------------------------*/
/* ARGSUSED */
static int
_kdms_copy_undefined(kobject obj1, kobject obj2, int assoc, int attrib,
		     kaddr clientData1, kaddr clientData2)
{
   /* -- try to fall through to a generic attribute -- */
   if (!kdms_copy_attribute(obj1, obj2, ktoken_to_string(assoc), 
			    ktoken_to_string(attrib)))
   {
      kinfo(KDEBUG, "kdms_copy_undefined : copy method for '%s' is not "
	            "defined", ktoken_to_string(attrib));
      return FALSE;
   }
   
   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _kdms_query_undefined
|       Purpose: used if a quasi-attribute query method is undefined
|    Written By: Steve Kubica
|          Date: Mar 21, 1994 16:21
------------------------------------------------------------*/
/* ARGSUSED */
static int
_kdms_query_undefined(kobject obj, int assoc, int attrib, kaddr clientData,
		      int *num_args, int *arg_size, int *data_type,
		      int *permanent)
{
   /* -- try to fall through to a generic attribute -- */
   if (!kdms_query_attribute(obj, ktoken_to_string(assoc), 
			     ktoken_to_string(attrib), num_args, arg_size,
			     data_type, permanent))
   {
      kinfo(KDEBUG, "kdms_query_undefined : query method for '%s' is not "
	            "defined", ktoken_to_string(attrib));
      return FALSE;
   }
   
   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _kdms_print_undefined
|       Purpose: used if a quasi-attribute print method is undefined
|    Written By: Steve Kubica
|          Date: Mar 21, 1994 15:28
------------------------------------------------------------*/
/* ARGSUSED */
static int
_kdms_print_undefined(kobject obj, int assoc, int attrib, kaddr clientData,
		      kfile *outfile)
{
   /* -- try to fall through to a generic attribute -- */
   if (!kdms_print_attribute(obj, ktoken_to_string(assoc), 
			     ktoken_to_string(attrib), outfile))
   {
      kinfo(KDEBUG, "kdms_print_undefined : print method for '%s' is not "
	            "defined", ktoken_to_string(attrib));
      return FALSE;
   }
   
   return TRUE;
}

/*
 * Centerline's CLCC doesn't honor the ARGSUSED *standard* so
 * this pragma turns back on warnings about unused arguments, which
 * occur in the above service routines.
 * are numerous in this file.
 */
#ifdef __CLCC__
#pragma Warning_level (4)
#endif

/*-----------------------------------------------------------
|
|  Routine Name: (static) _kdms_construct_quasi
|
|       Purpose: This routine will construct a quasi-type
|                attribute according to the given arguments.
|
|		 The attribute is distinguished by the attribute
|		 identifier, which is simply a tokenized
|		 version of the attribute name, such as "size".
|
|		 Functions are provided for each of the attribute's
|		 methods. If NULL is passed in for any of the
|		 functions, an undefined function will be used.
|
|		 Client data may be given to the attribute.
|		 This client data will be passed into each
|		 of the given attribute methods.  Note that
|		 a copy of this client data is _not_ made,
|		 so the address of a static variable must be
|		 provided.
|		 
|                Note that the constructors for each attribute
|                type are implemented as separate functions 
|                because of their wildly different function parameters.
|
|         Input: attribute   - attribute name
|		 get()       - get function to use
|  		 set()       - set function to use
|		 match()     - match function to use
| 		 copy()      - copy function to use 
|		 query()     - query function to use 
|		 print()     - print function to use 
|		 clientData  - client data to pass into the attribute's
|			       methods.
|        Output: none
|
|       Returns: a pointer to the constructed attribute on success,
|		 NULL otherwise
|
|    Written By: Steve Kubica
|          Date: Mar 21, 1994 15:28
| Modifications:
|
------------------------------------------------------------*/

static kdms_attribute *
_kdms_construct_quasi(
   char *attribute,
   int (*get)   PROTO((kobject, int, int, kaddr, kva_list *)),
   int (*set)   PROTO((kobject, int, int, kaddr, kva_list *)),
   int (*match) PROTO((kobject, kobject, int, int , kaddr, kaddr)),
   int (*copy)  PROTO((kobject, kobject, int, int , kaddr, kaddr)),
   int (*query) PROTO((kobject, int, int, kaddr, int *, int *, int *, int *)),
   int (*print) PROTO((kobject, int, int, kaddr, kfile *)),
   kaddr clientData)
{
   kdms_attribute *atr = NULL;

   /* allocate a new attribute structure */
   if ((atr = (kdms_attribute *) kcalloc(1, sizeof(kdms_attribute))) == NULL)
   {
      _kdms_set_error(KMEMORY_ALLOCATION);
      return NULL;
   }

   /* assign the base information */
   atr->quasi.type = KDMS_QUASI_ATTRIBUTE_TYPE;
   
   /* tokenized versions for handy reference */
   atr->quasi.attribute   = kstring_to_token(attribute);

   /* fill out the generic attribute methods */
   atr->quasi.get   = (get)   ? get   : _kdms_get_undefined;
   atr->quasi.set   = (set)   ? set   : _kdms_set_undefined;
   atr->quasi.match = (match) ? match : _kdms_match_undefined;
   atr->quasi.copy  = (copy)  ? copy  : _kdms_copy_undefined;
   atr->quasi.query = (query) ? query : _kdms_query_undefined;
   atr->quasi.print = (print) ? print : _kdms_print_undefined;

   /* assign the clientData for future use */
   atr->quasi.clientData = clientData;

   return atr;
}

/*
 *    ================================================================== 
 *    Attribute Methods
 *    ==================================================================
 */

/*-----------------------------------------------------------
|
|  Routine Name: (static) _kdms_destruct_attribute
|
|       Purpose: Destroy an attribute.  This function can
|		 destroy both generic and quasi attributes.
|		 All internal attribute data, as well as 
|		 the attribute itself, will be freed.
|
|         Input: atr - the attribute to free
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica
|          Date: Mar 17, 1994 10:47
| Modifications:
|
------------------------------------------------------------*/
static void
_kdms_destruct_attribute(kdms_attribute *atr)
{
   int i;

   /* -- trivial free -- */
   if (atr == NULL)
      return;

   /* -- this is only case where attribute contains something to free -- */  
   if (atr->base.type == KDMS_GENERIC_ATTRIBUTE_TYPE)
   {
      if (atr->generic.data_type == KSTRING)
      {
	 kstring *cdata = (kstring *) atr->generic.value;
	 for (i = 0; i < atr->generic.arg_size * atr->generic.num_args; i++)
	    kfree(cdata[i]);
      }
      kfree(atr->generic.value);
   }

   /* -- finally, free the attribute -- */
   atr->base.type = -1; /* -- sanity -- */

   kfree(atr);

   return;
}

/*
 *    ================================================================== 
 *    Attribute List Methods
 *    ==================================================================
 */

/*-----------------------------------------------------------
|
|  Routine Name: (static) _kdms_append_attribute
|
|       Purpose: Append an attribute to a given attribute list.
|                If the provided list is NULL, or contains no
|                attributes, the list will be initialized and
|                the attribute will be added.
|
|         Input: atr_list - the address of the list to add to
|                atr      - the attribute to add to the list
|
|        Output: none
|
|       Returns: atr_list - returns the pointer to the list
|		            after modification
|
|    Written By: Steve Kubica
|          Date: Mar 21, 1994 16:21
| Modifications:
|
------------------------------------------------------------*/
static kdms_attribute_list *
_kdms_append_attribute(
   kdms_attribute_list *atr_list,
   kdms_attribute      *atr)
{
   /* -- if there is no attribute to append, then do nothing -- */
   if (atr == NULL)
   {
      kinfo(KDEBUG, "_kdms_append_attribute : can not append null attribute");
      return atr_list;
   }

   /* -- if the list passed in is NULL, then create it -- */
   if (atr_list == NULL)
   {
      atr_list = (kdms_attribute_list *) 
	 kcalloc(1, sizeof(kdms_attribute_list));

      if (atr_list == NULL)
      {
	 _kdms_set_error(KMEMORY_ALLOCATION);
	 return NULL;
      }
      atr_list->nattributes = 0;
      atr_list->attributes = NULL;
   }

   /* -- increase the number of attributes on the attribute list -- */
   atr_list->nattributes++;
   atr_list->attributes = (kdms_attribute **) 
      krealloc(atr_list->attributes,
	       atr_list->nattributes *
	       sizeof(kdms_attribute *));

   if (atr_list->attributes == NULL)
   {
      _kdms_set_error(KMEMORY_ALLOCATION);
      return NULL;
   }

   /* -- finally, append the attribute to the list -- */
   atr_list->attributes[atr_list->nattributes - 1] = atr;

   return atr_list;
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _kdms_locate_attribute
|
|       Purpose: Given an attribute identifier, locate the 
|                attribute in a given list and return it.
|
|         Input: atr_list  - the address of the list to search
|                attribute - identifier of the attribute to search for
|
|        Output: none
|
|       Returns: the attribute if it is on the list, NULL otherwise
|
|    Written By: Steve Kubica
|          Date: Mar 21, 1994 16:21
| Modifications:
|
------------------------------------------------------------*/
static kdms_attribute *
_kdms_locate_attribute(
   kdms_attribute_list *atr_list,
   int                  attribute)
{
   int i;

   if (atr_list == NULL)
      return NULL;

   /* check each attribute and return the one that matches */
   for (i = 0; i < atr_list->nattributes; i++)
      if (atr_list->attributes[i]->base.attribute == attribute)
	 return atr_list->attributes[i];

   return NULL;
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _kdms_delete_attribute
|
|       Purpose: Given an attribute identifier, locate the 
|                attribute in a given list and delete it.
|
|         Input: atr_list  - the address of the list to delete from
|                attribute - identifier of the attribute to delete
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica
|          Date: Mar 21, 1994 16:21
| Modifications:
|
------------------------------------------------------------*/
static int
_kdms_delete_attribute(
   kdms_attribute_list *atr_list,
   int                  attribute)
{
   int i = 0;
   int j;
   int found = FALSE;

   if (atr_list == NULL)
      return FALSE;

   /* -- see if it is on the list, and where -- */
   do
   {
      if (atr_list->attributes[i]->base.attribute == attribute)
	 found = TRUE;
      else
	 i++;
   }
   while (i < atr_list->nattributes && !found);

   /* if it's on the list, then delete it */
   if (found)
   {
      _kdms_destruct_attribute(atr_list->attributes[i]);

      /* -- shorten the list -- */
      atr_list->nattributes--;

      /* -- compress rest of list down - this works even at end of list -- */
      for (j = i; j < atr_list->nattributes; j++)
	 atr_list->attributes[j] = atr_list->attributes[j + 1];

      atr_list->attributes = (kdms_attribute **) 
	 krealloc(atr_list->attributes,
		  atr_list->nattributes *
		  sizeof(kdms_attribute *));
   }
   else
   {
      _kdms_set_error(KDMS_EATTR_NOT_FOUND);
      kinfo(KDEBUG, "_kdms_delete_attribute : unable to find '%s' on the list",
	    ktoken_to_string(attribute));
      return FALSE;
   }

   return TRUE;
}


/*
 *    ================================================================== 
 *    Attribute Definition Methods
 *    ==================================================================
 */

/*-----------------------------------------------------------
|
|  Routine Name: (static) _kdms_construct_generic_defin
|
|       Purpose: This routine will construct a generic-type
|                attribute definition according to the given arguments.
|
|		 The attribute definition is distinguished by its
|		 attribute identifier, which is simply a tokenized
|		 version of the attribute name.
|
|		 This attribute definition will be used as a template
|                for creating (establishing) instances of the attribute
|		 on an actual data object.  Attributes can be 
|		 established either on the main object (object-level)
|		 or on any of the object's data segments (segment-level).
|		 The definition's "association" determines where it is
| 	         legal to establish the attribute.
|
|                The association is simply a tokenized string.  The 
|                string itself indicates where it is legal to establish
|                an attribute.  A NULL association indicates that the
|		 attribute should established at the object-level.
|		 An association of KDMS_ALL_SEGMENTS indicates that the
|		 attribute can be established at the segment level 
|		 on any given segment name.  If a specific segment name,
| 		 such as "value" is given for the association, then
|		 the attribute can only established on the "value" segment.
|		
|		 In addition to being established either at the
|		 object or segment level, an attribute can also
|		 be established at the physical or presentation level.
|
|		 Changes to physical level attributes are visible
|		 to all references of the data object and are termed
|		 as "shared" attributes.  Changes to presentation
|		 level attributes are visible only to the reference
|		 object on which the change was made, and thus 
|		 the attribute is termed as "unshared". 
|
|		 A default data block may be provided for
|		 initializing the value of any attributes created
|                from this definition template. If a default value 
|		 is provided, a copy of that value will be made and 
|		 assigned to the definition. If no default is 
|		 provided (NULL), then the the default value will
|		 be initialized to be NULL.
|
|                Note that the constructors for each attribute
|                definition type are implemented as separate functions 
|                because of their wildly different function parameters.
|
|         Input: attribute   - attribute identifier 
|		 association - attribute association
|		 data_type   - data type of the attribute
|  		 arg_size    - size of each attribute argument
|		 num_args    - number of arguments in attribute
| 		 permanent   - TRUE if attribute is permanent
|	         shared      - TRUE if attribute is shared
|		 def         - default values to use when
|                              initializing the attribute value.
|        Output: none
|
|       Returns: a pointer to the constructed attribute definition 
|		 on success, NULL otherwise
|
|    Written By: Steve Kubica
|          Date: Mar 17, 1994 10:47
| Modifications:
|
------------------------------------------------------------*/
static kdms_attribute_defin *
_kdms_construct_generic_defin(
   char *attribute,
   char *association,
   int   data_type,
   int   arg_size,
   int   num_args,
   int   permanent,
   int   shared,
   kaddr def)
{
   kdms_attribute_defin *defin;

   /* -- allocate a new attribute definition structure -- */
   if ((defin = (kdms_attribute_defin *)
	kcalloc(1, sizeof(kdms_attribute_defin))) == NULL)
   {
      _kdms_set_error(KMEMORY_ALLOCATION);
      return NULL;
   }

   /* -- fill out the attribute definition structure -- */
   defin->attribute   = kstrdup(attribute);
   defin->association = kstrdup(association);
   defin->shared      = shared;

   defin->atr = _kdms_construct_generic(kstring_to_token(attribute), 
					data_type, arg_size,
					num_args, permanent, def);
   if (defin->atr == NULL)
   {
      kfree(defin);
      kinfo(KDEBUG, "_kdms_construct_generic_defin : unable to construct "
	    "a generic attribute for definition of '%s'.", attribute);
      return NULL;
   }

   return defin;
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _kdms_construct_quasi_defin
|
|       Purpose: This routine will construct a quasi-type
|                attribute definition according to the given arguments.
|
|		 The attribute definition is distinguished by its
|		 attribute identifier, which is simply a tokenized
|		 version of the attribute name. 
|
|		 Quasi-attributes are essentially "action" attributes.
|	         They technically have no data associated with them.
|		 They instead consist of a number of attribute functions, 
|		 or methods.  These methods, provided when calling
|		 this function, provide specific actions which are invoked
|		 from higher level public functions.  
|
|		 Specifically, these functions are :
|
|		 !   get method   - kdms_get_attribute(s)
|		 !   set method   - kdms_set_attribute(s)
|		 !   match method - kdms_match_attribute(s)
|		 !   copy method  - kdms_copy_attribute(s)
|		 !   query method - kdms_query_attribute
|		 !   print method - kdms_print_attribute
|		 
|                Each of these higher-level functions uses a "segment"
|		 parameter to indicate whether the attribute is 
|		 considered to be associated with the main object
|		 (object-level), or with a specific segment (segment-level).
|		 While this notion is purely conceptual for a 
|		 quasi-attribute, it provides a means for defining
|	         identically named attributes at different levels
|		 or for different segments. 
|		 
|		 The "association" provided for the definition will be 
|		 used to check against the provided association from the 
|		 higher-level segment argument.  The association must 
|		 match before the attribute's method is called. 
|
|                The association provided is simply a tokenized string.
|                The string itself indicates what associations are legal
|		 for invoking the quasi-attribute methods.  A NULL
|                association indicates that the quasi-attribute methods
|		 can be evoked at the object level, but not at the 
|		 segment level.
|
|		 An association of KDMS_ALL_SEGMENTS indicates that the
|		 attribute can be evoked at the segment level 
|		 on any given segment name.  If a specific segment name,
| 		 such as "value" is given for the association, then
|		 the attribute can only established on the "value" segment.
|
|		 Functions are provided for each of the attribute's
|		 methods. If NULL is passed in for any of the
|		 functions, an undefined function will be used.
|
|		 Client data may be given to the attribute definition.
|		 This client data will be passed into each of the given 
|		 attribute methods.  Note that a copy of this client 
|		 data is _not_ made, so the address of a static 
|		 variable must be provided.
|
|                Note that the constructors for each attribute
|                definition type are implemented as separate functions 
|                because of their wildly different function parameters.
|
|         Input: attribute   - attribute identifier 
|                association - attribute association
|		 get()       - get function to use
|  		 set()       - set function to use
|		 match()     - match function to use
| 		 copy()      - copy function to use 
|		 query()     - query function to use 
|		 print()     - print function to use 
|		 clientData  - client data to pass into the attribute's
|			       methods.
|        Output: none
|
|       Returns: a pointer to the constructed attribute definition
|		 on success, NULL otherwise
|
|    Written By: Steve Kubica
|          Date: Mar 17, 1994 10:47
| Modifications:
|
------------------------------------------------------------*/
static kdms_attribute_defin *
_kdms_construct_quasi_defin(
   char *attribute,
   char *association,
   int (*get)   PROTO((kobject, int, int, kaddr, kva_list *)),
   int (*set)   PROTO((kobject, int, int, kaddr, kva_list *)),
   int (*match) PROTO((kobject, kobject, int, int, kaddr, kaddr)),
   int (*copy)  PROTO((kobject, kobject, int, int, kaddr, kaddr)),
   int (*query) PROTO((kobject, int, int,kaddr,int *,int *,int *,int *)),
   int (*print) PROTO((kobject, int, int, kaddr, kfile *)),
   kaddr clientData)
{
   kdms_attribute_defin *defin;

   /* -- allocate a new attribute definition structure -- */
   if ((defin = (kdms_attribute_defin *)
	kcalloc(1, sizeof(kdms_attribute_defin))) == NULL)
   {
      _kdms_set_error(KMEMORY_ALLOCATION);
      return NULL;
   }

   /* -- fill out the attribute definition structure -- */
   defin->attribute   = kstrdup(attribute);
   defin->association = kstrdup(association);
   defin->shared      = TRUE;

   defin->atr = _kdms_construct_quasi(attribute, 
				      get, set, match, copy,
				      query, print, clientData);
   if (defin->atr == NULL)
   {
      kinfo(KDEBUG, "_kdms_construct_quasi_defin : unable to construct "
	    "quasi attribute %s.", attribute);
      kfree(defin);
      return NULL;
   }

   return defin;
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _kdms_destruct_attribute_defin
|
|       Purpose: Destroy an attribute definition.  This 
|		 function can destroy both generic and quasi 
|		 attribute definitions.  All internal attribute 
|		 data, as well as the attribute definition 
|		 itself, will be freed.
|
|         Input: defin - the attribute definition to free
|
|        Output: none
|
|       Returns: none
|
|    Written By: Steve Kubica
|          Date: Mar 17, 1994 10:47
| Modifications:
|
------------------------------------------------------------*/
static void
_kdms_destruct_attribute_defin(kdms_attribute_defin *defin)
{
   /* -- trivial free -- */
   if (defin == NULL)
      return;

   /* -- free the internals of the definition -- */
   kfree(defin->attribute);
   kfree(defin->association);

   _kdms_destruct_attribute(defin->atr);

   /* -- finally free the attribute definition -- */
   kfree(defin);

   return;
}

/*
 *    ================================================================== 
 *    Attribute Definition Hash Table Methods
 *    ==================================================================
 */

/*-----------------------------------------------------------
|
|  Routine Name: (static) _kdms_hash_attribute_defin
|
|       Purpose: Hash an attribute definition to the static
|		 attribute definition hash table. 
|
|         Input: defin - attribute definition to hash
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica
|          Date: Mar 21, 1994 16:21
| Modifications:
|
------------------------------------------------------------*/
static int
_kdms_hash_attribute_defin(
   kdms_attribute_defin *defin)
{
   int indx;
   int hash;

   
   /* -- if there is no attribute define to append, then do nothing -- */
   if (defin == NULL)
   {
      kinfo(KDEBUG, "_kdms_hash_attribute_defin : "
	            "can not hash NULL attribute definition");
      return FALSE;
   }

   /* -- first try to see if the definition already exists... --  */
   if (defin->attribute == NULL)
      return FALSE;
   else if (_kdms_find_defin(defin->attribute, 
			     defin->association, &indx) != NULL)
      return FALSE;

   /* -- make a new entry into the hash table -- */
   hash = hash_next++;
   
   /* -- put the entry into the hash table -- */
   hash_table[indx] = klist_add(hash_table[indx], (kaddr) hash,
				  (kaddr) defin);
   return TRUE;
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _kdms_locate_attribute_defin
|
|       Purpose: Given an attribute identifier, locate the 
|                attribute definition on the static attribute
|		 hash table and return it.
|
|         Input: attribute   - identifier of attribute definition to search for
|		 association - association to use when searching
|
|        Output: none
|
|       Returns: the attribute definition if it is on the list, NULL otherwise
|
|    Written By: Steve Kubica
|          Date: Mar 21, 1994 16:21
| Modifications:
|
------------------------------------------------------------*/
static kdms_attribute_defin *
_kdms_locate_attribute_defin(
   char *attribute,
   char *association)
{
   int indx;
   kdms_attribute_defin *defin = NULL;

   if (!attribute)
      return NULL;
   else if ((defin = _kdms_find_defin(attribute, association, &indx)) != NULL)
      return defin;
   else 
      return NULL;
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _kdms_delete_attribute_defin
|
|       Purpose: Given an attribute and association, locate the 
|                attribute definition on the static attribute
|		 hash table and delete it.
|
|         Input: attribute   - name of attribute definition to delete.
|                association - corresponding attribute association.
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica
|          Date: Mar 21, 1994 16:21
| Modifications:
|
------------------------------------------------------------*/
static int
_kdms_delete_attribute_defin(
   char *attribute,
   char *association)
{
   kdms_attribute_defin *defin;
   klist *list;
   int    indx;
   int    id = -1;
   
   
   /* -- make sure we have a valid attribute name -- */
   if (attribute == NULL)
      return FALSE;

   /* -- get the index into the hash table -- */
   else if ((defin = _kdms_find_defin(attribute, association, &indx)) == NULL)
      return TRUE;

   for (list = hash_table[indx]; list != NULL; list = klist_next(list))
   {
      /* -- find the definition id on the list -- */
      if (defin == list->client_data)
	 id = (int) list->identifier;
   }

   /*
    * If the id was not found, then simply return...
    */
   if (id == -1)
     return (FALSE);
   
   /* -- delete the hash table entry ...  */
   hash_table[indx] = klist_delete(hash_table[indx], (kaddr) id);
   
   /* -- ... and free the definition structure */
   _kdms_destruct_attribute_defin(defin);

   return TRUE;
}

/*-----------------------------------------------------------
|
|  Routine Name: _kdms_free_list_attribute_defin
|
|       Purpose: Free up a list node from a klist structure
|		 under the assumption that the client_data of
|		 the list node is an attribute defintion.
|
|         Input: list
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Steve Kubica
|          Date: Dec 19, 1994 16:21
| Modifications:
|
------------------------------------------------------------*/
static void
_kdms_free_list_attribute_defin(klist *list)
{
   if (list == NULL)
      return;
   
   _kdms_destruct_attribute_defin((kdms_attribute_defin *)list->client_data);

   return;
}

/*-----------------------------------------------------------
|
|  Routine Name: kdms_free_attribute_defins
|
|       Purpose: Free up the static attribute definition hash table.
|		 All the definitions are freed and then the table
|		 itself is freed.
|
|         Input: none
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Steve Kubica
|          Date: Mar 21, 1994 16:21
| Modifications:
|
------------------------------------------------------------*/
void
kdms_free_attribute_defins(void)
{
   int i;

   for (i = 0; i < TABLESIZE; i++)
      klist_free(hash_table[i], _kdms_free_list_attribute_defin);

   kfree(hash_table);

   return;
}


/*
 *    ================================================================== 
 *    Attribute Management at the kobject Level 
 *    ==================================================================
 */

/*-----------------------------------------------------------
|
|  Routine Name: kdms_establish_attribute
|
|       Purpose: This routine will establish an attribute on
|		 a given attribute list which is a copy of
|		 a given attribute.  The established 
|	         attribute will be returned.
|
|		 Only generic attributes may be 
|		 physically instantiated on the list.  If a 
|		 quasi attribute is passed into this routine,
|		 and error will occur.
|			
|         Input: atr_list - the address of a pointer to the list
| 		            to establish the attribute on.
|		 atr      - the attribute to use as a template
|			    for establishing the new attribute
|		 	    to be established.
|
|        Output: none
|
|       Returns: the established attribute on success, NULL otherwise
|
|    Written By: Jeremy Worley and Steve Kubica
|          Date: Oct 04, 1993 12:58
| Modifications:
|
------------------------------------------------------------*/
kdms_attribute *
kdms_establish_attribute(
   kdms_attribute_list **atr_list,
   kdms_attribute       *atr)
{
   kdms_attribute *new_atr = NULL;

   if (atr_list == NULL || atr == NULL)
   {
      kinfo(KDEBUG, "kdms_establish_attribute : attribute or list is NULL");
      return NULL;
   }

   /* -- generic attributes need to be instantiated -- */
   if (atr->base.type == KDMS_GENERIC_ATTRIBUTE_TYPE)
   {
      new_atr = _kdms_construct_generic(atr->generic.attribute,
					atr->generic.data_type,
					atr->generic.arg_size,
					atr->generic.num_args,
					atr->generic.permanent,
					atr->generic.value);

      /* -- append attribute to the given attribute list -- */
      *atr_list = _kdms_append_attribute(*atr_list, new_atr);
   }

   /* -- quasi-attributes can not be established -- */
   else 
   {
      kerror("kdataman", "kdms_establish_attribute", 
	     "Bogus attempt to establish quasi attribute %s\n",
	     ktoken_to_string(atr->quasi.attribute));
      return NULL;
   }

   return new_atr;
}


/*-----------------------------------------------------------
|
|  Routine Name: (static) _kdms_locate_attribute_list
|
|       Purpose: Locate the attribute list in an object for
|	         a given association.   Either the physical
|		 or presentation attribute list for that
|		 association will be returned depending on
|		 the shared parameter.
|
|		 Remember, attributes at the physical level
|		 are considered to be shared among all
|		 reference objects.  Attributes at the 
|		 presentation level are specific to the
|		 presentation of that reference object and
|		 are thus unshared.
|
|         Input: obj    - the object to retrieve the list from
|                assoc  - the association of the list to retrieve
|                shared - TRUE if the physical list is desired
|                         FALSE if the presentation list is desired
|
|        Output: none
|
|       Returns: the attribute list for the given association,
| 		 NULL otherwise
|
|    Written By: Jeremy Worley and Steve Kubica
|          Date: Oct 04, 1993 12:58
| Modifications:
|
------------------------------------------------------------*/
static
kdms_attribute_list **
_kdms_locate_attribute_list(
   kobject obj,
   int     assoc,
   int     shared)
{
   kdms_attribute_list **at_list = NULL;
   kpresentation *pres = NULL;

   if (assoc == 0)
   {
      if (shared)		/* at physical */
	 at_list = &(obj->phys->attributes);
      else			/* at presentation */
	 at_list = &(obj->attributes);
   }
   else if (_kdms_get_segment_info(obj, assoc, &pres) && pres)
   {
      if (shared)		/* at physical */
	 at_list = &(pres->segment->attributes);
      else			/* at presentation */
	 at_list = &(pres->attributes);
   }
   else
   {
      _kdms_set_error(KDMS_ESEG_NONEXIST);
      kinfo(KDEBUG,
	    "_kdms_locate_attribute_list: The segment '%s' doesn't exist.\n",
	    ktoken_to_string(assoc));
      return NULL;
   }

   return at_list;
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _kdms_retrieve_attribute
|
|       Purpose: Retrieve an attribute from an object for a given
|	         association.  The association determines whether
|	         the attribute should be on the object level 
|		 attribute list or on a segment level attribute list.
|		 
|		 The presentation attribute list will be searched first, 
|		 followed by the physical attribute list.  If the
|		 attribute is not found on either of those lists,
|		 it will be established on one of the lists according
|		 to its definition.  If the attribute is shared,
|		 it will be established on the physical list. If it
|		 is not shared, it will be established on the presentation
|		 list.
|		 
|		 The resulting attribute, whether it was found, or
|		 established, will be returned.
|
|         Input: obj         - the object to retrieve the attribute from
|                attribute   - the attribute to retrieve
|                association - the attribute association
|
|        Output: none
|
|       Returns: the attribute on success, NULL otherwise
|
|    Written By: Jeremy Worley and Steve Kubica
|          Date: Oct 04, 1993 12:58
| Modifications:
|
------------------------------------------------------------*/
static kdms_attribute *
_kdms_retrieve_attribute(
   kobject obj,
   char   *attribute,
   char   *association)
{
   int attrib;
   int assoc;
   kdms_attribute_defin *defin;
   kdms_attribute_list **pres_list = NULL;
   kdms_attribute_list **phys_list = NULL;
   kdms_attribute *atr = NULL;

   /* -- try the definition first -- */
   defin = _kdms_locate_attribute_defin(attribute, association);

   /* -- if we have a quasi-attribute definition, use that -- */
   if (defin)
      if (defin->atr->base.type == KDMS_QUASI_ATTRIBUTE_TYPE)
	 return(defin->atr);
 
   /* -- at this point, we know attribute is a generic attribute -- */
   attrib = kstring_to_token(attribute);
   assoc  = kstring_to_token(association);
   
   /* -- try to get the attribute list from the presentation first -- */
   pres_list = _kdms_locate_attribute_list(obj, assoc, FALSE);

   /* 
    *   if the attribute was on the presentation list, then return it.
    *   otherwise, get the physical list so we can check there.  
    */
   if (pres_list == NULL ||
       (pres_list != NULL &&
	(atr = _kdms_locate_attribute(*pres_list, attrib)) == NULL))
      phys_list = _kdms_locate_attribute_list(obj, assoc, TRUE);
   else if (pres_list != NULL)
      return atr;


   /* 
    *   if the attribute was on the physical list, then return it.
    *   otherwise, need to get the definition and establish the attribute
    */
   if (phys_list == NULL ||
       (phys_list != NULL &&
	(atr = _kdms_locate_attribute(*phys_list, attrib)) == NULL))
   {
      if (defin == NULL)
	 return NULL; /* not an error...query uses this for existence */

      /* estabish the attribute at either the physical or presentation level */
      if (defin->shared)
	 atr = kdms_establish_attribute(phys_list, defin->atr);
      else
	 atr = kdms_establish_attribute(pres_list, defin->atr);

   }
   else
      return atr;

   return atr;
}

/*-----------------------------------------------------------
|
|  Routine Name: _kdms_copy_attribute
|
|       Purpose: This routine will copy an established attribute
|		 from one object to another. 
|  
|         Input: obj1        - the object containing the source list
|                obj2        - the object containing the destination list
|                association - the association of the list
|                attribute   - the attribute to copy
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley and Steve Kubica
|          Date: Apr 06, 1994 11:12
| Modifications:
|
------------------------------------------------------------*/
static int
_kdms_copy_attribute(
   kobject obj1,
   kobject obj2,
   char   *association,
   char   *attribute)
{
   kdms_attribute_list **pres_list = NULL;
   kdms_attribute_list **phys_list = NULL;
   kdms_attribute_list **obj2_list = NULL;

   kdms_attribute_defin *defin = NULL;
   kdms_attribute *atr1 = NULL, *atr2 = NULL;
   int status = TRUE;
   int shared = FALSE;

   /* get the attribute definition */
   defin = _kdms_locate_attribute_defin(attribute, association);

   if (defin != NULL && defin->atr->base.type == KDMS_QUASI_ATTRIBUTE_TYPE) 
   {
      status &= defin->atr->quasi.copy(obj1, obj2, 
				       kstring_to_token(association), 
				       defin->atr->quasi.attribute,
				       defin->atr->quasi.clientData, 
				       defin->atr->quasi.clientData);
   }
   else if (defin != NULL)  /* simple define-based copy */
   {
      /* -- retrieve the attribute for object1 -- */
      if ((atr1 = _kdms_retrieve_attribute(obj1, attribute, association))
	  == NULL) 
      {
	 _kdms_set_error(KDMS_EATTR_NODEF);
         return FALSE;
      }
      
      /* -- retrieve the attribute for object2 -- */
      if ((atr2 = _kdms_retrieve_attribute(obj2, attribute, association))
	  == NULL) 
      {
	 _kdms_set_error(KDMS_EATTR_NODEF);
         return FALSE;
      }
      
      _kdms_copy_generic(&atr1->generic, &atr2->generic);
   } 
   else  /* -- this is an attribute for which we have no definition -- */
   {
      int attrib = kstring_to_token(attribute);
      int assoc  = kstring_to_token(association);
      
      /* -- try to get the attribute list from the presentation first -- */
      pres_list = _kdms_locate_attribute_list(obj1, assoc, FALSE);
   
         /* 
          *   if the attribute was not on the presentation list, then
          *   check the physical list.
          */
      if ( (atr1 = _kdms_locate_attribute(*pres_list, attrib)) == NULL) {

         shared = TRUE;  /* since it isn't presentation, assume physical */
    
         phys_list = _kdms_locate_attribute_list(obj1, assoc, TRUE);
         if ( (atr1 = _kdms_locate_attribute(*phys_list, attrib)) == NULL) {
            _kdms_set_error(KDMS_EATTR_NODEF);
            return FALSE;
         }     
      } 

         /* 
	  *   at this point, we have the attribute from obj1 and we
          *   know if it was shared or not.  get the corrsponding list
          *   from obj2 and establish a copy of atr on it.
          */
      obj2_list = _kdms_locate_attribute_list(obj2,assoc,shared);
      
      if ((atr1 = kdms_establish_attribute(obj2_list, atr1)) == NULL) {
         _kdms_set_error(KDMS_EATTR_COPY_FAILED);
         kinfo(KDEBUG,"_kdms_copy_attribute : failed to establish "
                      "the attribute '%s' on the destination object.",
                      ktoken_to_string(attrib));
         return FALSE;
      } else
         return TRUE;
   }

   return TRUE;
}


/*-----------------------------------------------------------
|
|  Routine Name: _kdms_copy_attribute_list
|
|       Purpose: This routine will visit each attribute in a
|		 a given attribute list, and copy them to a 
|		 second attribute list.  For each attribute on
|		 the first list, the second list is checked.
|		 If the attribute does not exist on the 
|		 second list, a duplicate is constructed
|		 and placed on the second list.
|   
|		 The copy method of the attribute is then
|		 used to actually perform the copy.  Since
|                the quasi-copy requires the objects and the
|	         association, those parameters need to be
|	         passed in as well. 
|
|         Input: obj1  - the object containing the source list
|                obj2  - the object containing the destination list
|                assoc - the association of the list
|                list1 - the source list to copy
|
|        Output: list2 - the destination list
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley and Steve Kubica
|          Date: Apr 06, 1994 11:12
| Modifications:
|
------------------------------------------------------------*/
static int
_kdms_copy_attribute_list(
   kobject obj1,
   kobject obj2,
   int     assoc,
   kdms_attribute_list  *list1,
   kdms_attribute_list **list2)
{
   kdms_attribute *atr = NULL;
   int i;
   int status = TRUE;

   /* -- trivial copy -- */
   if (list1 == NULL || list1->nattributes == 0)
      return TRUE;

   /* 
    *   check for the existence of each attribute in list2,
    *   if it doesn't exist, then establish it (make an empty copy
    *   of the existing one).  once we are sure it is in the list2, 
    *   then copy it with the attribute copy method.
    */
   for (i = 0; i < list1->nattributes; i++)
   {

      atr = _kdms_locate_attribute(*list2, 
				   list1->attributes[i]->base.attribute);

      /* 
       *   if the attribute is not on the second list, establishing it will
       *   generate a copy.  if it is on the second list, then simply copy
       *   the attribute.
       */
      if (atr == NULL)
	 kdms_establish_attribute(list2, list1->attributes[i]);
      else
      {

	 /* copy it according to type */
	 if (atr->base.type == KDMS_GENERIC_ATTRIBUTE_TYPE)
	    status &= _kdms_copy_generic(&((list1->attributes[i])->generic),
					 &(atr->generic));


	 else if (atr->base.type == KDMS_QUASI_ATTRIBUTE_TYPE)
	 {
	    status &= atr->quasi.copy(obj1, obj2, assoc,
				      0,	 /* MAJOR KLUDGE! */
				      (list1->attributes[i])->quasi.clientData,
				      atr->quasi.clientData);
	 }
	 
	 else
	    status = FALSE;
      }
   }

   return status;
}


/*
 *    ================================================================== 
 *    Public Routines for Attributes 
 *    ==================================================================
 */

/************************************************************
*
*  Routine Name: kdms_define_quasi_attribute - define a quasi attribute
*
*       Purpose: This routine will define a data services quasi-attribute 
*		 for a session.  A quasi-attribute is one which does not
*		 technically have any data associated with it, but one for
*		 which a given action will be invoked.  From the higher
*		 level programmer's point of view, a quasi-attribute will 
*		 behave identically to a generic attribute. The higher-level
*		 function calls will be identical. That is, the same get, 
*		 set, match, query, and print attribute functions will work on
*		 both quasi-attributes and generic attributes.
*		 Quasi-attributes are provided simply to allow
*		 an application service programmer the ability to 
*		 provide their own custom functionality for an attribute.
*		 
* 		 This quasi-attribute definition will be used to determine
*		 what action should be taken on a get, set, match, copy
*		 or print attribute call.  On any kdms attribute call for
*	 	 this attribute, the relevant action handler provided 
*		 in this definition will be called or invoked. 
*
*	         This attribute definition is distinguished by the
*		 attribute name provided here.  The name must be a string,
*		 unique for the given association.
*
*		 In each attribute call, a segment argument will
*		 be provided to give scope to the attribute.
*		 The association given in this definition is used
*		 to determine the allowable scope of the attribute.
*		 Attributes can be either scoped to the main object 
*		 (object-level) or scoped to any of the 
*		 object's data segments (segment-level).  
*
*		 The provided association is simply a string.
*                A NULL association indicates that the attribute 
*		 can only be invoked when NULL is passed in as the
*		 segment argument.  An association of KDMS_ALL_SEGMENTS 
*		 indicates that the attribute can be invoked at the 
*		 segment level for any segment name. If a specific 
*		 segment name, such as "value" is given for the 
*		 association, then the attribute can only invoked 
*		 when "value" is provided as the segment name.
*
*		 This quasi-attribute can be undefined with the 
*		 kdms_undefine_attribute call.
*
*         Input: association    - string indicating where it is legal to
*				  to invoke the atttribute.  NULL implies
*				  the attribute can be invoked at
*				  the object level.  A segment name
*			          implies that the attribute can only
*			          be invoked on that segment.  The
*			          identifier KDMS_ALL_SEGMENTS implies
*				  that the attribute can be invoked
*				  for any segment, but not for the object.
*
*		 attribute      - attribute string identifier.
*
*		 clientData     - pointer to client data.  This client
*				  data will be passed in to all the 
*			          the routines for this attribute.
*
*   		 get            - get handler function for this attribute.
*			          This function will be invoked whenever 
*				  a kdms_get_attribute, kdms_get_attributes, 
*				  kdms_vget_attribute, or kdms_vget_attributes
*				  function is called and the given segment 
*				  matches the definition's association.
*
*       !The get handler declaration is of the form :
*       !
*       !   int get_handler(
*       !       kobject   object,
*       !       int       association,
*       !       int       attribute,
*       !       kaddr     clientData,
*       !       kva_list *list)
*
*                                 The object is passed from the calling get
*                                 function. The association is the tokenized
*                                 representation of the segment argument
*                                 from the get function.  The attribute is
*                                 the tokenized representation of the
*                                 attribute name.  The clientData is the
*                                 clientData from this definition.  The
*                                 variable argument list is an already
*                                 opened varargs list which contains the
*                                 pointers with which to return the 
*                                 attribute data.  The variable arguments
*                                 can be pulled off with the kva_arg()
*                                 function.  The list will be closed after
*                                 this handler returns.  The return value
*				  of this handler will also be propogated
*			          up to be the return value of the calling
*				  get function.
*
*		 set            - set handler function for this attribute.
*			          This function will be invoked whenever 
*				  a kdms_set_attribute, kdms_set_attributes, 
*				  kdms_vset_attribute, or kdms_vset_attributes
*				  function is called and the given segment 
*				  matches the definition's association.
*
*	!The set handler declaration is of the form :
*       !
*	!   int set_handler(
*	!       kobject   object,
*       !       int       association,
*       !       int       attribute,
*       !       kaddr     clientData,
*       !       kva_list *list)
*
*                                 The object is passed from the calling set
*                                 function. The association is the tokenized
*                                 representation of the segment argument
*                                 from the set function.  The attribute is
*                                 the tokenized representation of the
*                                 attribute name.  The clientData is the
*                                 clientData from this definition.  The
*                                 variable argument list is an already
*                                 opened varargs list which contains the
*                                 data with which to set the attribute
*                                 data.  The variable arguments
*                                 can be pulled off with the kva_arg()
*                                 function.  The list will be closed after
*                                 this handler returns.  The return value
*                                 of this handler will also be propogated
*                                 up to be the return value of the calling
*                                 set function.
*
*		 match          - match handler function for this attribute.
*			          This function will be invoked whenever 
*				  a kdms_match_attribute, 
*				  kdms_match_attributes, or 
*			          kdms_vmatch_attributes
*				  function is called and the given segment 
*				  matches the definition's association.
*
*	!The match handler declaration is of the form :
*       !
*	!   int match_handler(
*	!       kobject   object1,
*	!       kobject   object2,
*       !       int       association,
*       !       int       attribute,
*       !       kaddr     clientData1,
*       !       kaddr     clientData2)
*
*                                 The object1 and object2 arguments are 
*                                 passed from the calling match function. 
*                                 The association is the tokenized
*                                 representation of the segment argument
*                                 from the match function.  The attribute is
*                                 the tokenized representation of the
*                                 attribute name.  The clientData1 and
*                                 clientData2 arguments are from 
*                                 from this definition. The return value
*                                 of this handler will also be propogated
*                                 up to be the return value of the calling
*                                 match function.
*
*		 copy           - copy handler function for this attribute.
*			          This function will be invoked whenever 
*				  a kdms_copy_attribute, 
*				  kdms_copy_attributes, or 
*			          kdms_vcopy_attributes
*				  function is called and the given segment 
*				  matches the definition's association.
*
*	!The copy handler declaration is of the form :
*       !
*	!   int copy_handler(
*	!       kobject   object1,
*	!       kobject   object2,
*       !       int       association,
*       !       int       attribute,
*       !       kaddr     clientData1,
*       !       kaddr     clientData2)
*
*                                 The object1 and object2 arguments are 
*                                 passed from the calling copy function. 
*                                 The association is the tokenized
*                                 representation of the segment argument
*                                 from the copy function.  The attribute is
*                                 the tokenized representation of the
*                                 attribute name.  The clientData1 and
*                                 clientData2 arguments are from 
*                                 from this definition. The return value
*                                 of this handler will also be propogated
*                                 up to be the return value of the calling
*                                 copy function.
*
*		 query          - query handler function for this attribute.
*			          This function will be invoked whenever 
*				  a kdms_query_attribute function is 
*				  called and the given segment 
*				  matches the definition's association.
*
*	!The query handler declaration is of the form :
*       !
*	!   int query_handler(
*	!       kobject   object,
*       !       int       association,
*       !       int       attribute,
*       !       kaddr     clientData,
*       !       int      *num_args,
*       !       int      *arg_size,
*       !       int      *data_type,
*       !       int      *permanent)
*
*                                 The object is passed in from the 
*                                 calling query function.  The association 
*                                 is the tokenized representation of the 
*                                 segment argument from the query function.  
*                                 The attribute is the tokenized 
*                                 representation of the attribute name.  
*                                 The clientData argument is from 
*                                 from this definition.  The num_args
*                                 argument can be used to return the
*                                 expected number of arguments for this
*                                 attribute.  The arg_size argument can
*                                 be used to return the expected argument
*                                 size for this attribute.  The data_type
*                                 argument can be used to return the 
*                                 expected data type for this attribute.
*			          The permanent argument can be used to
*				  indicate if this attribute is stored
*				  or not.  Note that it is up to the 
*				  programmer to ensure that the values which 
*				  the handler returns for any of those
*				  arguments matches what is going to be
*				  processed in the get and set handlers.
*				  The return value of this handler will 
*				  be propogated up to be the return value 
*				  of the calling query function.  In general,
*			          a value of TRUE is interpreted to mean 
*				  that the attribute 'exists'.
*
*		 print          - print handler function for this attribute.
*			          This function will be invoked whenever 
*				  a kdms_print_attribute function is 
*				  called and the given segment 
*				  matches the definition's association.
*
*	!The print handler declaration is of the form :
*       !
*	!   int print_handler(
*	!       kobject   object,
*       !       int       association,
*       !       int       attribute,
*       !       kaddr     clientData,
*       !       kfile    *outfile)
*
*                                 The object is passed in from the 
*                                 calling print function.  The association 
*                                 is the tokenized representation of the 
*                                 segment argument from the print function.  
*                                 The attribute is the tokenized 
*                                 representation of the attribute name.  
*                                 The clientData argument is from 
*                                 from this definition.  The outfile
*                                 argument is the (hopefully) open 
*                                 khoros transort which this handler
*                                 can use to print the attribute to.
*                                 The return value of this handler will 
*                                 be propogated up to be the return value 
*                                 of the calling print function.
*        Output: none
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Jeremy Worley and Steve Kubica
*          Date: Mar 25, 1994 13:13
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kdms_define_quasi_attribute(
*                !      char *association,
*   		 !      char *attribute,
*   		 !      kaddr clientData,
*   		 !      kfunc_int *get,
*   		 !      kfunc_int *set,
*   		 !      kfunc_int *match,
*   		 !      kfunc_int *copy,
*   		 !      kfunc_int *query,
*		 !      kfunc_int *print)
*
*************************************************************/
int
kdms_define_quasi_attribute(
   char *association,
   char *attribute,
   kaddr clientData,
   int (*get)   PROTO((kobject, int, int, kaddr, kva_list *)),
   int (*set)   PROTO((kobject, int, int, kaddr, kva_list *)),
   int (*match) PROTO((kobject, kobject, int, int, kaddr, kaddr)),
   int (*copy)  PROTO((kobject, kobject, int, int, kaddr, kaddr)),
   int (*query) PROTO((kobject, int, int, kaddr, int *, int *, int *, int *)),
   int (*print) PROTO((kobject, int, int , kaddr, kfile *)))
{
   kdms_attribute_defin *defin = NULL;
   kdms_attribute_defin *old_defin = NULL;

   if (!_kdms_compiled)
      _kdms_init();

   old_defin = _kdms_locate_attribute_defin(attribute, association);
   
   if (old_defin != NULL)
   {
      _kdms_set_error(KDMS_EATTR_DEF_EXISTS);
      kinfo(KDEBUG, "kdms_define_quasi_attribute: attempt to redefine the " 
	            "attribute '%s'.", attribute);
      return FALSE;
   }

   defin = _kdms_construct_quasi_defin(attribute, association, get,
				       set, match, copy, query, print,
				       clientData);

   return _kdms_hash_attribute_defin(defin);
}

/************************************************************
*
*  Routine Name: kdms_define_attribute - define an attribute for
*					 for a session          
*
*	Purpose: This routine will define a data services attribute 
*		 for a session.  This attribute definition will be used
*		 as a template for creating instances of the attribute
*                on an actual data object.  This creation will be
*                implicit when the attribute is accessed by any other 
*		 kdms attribute routine.
*
*	         The attribute definition is distinguished by the
*		 given attribute name.  The name must be a string,
*		 unique for the given association.
*		 The attribute acts as storage for multiple argument 
*		 values.  Each attribute argument can be of any size 
*		 or data type.  The number of arguments, the argument
*	         size, and the data type are all specified in this
*		 definition.   
*
*		 Attributes can be permanent, implying that they are 
*		 stored to the disk when the object is closed.  
*		 Technically, a permanent attribute will only be
*		 stored if the underlying file format has the
*		 capacity for storing it.   See the man page for
*		 kdatafmt for more information of file formats.
*		 
*		 Attributes can be either created on the main object 
*		 (object-level) or created on any of the 
*		 object's data segments (segment-level).  
*		 Whether an attribute is created at the object-level
*		 or at the segment-level is determined by the 
*		 attribute definition's association.  
*
*		 The provided association is simply a string.
*                A NULL association indicates that the attribute 
*		 can only be created at the object-level.  An 
*		 association of KDMS_ALL_SEGMENTS indicates that the 
*		 attribute can be created at the segment level on 
*		 any segment name. If a specific segment name, such 
*		 as "value" is given for the association, then the 
*		 attribute can only created on the "value" segment.  
*
*		 The attribute can be either permanent or transient.
*		 Permanent impiles that when the object is closed,
*		 a representation of the attribute will be written
*		 out to the disk.  Note that this representation completely
*		 describes the attribute, and it will be possible to
*		 read this attribute later in another session even if it
*		 has not been defined there.
*
*		 In addition to being created either at the
*		 object or segment level, an attribute can also
*		 be created at the physical or presentation level.
*
*		 Changes to physical level attributes are visible
*		 to all references of the data object and are termed
*		 as "shared" attributes.  Changes to presentation
*		 level attributes are visible only to the reference
*		 object on which the change was made, and thus 
*		 the attribute is termed as "unshared".  Since all
*		 references may have potentially different values,
*		 unshared attributes may not be permanent.
*
*		 A default must be provided.  This default must consist
*		 of the proper number of arguments where each argument is of
*	         the same size and data type specified in the definition.
*		 This default will be used when initializing any attributes 
*		 created from this definition. 
*
*		 This attribute can be undefined with the 
*		 kdms_undefine_attribute call.
*
*         Input: association    - string indicating where it is legal to
*				  to create the atttribute.  NULL implies
*				  the attribute can be instantiated at
*				  the object level.  A segment name
*			          implies that the attribute can only
*			          be established on that segment.  The
*			          identifier KDMS_ALL_SEGMENTS implies
*				  that the attribute can be established
*				  on any segment, but not on the object.
*
*		 attribute      - attribute string identifier
*
*		 num_args       - number of arguments in the attribute
*
*  		 arg_size       - size of each attribute argument
*
*		 data_type      - data type of the attribute can be
*		 	           KBIT, KUBYTE, KBYTE,
*		 	           KUSHORT, KSHORT, KUINT, 
*			           KINT, KULONG, KLONG, KFLOAT
*			           KDOUBLE, KCOMPLEX, 
*			           KDCOMPLEX, or KSTRING  
*
* 		 permanent      - TRUE if attribute is permanent
*
*	         shared         - TRUE if attribute is shared
*
*                kvalist        - the default value in a variable 
*				  argument list containing data in the form :
*
*		 	      value1 [, value2, ...]
*                	
*                	      The list should consist of "num_arg" arguments,
*                	      where each argument is of "arg_size" size of
*                	      "data_type" data type.
*
*        Output: None
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Jeremy Worley and Steve Kubica
*          Date: Sep 23, 1993 13:36
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kdms_define_attribute(
*                !      char *association,
*                !      char *attribute,
*                !      int   num_args,
*                !      int   arg_size,
*                !      int   data_type,
*                !      int   permanent,
*                !      int   shared,
*                !      kvalist)
*
*************************************************************/
int
kdms_define_attribute(
   char *association,
   char *attribute,
   int   num_args,
   int   arg_size,
   int   data_type,
   int   permanent,
   int   shared,
   kvalist)
{
   kdms_attribute_defin *defin = NULL;
   kdms_attribute_defin *old_defin = NULL;
   kva_list list;
   kaddr def = NULL;
   int status = TRUE;

   if (!_kdms_compiled)
      _kdms_init();

   /* 
    *   do not allow stored attributes to exist only at the
    *   presentation layer or they will not be written out
    */
   if (permanent && !shared)
   {
      kerror("kdataman", "kdms_define_attribute",
	     "Illegal Definition for attribute '%s' : Unshared "
	     "attributes can not be permanent.", attribute);
      return FALSE;
   }

   /* -- has this been defined previously? -- */
   old_defin = _kdms_locate_attribute_defin(attribute, association);

   if (old_defin != NULL)
   {
      /* -- if previous defin was quasi, it's okay to redefine as generic -- */
      if (old_defin->atr->base.type != KDMS_QUASI_ATTRIBUTE_TYPE)
      {
	 _kdms_set_error(KDMS_EATTR_DEF_EXISTS);
	 kinfo(KDEBUG, "kdms_define_attribute: attempt to redefine the "
	               "attribute '%s'.", attribute);
	 return FALSE;
      }
   }

   /* -- start the variable arguments list after the permanent argument -- */
   kva_start(list, shared);

   /* -- pull the default off of the varargs list -- */
   status &= _kdms_vargs_to_attribute_data(&def, data_type, arg_size,
					   num_args, &list);

   /* -- construct the definition -- */
   defin = _kdms_construct_generic_defin(attribute, association, 
					 data_type, arg_size,
					 num_args, permanent, shared, def);

   /* -- free up our copy of the default -- */
   kfree(def);

   /* -- add the definition to the definition list -- */
   status &= _kdms_hash_attribute_defin(defin);


   /* -- terminate the variable arguments list -- */
   kva_end(list);

   return status;
}

/************************************************************
*
*  Routine Name: kdms_undefine_attribute - undefine a defined attribute
*
*       Purpose: This function will remove an attribute definition 
*		 from the definition list.  The attribute definition
*		 corresponding to the given association and attribute
*		 name will be removed.  This has no effect on any
*		 attribute which may have already been instantiated
*		 on an object.  This function can be used to undefine
*		 both generic and quasi attributes.
*
*         Input: association    - string indicating the scope of the
*				  attribute.  The same association
*			          that was used to define the attribute
*			          must be used here.
*
*		 attribute      - attribute string identifier
*
*        Output: none
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Steve Kubica
*          Date: Mar 25, 1994 13:37
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kdms_undefine_attribute(
*                !      char *association,
*                !      char *attribute)
*
*************************************************************/
int
kdms_undefine_attribute(
   char *association,
   char *attribute)
{
   if (!_kdms_compiled)
      _kdms_init();

   return _kdms_delete_attribute_defin(attribute, association);
}

/************************************************************
*
*  Routine Name: kdms_query_attribute_definition - determines if an attribute
*						   is defined.
*
*       Purpose: This function will check to see if an attribute is
*		 defined on the definition list.  The attribute definition
*		 corresponding to the given association and attribute
*		 name will be searched for. 
*
*         Input: association    - string indicating the scope of the
*				  attribute.  The same association
*			          that was used to define the attribute
*			          must be used here.
*
*		 attribute      - attribute string identifier
*
*        Output: none
*
*       Returns: TRUE (1) if attribute is defined, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Steve Kubica
*          Date: Mar 25, 1994 13:37
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kdms_query_attribute_definition(
*                !      char *association,
*                !      char *attribute)
*
*************************************************************/
int
kdms_query_attribute_definition(
   char *association,
   char *attribute)
{
   if (!_kdms_compiled)
      _kdms_init();

   if (_kdms_locate_attribute_defin(attribute, association) == NULL)
      return FALSE;
   else
      return TRUE;
}

/************************************************************
*
*  Routine Name: kdms_create_attribute - instantiate an attribute
*
*       Purpose: This routine provides the programmer with a 
*		 mechanism for creating attributes specific
*		 to a program being written.
*
*		 The attribute acts as storage for multiple argument 
*		 values.  Each attribute argument can be of any size 
*		 or data type.  The number of arguments, the argument
*	         size, and the data type are all specified here.
*
*		 The attributes can be either created on the 
*		 main object (object-level) or created on any of the 
*		 object's data segments (segment-level).  
*		 Whether the attribute is created at the object-level
*		 or at the segment-level is determined by the 
*		 segment argument.  If the segment argument is
*		 NULL, then the attribute will be instantiated
*		 at the object-level.  Otherwise, the attribute
*		 will be instantiated on the specified segment, if
*		 it exists.  If an attribute with the same name
*		 and scope already exists, then this function will fail.
*
*		 The attribute can be either permanent or transient.
*		 Permanent impiles that when the object is closed,
*		 a representation of the attribute will be written
*		 out to the disk.  
*
*		 In addition to being created either at the
*		 object or segment level, the attribute can also
*		 be created at the physical or presentation level.
*
*		 Changes to physical level attributes are visible
*		 to all references of the data object and are termed
*		 as "shared" attributes.  Changes to presentation
*		 level attributes are visible only to the reference
*		 object on which the change was made, and thus 
*		 the attribute is termed as "unshared". Since all
*		 references may have potentially different values,
*		 unshared attributes may not be permanent.
*
*		 An initial value must be provided.  This initial
*		 value  must consist of the proper number of arguments 
*		 where each argument is of the same size and data type 
*		 specified in this call.
*
*		 The attribute can be destroyed with the
*		 kdms_destroy_attribute function.
*
*        Input:  object         - the object on which to instantiate
*				  the new attribute.
*
*		 segment        - segment identifier string specifying
*				  which segment to create the attribute in.  
*				  If NULL, then the attribute will be
*				  created at the object level.
*
*		 attribute      - attribute identifier string.  This
*				  identifier must be unique for the
*			 	  given segment.
*
*		 num_args       - number of arguments in the attribute
*
*  		 arg_size       - size of each attribute argument
*
*		 data_type      - data type of the attribute can be
*		 	           KBIT, KUBYTE, KBYTE,
*		 	           KUSHORT, KSHORT, KUINT, 
*			           KINT, KULONG, KLONG, KFLOAT
*			           KDOUBLE, KCOMPLEX, 
*			           KDCOMPLEX, or KSTRING  
*
* 		 permanent      - TRUE if attribute is permanent.
*
*	         shared         - TRUE if attribute is shared
*
*        Output: none
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: This attribute will override any defined attributes with
*		 the same name and scope.  
*    Written By: Jeremy Worley and Steve Kubica
*          Date: Dec 14, 1992 15:46
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: int kdms_create_attribute(
*                !      kobject *object,
*		 !      char    *segment,
*                !      char    *attribute,
*                !      int      num_args,
*                !      int      arg_size,
*                !      int      data_type,
*                !      int      permanent,
*                !      int      shared)
*
*************************************************************/
int
kdms_create_attribute(
   kobject object,
   char *segment,
   char *attribute,
   int   num_args,
   int   arg_size,
   int   data_type,
   int   permanent,
   int   shared)
{
   int assoc = kstring_to_token(segment);
   int attrib = kstring_to_token(attribute);

   kdms_attribute_list **atr_list = NULL;
   kdms_attribute *atr = NULL;

   if (!_kdms_compiled)
      _kdms_init();

   /* 
    *   do not allow stored attributes to exist only at the
    *   presentation layer or they will not be written out
    */
   if (permanent && !shared)
   {
      kerror("kdataman", "kdms_define_attribute",
	     "Can not create attribute '%s' : Unshared "
	     "attributes can not be permanent.", attribute);
      return FALSE;
   }

   /* -- sanity check -- */
   if (object == NULL || object->type != KOBJ_DATA)
   {
      _kdms_set_error(KDMS_EOBJ_INVALID);
      return FALSE;
   }

   /* -- get the attribute list for the given association -- */
   atr_list = _kdms_locate_attribute_list(object, assoc, shared);

   if (atr_list == NULL)
      return FALSE;		/* errno set already */

   /* -- make sure this attribute does not already exist -- */
   if ((atr = _kdms_locate_attribute(*atr_list, attrib)) != NULL)
   {
      _kdms_set_error(KDMS_EATTR_ALREADY_EXISTS);
      kinfo(KDEBUG, "kdms_create_attribute: attempt to create attribute '%s' "
	    "which already exists for the association '%s'.",
	    attribute, segment == NULL ? "null" : segment);
      return FALSE;
   }

   /* -- construct a generic attribute according to what they wanted -- */
   atr = _kdms_construct_generic(attrib, data_type, arg_size, num_args,
				 permanent, NULL);

   /* -- add the attribute to the attribute list -- */
   *atr_list = _kdms_append_attribute(*atr_list, atr);

   /* -- if the list is not NULL, then it worked! -- */
   return (*atr_list != NULL);
}

/************************************************************
*
*  Routine Name: kdms_destroy_attribute - destroy an attribute
*
*       Purpose: This routine provides the programmer with a
*		 mechanism for destroying an instantiation
*	         of an attribute.
*
*		 The segment argument is used to indicate which
*		 segment the attribute exists in. If the segment 
*		 argument is NULL, then	the attribute is assumed
*		 to exist at the object level.
*
*		 If the attribute is not physically instantiated
*		 on the object, then it can not be destroyed.
*		 In general, this routine should only be used
*		 to destroy attributes that have been explicitly
*		 created with a kdms_create_attribute call.
*
*        Input:  object    - the object with the attribute to destroy
*
*		 segment   - the segment which contains the
*			     attribute.  If this is NULL, then
*			     the attribute is assumed to exist
*			     at the object level.
*
*		 attribute - string identifying the name of the 
*			     attribute to destroy.
*
*        Output: none
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Jeremy Worley and Steve Kubica
*          Date: Feb 15, 1993 08:38
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: int kdms_destroy_attribute(
*                !      kobject *object,
*		 !      char    *segment,
*                !      char    *attribute)
*
*************************************************************/
int
kdms_destroy_attribute(
   kobject object,
   char   *segment,
   char   *attribute)
{
   int assoc = kstring_to_token(segment);
   int attrib = kstring_to_token(attribute);

   kdms_attribute_list **atr_list = NULL;
   kdms_attribute *atr = NULL;

   if (!_kdms_compiled)
      _kdms_init();

   /* -- sanity check -- */
   if (object == NULL || object->type != KOBJ_DATA)
   {
      _kdms_set_error(KDMS_EOBJ_INVALID);
      return FALSE;
   }

   /* -- try to get the attribute list from the presentation first -- */
   atr_list = _kdms_locate_attribute_list(object, assoc, FALSE);

   if (atr_list == NULL)
      return FALSE;		/* errno set already */

   /* 
    *   if the attribute was on the presentation list, then delete it.
    *   otherwise, get the physical list so we can check there.  
    */
   atr = _kdms_locate_attribute(*atr_list, attrib);
   if (atr == NULL)
      atr_list = _kdms_locate_attribute_list(object, assoc, TRUE);
   else
      return _kdms_delete_attribute(*atr_list, attrib);

   /* 
    *   if the attribute was on the physical list, then delete it.
    *   otherwise, attribute does not exist anyway.
    */
   if ((atr = _kdms_locate_attribute(*atr_list, attrib)) != NULL)
      return _kdms_delete_attribute(*atr_list, attrib);
   else
   {
      kinfo(KDEBUG, "kdms_destroy_attribute: deleting the non-existent "
	            "attribute '%s'.", attribute);
      return FALSE;
   }
}


/************************************************************
*
*  Routine Name: kdms_vset_attribute - open varargs set attribute
*
*       Purpose: This routine allows the programmer to
*		 set the value of an attribute associated
*		 with a data object.
*
*		 Data services manages two sets of attributes
*		 with each object.  Internally, they are referred
*		 to as the physical attributes and the presentation
*		 attributes.  The presentation version of an attribute
*		 is the value that is desired by the programmer.  The
*		 corresponding physical attribute is what is physically
*		 stored on the file or transport associated with the
*		 object.  If the two versions of an attribute result
*		 in different presentations of data, then data services
*		 automatically translates the data from the physical
*		 interpretation to the presentation interpretation, 
*		 or visa-versa, as appropriate.  For example, if the
*		 data type attribute (KDMS_DATA_TYPE) is physically
*		 unsigned int (KULONG), but the presentation value
*		 of that attribute is float (KFLOAT), then data services
*		 will retrieve the data from the object and and cast it to
*		 float before returning it to the user.  Similar behavior
*		 will occur for attributes such as index order 
*		 (KDMS_INDEX_ORDER) and size (KDMS_SIZE).
*
*		 The presentation attributes are settable
*		 at any time throughout a object's lifetime.  The
*		 physical attributes, however, at some point are
*		 locked and thereafter cannot change.  An object
*		 that was opened as an input object (via kdms_input
*		 or kdms_open) has its physical attributes locked
*		 immediately.
*
*                The effect of setting the attribute is
*                immediate.  For example, if this routine
*                is used to set the KDMS_DATATYPE attribute,
*                then any future access of the data in this
*                image will involve data of the specified
*                data type.
*
*		 If the segment argument is NULL, then
*		 this instructs kdms_set_attribute to set
*		 a global attribute, if available.  In either
*		 case, if the attribute does not exist, then
*		 this routine returns FALSE, indicating a
*		 failure.
*
*         Input: object  - object to set the attribute on
*		 segment - name of segment to set the attribute on.
*		 attribute - name of the attribute to set
*		 va_list - a valist containing the values of the
*			   attribute.
*
*         Input: 
*
*        Output: 
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Jeremey Worley and Steve Kubica
*          Date: Mar 25, 1994 13:37
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/
int
kdms_vset_attribute(
   kobject   object,
   char     *segment,
   char     *attribute,
   kva_list *list)
{
   kdms_segment_defin *segd = NULL;
   kdms_attribute *atr = NULL;
   int status = TRUE;

   if (!_kdms_compiled)
      _kdms_init();

   /* -- sanity check -- */
   if (object == NULL || object->type != KOBJ_DATA)
   {
      _kdms_set_error(KDMS_EOBJ_INVALID);
      return FALSE;
   }

   /* -- if there is a defined segment set_atr routine, then use that -- */
   if ((segd = kdms_locate_segment_defin(segment)) != NULL)
      if (segd->set_atr)
      {
	 segd->lock = TRUE;
	 status = segd->set_atr(object, segment, attribute, list);
	 segd->lock = FALSE;
	 return status;
      }

   /* -- otherwise, do a normal set attribute -- */   

   /* -- retrieve the attribute definition -- */
   if ((atr = _kdms_retrieve_attribute(object, attribute, segment)) == NULL)
   {
      _kdms_set_error(KDMS_EATTR_NODEF);
      return FALSE;
   }

   /* -- now we can set it - set it according to type -- */
   if (atr->base.type == KDMS_GENERIC_ATTRIBUTE_TYPE)
      status = _kdms_set_generic(&(atr->generic), list);

   else if (atr->base.type == KDMS_QUASI_ATTRIBUTE_TYPE)
      status = atr->quasi.set(object, kstring_to_token(segment),
			      atr->quasi.attribute,
			      atr->quasi.clientData, list);
   else
      status = FALSE;

   if (!status)
   {
      _kdms_set_error(KDMS_EATTR_SET_FAILED);
      kinfo(KDEBUG, "kdms_vset_attribute : "
	    "failed to set attribute '%s'.", attribute);
      return FALSE;
   }

   return status;
}


/************************************************************
*
*  Routine Name: kdms_vset_attributes - set attributes on a kvalist
*
*       Purpose: This routine allows the programmer to
*		 set the values of multiple attributes associated
*		 with a data object.
*
*		 Data services manages two sets of attributes
*		 with each object.  Internally, they are referred
*		 to as the physical attributes and the presentation
*		 attributes.  The presentation version of an attribute
*		 is the value that is desired by the programmer.  The
*		 corresponding physical attribute is what is physically
*		 stored on the file or transport associated with the
*		 object.  If the two versions of an attribute result
*		 in different presentations of data, then data services
*		 automatically translates the data from the physical
*		 interpretation to the presentation interpretation, 
*		 or visa-versa, as appropriate.  For example, if the
*		 data type attribute (KDMS_DATA_TYPE) is physically
*		 unsigned int (KULONG), but the presentation value
*		 of that attribute is float (KFLOAT), then data services
*		 will retrieve the data from the object and and cast it to
*		 float before returning it to the user.  Similar behavior
*		 will occur for attributes such as index order 
*		 (KDMS_INDEX_ORDER) and size (KDMS_SIZE).
*
*		 The presentation attributes are settable
*		 at any time throughout a object's lifetime.  The
*		 physical attributes, however, at some point are
*		 locked and thereafter cannot change.  An object
*		 that was opened as an input object (via kdms_input
*		 or kdms_open) has its physical attributes locked
*		 immediately.
*
*                The effect of setting the attribute is
*                immediate.  For example, if this routine
*                is used to set the KDMS_DATATYPE attribute,
*                then any future access of the data in this
*                image will involve data of the specified
*                data type.
*
*		 If the segment argument is NULL, then
*		 this instructs kdms_set_attribute to set
*		 a global attribute, if available.  In either
*		 case, if the attribute does not exist, then
*		 this routine returns FALSE, indicating a
*		 failure.
*
*         Input: object  - object to set the attribute on
*		 segment - name of segment to set the attribute on.
*		 va_list - a valist containing attributes and their
*			   values.
*        Output: 
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Jeremey Worley
*          Date: Mar 25, 1994 13:37
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/
int
kdms_vset_attributes(
   kobject   object,
   char     *segment,
   kva_list *list)
{
   char *attribute;

   while ((attribute = kva_arg(*list, char *)) != NULL)
      if (!kdms_vset_attribute(object, segment, attribute, list))
	    return FALSE;	/* errno set already */

   return TRUE;
}

/************************************************************
*
*  Routine Name: kdms_set_attribute - set the value of an attribute
*
*       Purpose: This routine allows the programmer to
*		 set the value of an attribute associated
*		 with a data object.
*
*		 Data services manages two sets of attributes
*		 with each object.  Internally, they are referred
*		 to as the physical attributes and the presentation
*		 attributes.  The presentation version of an attribute
*		 is the value that is desired by the programmer.  The
*		 corresponding physical attribute is what is physically
*		 stored on the file or transport associated with the
*		 object.  If the two versions of an attribute result
*		 in different presentations of data, then data services
*		 automatically translates the data from the physical
*		 interpretation to the presentation interpretation, 
*		 or visa-versa, as appropriate.  For example, if the
*		 data type attribute (KDMS_DATA_TYPE) is physically
*		 unsigned int (KULONG), but the presentation value
*		 of that attribute is float (KFLOAT), then data services
*		 will retrieve the data from the object and and cast it to
*		 float before returning it to the user.  Similar behavior
*		 will occur for attributes such as index order 
*		 (KDMS_INDEX_ORDER) and size (KDMS_SIZE).
*
*		 The presentation attributes are settable
*		 at any time throughout a object's lifetime.  The
*		 physical attributes, however, at some point are
*		 locked and thereafter cannot change.  An object
*		 that was opened as an input object (via kdms_input
*		 or kdms_open) has its physical attributes locked
*		 immediately.
*
*                The effect of setting the attribute is
*                immediate.  For example, if this routine
*                is used to set the KDMS_DATATYPE attribute,
*                then any future access of the data in this
*                image will involve data of the specified
*                data type.
*
*		 If the segment argument is NULL, then
*		 this instructs kdms_set_attribute to set
*		 a global attribute, if available.  In either
*		 case, if the attribute does not exist, then
*		 this routine returns FALSE, indicating a
*		 failure.
*
*
*        INPUT:  object    - the object that is involved in
*			     the set attribute operation.
*		 segment   - the data segment whose attribute
*			     is being set.
*		  va_alist - variable argument list, that
*			     contains an attribute followed
*			     by any values associated with
*			     that attribute.  It takes the
*			     form:
*
*			     ATTRIBUTE_NAME, value1 [, value2, ...]
*
*			     The number of value arguments in
*			     the variable argument list depends
*			     on the specific attribute.  For
*			     example, KDMS_DATATYPE takes only
*			     one value, but KDMS_SIZE takes
*			     five values.
*
*        Output: none
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions:
*    Written By: Mark Young and Jeremy Worley
*          Date: Aug 11, 1992 00:45
*      Verified:
*  Side Effects: 
* Modifications:
*
*************************************************************/
int
kdms_set_attribute(
   kobject object,
   char   *segment,
   char   *attribute,
   kvalist)
{
   kva_list list;
   int status;

   if (!_kdms_compiled)
      _kdms_init();

   /* -- start the argument list to begin after the attribute argument -- */
   kva_start(list, attribute);

   /* -- call the variable argument version of kdms_set_attribute -- */
   status = kdms_vset_attribute(object, segment, attribute, &list);

   /* -- terminate the argument list -- */
   kva_end(list);

   return status;
}


/************************************************************
*
*  Routine Name: kdms_set_attributes - sets the values of multiple attributes
*
*       Purpose: The purpose of this routine is to set the values of 
*		 multiple attributes of a data object.
*
*		 Data services manages two sets of attributes
*		 with each object.  Internally, they are referred
*		 to as the physical attributes and the presentation
*		 attributes.  The presentation version of an attribute
*		 is the value that is desired by the programmer.  The
*		 corresponding physical attribute is what is physically
*		 stored on the file or transport associated with the
*		 object.  If the two versions of an attribute result
*		 in different presentations of data, then data services
*		 automatically translates the data from the physical
*		 interpretation to the presentation interpretation, 
*		 or visa-versa, as appropriate.  For example, if the
*		 data type attribute (KDMS_DATA_TYPE) is physically
*		 unsigned int (KULONG), but the presentation value
*		 of that attribute is float (KFLOAT), then data services
*		 will retrieve the data from the object and and cast it to
*		 float before returning it to the user.  Similar behavior
*		 will occur for attributes such as index order 
*		 (KDMS_INDEX_ORDER) and size (KDMS_SIZE).
*
*		 The presentation attributes are settable
*		 at any time throughout a object's lifetime.  The
*		 physical attributes, however, at some point are
*		 locked and thereafter cannot change.  An object
*		 that was opened as an input object (via kdms_input
*		 or kdms_open) has its physical attributes locked
*		 immediately.
*
*                The effect of setting an attribute is
*                immediate.  For example, if this routine
*                is used to set the KDMS_DATA_TYPE attribute,
*                then any future access of the data in this
*                image will involve data of the specified
*                data type.
*
*		 If the segment argument is NULL, then
*		 this instructs kdms_set_attributes to get
*		 a series of global attributes, if available.
*		 In either case, if an attribute does not exist,
*		 then this routine returns FALSE, indicating a
*		 failure.
*
*         Input: object    - the object that is involved in
*			     the set attribute operation.
*		 segment   - the data segment whose attribute
*			     is being set.
*		 va_alist  - variable argument list, that
*			     contains a set of attributes, each
*			     followed by any values associated
*			     with that attribute.  It takes the
*			     form:
*
* 			     ATTRIBUTE_NAME1, value1 [, value2, ...],
*			     ATTRIBUTE_NAME2, value1,[, value2, ...],
*			     ..., NULL.
*
*			     The number of value arguments in
*			     variable argument list for each
*			     attribute depends on the specific
*			     attribute.  For
*			     example, KDMS_DATATYPE takes only
*			     one value, but KDMS_SIZE takes
*			     five values.  The NULL at the
*			     end of the variable argument
*			     list serves as a flag indicating
*			     the end of the list to kdms_set_attributes.
*
*        Output:
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Jeremy Worley and Mark Young
*          Date: Aug 11, 1992 00:45
*      Verified:
*  Side Effects: 
* Modifications:
*
*************************************************************/
int
kdms_set_attributes(
   kobject object,
   char   *segment,
   kvalist)
{
   kva_list list;
   int status;

   if (!_kdms_compiled)
      _kdms_init();

   /* -- set the argument list to start after the segment argument -- */
   kva_start(list, segment);

   /* -- call the variable arguments version of this routine -- */
   status = kdms_vset_attributes(object, segment, &list);

   /* -- terminate the argument list -- */
   kva_end(list);

   return status;
}

/************************************************************
*
*  Routine Name: kdms_vget_attribute - get a single attribute on a kvalist
*
*       Purpose: The purpose of this routine is to allow the programmer to
*		 get the value of an attribute associated
*		 with a data object.
*
*		 Data services manages two sets of attributes
*		 with each object.  Internally, they are referred
*		 to as the physical attributes and the presentation
*		 attributes.  The presentation version of an attribute
*		 is the value that is desired by the programmer.  The
*		 corresponding physical attribute is what is physically
*		 stored on the file or transport associated with the
*		 object.  If the two versions of an attribute result
*		 in different presentations of data, then data services
*		 automatically translates the data from the physical
*		 interpretation to the presentation interpretation, 
*		 or visa-versa, as appropriate.  For example, if the
*		 data type attribute (KDMS_DATA_TYPE) is physically
*		 unsigned int (KULONG), but the presentation value
*		 of that attribute is float (KFLOAT), then data services
*		 will retrieve the data from the object and and cast it to
*		 float before returning it to the user.  Similar behavior
*		 will occur for attributes such as index order 
*		 (KDMS_INDEX_ORDER) and size (KDMS_SIZE).
*
*		 The presentation attributes are settable
*		 at any time throughout a object's lifetime.  The
*		 physical attributes, however, at some point are
*		 locked and thereafter cannot change.  An object
*		 that was opened as an input object (via kdms_input
*		 or kdms_open) has its physical attributes locked
*		 immediately.
*
*		 If the segment argument is NULL, then
*		 this instructs kdms_get_attribute to get
*		 a global attribute, if available.  In either
*		 case, if the attribute does not exist, then
*		 this routine returns FALSE, indicating a
*		 failure.
*
*         Input: 
*
*        Output: 
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Jeremey Worley and Steve Kubica
*          Date: Mar 25, 1994 13:37
*      Verified:
*  Side Effects:
* Modifications:
*
*
*************************************************************/
int
kdms_vget_attribute(
   kobject   object,
   char     *segment,
   char     *attribute,
   kva_list *list)
{
   kdms_segment_defin *segd = NULL;
   kdms_attribute *atr = NULL;
   int status = TRUE;

   if (!_kdms_compiled)
      _kdms_init();

   /* sanity check */
   if (object == NULL || object->type != KOBJ_DATA)
   {
      _kdms_set_error(KDMS_EOBJ_INVALID);
      return FALSE;
   }

   /* -- if there is a defined segment get_atr routine, then use that -- */
   if ((segd = kdms_locate_segment_defin(segment)) != NULL)
      if (segd->get_atr)
      {
	 segd->lock = TRUE;
	 status = segd->get_atr(object, segment, attribute, list);
	 segd->lock = FALSE;
	 return status;
      }
   /* -- otherwise, do a normal get attribute -- */   


   /* retrieve the attribute definition */
   if ((atr = _kdms_retrieve_attribute(object, attribute, segment)) == NULL)
   {
      _kdms_set_error(KDMS_EATTR_NODEF);
      return FALSE;
   }

   /* now we can get it - get it according to type */
   if (atr->base.type == KDMS_GENERIC_ATTRIBUTE_TYPE)
      status = _kdms_get_generic(&(atr->generic), list);

   else if (atr->base.type == KDMS_QUASI_ATTRIBUTE_TYPE)
      status = atr->quasi.get(object, kstring_to_token(segment),
			      atr->quasi.attribute,
			      atr->quasi.clientData, list);
   else
      status = FALSE;

   if (!status)
   {
      _kdms_set_error(KDMS_EATTR_GET_FAILED);
      kinfo(KDEBUG, "kdms_vset_attribute : "
	    "failed to get attribute '%s'.", attribute);
      return FALSE;
   }

   return status;
}

/************************************************************
*
*  Routine Name: kdms_vget_attributes - get attributes on a kvalist
*
*       Purpose: The purpose of this routine is to allow the programmer to
*		 get the values of multiple attributes associated
*		 with a data object
*
*		 Data services manages two sets of attributes
*		 with each object.  Internally, they are referred
*		 to as the physical attributes and the presentation
*		 attributes.  The presentation version of an attribute
*		 is the value that is desired by the programmer.  The
*		 corresponding physical attribute is what is physically
*		 stored on the file or transport associated with the
*		 object.  If the two versions of an attribute result
*		 in different presentations of data, then data services
*		 automatically translates the data from the physical
*		 interpretation to the presentation interpretation, 
*		 or visa-versa, as appropriate.  For example, if the
*		 data type attribute (KDMS_DATA_TYPE) is physically
*		 unsigned int (KULONG), but the presentation value
*		 of that attribute is float (KFLOAT), then data services
*		 will retrieve the data from the object and and cast it to
*		 float before returning it to the user.  Similar behavior
*		 will occur for attributes such as index order 
*		 (KDMS_INDEX_ORDER) and size (KDMS_SIZE).
*
*		 The presentation attributes are settable
*		 at any time throughout a object's lifetime.  The
*		 physical attributes, however, at some point are
*		 locked and thereafter cannot change.  An object
*		 that was opened as an input object (via kdms_input
*		 or kdms_open) has its physical attributes locked
*		 immediately.
*
*		 If the segment argument is NULL, then
*		 this instructs kdms_set_attributes to get
*		 a series of global attributes, if available.
*		 In either case, if an attribute does not exist,
*		 then this routine returns FALSE, indicating a
*		 failure.
*
*         Input: 
*
*        Output: 
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Jeremy Worley
*          Date: Mar 25, 1994 13:37
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/
int
kdms_vget_attributes(
   kobject   object,
   char     *segment,
   kva_list *list)
{
   char *attribute;

   while ((attribute = kva_arg(*list, char *)) != NULL)
      if (!kdms_vget_attribute(object, segment, attribute, list))
	    return FALSE;	/* errno set already */

   return TRUE;
}

/************************************************************
*
*  Routine Name: kdms_get_attribute - get the value of an
*		 attribute within a segment of an abstract
*		 object.
*
*       Purpose: The purpose of this routine is to allow the programmer to
*		 get the value of an attribute associated
*		 with a data object.
*
*		 Data services manages two sets of attributes
*		 with each object.  Internally, they are referred
*		 to as the physical attributes and the presentation
*		 attributes.  The presentation version of an attribute
*		 is the value that is desired by the programmer.  The
*		 corresponding physical attribute is what is physically
*		 stored on the file or transport associated with the
*		 object.  If the two versions of an attribute result
*		 in different presentations of data, then data services
*		 automatically translates the data from the physical
*		 interpretation to the presentation interpretation, 
*		 or visa-versa, as appropriate.  For example, if the
*		 data type attribute (KDMS_DATA_TYPE) is physically
*		 unsigned int (KULONG), but the presentation value
*		 of that attribute is float (KFLOAT), then data services
*		 will retrieve the data from the object and and cast it to
*		 float before returning it to the user.  Similar behavior
*		 will occur for attributes such as index order 
*		 (KDMS_INDEX_ORDER) and size (KDMS_SIZE).
*
*		 The presentation attributes are settable
*		 at any time throughout a object's lifetime.  The
*		 physical attributes, however, at some point are
*		 locked and thereafter cannot change.  An object
*		 that was opened as an input object (via kdms_input
*		 or kdms_open) has its physical attributes locked
*		 immediately.
*
*		 If the segment argument is NULL, then
*		 this instructs kdms_get_attribute to get
*		 a global attribute, if available.  In either
*		 case, if the attribute does not exist, then
*		 this routine returns FALSE, indicating a
*		 failure.
*
*
*        Input:  object    - the object that is involved in
*			     the get attribute operation.
*		 segment   - the data segment whose attribute
*			     is being retrieved.
*		  va_alist - variable argument list, that
*			     contains an attribute followed
*			     by any addresses that will be
*			     filled out with the values
*			     associated with that attribute.
*			     The variable argument list takes
*			     the form:
*
*			     ATTRIBUTE_NAME, &value1 [, &value2, ...]
*
*			     The number of value arguments in
*			     variable argument list depends
*			     on the specific attribute.  For
*			     example, KDMS_DATATYPE requires
*			     one value, but KDMS_SIZE requires
*			     five values.
*
*        Output: See the va_list description in Input.
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions:
*    Written By: Mark Young and Jeremy Worley
*          Date: Aug 11, 1992 00:45
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/
int
kdms_get_attribute(
   kobject object,
   char   *segment,
   char   *attribute,
   kvalist)
{
   kva_list list;
   int status;

   if (!_kdms_compiled)
      _kdms_init();

   /* -- start the argument list after the attribute argument -- */
   kva_start(list, attribute);

   /* -- call the variable argument version of this routine -- */
   status = kdms_vget_attribute(object, segment, attribute, &list);

   /* -- terminate the argument list -- */
   kva_end(list);
   return status;
}


/************************************************************
*
*  Routine Name: kdms_get_attributes - gets the values of a
*		 variable number of attributes within a
*		 single segment of an object.
*
*       Purpose: The purpose of this routine is to allow the programmer to
*		 get the values of multiple attributes associated
*		 with a data object
*
*		 Data services manages two sets of attributes
*		 with each object.  Internally, they are referred
*		 to as the physical attributes and the presentation
*		 attributes.  The presentation version of an attribute
*		 is the value that is desired by the programmer.  The
*		 corresponding physical attribute is what is physically
*		 stored on the file or transport associated with the
*		 object.  If the two versions of an attribute result
*		 in different presentations of data, then data services
*		 automatically translates the data from the physical
*		 interpretation to the presentation interpretation, 
*		 or visa-versa, as appropriate.  For example, if the
*		 data type attribute (KDMS_DATA_TYPE) is physically
*		 unsigned int (KULONG), but the presentation value
*		 of that attribute is float (KFLOAT), then data services
*		 will retrieve the data from the object and and cast it to
*		 float before returning it to the user.  Similar behavior
*		 will occur for attributes such as index order 
*		 (KDMS_INDEX_ORDER) and size (KDMS_SIZE).
*
*		 The presentation attributes are settable
*		 at any time throughout a object's lifetime.  The
*		 physical attributes, however, at some point are
*		 locked and thereafter cannot change.  An object
*		 that was opened as an input object (via kdms_input
*		 or kdms_open) has its physical attributes locked
*		 immediately.
*
*		 If the segment argument is NULL, then
*		 this instructs kdms_set_attributes to get
*		 a series of global attributes, if available.
*		 In either case, if an attribute does not exist,
*		 then this routine returns FALSE, indicating a
*		 failure.
*
*         Input: object    - the object that is involved in
*			     the set attribute operation.
*		 segment   - the data segment whose attribute
*			     is being retrieved.
*		 va_alist  - a variable argument list that
*			     contains a set attributes, each
*			     followed by addresses to variables
*			     that will be assigned with values
*			     associated with that attribute.
*			     The variable argument list takes the
*			     form:
*
*			     ATTRIBUTE_NAME1, &value1 [, &value2, ...],
*			     ATTRIBUTE_NAME2, &value1,[, &value2, ...],
*			     ..., NULL.
*
*			     The number of value arguments in
*			     the variable argument list for each
*			     attribute depends on the specific
*			     attribute.  For
*			     example, KDMS_DATATYPE takes only
*			     one value, but KDMS_SIZE takes
*			     multiple values.  The NULL at the
*			     end of the variable argument
*			     list serves as a flag indicating
*			     the end of the list to kdms_get_attributes.
*
*        Output: See the va_alist description under Input.
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: Use only as directed...
*
*    Written By: Mark Young and Jeremy Worley
*          Date: Aug 11, 1992 00:45
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/
int
kdms_get_attributes(
   kobject object,
   char   *segment,
   kvalist)
{
   kva_list list;
   int status;

   if (!_kdms_compiled)
      _kdms_init();

   /* -- set the argument list to start after the segment argument -- */
   kva_start(list, segment);

   /* -- call the variable argument version of this routine -- */
   status = kdms_vget_attributes(object, segment, &list);

   /* -- terminate the argument list -- */
   kva_end(list);

   return status;
}


/************************************************************
*
*  Routine Name: kdms_match_attribute - returns TRUE if the
*		 same segment attribute in two abstract
*		 data objects match.
*
*       Purpose: The purpose of this routine is to allow the programmer 
*		 to compare a single attribute between two data objects.
*
*		 Data services manages two sets of attributes
*		 with each object.  Internally, they are referred
*		 to as the physical attributes and the presentation
*		 attributes.  The presentation version of an attribute
*		 is the value that is desired by the programmer.  The
*		 corresponding physical attribute is what is physically
*		 stored on the file or transport associated with the
*		 object.  If the two versions of an attribute result
*		 in different presentations of data, then data services
*		 automatically translates the data from the physical
*		 interpretation to the presentation interpretation, 
*		 or visa-versa, as appropriate.  For example, if the
*		 data type attribute (KDMS_DATA_TYPE) is physically
*		 unsigned int (KULONG), but the presentation value
*		 of that attribute is float (KFLOAT), then data services
*		 will retrieve the data from the object and and cast it to
*		 float before returning it to the user.  Similar behavior
*		 will occur for attributes such as index order 
*		 (KDMS_INDEX_ORDER) and size (KDMS_SIZE).
*
*		 The presentation attributes are settable
*		 at any time throughout a object's lifetime.  The
*		 physical attributes, however, at some point are
*		 locked and thereafter cannot change.  An object
*		 that was opened as an input object (via kdms_input
*		 or kdms_open) has its physical attributes locked
*		 immediately.
*
*		 This routine will return TRUE if the
*		 specified attribute has the same value
*		 in both of the abstract data objects.  This
*		 routine will return FALSE if the attribute
*		 does not have the same value in both of
*		 of the objects kdms_match_attribute will
*		 also return FALSE if the attribute does
*		 not exist in either or both of the objects.
*
*		 If the segment argument is NULL, then
*		 that implies that the attribute is a
*		 a global attribute in each of the abstract
*		 data objects.
*
*        INPUT:   object1 - the first abstract data object on
*			    which to match the specified attribute
*		  object2 - the second abstract data object on
*			    which to match the specified attribute
*		  segment  - the data segment in each abstract
*			     data object in which we will be
*			     matching the attribute.
*		  attribute - the attribute that will be
*			      compared in the two objects.
*
*        Output: none
*
*       Returns: There are three ways for this routine to
*		 return a FALSE:  (1) if the attribute
*		 in the two objects does not match; (2) if
*		 either object does not contain the specified 
*		 attribute; (3) an error condition
*		 resulting from an invalid object or segment.
*		 If none of these three conditions exist, then
*		 this function will return TRUE.
*
*  Restrictions: 
*
*    Written By: Mark Young and Jeremy Worley
*          Date: Aug 11, 1992 00:45
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/
int
kdms_match_attribute(
   kobject object1,
   kobject object2,
   char   *segment,
   char   *attribute)
{
   kdms_segment_defin *segd = NULL;
   kdms_attribute *atr1 = NULL;
   kdms_attribute *atr2 = NULL;
   int status = TRUE;

   if (!_kdms_compiled)
      _kdms_init();

   /* -- sanity check -- */
   if (object1 == NULL || object1->type != KOBJ_DATA ||
       object2 == NULL || object2->type != KOBJ_DATA)
   {
      _kdms_set_error(KDMS_EOBJ_INVALID);
      return FALSE;
   }

   /* -- if there is a defined segment match_atr routine, then use that -- */
   if ((segd = kdms_locate_segment_defin(segment)) != NULL)
      if (segd->match_atr)
      {
	 segd->lock = TRUE;
	 status = segd->match_atr(object1, object2, segment, attribute);
	 segd->lock = FALSE;
	 return status;
      }
   /* -- otherwise, do a normal match attribute -- */   


   /* -- retrieve the attribute definitions -- */
   if ((atr1 = _kdms_retrieve_attribute(object1, attribute, segment)) == NULL) 
   {
      _kdms_set_error(KDMS_EATTR_NODEF);
      return FALSE;
   }

   if ((atr2 = _kdms_retrieve_attribute(object2, attribute, segment)) == NULL)
   {
      _kdms_set_error(KDMS_EATTR_NODEF);
      return FALSE;
   }

   /* -- now we can match it - match it according to type -- */
   if (atr1->base.type == KDMS_GENERIC_ATTRIBUTE_TYPE)
      status = _kdms_match_generic(&(atr1->generic), &(atr2->generic));

   else if (atr1->base.type == KDMS_QUASI_ATTRIBUTE_TYPE)
      status = atr1->quasi.match(object1, object2, 
   			         kstring_to_token(segment),
				 atr1->quasi.attribute,
				 atr1->quasi.clientData,
				 atr2->quasi.clientData);
   else
      status = FALSE;		/* this should never happen */

   return status;		/* FALSE is not an error */
}

/************************************************************
*
*  Routine Name: kdms_vmatch_attributes - returns true if the
*		 vararg list of segment attributes in two
*		 abstract data objects match.
*
*       Purpose: The purpose of this routine
*		 is to allow the programmer to compare multiple
*		 attributes in two object.
*
*		 This routine will return TRUE if all of the
*		 specified attributes have the same value
*		 in the objects.  This routine will return
*		 FALSE if any of the attributes do not
*		 match  kdms_match_attributes will also
*		 return FALSE if any of the attributes do
*		 not exist in either or both of the two objects.
*
*		 Data services manages two sets of attributes
*		 with each object.  Internally, they are referred
*		 to as the physical attributes and the presentation
*		 attributes.  The presentation version of an attribute
*		 is the value that is desired by the programmer.  The
*		 corresponding physical attribute is what is physically
*		 stored on the file or transport associated with the
*		 object.  If the two versions of an attribute result
*		 in different presentations of data, then data services
*		 automatically translates the data from the physical
*		 interpretation to the presentation interpretation, 
*		 or visa-versa, as appropriate.  For example, if the
*		 data type attribute (KDMS_DATA_TYPE) is physically
*		 unsigned int (KULONG), but the presentation value
*		 of that attribute is float (KFLOAT), then data services
*		 will retrieve the data from the object and and cast it to
*		 float before returning it to the user.  Similar behavior
*		 will occur for attributes such as index order 
*		 (KDMS_INDEX_ORDER) and size (KDMS_SIZE).
*
*		 The presentation attributes are settable
*		 at any time throughout a object's lifetime.  The
*		 physical attributes, however, at some point are
*		 locked and thereafter cannot change.  An object
*		 that was opened as an input object (via kdms_input
*		 or kdms_open) has its physical attributes locked
*		 immediately.
*
*		 If the segment argument is NULL, then
*		 that implies that the attributes are
*		 global attributes in each of the abstract
*		 data objects.
*
*        INPUT:   object1  - the first abstract data object on
*			     which to match the specified attributes
*		  object2  - the second abstract data object on
*			     which to match the specified attributes
*		  segment  - the data segment in each abstract
*			     data object in which we will be
*			     matching the attributes.
*		  list     - variable argument list, that
*			     contains an arbitrarily long
*			     list of attributes followed
*			     a NULL.  It takes the form:
*
*			     ATTRIBUTE_NAME1, ATTRIBUTE_NAME2, ..., NULL
*
*        Output: none
*       Returns: There are three ways for this routine to
*		 return a FALSE:  (1) if any of the attributes
*		 between the two objects do not match; (2) if
*		 either object does not contain one or more of
*		 the specified attributes; (3) an error condition
*		 resulting from an invalid object or segment.
*		 If none of these three conditions exist, then
*		 this function will return TRUE.
*
*  Restrictions: 
*
*    Written By: Jeremy Worley & John Salas
*          Date: Oct 06, 1993 19:51
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/
int
kdms_vmatch_attributes(
   kobject   object1,
   kobject   object2,
   char     *segment,
   kva_list *list)
{
   char *attribute;

   if (!_kdms_compiled)
      _kdms_init();

   while ((attribute = kva_arg(*list, char *)) != NULL)
      if (!kdms_match_attribute(object1, object2, segment, attribute))
      {
	 kva_end(*list);
	 return FALSE;	/* not an error */
      }

   return TRUE;
}

/************************************************************
*
*  Routine Name: kdms_match_attributes - returns true if the
*		 list of segment attributes in two
*		 abstract data objects match.
*
*       Purpose: The purpose of this routine
*		 is to allow the programmer to compare multiple
*		 attributes in two object.
*
*		 This routine will return TRUE if all of the
*		 specified attributes have the same value
*		 in the objects.  This routine will return
*		 FALSE if any of the attributes do not
*		 match  kdms_match_attributes will also
*		 return FALSE if any of the attributes do
*		 not exist in either or both of the two objects.
*
*		 Data services manages two sets of attributes
*		 with each object.  Internally, they are referred
*		 to as the physical attributes and the presentation
*		 attributes.  The presentation version of an attribute
*		 is the value that is desired by the programmer.  The
*		 corresponding physical attribute is what is physically
*		 stored on the file or transport associated with the
*		 object.  If the two versions of an attribute result
*		 in different presentations of data, then data services
*		 automatically translates the data from the physical
*		 interpretation to the presentation interpretation, 
*		 or visa-versa, as appropriate.  For example, if the
*		 data type attribute (KDMS_DATA_TYPE) is physically
*		 unsigned int (KULONG), but the presentation value
*		 of that attribute is float (KFLOAT), then data services
*		 will retrieve the data from the object and and cast it to
*		 float before returning it to the user.  Similar behavior
*		 will occur for attributes such as index order 
*		 (KDMS_INDEX_ORDER) and size (KDMS_SIZE).
*
*		 The presentation attributes are settable
*		 at any time throughout a object's lifetime.  The
*		 physical attributes, however, at some point are
*		 locked and thereafter cannot change.  An object
*		 that was opened as an input object (via kdms_input
*		 or kdms_open) has its physical attributes locked
*		 immediately.
*
*		 If the segment argument is NULL, then
*		 that implies that the attributes are
*		 global attributes in each of the abstract
*		 data objects.
*
*        INPUT:   object1 - the first abstract data object on
*			    which to match the specified attributes
*		  object2 - the second abstract data object on
*			    which to match the specified attributes
*		  segment  - the data segment in each abstract
*			     data object in which we will be
*			     matching the attributes.
*		  va_alist - variable argument list, that
*			     contains an arbitrarily long
*			     list of attributes followed
*			     a NULL.  It takes the form:
*
*			     ATTRIBUTE_NAME1, ATTRIBUTE_NAME2, ..., NULL
*
*        Output: none
*       Returns: There are three ways for this routine to
*		 return a FALSE:  (1) if any of the attributes
*		 between the two objects do not match; (2) if
*		 either object does not contain one or more of
*		 the specified attributes; (3) an error condition
*		 resulting from an invalid object or segment.
*		 If none of these three conditions exist, then
*		 this function will return TRUE.
*
*  Restrictions: 
*
*    Written By: Mark Young and Jeremy Worley
*          Date: Aug 11, 1992 00:45
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/
int
kdms_match_attributes(
   kobject object1,
   kobject object2,
   char   *segment,
   kvalist)
{
   kva_list list;
   int status;

   if (!_kdms_compiled)
      _kdms_init();

   /* -- set the argument list to start after the segment argument -- */
   kva_start(list, segment);

   /* -- call the varargs version of the routine -- */
   status = kdms_vmatch_attributes(object1, object2, segment, &list);

   /* -- terminate the argument list -- */
   kva_end(list);

   return status;
}



/************************************************************
*
*  Routine Name: kdms_copy_attribute - copy an attribute from a 
*			source object to a destination object.
*
*       Purpose: This function is used to copy a single 
*		 attribute from one object to another object.
*
*		 Data Services manages two versions of some of the
*		 attributes associated with each object.  These
*		 attributes are the size and data type.  Internally,
*		 the two versions of these attributes are referred to
*		 as the physical attribute and the presentation
*		 attribute.  Typically, the programmer has access
*		 to only the presentation versions of these attributes.
*		 The physical attributes are set indirectly depending
*		 on the setting of KPDS_VALUE_COUPLING.  See kpds_get_data
*		 for a description of how the presentation and physical
*		 attributes affect interaction with the data object.
*
*	         Other attributes are classified as either shared or
*	         unshared.  Shared attributes are stored at the
*	         physical layer of the attribute, and thus can be
*	         shared by multiple references of the data object (see
*	         kpds_reference_object for more information about
*	         references).  Unshared attributes, on the other hand,
*	         can only be used by the local object (see
*	         kpds_query_attribute for information on how to
*	         determine whether an attribute is shared or
*	         unshared).
*
*		 This function accepts a source object, a
*		 destination object, and an attribute name.
*		 If the attribute exists in the source object,
*		 then it will be copied to the destination object.
*		 If the attribute does not exist in the source
*		 object, then an error condition is returned.
*		
*         Input: object1 - the source for the copy operation
*		 object2 - the destination for the copy operation
*		 segment - the segment which contains the object to
*		 	   be copied.
*		 attribute - the attribute to be copied.
*
*        Output: 
*       Returns: TRUE on success, FALSE otherwise.
*  Restrictions: 
*
*    Written By: Jeremy Worley and Steve Kubica
*          Date: Aug 11, 1992 00:45
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/
int
kdms_copy_attribute(
   kobject object1,
   kobject object2,
   char   *segment,
   char   *attribute)
{
   kdms_segment_defin *segd = NULL;
   int status = TRUE;

   if (!_kdms_compiled)
      _kdms_init();

   /* -- sanity check -- */
   if (object1 == NULL || object1->type != KOBJ_DATA ||
       object2 == NULL || object2->type != KOBJ_DATA)
   {
      _kdms_set_error(KDMS_EOBJ_INVALID);
      return FALSE;
   }

   /* -- if there is a defined segment copy_atr routine, then use that -- */
   if ((segd = kdms_locate_segment_defin(segment)) != NULL)
      if (segd->copy_atr)
      {
	 segd->lock = TRUE;
	 status = segd->copy_atr(object1, object2, segment, attribute);
	 segd->lock = FALSE;
	 return status;
      }
   /* -- otherwise, do a normal copy attribute -- */   


   return _kdms_copy_attribute(object1, object2, segment, attribute);
}

/************************************************************
*
*  Routine Name: kdms_vcopy_attributes - copy attributes given in a kvalist
*
*       Purpose: This function is used to copy multiple attributes
*		 given a variable argument list containing a list
*		 attributes to copy.
*
*		 Data Services manages two versions of some of the
*		 attributes associated with each object.  These
*		 attributes are the size and data type.  Internally,
*		 the two versions of these attributes are referred to
*		 as the physical attribute and the presentation
*		 attribute.  Typically, the programmer has access
*		 to only the presentation versions of these attributes.
*		 The physical attributes are set indirectly depending
*		 on the setting of KPDS_VALUE_COUPLING.  See kpds_get_data
*		 for a description of how the presentation and physical
*		 attributes affect interaction with the data object.
*
*	         Other attributes are classified as either shared or
*	         unshared.  Shared attributes are stored at the
*	         physical layer of the attribute, and thus can be
*	         shared by multiple references of the data object (see
*	         kpds_reference_object for more information about
*	         references).  Unshared attributes, on the other hand,
*	         can only be used by the local object (see
*	         kpds_query_attribute for information on how to
*	         determine whether an attribute is shared or
*	         unshared).
*
*		 This function accepts a source object, a
*		 destination object, and an attribute name.
*		 If the attribute exists in the source object,
*		 then it will be copied to the destination object.
*		 If the attribute does not exist in the source
*		 object, then an error condition is returned.
*		
*         Input: object1 - the source for the copy operation
*		 object2 - the destination for the copy operation
*		 segment - the segment which contains the object to
*		 	   be copied.
*		 kvalist - A NULL terminated variable argument 
*		 	   list of attributes to be copied.
*
*        Output: 
*       Returns: TRUE on success, FALSE otherwise.
*
*  Restrictions: 
*
*    Written By: Jeremy Worley & Steve Kubica
*          Date: Oct 06, 1993 19:51
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/
int
kdms_vcopy_attributes(
   kobject   object1,
   kobject   object2,
   char     *segment,
   kva_list *list)
{
   char *attribute;

   if (!_kdms_compiled)
      _kdms_init();

   while ((attribute = kva_arg(*list, char *)) != NULL)
      if (!kdms_copy_attribute(object1, object2, segment, attribute))
      {
	 kva_end(*list);
	 return FALSE;	/* errno set already */
      }

   return TRUE;
}

/************************************************************
*
*  Routine Name: kdms_copy_attributes - copy attributes from a 
*		 		        source object to a destination object.
*
*       Purpose: This function is used to copy multiple attributes
*		 given a variable argument list containing a list
*		 attributes to copy.
*
*		 Data Services manages two versions of some of the
*		 attributes associated with each object.  These
*		 attributes are the size and data type.  Internally,
*		 the two versions of these attributes are referred to
*		 as the physical attribute and the presentation
*		 attribute.  Typically, the programmer has access
*		 to only the presentation versions of these attributes.
*		 The physical attributes are set indirectly depending
*		 on the setting of KPDS_VALUE_COUPLING.  See kpds_get_data
*		 for a description of how the presentation and physical
*		 attributes affect interaction with the data object.
*
*	         Other attributes are classified as either shared or
*	         unshared.  Shared attributes are stored at the
*	         physical layer of the attribute, and thus can be
*	         shared by multiple references of the data object (see
*	         kpds_reference_object for more information about
*	         references).  Unshared attributes, on the other hand,
*	         can only be used by the local object (see
*	         kpds_query_attribute for information on how to
*	         determine whether an attribute is shared or
*	         unshared).
*
*		 This function accepts a source object, a
*		 destination object, and an attribute name.
*		 If the attribute exists in the source object,
*		 then it will be copied to the destination object.
*		 If the attribute does not exist in the source
*		 object, then an error condition is returned.
*		
*         Input: object1 - the source for the copy operation
*		 object2 - the destination for the copy operation
*		 segment - the segment which contains the object to
*		 	   be copied.
*		 kvalist - A NULL terminated variable argument 
*		 	   list of attributes to be copied.
*
*        Output:
*       Returns: TRUE on success, FALSE otherwise.
*		
*  Restrictions: 
*
*    Written By: Jeremy Worley and Steve Kubica
*          Date: Aug 11, 1992 00:45
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/
int
kdms_copy_attributes(
   kobject object1,
   kobject object2,
   char   *segment,
   kvalist)
{
   kva_list list;
   int status;

   if (!_kdms_compiled)
      _kdms_init();

   /* -- set the argument list to start after the segment argument -- */
   kva_start(list, segment);

   /* -- call the varargs version of the routine -- */
   status = kdms_vcopy_attributes(object1, object2, segment, &list);

   /* -- terminate the argument list -- */
   kva_end(list);

   return status;
}

/************************************************************
*
*  Routine Name: kdms_query_attribute - get information about an attribute
*
*       Purpose: This function is used for two purposes:  
*		 (1) to determine the existence of an attribute; and
*		 (2) to obtain the characteristics of the attribute.
*
*		 Data Services manages two versions of some of the
*		 attributes associated with each object.  These
*		 attributes are the size and data type.  Internally,
*		 the two versions of these attributes are referred to
*		 as the physical attribute and the presentation
*		 attribute.  Typically, the programmer has access
*		 to only the presentation versions of these attributes.
*		 The physical attributes are set indirectly depending
*		 on the setting of KDMS_COUPLING.  See kpds_get_data
*		 for a description of how the presentation and physical
*		 attributes affect interaction with the data object.
*
*	         Other attributes are classified as either shared or
*	         unshared.  Shared attributes are stored at the
*	         physical layer of the attribute, and thus can be
*	         shared by multiple references of the data object (see
*	         kpds_reference_object for more information about
*	         references).  Unshared attributes, on the other hand,
*	         can only be used by the local object.
*
*		 The difference between shared and unshared attributes
*		 is abstracted from the user at the PDS level.  The
*		 permanent attributes are generally shared, and the
*		 non-permanent attributes are generally non-shared.
*		 Permanent attributes are attributes that will be stored
*		 as part of an output object when it is written.  Any
*		 attributes that are retrieved when an object is opened
*		 are also permanent attributes.  Non-permanent attributes
*		 exist only while the program that is operating on the
*		 object is executing.
*
*		 The datatype argument indicates what kind of
*		 information is stored in the attribute. Attributes
*		 can be one of the following data types: KBYTE,
*		 KUBYTE, KSHORT, KUSHORT, KINT, KUINT, KLONG, KULONG,
*		 KFLOAT, KDOUBLE, KCOMPLEX, or KDCOMPLEX.  
*
*		 The num_args argument indicates how many arguments
*		 must be passed in an argument list to one of the
*		 attribute functions.
*
*		 The size arguments indicates the number of
*		 units of the data type there are in each argument.
*		 This argument allows arrays of information to be
*		 stored as attributes.
*
*         Input: object      - the object with the attribute being
*			       queried
*				
*		 segment     - the segment that the attribute
*			       is stored in.  If this argument
*			       is NULL, then the attribute is
*			       global to the object.
*		 name        - name of the attribute to be
*			       queried.
*		 datatype    - datatype of the attribute
*		 num_args    - number of arguments in this attribute
*		 size        - size of each argument in this
*		 	       attribute.
*		 permanent - is the attribute stored or
*			       transient?  The return value
*			       will be either TRUE
*			       or FALSE
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Jeremy Worley and Steve Kubica
*          Date: Sep 23, 1993 16:04
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/
int
kdms_query_attribute(
   kobject  object,
   char    *segment,
   char    *attribute,
   int     *num_args,
   int     *arg_size,
   int     *data_type,
   int     *permanent)
{ 
   kdms_segment_defin *segd = NULL;
   kdms_attribute *atr = NULL;
   int status = TRUE;

   if (!_kdms_compiled)
      _kdms_init();

   /* -- sanity check -- */
   if (object == NULL || object->type != KOBJ_DATA)
   {
      _kdms_set_error(KDMS_EOBJ_INVALID);
      return FALSE;
   }

   /* -- if there is a defined segment query_atr routine, then use that -- */
   if ((segd = kdms_locate_segment_defin(segment)) != NULL)
      if (segd->query_atr)
      {
	 segd->lock = TRUE;
	 status = segd->query_atr(object, segment, attribute, num_args, 
				  arg_size, data_type, permanent);
	 segd->lock = FALSE;
	 return status;
      }
   /* -- otherwise, do a normal query attribute -- */   

   /* -- retrieve the attribute definition -- */
   if ((atr = _kdms_retrieve_attribute(object, attribute, segment)) == NULL)
      return FALSE; /* this is not an error...false == non-exist */


   /* -- now we can query it - query it according to type -- */
   if (atr->base.type == KDMS_GENERIC_ATTRIBUTE_TYPE)
      status = _kdms_query_generic(&(atr->generic), num_args, arg_size,
				   data_type, permanent);

   else if (atr->base.type == KDMS_QUASI_ATTRIBUTE_TYPE)
      status = atr->quasi.query(object, kstring_to_token(segment),
				atr->quasi.attribute,
				atr->quasi.clientData, num_args,
				arg_size, data_type, permanent);
   else
      status = FALSE;

   return status;
}

/************************************************************
*
*  Routine Name: kdms_print_attribute - print the value of an attribute
*
*       Purpose: This function is used to print a single
*		 attribute to an open kfile.
*
*		 This function is typically used by such programs as
*		 kprdata to print out the values of attributes in
*		 an object.
*
*         Input: object     - the object containing the attribute
*		 segment    - the segment in the object which contains
*		 	      the attribute.
*		 attribute  - the attribute to print
*		 printfile  - the open kfile to print to
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Steve Kubica
*          Date: 
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/
int
kdms_print_attribute(
   kobject object,
   char   *segment,
   char   *attribute,
   kfile  *printfile)
{
   kdms_segment_defin *segd = NULL;
   kdms_attribute *atr = NULL;
   int status = TRUE;

   if (!_kdms_compiled)
      _kdms_init();

   /* -- sanity check -- */
   if (object == NULL || object->type != KOBJ_DATA)
   {
      _kdms_set_error(KDMS_EOBJ_INVALID);
      return FALSE;
   }

   if (printfile == NULL)
   {
      _kdms_set_error(KINVALID_FILE);
      return FALSE;
   }

   /* -- if there is a defined segment print_atr routine, then use that -- */
   if ((segd = kdms_locate_segment_defin(segment)) != NULL)
      if (segd->print_atr)
      {
	 segd->lock = TRUE;
	 status = segd->print_atr(object, segment, attribute, printfile);
	 segd->lock = FALSE;
	 return status;
      }
   /* -- otherwise, do a normal print attribute -- */   


   /* -- retrieve the attribute definition -- */
   if ((atr = _kdms_retrieve_attribute(object, attribute, segment)) == NULL)
   {
      _kdms_set_error(KDMS_EATTR_NODEF);
      return FALSE;
   }

   /* -- now we can print it - print it according to type -- */
   if (atr->base.type == KDMS_GENERIC_ATTRIBUTE_TYPE)
      status = _kdms_print_generic(&(atr->generic), printfile);

   else if (atr->base.type == KDMS_QUASI_ATTRIBUTE_TYPE)
      status = atr->quasi.print(object, kstring_to_token(segment),
				atr->quasi.attribute,
				atr->quasi.clientData, printfile);
   else
      status = FALSE;

   if (!status)
   {
      kinfo(KDEBUG, "kdms_print_attribute : failed to print "
	            "the attribute '%s'.", attribute);
      return FALSE;
   }

   return status;
}

/*
 *    ================================================================== 
 *    Attribute Utility Functions
 *    ==================================================================
 */

/************************************************************
*
*  Routine Name: kdms_get_attribute_names - get a list of attributes 
*	                                    from an object.
*
*       Purpose: This function returns a list of attributes
*		 associated with the specified segment.  If
*		 the segment is NULL, then it returns a list
*		 of the attributes for the object.
*
*		 The allowable regular expression syntax is :
*
* !   .             Match any single character except newline
* !
* !   *             Match the preceding character or range 
* !                 of characters 0 or more times.  The 
* !                 matching includes items within a [...].
* !
* ! [...] or [^..]  Matches any one character contained within 
* !                 the brackets. If the first character after 
* !                 the '[' is the ']', then it is included in 
* !                 the characters to match.  If the first 
* !                 character after the '[' is a '^', then it 
* !                 will match all characters NOT included in 
* !                 the [].  The '-' will indicate a range of 
* !                 characters.  For example, [a-z] specifies 
* !                 all characters between and including the 
* !                 ascii values 'a' and 'z'.  If the '-' 
* !                 follows the '[' or is right before the ']' 
* !                 then it is interpreted literally.
* !             
* !   ^             If this is the first character of the 
* !                 regular expression, it matches the beginning
* !                 of the line.
* !  
* !   $             If this is the last character of the 
* !                 regular expression, it matches the end of
* !                 the line. 
* !
* !   \\            This escapes the meaning of a special character.
* ! 
*
*		 The array that is returned must be freed by
*		 the user using the call karray_free.
*
*         Input: object      - the object to get the attribute names from. 
*
*	         segment     - the segment to get the attribute names from.
*			       If NULL, then get the object-level names.
*
*                filter      - a regular expression filter.  Only
*			       attribute names passing the filter
*			       will be returned in the list.
*
*                permanent   - TRUE if only permanent attributes 
*                              should be included in the list
*
*        Output: number - the number of attribute names that
*		          are being returned
*
*       Returns: an array of attribute names
*
*  Restrictions: 
*    Written By: Steve Kubica and Jeremy Worley
*          Date: Apr 30, 1994 22:11
*      Verified:
*  Side Effects:
*    Modifications:
*
*************************************************************/
char **
kdms_get_attribute_names(
   kobject object,
   char   *segment,
   char   *filter,
   int     permanent,
   int    *number)
{
   int assoc = kstring_to_token(segment);
   char **attributes = NULL;
   char *a = NULL;
   int nattributes = 0;
   int i;
   int pass;
   int have_filter = FALSE;

   kdms_attribute_list **atr_list = NULL;

   /* -- compile the filter if it is present -- */
   if (filter != NULL)
   {
      if (kre_comp(filter) != NULL)
      {
	 kinfo(KDEBUG, "kdms_get_attribute_list: the filter '%s' does not "
	               "compile into a valid regular expression\n", filter);
	 kinfo(KDEBUG, "Filter not being used");
      }
      else
	 have_filter = TRUE;
   }

   /* -- permenent attributes are never at the presentation -- */
   if (!permanent)
   {

      /* -- get the presentation attribute list for the given association -- */
      atr_list = _kdms_locate_attribute_list(object, assoc, FALSE);

      if (*atr_list != NULL)
	 for (i = 0; i < (*atr_list)->nattributes; i++)
	 {
	    a = ktoken_to_string((*atr_list)->attributes[i]->base.attribute);

	    if (have_filter)
	       pass = kre_exec(a);
	    else
	       pass = TRUE;

	    if (pass)
	    {
	       attributes = karray_add(attributes, kstrdup(a), nattributes);
	       nattributes++;
	    }
	 }
   }

   /* -- get the physical attribute list for the given association -- */
   atr_list = _kdms_locate_attribute_list(object, assoc, TRUE);

   if (*atr_list != NULL)
      for (i = 0; i < (*atr_list)->nattributes; i++)
      {

	 /* -- only get permanent generic attributes -- */
	 if (permanent)
	 {
	    if (((*atr_list)->attributes[i]->base.type ==
		 KDMS_GENERIC_ATTRIBUTE_TYPE) &&
		((*atr_list)->attributes[i]->generic.permanent == TRUE))
	    {

	       a = ktoken_to_string(
		  (*atr_list)->attributes[i]->base.attribute);

	       if (have_filter)
		  pass = kre_exec(a);
	       else
		  pass = TRUE;

	       if (pass)
	       {
		  attributes = karray_add(attributes, kstrdup(a), nattributes);
		  nattributes++;
	       }
	    }
	 }
	 else
	 {
	    a = ktoken_to_string(
	       (*atr_list)->attributes[i]->base.attribute);

	    if (have_filter)
	       pass = kre_exec(a);
	    else
	       pass = TRUE;

	    if (pass)
	    {
	       attributes = karray_add(attributes, kstrdup(a), nattributes);
	       nattributes++;
	    }
	 }
      }

   if (number != NULL)
      *number = nattributes;

   return attributes;
}

/*-----------------------------------------------------------
|
|  Routine Name: kdms_get_Vattribute
|
|       Purpose: 
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Nov 17, 1993 12:28
| Modifications:
|
------------------------------------------------------------*/
int
kdms_get_Vattribute(
   kobject object,
   char   *segment,
   char   *attribute,
   kaddr  *dude)
{
   kdms_attribute *atr = NULL;

   /* -- retrieve the attribute definition -- */
   if ((atr = _kdms_retrieve_attribute(object, attribute, segment)) == NULL) 
   {
      _kdms_set_error(KDMS_EATTR_NODEF);
      return FALSE;		/* errno set already */
   }

   /* -- this only works for generic attributes -- */
   if (atr->base.type == KDMS_GENERIC_ATTRIBUTE_TYPE)
      *dude = (kaddr) atr->generic.value;
   else
      return FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|
|  Routine Name: kdms_set_Vattribute
|
|       Purpose: 
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Nov 17, 1993 12:28
| Modifications:
|
------------------------------------------------------------*/
int
kdms_set_Vattribute(
   kobject object,
   char   *segment,
   char   *attribute,
   kaddr   dude)
{
   kdms_attribute *atr = NULL;

   /* -- retrieve the attribute definition -- */
   if ((atr = _kdms_retrieve_attribute(object, attribute, segment)) == NULL)
   {
      _kdms_set_error(KDMS_EATTR_NODEF);
      return FALSE;
   }

   /* -- this only works for generic attributes -- */
   if (atr->base.type == KDMS_GENERIC_ATTRIBUTE_TYPE)
   {
      if (!_kdms_copy_attribute_data(dude, &(atr->generic.value),
				     atr->generic.data_type,
				     atr->generic.arg_size,
				     atr->generic.num_args))
      {
	 _kdms_set_error(KDMS_EATTR_SET_FAILED);
	 return FALSE;
      }
   }
   return TRUE;
}


/*-----------------------------------------------------------
|
|  Routine Name: kdms_free_segment_attributes
|
|       Purpose: This routine is used to delete all of the
|		 instantiated attributes for a particular
|		 segment of a given data object.
|
|	         This routine should only be called when deleting
|	         the physical segment.
|
|         Input: presentation - presentation to delete 
|		                attributes from.
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Jan 20, 1993 11:05
| Modifications:
|
------------------------------------------------------------*/
int
kdms_free_segment_attributes(kpresentation * pres)
{
   int i;

   if (pres->segment->attributes == NULL)
      return TRUE;

   for (i = 0; i < pres->segment->attributes->nattributes; i++)
   {
      if (!(pres->segment->attributes->attributes[i] == NULL ||
	    pres->segment->attributes->attributes[i]->base.attribute == 0))
	 _kdms_destruct_attribute(pres->segment->attributes->attributes[i]);
   }

   /* -- free internal attribute list, and then attribute list itself -- */
   kfree(pres->segment->attributes->attributes);
   kfree(pres->segment->attributes);
   pres->segment->attributes = NULL;

   return TRUE;
}

/*-----------------------------------------------------------
|
|  Routine Name: kdms_free_pres_attributes
|
|       Purpose: This routine is used to delete all of the
|		 instantiated attributes for a particular
|		 presentation of a given data object.
|
|         Input: presentation - presentation to delete 
|		                attributes from.
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Jan 20, 1993 11:05
| Modifications:
|
------------------------------------------------------------*/
int
kdms_free_pres_attributes(kpresentation * pres)
{
   int i;

   if (pres->attributes == NULL)
      return TRUE;

   for (i = 0; i < pres->attributes->nattributes; i++)
   {
      if (!(pres->attributes->attributes[i] == NULL ||
	    pres->attributes->attributes[i]->base.attribute == 0))
	 _kdms_destruct_attribute(pres->attributes->attributes[i]);
   }

   /* -- free internal attribute list, and then attribute list itself -- */
   kfree(pres->attributes->attributes);
   kfree(pres->attributes);
   pres->attributes = NULL;

   return TRUE;
}


/*-----------------------------------------------------------
|
|  Routine Name: kdms_free_object_attributes
|
|       Purpose: This routine is used to delete all user
|		 attributes within a data object.
|
|         Input: object - object to free the attributes from
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley and Steve Kubica
|          Date: Jan 20, 1993 11:05
| Modifications:
|
------------------------------------------------------------*/
int
kdms_free_object_attributes(kobject object)
{
   int i;

   if (object->phys->attributes == NULL)
      return TRUE;

   for (i = 0; i < object->phys->attributes->nattributes; i++)
   {
      if (!(object->phys->attributes->attributes[i] == NULL ||
	    object->phys->attributes->attributes[i]->base.attribute == 0))
	 _kdms_destruct_attribute(object->phys->attributes->attributes[i]);
   }

   /* free the internal attribute list, and then the attribute list itself */
   kfree(object->phys->attributes->attributes);
   kfree(object->phys->attributes);
   object->phys->attributes = NULL;

   return TRUE;
}

/************************************************************
*
*  Routine Name: kdms_copy_segment_attributes - copy attributes 
*                from a segment in a source object to the segment 
*	  	 of a destination object.
*
*       Purpose: This routine will copy the list of instantiated
*		 attributes from one segment in one object to
*		 one segment in another.  This routine can also
*		 copy the object level attributes, if NULL is
*		 passed in for the source segment name.
*
*		 The presentation of the given object's segment
*		 size, index order, and data type are set as 
*		 the size, index order, and data type of the 
*		 destination segment.
*
*		 Currently, only the physically instantiated 
*		 user-attributes are copied.
*
*         Input: src_object   - the object to copy from
*
*		 src_segment  - the segment to copy from.
*			        If this is NULL, then the
*			        object level attributes will 
*			        be copied.  In this case, the
*			        dest_segment will be ignored.
*
*                dest_object  - the object to copy into
*
*		 dest_segment - the segment to copy to.
*			        If this is NULL, then the
*			        object level attributes will be copied.
*
*        Output: none
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Steve Kubica
*          Date: Mar 30, 1994 18:32
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/
int
kdms_copy_segment_attributes(
   kobject src_object,
   char   *src_segment,
   kobject dest_object,
   char   *dest_segment)
{
   kpresentation *src_pres = NULL;
   kpresentation *dest_pres = NULL;
   int new_segment = FALSE;
   int src_assoc;
   int dest_assoc;
   int i;
   int transferable;
   
   /* -- sanity check -- */
   if (src_object == NULL || src_object->type != KOBJ_DATA ||
       dest_object == NULL || dest_object->type != KOBJ_DATA)
   {
      _kdms_set_error(KDMS_EOBJ_INVALID);
      return FALSE;
   }

   /* -- copy only the object level attributes -- */
   if (src_segment == NULL)
   {
      /* -- copy all of the physical object-level attributes -- */
      _kdms_copy_attribute_list(src_object, dest_object, 0,
				src_object->phys->attributes,
				&dest_object->phys->attributes);

      /* -- copy all of the not-so-physical object-level attributes -- */
      _kdms_copy_attribute_list(src_object, dest_object, 0,
				src_object->attributes,
				&dest_object->attributes);
      return TRUE;
   }

   if (!kdms_get_attribute(src_object, src_segment, KDMS_TRANSFERABLE, 
			    &transferable))
      return FALSE;

   /* -- if the segment is not transferable, then drop out now -- */
   if (!transferable)
      return TRUE;
   
   /* -- if a second segment was not provided, use the first one for both -- */
   if (dest_segment == NULL)
      dest_segment = src_segment;

   src_assoc = kstring_to_token(src_segment);
   dest_assoc = kstring_to_token(dest_segment);

   /* -- get the presentation associated with the first segment -- */
   if (!_kdms_get_segment_info(src_object, src_assoc, &src_pres) || !src_pres)
      return FALSE;		/* errno set already */

   /*
    * try to get the presentation associated with the second segment. if it
    * does not exist, then create one.  Otherwise, copy the attributes.
    */
   if (!_kdms_get_segment_info(dest_object, dest_assoc, &dest_pres) ||
       !dest_pres)
   {

      /* -- create the presentation structure -- */
      _kdms_create_segment(dest_object, dest_assoc, NULL);
      if (!_kdms_get_segment_info(dest_object, dest_assoc, &dest_pres) ||
	  !dest_pres)
	 return FALSE;

      _kdms_set_segment_info(dest_object, dest_assoc, dest_pres);
      new_segment = TRUE;
   }

   /* -- copy over all the presentation attributes -- */
   dest_pres->datatype = src_pres->datatype;
   dest_pres->dimension = src_pres->dimension;
   dest_pres->ragged = src_pres->ragged;

   dest_pres->complex_convert = src_pres->complex_convert;
   dest_pres->pad_real        = src_pres->pad_real;
   dest_pres->pad_imag        = src_pres->pad_imag;

   for (i = 0; i < src_pres->dimension; i++)
   {
      dest_pres->size[i] = src_pres->size[i];
      dest_pres->order[i] = src_pres->order[i];
   }

  /*
    * if the dimension, size, data type, or index order at the
    * physical layer has not yet been initialized and the coupling is
    * either demand or coupled, then go ahead and set the physical
    * size as well.  Otherwise, mark the segment for update if its
    * coupled. 
    */

   if (dest_pres->segment->dimension <= 0)
      dest_pres->segment->dimension = dest_pres->dimension;
   if (!_kdms_legal_size(dest_pres->segment->size, dest_pres->dimension) &&
       dest_pres->coupling != KUNCOUPLED)
      kmemcpy(dest_pres->segment->size, dest_pres->size, 
	      dest_pres->dimension * sizeof(int));
   if (!_kdms_legal_order(dest_pres->segment->order, dest_pres->dimension))
      kmemcpy(dest_pres->segment->order, dest_pres->order, 
	      dest_pres->dimension*sizeof(int));
   if (!_kdms_legal_datatype(dest_pres->segment->datatype))
      dest_pres->segment->datatype = dest_pres->datatype;

   /* -- copy all of the user defined attributes -- */
   _kdms_copy_attribute_list(src_object, dest_object, 0,
			     src_pres->segment->attributes,
			     &dest_pres->segment->attributes);

   _kdms_copy_attribute_list(src_object, dest_object, 0,
			     src_pres->attributes,
			     &dest_pres->attributes);

   if (new_segment)
      _kdms_update_segment(dest_object, dest_pres);

   return TRUE;
}

/*-----------------------------------------------------------
|
|  Routine Name: kdms_get_stored_attribute_names
|
|       Purpose: Gets a list of stored attributes
|		 from an object or segment.
|		 THIS IS A PRIVATE FUNCTION THAT SHOULD NOT
|		 BE USED.  THIS FUNCTIONALITY WILL BE ROLLLED
|		 INTO A MORE GENERAL MECHANISM IN THE NEAR 
|		 FUTURE.
|         Input:
|        Output:
|       Returns:
|    Written By: Jeremy Worley
|          Date: Aug 29, 1994 08:45
| Modifications:
|
------------------------------------------------------------*/
char **
kdms_get_stored_attribute_names(
   kobject obj,
   char *segment,
   int *number)
{
   char **attributes = NULL;
   int nattributes = 0;
   kpresentation *pres;
   int i;
   char *a;
   
   if (segment == NULL)
   {
      for (i = 0; i < obj->phys->attributes->nattributes; i++)
      {
	 if (obj->phys->attributes->attributes[i]->generic.permanent)
	 {
	    a = ktoken_to_string(
		     obj->phys->attributes->attributes[i]->base.attribute);

	    attributes = karray_add(attributes, kstrdup(a), nattributes);
	    nattributes++;
	 }
      }
   }
   else if (_kdms_get_segment_info(obj, kstring_to_token(segment), &pres) &&
	    (pres->segment->attributes != NULL))
   {
      for (i = 0; i < pres->segment->attributes->nattributes; i++)
      {
	 if (pres->segment->attributes->attributes[i]->generic.permanent)
	 {
	    a = ktoken_to_string(
	       pres->segment->attributes->attributes[i]->base.attribute);
	    
	    attributes = karray_add(attributes, kstrdup(a), nattributes);
	    nattributes++;
	 }
      }
   }
   
   *number = nattributes;
   
   return attributes;
}
