/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";

#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Khoros Data Service Routines
   >>>> These are routines that correspond to the public
   >>>> API.
   >>>>
   >>>>   Static:
   >>>>             exit_data_services()
   >>>>             initialize_data_services()
   >>>>             object_initialize()
   >>>>  Private:
   >>>>             _kdms_create()
   >>>>             _kdms_open()
   >>>>             _kdms_close()
   >>>>             _kdms_duplicate()
   >>>>             _kdms_copy_data()
   >>>>             _kdms_free()
   >>>>             _kdms_locate()
   >>>>
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"

extern int _kdms_compiled;

klist *kdms_object_list = NULL;

/*-----------------------------------------------------------
|
|  Routine Name: (static) object_initialize -
|
|       Purpose:
|
|         Input: routines - data services routines associated with the
|                           object.
|
|        Output:
|       Returns: new data object on success, NULL otherwise
|
|    Written By:
|          Date:
| Modifications:
|
------------------------------------------------------------*/

static kobject
object_initialize(DataServiceInformation * routines, int phys)
{
   kobject object;

   /*
    * Malloc the object structure.
    */
   if ((object = (kobject) kcalloc(1, sizeof(struct _kobject))) == NULL)
   {
      return (NULL);
   }
   if (phys)
   {
      if ((object->phys = (kphysobj *) kcalloc(1, sizeof(kphysobj))) == NULL)
      {
         return (NULL);
      }
      object->phys->routines = routines;
      object->phys->filename = NULL;
   }

   object->type = KOBJ_DATA;

   kdms_object_list = klist_add(kdms_object_list, (kaddr) object, NULL);
   return (object);
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _kdms_free - frees all memory associated with
|                               an object
|
|       Purpose: This function frees all associated memory with
|                the current instantiation of the data object and
|                it's private resources.
|
|         Input: object - the file to be freed which was opened earlier
|                         with kdms_open().
|
|        Output: Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: Aug 17, 1992 10:11
| Modifications:
|
------------------------------------------------------------*/

static int
_kdms_free(kobject object)
{
   if (!object->phys->list)
   {
      /*
       * remove all of the callbacks associated with the object.
       */
      _kdms_remove_all_callbacks(object);

      _kdms_glue_destroy(object);

      kfree(object->phys->filename);
      kdms_free_object_attributes(object);
      kfree(object->phys);
   }

   /*
    * this thing is smart enough not to delete the physical layers 
    * of the segments unless they are the last ones around.
    */
   _kdms_destroy_segments(object);

   /*
    * delete the object from our object list which is used by
    * exit_data_services to make sure that all objects are closed.
    */
   kdms_object_list = klist_delete(kdms_object_list, (kaddr) object);

   /*
    * Free the object
    */
   object->type = KNONE;
   kfree(object);

   return (TRUE);
}


/*-----------------------------------------------------------
|
|  Routine Name: _kdms_create - create an abstract data object
|
|       Purpose: This function is used to instantiate an abstract
|                data object (kobject).  For more details, see
|                the public routine kdms_create.
|
|         Input: none
|
|        Output: none
|       Returns: kobject on success, NULL upon failure
|
|    Written By: Mark Young and Jeremy Worley
|          Date: Jul 13, 1992 15:55
|      Verified:
|  Side Effects:
| Modifications:
|
------------------------------------------------------------*/

kobject
_kdms_create(void)
{
   kobject object;
   DataServiceInformation *routines;

   /*
    * initialize data services if necessary
    */
   if (_kdms_compiled == FALSE)
      _kdms_init();

   /*
    * According to the specified type of object get the appropriate data
    * service routines.
    */
   routines = _kdms_glue_routines(NULL);

   if (routines == NULL)
      return (NULL);

   object = object_initialize(routines, TRUE);

   /*
    * this sanity check seems odd, for robustness, lets do it anyway.
    * Also, if there is a problem make sure to delete the object from
    * our object list. 
    */
   if (object == NULL || object->type != KOBJ_DATA)
   {
      kdms_object_list = klist_delete(kdms_object_list, (kaddr) object);
      return (NULL);
   }

   /*
    * set some misc. fields in the data object
    */
   object->phys->filename = NULL;
   object->phys->temporary = TRUE;

   /*
    * very important to make sure that we add the reference to the list
    */
   object->phys->list = klist_add(object->phys->list, object, NULL);

   return (object);
}

/*-----------------------------------------------------------
|
|  Routine Name: _kdms_open - open or create an object for
|                               reading and/or writing
|
|       Purpose: This function is used to
|                open an abstract data object of unknown
|                format.  See kdms_open for more details.
|
|         Input: name - is the string containing the file path name
|                       to the desired object to be open.
|
|                flags - how the object is to be opened.
|
|        Output: none
|       Returns: kobject structure on success, NULL upon failure
|
|    Written By: Mark Young and Jeremy Worley
|          Date: Jul 13, 1992 15:55
|      Verified:
|  Side Effects:
| Modifications:
|
------------------------------------------------------------*/

kobject
_kdms_open(char *name, int flags)
{
   kobject object;
   DataServiceInformation *routines = NULL;

   /*
    * initialize data services if necessary
    */
   if (_kdms_compiled == FALSE)
      _kdms_init();

   if ((KOBJ_WRITE & flags) && (KOBJ_READ & flags))
   {
      _kdms_set_error(KDMS_EREADWRITE_LIMITATION);
      return (NULL);
   }

   /*
    * According to the specified type of object get the appropriate data
    * service routines.  These two function calls assume that the file has
    * already been opened.
    */
   if ((KOBJ_WRITE & flags) && !(KOBJ_READ & flags))
   {
      routines = _kdms_glue_routines(NULL);

      if (routines == NULL)
      {
	 kinfo(KSTANDARD,"The default file format does not exist.  This "
		"could be caused by a bogus setting of KHOROS_KDMS_FORMAT.\n");
	 return (NULL);
      }
      
   }

   /*
    * initialize the object with the format service
    */
   object = object_initialize(routines, TRUE);

   if (object == NULL || object->type != KOBJ_DATA)
      return (NULL);

   /*
    * set some misc. fields in the data object
    */
   object->phys->filename = kstring_copy(name, NULL);
   object->phys->flags = flags;


   if ((KOBJ_READ & flags) && !_kdms_glue_input(object, name, flags))
   {
      _kdms_free(object);
      return (NULL);
   }

   if (!kstrcmp(object->phys->routines->identifier, KRAW))
      object->phys->headerless = TRUE;
   else
      object->phys->headerless = FALSE;

   /*
    * Make sure that the protected flag gets set.  This is necessary
    * to prevent writing to an output file.  Also, this must be set
    * after the call to routines->input so that the file pointer gets
    * set correctly. 
    */
   if (KOBJ_READ & flags)
   {
      if (!object->phys->headerless)
         object->phys->locked = TRUE;
      object->phys->protected = TRUE;
   }

   /*
    * very important to make sure that we add the reference to the list
    */
   object->phys->list = klist_add(object->phys->list, object, NULL);

   return (object);
}

/*-----------------------------------------------------------
|
|  Routine Name: _kdms_close - close an abstract data object
|
|       Purpose: _kdms_close is part of the Data Management
|                Services layer of the Distributed Data
|                Services Toolkit.  This function closes
|                an abstract data object, which will free
|                all of the resources associated with the
|                object.
|
|                If the object is a duplicate of another
|                object, or shares resources with another
|                object, then if the other object is still
|                open, this kdms_close will simply
|                detach those shared resources from the
|                object rather than freeing them.
|
|         Input: object - the object to be closed.
|
|        Output: none
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|  Restrictions:
|    Written By: Mark Young and Jeremy Worley
|          Date: Jul 13, 1992 15:59
|      Verified:
|  Side Effects:
| Modifications:
|
------------------------------------------------------------*/

int
_kdms_close(kobject object, int exit_status)
{
   int status = TRUE;

   /*
    * If object is NULL or the object routines are NULL then return.
    */
   if (object == NULL)
   {
      _kdms_set_error(KDMS_EOBJ_INVALID);
      return (FALSE);
   }

   if (object->type != KOBJ_DATA)
   {
      _kdms_set_error(KDMS_EOBJ_INVALID);
      kinfo(KSTANDARD, "Deleting object from list anyway.");
      kdms_object_list = klist_delete(kdms_object_list, (kaddr) object);
      return (FALSE);
   }

   /*
    * fire off any save callbacks for the object.
    */
   if (object->phys->num_callbacks != 0)
      _kdms_process_object_callbacks(object, KDMS_CALLBACK_SAVE);
   
   /*
    * don't write the silly thing if the exit status is fail.
    */
   if (exit_status == KEXIT_FAILURE)
   {
      _kdms_free(object);
      return (TRUE);
   }

   /*
    * if there are reference objects, don't do any writing
    */
   object->phys->list = klist_delete(object->phys->list, object);
   if (object->phys->list == NULL)
   {
      /*
       * if the routines do not exist, exit.
       */
      if (object->phys->routines == NULL)
      {
         _kdms_set_error(KDMS_EINTERNAL);
         return (FALSE);
      }

      if (object->phys->filename != NULL)
      {                         /* a NULL filename is a temp obj */

         /*
          * call the close hook if it exists. if the call hook fails,
          * then fail too.
          */
         if (object->phys->close_hook != NULL)
            if (!(object->phys->close_hook(object)))
            {
               _kdms_free(object);
               return (FALSE);
            }

         /*
          * flush all of the data segments prior to closing the object.
          */
         if (object->phys->flags & KOBJ_WRITE && object->phys->resources == NULL)
            _kdms_lock_segments(object);

         if (object->phys->flags & KOBJ_WRITE)
            _kdms_flush_segments(object);

         /*
          * if the object->phys->resources is *still* null, then that
          * means that there
          * were no segments to write, and thus the header wasn't
          * written.  Really, we oughta check with the glue to make sure
          * that the format can support
          * a data set with no data...but that'll have to come later.
          */
         if (object->phys->resources == NULL)
	 {
	    _kdms_glue_lock(object, KLOCK_EX);
            _kdms_glue_output(object, object->phys->filename,
                              object->phys->flags);
	 }
	 
         /*
          * close the object data transport
          */
         if (object->phys->temporary && !object->phys->protected &&
             (object->phys->flags & KOBJ_WRITE) &&
             !(object->phys->flags & KOBJ_STREAM))
            _kdms_merge_temps(object);

         if (object->phys->resources == NULL)
	    _kdms_glue_lock(object, KLOCK_UN);
	 
      }
   }

   /*
    * free up the object.  _kdms_free is smart enough not to free the
    * physical layer if it is inappropriate to do so.
    */
   _kdms_free(object);

   return (status);
}

/*-----------------------------------------------------------
|
|  Routine Name: _kdms_lock
|
|       Purpose: This function is used to lock the physical
|                layer of the object.
|
|         Input: object - object to be locked.
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Jun 01, 1993 13:10
| Modifications:
|
------------------------------------------------------------*/

int
_kdms_lock(kobject object)
{
   object->phys->locked = TRUE;
   return (_kdms_lock_segments(object));
}

/*-----------------------------------------------------------
|
|  Routine Name: _kdms_reference - create a reference of an object.
|
|       Purpose: _kdms_reference is part of the Data Management
|                Services layer of the Distributed Data
|                Services Toolkit.
|
|         Input: object - the abstract data object to be
|                         reference.
|
|        Output: none
|       Returns: a kobject that is a duplicate of the input
|                object on success, NULL upon failure
|
|  Restrictions: In the case of a non-permanent transport
|                such as a stream, Data Services is likely
|                to actually duplicate the resources in order
|                to prevent complications with managing
|                the transport.
|
|    Written By: Mark Young and Jeremy Worley
|          Date: Aug 20, 1992 15:51
|      Verified:
|  Side Effects:
| Modifications:
|
------------------------------------------------------------*/

kobject
_kdms_reference(kobject object)
{
   kobject reference;
   int i;

   /*
    * If object is NULL or the object routines are NULL then return.
    */
   if (object == NULL || object->type != KOBJ_DATA)
   {
      _kdms_set_error(KDMS_EOBJ_INVALID);
      return (NULL);
   }

   reference = object_initialize(NULL, FALSE);

   if (reference == NULL)
      return (NULL);

   /*
    * Initialize the general information such as the name, kfile structure,
    * id, etc.
    */
   *reference = (*object);

   /* copy all of the user defined attributes */
   reference->attributes = NULL;
   if (object->attributes != NULL)
      for (i = 0; i < object->attributes->nattributes; i++)
         kdms_establish_attribute(&(reference->attributes),
                                  object->attributes->attributes[i]);
   /*
    * very important to make sure that we add the reference to the list
    */
   object->phys->list = klist_add(object->phys->list, reference, NULL);

   reference->presentations = NULL;

   if (!_kdms_reference_segments(object, reference))
   {
      kdms_close(reference);
      return (NULL);
   }
   return (reference);
}

/*-----------------------------------------------------------
|
|  Routine Name: _kdms_locate - locate an object
|
|       Purpose: locate an object given a filename
|         Input: filename - the filename
|        Output:
|       Returns: kobject structure on success, NULL upon failure
|
|    Written By: Mark Young and Jeremy Worley
|          Date: May 19, 1993 13:47
| Modifications:
|
------------------------------------------------------------*/

kobject
_kdms_locate(char *filename)
{
   kobject object;
   klist *list = klist_head(kdms_object_list);
   char temp[KLENGTH],
      path[KLENGTH];

   /*
    * race thru the list searching for the filename that matches
    */
   kfullpath(filename, NULL, path);
   while (list != NULL)
   {
      object = (kobject) klist_identifier(list);
      if (object->phys->filename != NULL &&
          kstrcmp(path, kfullpath(object->phys->filename, NULL, temp)) == 0)
      {
         return (object);
      }
      list = klist_next(list);
   }
   return (NULL);               /* not an error...just no match found */
}


/*-----------------------------------------------------------
|
|  Routine Name: _kdms_sync_segment
|
|       Purpose: 
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Aug 24, 1993 09:04
| Modifications:
|
------------------------------------------------------------*/

static int
_kdms_sync_segment(kobject object, int segment, int direction)
{
   kpresentation *pres = NULL;
   int i;

   if (!_kdms_get_segment_info(object, segment, &pres) || !pres)
      return (FALSE);

   if (pres->coupling == KUNCOUPLED)
   {
      _kdms_set_error(KDMS_ESYNC_NOT);
      return (FALSE);
   }

   if (direction == KPRES2PHYS)
   {
      return (_kdms_update_segment(object, pres));
   }
   else if (direction == KPHYS2PRES)
   {
      if (pres->segment->update)
         _kdms_update_segment(object, pres);

      pres->datatype = pres->segment->datatype;
      for (i = 0; i < pres->segment->dimension; i++)
      {
	 pres->size[i] = pres->segment->size[i];
	 pres->order[i] = pres->segment->order[i];
      }
      
      return (TRUE);
   }

   _kdms_set_error(KDMS_ESYNC_INVALID);
   return (FALSE);
}

/*-----------------------------------------------------------
|
|  Routine Name: _kdms_sync_segments
|
|       Purpose: 
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Aug 24, 1993 09:08
| Modifications:
|
------------------------------------------------------------*/

static int
_kdms_sync_segments(kobject object, int direction)
{
   klist *list;

   list = object->presentations;
   while (list != NULL)
   {
      _kdms_sync_segment(object, klist_token(list), direction);
      list = klist_next(list);
   }

   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: _kdms_sync
|
|       Purpose: 
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Aug 24, 1993 09:01
| Modifications:
|
------------------------------------------------------------*/

int 
_kdms_sync(kobject object, char *segment, int direction)
{
   if (segment == NULL)
      return (_kdms_sync_segments(object, direction));
   else
      return (_kdms_sync_segment(object, kstring_to_token(segment), direction));
}

/*-----------------------------------------------------------
|
|  Routine Name: _kdms_update_references
|
|       Purpose: 
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Oct 26, 1993 14:11
| Modifications:
|
------------------------------------------------------------*/

int
_kdms_update_references(kobject object, char *segment)
{
   klist *list;
   kpresentation *template,
     *pres;
   int tok = kstring_to_token(segment);

   /*
    * get the "template" presentation
    */
   if (!_kdms_get_segment_info(object, tok, &template) || !template)
      return (FALSE);

   /*
    * sweep through all reference objects and update each of them
    */
   for (list = klist_head(object->phys->list); list != NULL;
        list = klist_next(list))
      if ((kobject) list->identifier != object)
      {
         if (!_kdms_get_segment_info(list->identifier, tok, &pres) || !pres)
            _kdms_create_segment((kobject) list->identifier, tok, template);
         else
            _kdms_copy_pres_struct(template, pres);
      }

   /*
    * all done
    */
   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: _kdms_reopen
|
|       Purpose: 
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Oct 26, 1993 14:11
| Modifications:
|
------------------------------------------------------------*/

kobject
_kdms_reopen(kobject object, char *name, int flags)
{
   if (object == NULL || object->type != KOBJ_DATA || !(flags & KOBJ_READ))
      return (NULL);

   /*
    * clean up the old object
    */
   _kdms_glue_destroy(object);
   object->phys->resources = NULL;
   object->phys->routines = NULL;

   kfree(object->phys->filename);

   /*
    * set some misc. fields in the data object
    */
   object->phys->filename = kstring_copy(name, NULL);
   object->phys->flags = flags;


   /*    
    *  should really be a _kdms_flush_data(  in data.c or transport.c.);
    * 
    *   need to NULL out the actual data and begin and end
    */
   
   /* 
    * quick hack  zero out the data and the begin and end by hand..  - Steve K
    */
   {
     kpresentation *pres = NULL;
     klist *list;
     int i;

     list = object->presentations;
     /* go through every presentation */
     while (list != NULL)
     {
       /* get the presentation */
       if (_kdms_get_segment_info(object, klist_token(list),
				  &pres) && pres && !pres->segment->temporary)
       {
	 kfree(pres->segment->data);

	 for (i = 0; i < 5; i++)
	 {
	   pres->segment->begin[i] = 0;
	   pres->segment->end[i] = 0;
	 }
       }
       list = klist_next(list);
     }
   }
	   

   if ((KOBJ_READ & flags) && !_kdms_glue_input(object, name, flags))
   {
      _kdms_free(object);
      return (NULL);
   }

   if (!kstrcmp(object->phys->routines->identifier, KRAW))
      object->phys->headerless = TRUE;
   else
      object->phys->headerless = FALSE;

   /*
    * Make sure that the protected flag gets set.  This is necessary
    * to prevent writing to an output file.  Also, this must be set
    * after the call to routines->input so that the file pointer gets
    * set correctly. 
    */
   if (KOBJ_READ & flags)
   {
      if (!object->phys->headerless)
         object->phys->locked = TRUE;
      object->phys->protected = TRUE;
   }

   /*
    * very important to make sure that we add the reference to the list
    */
   object->phys->list = klist_add(object->phys->list, object, NULL);

   return (object);
}
