 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";

#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Khoros ASCII Data Service Routines
   >>>>
   >>>>   Static:
   >>>>  Private:
   >>>>             ascii_check()
   >>>>             ascii_input()
   >>>>             ascii_output()
   >>>>             _init()
   >>>>             ascii_destroy()
   >>>>   Public:
   >>>>             None - no public should ever exist as these are
   >>>>                    internal routines only accessible via the
   >>>>                    DataServiceInformation *services[] structure.
   >>>>
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"


#if !defined(KASCII_DEF)
DataServiceInformation ascii_format[] =
{NULL};

#else

#include "kdatafmt/ascii.h"


static int ascii_check PROTO((int));
static int ascii_input PROTO((kobject, int, int));
static int ascii_output PROTO((kobject, int, int));
static int ascii_destroy PROTO((kobject));
static int ascii_order PROTO((char *, int *));

DataServiceInformation ascii_format[] =
{
   {
      "Ascii File Format (ascii)",
      "ascii",
      ascii_check,
      ascii_input,
      ascii_output,
      ascii_destroy,
      NULL,
      NULL,
      ascii_order,
      NULL,
      NULL,
      NULL,
   }
};

/*
 *  Internal Resource Structure for the following data services
 *
 *              ascii  -  Sun Rasterfile File Format Utilities (ascii)
 */
typedef struct
{
   kfile *file;
   ascii *image;
}
ResourceStruct;


/*-----------------------------------------------------------
|
|  Routine Name: _init - Creates an ascii image
|
|       Purpose: This function is used to create an ascii image.  Which
|                means that we use the current dimension values in order
|                to create a proper ascii.
|
|         Input: data - initial sample of data
|                num  - number of bytes in the sample
|
|        Output: Returns: returns TRUE or FALSE if an error occurs
|
|    Written By: Mark Young
|          Date: Jul 13, 1992 15:50
| Modifications:
|
------------------------------------------------------------*/

static int
_init(kobject object)
{
   ResourceStruct *resources;


   /*
    * Create the image to the object file
    */
   resources = (ResourceStruct *) kcalloc(1, sizeof(ResourceStruct));

   if (resources == NULL)
      return (FALSE);

   _kdms_glue_set_resources(object, resources);
   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: ascii_check - Checks to see if the file is
|                                an ascii
|
|       Purpose: This function is used to check the first few bytes of
|                the stream to see if it is an ascii.
|
|         Input: object - the ascii object to be initialized
|
|        Output: Returns: returns TRUE or FALSE if an error occurs
|
|    Written By: Mark Young
|          Date: Jul 13, 1992 15:50
| Modifications:
|
------------------------------------------------------------*/

static int
ascii_check(int fid)
{
   ascii *image;

   /*
    * Return whether this is an ascii file or not
    */
   image = ascii_fdread(fid);

   if (image == NULL)
      return (FALSE);

   ascii_free(image);
   return (TRUE);
}



/*-----------------------------------------------------------
|
|  Routine Name: ascii_input - Read an ascii image
|
|       Purpose: This function is used to read in the image if the
|                supplied data is ascii image.  The data was checked
|                by the ascii_check() routine in which the ascii
|                identifier indicated that the data is a valid ascii.
|                Since the data was valid we can assume simply read
|                the header in and initialize that data segments.
|
|         Input: object - the ascii object to be initialized
|
|        Output: Returns: returns TRUE or FALSE if an error occurs
|
|    Written By: Mark Young
|          Date: Jul 13, 1992 15:50
| Modifications:
|
------------------------------------------------------------*/

/*ARGSUSED */
static int
ascii_input(kobject object, int fid, int flags)
{
   ResourceStruct *resources;
   ascii *image;

   int i,
      size[KDMS_MAX_DIM],
      order[KDMS_MAX_DIM],
      begin[KDMS_MAX_DIM],
      end[KDMS_MAX_DIM];

   if (!_kdms_initialized(object))
      _init(object);

   resources = (ResourceStruct *) _kdms_glue_get_resources(object);

   /*
    * Read the image from the object file and go ahead and set the ascii image
    * within the resource structure.
    */
   image = resources->image = ascii_fdread(fid);
   if (image == NULL)
   {
      _kdms_set_error(KDMS_EFMT_FAILURE);
      return (FALSE);
   }

   /*
    * Initialize the data segments according to the data found in the ascii
    * structure.
    * 
    * The ascii structure only contains value (image->data)
    */

   /* check the value data */
   if (image->width != 0 && image->height != 0)
   {
      size[0] = image->width;
      size[1] = image->height;
      size[2] = size[3] = size[4] = 1;

      order[0] = KWIDTH;
      order[1] = KHEIGHT;
      order[2] = KDEPTH;
      order[3] = KTIME;
      order[4] = KELEMENTS;

      for (i = 0; i < 5; i++)
      {
         end[i] = size[i] - 1;
         begin[i] = 0;
      }

      _kdms_set_segment(object, KDMS_SEGMENT_VALUE, image->data, image->type,
                        order, size, begin, end, 5);
   }
   return (TRUE);
}



/*-----------------------------------------------------------
|
|  Routine Name: ascii_output - Closes the ascii image
|
|       Purpose: This function is used to close the ascii image.  Which
|                means if the image was modified we then re-write image
|                back out.
|
|         Input: object - the ascii object to be closed
|
|        Output: Returns: returns TRUE or FALSE if an error occurs
|
|    Written By: Mark Young
|          Date: Jul 13, 1992 15:50
| Modifications:
|
------------------------------------------------------------*/

/*ARGSUSED */
static int
ascii_output(kobject object, int fid, int flags)
{
   ResourceStruct *resources;
   ascii *image;
   int i,
     *size,
      type,
      begin[KDMS_MAX_DIM],
      end[KDMS_MAX_DIM];
   int order[KDMS_MAX_DIM] =
   {KWIDTH, KHEIGHT, KDEPTH, KTIME, KELEMENTS};

   if (!_kdms_initialized(object))
      _init(object);

   resources = (ResourceStruct *) _kdms_glue_get_resources(object);

   if (resources->image == NULL)
   {
      resources->image = ascii_create();
      if (resources->image == NULL)
      {
         _kdms_traceback("ascii_output");
         _kdms_set_error(KDMS_EFMT_FAILURE);
         return (FALSE);
      }
   }
   image = resources->image;

   /* 
    * Force the size of the presentation to match the physical
    */
   if (!kdms_set_attribute(object, NULL, KDMS_COUPLING, KCOUPLED))
   {
      _kdms_traceback("ascii_output");
      _kdms_set_error(KDMS_EFMT_FAILURE);
      return (FALSE);
   }

   if (!kdms_sync(object, KDMS_SEGMENT_VALUE, KPHYS2PRES))
   {
      _kdms_traceback("ascii_output");
      _kdms_set_error(KDMS_EFMT_FAILURE);
      return (FALSE);
   }


   /* 
    * Get the size
    */
   if (!kdms_get_attributes(object, KDMS_SEGMENT_VALUE, KDMS_DATA_TYPE, &type,
			    NULL))
   {
      _kdms_traceback("ascii_output");
      _kdms_set_error(KDMS_EFMT_FAILURE);
      return (FALSE);
   }

   if (type == KCOMPLEX || type == KDCOMPLEX)
      type = KDCOMPLEX;
   else
      type = KDOUBLE;

   /*
    * Set the index order and data type to the ones we support 
    */
   if (!kdms_set_attributes(object, KDMS_SEGMENT_VALUE, KDMS_INDEX_ORDER, order,
                            KDMS_DATA_TYPE, type, NULL))
   {
      _kdms_traceback("ascii_output");
      _kdms_set_error(KDMS_EFMT_FAILURE);
      return (FALSE);
   }

   /* 
    * Get the size
    */
   if (!kdms_get_attributes(object, KDMS_SEGMENT_VALUE, KDMS_SIZE, &size, NULL))
   {
      _kdms_traceback("ascii_output");
      _kdms_set_error(KDMS_EFMT_FAILURE);
      return (FALSE);
   }

   /* 
    * Set up the begin and end arrays for retrieving the data 
    */
   begin[0] = begin[1] = 0;
   end[0] = size[0] - 1;
   end[1] = size[1] - 1;
   for (i = 2; i < KDMS_MAX_DIM; i++)
      end[i] = begin[i] = 0;

   /* 
    * Retrieve the data in the expected index order
    */
   if ((image->data = kdms_get_data(object, KDMS_SEGMENT_VALUE, begin, end,
                                     NULL)) == NULL)
   {
      _kdms_traceback("ascii_output");
      _kdms_set_error(KDMS_EFMT_FAILURE);
      return (FALSE);
   }

   /*
    * Set the type & size fields
    */
   image->type   = type;
   image->width  = (int)size[0];
   image->height = (int)size[1];

   /*
    * Write the ascii structure out to the specified filename
    */
   return(ascii_fdwrite(fid, image));
}

/*-----------------------------------------------------------
|
|  Routine Name: ascii_destroy - Frees an ascii image
|
|       Purpose: This function is used to create an ascii image.  Which
|                means that we use the current dimension values in order
|                to create a proper ascii.
|
|         Input: data - initial sample of data
|                num  - number of bytes in the sample
|
|        Output: Returns: returns TRUE or FALSE if an error occurs
|
|    Written By: Mark Young
|          Date: Jul 13, 1992 15:50
| Modifications:
|
------------------------------------------------------------*/

static int
ascii_destroy(kobject object)
{
   ResourceStruct *resources;

   resources = (ResourceStruct *) _kdms_glue_get_resources(object);

   /*
    * free the asciier structure
    */
   /* kludge */resources->image->data = NULL;
   ascii_free(resources->image);

   /*
    * free the resources
    */
   kfree(resources);

   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: ascii_order
|
|       Purpose:
|
|         Input:
|
|        Output:
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Mar 12, 1993 13:12
| Modifications:
|
------------------------------------------------------------*/

static int
ascii_order(char *seg, int *ord)
{
   if (kstring_to_token(seg) == kstring_to_token(KDMS_SEGMENT_MAP))
   {
      ord[0] = KMAP_HEIGHT;
      ord[1] = KMAP_WIDTH;
      ord[2] = KMAP_HEIGHT;
      ord[3] = KMAP_DEPTH;
      ord[4] = KMAP_ELEMENTS;
   }
   return (TRUE);
}

#endif /* KASCII_DEF */

/* don`t add after the endif */
