 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            X11 Window Dump File Format Utilities
   >>>>
   >>>>	    The "xwd" format describes the conversion used for
   >>>>	    the format:
   >>>>
   >>>>			X11 Window Dump File Format
   >>>>
   >>>>	    The format was orignially developed by the company/persons:
   >>>>
   >>>>			MIT Project Athena
   >>>>	    
   >>>>	    The following routines are used by the conversion routines
   >>>>	    to read and write the xwd files.  These routines are not a
   >>>>	    standard, just an a set of routines used by the Khoros routines
   >>>>	    to manipulate X11 Window Dump File Format.
   >>>>
   >>>>
   >>>>  Private:
   >>>>   Static:
   >>>>   Public:
   >>>>             xwd_readheader()
   >>>>             xwd_read()
   >>>>             xwd_writeheader()
   >>>>             xwd_write()
   >>>>             xwd_free()
   >>>>             xwd_create()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"
#include <kdatafmt/xwd.h>
#define  KDATAFMT "kdatafmt"


/************************************************************
*
*  Routine Name: xwd_readheader - reads a xwd header structure
*				      from the specified kfile id
*
*       Purpose: This routines reads a xwd header structure and it's
*		 data into the specified filename.  The routine
*		 uses the ktransport library which allows for
*		 remote reading of the xwd header.
*
*         Input: fid - the kfile id opened previously with kopen()
*
*        Output: none
*
*       Returns: image - explanation
*
*  Restrictions: 
*    Written By: Mark Young 
*          Date: Sep 30, 1992 17:53 
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

xwd *xwd_readheader(
   int fid)
{
	xwd *image;


	/* 
	 * Sanity check to make sure that the image pointer is not NULL
	 */
	if ((image = (xwd *) kcalloc(1, sizeof(xwd))) == NULL)
	{
	   return(NULL);
	}

	/*
	 *  Data is always stored in Sun format so we need to set the
	 *  descriptor machine type to be of type KMACH_SPARC.
	 */
	kfile_setmachtype(fid, KMACH_SPARC);

	kread_ulong(fid, &image->header_size, 22);
	if (image->file_version != XWD_FILE_VERSION)
	{
	   kfree(image);
	   return(NULL);
	}
	kread_long(fid, &image->x, 2);
	kread_ulong(fid, &image->bdrwidth, 1);

	/*
	 * jump over the window name
	 */
	klseek(fid, (int) image->header_size, SEEK_SET);

	return(image);
}

/************************************************************
*
*  Routine Name: xwd_fdread - read a xwd structure from the
*				specified file descriptor
*
*       Purpose: This routines reads a xwd structure and it's
*		 data from the specified file descriptor.  The routine
*		 uses the ktransport library which allows for
*		 remote reading of the data and it's header.
*
*         Input: filename - filename in which we will be reading
*			    the xwd
*
*        Output: none
*
*       Returns: returns the newly read xwd or NULL upon failure
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Sep 30, 1992 17:53
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

xwd *xwd_fdread(int fid)
{
   xwd  *image;
   int  i, size, type;

   /*
    *  Read the X11 Window Dump File Format header
    */
   if ((image = xwd_readheader(fid)) == NULL)
      return(FALSE);

   /*
    *  Now read the data associated with the
    *  X11 Window Dump File Format header
    */
   if (image->ncolors > 0)
   {
      image->maps = (xwdcolor *) kmalloc(image->ncolors*sizeof(xwdcolor));

      for (i = 0; i < image->ncolors; i++)
      {
	 kread_ulong(fid, &image->maps[i].pixel, 1);
	 kread_ushort(fid, &image->maps[i].red, 3);
	 kread_ubyte(fid, &image->maps[i].flags, 2);
      }
   }

   size = 0;

   if (image->bits_per_pixel == 1)
   {
      type = KBIT;
      size = image->pixmap_height * (image->pixmap_width + 7)/8;
   }
   else if (image->bits_per_pixel == 8)
   {
      type = KUBYTE;
      size = image->bytes_per_line * image->pixmap_height;
   }
   else if (image->bits_per_pixel == 16)
   {
      type = KUSHORT;
      size = image->bytes_per_line * image->pixmap_height;
   }
   else if (image->bits_per_pixel == 32)
   {
      type = KUINT;
      size = image->bytes_per_line * image->pixmap_height;
   }
   else
   {
      kfree(image->maps);
      kfree(image);
      return(NULL);
   }
   
   if (size > 0)
   {
      if ((image->data = kmalloc((unsigned) size)) == NULL)
      {
	 kfree(image->maps);
	 kfree(image);
	 return(NULL);
      }
      size /= (image->bits_per_pixel >> 3);
      kread_generic(fid, image->data, size, type);
   }
   return image;
}

/************************************************************
*
*  Routine Name: xwd_read - read a xwd structure from the
*				specified filename
*
*       Purpose: This routines reads a xwd structure and it's
*		 data from the specified filename.  The routine
*		 uses the ktransport library which allows for
*		 remote reading of the data and it's header.
*
*         Input: filename - filename in which we will be reading
*			    the xwd
*
*        Output: none
*
*       Returns: returns the newly read xwd or NULL upon failure
*
*  Restrictions:
*    Written By: Jeremy Worley
*          Date: Jan 05, 1995 13:54
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

xwd *xwd_read(
   char *filename)
{
   int fid;
   xwd *image;
 
   /* 
    * Open the input file 
    */
   if ((fid = kopen(filename, O_RDONLY, 0664)) < 0)
      return(FALSE);

   /*
    *  Lock the transport file descriptor if the transport has
    *  data permanence.
    */
   if (kfile_getpermanence(filename) == TRUE)
      kflock(fid, KLOCK_SH);

   image = xwd_fdread(fid);
   
   kclose(fid);

   return image;
}


/************************************************************
*
*  Routine Name: xwd_fdwrite - write a xwd structure to the
*				 specified file descriptor
*
*       Purpose: This routines writes a xwd structure and it's
*		 data into the specified file descriptor.  The routine
*		 uses the ktransport library which allows for
*		 remote writing of the data.
*
*         Input: fid  - the file descriptor in which we will be writing
*			    the xwd image and associated data
*                image    - the xwd structure to be written
*
*        Output: none
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions:
*    Written By: Mark Young 
*          Date: Sep 30, 1992 17:53 
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int xwd_fdwrite(int fid, xwd *image)
{
   int i;
   int type;
   int size;
   
   /*
    *  Write the X11 Window Dump File Format header
    */
   if (!xwd_writeheader(fid, image))
      return(FALSE);

   /*
    *  Now write the data associated with the
    *  X11 Window Dump File Format header
    */
   for (i = 0; i < image->ncolors; i++)
   {
      kwrite_ulong(fid, &image->maps[i].pixel, 1);
      kwrite_ushort(fid, &image->maps[i].red, 3);
      kwrite_ubyte(fid, &image->maps[i].flags, 2);
   }

   if (image->bits_per_pixel == 1)
      type = KBIT;
   else if (image->bits_per_pixel == 8)
      type = KUBYTE;
   else if (image->bits_per_pixel == 16)
      type = KUSHORT;
   else if (image->bits_per_pixel == 32)
      type = KUINT;
   else
      return(FALSE);

   size =  image->pixmap_height * image->pixmap_width;
   kwrite_generic(fid, image->data, size, type);

   return TRUE;
}

/************************************************************
*
*  Routine Name: xwd_write - write a xwd structure to the
*				 specified filename
*
*       Purpose: This routines writes a xwd structure and it's
*		 data into the specified filename.  The routine
*		 uses the ktransport library which allows for
*		 remote writing of the data.
*
*         Input: filename - the filename in which we will be writing
*			    the xwd image and associated data
*                image    - the xwd structure to be written
*
*        Output: none
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions:
*    Written By: Mark Young 
*          Date: Sep 30, 1992 17:53 
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int xwd_write(
   char *filename,
   xwd  *image)
{
   int fid;
   int status;
   
   /* 
    * Sanity check to make sure that the image pointer is not NULL
    */
   if (image == NULL)
   {
      errno = KNULL_PARAMETER;
      return(FALSE);
   }

   /* 
    * Open the output file 
    */
   if ((fid = kopen(filename, O_WRONLY|O_TRUNC|O_CREAT, 0664)) < 0)
      return(FALSE);

   /*
    *  Lock the transport file descriptor if the transport has
    *  data permanence.
    */
   if (kfile_getpermanence(filename) == TRUE)
      kflock(fid, KLOCK_EX);

   status = xwd_fdwrite(fid, image);
   
   kclose(fid);

   return status;
}


/************************************************************
*
*  Routine Name: xwd_writeheader - writes a xwd header structure
*				      from the specified kfile id
*
*       Purpose: This routines writes a xwd header structure and it's
*		 data into the specified filename.  The routine
*		 uses the ktransport library which allows for
*		 remote reading of the xwd header.
*
*         Input: fid - the kfile id opened previously with kopen()
*		 image - the image header to be written
*
*        Output: none
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Mark Young 
*          Date: Sep 30, 1992 17:53 
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int xwd_writeheader(
   int fid,
   xwd *image)
{

	/*
	 *  Write the (xwd) header out
	 */

	/*
	 *  Data is always stored in Sun format so we need to set the
	 *  descriptor machine type to be of type KMACH_SPARC.
	 */
	kfile_setmachtype(fid, KMACH_SPARC);

	kwrite_ulong(fid, &image->header_size, 22);
	kwrite_long(fid, &image->x, 2);
	kwrite_ulong(fid, &image->bdrwidth, 1);
	return(TRUE);
}


/************************************************************
*
*  Routine Name: xwd_free - frees a xwd structure and it's
*				associated data
*
*       Purpose: This routine frees an khoros xwd structure and
*		 it's associated data.
*
*         Input: image - a pointer to an khoros xwd structure that
*			 contains the image structure to be freed.
*
*        Output: none
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Mark Young 
*          Date: Sep 30, 1992 17:53
*      Verified:
*  Side Effects: Once this routine is called no further reference
*		 the image should be made.
* Modifications:
*
*************************************************************/

int xwd_free(
   xwd *image)
{
	/*
	 *  Free the data and all associated resources associated with
	 *  the image.
	 */
	kfree(image->maps);
	kfree(image->data);

	/*
	 *  Free the xwd image structure itself, which was previously
	 */
	kfree(image);
	return(TRUE);
}


/************************************************************
*
*  Routine Name: xwd_create - creates a xwd structure and it's
*				associated data
*
*       Purpose: This routine creates an khoros xwd structure and
*		 it's associated data.
*
*         Input: image - a pointer to an khoros xwd structure that
*			 contains the image structure to be freed.
*
*        Output: none
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Mark Young 
*          Date: Sep 30, 1992 17:53
*      Verified:
*  Side Effects: 
* Modifications:
*
*************************************************************/

xwd *xwd_create(void)
{
	xwd *image;

	/* 
	 * Sanity check to make sure that the image pointer is not NULL
	 */
	if ((image = (xwd *) kcalloc(1, sizeof(xwd))) == NULL)
	   return(NULL);

	image->file_version = XWD_FILE_VERSION;
	image->header_size = 100;

	return(image);
}
