 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            X11 Pixmap Format Utilities
   >>>>
   >>>>	    The "xpm" format describes the conversion used for
   >>>>	    the format:
   >>>>
   >>>>			X11 Pixmap Format
   >>>>
   >>>>	    The format was orignially developed by the company/persons:
   >>>>
   >>>>			GROUPE BULL
   >>>>	    
   >>>>	    The following routines are used by the conversion routines
   >>>>	    to read and write the xpm files.  These routines are not a
   >>>>	    standard, just an a set of routines used by the Khoros routines
   >>>>	    to manipulate X11 Pixmap Format.
   >>>>
   >>>>
   >>>>  Private:
   >>>>   Static:
   >>>>		    read_comment()
   >>>>		    read_rgbcolor()
   >>>>   Public:
   >>>>             xpm_readheader()
   >>>>             xpm_read()
   >>>>             xpm_writeheader()
   >>>>             xpm_write()
   >>>>             xpm_free()
   >>>>             xpm_create()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"
#include <kdatafmt/xpm.h>
#define  KDATAFMT "kdatafmt"

static char *xpm_printables =
" .XoO+@#$%&*=-;:>,<1234567890qwertyuipasdfghjklzxcvbnmMNBVCZ"
"ASDFGHJKLPIUYTREWQ!~^/()_`'][{}|";	

/*-----------------------------------------------------------
|
|  Routine Name: (static) _xpm_get_color
|       Purpose: Returns a colorstring.
|         Input:
|        Output:
|       Returns:
|    Written By: Jeremy Worley
|          Date: Aug 16, 1994 10:15
| Modifications:
|
------------------------------------------------------------*/

static char *_xpm_get_color(int idx, int nchars)
{
   int i;
   int l = kstrlen(xpm_printables);
   
   static char tmp[4];
   
   for (i = 0; i < nchars; i++, idx /= l)
      tmp[i] = xpm_printables[idx % l];
   
   return(tmp);
}

/*-----------------------------------------------------------
|
|  Routine Name: read_comment - reads the rest of the comment line
|
|       Purpose: This routine reads the rest of the comment line
|         Input: file - the file to be read.
|		 line - the rest of the line
|        Output: 
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: Dec 13, 1993
| Modifications:
|
------------------------------------------------------------*/

static int read_comment(
   kfile *file,
   char  *line)
{
	int  c;
	char *tmp = line;


	/*
	 *  Read to the end of the comment.
	 */
	while (*tmp != '\0')
	{
	   if (*tmp++ == '*')
	   {
	      if (*tmp == '/')
		 break;
	   }
	}

	if (*tmp == '\0')
	{
	   for (;;)
	   {
	      while ((c = kfgetc(file)) != '*' && c != EOF && c != '\0')
				;

	      if (c == '*')
	      {
		 if ((c = kfgetc(file)) == '/')
		 {
		    kfgets(line, KLENGTH, file);
		    break;
		 }
	      }
	      else break;
	   }
	}
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: read_rgbcolor - reads the color from the rgb.txt file
|
|       Purpose: This routine reads the rgb name from the rgb.txt file
|         Input: color - the desired color
|        Output: red   - returns the red value
|		 green - returns the green value
|		 blue  - returns the blue value
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: Dec 13, 1993
| Modifications:
|
------------------------------------------------------------*/

static int read_rgbcolor(
   char  *color,
   long  *rval,
   long  *gval,
   long  *bval)
{
	kfile *file;
	int   i, status, red, green, blue;
	char  temp[KLENGTH], *filename = "$BOOTSTRAP/repos/bitmaps/rgb.txt";

	static int  num = 0;
	static char **list = NULL;
	static int  initialized = FALSE;


	if (!initialized)
	{
	   initialized = TRUE;

	   file = kfinput(filename);
           if ((list = kparse_file_scan_delimit(file, KPARSE_BOF,
		KPARSE_EOF, KLITERAL, "\n", NULL, NULL, &num, &status)) == NULL)
           {
              kerror(NULL, "read_rgbcolor", "Cannot read contents of  RGB \
color file '%s' in order to convert xpm color name '%s' to RGB color values.",
			filename, temp);
              return(FALSE);
	   }
           kfclose(file);
        }

	for (i = 0; i < num; i++)
        {
	   if (ksscanf(list[i], "%d %d %d %[^$]",&red,&green,&blue,temp) == 4 &&
	       kstrcasecmp(color, temp) == 0)
	   {
	      *rval = red << 8;
	      *gval = green << 8;
	      *bval = blue << 8;
	      return(TRUE);
	   }
        }
	return(FALSE);
}


/************************************************************
*
*  Routine Name: xpm_readheader - reads a xpm header structure
*				      from the specified kfile id
*
*       Purpose: This routines reads a xpm header structure and it's
*		 data into the specified filename.  The routine
*		 uses the ktransport library which allows for
*		 remote reading of the xpm header.
*
*         Input: fid - the kfile id opened previously with kopen()
*        Output: none
*       Returns: image - explanation
*
*  Restrictions: 
*    Written By: Mark Young 
*          Date: Oct 07, 1993 13:50 
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

xpm *xpm_readheader(
   int	fid)
{
	xpm   *image;

	kfile *file;
	int   w, h, n, i, done = FALSE;
	char  line[KLENGTH], temp[KLENGTH], character[2];


	/* 
	 * Sanity check to make sure that the image pointer is not NULL
	 */
	if ((image = (xpm *) kcalloc(1, sizeof(xpm))) == NULL)
	{
	   kerror(KDATAFMT, "xpm_read",
		"Unable to allocate %d bytes for the xpm \
(X11 Pixmap Format) header", sizeof(xpm));
	   return(NULL);
	}
	file = kfdopen(fid, "r");

	kfgets(line, KLENGTH, file);
	kstring_cleanup(line, line);
	if (kstrcmp(line, "/* XPM */") != 0)
	{
	   xpm_free(image);
	   return(NULL);
	}

	while (!done)
	{
	    if (!kfgets(line, KLENGTH, file))
	    {
	       xpm_free(image);
	       return(NULL);
	    }

	    if (ksscanf(line, "static char *%[^_]", temp) == 1)
	    {
	       image->name = kstrdup(temp);
	    }
	    else if (ksscanf(line," \" %d %d %d %d \",", &w, &h, &n, &i) == 4)
	    {
	       image->width   = w;
	       image->height  = h;
	       image->ncolors = n;
	       image->size    = i;
	       image->data = (char *) kmalloc((unsigned) w * h);
	       image->maps = (xpmcolor *) kcalloc((unsigned) n*3,
					sizeof(xpmcolor));
	       done = TRUE;
	    }
	    else if (ksscanf(line, " %[/]%[*]", character, character) == 2)
	    {
	       read_comment(file, line);
	    }
	    else if (kstring_cleanup(line, line) != NULL)
	    {
	       kfree(image);
	       return(NULL);
	    }
	}
	return(image);
}


/************************************************************
*
*  Routine Name: xpm_fdread - read a xpm structure from the
*				specified file descriptor
*
*       Purpose: This routines reads a xpm structure and it's
*		 data from the specified file descriptor.  The routine
*		 uses the ktransport library which allows for
*		 remote reading of the data and it's header.
*
*         Input: fid - file descriptor in which we will be reading
*			    the xpm
*
*        Output: none
*
*       Returns: returns the newly read xpm or NULL upon failure
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Oct 07, 1993 13:50
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

xpm *xpm_fdread(int fid)
{
   xpm   *image;
   kfile *file;
   long  rval, gval, bval;
   int   i, j, k, size, mult = 0;
   char  line[5*KLENGTH], temp[5*KLENGTH], name[KLENGTH], character[2],
      red[10], green[10], blue[10], *data, *tmp;

   /*
    *  Read the X11 Pixmap Format header
    */
   if ((image = xpm_readheader(fid)) == NULL)
      return(FALSE);

   /*
    *  Now read the data associated with the
    *  X11 Pixmap Format header
    */
   file = kfdopen(fid, "r");

   i = 0;
   while (i < image->ncolors)
   {
      if (!kfgets(line, sizeof(line), file))
      {
	 xpm_free(image);
	 return(NULL);
      }

      if ((tmp = kstrchr(line, '"')) != NULL)
      {
	 image->maps[i].val = kstring_ncopy(++tmp, image->size, NULL);
	 data = tmp+image->size;
	 if ((tmp = kstrstr(data, " c ")) != NULL ||
	     (tmp = kstrstr(data, "\tc ")) != NULL)
	 {
	    kstring_cleanup(tmp+3, tmp);
	    if ((data = kstrstr(tmp, " m ")) != NULL ||
		(data = kstrstr(tmp, "\tm ")) != NULL)
	       *data = '\0';
	    if ((data = kstrstr(tmp, " s ")) != NULL ||
		(data = kstrstr(tmp, "\ts ")) != NULL)
	       *data = '\0';
	    
	    if (ksscanf(tmp, "#%[0-9A-Fa-f]", temp) == 1 &&
		((size = kstrlen(temp)) == 12 || size == 6 || size  == 3))
	    {
	       if (size == 12)
		  mult = 1;
	       else if (size == 6)
		  mult = 256;
	       else if (size == 3)
		  mult = 4096;
	       
	       size /= 3;
	       kstring_ncopy(temp, size, red);
	       kstring_ncopy(&temp[size],   size, green);
	       kstring_ncopy(&temp[size*2], size, blue);
	       rval = strtol(red, NULL, 16)   * mult;
	       gval = strtol(green, NULL, 16) * mult;
	       bval = strtol(blue, NULL, 16)  * mult;
	    }
	    else if (ksscanf(tmp, "%[a-zA-Z0-9 ]", name) == 1)
	    {
	       kstring_cleanup(name, name);
	       if (kstrcasecmp(name, "None") == 0 ||
		   !read_rgbcolor(name, &rval, &gval, &bval))
	       {
		  if (kstrcasecmp(name, "None") != 0)
		     kprintf("Unknown color format... '%s'\n", name);
		  
		  image->maps[i++].flags = 0xff;
		  continue;
	       }
	    }
	    else
	    {
	       kprintf("Could not parse color format... '%s'\n", tmp);
	       image->maps[i++].flags = 0xff;
	       continue;
	    }
	 }
	 image->maps[i].red   = (unsigned short)rval;
	 image->maps[i].green = (unsigned short)gval;
	 image->maps[i++].blue  = (unsigned short)bval;
      }
      else if (ksscanf(line, " %[/]%[*]", character, character) == 2)
      {
	 read_comment(file, line);
      }
      else if (kstring_cleanup(line, line) != NULL)
      {
	 kfree(image);
	 xpm_free(image);
	 return(NULL);
      }
   }
   
   i = 0; data = image->data;
   while (i < image->height)
   {
      if (!kfgets(line, sizeof(line), file))
      {
	 xpm_free(image);
	 return(NULL);
      }
      else if (ksscanf(line, " \"%[^\"]", temp) == 1)
      {
	 for (j = 0, tmp = temp; j < image->width; j++)
	 {
	    for (k = 0; k < image->ncolors; k++)
	    {
	       if (kstrncmp(tmp, image->maps[k].val, image->size) == 0)
	       {
		  *data++ = k;
		  break;
	       }
	    }
	    tmp += image->size;
	    if (k == image->ncolors)
	       *data++ = 0;
	 }
	 i++;
      }
      else if (ksscanf(line, " %[/]%[*]", character, character) == 2)
      {
	 read_comment(file, line);
      }
      else if (kstring_cleanup(line, line) != NULL)
      {
	 xpm_free(image);
	 return(NULL);
      }
   }

   return(image);
}

/************************************************************
*
*  Routine Name: xpm_read - read a xpm structure from the
*				specified filename
*
*       Purpose: This routines reads a xpm structure and it's
*		 data from the specified filename.  The routine
*		 uses the ktransport library which allows for
*		 remote reading of the data and it's header.
*
*         Input: filename - filename in which we will be reading
*			    the xpm
*
*        Output: none
*
*       Returns: returns the newly read xpm or NULL upon failure
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Oct 07, 1993 13:50
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

xpm *xpm_read(
   char	*filename)
{
   int   fid;
   xpm *image;
   
   /* 
    * Open the input file 
    */
   if ((fid = kopen(filename, O_RDONLY, 0664)) < 0)
   {
      kerror(KDATAFMT, "xpm_read",
	     "unable to access %s for reading", filename);
      return(FALSE);
   }
   
   /*
    *  Lock the transport file descriptor if the transport has
    *  data permanence.
    */
   if (kfile_getpermanence(filename) == TRUE)
      kflock(fid, KLOCK_SH);
   
   image = xpm_fdread(fid);
   
   kclose(fid);
   
   return image;
}

/************************************************************
*
*  Routine Name: xpm_fdwrite - write a xpm structure to the
*				 specified file descriptor
*
*       Purpose: This routines writes a xpm structure and it's
*		 data into the specified file descriptor.  The routine
*		 uses the ktransport library which allows for
*		 remote writing of the data.
*
*         Input: fid - the file descriptor in which we will be writing
*			    the xpm image and associated data
*                image    - the xpm structure to be written
*        Output: none
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions:
*    Written By: Mark Young 
*          Date: Oct 07, 1993 13:50 
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int xpm_fdwrite(int fid, xpm *image)
{
   int i;
   int j;
   kfile *f;
   int nchars = (int)kceil(klogn((double)image->ncolors,
				 (double)kstrlen(xpm_printables)));
   unsigned char *d = (unsigned char *)image->data;
   
   f = kfdopen(fid, "w");
	
   /*
    *  Read the X11 Pixmap Format header
    */
   if (!xpm_writeheader(fid, image))
      return(FALSE);

   for (i = 0; i < image->height; i++)
   {
      kfprintf(f, "\"");
      for (j = 0; j < image->width; j++)
	 kfprintf(f, "%s", _xpm_get_color((int)d[j + i * image->width], 
					  nchars));
      kfprintf(f, "\",\n");
   }

   kfprintf(f, "};\n");

   return(TRUE);
}

/************************************************************
*
*  Routine Name: xpm_write - write a xpm structure to the
*				 specified filename
*
*       Purpose: This routines writes a xpm structure and it's
*		 data into the specified filename.  The routine
*		 uses the ktransport library which allows for
*		 remote writing of the data.
*
*         Input: filename - the filename in which we will be writing
*			    the xpm image and associated data
*                image    - the xpm structure to be written
*        Output: none
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions:
*    Written By: Mark Young 
*          Date: Oct 07, 1993 13:50 
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int xpm_write(
   char	*filename,
   xpm	*image)
{
   int fid;
   int status;
   
   /* 
    * Sanity check to make sure that the image pointer is not NULL
    */
   if (image == NULL)
   {
      errno = KNULL_PARAMETER;
      kerror(KDATAFMT, "xpm_write",
	     "NULL image passed into write X11 Pixmap Format routine");
      return(FALSE);
   }
   
   /* 
    * Open the output file 
    */
   if ((fid = kopen(filename, O_WRONLY|O_TRUNC|O_CREAT, 0664)) < 0)
   {
      kerror(KDATAFMT, "xpm_write",
	     "unable to access %s for writing", filename);
      return(FALSE);
   }
   
   /*
    *  Lock the transport file descriptor if the transport has
    *  data permanence.
    */
   if (kfile_getpermanence(filename) == TRUE)
      kflock(fid, KLOCK_EX);
   
   status = xpm_fdwrite(fid, image);
   
   kclose(fid);
   
   return status;
}

/************************************************************
*
*  Routine Name: xpm_writeheader - writes a xpm header structure
*				      from the specified kfile id
*
*       Purpose: This routines writes a xpm header structure and it's
*		 data into the specified filename.  The routine
*		 uses the ktransport library which allows for
*		 remote writing of the xpm header.
*
*         Input: fid - the kfile id opened previously with kopen()
*		 image - the image header to be written
*
*        Output: none
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Jeremy Worley
*          Date: Aug 16, 1994 09:48
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int xpm_writeheader(
   int		fid,
   xpm	*image)
{
   kfile *f = kfdopen(fid, "w");
   int i;
   int nchars = (int)kceil(klogn((double)image->ncolors,
				 (double)kstrlen(xpm_printables)));

   /*
    *  Write the (xpm) header out
    */
   kfprintf(f, "/* XPM */\nstatic char * %s[] = {\n/*  pixmap\n * width "
	    "height ncolors chars_per_pixel */\n\"%d %d %d %d \",\n",
	    image->name == NULL? "noname" : image->name, image->width, 
	    image->height, image->ncolors, nchars);

   for (i = 0; i < image->ncolors; i++)
      kfprintf(f, "\"%s c #%.4x%.4x%.4x\",\n", _xpm_get_color(i, nchars),
	       (int)image->maps[i].red, (int)image->maps[i].green, 
	       (int)image->maps[i].blue);

   kfprintf(f, "/* pixels */\n");
   
   return(TRUE);
}


/************************************************************
*
*  Routine Name: xpm_free - frees a xpm structure and it's
*				associated data
*
*       Purpose: This routine frees an khoros xpm structure and
*		 it's associated data.
*
*         Input: image - a pointer to an khoros xpm structure that
*			 contains the image structure to be freed.
*
*        Output: none
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Mark Young 
*          Date: Oct 07, 1993 13:50
*      Verified:
*  Side Effects: Once this routine is called no further reference
*		 the image should be made.
* Modifications:
*
*************************************************************/

int xpm_free(
   xpm	*image)
{
	if (!image)
	   return(FALSE);

	/*
	 *  Free the data and all associated resources associated with
	 *  the image.
	 */
	kfree(image->data);
	kfree(image->name);
	kfree(image->maps);

	/*
	 *  Free the xpm image structure itself, which was previously
	 */
	kfree(image);
	return(TRUE);
}


/************************************************************
*
*  Routine Name: xpm_create - creates a xpm structure and it's
*				associated data
*
*       Purpose: This routine creates an khoros xpm structure and
*		 it's associated data.
*
*         Input: image - a pointer to an khoros xpm structure that
*			 contains the image structure to be freed.
*
*        Output: none
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Mark Young 
*          Date: Oct 07, 1993 13:50
*      Verified:
*  Side Effects: 
* Modifications:
*
*************************************************************/

xpm *xpm_create(
   void)
{
	xpm *image;

	/* 
	 * Sanity check to make sure that the image pointer is not NULL
	 */
	if ((image = (xpm *) kcalloc(1, sizeof(xpm))) == NULL)
	{
	   kerror(KDATAFMT, "xpm_create",
		"Unable to allocate %d bytes for the xpm \
(X11 Pixmap Format) header", sizeof(xpm));
	   return(NULL);
	}
	return(image);
}
