 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for kinset
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 	kapu_inset
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/*
 * This turns off the Global Subexpression Evaluation optimization in
 * Centerline CC.   We do this because clcc turns it off anyway because
 * kapu_inset is too long and some table overflows.  The other side
 * effect is that clcc insists on blurting out a ridiculous message
 * about the problem.  Turning off the optimization gets rid of the
 * message.  -jw Thu Mar 23 16:45:17 1995
 */
#if defined(__CLCC__)
#  pragma Off(Global_CSE)
#endif

/****************************************************************
* 
*  Routine Name: kapu_inset - inset source object 2 into source object 1
* 
*       Purpose: kapu_inset inserts the data from src2_obj into src1_obj
*                at the specified position, replacing the data in
*                src1_obj.  The resulting destination object (dst_obj)
*                size will be at least the size of src1_obj, and may be 
*		 larger, depending on the specified offset values (w_pos,
*		 h_pos, d_pos, t_pos, e_pos) and the size of src2_obj.
*
*		 The only_inset_valid parameter defines whether only
*		 data from src2 that is marked as valid by the validity
*		 mask is inset into src1.  If only_inset_valid is FALSE,
*		 all data and the mask from src2 are inset into src1.
*		 If only_inset_valid is TRUE, the src2 mask is checked
*		 for each point, and only valid points are inset into
*		 src1.  The src2 mask is not propagated to the 
*		 destination.
*
*                Lkinset is based on implicit indexing.  This means
*                that if location or time data exist, the inset
*                operation is not done in terms an interpretation of
*                the location/time data values, but in terms of the
*                implicit indexing of these data (which is specified
*                by the width, height, depth, time, elements indices
*                of the polymorphic data model).  
*
*                If location and/or time data exist, src1_obj is 
*		 considered the dominant object.  So the location/time 
*		 data and attributes from src1_obj will be propagated
*		 to the destination object.  If location/time does not 
*                exist in the destination object, it will be created there.
*
*                If src2_obj contains location or time data, the inset
*		 operation will still be performed, but the location/time
*                data from src2_obj will not be preserved.
*
*                The position in src1_obj where src2_obj will be inserted
*                is initially obtained from the KPDS_SUBOBJECT_POSITION
*                attribute stored in src2_obj.  Then, for any flag
*                (w_flag, h_flag, d_flag, t_flag, e_flag) that is TRUE,
*                the position for the corresponding dimension is reset
*                to the position value passed in (w_pos, h_pos, d_pos,
*                t_pos, e_pos).
*
*                Padding - In some instances, padding is necessary to
*                maintain the integrity of the polymorphic data model -
*                for example when the final size of the destination
*                object is larger than the base source object.  The real 
*                and imag parameters passed in define the values that 
*		 will be used when padding.
*
*                If padding occurs in the destination object, and the
*                valid flag is TRUE, padded data is considered valid
*                and, if the destination object contains a validity mask,
*                the mask value is set accordingly.  If the valid flag
*                is FALSE, the padded data will be masked as invalid.
*                In this case, if the destination object does not have
*                a validity mask, one will be created.
*
*                Treatment of maps, and how the data is represented in 
*                the destination object, depends on the mapping parameter 
*                passed in.  Possible mapping options are KAPU_MAP_ALL_DATA 
*                and KAPU_USE_COMMON_MAP.  If neither source objects has 
*		 a map, the mapping option is ignored.  If there are doubts 
*                about which mapping option to use, the safest bet is to 
*                map the data thru the maps.
*                .RS
*                .IP KAPU_MAP_ALL_DATA
*                Map Data Thru Maps:
*                All data will be mapped before the inset operation is
*                performed.  The destination object will not have a map.
*
*                .IP KAPU_USE_COMMON_MAP
*                Use First Map Only:
*                In this case, the map data and color attributes of the
*                first source object that has a map are directly transferred
*                to the destination object.  Note that by selecting this 
*                mapping option, you are assuming that the value segments 
*                of both objects have valid indices into that map.
*                .RE
*
*         Input: src1_obj - base object
*                src2_obj - object to inset into base object
*                w_flag   - if TRUE, use w_pos for width insert position
*                w_pos    - insert coordinate for width dimension
*                h_flag   - if TRUE, use h_pos for height insert position
*                h_pos    - insert coordinate for height dimension
*                d_flag   - if TRUE, use d_pos for depth insert position
*                d_pos    - insert coordinate for depth dimension
*                t_flag   - if TRUE, use t_pos for time insert position
*                t_pos    - insert coordinate for time dimension
*                e_flag   - if TRUE, use e_pos for elements insert position
*                e_pos    - insert coordinate for elements dimension
*                real     - real pad value
*                imag     - imaginary pad value
*                pad_is_valid    - if TRUE, padded data is considered valid
*                only_inset_valid - if TRUE, check mask; only inset valid data
*                mapping - if sources have maps, defines how map is used
*        Output: dst_obj  - resulting data object
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Donna Koechner and Mark Young
*          Date: Aug 30, 1994
*      Verified: 
*  Side Effects: 
* Modifications: 
****************************************************************/

int kapu_inset(
   kobject src1_obj,
   kobject src2_obj,
   int	   w_flag,
   int     w_pos,
   int	   h_flag,
   int     h_pos,
   int	   d_flag,
   int     d_pos,
   int	   t_flag,
   int     t_pos,
   int	   e_flag,
   int     e_pos,
   double  real,
   double  imag,
   int	   pad_is_valid,
   int     only_inset_valid,
   int     mapping,
   kobject dst_obj)
{
        klist *objlist=NULL;
	int   grid_type=KNONE;
	int   mask; 
        int   w1=1, h1=1, d1=1, t1=1, e1=1; 
	int   w2=1, h2=1, d2=1, t2=1, e2=1;
	int   type1, type2, type;

        kaddr data = NULL, bgdata=NULL;
	int   wd=1, hd=1, dd=1, td=1, ed=1;
	int   wp=0, hp=0, dp=0, tp=0, ep=0;
	int   wdp=0, hdp=0, ddp=0, tdp=0, edp=0;
	int   w_rgn=1, h_rgn=1, d_rgn=1, t_rgn=1, e_rgn=1, num_rgns=1;
	int   i, j;
	int   transfer_map1=FALSE, transfer_map2=FALSE;
	int   transfer_loc1=FALSE, transfer_time1=FALSE;
	double mask_pad_value;
	char  *rtn="kapu_inset", *lib="kapputils";
	unsigned char	*ub_data=NULL, *ub_bgdata=NULL, *mdata=NULL;
	unsigned long	*ul_data=NULL, *ul_bgdata=NULL;
	long		*l_data=NULL, *l_bgdata=NULL;
	double		*d_data=NULL, *d_bgdata=NULL;
	kdcomplex	*dc_data=NULL, *dc_bgdata=NULL;

/*------------------------------------------------------------------------
|  INITIAL ERROR CHECKING
------------------------------------------------------------------------*/
 
        /*----------------------------------------------------------------
        | if an illegal mapping type is passed in, set to KAPU_MAP_ALL_DATA
        | and print warning message.
        ----------------------------------------------------------------*/
        if ((mapping != KAPU_USE_COMMON_MAP) && (mapping != KAPU_MAP_ALL_DATA))
	{
           kinfo(KSTANDARD, "Warning: unknown mapping type (%d) passed to "
                            "kapu_inset.  Setting to KAPU_MAP_ALL_DATA (%d)",
                            mapping, KAPU_MAP_ALL_DATA);
           mapping = KAPU_MAP_ALL_DATA;
        }

        /*----------------------------------------------------------------
        | Check if sources have location. If source 1 does, will transfer
	| it to the destination.
        ----------------------------------------------------------------*/
	transfer_loc1 = kapu_multi_input_loc(&grid_type, 1, src1_obj);
	
        /*----------------------------------------------------------------
        | Check if sources have time. If source 1 does, transfer
	| it to the destination.
        ----------------------------------------------------------------*/
	transfer_time1 = kapu_multi_input_time(1, src1_obj);
	
        /*----------------------------------------------------------------
        | Both source and destination objects must have value data for 
	| inset operation.
        ----------------------------------------------------------------*/
	if (!kpds_query_value(src1_obj) || !kpds_query_value(src2_obj))
        {
           kerror(lib,rtn, "Source objects must have value data.");
           return(FALSE);
        }
	if (!kpds_query_value(dst_obj) && !kpds_create_value(dst_obj))
        {
           kerror(lib,rtn, "Unable to create value segment for dst_obj.");
           return(FALSE);
        }

        /*----------------------------------------------------------------
        | Reference the input objects to avoid side effects, then add the
        | reference to the list of goodies to be autmatically free'd on
        | error. 
        ----------------------------------------------------------------*/
        KCALL(((src1_obj = kpds_reference_object(src1_obj)) != NULL));
        objlist = klist_add(objlist,src1_obj,"KOBJECT");

        KCALL(((src2_obj = kpds_reference_object(src2_obj)) != NULL));
        objlist = klist_add(objlist,src2_obj,"KOBJECT");

        /*----------------------------------------------------------------
        | If input objects have maps, map the data before proceeding.
	| Also destroy it in the destination object, if it exists.
        ----------------------------------------------------------------*/
	if (mapping == KAPU_USE_COMMON_MAP) 
	{
	   if (kpds_query_map(src1_obj))
	      transfer_map1 = TRUE;
	   else if (kpds_query_map(src2_obj))
	      transfer_map2 = TRUE;
	}
	else
	{
	   if (kpds_query_map(src1_obj))
              KCALL(kpds_set_attribute(src1_obj,KPDS_MAPPING_MODE, KMAPPED));
	
	   if (kpds_query_map(src2_obj))
              KCALL(kpds_set_attribute(src2_obj,KPDS_MAPPING_MODE, KMAPPED));
	
	   if (kpds_query_map(dst_obj))
	      KCALL(kpds_destroy_map(dst_obj));
	}

        /*----------------------------------------------------------------
        | Get size and type attributes from source objects (also get 
        | subobject position from src2_obj).
        ----------------------------------------------------------------*/
        KCALL(kpds_get_attribute(src1_obj,KPDS_VALUE_SIZE,&w1,&h1,&d1,&t1,&e1));
        KCALL(kpds_get_attribute(src1_obj, KPDS_VALUE_DATA_TYPE, &type1));

        KCALL(kpds_get_attribute(src2_obj,KPDS_VALUE_SIZE,&w2,&h2,&d2,&t2,&e2));
        KCALL(kpds_get_attribute(src2_obj, KPDS_SUBOBJECT_POSITION, 
						&wp, &hp, &dp, &tp, &ep));
        KCALL(kpds_get_attribute(src2_obj, KPDS_VALUE_DATA_TYPE, &type2));
        /*----------------------------------------------------------------
        | If position flags set, use values provided for those dimensions.
        ----------------------------------------------------------------*/
	if (w_flag)  wp = w_pos;
   	if (h_flag)  hp = h_pos;
   	if (d_flag)  dp = d_pos;
   	if (t_flag)  tp = t_pos;
   	if (e_flag)  ep = e_pos;

        /*----------------------------------------------------------------
        | Calculate the final object size (position may be negative)
        ----------------------------------------------------------------*/
/*
	if ((wp == 0) && (w1 == w2))  wd = w1;
*/
	if (wp < 0)  wd = (w1-wp > w2)    ? w1-wp : w2;
        else         wd = (w1    > w2+wp) ? w1    : w2+wp;

	if (hp < 0)  hd = (h1-hp > h2)    ? h1-hp : h2;
        else         hd = (h1    > h2+hp) ? h1    : h2+hp;

        if (dp < 0)  dd = (d1-dp > d2)    ? d1-dp : d2;
	else         dd = (d1    > d2+dp) ? d1    : d2+dp;

        if (tp < 0)  td = (t1-tp > t2)    ? t1-tp : t2;
	else         td = (t1    > t2+tp) ? t1    : t2+tp;

        if (ep < 0)  ed = (e1-ep > e2)    ? e1-ep : e2;
	else         ed = (e1    > e2+ep) ? e1    : e2+ep;

        /*----------------------------------------------------------------
        | Set size of destination object.
        ----------------------------------------------------------------*/
        KCALL(kpds_set_attribute(dst_obj, KPDS_VALUE_SIZE, wd,hd,dd,td,ed));

        /*----------------------------------------------------------------
        | Set value data type of objects to the highest source object type.
        ----------------------------------------------------------------*/
	type = kdatatype_cast_output(type1, type2);
	KCALL(kpds_set_attribute(src1_obj, KPDS_VALUE_DATA_TYPE, type));
	KCALL(kpds_set_attribute(src2_obj, KPDS_VALUE_DATA_TYPE, type));
	KCALL(kpds_set_attribute(dst_obj,  KPDS_VALUE_DATA_TYPE, type));

        /*----------------------------------------------------------------
	| If the destination size is greater than src1_obj size, padding
	| will occur.  Also, if the valid parameter passed in is FALSE, the
	| padded data should be masked as invalid.  Therefore, a mask must
	| be created. 
        ----------------------------------------------------------------*/
	mask = FALSE;
	if (!pad_is_valid && ( (wd>w1)||(hd>h1)||(dd>d1)||(td>t1)||(ed>e1) ))
	   mask = TRUE;
	   
        /*----------------------------------------------------------------
        | If either source object has a mask, create destination mask.
        ----------------------------------------------------------------*/
	if (kpds_query_mask(src1_obj) || kpds_query_mask(src2_obj) || (mask))
	{
	   type1=type2=KUBYTE;

	   if (kpds_query_mask(src1_obj))
	       KCALL(kpds_get_attribute(src1_obj,KPDS_MASK_DATA_TYPE,&type1));
	   
	   if (kpds_query_mask(src2_obj))
	       KCALL(kpds_get_attribute(src2_obj,KPDS_MASK_DATA_TYPE,&type2));

	   if (!kpds_query_mask(src1_obj))
	   {
	      type1 = type2;
	      KCALL(kpds_create_mask(src1_obj));
	      KCALL(kpds_set_attribute(src1_obj,KPDS_MASK_DATA_TYPE,type1));
	      KCALL(kpds_initialize_mask(src1_obj, 1.0));
	   }

	   if (!kpds_query_mask(src2_obj))
	   {
	      type2 = type1;
	      KCALL(kpds_create_mask(src2_obj));
	      KCALL(kpds_set_attribute(src2_obj,KPDS_MASK_DATA_TYPE,type2));
	      KCALL(kpds_initialize_mask(src2_obj, 1.0));
	   }

	   if (!kpds_query_mask(dst_obj))
	   {
	      KCALL(kpds_create_mask(dst_obj));
	      KCALL(kpds_set_attribute(dst_obj,KPDS_MASK_SIZE,wd,hd,dd,td,ed));
	   }

           /*-------------------------------------------------------------
           | Set all objects to highest mask data type 
           -------------------------------------------------------------*/
	   type = kdatatype_cast_output(type1, type2);
	   KCALL(kpds_set_attribute(src1_obj, KPDS_MASK_DATA_TYPE, type));
	   KCALL(kpds_set_attribute(src2_obj, KPDS_MASK_DATA_TYPE, type));
	   KCALL(kpds_set_attribute(dst_obj,  KPDS_MASK_DATA_TYPE, type));
	   mask = TRUE;
	}

	else if (kpds_query_mask(dst_obj)) 
	   KCALL(kpds_destroy_mask(dst_obj));

        /*----------------------------------------------------------------
        | Set size of the first source object to the destination size.
	| Need to resize src1_obj for the pad value to be applied. 
        ----------------------------------------------------------------*/
        KCALL(kpds_set_attribute(src1_obj, KPDS_VALUE_SIZE, wd,hd,dd,td,ed));

        /*----------------------------------------------------------------
        | Determine and set the optimal value region size for src1_obj.
        ----------------------------------------------------------------*/
	KCALL(kpds_get_attribute(src1_obj, KPDS_VALUE_OPTIMAL_REGION_SIZE, 
			&w_rgn, &h_rgn, &d_rgn, &t_rgn, &e_rgn, &num_rgns));

        /*----------------------------------------------------------------
        | Calculate the offset to be applied to src1_obj.
        ----------------------------------------------------------------*/
	wdp=hdp=ddp=tdp=edp=0;
	if (wp < 0)  wdp = wp;
	if (hp < 0)  hdp = hp;
	if (dp < 0)  ddp = dp;
	if (tp < 0)  tdp = tp;
	if (ep < 0)  edp = ep;

        /*----------------------------------------------------------------
        | Set the region size, padding, offset, and position for src1_obj.
        ----------------------------------------------------------------*/
	KCALL(kpds_set_attribute(src1_obj, 
	      KPDS_VALUE_REGION_SIZE, w_rgn, h_rgn, d_rgn, t_rgn, e_rgn));
	KCALL(kpds_set_attribute(src1_obj, KPDS_VALUE_POSITION, 0,0,0,0,0));
	KCALL(kpds_set_attribute(src1_obj, 
	      KPDS_VALUE_OFFSET, wdp, hdp, ddp, tdp, edp));
	KCALL(kpds_set_attribute(src1_obj, KPDS_VALUE_INTERPOLATE, KPAD));
	KCALL(kpds_set_attribute(src1_obj, KPDS_VALUE_PAD_VALUE, real, imag));

        /*----------------------------------------------------------------
        | Reference the destination object
        ----------------------------------------------------------------*/
        KCALL(((dst_obj = kpds_reference_object(dst_obj)) != NULL));
        objlist = klist_add(objlist,dst_obj,"KOBJECT");

        /*----------------------------------------------------------------
        | Set the region size and position for data in dst_obj.
        ----------------------------------------------------------------*/
	KCALL(kpds_set_attribute(dst_obj, 
		KPDS_VALUE_REGION_SIZE, w_rgn, h_rgn, d_rgn, t_rgn, e_rgn));
	KCALL(kpds_set_attribute(dst_obj, KPDS_VALUE_POSITION, 0, 0, 0, 0, 0));

/*
kpds_print_attribute(src1_obj, KPDS_VALUE_SIZE, kstdout);
kinfo(KSTANDARD," <-- new src 1 size");
kpds_print_attribute(dst_obj, KPDS_VALUE_SIZE, kstdout);
kinfo(KSTANDARD," <-- destination size");
kpds_print_attribute(src1_obj, KPDS_VALUE_REGION_SIZE, kstdout);
kinfo(KSTANDARD," <-- src 1 region size");
kpds_print_attribute(dst_obj, KPDS_VALUE_REGION_SIZE, kstdout);
kinfo(KSTANDARD," <-- dst region size");
kinfo(KSTANDARD," number of regions = %d", num_rgns);
*/

        /*----------------------------------------------------------------
        | Get all of the value data from src1_obj and put it in dst_obj.
        ----------------------------------------------------------------*/
        objlist = klist_add(objlist,data,"KMALLOC");
	for (i=0; i<num_rgns; i++)
	{
	   if ((data = kpds_get_data(src1_obj,KPDS_VALUE_REGION,data)) == NULL)
	   {
              kerror(lib,rtn, "Unable to get src1_obj value data.");
	      (void)klist_free(objlist, (kfunc_void)lkcall_free);
              return(FALSE);
           }
	   if (!kpds_put_data(dst_obj, KPDS_VALUE_REGION, data))
	   {
              kerror(lib,rtn, "Unable to put value data to dst_obj.");
	      (void)klist_free(objlist, (kfunc_void)lkcall_free);
              return(FALSE);
           }
	}

        /*----------------------------------------------------------------
        | Get all of the mask data from src1_obj and put it in dst_obj.
        ----------------------------------------------------------------*/
	if (mask)
	{
	   kfree(data);
	   if (pad_is_valid)  	mask_pad_value = 1.0;
	   else         	mask_pad_value = 0.0;

           /*-------------------------------------------------------------
           | Set the region size, padding, offset, and position for src1_obj.
           -------------------------------------------------------------*/
	   KCALL(kpds_set_attribute(src1_obj, 
		 KPDS_MASK_REGION_SIZE, w_rgn, h_rgn, d_rgn, t_rgn, e_rgn));
	   KCALL(kpds_set_attribute(src1_obj, KPDS_MASK_POSITION, 0,0,0,0,0));
	   KCALL(kpds_set_attribute(src1_obj, 
		 KPDS_MASK_OFFSET, wdp, hdp, ddp, tdp, edp));
	   KCALL(kpds_set_attribute(src1_obj, KPDS_MASK_INTERPOLATE, KPAD));
	   KCALL(kpds_set_attribute(src1_obj, 
		 KPDS_MASK_PAD_VALUE, mask_pad_value));
           
	   KCALL(kpds_set_attribute(dst_obj, KPDS_MASK_POSITION, 0,0,0,0,0));
	   KCALL(kpds_set_attribute(dst_obj, 
		 KPDS_MASK_REGION_SIZE, w_rgn, h_rgn, d_rgn, t_rgn, e_rgn));
           
	   for (i=0; i<num_rgns; i++)
	   {
	      if ((data=kpds_get_data(src1_obj,KPDS_MASK_REGION,data)) == NULL)
	      {
                 kerror(lib,rtn, "Unable to get src1_obj mask data.");
	         (void)klist_free(objlist, (kfunc_void)lkcall_free);
                 return(FALSE);
              }
	      if (!kpds_put_data(dst_obj, KPDS_MASK_REGION, data))
	      {
                 kerror(lib,rtn, "Unable to put mask data to dst_obj.");
	         (void)klist_free(objlist, (kfunc_void)lkcall_free);
                 return(FALSE);
              }
	   }
	}
	
        /*----------------------------------------------------------------
        | Determine the optimal size region to get from src2_obj.
        ----------------------------------------------------------------*/
	KCALL(kpds_get_attribute(src2_obj, KPDS_VALUE_OPTIMAL_REGION_SIZE, 
			&w_rgn, &h_rgn, &d_rgn, &t_rgn, &e_rgn, &num_rgns));
        
	KCALL(kpds_set_attribute(src2_obj, 
		KPDS_VALUE_REGION_SIZE, w_rgn, h_rgn, d_rgn, t_rgn, e_rgn));
	KCALL(kpds_set_attribute(src2_obj, KPDS_VALUE_OFFSET, 0, 0, 0, 0, 0));
	KCALL(kpds_set_attribute(src2_obj, KPDS_VALUE_POSITION, 0, 0, 0, 0, 0));

        /*----------------------------------------------------------------
        | Set the region size and position for data in dst_obj 
        ----------------------------------------------------------------*/
	wdp=hdp=ddp=tdp=edp=0;
	if (wp > 0)  wdp = wp;
	if (hp > 0)  hdp = hp;
	if (dp > 0)  ddp = dp;
	if (tp > 0)  tdp = tp;
	if (ep > 0)  edp = ep;
	KCALL(kpds_set_attribute(dst_obj, KPDS_VALUE_POSITION, 0, 0, 0, 0, 0));
	KCALL(kpds_set_attribute(dst_obj, 
		KPDS_VALUE_OFFSET, wdp, hdp, ddp, tdp, edp));
	KCALL(kpds_set_attribute(dst_obj, 
		KPDS_VALUE_INCREMENT_SIZE, w2, h2, d2, t2, e2));
	KCALL(kpds_set_attribute(dst_obj, 
		KPDS_VALUE_REGION_SIZE, w_rgn,h_rgn,d_rgn,t_rgn,e_rgn));

        /*----------------------------------------------------------------
        | Get all of the data from src2_obj and put it in dst_obj
        ----------------------------------------------------------------*/
	for (i=0; i<num_rgns; i++)
	{
	   kfree(data);
	   if (!only_inset_valid || !mask)
	   {
	      if ((data = kpds_get_data(src2_obj,KPDS_VALUE_REGION,data)) 
			== NULL)
	      {
                 kerror(lib,rtn, "Unable to get src2_obj value data.");
	         (void)klist_free(objlist, (kfunc_void)lkcall_free);
                 return(FALSE);
              }
	      if (!kpds_put_data(dst_obj, KPDS_VALUE_REGION, data))
	      {
                 kerror(lib,rtn, "Unable to put value data to dst_obj.");
	         (void)klist_free(objlist, (kfunc_void)lkcall_free);
                 return(FALSE);
              }
	   }
	   else
	   {
              /*----------------------------------------------------------
              | If we're insetting only data marked as valid by mask, set 
	      | data of source and destination objects type
              ----------------------------------------------------------*/
              type = kdatatype_cast_process(type, type,
                        (KUBYTE | KLONG | KULONG | KDOUBLE | KDCOMPLEX));

	      KCALL(kpds_set_attribute(dst_obj, KPDS_VALUE_DATA_TYPE, type));

	      KCALL(kpds_set_attribute(src2_obj, KPDS_VALUE_DATA_TYPE, type));
	      KCALL(kpds_set_attribute(src2_obj, KPDS_MASK_DATA_TYPE, KUBYTE));
	      KCALL(kpds_set_attribute(src2_obj, 
			KPDS_MASK_REGION_SIZE, w_rgn,h_rgn,d_rgn,t_rgn,e_rgn));
              
	      if ((data = kpds_get_data(src2_obj,KPDS_VALUE_REGION,data)) 
			 == NULL)
	      {
                 kerror(lib,rtn, "Unable to get src2_obj value data.");
	         (void)klist_free(objlist, (kfunc_void)lkcall_free);
                 return(FALSE);
              }
              
	      if ((mdata = kpds_get_data(src2_obj,KPDS_MASK_REGION,mdata)) 
			== NULL)
	      {
                 kerror(lib,rtn, "Unable to get src2_obj value data.");
	         (void)klist_free(objlist, (kfunc_void)lkcall_free);
	   	 if (mdata)	kfree(mdata);
                 return(FALSE);
              }
              
	      if ((bgdata = kpds_get_data(dst_obj,KPDS_VALUE_REGION,bgdata)) 
			   == NULL)
	      {
                 kerror(lib,rtn, "Unable to get dst_obj value data.");
	         (void)klist_free(objlist, (kfunc_void)lkcall_free);
	   	 if (bgdata)	kfree(bgdata);
	   	 if (mdata)	kfree(mdata);
                 return(FALSE);
              }

	      if (type == KUBYTE)
              {
	         ub_data = (unsigned char *) data;
	         ub_bgdata = (unsigned char *) bgdata;
	         for (j=0; j<(w_rgn*h_rgn*d_rgn*t_rgn*e_rgn); j++)
		    ub_data[j] = (mdata[j]) ? ub_data[j] : ub_bgdata[j];
              }
              else if (type1 == KULONG)
              {
	         ul_data = (unsigned long *) data;
	         ul_bgdata = (unsigned long *) bgdata;
	         for (j=0; j<(w_rgn*h_rgn*d_rgn*t_rgn*e_rgn); j++)
		    ul_data[j] = (mdata[j]) ? ul_data[j] : ul_bgdata[j];
              }
              else if (type1 == KLONG)
              {
	         l_data = (long *) data;
	         l_bgdata = (long *) bgdata;
	         for (j=0; j<(w_rgn*h_rgn*d_rgn*t_rgn*e_rgn); j++)
		    l_data[j] = (mdata[j]) ? l_data[j] : l_bgdata[j];
              }
              else if (type1 == KDOUBLE)
              {
	         d_data = (double *) data;
	         d_bgdata = (double *) bgdata;
	         for (j=0; j<(w_rgn*h_rgn*d_rgn*t_rgn*e_rgn); j++)
		    d_data[j] = (mdata[j]) ? d_data[j] : d_bgdata[j];
              }
              else if (type1 == KDCOMPLEX)
              {
	         dc_data = (kdcomplex *) data;
	         dc_bgdata = (kdcomplex *) bgdata;
	         for (j=0; j<(w_rgn*h_rgn*d_rgn*t_rgn*e_rgn); j++)
		    dc_data[j] = (mdata[j]) ? dc_data[j] : dc_bgdata[j];
              }

	      if (!kpds_put_data(dst_obj, KPDS_VALUE_REGION, data))
	      {
                 kerror(lib,rtn, "Unable to put value data to dst_obj.");
	         (void)klist_free(objlist, (kfunc_void)lkcall_free);
	         if (bgdata)	kfree(bgdata);
	         if (mdata)	kfree(mdata);
                 return(FALSE);
              }
	   }
	   if (bgdata)	kfree(bgdata);
	   if (mdata)	kfree(mdata);
	}
	
        /*----------------------------------------------------------------
        | Get all of the mask data from src2_obj and put it in dst_obj.
	| If we are only insetting valid data, remove 2nd object's
	| mask before continuing. 
        ----------------------------------------------------------------*/
	if (mask && !only_inset_valid)
	{
	   KCALL(kpds_set_attribute(dst_obj, KPDS_MASK_OFFSET, 0, 0, 0, 0, 0));
	   KCALL(kpds_set_attribute(dst_obj, 
		KPDS_MASK_POSITION, wdp, hdp, ddp, tdp, edp));
	   KCALL(kpds_set_attribute(dst_obj, 
		KPDS_MASK_INCREMENT_SIZE, w2, h2, d2, t2, e2));
	   KCALL(kpds_set_attribute(dst_obj, 
		KPDS_MASK_REGION_SIZE, w_rgn,h_rgn,d_rgn,t_rgn,e_rgn));

	   KCALL(kpds_set_attribute(src2_obj, KPDS_MASK_POSITION, 0,0,0,0,0));
	   KCALL(kpds_set_attribute(src2_obj, KPDS_MASK_OFFSET,   0,0,0,0,0));
	   KCALL(kpds_set_attribute(src2_obj, 
		KPDS_MASK_REGION_SIZE, w_rgn,h_rgn,d_rgn,t_rgn,e_rgn));

	   kfree(data);
	   for (i=0; i<num_rgns; i++)
	   {
	      if ((data=kpds_get_data(src2_obj,KPDS_MASK_REGION,data)) == NULL)
	      {
                 kerror(lib,rtn, "Unable to get src2_obj mask data.");
	         (void)klist_free(objlist, (kfunc_void)lkcall_free);
                 return(FALSE);
              }
	      if (!kpds_put_data(dst_obj, KPDS_MASK_REGION, data))
	      {
                 kerror(lib,rtn, "Unable to put mask data to dst_obj.");
	         (void)klist_free(objlist, (kfunc_void)lkcall_free);
                 return(FALSE);
              }
	   }
	}

        /*----------------------------------------------------------------
        | If we are transferring src1_obj's map, copy the map.
        ----------------------------------------------------------------*/
	if (transfer_map1 || transfer_map2)
	{
	   if (!kpds_query_map(dst_obj))
	      KCALL(kpds_create_map(dst_obj));
	   if (transfer_map1)
	   {
	      KCALL(kcolor_copy_attributes(src1_obj, dst_obj,  
				KCOLOR_MAP_AUTOCOLOR, KCOLOR_COLORSPACE,
                                KCOLOR_HAS_ALPHA, NULL));
	      KCALL(kpds_copy_map(src1_obj, dst_obj, TRUE));
	   }
	   else if (transfer_map2)
	   {
	      KCALL(kcolor_copy_attributes(src2_obj, dst_obj,  
				KCOLOR_MAP_AUTOCOLOR, KCOLOR_COLORSPACE,
                                KCOLOR_HAS_ALPHA, NULL));
	      KCALL(kpds_copy_map(src2_obj, dst_obj, TRUE));
	   }
	}

        /*----------------------------------------------------------------
        | If source 1 has location or time data, transfer it to destination
        ----------------------------------------------------------------*/
	if (transfer_loc1)
	{
           if (!kpds_query_location(dst_obj))
              KCALL(kpds_create_location(dst_obj));
	   KCALL(kpds_copy_location(src1_obj, dst_obj, TRUE));
	}

	if (transfer_time1)
	{
           if (!kpds_query_time(dst_obj))
              KCALL(kpds_create_time(dst_obj));
	   KCALL(kpds_copy_time(src1_obj, dst_obj, TRUE));
	}

	(void)klist_free(objlist, (kfunc_void)lkcall_free);
        return(TRUE);
}
