 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Polymophic Model Data Handlers
   >>>>
   >>>>   Static:
   >>>>			_get_type()
   >>>>
   >>>>                 _size_get()
   >>>>                 _size_set()
   >>>>                 _size_match()
   >>>>                 _size_copy()
   >>>>                 _size_query()
   >>>>                 _size_print() 
   >>>>             
   >>>>                 _type_get()
   >>>>                 _type_set()
   >>>>                 _type_match()
   >>>>                 _type_copy()
   >>>>                 _type_query()
   >>>>                 _type_print() 
   >>>>
   >>>>			_create_value()
   >>>>			_get_value_data()
   >>>>			_put_value_data()
   >>>>  Private:
   >>>>			kpds_init_value_segment()
   >>>>   Public:
   >>>>			kpds_create_value()
   >>>>			kpds_query_value()
   >>>>			kpds_destroy_value()
   >>>>			kpds_create_value()
   >>>>			kpds_copy_value_data()
   >>>>			kpds_copy_value_attr()
   >>>>			kpds_copy_value()  
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"

/* -- pds_attrib.c -- */
extern int sync_sizes;

/* 
 *    ================================================================== 
 *    VALUE utilities
 *    ==================================================================
 */

/*-----------------------------------------------------------
|  Routine Name: (static) _get_type
|       Purpose: retrieve the value type
|       Returns: the value type
|    Written By: Steve Kubica
------------------------------------------------------------*/
static 
int _get_type(kobject obj)
{
   int type;
   
   if (!kdms_get_attribute(obj, KPDS_SEGMENT_VALUE, KDMS_DATA_TYPE, &type))
      return FALSE;
   else
      return type;
}

/* 
 *    ================================================================== 
 *    VALUE attribute handlers
 *    ==================================================================
 */

/*********** ---------------------------------------------------------------
 ***********  KPDS_VALUE_SIZE
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: (static) _size_get
|       Purpose: get the value size attribute
|    Written By: Steve Kubica and Jeremy Worley
|          Date: Apr 27, 1994 21:34
------------------------------------------------------------*/
/* ARGSUSED */
static int
_size_get(kobject obj, int assoc, int attr, kaddr clientData, kva_list *list)
{
   int *w = kva_arg(*list, int *);
   int *h = kva_arg(*list, int *);
   int *d = kva_arg(*list, int *);
   int *t = kva_arg(*list, int *);
   int *e = kva_arg(*list, int *);
   int *size;


   if (!kdms_query_segment(obj, KPDS_SEGMENT_VALUE))
      return FALSE; /* error -- must create value first */

   if (!kdms_get_attribute(obj, KPDS_SEGMENT_VALUE, KDMS_SIZE, &size))
      return FALSE; 

   if (w) *w = size[KWID];
   if (h) *h = size[KHGT];
   if (d) *d = size[KDEP];
   if (t) *t = size[KTIM];

   /* -- if there is a map, then map width is element size -- */
   if (kaps_mapping_mode(obj) && kdms_query_segment(obj, KPDS_SEGMENT_MAP))
   {
      int map_wid;

      if (!kpds_get_attribute(obj, KPDS_MAP_SIZE, 
			      &map_wid, NULL, NULL, NULL, NULL))
	 return FALSE;

      if (e) *e = size[KELM] * map_wid;
   }
   else
      if (e) *e = size[KELM];

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _size_set
|       Purpose: set the value size attribute
|    Written By: Steve Kubica and Jeremy Worley
|          Date: Apr 27, 1994 21:34
------------------------------------------------------------*/
/* ARGSUSED */
static int
_size_set(kobject obj, int assoc, int attr, kaddr clientData, 
	  kva_list *list)
{
   int  w = kva_arg(*list, int);
   int  h = kva_arg(*list, int);
   int  d = kva_arg(*list, int);
   int  t = kva_arg(*list, int);
   int  e = kva_arg(*list, int);
   int  size[KPDS_VALUE_DIM] = {-1, -1, -1, -1, -1};

   if (!kdms_query_segment(obj, KPDS_SEGMENT_VALUE))
      return FALSE; /* error -- must create value first */

   size[KWID] = w;
   size[KHGT] = h;
   size[KDEP] = d;
   size[KTIM] = t;
   size[KELM] = e;

   if (sync_sizes)
   {
      sync_sizes = FALSE;  /* -- avoid calling here again -- */

      /* -- set the shared size of the mask segment -- */
      if (kdms_query_segment(obj, KPDS_SEGMENT_MASK))
	 if (!kpds_set_attribute(obj, KPDS_MASK_SIZE, w, h, d, t, e))
	    return FALSE;
   
      /* -- set the shared size of the location segment -- */
      if (kdms_query_segment(obj, KPDS_SEGMENT_LOCATION))
	 if (!kpds_set_attribute(obj, KPDS_LOCATION_SIZE, w, h, d, -1))
	    return FALSE;
      
      /* -- set the shared size of the time segment -- */
      if (kdms_query_segment(obj, KPDS_SEGMENT_TIME))
	 if (!kpds_set_attribute(obj, KPDS_TIME_SIZE, t))
	    return FALSE;

      /* -- set the shared size of the map segment -- */
      if (kdms_query_segment(obj, KPDS_SEGMENT_MAP))
      {
	 int *osize;

	 /* -- track size of value depth, time, and element only if previous
	  *    map size is not 1 --  */
	 if (!kdms_get_attribute(obj, KPDS_SEGMENT_MAP, KDMS_SIZE, &osize))
	    return FALSE;

	 if (osize[KMAP_DEP]!=1 || osize[KMAP_TIM]!=1 || osize[KMAP_ELM]!=1)
	 {
	    int msize[KPDS_MAP_DIM] = {-1, -1, -1, -1, -1};
	 
	    msize[KMAP_DEP] = (osize[KMAP_DEP] == 1) ? -1 : d;
	    msize[KMAP_TIM] = (osize[KMAP_TIM] == 1) ? -1 : t;
	    msize[KMAP_ELM] = (osize[KMAP_ELM] == 1) ? -1 : e;
	 
	    if (!kdms_set_attribute(obj, KPDS_SEGMENT_MAP, KDMS_SIZE, msize))
	       return FALSE;
	 }
      }

      /* -- if mapping mode is on, then map width is affected by esize -- */
      if (kaps_mapping_mode(obj) && kdms_query_segment(obj, KPDS_SEGMENT_MAP) 
	  && e != -1)  /* e of -1 implies that the esize should not change */
      {
	 int *old_s;
	 int  map_w;
	 
 	 /* 
	  *  the new element size will only decompose properly if the
	  *  true value is 1 or the map width is 1 ... otherwise, the
	  *  map size will get screwed up :-(
	  */
	 if (!kdms_get_attribute(obj, KPDS_SEGMENT_VALUE, KDMS_SIZE, &old_s))
	    return FALSE;

	 /* -- if value element size == 1, then map width takes on change -- */
	 if (old_s[KELM] == 1)
	 {
	    if (!kpds_set_attribute(obj, KPDS_MAP_SIZE, e, -1, -1, -1, -1))
	       return FALSE;

	    /* -- the element size will become 1 now -- */
	    size[KELM] = 1;
	 }
	 else  /* -- our only hope is if the map width is 1 -- */
	 {
	    if (!kpds_get_attribute(obj, KPDS_MAP_SIZE, &map_w, 
				    NULL, NULL, NULL, NULL))
	       return FALSE;

	    /* -- if map_w == 1, then value elements can take on change -- */
	    if (map_w != 1)
	       kerror("kappserv", "kpds_set_attribute", 
		      "Unable to alter size of value data with map.  "
		      "Data must be mapped explicitly before "
		      "operation will work.");
	 }
      }
      sync_sizes = TRUE;
   }
   
   /* -- set the size of the value segment -- */
   if (!kdms_set_attribute(obj, KPDS_SEGMENT_VALUE, KDMS_SIZE, size))
      return FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _size_match
|       Purpose: match the value size attribute
|    Written By: Steve Kubica
|          Date: Apr 28, 1994 17:37
------------------------------------------------------------*/
/* ARGSUSED */
static int
_size_match(kobject obj1, kobject obj2, int assoc, int attrib,
	    kaddr clientData1, kaddr clientData2)
{
   int w1, h1, d1, t1, e1;
   int w2, h2, d2, t2, e2;


   if (!kpds_get_attribute(obj1, KPDS_VALUE_SIZE, &w1, &h1, &d1, &t1, &e1))
      return FALSE;

   if (!kpds_get_attribute(obj2, KPDS_VALUE_SIZE, &w2, &h2, &d2, &t2, &e2))
      return FALSE;

   return ((w1 == w2) && (h1 == h2) && (d1 == d2) && (t1 == t2) && (e1 == e2));
}

/*-----------------------------------------------------------
|  Routine Name: (static) _size_copy
|       Purpose: copy the value size attribute
|    Written By: Steve Kubica
|          Date: Apr 28, 1994 17:37
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_size_copy(kobject obj1, kobject obj2, int assoc, int attrib,
	   kaddr clientData1, kaddr clientData2)
{
   int w, h, d, t, e;


   if (!kpds_get_attribute(obj1, KPDS_VALUE_SIZE, &w, &h, &d, &t, &e))
      return FALSE;

   if (!kpds_set_attribute(obj2, KPDS_VALUE_SIZE, w, h, d, t, e))
      return FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _size_query
|       Purpose: query the value size attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_size_query(kobject obj, int assoc, int attrib, kaddr clientData,
	    int *num_args, int *arg_size, int *data_type, int *permanent)
{
   if (!kdms_query_segment(obj, KPDS_SEGMENT_VALUE))
      return FALSE;		/* not an error */

   if (num_args)
      *num_args = 5;
   if (arg_size)
      *arg_size = 1;
   if (data_type)
      *data_type = KINT;
   if (permanent)
      *permanent = TRUE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _size_print
|       Purpose: print the value size attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_size_print(kobject obj, int assoc, int attrib, kaddr clientData,
	    kfile * outfile)
{
   int w, h, d, t, e;


   if (!kpds_get_attribute(obj, KPDS_VALUE_SIZE, &w, &h, &d, &t, &e))
      return FALSE;

   if (outfile)
      kfprintf(outfile, "%d %d %d %d %d", w, h, d, t, e);

   return TRUE;
}

/*********** ---------------------------------------------------------------
 ***********  KPDS_VALUE_DATA_TYPE
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: (static) _type_get
|       Purpose: get the type attribute
|    Written By: Jeremy Worley, John Salas, and Steve Kubica
|          Date: Apr 28, 1994 17:39
------------------------------------------------------------*/
/* ARGSUSED */
static int
_type_get(kobject obj, int assoc, int attr, kaddr clientData, kva_list *list)
{
   int *rtyp = kva_arg(*list, int *);
   int  typ;


   if (!kdms_query_segment(obj, KPDS_SEGMENT_VALUE))
      return FALSE;

   if (kaps_mapping_mode(obj) && kdms_query_segment(obj, KPDS_SEGMENT_MAP))
   {
      if (!kdms_get_attribute(obj, KPDS_SEGMENT_MAP, KDMS_DATA_TYPE, &typ))
	 return FALSE;
   }
   else
   {
      if (!kdms_get_attribute(obj, KPDS_SEGMENT_VALUE, KDMS_DATA_TYPE, &typ))
	    return FALSE;
   }

   if (rtyp) *rtyp = typ;
   
   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _type_set
|       Purpose: set the type attribute
|    Written By: Jeremy Worley, John Salas, and Steve Kubica
|          Date: Apr 28, 1994 17:39
------------------------------------------------------------*/
/* ARGSUSED */
static int
_type_set(kobject obj, int assoc, int attr, kaddr clientData, kva_list *list)
{
   int typ = kva_arg(*list, int);

   if (!kdms_query_segment(obj, KPDS_SEGMENT_VALUE))
      return FALSE;

   if (kaps_mapping_mode(obj) && kdms_query_segment(obj, KPDS_SEGMENT_MAP))
   {
      if (!kdms_set_attribute(obj, KPDS_SEGMENT_MAP, KDMS_DATA_TYPE, typ))
	    return FALSE;
   }
   else
   {
      if (!kdms_set_attribute(obj, KPDS_SEGMENT_VALUE, KDMS_DATA_TYPE, typ))
	    return FALSE;
   }

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _type_match
|       Purpose: match the type attribute
|    Written By: Steve Kubica and John Salas
|          Date: Apr 28, 1994 17:39
------------------------------------------------------------*/
/* ARGSUSED */
static int
_type_match(kobject obj1, kobject obj2, int assoc, int attrib,
	    kaddr clientData1, kaddr clientData2)
{
   int typ1;
   int typ2;

   if (!kdms_query_segment(obj1, KPDS_SEGMENT_VALUE) ||
       !kdms_query_segment(obj2, KPDS_SEGMENT_VALUE))
      return FALSE;

   /* -- use pds to handle mapping mode -- */
   if (!kpds_get_attribute(obj1, KPDS_VALUE_DATA_TYPE, &typ1))
      return FALSE;

   if (!kpds_get_attribute(obj2, KPDS_VALUE_DATA_TYPE, &typ2))
      return FALSE;
   
   return (typ1 == typ2);
}

/*-----------------------------------------------------------
|  Routine Name: (static) _type_copy
|       Purpose: copy the type attribute
|    Written By: Steve Kubica
|          Date: Apr 28, 1994 17:43
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_type_copy(kobject obj1, kobject obj2, int assoc, int attrib,
	   kaddr clientData1, kaddr clientData2)
{
   int typ;

   if (!kdms_query_segment(obj1, KPDS_SEGMENT_VALUE) ||
       !kdms_query_segment(obj2, KPDS_SEGMENT_VALUE))
      return FALSE;

   /* -- use pds to handle mapping mode -- */
   if (!kpds_get_attribute(obj1, KPDS_VALUE_DATA_TYPE, &typ))
      return FALSE;

   if (!kpds_set_attribute(obj2, KPDS_VALUE_DATA_TYPE, typ))
      return FALSE;
   
   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _type_query
|       Purpose: query the type attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_type_query(kobject obj, int assoc, int attrib, kaddr clientData,
	    int *num_args, int *arg_size, int *data_type, int *permanent)
{
   if (!kdms_query_segment(obj, KPDS_SEGMENT_VALUE))
      return FALSE;

   if (num_args)
      *num_args = 1;
   if (arg_size)
      *arg_size = 1;
   if (data_type)
      *data_type = KINT;
   if (permanent)
      *permanent = TRUE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _type_print
|       Purpose: print the type attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_type_print(kobject obj, int assoc, int attrib, kaddr clientData,
	    kfile * outfile)
{
   int typ;

   if (!kdms_query_segment(obj, KPDS_SEGMENT_VALUE))
      return FALSE;

   /* -- use pds to handle mapping mode -- */
   if (!kpds_get_attribute(obj, KPDS_VALUE_DATA_TYPE, &typ))
      return FALSE;

   if (outfile)
      kfprintf(outfile, "%s (%d)", kdefine_to_datatype(typ), typ);

   return TRUE;
}

/* 
 *    ================================================================== 
 *    VALUE segment handlers
 *    ==================================================================
 */

/*-----------------------------------------------------------
|
|  Routine Name: (static) _create_value
|
|       Purpose: This routine will create the value segment and
|		 initialize its dimensionality and index order.
|
|         Input: obj     - the obj in which to create the value segment
|		 segment - the name of the value segment.
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica & Jeremy Worley
|          Date: Jun 01, 1994 16:23
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static int
_create_value(
   kobject  obj,
   char    *segment)
{
   int  order[KPDS_VALUE_DIM] = {KWIDTH, KHEIGHT, KDEPTH, KTIME, KELEMENTS};
   int *vorder = NULL;
   int  oldmode;
   int  w = -1;
   int 	h = -1;
   int 	d = -1;
   int 	t = -1;
   int 	e = -1;


   if (!kdms_create_segment(obj, KPDS_SEGMENT_VALUE))
      return FALSE;
   
   if (!kdms_set_attributes(obj, KPDS_SEGMENT_VALUE, 
			         KDMS_DIMENSION, KPDS_VALUE_DIM,
			    	 KDMS_INDEX_ORDER, order, 
			    	 NULL))
      return FALSE;

   /* -- don't want esize to be mapped -- */
   kpds_get_attribute(obj, KPDS_MAPPING_MODE, &oldmode);
   kpds_set_attribute(obj, KPDS_MAPPING_MODE, KUNMAPPED);
      
   /* -- get initial size and index order from existing segments -- */
   if (kdms_query_segment(obj, KPDS_SEGMENT_MASK))
   {
      kpds_get_attribute(obj, KPDS_MASK_SIZE, &w, &h, &d, &t, &e);

      kdms_get_attribute(obj, KPDS_SEGMENT_MASK, KDMS_INDEX_ORDER, &vorder);
      kdms_set_attribute(obj, KPDS_SEGMENT_VALUE,  KDMS_INDEX_ORDER, vorder);
   }
   else if (kdms_query_segment(obj, KPDS_SEGMENT_LOCATION))
   {
      kpds_get_attribute(obj, KPDS_LOCATION_SIZE, &w, &h, &d, NULL);

      kdms_get_attribute(obj, KPDS_SEGMENT_LOCATION, KDMS_INDEX_ORDER,&vorder);
      order[0] = vorder[0];
      order[1] = vorder[1];
      order[2] = vorder[2];
      order[3] = KTIME;
      order[4] = KELEMENTS;
      kdms_set_attribute(obj, KPDS_SEGMENT_VALUE, KDMS_INDEX_ORDER, order);
   }

   else if (kdms_query_segment(obj, KPDS_SEGMENT_TIME))
      kpds_get_attribute(obj, KPDS_TIME_SIZE, &t);

   else if (kdms_query_segment(obj, KPDS_SEGMENT_MAP))
      kpds_get_attribute(obj, KPDS_MAP_SIZE, NULL, NULL, &d, &t, &e);
   

   /* -- set initial size of the value segment -- */
   kpds_set_attribute(obj, KPDS_VALUE_SIZE, w, h, d, t, e);

   /* -- restore mapping mode -- */
   kpds_set_attribute(obj, KPDS_MAPPING_MODE, oldmode);

   return TRUE;
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _get_value_data
|
|       Purpose: This routine gets data from the value segment.
|                It is "map-aware" in that it understands the 
|		 mapping mode and will map the value data
|		 appropriately.
|
|         Input: object  - the object to retrieve data from
|	         segment - segment to retrieve data from
|		 begin   - the begin point for region to get :
|			   if the data is to be mapped, this
|		           begin point represents the mapped begin
|		 end     - the end point for the region to get :
|			   if the data is to be mapped, this
|		           end point represents the mapped end
|
|        Output: data   - pointer to already allocated data,
|			  or NULL if we should perform the allocation
|
|       Returns: pointer to the data being returned on success,
|		 NULL otherwise
|
|    Written By: Steve Kubica
|          Date: Apr 29, 1994 11:17
| Modifications:
|
------------------------------------------------------------*/
static kaddr
_get_value_data(
   kobject obj,
   char   *segment,
   int    *begin,
   int    *end,
   kaddr   data)
{
   kaddr vdata;
   kaddr rdata;

   /* -- verify that value segment exists -- */
   if (!kdms_query_segment(obj, segment))
      return NULL;
   
   if (kaps_mapping_mode(obj))
      vdata = kaps_get_mapped_value_data(obj, begin, end, data);
   else
      vdata = kdms_get_data(obj, segment, begin, end, data);

   if (kaps_masked_presentation(obj) == KUSE_SUBSTITUTE_VALUE)
      rdata = kaps_mask_data(obj, begin, end, vdata);
   else
      rdata = vdata;

   if (kaps_elements_first(obj))
   {
      kaddr ldata;
      
      ldata = kaps_transpose_elements_first(rdata, begin, end, _get_type(obj),
					    KPDS_VALUE_DIM);
      return ldata;
   }
      
   return rdata;
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _put_value_data
|
|       Purpose: This routine puts data into the value segment.
|                It is "map-aware" in that it understands the 
|		 mapping mode. It actually ignores the mapping 
|		 mode, since it is meaningless for a put,
|		 but it know enough to give debug warnings.
|
|         Input: obj - the object to put data into
|	         segment - the segment to put the data into
|		 begin   - the begin point for the put
|		 end     - the end point for the put
|		 data    - the data to be put
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica
|          Date: Apr 29, 1994 11:17
| Modifications:
|
------------------------------------------------------------*/
static int
_put_value_data(
   kobject obj,
   char   *segment,
   int    *begin,
   int    *end,
   kaddr   data)
{
   kaddr pdata = data;
   
   /* -- verify that the value segment exists -- */
   if (!kdms_query_segment(obj, segment))
      return FALSE;

   /* -- transpose if elements are first -- */
   if (kaps_elements_first(obj))
      pdata = kaps_transpose_elements_last(data, begin, end, _get_type(obj),
					   KPDS_VALUE_DIM);

   /* -- put the data the same way, ignoring mapping mode -- */
   return kdms_put_data(obj, segment, begin, end, pdata);
}


/* 
 *    ================================================================== 
 *    PRIVATE VALUE functions
 *    ==================================================================
 */

/*-----------------------------------------------------------
|
|  Routine Name: kpds_init_value_segment()
|
|       Purpose: Define the value segment.
|
|         Input: none
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica
|          Date: Dec 26, 1994
| Modifications:
|
------------------------------------------------------------*/
int
kpds_init_value_segment(void)
{
   int status = TRUE;

   status &= kdms_define_quasi_attribute(KDMS_OBJECT, 
					 KPDS_VALUE_SIZE, NULL,
					 _size_get,   _size_set, 
					 _size_match, _size_copy, 
					 _size_query, _size_print);


   status &= kdms_define_quasi_attribute(KDMS_OBJECT, 
					 KPDS_VALUE_DATA_TYPE, NULL,
					 _type_get,   _type_set, 
					 _type_match, _type_copy,
					 _type_query, _type_print);


   status &= kdms_define_segment(KPDS_SEGMENT_VALUE, _create_value,
				 NULL, NULL,
				 _get_value_data, _put_value_data,
				 NULL, NULL, NULL, NULL, NULL, NULL);

   return status;
}

/* 
 *    ================================================================== 
 *    PUBLIC VALUE functions
 *    ==================================================================
 */

/************************************************************
*
*  Routine Name: kpds_create_value - create a value segment within a 
*				     data object.
*
*       Purpose: This function is used to create a value segment
*		 within a specified data object.  The size of the
*		 value segment will be initialized to match any
*		 the sizes shared with any other polymorphic segments.
*
*		 It is considered an error to create a value segment
*		 if the object already contains one.
*
*         Input: object  - object in which to create the value segment.
*
*        Output: none
*
*       Returns: TRUE (1) if the value segment was successfully created, 
*		 FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Jeremy Worley & John Salas
*          Date: Sep 20, 1993 08:33
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kpds_create_value(
*		!   kobject object)
*
*************************************************************/
int
kpds_create_value(kobject object)
{
   return (kpds_init() && kdms_create_segment(object, KPDS_SEGMENT_VALUE));
}

/************************************************************
*
*  Routine Name: kpds_query_value - determine if the value segment
*				    exists in a data object.
*
*       Purpose: This function is used to determine if the value
*		 segment exists in a data object.  If value segment
*		 exists in the specified object, then this function
*		 will return TRUE.  If the object is invalid, or value
*		 data does not exist in the object, then this function
*		 will return FALSE.
*
*         Input: object  - data object to be queried.
*
*        Output: none
*
*       Returns: TRUE (1) if the value segment exists, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Jeremy Worley & John Salas
*          Date: Sep 20, 1993 08:33
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kpds_query_value(
*		!   kobject object)
*
*************************************************************/
int
kpds_query_value(kobject object)
{
   return (kpds_init() && kdms_query_segment(object, KPDS_SEGMENT_VALUE));
}

/************************************************************
*
*  Routine Name: kpds_destroy_value - destroy the value segment in a
*				      data object.
*
*       Purpose: This function is used to destroy the value segment
*		 contained within an object.  Once the value segment
*		 has been destroyed, any data or attributes associated
*		 with the value data will be lost forever.  A new
*		 value segment can be created in its place with the
*		 function kpds_create_value.
*
*		 If the value segment does not exist in the specified
*		 object, it is considered to be an error.
*
*         Input: object  - object from which to remove the value segment.
*
*        Output: value segment is destroyed
*
*       Returns: TRUE (1) if the value segment is successfully destroyed, 
*		 FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Jeremy Worley & John Salas
*          Date: Sep 20, 1993 08:33
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kpds_destroy_value(
*		!   kobject object)
*
*************************************************************/
int
kpds_destroy_value(kobject object)
{
   return (kpds_init() && kdms_destroy_segment(object, KPDS_SEGMENT_VALUE));
}

/************************************************************
*
*  Routine Name: kpds_initialize_value - preset value segment to constant
*
*       Purpose: This routine initializes the entire value segment to
*		 a specified value.  The last two arguments are real
*		 and imaginary components of a complex number that is
*		 the value that the data will be initialized to.  The
*		 data type of the value segment at the time this
*		 function is called will determine how the complex
*		 number is used.  A non-complex datatype means that
*		 the real part only will be used.  A
*		 non-floating-point data type means that the integer
*		 part of the real argument will be used.
*
*         Input: object - object to initialize value data on
*		 real   - real part of constant
*		 imag   - imaginary part of constant
*
*        Output: value segment is modified in place
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions:
*    Written By: Jeremy Worley
*          Date: Aug 03, 1994 14:53
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kpds_initialize_value(
*                !   kobject object,
*                !   double real,
*                !   double imag)
*
*************************************************************/
int
kpds_initialize_value(
   kobject object, 
   double  real, 
   double  imag)
{
   int  status = TRUE;
   int  oldmode;

   if (!kpds_init())
      return FALSE;

   /* -- don't want esize to be mapped -- */
   status &= kpds_get_attribute(object, KPDS_MAPPING_MODE, &oldmode);
   status &= kpds_set_attribute(object, KPDS_MAPPING_MODE, KUNMAPPED);
   
   status &= kdut_initialize_segment(object, KPDS_SEGMENT_VALUE, real, imag);

   /* -- restore mapping mode -- */
   status &= kpds_set_attribute(object, KPDS_MAPPING_MODE, oldmode);

   return status;
}

/************************************************************
*
*  Routine Name: kpds_copy_value_data - copy all value data from
*                                       one object to another object.
*
*       Purpose: This function copies all of the data contained in the
*		 value segment of one object into the value segment
*		 contained in another object.  If the value segment
*		 exists in the destination object, then its data will
*		 be replaced with the data from the source object.  If
*		 the value segment does not exist in the destination
*		 object, it will be created.
*
*		 The value data can be optionally copied through the
*		 presentations of the two data objects.  This implies
*		 that any presentation stages which are normally
*		 invoked during a kpds_get_data on the source object
*		 or a kpds_put_data call on the destination object
*		 will be used for the copy.
*
*		 Any of the following presentation stages can be
*		 invoked on either the source object or the
*		 destination object : casting, scaling, normalizing,
*		 padding, and interpolating.
*		 
*		 The following presentation stages can be invoked on
*		 only the source object : mapping, axis assignment,
*		 position, and offset.  The resulting copy will be
*		 mapped through the map and transposed to be in the
*		 default axis ordering.  The resulting copy will also
*		 appear to be shifted by the source position and
*		 offset.
*
*		 These presentation stages are brought into the data
*		 pipeline by setting the appropriate presentation
*		 attributes on the data objects. Please see the
*		 chapter on polymorphic data services in the data
*		 services manual for more information on the various
*		 presentation attributes.
*
*		 The following example may help in visualizing the
*		 process.  The source object has mapping mode set
*		 to map the data, a presentation data type different
*		 from the physical data type, and a presentation size
*		 different from the physical size.  The destination
*		 object only has a different presentation and physical
*		 data type.
*
*		 !	        _______________________
*		 !	       | _____________________ |
*		 !	  _____||______  ---->        ||
*		 !	 |___mapping___|              ||
*		 !	  _____||______               ||
*		 !	 |interpolating|              ||
*		 !	  _____||______          _____||______
*		 !	 |___casting___|        |___casting___|
*		 !	   ____||_____            ____||_____
*		 !	  |           |          |           |
*		 !	  |  source   |        	 |destination|
*		 !	  |   data    |        	 |   data    |
*		 !	  |___________|        	 |___________|
*	         !
*
*		 In the above example, the data resulting from the
*		 copy will be interpolated to a new size, cast to a
*		 new data type, and mapped through the object's map
*		 before being copied into the destination object.
*		 When being copied into the destination object, the
*		 data will be cast yet again before it finally is
*		 stored.
*
*		 If the presentation and physical layers of the
*		 destination object are coupled then the destination
*		 attributes will be ignored and the source
*		 presentation attributes will be propogated to the
*		 destination physical layer.  The copy, in essence,
*		 will be performed only through the source
*		 presentation, with the destination physical and
*		 presentation layers taking on the characteristics of
*		 the source presentation.  By default, output objects
*		 are not coupled, so this behavior is typical.
*		 
*		 The copy need not be performed using the presentation.
*		 If the data is not copied through the presentation,
*		 the destination data will be a direct copy of the
*		 physical source data.  The physical size and data
*		 type of the value segment will be reflected in the
*		 destination data object from the source to the
*		 destination since they describe the physical state of
*		 the data.
*
*         Input: object1 - the object that serves as a source for the data.
*
*		 copy_through_presentation - if set to TRUE, the copy will
*			 		     be performed through the
*				  	     presentation of the source 
*					     and destination objects.
*					     if set to FALSE, the copy will
*					     be a direct copy of the physical
*					     data.
*
*        Output: object2 - the object that will serve as a destination for
*			   the copy operation.
*
*       Returns: TRUE (1) if copy was successful, FALSE (0) otherwise
*
*  Restrictions:
*    Written By: Jeremy Worley and Steve Kubica
*          Date: Jan 2, 1995
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kpds_copy_value_data(
*               !   kobject object1,
*               !   kobject object2,
*		!   int     copy_through_presentation)
*
*************************************************************/
int
kpds_copy_value_data(
   kobject object1,
   kobject object2,
   int     copy_through_presentation)
{
   kobject src = NULL;
   kobject dest = object2;
   
   int w, h, d, t, e;
   int ow, oh, od, ot, oe;
   int wps, hps, dps, tps, eps;
   int wof, hof, dof, tof, eof;
   int i;
   int typ;
   int num_regions;
   int old_coupling;
   int old_masking;
   kaddr data = NULL;
   int order[KPDS_VALUE_DIM] = {KWIDTH, KHEIGHT, KDEPTH, KTIME, KELEMENTS};

   if (!kpds_init())
      return FALSE;

   /* -- sanity check -- */
   if (object1 == NULL || object2 == NULL)
   {
      /* -- set invalid object errorno here -- */
      return FALSE;
   }

   if (!copy_through_presentation)
      return kdut_copy_segment_data(object1, object2, KPDS_SEGMENT_VALUE);

   /* -- make sure the segment exists in the source -- */
   if (!kdms_query_segment(object1, KPDS_SEGMENT_VALUE))
   {
      /* -- set cool errno here -- */
      return FALSE;
   }

   /* -- make sure the segment exists in the destination -- */
   if (!kdms_query_segment(object2, KPDS_SEGMENT_VALUE))
      if (!kdms_create_segment(object2, KPDS_SEGMENT_VALUE))
	 return FALSE;

   /* -- get the position and offset before we make the reference -- */
   if (!kpds_get_attributes(object1,
			    KPDS_VALUE_POSITION, &wps, &hps, &dps, &tps, &eps,
			    KPDS_VALUE_OFFSET,   &wof, &hof, &dof, &tof, &eof,
			    NULL))
      return FALSE;

   /* -- create a reference to avoid side effects on the source -- */
   src  = kdms_reference(object1);

   /* -- get the old coupling on the dest so we can restore it later -- */
   if (!kdms_get_attribute(dest, KPDS_SEGMENT_VALUE, 
			         KDMS_COUPLING, &old_coupling))
      return FALSE;
   
   /* -- want to affect physical layer on the destination -- */
   if (!kdms_set_attribute(dest, KPDS_SEGMENT_VALUE, KDMS_COUPLING, KCOUPLED))
      return FALSE;

   /* -- the destination should not be mapped -- */
   if (!kpds_set_attribute(dest, KPDS_MAPPING_MODE, KUNMAPPED))
      return FALSE;

   /* -- the resulting copy will have the default index order -- */
   kdms_set_attribute(dest, KPDS_SEGMENT_VALUE, KDMS_INDEX_ORDER, order);

   /* -- get the old masked value presentation so we can restore it later -- */
   if (!kpds_get_attribute(dest, KPDS_MASKED_VALUE_PRESENTATION, 
			   &old_masking))
      return FALSE;

#if 0      
   /* -- want to use the original value data for copying -- */
   if (!kpds_set_attribute(src, KPDS_MASKED_VALUE_PRESENTATION,  
			   KUSE_ORIGINAL))
      return FALSE;
#endif

   /* -- set up region sizes - also copy size and data type -- */
   if (!kpds_get_attributes(src,
			    KPDS_VALUE_SIZE, &w, &h, &d, &t, &e,
			    KPDS_VALUE_DATA_TYPE, &typ,
			    KPDS_VALUE_OPTIMAL_REGION_SIZE, &ow, &oh,
			    &od, &ot, &oe, &num_regions, 
			    NULL))
      return FALSE;
   if (!kpds_set_attributes(src,
			    KPDS_VALUE_POSITION, wps, hps, dps, tps, eps,
			    KPDS_VALUE_OFFSET,   wof, hof, dof, tof, eof,
			    KPDS_VALUE_REGION_SIZE, ow, oh, od, ot, oe,
			    NULL))
      return FALSE;
   if (!kpds_set_attributes(dest,
			    KPDS_VALUE_SIZE, w, h, d, t, e,
			    KPDS_VALUE_DATA_TYPE, typ,
			    KPDS_VALUE_REGION_SIZE, ow, oh, od, ot, oe,
			    KPDS_VALUE_POSITION, 0, 0, 0, 0, 0,
			    KPDS_VALUE_OFFSET, 0, 0, 0, 0, 0,
			    NULL))
      return FALSE;

   /* -- copy the data using the pds get'n'put data calls -- */
   for (i = 0; i < num_regions; i++)
   {
      data = kpds_get_data(src, KPDS_VALUE_REGION, data);
      kpds_put_data(dest, KPDS_VALUE_REGION, data);
   }
   kfree(data);
   
   /* reset the masked value presentation on the original */
   if (!kpds_set_attribute(src, KPDS_MASKED_VALUE_PRESENTATION, old_masking))
      return FALSE;

   /* -- reset the position and offset to be zero -- */
   if (!kpds_set_attributes(dest, 
			    KPDS_VALUE_POSITION, 0, 0, 0, 0, 0,
			    KPDS_VALUE_OFFSET, 0, 0, 0, 0, 0, 
			    NULL))
      return FALSE;

   /* -- set the old coupling back on the segment -- */
   kdms_set_attribute(dest, KPDS_SEGMENT_VALUE, KDMS_COUPLING, old_coupling);

   /* -- close the source reference -- */
   kdms_close(src);

   return TRUE;
}

/************************************************************
*
*  Routine Name: kpds_copy_value_attr - copy all value attributes from
*	                                one object to another object.
*
*       Purpose: This function copies all the value attributes from 
*		 one object to another object.
*
*		 If the destination object does not contain a value
*		 segment, then this function will create a value
*		 segment, and initialize its size and data type
*		 attributes to those in the source object.  If the
*		 value data already exists in the destination object,
*		 then the presentation attributes will be set to the
*		 source object's settings.
*
*		 The destination's physical attributes will change
*		 depending on the coupling of the data objects.
*		 If the destination is coupled, then the physical
*		 attributes will be changed as well.  Any data
*		 contained in the destination will be cast, rescaled,
*		 or resized to match its new physical attributes.
*
*         Input: source_object - the object that serves as a
*				 source for the attributes.
*
*        Output: destination_object - the object that serves as
*				      a destination for the
*				      operation.
*
*       Returns: TRUE (1) if copy was successful, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Jeremy Worley & John Salas
*          Date: Sep 19, 1993 15:43
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kpds_copy_value_attr(
*		!   kobject object1,
*		!   kobject object2)
*
*************************************************************/
int
kpds_copy_value_attr(
   kobject object1,
   kobject object2)
{
   return (kpds_init() && 
	   kdut_copy_segment_attr(object1, object2, KPDS_SEGMENT_VALUE));
}

/************************************************************
*
*  Routine Name: kpds_copy_value - copy the value segment from
*		                   one object to another.
*
*       Purpose: This function copies all of the attributes and data
*		 contained in the value segment of one object into the
*		 value segment contained in another object.  If the
*		 value segment exists in the destination object, then
*		 its data will be replaced with the data from the
*		 source object.  If the value segment does not exist
*		 in the destination object, it will be created.
*
*		 All value segment attributes will be copied from one
*		 object to the other.  This includes presentation
*		 attributes, such as position and offset.
*
*		 The value data can be optionally copied through the
*		 presentations of the two data objects.  This implies
*		 that any presentation stages which are normally
*		 invoked during a kpds_get_data on the source object
*		 or a kpds_put_data call on the destination object
*		 will be used for the copy.
*
*		 Any of the following presentation stages can be
*		 invoked on either the source object or the
*		 destination object : casting, scaling, normalizing,
*		 padding, and interpolating.
*		 
*		 The following presentation stages can be invoked on
*		 only the source object : mapping, axis assignment,
*		 position, and offset.  The resulting copy will be
*		 mapped through the map and transposed to be in the
*		 default axis ordering.  The resulting copy will also
*		 appear to be shifted by the source position and
*		 offset.
*
*		 These presentation stages are brought into the data
*		 pipeline by setting the appropriate presentation
*		 attributes on the data objects. Please see the
*		 chapter on polymorphic data services in the data
*		 services manual for more information on the various
*		 presentation attributes.
*
*		 The following example may help in visualizing the
*		 process.  The source object has mapping mode set
*		 to map the data, a presentation data type different
*		 from the physical data type, and a presentation size
*		 different from the physical size.  The destination
*		 object only has a different presentation and physical
*		 data type.
*
*		 !	        _______________________
*		 !	       | _____________________ |
*		 !	  _____||______  ---->        ||
*		 !	 |___mapping___|              ||
*		 !	  _____||______               ||
*		 !	 |interpolating|              ||
*		 !	  _____||______          _____||______
*		 !	 |___casting___|        |___casting___|
*		 !	   ____||_____            ____||_____
*		 !	  |           |          |           |
*		 !	  |  source   |        	 |destination|
*		 !	  |   data    |        	 |   data    |
*		 !	  |___________|        	 |___________|
*	         !
*
*		 In the above example, the data resulting from the
*		 copy will be interpolated to a new size, cast to a
*		 new data type, and mapped through the object's map
*		 before being copied into the destination object.
*		 When being copied into the destination object, the
*		 data will be cast yet again before it finally is
*		 stored.
*
*		 If the presentation and physical layers of the
*		 destination object are coupled then the destination
*		 attributes will be ignored and the source
*		 presentation attributes will be propogated to the
*		 destination physical layer.  The copy, in essence,
*		 will be performed only through the source
*		 presentation, with the destination physical and
*		 presentation layers taking on the characteristics of
*		 the source presentation.  By default, output objects
*		 are not coupled, so this behavior is typical.
*		 
*		 The copy need not be performed using the presentation.
*		 If the data is not copied through the presentation,
*		 the destination data will be a direct copy of the
*		 physical source data.  The physical size and data
*		 type of the value segment will be reflected in the
*		 destination data object from the source to the
*		 destination since they describe the physical state of
*		 the data.
*
*		 This function is equivalent to performing successive
*		 calls to kpds_copy_value_attr and kpds_copy_value_data.
*
*         Input: object1 - the object that serves as a source for the value
*			   segment.
*
*		 copy_through_presentation - if set to TRUE, the copy will
*			 		     be performed through the
*				  	     presentation of the source 
*					     and destination objects.
*					     if set to FALSE, the copy will
*					     be a direct copy of the physical
*					     data.
*
*        Output: object2 - the object that will serve as a destination for
*			   the copy operation.
*
*       Returns: TRUE (1) if copy was successful, FALSE (0) otherwise
*
*  Restrictions:
*    Written By: Jeremy Worley & Steve Kubica
*          Date: Sep 19, 1993 15:40
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kpds_copy_value(
*               !   kobject object1,
*               !   kobject object2,
*		!   int     copy_through_presentation)
*
*************************************************************/
int
kpds_copy_value(
   kobject object1,
   kobject object2,
   int     copy_through_presentation)
{
   return (kpds_init() && 
	   kpds_copy_value_attr(object1, object2) &&
	   kpds_copy_value_data(object1, object2, copy_through_presentation));
}
