 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>          Polymorphic Mapping Information
   >>>>
   >>>>   Static:
   >>>>			_kaps_get_map_index()
   >>>>  Private:
   >>>>			kaps_map_data()
   >>>>			kaps_get_mapped_value_data() 
   >>>>			kaps_get_mapped_mask_data()
   >>>>			kaps_mask_data()
   >>>>			kaps_get_rectilinear_data()
   >>>>			kaps_transpose_elements_first()
   >>>>			kaps_mapping_mode()
   >>>>			kaps_masked_presentation()
   >>>>			kaps_location_grid()
   >>>>			kaps_elements_first()
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"

extern int width_token;
extern int height_token;
extern int depth_token;

#define NUM_ELEMENTS(a) (a[0] * a[1] * a[2] * a[3] * a[4])

/*-----------------------------------------------------------
|
|  Routine Name: _kaps_get_map_index
|
|       Purpose: This routine determines the appropriate index
|                into the linear map array which will return the
|                desired mapped value.  It does this by determining
|		 the appropriate map width position and map height
|		 position. The map width position is determined
|		 to be the modulo of the current destination element
|		 position, and the map height position is determined
|		 by the value in the value data at the current
|		 unmapped value position.  Still confused?  Then
|		 look at the code... it's not as bad as it sounds.
|
|		 If the value being mapped falls outside of the
|		 range of the map height, then the map_index returned
|		 is set to -1.
|
|         Input: index  - the linear index into the destination
|                size   - the size of the destination - this represents
|			  the mapped value size of the data being 
|			  retrieved.
|
|		 value  - the value data to look into.  the zero
|			  point of this value data "aligns" with
|			  the zero point of the destination.
|
|		 vsize     - the size of the value data 
|		 map_size  - the size of the map data
|
|        Output: none 
|
|       Returns: the index into the map data on success,
|		 -1 otherwise
|
|    Written By: Jeremy Worley and Steve Kubica
|          Date: Nov 22, 1993 07:35
| Modifications:
|
------------------------------------------------------------*/
static int
_kaps_get_map_index(
   int index,
   int *mapped_value_size,
   int *value,
   int *value_size,
   int *map_size)
{
   int p[KPDS_VALUE_DIM];
   int acc = 1;
   int mw;
   int mh;
   int me;
   int md;
   int mt;


   /* 
    *  compute the position into the destination according 
    *  to the given index and the destination size
    *  note that this size is the *mapped* size!
    */
   p[KWID] = index % mapped_value_size[KWID];
   p[KHGT] = index/ (acc *= mapped_value_size[KWID]) % mapped_value_size[KHGT];
   p[KDEP] = index/ (acc *= mapped_value_size[KHGT]) % mapped_value_size[KDEP];
   p[KTIM] = index/ (acc *= mapped_value_size[KDEP]) % mapped_value_size[KTIM];
   p[KELM] = index/ (acc *= mapped_value_size[KTIM]) % mapped_value_size[KELM];

   /* we can now determine where we are with respect to the map width */
   mw = p[KELM] % map_size[KMAP_WID];

   /* unmap the element position */
   p[KELM] = p[KELM] / map_size[KMAP_WID];     

   /* find the value "value" that is used to index into the map height */
   mh = value[p[KWID] + value_size[KWID] * (p[KHGT] +
					    value_size[KHGT] * (p[KDEP] +
					    value_size[KDEP] * (p[KTIM] +
					    value_size[KTIM] * p[KELM])))];

   /* if this value is beyond the extent of the map height, then fail */
   if (mh < 0 || mh >= map_size[KMAP_HGT])
      return -1;

   /* 
    * the index into the map depth, time, and elements will be 
    * equal the current unmapped value element position unless the 
    * map element size is equal to 1, in which case it's just 1.
    */
   md = (map_size[KMAP_DEP] == 1) ? 0 : p[KDEP];
   mt = (map_size[KMAP_TIM] == 1) ? 0 : p[KTIM];
   me = (map_size[KMAP_ELM] == 1) ? 0 : p[KELM];	/* ok to be unmapped */

   return (mw + map_size[KMAP_WID] * (mh +
		map_size[KMAP_HGT] * (md +
		map_size[KMAP_DEP] * (mt +
		map_size[KMAP_TIM] * me))));
}

/*-----------------------------------------------------------
|
|  Routine Name: kaps_map_data
|
|       Purpose: This routine gets mapped value data through a given map.
|
|                The map indexing is always assumed to be 
|		 KMAP_WIDTH, KMAP_HEIGHT, KMAP_ELEMENTS, KMAP_DEPTH,
|		 and KMAP_TIME.   The mapping is performed by 
|		 replacing each element in the value data with the
|		 the KMAP_WIDTH vector indicated by the KMAP_HEIGHT
|		 equal to the value of the element.  The elements,
|		 depth, and time are all made equal to the element,
|		 Depth, and time position of the original value.
|
|         Input: value      - the value data to be mapped cast to integers
|                value_size - the size of the value data
|
|                map        - the map to map the values through
|			      this data will only be a subset of the
|		              original map but will contain enough
|			      of the map to map all the values we have.
|
|                map_size   - the size of the map
|                map_type   - the data type of the map
|
|                dest_size  - the mapped size of the destination data
|	
|		 mpad_r     - real map pad value to use when map indices
|		 	      go out of bounds
|
|		 mpad_i     - imaginary map pad value to use when map indices
|		 	      go out of bounds
|
|        Output: dest       - pointer to the mapped data.  it is assumed
|			      that the enought space has been allocated
|			      to contain the mapped data.
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica
|          Date: Apr 29, 1994 11:17
| Modifications:
|
------------------------------------------------------------*/
int
kaps_map_data(
   int *mapped_value_size,
   int *value,
   int *value_size,
   kaddr map,
   int *map_size,
   int map_type,
   int num_elements,
   kaddr dest,
   double mpad_r,
   double mpad_i)
{
   int i;
   int mi;


   switch (map_type)
   {
      case KBYTE:
	 {
	    char *mp = (char *)map;
	    char *data = (char *)dest;

	    /* index through the elements in the destination */
	    for (i = 0; i < num_elements; i++)
	    {
	       mi = _kaps_get_map_index(i, mapped_value_size, value,
					value_size, map_size);

	       data[i] = (mi == -1) ? (char)mpad_r : mp[mi];
	    }

	 }
	 break;

      case KUBYTE:
	 {
	    unsigned char *mp = (unsigned char *)map;
	    unsigned char *data = (unsigned char *)dest;

	    /* index through the elements in the destination */
	    for (i = 0; i < num_elements; i++)
	    {
	       mi = _kaps_get_map_index(i, mapped_value_size, value,
					value_size, map_size);

	       data[i] = (mi == -1) ? (unsigned char)mpad_r : mp[mi];
	    }

	 }
	 break;

      case KSHORT:
	 {
	    short *mp = (short *)map;
	    short *data = (short *)dest;

	    /* index through the elements in the destination */
	    for (i = 0; i < num_elements; i++)
	    {
	       mi = _kaps_get_map_index(i, mapped_value_size, value,
					value_size, map_size);

	       data[i] = (mi == -1) ? (short)mpad_r : mp[mi];
	    }

	 }
	 break;

      case KUSHORT:
	 {
	    unsigned short *mp = (unsigned short *)map;
	    unsigned short *data = (unsigned short *)dest;

	    /* index through the elements in the destination */
	    for (i = 0; i < num_elements; i++)
	    {
	       mi = _kaps_get_map_index(i, mapped_value_size, value,
					value_size, map_size);

	       data[i] = (mi == -1) ? (unsigned short)mpad_r : mp[mi];
	    }

	 }
	 break;

      case KINT:
	 {
	    int *mp = (int *)map;
	    int *data = (int *)dest;

	    /* index through the elements in the destination */
	    for (i = 0; i < num_elements; i++)
	    {
	       mi = _kaps_get_map_index(i, mapped_value_size, value,
					value_size, map_size);

	       data[i] = (mi == -1) ? (int)mpad_r : mp[mi];
	    }

	 }
	 break;

      case KUINT:
	 {
	    unsigned int *mp = (unsigned int *)map;
	    unsigned int *data = (unsigned int *)dest;

	    /* index through the elements in the destination */
	    for (i = 0; i < num_elements; i++)
	    {
	       mi = _kaps_get_map_index(i, mapped_value_size, value,
					value_size, map_size);

	       data[i] = (mi == -1) ? (unsigned int)mpad_r : mp[mi];
	    }

	 }
	 break;

      case KLONG:
	 {
	    long *mp = (long *)map;
	    long *data = (long *)dest;

	    /* index through the elements in the destination */
	    for (i = 0; i < num_elements; i++)
	    {
	       mi = _kaps_get_map_index(i, mapped_value_size, value,
					value_size, map_size);

	       data[i] = (mi == -1) ? (long)mpad_r : mp[mi];
	    }

	 }
	 break;

      case KULONG:
	 {
	    unsigned long *mp = (unsigned long *)map;
	    unsigned long *data = (unsigned long *)dest;

	    /* index through the elements in the destination */
	    for (i = 0; i < num_elements; i++)
	    {
	       mi = _kaps_get_map_index(i, mapped_value_size, value,
					value_size, map_size);

	       data[i] = (mi == -1) ? (unsigned long)mpad_r : mp[mi];
	    }

	 }
	 break;

      case KFLOAT:
	 {
	    float *mp = (float *)map;
	    float *data = (float *)dest;

	    /* index through the elements in the destination */
	    for (i = 0; i < num_elements; i++)
	    {
	       mi = _kaps_get_map_index(i, mapped_value_size, value,
					value_size, map_size);

	       data[i] = (mi == -1) ? (float)mpad_r : mp[mi];
	    }

	 }
	 break;

      case KDOUBLE:
	 {
	    double *mp = (double *)map;
	    double *data = (double *)dest;

	    /* index through the elements in the destination */
	    for (i = 0; i < num_elements; i++)
	    {
	       mi = _kaps_get_map_index(i, mapped_value_size, value,
					value_size, map_size);

	       data[i] = (mi == -1) ? (double)mpad_r : mp[mi];
	    }

	 }
	 break;

      case KCOMPLEX:
	 {
	    kcomplex *mp = (kcomplex *) map;
	    kcomplex *data = (kcomplex *) dest;

	    /* index through the elements in the destination */
	    for (i = 0; i < num_elements; i++)
	    {
	       mi = _kaps_get_map_index(i, mapped_value_size, value,
					value_size, map_size);

	       (data[i]).r = (mi == -1) ? (float)mpad_r : (mp[mi]).r;
	       (data[i]).i = (mi == -1) ? (float)mpad_i : (mp[mi]).i;
	    }

	 }
	 break;

      case KDCOMPLEX:
	 {
	    kdcomplex *mp = (kdcomplex *) map;
	    kdcomplex *data = (kdcomplex *) dest;

	    /* index through the elements in the destination */
	    for (i = 0; i < num_elements; i++)
	    {
	       mi = _kaps_get_map_index(i, mapped_value_size, value,
					value_size, map_size);

	       (data[i]).r = (mi == -1) ? (double)mpad_r : (mp[mi]).r;
	       (data[i]).i = (mi == -1) ? (double)mpad_i : (mp[mi]).i;
	    }

	 }
	 break;

      default:
	 errno = KINVALID_DATATYPE;
	 kfree(value);
	 kfree(map);
	 return FALSE;
   }
   return TRUE;
}

/*-----------------------------------------------------------
|
|  Routine Name: kaps_get_mapped_value_data
|
|       Purpose: This routine gets mapped data through
|		 the value segment.  The given begin and
|                end positions reflect the "mapped" size
|                of the value segment and it is up to us
|                to resolve where these points go in the
|                actual value and map segment.
|
|         Input: obj    - the object to retrieve data from
|                begin  - the mapped begin point into the value
|                end    - the mapped end point into the value
|		 dest   - pointer to already allocated data,
|			  or NULL if we should perform the allocation
|        Output: none
|
|       Returns: pointer to the mapped data on sucess, 
|		 NULL otherwise
|
|    Written By: Steve Kubica
|          Date: Apr 29, 1994 11:17
| Modifications:
|
------------------------------------------------------------*/
kaddr
kaps_get_mapped_value_data(
   kobject obj,
   int *begin,
   int *end,
   kaddr dest)
{
   kaddr map;
   int *value;
   int  value_size[KPDS_VALUE_DIM];
   int  mapped_value_size[KPDS_VALUE_DIM];
   int  map_size[KPDS_MAP_DIM];
   int *msize;
   int  mtype;
   int  vtype;
   int  mb[KPDS_MAP_DIM];
   int  me[KPDS_MAP_DIM];
   int  i;
   int  total_size;
   int  status;

   double vpad_r;
   double vpad_i;

   double mpad_r;
   double mpad_i;

   int unmapped_begin_e;
   int unmapped_end_e;

   int  abeg[KPDS_MAP_DIM];
   int  aend[KPDS_MAP_DIM];

   /* get our own copy of these */
   kmemcpy(abeg, begin, KPDS_VALUE_DIM * sizeof(int));
   kmemcpy(aend, end,   KPDS_VALUE_DIM * sizeof(int));

   /* get a bunch of important map info */
   kdms_get_attributes(obj, KDMS_SEGMENT_MAP,
		       KDMS_DATA_TYPE, &mtype,
		       KDMS_SIZE, &msize, NULL);

   /* determine the amount of mapped value data which is expected back */
   for (i = 0; i < KPDS_VALUE_DIM; i++)
      mapped_value_size[i] = end[i] - begin[i] + 1;

   total_size = NUM_ELEMENTS(mapped_value_size);

   /*
    *   retrieve the map data we need first --
    *     we will determine the begin and points of the map data
    *     in order to get the minimum span of map data that is
    *     required to completely map the data.  This will always
    *     include all of the map height.
    */

   /* 
    * determine the begin and end points for the map data to retrieve --
    * we want to get as little as the map as possible, so look at where
    * the begining point is to determine how to minimize this 
    */
   unmapped_begin_e = begin[KELM] / msize[KMAP_WID];
   unmapped_end_e = end[KELM] / msize[KMAP_WID];

   /* 
    *  we need all of height because that is where the map indexes into;
    *  we need only the part of the width that corresponds to the span
    *  of mapped value elements which we are mapping.
    */
   mb[KMAP_WID] = begin[KELM] % msize[KMAP_WID]; /* span along map width */
   mb[KMAP_HGT] = 0;
   mb[KMAP_DEP] = (msize[KMAP_DEP] == 1) ? 0 : begin[KDEP];
   mb[KMAP_TIM] = (msize[KMAP_TIM] == 1) ? 0 : begin[KTIM];
   mb[KMAP_ELM] = (msize[KMAP_ELM] == 1) ? 0 : unmapped_begin_e;

   me[KMAP_WID] = end[KELM] % msize[KMAP_WID];  /* span along map width */
   me[KMAP_HGT] = msize[KMAP_HGT] - 1;
   me[KMAP_DEP] = (msize[KMAP_DEP] == 1) ? 0 : end[KDEP];
   me[KMAP_TIM] = (msize[KMAP_TIM] == 1) ? 0 : end[KTIM];
   me[KMAP_ELM] = (msize[KMAP_ELM] == 1) ? 0 : unmapped_end_e;

   /* compute the effective size of the map data which we will retreive */
   for (i = 0; i < KPDS_MAP_DIM; i++)
      map_size[i] = me[i] - mb[i] + 1;

   /* get map data - this is all the data necessary to fully map the data */
   map = kdms_get_data(obj, KDMS_SEGMENT_MAP, mb, me, NULL);


   /* "unmap" the begin and end points - this alters the begin and end! */
   abeg[KELM] = unmapped_begin_e;
   aend[KELM] = unmapped_end_e;

   /* determine the amount of unmapped value data which we need to map */
   for (i = 0; i < KPDS_VALUE_DIM; i++)
      value_size[i] = aend[i] - abeg[i] + 1;

   /* get the old value type and pad value so we can put them back later! */
   kdms_get_attributes(obj, KDMS_SEGMENT_VALUE, 
                            KDMS_PAD_VALUE, &vpad_r, &vpad_i,
			    KDMS_DATA_TYPE, &vtype,
			    NULL);

   /* set the value data type to int so that we can use it as an index */
   kdms_set_attribute(obj, KDMS_SEGMENT_VALUE, KDMS_DATA_TYPE, KINT);

   /* 
    *  we want to know if the value is padding so we can replace it with the
    *  map pad -- set it to something that won't map into the map at all.
    */
   kdms_set_attribute(obj, KDMS_SEGMENT_VALUE, KDMS_PAD_VALUE, -1.0, 0,0);

   /* now we can get the raw value data which we want to map */
   value = kdms_get_data(obj, KDMS_SEGMENT_VALUE, abeg, aend, NULL);

   /* set the value data type and pad value back to what they were before */
   kdms_set_attributes(obj, KDMS_SEGMENT_VALUE, 
                            KDMS_PAD_VALUE, vpad_r, vpad_i,
			    KDMS_DATA_TYPE, vtype,
			    NULL);
   /* 
    * the mapped data will be the number of mapped value points times the 
    * size of the map data type - allocate this if it hasn't been allocated
    */
   if (dest == NULL)
      dest = (kaddr) kmalloc((unsigned)(total_size * kdata_size(mtype)));

   if (dest == NULL)
   {
      errno = KMEMORY_ALLOCATION;
      return NULL;
   }

   /* in case the map goes out of bounds, we need to use the pad values */
   kdms_get_attribute(obj, KDMS_SEGMENT_MAP, KDMS_PAD_VALUE, &mpad_r, &mpad_i);

   /* do the mapping magic...  you may applaud at any time */
   status = kaps_map_data(mapped_value_size, value, value_size, map, map_size,
			  mtype, total_size, dest, mpad_r, mpad_i);

   /*
    * not the most efficient way to go through life, but if ya want magic
    * mapping, then you gotta pay the price...
    */
   kfree(map);
   kfree(value);

   if (status)
      return dest;
   else
      return NULL;
}

/*-----------------------------------------------------------
|
|  Routine Name: kaps_get_mapped_mask_data
|
|       Purpose: This routine gets mapped data from
|		 the mask segment.  The given begin and
|                end positions reflect the "mapped" size
|                of the mask segment and it is up to us
|                to resolve where these points go in the
|                actual mask segment.
|	
|		 the begin and end points are altered
|		 as a side effect of this routine.
|
|         Input: obj - the object to retrieve data from
|                begin  - the mapped begin point into the value
|                end    - the mapped end point into the value
|		 dest   - pointer to already allocated data,
|			  or NULL if we should perform the allocation
|        Output: none
|
|       Returns: pointer to the mapped data on sucess, 
|		 NULL otherwise
|
|    Written By: Steve Kubica
|          Date: Apr 29, 1994 11:17
| Modifications:
|
------------------------------------------------------------*/
kaddr
kaps_get_mapped_mask_data(
   kobject obj,
   int    *begin,
   int    *end,
   kaddr   dest)
{
   int   b[KPDS_MASK_DIM];
   int   e[KPDS_MASK_DIM];
   int  *mask_size;
   int   mask_type;
   int  *map_size;
   int   shift;
   int   num_blocks;
   int   bsize;
   int   mw;
   int   i;
   kaddr rdata;
   char *sptr;
   char *dptr;

   kdms_get_attribute(obj, KPDS_SEGMENT_MAP, KDMS_SIZE, &map_size);
   mw = map_size[KWID];
   
   kdms_get_attributes(obj, KPDS_SEGMENT_MASK, 
		       KDMS_SIZE,      &mask_size,
		       KDMS_DATA_TYPE, &mask_type, 
		       NULL);

   /*
    *  the mask basically just needs to be pixel-replicated in the
    *  elements direction so it appears that it actually appears
    *  to have element size which includes the map width.
    */

   /* -- spin off a couple of working copies of the begin and end array -- */
   kmemcpy(b, begin, KPDS_MASK_DIM * sizeof(int));
   kmemcpy(e, end,   KPDS_MASK_DIM * sizeof(int));

   /* -- shift the region we are getting so it has all the data we need -- */
   shift    = begin[KELM] - begin[KELM]/mw;
   b[KELM] -= shift;
   e[KELM] -= shift;

   /* -- data we get is all there, but it is in the wrong order :( -- */
   rdata = kdms_get_data(obj, KPDS_SEGMENT_MASK, b, e, dest);

   /* -- so we need to recopy blocks of odata so it comes out pixel-rep'd -- */

   /* -- the overall block size is everything *except* elements -- */
   bsize = (e[0] - b[0] + 1) * 
           (e[1] - b[1] + 1) *
           (e[2] - b[2] + 1) *
           (e[3] - b[3] + 1) * kdata_size(mask_type);

   num_blocks = e[4] - b[4] + 1;

   /* -- we may be done! -- */
   if (num_blocks == 1)
      return rdata;

   /* -- set up a destination pointer to be at the last block -- */
   dptr = ((char *) rdata) + (bsize * (num_blocks - 1));
   sptr = ((char *) rdata) + (bsize * (end[4] / mw));

   /* -- pixel-rep in place, filling out the end of the array first -- */
   for (i = end[4]; i >= begin[4]; i--, dptr -= bsize)
   {
      kmemcpy(dptr, sptr, bsize);
      
      if (i % mw)
	 sptr -= bsize;
   }
   
   return rdata;
}

/*-----------------------------------------------------------
|
|  Routine Name: kaps_mask_data
|
|       Purpose: This routine masks out a given data array
|		 according to the data from the mask segment.
|
|         Input: obj - the object to retrieve data from
|                begin  - the mapped begin point into the value
|                end    - the mapped end point into the value
|		 dest   - pointer to already allocated data,
|			  or NULL if we should perform the allocation
|        Output: none
|
|       Returns: pointer to the mapped data on sucess, 
|		 NULL otherwise
|
|    Written By: Steve Kubica
|          Date: Apr 29, 1994 11:17
| Modifications:
|
------------------------------------------------------------*/
kaddr
kaps_mask_data(
   kobject  obj,
   int     *begin,
   int     *end,
   kaddr    dest)
{
   int i;
   int num_elements = 1;
   int value_type;
   int mask_type;
   double msk_r;
   double msk_i;
   char *mdata = NULL;
   kaddr vdata = NULL;


   for (i = 0; i < KPDS_VALUE_DIM; i++)
      num_elements *= end[i] - begin[i] + 1;

   /* get the old mask data type so we can restore it later */
   if (!kpds_get_attribute(obj, KPDS_MASK_DATA_TYPE, &mask_type))
      return FALSE;
   
   /* set the presentation data type of the mask to byte */
   if (!kpds_set_attribute(obj, KPDS_MASK_DATA_TYPE, KBYTE))
      return FALSE;

   /* get the mask data */    
   mdata = kdms_get_data(obj, KDMS_SEGMENT_MASK, begin, end, NULL);

   /* set the presentation data type of the mask back to it's orginal */
   if (!kpds_set_attribute(obj, KPDS_MASK_DATA_TYPE, mask_type))
   {
      kfree(mdata);
      return FALSE;
   }
   
   /* get the mask substitue values */
   if (!kpds_get_attribute(obj, KPDS_MASK_SUBSTITUTE_VALUE, &msk_r, &msk_i))
   {
      kfree(mdata);
      return FALSE;
   }
   
   /* get the value data type -- this is what type we will be working in */
   if (!kpds_get_attribute(obj, KPDS_VALUE_DATA_TYPE, &value_type))
   {
      kfree(mdata);
      return FALSE;
   }

   /* if data is NULL, then allocate, else modify in place */
   if (dest == NULL)
      vdata = kcalloc((unsigned)num_elements, 
		      (unsigned)kdata_size(value_type));
   else
      vdata = dest;
   
   switch (value_type)
   {
      case KBYTE:
	 {
	    char *data = (char *)vdata;

	    /* index through the elements in the destination */
	    for (i = 0; i < num_elements; i++)
	       if (mdata[i] == 0) data[i] = (char) msk_r;
	 }
	 break;

      case KUBYTE:
	 {
	    unsigned char *data = (unsigned char *)vdata;

	    /* index through the elements in the destination */
	    for (i = 0; i < num_elements; i++)
	       if (mdata[i] == 0) data[i] = (unsigned char) msk_r;
	 }
	 break;

      case KSHORT:
	 {
	    short *data = (short *)vdata;

	    /* index through the elements in the destination */
	    for (i = 0; i < num_elements; i++)
	       if (mdata[i] == 0) data[i] = (short) msk_r;
	 }
	 break;

      case KUSHORT:
	 {
	    unsigned short *data = (unsigned short *)vdata;

	    /* index through the elements in the destination */
	    for (i = 0; i < num_elements; i++)
	       if (mdata[i] == 0) data[i] = (unsigned short) msk_r;
	 }
	 break;

      case KINT:
	 {
	    int *data = (int *)vdata;

	    /* index through the elements in the destination */
	    for (i = 0; i < num_elements; i++)
	       if (mdata[i] == 0) data[i] = (int) msk_r;
	 }
	 break;

      case KUINT:
	 {
	    unsigned int *data = (unsigned int *)vdata;

	    /* index through the elements in the destination */
	    for (i = 0; i < num_elements; i++)
	       if (mdata[i] == 0) data[i] = (unsigned int) msk_r;
	 }
	 break;

      case KLONG:
	 {
	    long *data = (long *)vdata;

	    /* index through the elements in the destination */
	    for (i = 0; i < num_elements; i++)
	       if (mdata[i] == 0) data[i] = (long) msk_r;
	 }
	 break;

      case KULONG:
	 {
	    unsigned long *data = (unsigned long *)vdata;

	    /* index through the elements in the destination */
	    for (i = 0; i < num_elements; i++)
	       if (mdata[i] == 0) data[i] = (unsigned long) msk_r;
	 }
	 break;

      case KFLOAT:
	 {
	    float *data = (float *)vdata;

	    /* index through the elements in the destination */
	    for (i = 0; i < num_elements; i++)
	       if (mdata[i] == 0) data[i] = (float) msk_r;
	 }
	 break;

      case KDOUBLE:
	 {
	    double *data = (double *)vdata;

	    /* index through the elements in the destination */
	    for (i = 0; i < num_elements; i++)
	       if (mdata[i] == 0) data[i] = (double) msk_r;
	 }
	 break;

      case KCOMPLEX:
	 {
	    kcomplex *data = (kcomplex *) vdata;

	    /* index through the elements in the destination */
	    for (i = 0; i < num_elements; i++)
	       if (mdata[i] == 0) 
	       {
		  data[i].r = (float) msk_r;
		  data[i].i = (float) msk_i;
	       }
	 }
	 break;

      case KDCOMPLEX:
	 {
	    kdcomplex *data = (kdcomplex *) vdata;

	    /* index through the elements in the destination */
	    for (i = 0; i < num_elements; i++)
	       if (mdata[i] == 0) 
	       {
		  data[i].r = (double) msk_r;
		  data[i].i = (double) msk_i;
	       }
	 }
	 break;

      default:
	 errno = KINVALID_DATATYPE;
	 kfree(vdata);
	 kfree(mdata);
	 return FALSE;
   }
	
   return vdata;
}


/*-----------------------------------------------------------
|
|  Routine Name: kaps_get_uniform_data
|
|       Purpose: This routine gets a region of curvilinear
|		 data as it is constructed from uniform data.
|
|         Input: obj    - the object to retrieve data from
|                begin  - the begin point into the location
|                end    - the end point into the location
|		 dest   - pointer to already allocated data,
|			  or NULL if we should perform the allocation
|        Output: none
|
|       Returns: pointer to the location data on success
|		 NULL otherwise
|
|    Written By: Steve Kubica
|          Date: Jul 08, 1994 19:40
| Modifications:
|
------------------------------------------------------------*/
kaddr
kaps_get_uniform_data(
   kobject  obj, 
   int     *begin,
   int     *end,
   kaddr    dest)
{
   kaddr  ldata = NULL;
   int    loc_type;
   int    num_elements;
   int 	  ws;
   int 	  hs;
   int 	  ds;
   int 	  ns;
   int    pad = FALSE;
   int    b[KPDS_LOCATION_DIM];
   int    e[KPDS_LOCATION_DIM];
   int    skip[KPDS_LOCATION_DIM] = {1, 0, 0, 0};

   int    k[3];
   int    n;
   int    offset = 0;

   double ubeg[3];
   double uend[3];
   double del[3];

   /* this routine is only called when we know we have uniform data */   

   /* get the location data type -- this is what type we will be working in */
   if (!kpds_get_attribute(obj, KPDS_LOCATION_DATA_TYPE, &loc_type))
      return FALSE;

   /* -- figure out requested region size -- */
   num_elements = (end[0]-begin[0]+1) * (end[1]-begin[1]+1) *
                  (end[2]-begin[2]+1) * (end[3]-begin[3]+1);

   /* if data is NULL, then allocate requested space, else modify in place */
   if (dest == NULL)
      ldata = kmalloc((unsigned)(num_elements * kdata_size(loc_type)));
   else
      ldata = dest;

   /* -- retreive the presentation location size -- */ 
   kpds_get_attribute(obj, KPDS_LOCATION_SIZE,  &ws, &hs, &ds, &ns);

   /* -- figure out if the requested region falls within the size -- */

   /* -- check for points before the origin -- */
   b[0] = (begin[0] < 0) ? pad=TRUE, 0 : begin[0];
   b[1] = (begin[1] < 0) ? pad=TRUE, 0 : begin[1];
   b[2] = (begin[2] < 0) ? pad=TRUE, 0 : begin[2];
   b[3] = (begin[3] < 0) ? pad=TRUE, 0 : begin[3];
   e[0] = (end[0]   < 0) ? pad=TRUE, 0 : end[0];
   e[1] = (end[1]   < 0) ? pad=TRUE, 0 : end[1];
   e[2] = (end[2]   < 0) ? pad=TRUE, 0 : end[2];
   e[3] = (end[3]   < 0) ? pad=TRUE, 0 : end[3];
   
   /* -- check for points after the size -- */
   b[0] = (begin[0] >= ws) ? pad=TRUE, ws-1 : b[0];
   b[1] = (begin[1] >= hs) ? pad=TRUE, hs-1 : b[1];
   b[2] = (begin[2] >= ds) ? pad=TRUE, ds-1 : b[2];
   b[3] = (begin[3] >= ns) ? pad=TRUE, ns-1 : b[3];
   e[0] = (end[0]   >= ws) ? pad=TRUE, ws-1 : e[0];
   e[1] = (end[1]   >= hs) ? pad=TRUE, hs-1 : e[1];
   e[2] = (end[2]   >= ds) ? pad=TRUE, ds-1 : e[2];
   e[3] = (end[3]   >= ns) ? pad=TRUE, ns-1 : e[3];

   /* -- b and e now point to corners of the data we need to generate ... */

   /* -- ... we also now know if we need to prepad the dataset -- */
   if (pad)
   {
      int    stride[KPDS_LOCATION_DIM];
      double real;
      double imag;
      
      if (!kdms_get_attribute(obj, KPDS_SEGMENT_UNIFORM,
			      KDMS_PAD_VALUE, &real, &imag))
      {
	 kfree(ldata);
	 return NULL;
      }

      /* -- prepad the data set to the pad value -- */
      kdata_fill(ldata, num_elements, loc_type, real, imag);

      /* -- if the begin and end points are outside space, return now -- */
      if (end[0] < 0 || end[1] < 0 || end[2] < 0 || end[3] < 0)
	 return ldata;
      
      if (begin[0] >= ws || begin[1] >= hs || begin[2] >= ds || begin[3] >= ns)
	 return ldata;

      /* -- calculate the necessary strides to traverse the destination -- */
      stride[0] = 1;
      stride[1] = stride[0] * (end[0] - begin[0] + 1);
      stride[2] = stride[1] * (end[1] - begin[1] + 1);
      stride[3] = stride[2] * (end[2] - begin[2] + 1);
      
      /* -- calculate the skip factors between dimensions -- */
      skip[0] = stride[0];
      skip[1] = stride[1] - stride[0] * (e[0] - b[0] + 1);
      skip[2] = stride[2] - stride[1] * (e[1] - b[1] + 1);
      skip[3] = stride[3] - stride[2] * (e[2] - b[2] + 1);

      /* -- compute initial offset -- */
      offset = (b[0]-begin[0]) +
	 (b[1]-begin[1])*ws    +
	 (b[2]-begin[2])*ws*hs +
	 (b[3]-begin[3])*ws*hs*ds;
   }

   /* -- determine the deltas between adjacent points (no 0 deltas!) -- */
   kpds_get_attribute(obj, KPDS_LOCATION_BEGIN, &ubeg[0], &ubeg[1], &ubeg[2]);
   kpds_get_attribute(obj, KPDS_LOCATION_END,   &uend[0], &uend[1], &uend[2]);

   /* -- determine the deltas between adjacent points (no 0 deltas!) -- */
   del[0] = (uend[0]-ubeg[0])/((double)((ws-1)==0)? 1:(ws-1));
   del[1] = (uend[1]-ubeg[1])/((double)((hs-1)==0)? 1:(hs-1));
   del[2] = (uend[2]-ubeg[2])/((double)((ds-1)==0)? 1:(ds-1));
   
   switch(loc_type)
   {
      case KBYTE :
      {
	 char *data = (char *)ldata;
	 
	 data += offset;
	 for (n = b[3]; n <= e[3]; n++, data += skip[3])
	    for (k[2] = b[2]; k[2] <= e[2]; k[2]++, data += skip[2])
	       for (k[1] = b[1]; k[1] <= e[1]; k[1]++, data += skip[1])
		  for (k[0] = b[0]; k[0] <= e[0]; k[0]++, data += skip[0])
		     *data = (char) ubeg[n] + del[n] * k[n];
      } break;

      case KUBYTE :
      {
	 unsigned char *data = (unsigned char *)ldata;
	 
	 data += offset;
	 for (n = b[3]; n <= e[3]; n++, data += skip[3])
	    for (k[2] = b[2]; k[2] <= e[2]; k[2]++, data += skip[2])
	       for (k[1] = b[1]; k[1] <= e[1]; k[1]++, data += skip[1])
		  for (k[0] = b[0]; k[0] <= e[0]; k[0]++, data += skip[0])
		     *data = (unsigned char) ubeg[n] + del[n] * k[n];
      } break;

      case KSHORT:
      {
	 short *data = (short *)ldata;
	 
	 data += offset;
	 for (n = b[3]; n <= e[3]; n++, data += skip[3])
	    for (k[2] = b[2]; k[2] <= e[2]; k[2]++, data += skip[2])
	       for (k[1] = b[1]; k[1] <= e[1]; k[1]++, data += skip[1])
		  for (k[0] = b[0]; k[0] <= e[0]; k[0]++, data += skip[0])
		     *data = (short) ubeg[n] + del[n] * k[n];
      } break;

      case KUSHORT:
      {
	 unsigned short *data = (unsigned short *)ldata;
	 
	 data += offset;
	 for (n = b[3]; n <= e[3]; n++, data += skip[3])
	    for (k[2] = b[2]; k[2] <= e[2]; k[2]++, data += skip[2])
	       for (k[1] = b[1]; k[1] <= e[1]; k[1]++, data += skip[1])
		  for (k[0] = b[0]; k[0] <= e[0]; k[0]++, data += skip[0])
		     *data = (unsigned short) ubeg[n] + del[n] * k[n];
      } break;
      
      case KINT:
      {
	 int *data = (int *)ldata;
	 
	 data += offset;
	 for (n = b[3]; n <= e[3]; n++, data += skip[3])
	    for (k[2] = b[2]; k[2] <= e[2]; k[2]++, data += skip[2])
	       for (k[1] = b[1]; k[1] <= e[1]; k[1]++, data += skip[1])
		  for (k[0] = b[0]; k[0] <= e[0]; k[0]++, data += skip[0])
		     *data = (int) ubeg[n] + del[n] * k[n];
      } break;

      case KUINT:
      {
	 unsigned int *data = (unsigned int *)ldata;
	 
	 data += offset;
	 for (n = b[3]; n <= e[3]; n++, data += skip[3])
	    for (k[2] = b[2]; k[2] <= e[2]; k[2]++, data += skip[2])
	       for (k[1] = b[1]; k[1] <= e[1]; k[1]++, data += skip[1])
		  for (k[0] = b[0]; k[0] <= e[0]; k[0]++, data += skip[0])
		     *data = (unsigned int) ubeg[n] + del[n] * k[n];
      } break;

      case KLONG:
      {
	 long *data = (long *)ldata;
	 
	 data += offset;
	 for (n = b[3]; n <= e[3]; n++, data += skip[3])
	    for (k[2] = b[2]; k[2] <= e[2]; k[2]++, data += skip[2])
	       for (k[1] = b[1]; k[1] <= e[1]; k[1]++, data += skip[1])
		  for (k[0] = b[0]; k[0] <= e[0]; k[0]++, data += skip[0])
		     *data = (long) ubeg[n] + del[n] * k[n];
      } break;

      case KULONG:
      {
	 unsigned long *data = (unsigned long *)ldata;
	 
	 data += offset;
	 for (n = b[3]; n <= e[3]; n++, data += skip[3])
	    for (k[2] = b[2]; k[2] <= e[2]; k[2]++, data += skip[2])
	       for (k[1] = b[1]; k[1] <= e[1]; k[1]++, data += skip[1])
		  for (k[0] = b[0]; k[0] <= e[0]; k[0]++, data += skip[0])
		     *data = (unsigned long) ubeg[n] + del[n] * k[n];
      } break;

      case KFLOAT:
      {
	 float *data = (float *)ldata;
	 
	 data += offset;
	 for (n = b[3]; n <= e[3]; n++, data += skip[3])
	    for (k[2] = b[2]; k[2] <= e[2]; k[2]++, data += skip[2])
	       for (k[1] = b[1]; k[1] <= e[1]; k[1]++, data += skip[1])
		  for (k[0] = b[0]; k[0] <= e[0]; k[0]++, data += skip[0])
		     *data = (float) ubeg[n] + del[n] * k[n];
      } break;

      case KDOUBLE:
      {
	 double *data = (double *)ldata;
	 
	 data += offset;
	 for (n = b[3]; n <= e[3]; n++, data += skip[3])
	    for (k[2] = b[2]; k[2] <= e[2]; k[2]++, data += skip[2])
	       for (k[1] = b[1]; k[1] <= e[1]; k[1]++, data += skip[1])
		  for (k[0] = b[0]; k[0] <= e[0]; k[0]++, data += skip[0])
		     *data = (double) ubeg[n] + del[n] * k[n];
      } break;

      case KCOMPLEX:
      {
	 kcomplex *data = (kcomplex *)ldata;

	 data += offset;
	 for (n = b[3]; n <= e[3]; n++, data += skip[3])
	    for (k[2] = b[2]; k[2] <= e[2]; k[2]++, data += skip[2])
	       for (k[1] = b[1]; k[1] <= e[1]; k[1]++, data += skip[1])
		  for (k[0] = b[0]; k[0] <= e[0]; k[0]++, data += skip[0])
		  {
		     (*data).r = (float) ubeg[n] + del[n] * k[n];
		     (*data).i = (float) 0;
		  }
      } break;

      case KDCOMPLEX:
      {
	 kdcomplex *data = (kdcomplex *)ldata;
	 
	 data += offset;
	 for (n = b[3]; n <= e[3]; n++, data += skip[3])
	    for (k[2] = b[2]; k[2] <= e[2]; k[2]++, data += skip[2])
	       for (k[1] = b[1]; k[1] <= e[1]; k[1]++, data += skip[1])
		  for (k[0] = b[0]; k[0] <= e[0]; k[0]++, data += skip[0])
		  {
		     (*data).r = (double) ubeg[n] + del[n] * k[n];
		     (*data).i = (double) 0;
		  }
      } break;

      default :
	    return NULL;
   }

   return ldata;
}

/*-----------------------------------------------------------
|
|  Routine Name: kaps_get_rectilinear_data
|
|       Purpose: This routine gets a region of rectilinear
|		 data and lays it out into an array as if it
|		 were curviliner data.
|
|         Input: obj    - the object to retrieve data from
|                begin  - the begin point into the location
|                end    - the end point into the location
|		 dest   - pointer to already allocated data,
|			  or NULL if we should perform the allocation
|        Output: none
|
|       Returns: pointer to the location data on success
|		 NULL otherwise
|
|    Written By: Steve Kubica
|          Date: Jul 08, 1994 19:40
| Modifications:
|
------------------------------------------------------------*/
kaddr
kaps_get_rectilinear_data(
   kobject  obj,
   int     *begin,
   int     *end,
   kaddr    dest)
{
   kaddr wid_data = NULL;
   kaddr hgt_data = NULL;
   kaddr dep_data = NULL;
   kaddr ldata = NULL;
   int   loc_type;
   int   wsize;
   int   hsize;
   int   dsize;
   int   esize;
   int   num_elements = 0;
   int   w;
   int   h;
   int   d;
   int   i;
   
   
   /* this routine is only called when we know we have rectilinear data */   

   /* figure out entire requested along with the sizes of each dimension */
   wsize = end[KWID] - begin[KWID] + 1;
   hsize = end[KHGT] - begin[KHGT] + 1;
   dsize = end[KDEP] - begin[KDEP] + 1;
   esize = end[KDIM] - begin[KDIM] + 1;
   num_elements = wsize * hsize * dsize * esize;

   /* get the location data type -- this is what type we will be working in */
   if (!kpds_get_attribute(obj, KPDS_LOCATION_DATA_TYPE, &loc_type))
      return FALSE;

   /* get the data from each of the rectilinear segments */
   wid_data = kdms_get_data(obj, KDMS_SEGMENT_WIDTH,
			    &begin[KWID], &end[KWID], NULL);
   hgt_data = kdms_get_data(obj, KDMS_SEGMENT_HEIGHT,
			    &begin[KHGT], &end[KHGT], NULL);
   dep_data = kdms_get_data(obj, KDMS_SEGMENT_DEPTH,
			    &begin[KDEP], &end[KDEP], NULL);
   
   if (wid_data == NULL || hgt_data == NULL || dep_data == NULL)
   {
      kfree(wid_data);      kfree(hgt_data);      kfree(dep_data);
      return NULL;
   }
   
   /* if data is NULL, then allocate, else modify in place */
   if (dest == NULL)
      ldata = kcalloc((unsigned)num_elements, (unsigned)kdata_size(loc_type));
   else
      ldata = dest;

   switch (loc_type)
   {
      case KBYTE:
	 {
	    char *wdata = (char *) wid_data;
	    char *hdata = (char *) hgt_data;
	    char *ddata = (char *) dep_data;
	    char *data  = (char *) ldata;
	    
	    /* construct the curvilinear representation */
	    for (w = 0; w < wsize; w++)
	       for (h = 0; h < hsize; h++)
		  for (d = 0; d < dsize; d++)
		     for (i = 0; i < esize; i++)
		     {
			if (i+begin[KDIM] == 0) 
			   data[w+wsize*(h+hsize*(d+dsize*i))] = wdata[w];
			else if (i+begin[KDIM] == 1)
			   data[w+wsize*(h+hsize*(d+dsize*i))] = hdata[h];
			else if (i+begin[KDIM] == 2) 
			   data[w+wsize*(h+hsize*(d+dsize*i))] = ddata[d];
		     }
	 } break;
      case KUBYTE:
	 {
	    unsigned char *wdata = (unsigned char *) wid_data;
	    unsigned char *hdata = (unsigned char *) hgt_data;
	    unsigned char *ddata = (unsigned char *) dep_data;
	    unsigned char *data  = (unsigned char *) ldata;
	    
	    /* construct the curvilinear representation */
	    for (w = 0; w < wsize; w++)
	       for (h = 0; h < hsize; h++)
		  for (d = 0; d < dsize; d++)
		     for (i = 0; i < esize; i++)
		     {
			if (i+begin[KDIM] == 0) 
			   data[w+wsize*(h+hsize*(d+dsize*i))] = wdata[w];
			else if (i+begin[KDIM] == 1)
			   data[w+wsize*(h+hsize*(d+dsize*i))] = hdata[h];
			else if (i+begin[KDIM] == 2) 
			   data[w+wsize*(h+hsize*(d+dsize*i))] = ddata[d];
		     }
	 } break;
      case KSHORT:
	 {
	    short *wdata = (short *) wid_data;
	    short *hdata = (short *) hgt_data;
	    short *ddata = (short *) dep_data;
	    short *data  = (short *) ldata;
	    
	    /* construct the curvilinear representation */
	    for (w = 0; w < wsize; w++)
	       for (h = 0; h < hsize; h++)
		  for (d = 0; d < dsize; d++)
		     for (i = 0; i < esize; i++)
		     {
			if (i+begin[KDIM] == 0) 
			   data[w+wsize*(h+hsize*(d+dsize*i))] = wdata[w];
			else if (i+begin[KDIM] == 1)
			   data[w+wsize*(h+hsize*(d+dsize*i))] = hdata[h];
			else if (i+begin[KDIM] == 2) 
			   data[w+wsize*(h+hsize*(d+dsize*i))] = ddata[d];
		     }
	 } break;
      case KUSHORT:
	 {
	    unsigned short *wdata = (unsigned short *) wid_data;
	    unsigned short *hdata = (unsigned short *) hgt_data;
	    unsigned short *ddata = (unsigned short *) dep_data;
	    unsigned short *data  = (unsigned short *) ldata;
	    
	    /* construct the curvilinear representation */
	    for (w = 0; w < wsize; w++)
	       for (h = 0; h < hsize; h++)
		  for (d = 0; d < dsize; d++)
		     for (i = 0; i < esize; i++)
		     {
			if (i+begin[KDIM] == 0) 
			   data[w+wsize*(h+hsize*(d+dsize*i))] = wdata[w];
			else if (i+begin[KDIM] == 1)
			   data[w+wsize*(h+hsize*(d+dsize*i))] = hdata[h];
			else if (i+begin[KDIM] == 2) 
			   data[w+wsize*(h+hsize*(d+dsize*i))] = ddata[d];
		     }
	 } break;
      case KINT:
	 {
	    int *wdata = (int *) wid_data;
	    int *hdata = (int *) hgt_data;
	    int *ddata = (int *) dep_data;
	    int *data  = (int *) ldata;
	    
	    /* construct the curvilinear representation */
	    for (w = 0; w < wsize; w++)
	       for (h = 0; h < hsize; h++)
		  for (d = 0; d < dsize; d++)
		     for (i = 0; i < esize; i++)
		     {
			if (i+begin[KDIM] == 0) 
			   data[w+wsize*(h+hsize*(d+dsize*i))] = wdata[w];
			else if (i+begin[KDIM] == 1)
			   data[w+wsize*(h+hsize*(d+dsize*i))] = hdata[h];
			else if (i+begin[KDIM] == 2) 
			   data[w+wsize*(h+hsize*(d+dsize*i))] = ddata[d];
		     }
	 } break;
      case KUINT:
	 {
	    unsigned int *wdata = (unsigned int *) wid_data;
	    unsigned int *hdata = (unsigned int *) hgt_data;
	    unsigned int *ddata = (unsigned int *) dep_data;
	    unsigned int *data  = (unsigned int *) ldata;
	    
	    /* construct the curvilinear representation */
	    for (w = 0; w < wsize; w++)
	       for (h = 0; h < hsize; h++)
		  for (d = 0; d < dsize; d++)
		     for (i = 0; i < esize; i++)
		     {
			if (i+begin[KDIM] == 0) 
			   data[w+wsize*(h+hsize*(d+dsize*i))] = wdata[w];
			else if (i+begin[KDIM] == 1)
			   data[w+wsize*(h+hsize*(d+dsize*i))] = hdata[h];
			else if (i+begin[KDIM] == 2) 
			   data[w+wsize*(h+hsize*(d+dsize*i))] = ddata[d];
		     }
	 } break;
      case KLONG:
	 {
	    long *wdata = (long *) wid_data;
	    long *hdata = (long *) hgt_data;
	    long *ddata = (long *) dep_data;
	    long *data  = (long *) ldata;
	    
	    /* construct the curvilinear representation */
	    for (w = 0; w < wsize; w++)
	       for (h = 0; h < hsize; h++)
		  for (d = 0; d < dsize; d++)
		     for (i = 0; i < esize; i++)
		     {
			if (i+begin[KDIM] == 0) 
			   data[w+wsize*(h+hsize*(d+dsize*i))] = wdata[w];
			else if (i+begin[KDIM] == 1)
			   data[w+wsize*(h+hsize*(d+dsize*i))] = hdata[h];
			else if (i+begin[KDIM] == 2) 
			   data[w+wsize*(h+hsize*(d+dsize*i))] = ddata[d];
		     }
	 } break;
      case KULONG:
	 {
	    unsigned long *wdata = (unsigned long *) wid_data;
	    unsigned long *hdata = (unsigned long *) hgt_data;
	    unsigned long *ddata = (unsigned long *) dep_data;
	    unsigned long *data  = (unsigned long *) ldata;
	    
	    /* construct the curvilinear representation */
	    for (w = 0; w < wsize; w++)
	       for (h = 0; h < hsize; h++)
		  for (d = 0; d < dsize; d++)
		     for (i = 0; i < esize; i++)
		     {
			if (i+begin[KDIM] == 0) 
			   data[w+wsize*(h+hsize*(d+dsize*i))] = wdata[w];
			else if (i+begin[KDIM] == 1)
			   data[w+wsize*(h+hsize*(d+dsize*i))] = hdata[h];
			else if (i+begin[KDIM] == 2) 
			   data[w+wsize*(h+hsize*(d+dsize*i))] = ddata[d];
		     }
	 } break;
      case KFLOAT:
	 {
	    float *wdata = (float *) wid_data;
	    float *hdata = (float *) hgt_data;
	    float *ddata = (float *) dep_data;
	    float *data  = (float *) ldata;
	    
	    /* construct the curvilinear representation */
	    for (w = 0; w < wsize; w++)
	       for (h = 0; h < hsize; h++)
		  for (d = 0; d < dsize; d++)
		     for (i = 0; i < esize; i++)
		     {
			if (i+begin[KDIM] == 0) 
			   data[w+wsize*(h+hsize*(d+dsize*i))] = wdata[w];
			else if (i+begin[KDIM] == 1)
			   data[w+wsize*(h+hsize*(d+dsize*i))] = hdata[h];
			else if (i+begin[KDIM] == 2) 
			   data[w+wsize*(h+hsize*(d+dsize*i))] = ddata[d];
		     }
	 } break;
      case KDOUBLE:
	 {
	    double *wdata = (double *) wid_data;
	    double *hdata = (double *) hgt_data;
	    double *ddata = (double *) dep_data;
	    double *data  = (double *) ldata;
	    
	    /* construct the curvilinear representation */
	    for (w = 0; w < wsize; w++)
	       for (h = 0; h < hsize; h++)
		  for (d = 0; d < dsize; d++)
		     for (i = 0; i < esize; i++)
		     {
			if (i+begin[KDIM] == 0) 
			   data[w+wsize*(h+hsize*(d+dsize*i))] = wdata[w];
			else if (i+begin[KDIM] == 1)
			   data[w+wsize*(h+hsize*(d+dsize*i))] = hdata[h];
			else if (i+begin[KDIM] == 2) 
			   data[w+wsize*(h+hsize*(d+dsize*i))] = ddata[d];
		     }
	 } break;

      case KCOMPLEX:
	 {
	    kcomplex *wdata = (kcomplex *) wid_data;
	    kcomplex *hdata = (kcomplex *) hgt_data;
	    kcomplex *ddata = (kcomplex *) dep_data;
	    kcomplex *data  = (kcomplex *) ldata;
	    
	    /* construct the curvilinear representation */
	    for (w = 0; w < wsize; w++)
	       for (h = 0; h < hsize; h++)
		  for (d = 0; d < dsize; d++)
		     for (i = 0; i < esize; i++)
		     {
			if (i+begin[KDIM] == 0) 
			   data[w+wsize*(h+hsize*(d+dsize*i))] = wdata[w];
			else if (i+begin[KDIM] == 1)
			   data[w+wsize*(h+hsize*(d+dsize*i))] = hdata[h];
			else if (i+begin[KDIM] == 2) 
			   data[w+wsize*(h+hsize*(d+dsize*i))] = ddata[d];
		     }
	 } break;
      case KDCOMPLEX:
	 {
	    kdcomplex *wdata = (kdcomplex *) wid_data;
	    kdcomplex *hdata = (kdcomplex *) hgt_data;
	    kdcomplex *ddata = (kdcomplex *) dep_data;
	    kdcomplex *data  = (kdcomplex *) ldata;
	    
	    /* construct the curvilinear representation */
	    for (w = 0; w < wsize; w++)
	       for (h = 0; h < hsize; h++)
		  for (d = 0; d < dsize; d++)
		     for (i = 0; i < esize; i++)
		     {
			if (i+begin[KDIM] == 0) 
			   data[w+wsize*(h+hsize*(d+dsize*i))] = wdata[w];
			else if (i+begin[KDIM] == 1)
			   data[w+wsize*(h+hsize*(d+dsize*i))] = hdata[h];
			else if (i+begin[KDIM] == 2) 
			   data[w+wsize*(h+hsize*(d+dsize*i))] = ddata[d];
		     }
	 } break;

      default :
	    return NULL;
   }

   return ldata;
}

/*-----------------------------------------------------------
|
|  Routine Name: kaps_transpose_elements_first
|
|       Purpose: This routine transposes a given array
|		 of data with elements as the trailing
|		 dimension into an array with the elements
|		 as the leading dimension.
|
|         Input: rdata  - data to transpose (this is freed!)
|                begin  - the begin point
|                end    - the end point 
|		 dim    - the dimensionality of the data
|
|        Output: none
|
|       Returns: pointer to the transposed data on success
|		 NULL otherwise
|
|    Written By: Steve Kubica
|          Date: Jul 08, 1994 19:40
| Modifications:
|
------------------------------------------------------------*/
kaddr
kaps_transpose_elements_first(
   kaddr rdata,
   int  *begin,
   int  *end,
   int   type,
   int   dim)
{
   int   block_size = 1;
   int   e = dim-1;
   
   kaddr tdata = NULL;
   int   i;
   int   j;

   /* -- if the element size is 1, then transpose is a nullop -- */
   if (end[e] - begin[e] == 0)
      return rdata;
   
   /* -- determine number of points in each 1 element block -- */
   for (i = 0; i < e; i++)
      block_size *= (end[i] - begin[i] + 1);

   /* -- allocate space for the transposed array -- */
   tdata = kcalloc((unsigned)(block_size * (end[e] - begin[e] + 1)), 
		   (unsigned)kdata_size(type));

   switch (type)
   {
      case KBYTE:
	 {
	    char *sdata = (char *) rdata;
	    char *ddata = (char *) tdata;

	    for (j = 0; j < block_size; j++)
	       for (i = 0; i <= end[e]; i++)
		  *ddata++ = sdata[i*block_size + j];
	 } break;
      case KUBYTE:
	 {
	    unsigned char *sdata = (unsigned char *) rdata;
	    unsigned char *ddata = (unsigned char *) tdata;
	    
	    for (j = 0; j < block_size; j++)
	       for (i = 0; i <= end[e]; i++)
		  *ddata++ = sdata[i*block_size + j];
	 } break;
      case KSHORT:
	 {
	    short *sdata = (short *) rdata;
	    short *ddata = (short *) tdata;
	    
	    for (j = 0; j < block_size; j++)
	       for (i = 0; i <= end[e]; i++)
		  *ddata++ = sdata[i*block_size + j];
	 } break;
      case KUSHORT:
	 {
	    unsigned short *sdata = (unsigned short *) rdata;
	    unsigned short *ddata = (unsigned short *) tdata;
	    
	    for (j = 0; j < block_size; j++)
	       for (i = 0; i <= end[e]; i++)
		  *ddata++ = sdata[i*block_size + j];
	 } break;
      case KINT:
	 {
	    int *sdata = (int *) rdata;
	    int *ddata = (int *) tdata;
	    
	    for (j = 0; j < block_size; j++)
	       for (i = 0; i <= end[e]; i++)
		  *ddata++ = sdata[i*block_size + j];
	 } break;
      case KUINT:
	 {
	    unsigned int *sdata = (unsigned int *) rdata;
	    unsigned int *ddata = (unsigned int *) tdata;
	    
	    for (j = 0; j < block_size; j++)
	       for (i = 0; i <= end[e]; i++)
		  *ddata++ = sdata[i*block_size + j];
	 } break;
      case KLONG:
	 {
	    long *sdata = (long *) rdata;
	    long *ddata = (long *) tdata;
	    
	    for (j = 0; j < block_size; j++)
	       for (i = 0; i <= end[e]; i++)
		  *ddata++ = sdata[i*block_size + j];
	 } break;
      case KULONG:
	 {
	    unsigned long *sdata = (unsigned long *) rdata;
	    unsigned long *ddata = (unsigned long *) tdata;
	    
	    for (j = 0; j < block_size; j++)
	       for (i = 0; i <= end[e]; i++)
		  *ddata++ = sdata[i*block_size + j];
	 } break;
      case KFLOAT:
	 {
	    float *sdata = (float *) rdata;
	    float *ddata = (float *) tdata;
	    
	    for (j = 0; j < block_size; j++)
	       for (i = 0; i <= end[e]; i++)
		  *ddata++ = sdata[i*block_size + j];

	 } break;
      case KDOUBLE:
	 {
	    double *sdata = (double *) rdata;
	    double *ddata = (double *) tdata;
	    
	    for (j = 0; j < block_size; j++)
	       for (i = 0; i <= end[e]; i++)
		  *ddata++ = sdata[i*block_size + j];

	 } break;
      case KCOMPLEX:
	 {
	    kcomplex *sdata = (kcomplex *) rdata;
	    kcomplex *ddata = (kcomplex *) tdata;
	    
	    for (j = 0; j < block_size; j++)
	       for (i = 0; i <= end[e]; i++)
		  *ddata++ = sdata[i*block_size + j];
	 } break;
      case KDCOMPLEX:
	 {
	    kdcomplex *sdata = (kdcomplex *) rdata;
	    kdcomplex *ddata = (kdcomplex *) tdata;
	    
	    for (j = 0; j < block_size; j++)
	       for (i = 0; i <= end[e]; i++)
		  *ddata++ = sdata[i*block_size + j];
	 } break;

      default :
	    return NULL;
   }

   kfree(rdata);

   return tdata;
}

/*-----------------------------------------------------------
|
|  Routine Name: kaps_transpose_elements_last
|
|       Purpose: This routine transposes a given array
|		 of data with elements in the leading
|		 dimension into an array with the elements
|		 in the trailing dimension.  The begin
|		 and end points are assumed to indicate
|		 that the elements are in the *trailing*
|		 dimension.
|
|         Input: rdata  - data to transpose (this is freed!)
|                begin  - the begin point
|                end    - the end point 
|		 dim    - the dimensionality of the data
|
|        Output: none
|
|       Returns: pointer to the transposed data on success
|		 NULL otherwise
|
|    Written By: Steve Kubica
|          Date: Jul 08, 1994 19:40
| Modifications:
|
------------------------------------------------------------*/
kaddr
kaps_transpose_elements_last(
   kaddr rdata,
   int  *begin,
   int  *end,
   int   type,
   int   dim)
{
   int   block_size = 1;
   int   e = dim-1;
   int   esize;
   
   kaddr tdata = NULL;
   int   i;
   int   j;

   /* -- if the element size is 1, then transpose is a nullop -- */
   if ((esize = (end[e] - begin[e]) + 1) == 0)
      return rdata;
   
   /* -- determine number of points in each 1 element block -- */
   for (i = 0; i < e; i++)
      block_size *= (end[i] - begin[i] + 1);

   /* -- allocate space for the transposed array -- */
   tdata = kcalloc((unsigned)(block_size * (end[e] - begin[e] + 1)), 
		   (unsigned)kdata_size(type));

   switch (type)
   {
      case KBYTE:
	 {
	    char *sdata = (char *) rdata;
	    char *ddata = (char *) tdata;

	    for (j = 0; j <= end[e]; j++)
	       for (i = 0; i < block_size; i++)
		  *ddata++ = sdata[i*esize + j];
	 } break;
      case KUBYTE:
	 {
	    unsigned char *sdata = (unsigned char *) rdata;
	    unsigned char *ddata = (unsigned char *) tdata;
	    
	    for (j = 0; j <= end[e]; j++)
	       for (i = 0; i < block_size; i++)
		  *ddata++ = sdata[i*esize + j];
	 } break;
      case KSHORT:
	 {
	    short *sdata = (short *) rdata;
	    short *ddata = (short *) tdata;
	    
	    for (j = 0; j <= end[e]; j++)
	       for (i = 0; i < block_size; i++)
		  *ddata++ = sdata[i*esize + j];
	 } break;
      case KUSHORT:
	 {
	    unsigned short *sdata = (unsigned short *) rdata;
	    unsigned short *ddata = (unsigned short *) tdata;
	    
	    for (j = 0; j <= end[e]; j++)
	       for (i = 0; i < block_size; i++)
		  *ddata++ = sdata[i*esize + j];
	 } break;
      case KINT:
	 {
	    int *sdata = (int *) rdata;
	    int *ddata = (int *) tdata;
	    
	    for (j = 0; j <= end[e]; j++)
	       for (i = 0; i < block_size; i++)
		  *ddata++ = sdata[i*esize + j];
	 } break;
      case KUINT:
	 {
	    unsigned int *sdata = (unsigned int *) rdata;
	    unsigned int *ddata = (unsigned int *) tdata;
	    
	    for (j = 0; j <= end[e]; j++)
	       for (i = 0; i < block_size; i++)
		  *ddata++ = sdata[i*esize + j];
	 } break;
      case KLONG:
	 {
	    long *sdata = (long *) rdata;
	    long *ddata = (long *) tdata;
	    
	    for (j = 0; j <= end[e]; j++)
	       for (i = 0; i < block_size; i++)
		  *ddata++ = sdata[i*esize + j];
	 } break;
      case KULONG:
	 {
	    unsigned long *sdata = (unsigned long *) rdata;
	    unsigned long *ddata = (unsigned long *) tdata;
	    
	    for (j = 0; j <= end[e]; j++)
	       for (i = 0; i < block_size; i++)
		  *ddata++ = sdata[i*esize + j];
	 } break;
      case KFLOAT:
	 {
	    float *sdata = (float *) rdata;
	    float *ddata = (float *) tdata;
	    
	    for (j = 0; j <= end[e]; j++)
	       for (i = 0; i < block_size; i++)
		  *ddata++ = sdata[i*esize + j];
	 } break;
      case KDOUBLE:
	 {
	    double *sdata = (double *) rdata;
	    double *ddata = (double *) tdata;
	    
	    for (j = 0; j <= end[e]; j++)
	       for (i = 0; i < block_size; i++)
		  *ddata++ = sdata[i*esize + j];
	 } break;
      case KCOMPLEX:
	 {
	    kcomplex *sdata = (kcomplex *) rdata;
	    kcomplex *ddata = (kcomplex *) tdata;
	    
	    for (j = 0; j <= end[e]; j++)
	       for (i = 0; i < block_size; i++)
		  *ddata++ = sdata[i*esize + j];
	 } break;
      case KDCOMPLEX:
	 {
	    kdcomplex *sdata = (kdcomplex *) rdata;
	    kdcomplex *ddata = (kdcomplex *) tdata;
	    
	    for (j = 0; j <= end[e]; j++)
	       for (i = 0; i < block_size; i++)
		  *ddata++ = sdata[i*esize + j];
	 } break;

      default :
	    return NULL;
   }

   return tdata;
}

/*-----------------------------------------------------------
|
|  Routine Name: kaps_mapping_mode
|
|       Purpose: This utility function returns the mapping mode
|	         It also considers the presence of a map, if there
|		 is no map segment, it will return that mapping mode is off
|		 even though it may really be turned on.
|
|         Input: obj - the object from which to query the mapping mode
|
|        Output: none
|
|       Returns: TRUE (1) if the mapping mode is on, FALSE (0) othewise
|
|    Written By: Steve Kubica
|          Date: Apr 28, 1994 15:15
| Modifications:
|
------------------------------------------------------------*/
int
kaps_mapping_mode(kobject obj)
{
   int mapping;

   /* if there is no map, then pretend that mapping mode is off */
   if (!kdms_query_segment(obj, KDMS_SEGMENT_MAP))
      return FALSE;   

   if (!kdms_get_attribute(obj, NULL, KPDS_MAPPING_MODE, &mapping))
      return FALSE;

   /* a sanity check requested by donna */
   if (mapping == TRUE && !kdms_query_segment(obj, KDMS_SEGMENT_VALUE))
      kinfo(KSYSLIB,"Mapping mode has been set, but no value data exists.\n");

   return mapping;
}

/*-----------------------------------------------------------
|
|  Routine Name: kaps_masked_presentation
|
|       Purpose: This utility function returns the masked value
|		 presentation. It also considers the presence of a mask, 
|		 if there is no mask segment, it will return that the
|		 masked value presentation is set to use the original
|		 even though it may really be set to use replacement values.
|
|         Input: obj - the object from which to query the masked value 
|		       presentation
|
|        Output: none
|
|       Returns: TRUE (1) if the the value is to be masked and a 
|		 mask is present, FALSE (0) othewise
|
|    Written By: Steve Kubica
|          Date: Apr 28, 1994 15:15
| Modifications:
|
------------------------------------------------------------*/
int
kaps_masked_presentation(kobject obj)
{
   int masking;

   if (!kdms_get_attribute(obj, NULL, KPDS_MASKED_VALUE_PRESENTATION,
			   &masking))
      return KUSE_ORIGINAL;

   /* if there is no mask, then pretend that masking mode is off */
   if (!kdms_query_segment(obj, KDMS_SEGMENT_MASK))
      return KUSE_ORIGINAL;   

   return masking;
}

/*-----------------------------------------------------------
|
|  Routine Name: kaps_location_grid
|
|       Purpose: This utility function returns the location grid.
|
|         Input: obj - the object from which to query the location grid
|
|        Output: none
|
|       Returns: TRUE (1) if the mapping mode is on, FALSE (0) othewise
|
|    Written By: Steve Kubica
|          Date: Apr 28, 1994 15:15
| Modifications:
|
------------------------------------------------------------*/
int
kaps_location_grid(kobject obj)
{
   int grid;


   if (!kdms_get_attribute(obj, NULL, KPDS_LOCATION_GRID, &grid))
      return FALSE;

   return grid;
}

/*-----------------------------------------------------------
|
|  Routine Name: kaps_elements_first
|
|       Purpose: This utility function returns the elements first attribute
|
|         Input: obj - the object from which to query the elements first 
|		       attribute
|
|        Output: none
|
|       Returns: TRUE (1) if the elements first is on, FALSE (0) othewise
|
|    Written By: Steve Kubica
|          Date: Apr 28, 1994 15:15
| Modifications:
|
------------------------------------------------------------*/
int
kaps_elements_first(kobject obj)
{
   int efirst;


   if (!kdms_get_attribute(obj, NULL, KPDS_ELEMENTS_FIRST, &efirst))
      return FALSE;

   return efirst;
}
