 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Geometry Application Services
   >>>>               Primitive List Routines 
   >>>>
   >>>>   Static:
   >>>>			_kgeom_query_quadmesh()
   >>>>			_kgeom_set_quadmesh_size()
   >>>>			_kgeom_query_octmesh()
   >>>>			_kgeom_set_octmesh_size()
   >>>>			_kgeom_query_texture2d()
   >>>>			_kgeom_set_texture2d_size()
   >>>>			_kgeom_query_texture3d()
   >>>>			_kgeom_set_texture3d_size()
   >>>>			_kgeom_init_mesh()
   >>>>
   >>>>			_quadmesh_size_get()
   >>>>			_quadmesh_size_set() 
   >>>>			_quadmesh_size_match() 
   >>>>			_quadmesh_size_copy()
   >>>>			_quadmesh_size_query() 
   >>>>			_quadmesh_size_print()
   >>>>			_octmesh_size_get()
   >>>>			_octmesh_size_set()
   >>>>			_octmesh_size_match()
   >>>>			_octmesh_size_copy()
   >>>>			_octmesh_size_query()
   >>>>			_octmesh_size_print()
   >>>>			_texture2d_size_get()
   >>>>			_texture2d_size_set()
   >>>>			_texture2d_size_match()
   >>>>			_texture2d_size_copy() 
   >>>>			_texture2d_size_query()
   >>>>			_texture2d_size_print()
   >>>>			_texture3d_size_get()
   >>>>			_texture3d_size_set()
   >>>>			_texture3d_size_match()
   >>>>			_texture3d_size_copy() 
   >>>>			_texture3d_size_query()
   >>>>			_texture3d_size_print()
   >>>>			_position2d_get() 
   >>>>			_position2d_set() 
   >>>>			_position2d_match() 
   >>>>			_position2d_copy() 
   >>>>			_position2d_query() 
   >>>>			_position2d_print() 
   >>>>			_position3d_get() 
   >>>>			_position3d_set() 
   >>>>			_position3d_match()
   >>>>			_position3d_copy()
   >>>>			_position3d_query() 
   >>>>			_position3d_print()
   >>>>  Private:
   >>>>			kgeom_get_uniform_mesh()
   >>>>	    		kgeom_put_uniform_mesh()
   >>>>			kgeom_get_mesh()
   >>>>	    		kgeom_put_mesh()
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "geom_internals.h"

/* 
 *  Mesh, Texture, and Map primitives
 * .......................................................................
 *
 *  note that primitives are numbers for geometry services
 */

kaps_primitive kgeom_meshes[] =
{

   /* 
    *  Quadmesh primitives
    * .......................................................................
    */
   
   {NULL, KGEOM_QUADMESH_LOCATION_ALL, 0, KGEOM_SEGMENT_LOCATION,
    {1, 1, 0, 1, 0},
    kaps_get_primitive, kaps_put_primitive},
   {NULL, KGEOM_QUADMESH_LOCATION_VERTEX, 0, KGEOM_SEGMENT_LOCATION,
    {0, 0, 0, 1, 0},
    kaps_get_primitive, kaps_put_primitive},
   {NULL, KGEOM_QUADMESH_LOCATION_ROW, 0, KGEOM_SEGMENT_LOCATION,
    {1, 0, 0, 1, 0},
    kaps_get_primitive, kaps_put_primitive},
   {NULL, KGEOM_QUADMESH_LOCATION_COLUMN, 0, KGEOM_SEGMENT_LOCATION,
    {0, 1, 0, 1, 0},
    kaps_get_primitive, kaps_put_primitive},

   {NULL, KGEOM_QUADMESH_LOCATION_MINIMUM, 0, KGEOM_SEGMENT_LOCATION,
    {0, 0, 0, 0, 0},
    kgeom_get_uniform_mesh, kgeom_put_uniform_mesh},
   {NULL, KGEOM_QUADMESH_LOCATION_MAXIMUM, 0, KGEOM_SEGMENT_LOCATION,
    {0, 0, 0, 0, 0},
    kgeom_get_uniform_mesh, kgeom_put_uniform_mesh},

   {NULL, KGEOM_QUADMESH_LOCATION_WIDTH_ALL, 0, KGEOM_SEGMENT_WIDTH,
    {1, 0, 0, 0, 0},
    kaps_get_primitive, kaps_put_primitive},
   {NULL, KGEOM_QUADMESH_LOCATION_WIDTH_POINT, 0, KGEOM_SEGMENT_WIDTH,
    {0, 0, 0, 0, 0},
    kaps_get_primitive, kaps_put_primitive},

   {NULL, KGEOM_QUADMESH_LOCATION_HEIGHT_ALL, 0, KGEOM_SEGMENT_HEIGHT,
    {1, 0, 0, 0, 0},
    kaps_get_primitive, kaps_put_primitive},
   {NULL, KGEOM_QUADMESH_LOCATION_HEIGHT_POINT, 0, KGEOM_SEGMENT_HEIGHT,
    {0, 0, 0, 0, 0},
    kaps_get_primitive, kaps_put_primitive},
   
   {NULL, KGEOM_QUADMESH_COLOR_ALL, 0, KPDS_SEGMENT_VALUE,
    {1, 1, 0, 0, 1 },
    kaps_get_primitive, kaps_put_primitive},
   {NULL, KGEOM_QUADMESH_COLOR_VERTEX, 0, KPDS_SEGMENT_VALUE,
    {0, 0, 0, 0, 1},
    kaps_get_primitive, kaps_put_primitive},
   {NULL, KGEOM_QUADMESH_COLOR_ROW, 0, KPDS_SEGMENT_VALUE,
    {1, 0, 0, 0, 1},
    kaps_get_primitive, kaps_put_primitive},
   {NULL, KGEOM_QUADMESH_COLOR_COLUMN, 0, KPDS_SEGMENT_VALUE,
    {0, 1, 0, 0, 1},
    kaps_get_primitive, kaps_put_primitive},
    
   {NULL, KGEOM_QUADMESH_NORMAL_ALL, 0, KGEOM_SEGMENT_NORMAL,
    {1, 1, 1, 0, 0},
    kaps_get_primitive, kaps_put_primitive},
   {NULL, KGEOM_QUADMESH_NORMAL_VERTEX, 0, KGEOM_SEGMENT_NORMAL,
    {1, 0, 0, 0, 0},
    kaps_get_primitive, kaps_put_primitive},
   {NULL, KGEOM_QUADMESH_NORMAL_ROW, 0, KGEOM_SEGMENT_NORMAL,
    {1, 1, 0, 0, 0},
    kaps_get_primitive, kaps_put_primitive},
   {NULL, KGEOM_QUADMESH_NORMAL_COLUMN, 0, KGEOM_SEGMENT_NORMAL,
    {1, 0, 1, 0, 0},
    kaps_get_primitive, kaps_put_primitive},
       
   {NULL, KGEOM_QUADMESH_TEXTURE_COORD_ALL, 0, KGEOM_SEGMENT_TEXTURE_COORD,
    {1, 1, 1, 0, 0},
    kaps_get_primitive, kaps_put_primitive},
   {NULL, KGEOM_QUADMESH_TEXTURE_COORD_VERTEX,0,KGEOM_SEGMENT_TEXTURE_COORD,
    {1, 0, 0, 0, 0},
    kaps_get_primitive, kaps_put_primitive},
   {NULL, KGEOM_QUADMESH_TEXTURE_COORD_ROW, 0, KGEOM_SEGMENT_TEXTURE_COORD,
    {1, 1, 0, 0, 0},
    kaps_get_primitive, kaps_put_primitive},
   {NULL, KGEOM_QUADMESH_TEXTURE_COORD_COLUMN, 0,KGEOM_SEGMENT_TEXTURE_COORD,
    {1, 0, 1, 0, 0},
    kaps_get_primitive, kaps_put_primitive},

   {NULL, KGEOM_OCTMESH_LOCATION_ALL, 0, KGEOM_SEGMENT_LOCATION,
    {1, 1, 1, 1, 0},
    kaps_get_primitive, kaps_put_primitive},
   {NULL, KGEOM_OCTMESH_LOCATION_VERTEX, 0, KGEOM_SEGMENT_LOCATION,
    {0, 0, 0, 1, 0},
    kaps_get_primitive, kaps_put_primitive},
   {NULL, KGEOM_OCTMESH_LOCATION_ROW, 0, KGEOM_SEGMENT_LOCATION,
    {1, 0, 0, 1, 0},
    kaps_get_primitive, kaps_put_primitive},
   {NULL, KGEOM_OCTMESH_LOCATION_COLUMN, 0, KGEOM_SEGMENT_LOCATION,
    {0, 1, 0, 1, 0},
    kaps_get_primitive, kaps_put_primitive},
   {NULL, KGEOM_OCTMESH_LOCATION_PLANE, 0, KGEOM_SEGMENT_LOCATION,
    {1, 1, 0, 1, 0},
    kaps_get_primitive, kaps_put_primitive},

   {NULL, KGEOM_OCTMESH_LOCATION_MINIMUM, 0, KGEOM_SEGMENT_LOCATION,
    {0, 0, 0, 0, 0},
    kgeom_get_uniform_mesh, kgeom_put_uniform_mesh},
   {NULL, KGEOM_OCTMESH_LOCATION_MAXIMUM, 0, KGEOM_SEGMENT_LOCATION,
    {0, 0, 0, 0, 0},
    kgeom_get_uniform_mesh, kgeom_put_uniform_mesh},

   {NULL, KGEOM_OCTMESH_LOCATION_WIDTH_ALL, 0, KGEOM_SEGMENT_WIDTH,
    {1, 0, 0, 0, 0},
    kaps_get_primitive, kaps_put_primitive},
   {NULL, KGEOM_OCTMESH_LOCATION_WIDTH_POINT, 0, KGEOM_SEGMENT_WIDTH,
    {0, 0, 0, 0, 0},
    kaps_get_primitive, kaps_put_primitive},

   {NULL, KGEOM_OCTMESH_LOCATION_HEIGHT_ALL, 0, KGEOM_SEGMENT_HEIGHT,
    {1, 0, 0, 0, 0},
    kaps_get_primitive, kaps_put_primitive},
   {NULL, KGEOM_OCTMESH_LOCATION_HEIGHT_POINT, 0, KGEOM_SEGMENT_HEIGHT,
    {0, 0, 0, 0, 0},
    kaps_get_primitive, kaps_put_primitive},

   {NULL, KGEOM_OCTMESH_LOCATION_DEPTH_ALL, 0, KGEOM_SEGMENT_DEPTH,
    {1, 0, 0, 0, 0},
    kaps_get_primitive, kaps_put_primitive},
   {NULL, KGEOM_OCTMESH_LOCATION_DEPTH_POINT, 0, KGEOM_SEGMENT_DEPTH,
    {0, 0, 0, 0, 0},
    kaps_get_primitive, kaps_put_primitive},
   
   {NULL, KGEOM_OCTMESH_COLOR_ALL, 0, KPDS_SEGMENT_VALUE,
    {1, 1, 1, 0, 1},
    kaps_get_primitive, kaps_put_primitive},
   {NULL, KGEOM_OCTMESH_COLOR_VERTEX, 0, KPDS_SEGMENT_VALUE,
    {0, 0, 0, 0, 1},
    kaps_get_primitive, kaps_put_primitive},
   {NULL, KGEOM_OCTMESH_COLOR_ROW, 0, KPDS_SEGMENT_VALUE,
    {1, 0, 0, 0, 1},
    kaps_get_primitive, kaps_put_primitive},
   {NULL, KGEOM_OCTMESH_COLOR_COLUMN, 0, KPDS_SEGMENT_VALUE,
    {0, 1, 0, 0, 1},
    kaps_get_primitive, kaps_put_primitive},
   {NULL, KGEOM_OCTMESH_COLOR_PLANE, 0, KPDS_SEGMENT_VALUE,
    {1, 1, 0, 0, 1},
    kaps_get_primitive, kaps_put_primitive},

   {NULL, KGEOM_OCTMESH_NORMAL_ALL, 0, KGEOM_SEGMENT_NORMAL,
    {1, 1, 1, 1, 0},
    kaps_get_primitive, kaps_put_primitive},
   {NULL, KGEOM_OCTMESH_NORMAL_VERTEX, 0, KGEOM_SEGMENT_NORMAL,
    {1, 0, 0, 0, 0},
    kaps_get_primitive, kaps_put_primitive},
   {NULL, KGEOM_OCTMESH_NORMAL_ROW, 0, KGEOM_SEGMENT_NORMAL,
    {1, 1, 0, 0, 0},
    kaps_get_primitive, kaps_put_primitive},
   {NULL, KGEOM_OCTMESH_NORMAL_COLUMN, 0, KGEOM_SEGMENT_NORMAL,
    {1, 0, 1, 0, 0},
    kaps_get_primitive, kaps_put_primitive},
   {NULL, KGEOM_OCTMESH_NORMAL_PLANE, 0, KGEOM_SEGMENT_NORMAL,
    {1, 1, 1, 0, 0},
    kaps_get_primitive, kaps_put_primitive},
       
   {NULL, KGEOM_OCTMESH_TEXTURE_COORD_ALL, 0, KGEOM_SEGMENT_TEXTURE_COORD,
    {1, 1, 1, 1, 0},
    kaps_get_primitive, kaps_put_primitive},
   {NULL, KGEOM_OCTMESH_TEXTURE_COORD_VERTEX,0,KGEOM_SEGMENT_TEXTURE_COORD,
    {1, 0, 0, 0, 0},
    kaps_get_primitive, kaps_put_primitive},
   {NULL, KGEOM_OCTMESH_TEXTURE_COORD_ROW, 0, KGEOM_SEGMENT_TEXTURE_COORD,
    {1, 1, 0, 0, 0},
    kaps_get_primitive, kaps_put_primitive},
   {NULL, KGEOM_OCTMESH_TEXTURE_COORD_COLUMN, 0,KGEOM_SEGMENT_TEXTURE_COORD,
    {1, 1, 0, 0, 0},
    kaps_get_primitive, kaps_put_primitive},
   {NULL, KGEOM_OCTMESH_TEXTURE_COORD_PLANE, 0, KGEOM_SEGMENT_TEXTURE_COORD,
    {1, 1, 1, 0, 0},
    kaps_get_primitive, kaps_put_primitive},

   {NULL, KGEOM_TEXTURE2D_ALL, 0, KPDS_SEGMENT_VALUE,
    {1, 1, 0, 0, 1},
    kaps_get_primitive, kaps_put_primitive},
   {NULL, KGEOM_TEXTURE2D_VERTEX, 0, KPDS_SEGMENT_VALUE,
    {0, 0, 0, 0, 1},
    kaps_get_primitive, kaps_put_primitive},
   {NULL, KGEOM_TEXTURE2D_ROW, 0, KPDS_SEGMENT_VALUE,
    {1, 0, 0, 0, 1},
    kaps_get_primitive, kaps_put_primitive},
   {NULL, KGEOM_TEXTURE2D_COLUMN, 0, KPDS_SEGMENT_VALUE,
    {0, 1, 0, 0, 1},
    kaps_get_primitive, kaps_put_primitive},

   {NULL, KGEOM_TEXTURE3D_ALL, 0, KPDS_SEGMENT_VALUE,
    {1, 1, 1, 0, 1},
    kaps_get_primitive, kaps_put_primitive},
   {NULL, KGEOM_TEXTURE3D_VERTEX, 0, KPDS_SEGMENT_VALUE,
    {0, 0, 0, 0, 1},
    kaps_get_primitive, kaps_put_primitive},
   {NULL, KGEOM_TEXTURE3D_ROW, 0, KPDS_SEGMENT_VALUE,
    {1, 0, 0, 0, 1},
    kaps_get_primitive, kaps_put_primitive},
   {NULL, KGEOM_TEXTURE3D_COLUMN, 0, KPDS_SEGMENT_VALUE,
    {0, 1, 0, 0, 1},
    kaps_get_primitive, kaps_put_primitive},
   {NULL, KGEOM_TEXTURE3D_PLANE, 0, KPDS_SEGMENT_VALUE,
    {1, 1, 0, 0, 1},
    kaps_get_primitive, kaps_put_primitive},

};

int kgeom_num_meshes = sizeof(kgeom_meshes) / sizeof(kgeom_meshes[0]);

/* ################################################################## */

/*
 *    ================================================================== 
 *    QUADMESH Routines
 *    ==================================================================
 */

/*-----------------------------------------------------------
|
|  Routine Name: (static) _kgeom_query_quadmesh
|
|       Purpose: Determines the existence of a quadmesh
|
|         Input: obj - the data object to check
|
|        Output: none
|
|       Returns: TRUE (1) if quadmesh exists , FALSE (0) otherwise
|
|    Written By: Steve Kubica
|          Date: Feb 13, 1993 16:17
| Modifications:
|
------------------------------------------------------------*/
static int 
_kgeom_query_quadmesh(kobject obj)
{
   int loc_present;
   int width;
   int height;
   int depth;
   

   KGEOM_INIT;

   if (obj == NULL)
      kgeom_fail_error(KGEOM_EOBJ_INVALID);

   /* -- mesh have at least location to be a quadmesh -- */
   loc_present = kdms_query_segment(obj, KGEOM_SEGMENT_LOCATION);
   if (!loc_present)
      return FALSE;
   
   kpds_get_attribute(obj, KPDS_LOCATION_SIZE,
		      &width, &height, &depth, NULL);
   
   /* -- must have depth = 1, otherwise it is really an octmesh -- */
   if (width < 2 || height < 2 || depth != 1)
      return FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _kgeom_set_quadmesh_size
|
|       Purpose: This routine is used to set the size
|		 of a quadmesh where a quadmesh is considered
|		 the collection of location, value, normal, and
|		 texture coord segments.  The element size and
|		 data types are also set on each segment according
|		 to the current geometry presentation.
|
|         Input: obj    - object with the quadmesh
|		 width  - width 
|		 height - height
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica
|          Date: Feb 20, 1994 14:09
| Modifications:
|
------------------------------------------------------------*/
static int 
_kgeom_set_quadmesh_size(
   kobject  obj,
   int      width,
   int      height)
{
   /* location segment */
   if (kdms_query_segment(obj, KGEOM_SEGMENT_LOCATION))
   {
      int loctype, esize;

      /* get all the location dependent attributes */
      kgeom_get_attributes(obj, KGEOM_OBJECT, 
			   KGEOM_LOCATION_DATA_TYPE, &loctype,
			   KGEOM_LOCATION_SIZE,      &esize, 
			   NULL);

      /* set the location segment characteristics */
      kaps_set_attributes(obj, KGEOM_SEGMENT_LOCATION,
			  KDMS_DATA_TYPE, loctype, NULL);

      kpds_set_attribute(obj, KPDS_LOCATION_SIZE, 
			 (width  < 1) ? 1 : width,
			 (height < 1) ? 1 : height, 1, 
			 (esize  < 1) ? 1 : esize);
   }

   /* value segment */
   if (kdms_query_segment(obj, KPDS_SEGMENT_VALUE))
   {
      int coltype, esize;
      int colsize[5] = {1, 1, 1, 1, 1};
      int nwidth, nheight;
      int mode;
      int map_width;
      int layout = kgeom_layout(obj);
      

      kgeom_get_attribute(obj, KGEOM_OBJECT, KGEOM_COLOR_DATA_TYPE, &coltype);

      /* determine the color vector size (this will be 1 if there is a map) */
      esize = kgeom_color_size(obj, KPDS_SEGMENT_VALUE);

      /* modify the width and height according to the layout */
      nwidth  = (layout == KPER_FACE) ?  width - 1 : width;
      nheight = (layout == KPER_FACE) ? height - 1 : height;

      /* 
       * get the old mapping mode and then turn it off so we don't go
       * changing the map size when we set the element size of the value.
       */
      kgeom_get_attribute(obj, KGEOM_OBJECT, KGEOM_MAPPING_MODE, &mode);
      kgeom_set_attribute(obj, KGEOM_OBJECT, KGEOM_MAPPING_MODE, KUNMAPPED);

      /* determine the initial physical size */
      colsize[0] = (width  < 1) ? 1 : width;
      colsize[1] = (height < 1) ? 1 : height;
      colsize[4] = (esize  < 1) ? 1 : esize;

      /* set the color segment characteristics */
      kdms_set_attributes(obj, KPDS_SEGMENT_VALUE,
			  KDMS_DATA_TYPE, coltype, 
			  KDMS_SIZE,      colsize, NULL);

      /*   WARNING /// WARNING /// WARNING /// WARNING
       *
       *   This is something of a kludge to allow for rmonster
       *   to directly interpret most pds data models properly.
       *
       *   What rmonster expects : RGB or RGBa floats ranging from 0.0->1.0
       *
       *   We can do a number of presentation tricks to get that :
       *    size : if we only have a value size of 1, then 
       *           either turn mapping mode on to make it look like 3,
       *           or, if we don't have a map, then turn interpolating
       *	   to pixel replication and make our element size 3.
       *    type : if we want float or double and we have byte, assume
       *           the byte ranges from 0->255 and scale it so it 
       *           comes out 0->1.  do this to either the map or the
       *           value depending on where we're getting the data from.
       *           also, if we have float value and a map, normalize the
       *           float value so it indexes into the map.
       */
      {
	 char *color_segment;
	 int  *phy_val_size;
	 int   phy_val_type;
	 int   phy_map_type;
	 int   phy_type;
	 int   has_map = kpds_query_map(obj);	 

	 kdms_get_attributes(obj, KPDS_SEGMENT_VALUE,
			     KDMS_PHYSICAL_SIZE, &phy_val_size,
			     KDMS_PHYSICAL_DATA_TYPE, &phy_val_type, NULL);
	 if (has_map)
	    kdms_get_attribute(obj, KPDS_SEGMENT_MAP,
			       KDMS_PHYSICAL_DATA_TYPE, &phy_map_type);

	 /* -- determine if we get colors from the map -- */
	 if (phy_val_size[4] == 1 && has_map)
	 {
	    color_segment = KPDS_SEGMENT_MAP;
	    phy_type = phy_map_type;
	    kpds_set_attribute(obj, KPDS_MAPPING_MODE, KMAPPED);
	    kgeom_set_attribute(obj, KGEOM_OBJECT, KGEOM_COLORSPACE, KRGB);
	 }
	 else
	 {
	    color_segment = KPDS_SEGMENT_VALUE;
	    phy_type = phy_val_type;
	 }

	 /* -- if we have no map, then make the value segment look big -- */
	 if (phy_val_size[4] == 1 && !has_map)
	 {
	    int psize[5] = {-1,-1,-1,-1,-1};

	    psize[4] = 3;
	    kdms_set_attributes(obj, KPDS_SEGMENT_VALUE, 
				KDMS_INTERPOLATE, KZERO_ORDER,
				KDMS_SIZE,        psize, NULL);

	    kgeom_set_attribute(obj, KGEOM_OBJECT, KGEOM_COLORSPACE, KRGB);
	 }
	 
	 /* -- normalize float value so it indexes into map -- */
	 if ((phy_val_type == KFLOAT || phy_val_type == KDOUBLE) && has_map)
	    kdms_set_attributes(obj, KPDS_SEGMENT_VALUE, 
				KDMS_SCALING,  KNORMALIZE, 
				KDMS_NORM_MIN, 0.0,
				KDMS_NORM_MAX, 255.0, NULL);

	 /* -- if we have byte or unsigned byte colors, make range 0-1 -- */
	 if ((phy_type == KBYTE || phy_type == KUBYTE) && coltype == KFLOAT)
	    kdms_set_attributes(obj, color_segment,
				KDMS_SCALING,      KSCALE, 
				KDMS_DATA_TYPE,    KFLOAT, 
				KDMS_SCALE_OFFSET, 0.0, 0.0,
				KDMS_SCALE_FACTOR, 1.0/256.0, NULL);

      }

#if 0
      /* also fix up the map width if we have a map */
      if ((map_width = kgeom_color_size(obj, KGEOM_SEGMENT_MAP)) != -1)
	 kpds_set_attribute(obj, KPDS_MAP_SIZE, map_width, -1, -1, -1, -1);

      /* pop back the old mapping mode */
      kgeom_set_attribute(obj, KGEOM_OBJECT, KGEOM_MAPPING_MODE, mode);
#endif     
   }
   

   /* normals segment */
   if (kdms_query_segment(obj, KGEOM_SEGMENT_NORMAL))
   {
      int normtype, esize;
      int normsize[4] = {1, 1, 1, 1};

      kgeom_get_attributes(obj, KGEOM_OBJECT, 
			   KGEOM_NORMAL_DATA_TYPE, &normtype,
			   KGEOM_LOCATION_SIZE,    &esize, NULL);

      /* determine the initial physical size */
      normsize[0] = (esize  < 1) ? 1 : esize;
      normsize[1] = (width  < 1) ? 1 : width;
      normsize[2] = (height < 1) ? 1 : height;

      /* set the value segment characteristics */
      kdms_set_attributes(obj, KGEOM_SEGMENT_NORMAL,
			  KDMS_DATA_TYPE,   normtype, 
			  KDMS_SIZE,        normsize,
			  NULL);
   }

   /* texture coord segment */
   if (kdms_query_segment(obj, KGEOM_SEGMENT_TEXTURE_COORD))
   {
      int textype, esize;
      int texsize[4] = {1, 1, 1, 1};

      kgeom_get_attributes(obj, KGEOM_OBJECT, 
			   KGEOM_TEXTURE_COORD_DATA_TYPE, &textype,
			   KGEOM_TEXTURE_COORD_SIZE,      &esize, 
			   NULL);

      /* determine the initial physical size */
      texsize[0] = (width  < 1) ? 1 : width;
      texsize[1] = (height < 1) ? 1 : height;
      texsize[3] = (esize  < 1) ? 1 : esize;

      /* create the texture coords segment and set up its characteristics */
      kdms_set_attributes(obj, KGEOM_SEGMENT_TEXTURE_COORD,
				  KDMS_DATA_TYPE,   textype, 
                                  KDMS_SIZE,        texsize,
				  NULL);
   }
   return TRUE;
}

/*
 *    ================================================================== 
 *    OCTMESH Routines
 *    ==================================================================
 */

/*-----------------------------------------------------------
|
|  Routine Name: (static) _kgeom_query_octmesh
|
|       Purpose: Determines the existence of a octmesh
|
|         Input: obj - the data object to check
|
|        Output: none
|
|       Returns: TRUE (1) if octmesh exists , FALSE (0) otherwise
|
|    Written By: Steve Kubica
|          Date: Feb 13, 1993 16:17
| Modifications:
|
------------------------------------------------------------*/
static int 
_kgeom_query_octmesh(kobject obj)
{
   int loc_present;
   int width;
   int height;
   int depth;
   

   KGEOM_INIT;

   if (obj == NULL)
      kgeom_fail_error(KGEOM_EOBJ_INVALID);

   /* -- mesh have at least location to be a quadmesh -- */
   loc_present = kdms_query_segment(obj, KGEOM_SEGMENT_LOCATION);
   if (!loc_present)
      return(FALSE);
   
   kpds_get_attribute(obj, KPDS_LOCATION_SIZE, 
		       &width, &height, &depth, NULL);
   
   /* -- must have depth > 1, otherwise it is really an quadmesh -- */
   if (width < 2 || height < 2 || depth < 2)
      return FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|
|  Routine Name: _kgeom_set_octmesh_size
|
|       Purpose: This routine is used to set the size
|		 of a octmesh where a octmesh is considered
|		 the collection of location, value, normal, and
|		 texture coord segments.  The element size and
|		 data types are also set on each segment according
|		 to the current geometry presentation.
|
|         Input: obj    - object with the octmesh
|		 width  - width 
|		 height - height
|		 depth  - depth
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica
|          Date: Feb 20, 1994 14:09
| Modifications:
|
------------------------------------------------------------*/
static int 
_kgeom_set_octmesh_size(
   kobject obj,
   int      width,
   int      height,
   int      depth)
{
   /* location segment */
   if (kdms_query_segment(obj, KGEOM_SEGMENT_LOCATION))
   {
      int loctype, esize;

      /* get all the location dependent attributes */
      kgeom_get_attributes(obj, KGEOM_OBJECT, 
			   KGEOM_LOCATION_DATA_TYPE, &loctype,
			   KGEOM_LOCATION_SIZE,      &esize, 
			   NULL);

      /* set the location segment characteristics */
      kaps_set_attributes(obj, KGEOM_SEGMENT_LOCATION,
			  KDMS_DATA_TYPE, loctype, NULL);

      kpds_set_attribute(obj, KPDS_LOCATION_SIZE, 
			 (width  < 1) ? 1 : width,
			 (height < 1) ? 1 : height,
			 (depth  < 1) ? 1 : depth,
			 (esize  < 1) ? 1 : esize);
   }

   /* value segment */
   if (kdms_query_segment(obj, KPDS_SEGMENT_VALUE))
   {
      int coltype, esize;
      int colsize[4] = {1, 1, 1, 1};
      int nwidth, nheight, ndepth;
      int mode;
      int map_width;
      int layout = kgeom_layout(obj);


      kgeom_get_attribute(obj, KGEOM_OBJECT, KGEOM_COLOR_DATA_TYPE, &coltype);

      /* determine the color vector size (this will be 1 if there is a map) */
      esize = kgeom_color_size(obj, KPDS_SEGMENT_VALUE);

      /* modify the width and height according to the layout */
      nwidth  = (layout == KPER_FACE) ?  width - 1 : width;
      nheight = (layout == KPER_FACE) ? height - 1 : height;
      ndepth  = (layout == KPER_FACE) ?  depth - 1 : depth;

      /* 
       * get the old mapping mode and then turn it off so we don't go
       * changing the map size when we set the element size of the value.
       */
      kgeom_get_attribute(obj, KGEOM_OBJECT, KGEOM_MAPPING_MODE, &mode);
      kgeom_set_attribute(obj, KGEOM_OBJECT, KGEOM_MAPPING_MODE, KUNMAPPED);

      /* determine the initial physical size */
      colsize[0] = (width  < 1) ? 1 : width;
      colsize[1] = (height < 1) ? 1 : height;
      colsize[2] = (depth  < 1) ? 1 : depth;
      colsize[4] = (esize  < 1) ? 1 : esize;

      /* set the color segment characteristics */
      kdms_set_attributes(obj, KPDS_SEGMENT_VALUE,
			  KDMS_DATA_TYPE, coltype, 
			  KDMS_SIZE,      colsize, NULL);


      /*   WARNING /// WARNING /// WARNING /// WARNING
       *
       *   This is something of a kludge to allow for rmonster
       *   to directly interpret most pds data models properly.
       *
       *   What rmonster expects : RGBa floats ranging from 0.0->1.0
       *
       *   We can do a number of presentation tricks to get that :
       *    size : if we only have a value size of 1, then 
       *           either turn mapping mode on to make it look like 3,
       *           or, if we don't have a map, then turn interpolating
       *	   to pixel replication and make our element size 3.
       *    type : if we want float or double and we have byte, assume
       *           the byte ranges from 0->255 and scale it so it 
       *           comes out 0->1.  do this to either the map or the
       *           value depending on where we're getting the data from.
       *           also, if we have float value and a map, normalize the
       *           float value so it indexes into the map.
       */
      {
	 char *color_segment;
	 int  *phy_val_size;
	 int   phy_val_type;
	 int   phy_map_type;
	 int   phy_type;
	 int   has_map = kpds_query_map(obj);	 

	 kdms_get_attributes(obj, KPDS_SEGMENT_VALUE,
			     KDMS_PHYSICAL_SIZE, &phy_val_size,
			     KDMS_PHYSICAL_DATA_TYPE, &phy_val_type, NULL);

	 if (has_map)
	    kdms_get_attribute(obj, KPDS_SEGMENT_MAP,
			       KDMS_PHYSICAL_DATA_TYPE, &phy_map_type);
	 
	 /* -- volume data must be rgba for rmonster -- */
	 if (phy_val_size[4] == 1 && has_map)
	 {
	    color_segment = KPDS_SEGMENT_MAP;
	    phy_type = phy_map_type;
	    
	    /* -- set up so that we end up with alpha values of 0.25 -- */
	    kpds_set_attribute(obj, KPDS_MAP_SIZE, 4, -1, -1, -1, -1);
	    kpds_set_attribute(obj, KPDS_MAP_INTERPOLATE, KPAD);
	    kpds_set_attribute(obj, KPDS_MAP_PAD_VALUE, 64.0, 0.0);

	    kgeom_set_attribute(obj, KGEOM_OBJECT, KGEOM_COLORSPACE, KRGB);
	    kgeom_set_attribute(obj, KGEOM_OBJECT, KGEOM_HAS_ALPHA, TRUE);
	    kpds_set_attribute(obj, KPDS_MAPPING_MODE, KMAPPED);
	 }
	 else
	 {
	    color_segment = KPDS_SEGMENT_VALUE;
	    phy_type = phy_val_type;
	 }

	 if (phy_val_size[4] == 1 && !has_map)
	 {
	    int psize[5] = {-1,-1,-1,-1,-1};

	    psize[4] = 4;
	    kdms_set_attributes(obj, KPDS_SEGMENT_VALUE, 
				KDMS_INTERPOLATE, KZERO_ORDER,
				KDMS_SIZE, psize, NULL);
	    kgeom_set_attribute(obj, KGEOM_OBJECT, KGEOM_COLORSPACE, KRGB);
	    kgeom_set_attribute(obj, KGEOM_OBJECT, KGEOM_HAS_ALPHA, TRUE);
	 }

	 /* -- normalize float value so it indexes into map -- */
	 if ((phy_val_type == KFLOAT || phy_val_type == KDOUBLE) && has_map)
	    kdms_set_attributes(obj, KPDS_SEGMENT_VALUE, 
				KDMS_SCALING,  KNORMALIZE, 
				KDMS_NORM_MIN, 0.0,
				KDMS_NORM_MAX, 255.0, NULL);

	 /* -- if we have byte or unsigned byte colors, make range 0-1 -- */
	 if ((phy_type == KBYTE || phy_type == KUBYTE) && coltype == KFLOAT)
	    kdms_set_attributes(obj, color_segment,
				KDMS_SCALING,      KSCALE, 
				KDMS_DATA_TYPE,    KFLOAT,
				KDMS_SCALE_OFFSET, 0.0, 0.0,
				KDMS_SCALE_FACTOR, 1.0/256.0, NULL);
      }
      
#if 0
      /* also fix up the map width if we have a map */
      if ((map_width = kgeom_color_size(obj, KGEOM_SEGMENT_MAP)) != -1)
	 kpds_set_attribute(obj, KPDS_MAP_SIZE, map_width, 1, 1, 1, 1);
      
      /* pop back the old mapping mode */
      kgeom_set_attribute(obj, KGEOM_OBJECT, KGEOM_MAPPING_MODE, mode);
#endif

   }

   /* normals segment */
   if (kdms_query_segment(obj, KGEOM_SEGMENT_NORMAL))
   {
      int normtype, esize;
      int normsize[4] = {1, 1, 1, 1};

      kgeom_get_attributes(obj, KGEOM_OBJECT, 
			   KGEOM_NORMAL_DATA_TYPE, &normtype,
			   KGEOM_LOCATION_SIZE,    &esize, NULL);

      /* determine the initial physical size */
      normsize[0] = (esize  < 1) ? 1 : esize;
      normsize[1] = (width  < 1) ? 1 : width;
      normsize[2] = (height < 1) ? 1 : height;
      normsize[3] = (depth  < 1) ? 1 : depth;

      /* set the value segment characteristics */
      kdms_set_attributes(obj, KGEOM_SEGMENT_NORMAL,
			  KDMS_DATA_TYPE,   normtype, 
			  KDMS_SIZE,        normsize,
			  NULL);
   }

   /* texture coord segment */
   if (kdms_query_segment(obj, KGEOM_SEGMENT_TEXTURE_COORD))
   {
      int textype, esize;
      int texsize[4] = {1, 1, 1, 1};

      kgeom_get_attributes(obj, KGEOM_OBJECT, 
			   KGEOM_TEXTURE_COORD_DATA_TYPE, &textype,
			   KGEOM_TEXTURE_COORD_SIZE,      &esize, 
			   NULL);

      /* determine the initial physical size */
      texsize[0] = (width  < 1) ? 1 : width;
      texsize[1] = (height < 1) ? 1 : height;
      texsize[2] = (depth  < 1) ? 1 : depth;
      texsize[3] = (esize  < 1) ? 1 : esize;

      /* create the texture coords segment and set up its characteristics */
      kdms_set_attributes(obj, KGEOM_SEGMENT_TEXTURE_COORD,
			  KDMS_DATA_TYPE,   textype, 
			  KDMS_SIZE,        texsize,
			  NULL);
   }
   return TRUE;
}

/*
 *    ================================================================== 
 *    TEXTURE2D Routines
 *    ==================================================================
 */

/*-----------------------------------------------------------
|
|  Routine Name: (static) _kgeom_query_texture2d
|
|       Purpose: Determines the existence of a 2d texture
|
|         Input: obj - the data object to check
|
|        Output: none
|
|       Returns: TRUE (1) if 2d texture exists , FALSE (0) otherwise
|
|    Written By: Steve Kubica
|          Date: Feb 13, 1993 16:17
| Modifications:
|
------------------------------------------------------------*/
static int
_kgeom_query_texture2d(kobject obj)
{
   int color_present;
   int depth;
   
   
   KGEOM_INIT;

   if (obj == NULL)
      kgeom_fail_error(KGEOM_EOBJ_INVALID);

   /* -- if there is location data present, then it's a mesh -- */
   if (kdms_query_segment(obj, KGEOM_SEGMENT_LOCATION))
	return FALSE;

   color_present = kdms_query_segment(obj, KPDS_SEGMENT_VALUE);
   if (!color_present)
      return FALSE;

   kpds_get_attribute(obj, KPDS_VALUE_SIZE, NULL, NULL, &depth, NULL, NULL);
   
   /* -- color must have depth = 1, otherwise really it is a 3D texture -- */
   if (depth != 1)
      return FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _kgeom_set_texture2d_size
|
|       Purpose: This routine is used to set the size
|		 of a 2d texture where a 2d texture is considered
|		 to be the value segment.  The element size and
|		 data types are also set on each segment according
|		 to the current geometry presentation.
|
|         Input: obj - object with the 2d texture
|		 width  - width 
|		 height - height
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica
|          Date: Feb 20, 1994 14:09
| Modifications:
|
------------------------------------------------------------*/
static int 
_kgeom_set_texture2d_size(
   kobject obj,
   int     width,
   int     height)
{
   /* value segment */
   if (kdms_query_segment(obj, KPDS_SEGMENT_VALUE))
   {
      int coltype, esize;
      int mode;
      int map_width;


      kgeom_get_attribute(obj, KGEOM_OBJECT, KGEOM_COLOR_DATA_TYPE, &coltype);

      /* determine the color vector size (this will be 1 if there is a map) */
      esize = kgeom_color_size(obj, KPDS_SEGMENT_VALUE);

      /* 
       * get the old mapping mode and then turn it off so we don't go
       * changing the map size when we set the element size of the value.
       */
      kgeom_get_attribute(obj, KGEOM_OBJECT, KGEOM_MAPPING_MODE, &mode);
      kgeom_set_attribute(obj, KGEOM_OBJECT, KGEOM_MAPPING_MODE, KUNMAPPED);

      /* set the color segment characteristics */
      kaps_set_attributes(obj, KPDS_SEGMENT_VALUE,
			  KDMS_DATA_TYPE, coltype, NULL);

      kpds_set_attribute(obj, KPDS_VALUE_SIZE, 
			 (width  < 1) ? 1 : width,
			 (height < 1) ? 1 : height, 1, 1, 
			 (esize  < 1) ? 1 : esize);

      /* also fix up the map width if we have a map */
      if ((map_width = kgeom_color_size(obj, KGEOM_SEGMENT_MAP)) != -1)
	 kpds_set_attribute(obj, KPDS_MAP_SIZE, map_width, 1, 1, 1, 1);

      /* pop back the old mapping mode */
      kgeom_set_attribute(obj, KGEOM_OBJECT, KGEOM_MAPPING_MODE, mode);
   }
   return TRUE;
}

/*
 *    ================================================================== 
 *    TEXTURE3D Routines
 *    ==================================================================
 */

/*-----------------------------------------------------------
|
|  Routine Name: (static) _kgeom_query_texture3d
|
|       Purpose: Determines the existence of a 3d texture
|
|         Input: obj - the data object to check
|
|        Output: none
|
|       Returns: TRUE (1) if 3d texture exists , FALSE (0) otherwise
|
|    Written By: Steve Kubica
|          Date: Feb 13, 1993 16:17
| Modifications:
|
------------------------------------------------------------*/
static int 
_kgeom_query_texture3d(kobject obj)
{
   int color_present;
   int depth;
   
   
   KGEOM_INIT;

   if (obj == NULL)
      kgeom_fail_error(KGEOM_EOBJ_INVALID);

   /* -- if there is location data present, then it's a mesh -- */
   if (kdms_query_segment(obj, KGEOM_SEGMENT_LOCATION))
	return FALSE;
   
   color_present = kdms_query_segment(obj, KPDS_SEGMENT_VALUE);
   if (!color_present)
      return FALSE;

   kpds_get_attribute(obj, KPDS_VALUE_SIZE, NULL, NULL, &depth, NULL, NULL);
   
   /* -- color must have depth = 1, otherwise really it is a 3D texture -- */
   if (depth < 2)
      return FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _kgeom_set_texture3d_size
|
|       Purpose: This routine is used to set the size
|		 of a 3d texture where a 3d texture is considered
|		 to be the value segment.  The element size and
|		 data types are also set on each segment according
|		 to the current geometry presentation.
|
|         Input: obj    - object with the 3d texture
|		 width  - width 
|		 height - height
|		 depth  - depth
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica
|          Date: Feb 20, 1994 14:09
| Modifications:
|
------------------------------------------------------------*/
static int 
_kgeom_set_texture3d_size(
   kobject  obj,
   int      width,
   int      height,
   int      depth)
{
   /* value segment */
   if (kdms_query_segment(obj, KPDS_SEGMENT_VALUE))
   {
      int coltype, esize;
      int mode;
      int map_width;


      kgeom_get_attribute(obj, KGEOM_OBJECT, KGEOM_COLOR_DATA_TYPE, &coltype);

      /* determine the color vector size (this will be 1 if there is a map) */
      esize = kgeom_color_size(obj, KPDS_SEGMENT_VALUE);

      /* 
       * get the old mapping mode and then turn it off so we don't go
       * changing the map size when we set the element size of the value.
       */
      kgeom_get_attribute(obj, KGEOM_OBJECT, KGEOM_MAPPING_MODE, &mode);
      kgeom_set_attribute(obj, KGEOM_OBJECT, KGEOM_MAPPING_MODE, KUNMAPPED);

      /* set the color segment characteristics */
      kaps_set_attributes(obj, KPDS_SEGMENT_VALUE,
			  KDMS_DATA_TYPE, coltype, NULL);

      kpds_set_attribute(obj, KPDS_VALUE_SIZE, 
			 (width  < 1) ? 1 : width,
			 (height < 1) ? 1 : height, 
			 (depth  < 1) ? 1 : depth,   1, 
			 (esize  < 1) ? 1 : esize);

      /* also fix up the map width if we have a map */
      if ((map_width = kgeom_color_size(obj, KGEOM_SEGMENT_MAP)) != -1)
	 kpds_set_attribute(obj, KPDS_MAP_SIZE, map_width, 1, 1, 1, 1);

      /* pop back the old mapping mode */
      kgeom_set_attribute(obj, KGEOM_OBJECT, KGEOM_MAPPING_MODE, mode);
   }
   return TRUE;
}

/* ################################################################## */

/*
 *    ================================================================== 
 *    Mesh Presentation Initialization
 *    ==================================================================
 */

/*-----------------------------------------------------------
|
|  Routine Name: (static) _kgeom_init_mesh
|
|       Purpose: This routine returns the index into
|                the mesh table for a given mesh primitive
|                name.  If the primitive is not found, 
|                then an errno is set.
|
|         Input: prim - the name of the primitive to look up
|
|        Output: none
|
|       Returns: the index into the primitive table on success, 
|                -1 otherwise.
|
|    Written By: Steve Kubica
|          Date: Feb 20, 1994 14:09
| Modifications:
|
------------------------------------------------------------*/
static int 
_kgeom_init_mesh(
   kobject       obj,
   int           primitive,
   int           put_operation)
{
   int  i;
   int  index = -1;


   /* find the appropriate specific primitive */
   for (i = 0; i < kgeom_num_meshes && index == -1; i++) 
      if (kgeom_meshes[i].token == primitive)
         index = i;

   if (index == -1)
   {
      kgeom_error(KGEOM_EINVALID_PRIMITIVE);  
      return -1;
   }

   /* go for broke(n) and try to create the segment here */
   if (put_operation)
      if (!kdms_query_segment(obj, kgeom_meshes[index].segment))
      {
	 if (!kdms_create_segment(obj, kgeom_meshes[index].segment))
	    return FALSE;
      }

   /* 
    *   depending on the class of primitive, syncronize the presentation
    *   (size and data type) and return the appropriate set of handlers
    */

   if (is_quadmesh(primitive)) 
   {
      int width, height;
      
      kgeom_get_attribute(obj, KGEOM_OBJECT, 
			  KGEOM_QUADMESH_SIZE, &width, &height);
      
      _kgeom_set_quadmesh_size(obj, width, height);
   }
   else if (is_octmesh(primitive))
   {
      int width, height, depth;
      
      kgeom_get_attribute(obj, KGEOM_OBJECT, 
			  KGEOM_OCTMESH_SIZE, &width, &height, &depth);

      _kgeom_set_octmesh_size(obj, width, height, depth);
   }
   else if (is_texture2d(primitive))
   {
      int width, height;

      kgeom_get_attribute(obj, KGEOM_OBJECT, 
			  KGEOM_TEXTURE2D_SIZE, &width, &height);
      
      _kgeom_set_texture2d_size(obj, width, height);
   }
   else if (is_texture3d(primitive))
   {
      int width, height, depth;

      kgeom_get_attribute(obj, KGEOM_OBJECT, 
			  KGEOM_TEXTURE3D_SIZE, &width, &height, &depth);
      
      _kgeom_set_texture3d_size(obj, width, height, depth);
   }

   return index;
}

/*
 *    ================================================================== 
 *    Mesh Uniform Location Data Routines
 *    ==================================================================
 */

/*-----------------------------------------------------------
|
|  Routine Name: kgeom_get_uniform_mesh
|
|       Purpose: This routine gets the uniform mesh primitives
|
|         Input: obj  - the object to retrieve the primitive from
|                prim - the primitive to retrieve
|                data - pointer to allocated data with which to
|			return the primitive, or NULL if we should
|			perform the allocation.   
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica
|          Date: Apr 29, 1994 11:17
| Modifications:
|
------------------------------------------------------------*/
kaddr
kgeom_get_uniform_mesh(
   kobject         obj,
   kaps_primitive *prim,
   kaddr           data)
{
   double w, h, d;
   int    dim;
   int    type;
   kaddr  rdata = NULL;
   
   
   /* -- depending on the primitive, get either the minimum or maximum -- */
   if (prim->token == KGEOM_QUADMESH_LOCATION_MINIMUM || 
       prim->token == KGEOM_OCTMESH_LOCATION_MINIMUM)
      kpds_get_attribute(obj, KPDS_LOCATION_BEGIN, &w, &h, &d);

   else if (prim->token == KGEOM_QUADMESH_LOCATION_MAXIMUM || 
	    prim->token == KGEOM_OCTMESH_LOCATION_MAXIMUM)
      kpds_get_attribute(obj, KPDS_LOCATION_END, &w, &h, &d);

   else
      return NULL;

   /* -- determine the dimensionality of the primitive -- */
   if (prim->token == KGEOM_QUADMESH_LOCATION_MINIMUM || 
       prim->token == KGEOM_QUADMESH_LOCATION_MAXIMUM)
      dim = 2;  /* quadmeshes are 2 */

   else if (prim->token == KGEOM_OCTMESH_LOCATION_MINIMUM || 
	    prim->token == KGEOM_OCTMESH_LOCATION_MAXIMUM)
      dim = 3;  /* octmeshes are 3 */
   
   kgeom_get_attribute(obj, KGEOM_OBJECT, KGEOM_LOCATION_DATA_TYPE, &type);
   
   /* -- allocate space if it doesn't look like they gave us any -- */
   if (data == NULL)
      rdata = kcalloc(dim, kdata_size(type));
   else
      rdata = data;
   
   /* -- fill out the data in the appropriate data type -- */
   switch (type)
   {
      case KBYTE:
	 {
	    char *ldata = (char *) rdata;

	    ldata[KWID] = (char) w;
	    ldata[KHGT] = (char) h;
	    if (dim == 3) ldata[KDEP] = d;
	 }
	 break;

      case KUBYTE:
	 {
	    unsigned char *ldata = (unsigned char *) rdata;

	    ldata[KWID] = (unsigned char) w;
	    ldata[KHGT] = (unsigned char) h;
	    if (dim == 3) ldata[KDEP] = (unsigned char) d;
	 }
	 break;

      case KSHORT:
	 {
	    short *ldata = (short *) rdata;

	    ldata[KWID] = (short) w;
	    ldata[KHGT] = (short) h;
	    if (dim == 3) ldata[KDEP] = (short) d;
	 }
	 break;

      case KUSHORT:
	 {
	    unsigned short *ldata = (unsigned short *) rdata;

	    ldata[KWID] = (unsigned short) w;
	    ldata[KHGT] = (unsigned short) h;
	    if (dim == 3) ldata[KDEP] = (unsigned short) d;
	 }
	 break;

      case KINT:
	 {
	    int *ldata = (int *) rdata;

	    ldata[KWID] = (int) w;
	    ldata[KHGT] = (int) h;
	    if (dim == 3) ldata[KDEP] = (int) d;
	 }
	 break;

      case KUINT:
	 {
	    unsigned int *ldata = (unsigned int *) rdata;

	    ldata[KWID] = (unsigned int) w;
	    ldata[KHGT] = (unsigned int) h;
	    if (dim == 3) ldata[KDEP] = (unsigned int) d;
	 }
	 break;

      case KLONG:
	 {
	    long *ldata = (long *) rdata;

	    ldata[KWID] = (long) w;
	    ldata[KHGT] = (long) h;
	    if (dim == 3) ldata[KDEP] = (long) d;
	 }
	 break;

      case KULONG:
	 {
	    unsigned long *ldata = (unsigned long *) rdata;

	    ldata[KWID] = (unsigned long) w;
	    ldata[KHGT] = (unsigned long) h;
	    if (dim == 3) ldata[KDEP] = (unsigned long) d;
	 }
	 break;

      case KFLOAT:
	 {
	    float *ldata = (float *) rdata;

	    ldata[KWID] = (float) w;
	    ldata[KHGT] = (float) h;
	    if (dim == 3) ldata[KDEP] = (float) d;
	 }
	 break;

      case KDOUBLE:
	 {
	    double *ldata = (double *) rdata;

	    ldata[KWID] = (double) w;
	    ldata[KHGT] = (double) h;
	    if (dim == 3) ldata[KDEP] = (double) d;
	 }
	 break;

      case KCOMPLEX:
	 {
	    kcomplex *ldata = (kcomplex *) rdata;

	    ldata[KWID].r = (float) w;
	    ldata[KWID].i = 0.0;
	    ldata[KHGT].r = (float) h;
	    ldata[KHGT].i = 0.0;
	    if (dim == 3) 
	    {
	       ldata[KDEP].r = (float) d;
	       ldata[KDEP].i = 0.0;
	    }
	 }
	 break;

      case KDCOMPLEX:
	 {
	    kdcomplex *ldata = (kdcomplex *) rdata;

	    ldata[KWID].r = (double) w;
	    ldata[KWID].i = 0.0;
	    ldata[KHGT].r = (double) h;
	    ldata[KHGT].i = 0.0;
	    if (dim == 3) 
	    {
	       ldata[KDEP].r = (double) d;
	       ldata[KDEP].i = 0.0;
	    }
	 }
	 break;

      default:
	 errno = KINVALID_DATATYPE;
	 return FALSE;
   }

   return rdata;
}

/*-----------------------------------------------------------
|
|  Routine Name: kgeom_put_uniform_mesh
|
|       Purpose: This routine puts a location minimum primitive
|
|         Input: obj  - the object to put the primitive into
|                prim - the primitive to retrieve
|                data - pointer to the primitive data being stored
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica
|          Date: Apr 29, 1994 11:17
| Modifications:
|
------------------------------------------------------------*/
int
kgeom_put_uniform_mesh(
   kobject         obj,
   kaps_primitive *prim,
   kaddr           data)
{
   double w = 1;
   double h = 1;
   double d = 1;
   int    dim;
   int    type;
   

   /* -- determine the dimensionality of the primitive -- */
   if (prim->token == KGEOM_QUADMESH_LOCATION_MINIMUM || 
       prim->token == KGEOM_QUADMESH_LOCATION_MAXIMUM)
      dim = 2;  /* quadmeshes are 2 */

   else if (prim->token == KGEOM_OCTMESH_LOCATION_MINIMUM || 
	    prim->token == KGEOM_OCTMESH_LOCATION_MAXIMUM)
      dim = 3;  /* octmeshes are 3 */

   kgeom_get_attribute(obj, KGEOM_OBJECT, KGEOM_LOCATION_DATA_TYPE, &type);

   /* -- fill out the data in the appropriate data type -- */
   switch (type)
   {
      case KBYTE:
	 {
	    char *ldata = (char *) data;

	    w = (char) ldata[KWID];
	    h = (char) ldata[KHGT];
	    if (dim == 3) d = (char) ldata[KDEP];
	 }
	 break;

      case KUBYTE:
	 {
	    unsigned char *ldata = (unsigned char *) data;

	    w = (unsigned char) ldata[KWID];
	    h = (unsigned char) ldata[KHGT];
	    if (dim == 3) d = (unsigned char) ldata[KDEP];
	 }
	 break;

      case KSHORT:
	 {
	    short *ldata = (short *) data;

	    w = (short) ldata[KWID];
	    h = (short) ldata[KHGT];
	    if (dim == 3) d = (short) ldata[KDEP];
	 }
	 break;

      case KUSHORT:
	 {
	    unsigned short *ldata = (unsigned short *) data;

	    w = (unsigned short) ldata[KWID];
	    h = (unsigned short) ldata[KHGT];
	    if (dim == 3) d = (unsigned short) ldata[KDEP];
	 }
	 break;

      case KINT:
	 {
	    int *ldata = (int *) data;

	    w = (int) ldata[KWID];
	    h = (int) ldata[KHGT];
	    if (dim == 3) d = (int) ldata[KDEP];
	 }
	 break;

      case KUINT:
	 {
	    unsigned int *ldata = (unsigned int *) data;

	    w = (unsigned int) ldata[KWID];
	    h = (unsigned int) ldata[KHGT];
	    if (dim == 3) d = (unsigned int) ldata[KDEP];
	 }
	 break;

      case KLONG:
	 {
	    long *ldata = (long *) data;

	    w = (long) ldata[KWID];
	    h = (long) ldata[KHGT];
	    if (dim == 3) d = (long) ldata[KDEP];
	 }
	 break;

      case KULONG:
	 {
	    unsigned long *ldata = (unsigned long *) data;

	    w = (unsigned long) ldata[KWID];
	    h = (unsigned long) ldata[KHGT];
	    if (dim == 3) d = (unsigned long) ldata[KDEP];
	 }
	 break;

      case KFLOAT:
	 {
	    float *ldata = (float *) data;

	    w = (float) ldata[KWID];
	    h = (float) ldata[KHGT];
	    if (dim == 3) d = (float) ldata[KDEP];
	 }
	 break;

      case KDOUBLE:
	 {
	    double *ldata = (double *) data;

	    w = (double) ldata[KWID];
	    h = (double) ldata[KHGT];
	    if (dim == 3) d = (double) ldata[KDEP];
	 }
	 break;

      case KCOMPLEX:
	 {
	    kcomplex *ldata = (kcomplex *) data;

	    /* -- lossy, but oh well -- */
	    w = (float) ldata[KWID].r;
	    h = (float) ldata[KHGT].r;
	    if (dim == 3) d = (float) ldata[KDEP].r;
	 }
	 break;

      case KDCOMPLEX:
	 {
	    kdcomplex *ldata = (kdcomplex *) data;

	    /* -- lossy, but oh well -- */
	    w = (double) ldata[KWID].r;
	    h = (double) ldata[KHGT].r;
	    if (dim == 3) d = (double) ldata[KDEP].r;
	 }
	 break;

      default:
	 errno = KINVALID_DATATYPE;
	 return FALSE;
   }

   /* -- depending on the primitive, set either the minimum or maximum -- */
   if (prim->token == KGEOM_QUADMESH_LOCATION_MINIMUM || 
       prim->token == KGEOM_OCTMESH_LOCATION_MINIMUM)
      return(kpds_set_attribute(obj, 
				KPDS_LOCATION_BEGIN, w, h, d));

   else if (prim->token == KGEOM_QUADMESH_LOCATION_MAXIMUM || 
	    prim->token == KGEOM_OCTMESH_LOCATION_MAXIMUM)
      return(kpds_get_attribute(obj, 
				KPDS_LOCATION_END, w, h, d));
   else
      return FALSE;
}

/*
 *    ================================================================== 
 *    Mesh Data Routines
 *    ==================================================================
 */

/*-----------------------------------------------------------
|
|  Routine Name: kgeom_get_mesh
|
|       Purpose: This routine gets a mesh.  The term mesh covers 
|		 all quadmesh, octmesh, and texture primitives.
|
|         Input: obj       - the object to retrieve the primitive from
|                primitive - the primitive to retrieve
|                kva_list  - already started variable argument list of 
|			     pointers by which to return the data
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica
|          Date: Apr 29, 1994 11:17
| Modifications:
|
------------------------------------------------------------*/
int
kgeom_get_mesh(
   kobject   obj,
   int       primitive,
   kva_list *list)
{
   int    prim = _kgeom_init_mesh(obj, primitive, FALSE);  
   kaddr *data = NULL;


   /* verify that this is a valid mesh primitive */
   if (prim < 0)
      return FALSE;  /* errno already set */

   /* get the pointer data for this segment */
   if ((data = (kaddr *) kva_arg(*list, kaddr *)) != NULL ) 
   {
      /* if this segment doesn't exist, then there is no data to return */
      if (!kdms_query_segment(obj, kgeom_meshes[prim].segment))
      {
	 *data = NULL;
	 return FALSE;
      }
      else
      {
	 /* want to transpose data for these segments to be elements first */
	 if ((kstrcmp(kgeom_meshes[prim].segment, KPDS_SEGMENT_VALUE) == 0) ||
	     (kstrcmp(kgeom_meshes[prim].segment, KPDS_SEGMENT_LOCATION) == 0))
	    kpds_set_attribute(obj, KPDS_ELEMENTS_FIRST, TRUE);

	 *data = (kgeom_meshes[prim].get(obj, &kgeom_meshes[prim], 
					 *data));
	 
	 /* enough of that tranposing */
	 if ((kstrcmp(kgeom_meshes[prim].segment, KPDS_SEGMENT_VALUE) == 0) ||
	     (kstrcmp(kgeom_meshes[prim].segment, KPDS_SEGMENT_LOCATION) == 0))
	    kpds_set_attribute(obj, KPDS_ELEMENTS_FIRST, FALSE);
      }
   }

   return TRUE;
}

/*-----------------------------------------------------------
|
|  Routine Name: kgeom_put_mesh
|
|       Purpose: This routine puts a mesh primitive. The term mesh 
|		 covers all quadmesh, octmesh, and texture primitives.
|
|         Input: obj       - the object to store the primitive in
|                primitive - the primitive being put
|                kva_list  - already started variable argument list of 
|			     pointers by which point to the data 
|		 	     being put.
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica
|          Date: Apr 29, 1994 11:17
| Modifications:
|
------------------------------------------------------------*/
int
kgeom_put_mesh(
   kobject   obj,
   int       primitive,
   kva_list *list)
{
   int   prim = _kgeom_init_mesh(obj, primitive, TRUE);
   kaddr data = NULL;
   int   status;


   /* verify that this is a valid mesh primitive */
   if (prim < 0)
      return FALSE;  /* errno already set */

   /* get the pointer data for this segment */
   if ((data = (kaddr *) kva_arg(*list, kaddr *)) != NULL ) 
   {
      /* this segment should exist, but check anyway */
      if (!kdms_query_segment(obj, kgeom_meshes[prim].segment))
	 return FALSE;

      /* want to transpose the data for these segments to be elements first */
      if ((kstrcmp(kgeom_meshes[prim].segment, KPDS_SEGMENT_VALUE) == 0) ||
	  (kstrcmp(kgeom_meshes[prim].segment, KPDS_SEGMENT_LOCATION) == 0))
	  kpds_set_attribute(obj, KPDS_ELEMENTS_FIRST, TRUE);

      status = (kgeom_meshes[prim].put(obj, &kgeom_meshes[prim], data));

      /* enough of that tranposing */
      if ((kstrcmp(kgeom_meshes[prim].segment, KPDS_SEGMENT_VALUE) == 0) ||
	  (kstrcmp(kgeom_meshes[prim].segment, KPDS_SEGMENT_LOCATION) == 0))
	 kpds_set_attribute(obj, KPDS_ELEMENTS_FIRST, FALSE);

      return status;
   }

   return TRUE;
}

/*
 *    ================================================================== 
 *    Mesh Attributes
 *    ==================================================================
 */

/*********** ---------------------------------------------------------------
 ***********  KGEOM_QUADMESH_SIZE
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: _quadmesh_size_get
|       Purpose: get the quadmesh size attribute 
|    Written By: Steve Kubica
|          Date: Jul 09, 1994 
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_quadmesh_size_get(kobject obj, int assoc, int attr, kaddr clientData, 
		   kva_list *list)
{
  int *rwidth  = kva_arg(*list, int *);
  int *rheight = kva_arg(*list, int *);
  int  width;
  int  height;

  if (!_kgeom_query_quadmesh(obj))
    return FALSE;

  kpds_get_attribute(obj, KPDS_LOCATION_SIZE, &width, &height, NULL, NULL);

  if (rwidth)  *rwidth  = width;
  if (rheight) *rheight = height;
 
  return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: _quadmesh_size_set
|       Purpose: set the quadmesh size attribute
|    Written By: Steve Kubica
|          Date: Dec 3, 1993
------------------------------------------------------------*/
/* ARGSUSED */
static int
_quadmesh_size_set(kobject obj, int assoc, int attr, kaddr clientData, 
		   kva_list *list)
{
   int width  = kva_arg(*list, int);
   int height = kva_arg(*list, int);

   /* if there is not yet a quadmesh, then create it */
   if (!_kgeom_query_quadmesh(obj))
   {
      /* -- a quadmesh consists of at least a location segment -- */
      if (!kdms_query_segment(obj, KGEOM_SEGMENT_LOCATION))
	 kdms_create_segment(obj, KGEOM_SEGMENT_LOCATION);
   }

   return (_kgeom_set_quadmesh_size(obj, width, height));
}

/*-----------------------------------------------------------
|  Routine Name: _quadmesh_size_match
|       Purpose: match the quadmesh size attribute
|    Written By: Steve Kubica 
|          Date: Dec 3, 1993
------------------------------------------------------------*/
/* ARGSUSED */
static int
_quadmesh_size_match(kobject obj1, kobject obj2, int assoc, int attrib,
                     kaddr clientData1, kaddr clientData2)
{
   int w1, h1;
   int w2, h2;

 
   if (!kpds_get_attribute(obj1, KPDS_LOCATION_SIZE, &w1, &h1, NULL, NULL))
      return FALSE;
 
   if (!kpds_get_attribute(obj2, KPDS_LOCATION_SIZE, &w2, &h2, NULL, NULL))
      return FALSE;
 
   return ((w1 == w2) && (h1 == h2));
}

/*-----------------------------------------------------------
|  Routine Name: (static) _quadmesh_size_copy
|       Purpose: copy the quadmesh size attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_quadmesh_size_copy(kobject obj1, kobject obj2, int assoc, int attrib,
                    kaddr clientData1, kaddr clientData2)
{
   int w, h;
   int status = TRUE;
   

   status &= kgeom_get_attribute(obj1, KGEOM_OBJECT, 
				 KGEOM_QUADMESH_SIZE, &w, &h);
   status &= kgeom_set_attribute(obj2, KGEOM_OBJECT, 
				 KGEOM_QUADMESH_SIZE, w, h);
   return status;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _quadmesh_size_query
|       Purpose: query the quadmesh size attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_quadmesh_size_query(kobject obj, int assoc, int attrib, kaddr clientData,
                     int *num_args, int *arg_size, int *data_type, 
                     int *persistence)
{
   /* these are fixed quantities - simply fill them out */
   if (num_args)    *num_args    = 2;
   if (persistence) *persistence = TRUE;
   if (arg_size)    *arg_size    = 1;
   if (data_type)   *data_type   = KINT;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _quadmesh_size_print
|       Purpose: print the quadmesh size attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_quadmesh_size_print(kobject obj, int assoc, int attrib, kaddr clientData,
                     kfile *outfile)
{
   int w, h;


   kgeom_get_attribute(obj, KGEOM_OBJECT, KGEOM_QUADMESH_SIZE, &w, &h);

   if (outfile)
      kfprintf(outfile,"%d x %d", w, h);

   return TRUE;
}

/*********** ---------------------------------------------------------------
 ***********  KGEOM_OCTMESH_SIZE
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: _octmesh_size_get
|       Purpose: get the octmesh size attribute 
|    Written By: Steve Kubica
|          Date: Jul 09, 1994 
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_octmesh_size_get(kobject obj, int assoc, int attr, kaddr clientData, 
		  kva_list *list)
{
  int *rwidth  = kva_arg(*list, int *);
  int *rheight = kva_arg(*list, int *);
  int *rdepth  = kva_arg(*list, int *);
  int  width;
  int  height;
  int  depth;

  if (!_kgeom_query_octmesh(obj))
    return FALSE;

  kpds_get_attribute(obj, KPDS_LOCATION_SIZE, &width, &height, &depth, NULL);

  if (rwidth)  *rwidth  = width;
  if (rheight) *rheight = height;
  if (rdepth)  *rdepth  = depth;

  return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: _octmesh_size_set
|       Purpose: set the octmesh size attribute
|    Written By: Steve Kubica
|          Date: Dec 3, 1993
------------------------------------------------------------*/
/* ARGSUSED */
static int
_octmesh_size_set(kobject obj, int assoc, int attr, kaddr clientData, 
		  kva_list *list)
{
   int width  = kva_arg(*list, int);
   int height = kva_arg(*list, int);
   int depth  = kva_arg(*list, int);

   if (!_kgeom_query_octmesh(obj))
   {
      /* -- octmesh must consist of at least location data -- */
      if (!kdms_query_segment(obj, KGEOM_SEGMENT_LOCATION))
	 kdms_create_segment(obj, KGEOM_SEGMENT_LOCATION);
   }
 
   return (_kgeom_set_octmesh_size(obj, width, height, depth));
}

/*-----------------------------------------------------------
|  Routine Name: _octmesh_size_match
|       Purpose: match the octmesh size attribute
|    Written By: Steve Kubica 
|          Date: Dec 3, 1993
------------------------------------------------------------*/
/* ARGSUSED */
static int
_octmesh_size_match(kobject obj1, kobject obj2, int assoc, int attrib,
		    kaddr clientData1, kaddr clientData2)
{
   int w1, h1, d1;
   int w2, h2, d2;

 
   if (!kpds_get_attribute(obj1, KPDS_LOCATION_SIZE, &w1, &h1, &d1, NULL))
      return FALSE;
 
   if (!kpds_get_attribute(obj2, KPDS_LOCATION_SIZE, &w2, &h2, &d2, NULL))
      return FALSE;
 
   return ((w1 == w2) && (h1 == h2) && (d1 == d2));
}

/*-----------------------------------------------------------
|  Routine Name: (static) _octmesh_size_copy
|       Purpose: copy the octmesh size attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_octmesh_size_copy(kobject obj1, kobject obj2, int assoc, int attrib,
		   kaddr clientData1, kaddr clientData2)
{
   int w, h, d;
   int status = TRUE;
   

   status &= kgeom_get_attribute(obj1, KGEOM_OBJECT, 
				 KGEOM_OCTMESH_SIZE, &w, &h, &d);
   status &= kgeom_set_attribute(obj2, KGEOM_OBJECT, 
				 KGEOM_OCTMESH_SIZE, w, h, d);
   return status;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _octmesh_size_query
|       Purpose: query the octmesh size attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_octmesh_size_query(kobject obj, int assoc, int attrib, kaddr clientData,
		    int *num_args, int *arg_size, int *data_type, 
		    int *persistence)
{
   /* these are fixed quantities - simply fill them out */
   if (num_args)    *num_args    = 3;
   if (persistence) *persistence = TRUE;
   if (arg_size)    *arg_size    = 1;
   if (data_type)   *data_type   = KINT;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _octmesh_size_print
|       Purpose: print the octmesh size attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_octmesh_size_print(kobject obj, int assoc, int attrib, kaddr clientData,
		    kfile *outfile)
{
   int w, h, d;


   kgeom_get_attribute(obj, KGEOM_OBJECT, KGEOM_OCTMESH_SIZE, &w, &h, &d);

   if (outfile)
      kfprintf(outfile,"%d x %d x %d", w, h, d);

   return TRUE;
}

/*********** ---------------------------------------------------------------
 ***********  KGEOM_TEXTURE2D_SIZE
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: _texture2d_size_get
|       Purpose: get the texture2d size attribute 
|    Written By: Steve Kubica
|          Date: Jul 09, 1994 
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_texture2d_size_get(kobject obj, int assoc, int attr, kaddr clientData, 
		    kva_list *list)
{
  int *rwidth  = kva_arg(*list, int *);
  int *rheight = kva_arg(*list, int *);
  int  width;
  int  height;

  if (!_kgeom_query_texture2d(obj))
      return FALSE;   /* need to create texture2d first */

  kpds_get_attribute(obj, KPDS_VALUE_SIZE, &width, &height, NULL, NULL, NULL);

  if (rwidth)  *rwidth  = width;
  if (rheight) *rheight = height;

  return (TRUE);
}

/*-----------------------------------------------------------
|  Routine Name: _texture2d_size_set
|       Purpose: set the texture2d size attribute
|    Written By: Steve Kubica
|          Date: Dec 3, 1993
------------------------------------------------------------*/
/* ARGSUSED */
static int
_texture2d_size_set(kobject obj, int assoc, int attr, kaddr clientData, 
		    kva_list *list)
{
   int width  = kva_arg(*list, int);
   int height = kva_arg(*list, int);

   if (!_kgeom_query_texture2d(obj))
   {
      /* -- texture 2d consists of only a value segment -- */
      if (!kdms_query_segment(obj, KPDS_SEGMENT_VALUE))
	 kdms_create_segment(obj, KPDS_SEGMENT_VALUE);
   }
 
   return (_kgeom_set_texture2d_size(obj, width, height));
}

/*-----------------------------------------------------------
|  Routine Name: _texture2d_size_match
|       Purpose: match the texture2d size attribute
|    Written By: Steve Kubica 
|          Date: Dec 3, 1993
------------------------------------------------------------*/
/* ARGSUSED */
static int
_texture2d_size_match(kobject obj1, kobject obj2, int assoc, int attrib,
		      kaddr clientData1, kaddr clientData2)
{
   int w1, h1;
   int w2, h2;

 
   if (!kpds_get_attribute(obj1, KPDS_VALUE_SIZE, &w1, &h1, NULL, NULL, NULL))
      return FALSE;
 
   if (!kpds_get_attribute(obj2, KPDS_VALUE_SIZE, &w2, &h2, NULL, NULL, NULL))
      return FALSE;
 
   return ((w1 == w2) && (h1 == h2));
}

/*-----------------------------------------------------------
|  Routine Name: (static) _texture2d_size_copy
|       Purpose: copy the texture2d size attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_texture2d_size_copy(kobject obj1, kobject obj2, int assoc, int attrib,
		     kaddr clientData1, kaddr clientData2)
{
   int w, h;
   int status = TRUE;
   

   status &= kgeom_get_attribute(obj1, KGEOM_OBJECT, 
				 KGEOM_TEXTURE2D_SIZE, &w, &h);
   status &= kgeom_set_attribute(obj2, KGEOM_OBJECT, 
				 KGEOM_TEXTURE2D_SIZE, w, h);
   return status;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _texture2d_size_query
|       Purpose: query the texture2d size attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_texture2d_size_query(kobject obj, int assoc, int attrib, kaddr clientData,
		      int *num_args, int *arg_size, int *data_type, 
		      int *persistence)
{
   /* these are fixed quantities - simply fill them out */
   if (num_args)    *num_args    = 2;
   if (persistence) *persistence = TRUE;
   if (arg_size)    *arg_size    = 1;
   if (data_type)   *data_type   = KINT;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _texture2d_size_print
|       Purpose: print the texture2d size attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_texture2d_size_print(kobject obj, int assoc, int attrib, kaddr clientData,
		      kfile *outfile)
{
   int w, h;


   kgeom_get_attribute(obj, KGEOM_OBJECT, KGEOM_TEXTURE2D_SIZE, &w, &h);

   if (outfile)
      kfprintf(outfile,"%d x %d", w, h);

   return TRUE;
}

/*********** ---------------------------------------------------------------
 ***********  KGEOM_TEXTURE3D_SIZE
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: _texture3d_size_get
|       Purpose: get the texture3d size attribute 
|    Written By: Steve Kubica
|          Date: Jul 09, 1994 
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_texture3d_size_get(kobject obj, int assoc, int attr, kaddr clientData, 
		    kva_list *list)
{
  int *rwidth  = kva_arg(*list, int *);
  int *rheight = kva_arg(*list, int *);
  int *rdepth  = kva_arg(*list, int *);
  int  width;
  int  height;
  int  depth;

  if (!_kgeom_query_texture3d(obj))
    return FALSE;

  kpds_get_attribute(obj, KPDS_VALUE_SIZE, &width, &height, &depth, NULL,NULL);

  if (rwidth)  *rwidth  = width;
  if (rheight) *rheight = height;
  if (rdepth)  *rdepth  = depth;

  return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: _texture3d_size_set
|       Purpose: set the texture3d size attribute
|    Written By: Steve Kubica
|          Date: Dec 3, 1993
------------------------------------------------------------*/
/* ARGSUSED */
static int
_texture3d_size_set(kobject obj, int assoc, int attr, kaddr clientData, 
		    kva_list *list)
{
   int width  = kva_arg(*list, int);
   int height = kva_arg(*list, int);
   int depth  = kva_arg(*list, int);

   if (!_kgeom_query_texture3d(obj))
   {
      /* -- texture 2d consists of only a value segment -- */
      if (!kdms_query_segment(obj, KPDS_SEGMENT_VALUE))
	 kdms_create_segment(obj, KPDS_SEGMENT_VALUE);
   }

   return (_kgeom_set_texture3d_size(obj, width, height, depth));
}

/*-----------------------------------------------------------
|  Routine Name: _texture3d_size_match
|       Purpose: match the texture3d size attribute
|    Written By: Steve Kubica 
|          Date: Dec 3, 1993
------------------------------------------------------------*/
/* ARGSUSED */
static int
_texture3d_size_match(kobject obj1, kobject obj2, int assoc, int attrib,
		      kaddr clientData1, kaddr clientData2)
{
   int w1, h1, d1;
   int w2, h2, d2;

 
   if (!kpds_get_attribute(obj1, KPDS_VALUE_SIZE, &w1, &h1, &d1, NULL, NULL))
      return FALSE;
 
   if (!kpds_get_attribute(obj2, KPDS_VALUE_SIZE, &w2, &h2, &d2, NULL, NULL))
      return FALSE;
 
   return ((w1 == w2) && (h1 == h2) && (d1 == d2));
}

/*-----------------------------------------------------------
|  Routine Name: (static) _texture3d_size_copy
|       Purpose: copy the texture3d size attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_texture3d_size_copy(kobject obj1, kobject obj2, int assoc, int attrib,
		     kaddr clientData1, kaddr clientData2)
{
   int w, h, d;
   int status = TRUE;
   

   status &= kgeom_get_attribute(obj1, KGEOM_OBJECT, 
				 KGEOM_TEXTURE3D_SIZE, &w, &h, &d);
   status &= kgeom_set_attribute(obj2, KGEOM_OBJECT, 
				 KGEOM_TEXTURE3D_SIZE, w, h, d);
   return status;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _texture3d_size_query
|       Purpose: query the texture3d size attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_texture3d_size_query(kobject obj, int assoc, int attrib, kaddr clientData,
		      int *num_args, int *arg_size, int *data_type, 
		      int *persistence)
{
   /* these are fixed quantities - simply fill them out */
   if (num_args)    *num_args    = 3;
   if (persistence) *persistence = TRUE;
   if (arg_size)    *arg_size    = 1;
   if (data_type)   *data_type   = KINT;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _texture3d_size_print
|       Purpose: print the texture3d size attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_texture3d_size_print(kobject obj, int assoc, int attrib, kaddr clientData,
                      kfile *outfile)
{
   int w, h, d;


   kgeom_get_attribute(obj, KGEOM_OBJECT, KGEOM_TEXTURE3D_SIZE, &w, &h, &d);

   if (outfile)
      kfprintf(outfile,"%d x %d x %d", w, h, d);

   return TRUE;
}

/*********** ---------------------------------------------------------------
 ***********  KGEOM_POSITION_2D
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: _position2d_get
|       Purpose: get 2D position attribute 
|    Written By: Steve Kubica
|          Date: Dec 3, 1993
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_position2d_get(kobject obj, int assoc, int attr, kaddr clientData, 
		kva_list *list)
{
   int *ret;
   int *position = NULL;

   if (!kdms_query_segment(obj, (char *)clientData))
      position = kaps_get_segment_position(obj, (char *)clientData);
   else
      return FALSE;

   if (position == NULL)
      return FALSE;

   /* return the width and height from the position array */
   if ( (ret = kva_arg(*list, int *)) != NULL) *ret = position[KWID];
   if ( (ret = kva_arg(*list, int *)) != NULL) *ret = position[KHGT];
 
   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: _position2d_set
|       Purpose: set the 2D position attribute
|    Written By: Steve Kubica
|          Date: Dec 3, 1993
------------------------------------------------------------*/
/* ARGSUSED */
static int
_position2d_set(kobject obj, int assoc, int attr, kaddr clientData, 
		kva_list *list)
{
   int position[5] = {0, 0, 0, 0, 0};


   /* get the new width and height position from the variable argument list */
   position[0] = kva_arg( *list, int );
   position[1] = kva_arg( *list, int );

   if (!kdms_query_segment(obj, (char *)clientData))
   {
      kaps_set_segment_position(obj, (char *)clientData, position);
      kaps_set_segment_inc_state(obj, (char *)clientData, 0, 0, 0);
   }

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: _position2d_match
|       Purpose: match the 2D position position attribute
|    Written By: Steve Kubica 
|          Date: Dec 3, 1993
------------------------------------------------------------*/
/* ARGSUSED */
static int
_position2d_match(kobject obj1, kobject obj2, int assoc, int attrib,
		  kaddr clientData1, kaddr clientData2)
{
   int *position1;
   int *position2;


   /* get the internal sizes */
   position1 = kaps_get_segment_position(obj1, (char *)clientData1);
   position2 = kaps_get_segment_position(obj2, (char *)clientData2);

   if (position1 == NULL || position2 == NULL)
      return FALSE;

   return ((position1[0] == position2[0]) && 
	   (position1[1] == position2[1]));
}

/*-----------------------------------------------------------
|  Routine Name: (static) _position2d_copy
|       Purpose: copy the 2D position attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_position2d_copy(kobject obj1, kobject obj2, int assoc, int attrib,
		 kaddr clientData1, kaddr clientData2)
{
   int *position;


   position = kaps_get_segment_position(obj1, (char *)clientData1);
   if (position == NULL)
      return FALSE;

   kaps_set_segment_position(obj2, (char *)clientData2, position);
   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _position2d_pos_query
|       Purpose: query the 2D position attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_position2d_query(kobject obj, int assoc, int attrib, kaddr clientData,
		  int *num_args, int *arg_size, int *data_type, 
		  int *persistence)
{
   /* these are fixed quantities - simply fill them out */
   if (num_args)    *num_args    = 2;
   if (persistence) *persistence = FALSE;
   if (arg_size)    *arg_size    = 1;
   if (data_type)   *data_type   = KINT;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _position2d_print
|       Purpose: print the 2D position attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_position2d_print(kobject obj, int assoc, int attrib, kaddr clientData,
		  kfile *outfile)
{
   int *position;
   

   position = kaps_get_segment_position(obj, (char *)clientData);
   if (position == NULL)
      return FALSE;
   
   if (outfile)
      kfprintf(outfile,"%d x %d", position[0], position[1]);

   return TRUE;
}

/*********** ---------------------------------------------------------------
 ***********  KGEOM_POSITION_3D
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: _position3d_get
|       Purpose: get 3D position attribute 
|    Written By: Steve Kubica
|          Date: Dec 3, 1993
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_position3d_get(kobject obj, int assoc, int attr, kaddr clientData, 
		kva_list *list)
{
   int *ret;
   int *position = NULL;

   if (!kdms_query_segment(obj, (char *)clientData))
      position = kaps_get_segment_position(obj, (char *)clientData);
   else
      return FALSE;

   if (position == NULL)
      return FALSE;

   /* return the width and height from the position array */
   if ( (ret = kva_arg(*list, int *)) != NULL) *ret = position[0];
   if ( (ret = kva_arg(*list, int *)) != NULL) *ret = position[1];
   if ( (ret = kva_arg(*list, int *)) != NULL) *ret = position[2];
 
   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: _position3d_set
|       Purpose: set the 3D position attribute
|    Written By: Steve Kubica
|          Date: Dec 3, 1993
------------------------------------------------------------*/
/* ARGSUSED */
static int
_position3d_set(kobject obj, int assoc, int attr, kaddr clientData, 
		kva_list *list)
{
   int position[5] = {0, 0, 0, 0, 0};


   /* get the new width, height, depth position from the var arg list */
   position[0] = kva_arg( *list, int );
   position[1] = kva_arg( *list, int );
   position[2] = kva_arg( *list, int );

   if (!kdms_query_segment(obj, (char *)clientData))
   {
      kaps_set_segment_position(obj, (char *)clientData, position);
      kaps_set_segment_inc_state(obj, (char *)clientData, 0, 0, 0);
   }

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: _position3d_match
|       Purpose: match the 3D position position attribute
|    Written By: Steve Kubica 
|          Date: Dec 3, 1993
------------------------------------------------------------*/
/* ARGSUSED */
static int
_position3d_match(kobject obj1, kobject obj2, int assoc, int attrib,
		  kaddr clientData1, kaddr clientData2)
{
   int *position1;
   int *position2;


   /* get the internal sizes */
   position1 = kaps_get_segment_position(obj1, (char *)clientData1);
   position2 = kaps_get_segment_position(obj2, (char *)clientData2);

   if (position1 == NULL || position2 == NULL)
      return FALSE;

   return ((position1[0] == position2[0]) && 
	   (position1[1] == position2[1]) &&
	   (position1[2] == position2[2]));
}

/*-----------------------------------------------------------
|  Routine Name: (static) _position3d_copy
|       Purpose: copy the 3D position attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_position3d_copy(kobject obj1, kobject obj2, int assoc, int attrib,
		 kaddr clientData1, kaddr clientData2)
{
   int *position;


   position = kaps_get_segment_position(obj1, (char *)clientData1);
   if (position == NULL)
      return FALSE;

   kaps_set_segment_position(obj2, (char *)clientData2, position);

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _position3d_query
|       Purpose: query the 3D position attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_position3d_query(kobject obj, int assoc, int attrib, kaddr clientData,
		  int *num_args, int *arg_size, int *data_type, 
		  int *persistence)
{
   /* these are fixed quantities - simply fill them out */
   if (num_args)    *num_args    = 3;
   if (persistence) *persistence = FALSE;
   if (arg_size)    *arg_size    = 1;
   if (data_type)   *data_type   = KINT;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _position3d_print
|       Purpose: print the 3D position attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_position3d_print(kobject obj, int assoc, int attrib, kaddr clientData,
		  kfile *outfile)
{
   int *position;
      

   position = kaps_get_segment_position(obj, (char *)clientData);
   if (position == NULL)
      return FALSE;

   if (outfile)
      kfprintf(outfile,"%d x %d x %d", position[0], position[1], position[2]);

   return TRUE;
}

/*
 *    ================================================================== 
 *    Mesh Initialization
 *    ==================================================================
 */

/*-----------------------------------------------------------
|
|  Routine Name: kgeom_define_meshes
|
|       Purpose: This routine initializes all the segment
|		 tokens in the static mesh primitive list.
|
|         Input: none
|
|        Output: none
|
|       Returns: none
|
|    Written By: Steve Kubica
|          Date: Jul 09, 1994 18:12
| Modifications:
|
------------------------------------------------------------*/
void
kgeom_define_meshes(void)
{
   int zero[5] = {0, 0, 0, 0, 0};
   int i;

 
   for (i = 0; i < kgeom_num_meshes; i++)
      kgeom_meshes[i].seg_token = kstring_to_token(kgeom_meshes[i].segment);


   /*
    *   Mesh Attributes
    *   .....................................................................
    */
   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KGEOM_QUADMESH_SIZE, NULL, 
			       _quadmesh_size_get,   _quadmesh_size_set,
			       _quadmesh_size_match, _quadmesh_size_copy,
			       _quadmesh_size_query, _quadmesh_size_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KGEOM_OCTMESH_SIZE, NULL, 
			       _octmesh_size_get,   _octmesh_size_set, 
			       _octmesh_size_match, _octmesh_size_copy,
			       _octmesh_size_query, _octmesh_size_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KGEOM_QUADMESH_LOCATION_POSITION, 
			       KGEOM_SEGMENT_LOCATION, 
			       _position2d_get,   _position2d_set,
			       _position2d_match, _position2d_copy,
			       _position2d_query, _position2d_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KGEOM_QUADMESH_COLOR_POSITION, 
			       KPDS_SEGMENT_VALUE, 
			       _position2d_get,   _position2d_set,
			       _position2d_match, _position2d_copy,
			       _position2d_query, _position2d_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KGEOM_QUADMESH_NORMAL_POSITION, 
			       KGEOM_SEGMENT_NORMAL, 
			       _position2d_get,   _position2d_set,
			       _position2d_match, _position2d_copy,
			       _position2d_query, _position2d_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KGEOM_QUADMESH_TEXTURE_COORD_POSITION, 
			       KGEOM_SEGMENT_TEXTURE_COORD, 
			       _position2d_get,   _position2d_set,
			       _position2d_match, _position2d_copy,
			       _position2d_query, _position2d_print);


   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KGEOM_OCTMESH_LOCATION_POSITION, 
			       KGEOM_SEGMENT_LOCATION, 
			       _position3d_get,   _position3d_set,
			       _position3d_match, _position3d_copy,
			       _position3d_query, _position3d_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KGEOM_OCTMESH_COLOR_POSITION, 
			       KPDS_SEGMENT_VALUE, 
			       _position3d_get,   _position3d_set,
			       _position3d_match, _position3d_copy,
			       _position3d_query, _position3d_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KGEOM_OCTMESH_NORMAL_POSITION, 
			       KGEOM_SEGMENT_NORMAL, 
			       _position3d_get,   _position3d_set,
			       _position3d_match, _position3d_copy,
			       _position3d_query, _position3d_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KGEOM_OCTMESH_TEXTURE_COORD_POSITION, 
			       KGEOM_SEGMENT_TEXTURE_COORD, 
			       _position3d_get,   _position3d_set,
			       _position3d_match, _position3d_copy,
			       _position3d_query, _position3d_print);

   /*  internal position and offset information  */
   kdms_define_attribute(KGEOM_SEGMENT_NORMAL, _INTERNAL_POSITION, 1, 4, KINT,
                         FALSE, FALSE, zero);
   kdms_define_attribute(KGEOM_SEGMENT_NORMAL, _INTERNAL_OFFSET, 1, 4, KINT,
                         FALSE, FALSE, zero);

   kdms_define_attribute(KGEOM_SEGMENT_TEXTURE_COORD, _INTERNAL_POSITION, 
			 1, 4, KINT, FALSE, FALSE, zero);
   kdms_define_attribute(KGEOM_SEGMENT_TEXTURE_COORD, _INTERNAL_OFFSET, 
			 1, 4, KINT, FALSE, FALSE, zero);



   /*
    *   Texture Attributes
    *   .....................................................................
    */

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KGEOM_TEXTURE2D_SIZE, NULL, 
			       _texture2d_size_get,   _texture2d_size_set,
			       _texture2d_size_match, _texture2d_size_copy,
			       _texture2d_size_query, _texture2d_size_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KGEOM_TEXTURE3D_SIZE, NULL, 
			       _texture3d_size_get,   _texture3d_size_set,
			       _texture3d_size_match, _texture3d_size_copy,
			       _texture3d_size_query, _texture3d_size_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KGEOM_TEXTURE2D_POSITION, 
			       KPDS_SEGMENT_VALUE,
			       _position2d_get,   _position2d_set, 
			       _position2d_match, _position2d_copy,
			       _position2d_query, _position2d_print);

   kdms_define_quasi_attribute(KDMS_OBJECT,
			       KGEOM_TEXTURE3D_POSITION, 
			       KPDS_SEGMENT_VALUE,
			       _position3d_get,   _position3d_set, 
			       _position3d_match, _position3d_copy,
			       _position3d_query, _position3d_print);
   return;
}

