 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Geometry Application Services
   >>>>            Primitive Definition Routines 
   >>>>
   >>>>   Static:
   >>>>			_get_component()  
   >>>>			_put_component()  
   >>>>
   >>>>			_kgeom_construct_component_defin()  
   >>>>			_kgeom_free_component_defin()  
   >>>>			_kgeom_construct_primitive_defin()  
   >>>>			_kgeom_free_primitive_defin()  
   >>>>			_kgeom_add_primitive_defin()  
   >>>>			_kgeom_delete_primitive_defin()  
   >>>>  Private:
   >>>>			kgeom_locate_primitive_defin()  
   >>>>   Public:
   >>>>			kgeom_define_primitive()
   >>>>			kgeom_undefine_primitive()
   >>>>			kgeom_primitive_name()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "geom_internals.h"

/* 
 *    ================================================================== 
 *    Geometry Primitive Definition List
 *    ==================================================================
 */

kgeom_primitive_defin **kgeom_primitive_defins     = NULL;
int                     kgeom_num_primitive_defins = 0;

/* 
 *    ================================================================== 
 *    Generic get and put Methods
 *    ==================================================================
 */

/*-----------------------------------------------------------
|
|  Routine Name: (static) _get_component
|
|       Purpose: This routine gets data from a data component
|		 given the storage information as well as the 
|		 current presentation size and data type.
|
|		 The data is assumed to be two dimensional,
|		 consisting of a number of points, which each
|		 point containing a number of element values.
|		 The first dimension in the segment is assumed
|		 to be the elements, and the second dimension 
|		 is assumed to be down the number of points.
|		 
|		 The storage information consists of the stored
|		 size in terms of element size and number of points
|		 along with the begin point which marks where the 
|		 desired data is located along the second dimension 
|		 in the segment.
|
|         Input: obj       - object to be synced to presentation
|		 segment   - segment to be synced to presentation 
|		 position  - the primitive position of the primitive 
|			     which contains component to be synced
|		 component - the component number within the primitive
|
|		 The following info is from the presentation function :
|
|		   pres_size - array of two numbers indicating the 
|			       presentation element size and number
|			       of points
|		   data_type - presentation data type 
|
|		 The following info is from the primitive manifest :
|		 
|		   size        - array of two numbers indicating the 
|			         stored element size and number
|			         of points
|        	   begin_point - the physical begining index 
|				 along the second dimension
|
|		 data  - a pointer to already allocated memory in which 
|			 to return the data being retrieved.  If this
|		 	 is NULL, then new space will be allocated.
|
|       Returns: the requested data component
|
|    Written By: Steve Kubica
|          Date: Feb 04, 1995 15:29
| Modifications:
------------------------------------------------------------*/
/* ARGSUSED */
static kaddr 
_get_component(
   kobject obj,
   char   *segment,
   int     position,
   int     component,
   int    *pres_size, 
   int     data_type,
   int    *size,      
   int     begin_point,
   kaddr   data)        
{
   int  *internal_size;
   int   old_overall_size[KGEOM_MAX_DIM];
   int   new_overall_size[KGEOM_MAX_DIM];
   float factor;
   
   int   beg[KGEOM_MAX_DIM] = {0,0,0,0};
   int   end[KGEOM_MAX_DIM] = {0,0,0,0};
   int   pres_begin_point;
   
   kaddr rdata    = NULL;

   /* -- sanity check -- */
   if (segment == NULL || size == NULL)
      return NULL;

   /* == handle presentation issues : size and data type == */

   /* -- need to first get the old overall size -- */
   if (!kdms_get_attribute(obj, segment, KDMS_SIZE, &internal_size))
      return NULL;

   /* -- spin off a few working copies of the internal size array -- */
   kmemcpy(old_overall_size, internal_size, KGEOM_MAX_DIM * sizeof(int));
   kmemcpy(new_overall_size, internal_size, KGEOM_MAX_DIM * sizeof(int));

   /*   
    *   trickiness in presentation size change will be along the 
    *   number of points since that is where sharing occurs across
    *   different primitives.   
    *
    *   anyway, here's the basic idea : 
    *
    *      factor = presentation size[1] / manifest size[1]
    *
    *      new overall_size[1] = old overall_size[1] * factor
    *      new begin point     = old begin point * factor
    *
    */
   factor = pres_size[1] / size[1];

   /* -- determine the appropriate presentation size -- */
   new_overall_size[1] = old_overall_size[1] * factor;

   /* -- modify the begin point so it indexes into the new size -- */
   pres_begin_point = begin_point * factor;

   /* -- set the presentation size and data type on the segment -- */
   if (!kdms_set_attributes(obj, segment, 
			    KDMS_SIZE,      new_overall_size,
			    KDMS_DATA_TYPE, data_type, NULL))
      return NULL;


   /* ===  retreive the data from the segment  === */

   /* -- calculate the begin and end points for the data access -- */
   end[0] = pres_size[0] - 1;

   /* -- use the begin and end corners dictated by prior allocation -- */
   beg[1] = pres_begin_point;
   end[1] = pres_begin_point + pres_size[1] - 1;

   /* -- retrieve the data -- */
   rdata = kdms_get_data(obj, segment, beg, end, data);   


   /* -- return the presentation size to normal -- */
   if (!kdms_set_attribute(obj, segment, KDMS_SIZE, old_overall_size))
      return NULL;

   return rdata;
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _put_component
|
|       Purpose: This routine puts data into a data component
|		 according to the size and data type of a given
|		 presentation.   The space in which to put
|		 the data is allocated by this routine.
|	 	 If the segment in which to store the component
|		 does not yet exist, it will be created by this 
|		 routine.
|		
|		 The assumptions are made that the allocated 
|		 segment will be at least two dimensional, 
|		 that the first dimension will be the element size,
|		 and that the second dimension will mark the direction
|		 for allocation.  This routine forces all other 
|		 dimensions to be of size one.
|
|		 The begin point marking where the allocated space 
|		 is located along the second dimension will be returned.
|
|         Input: obj       - object to put component data into
|		 segment   - segment to put component data into
|		 position  - the primitive position of the primitive 
|			     which will be containing this component
|		 component - the component number within the primitive
|		 size      - array of two numbers indicating the element
|			     size and the number of points to be allocated
|		 data_type - data type to the data is in the segment to
|		 data      - generic pointer to the data to be stored into 
|			     the component
|
|        Output: begin_point - the begining index along the second dimension
|			       marking where the data has been stored
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica
|          Date: Feb 04, 1995 15:29
| Modifications:
------------------------------------------------------------*/
/* ARGSUSED */
static int 
_put_component(
   kobject obj,
   char   *segment,
   int     position,
   int     component,
   int    *size,
   int     data_type,
   int    *begin_point,
   kaddr   data)
{
   int  beg[KGEOM_MAX_DIM] = {0,0,0,0};
   int  end[KGEOM_MAX_DIM] = {0,0,0,0};
   int  new_size[KGEOM_MAX_DIM] = {1, 1, 1, 1};
   int *old_size;
   int  used;
   
   /* -- sanity check -- */
   if (segment == NULL || size == NULL)
      kgeom_fail_error(KGEOM_EINVALID_PRESENTATION);

   /* ===  allocate space in the segment for this data  === */

   /* -- if segment does not yet exist, then create it -- */
   if (!kdms_query_segment(obj, segment))
   {
      if (!kdms_create_segment(obj, segment))
	 return FALSE;

      if (!kdms_create_attribute(obj, segment, "internalUsed", 1, 1,
				 KINT, TRUE, TRUE))
	 return FALSE;
   }

   /* -- get old size with assumption that dimension is at least two -- */
   if (!kdms_get_attribute(obj, segment, KDMS_SIZE, &old_size))
      return FALSE;

   /* -- get old used space -- */
   if (!kdms_get_attribute(obj, segment, "internalUsed", &used))
      return FALSE;

   /* -- see if it is necessary to allocate new space -- */
   if (used + size[1] >= old_size[1])
   {
      /* 
       *  this appears confusing, so read carefully :
       *
       *  what we are wanting to do is incrment the height dimension
       *  by a number larger than the amount of space we actually need
       *  so we can avoid allocating new space all the time.  to be
       *  clever, we will determine a number that is a factor of 2048
       *  which is greater than the amount of space we need.  the
       *  space we need, recall, is stored in size[1];
       *
       *  recall that 2^11 = 2048, and you'll see that the functional
       *  equivalent of this equation is : ((size[1]/2048) + 1) * 2048
       *
       */

      /* -- the 'allocation' will occur along height -- */
      new_size[1] = old_size[1] + ((size[1]/10000)+1)*10000;
/*       new_size[1] = old_size[1] + size[1]; */
      new_size[0] = size[0];               /* element size */   

      /* -- set the new size -- */
      if (!kdms_set_attribute(obj, segment, KDMS_SIZE, new_size))
	 return FALSE;
   }
   
   /* -- return the begin point from the allocation -- */
   *begin_point = used;

   /* -- set the new data type -- */
   if (!kdms_set_attribute(obj, segment, KDMS_DATA_TYPE,  data_type))
      return FALSE;

   /* ===  store the data in the allocated space  === */

   /* -- calculate the begin and end points for the data access -- */
   end[0] = size[0] - 1;

   /* -- use the begin and end corners dictated by prior allocation -- */
   beg[1] = *begin_point;
   end[1] = *begin_point + size[1] - 1;

   /* -- used space increases by the amount being put -- */
   used += size[1];
   if (!kdms_set_attribute(obj, segment, "internalUsed", used))
      return FALSE;

   /* -- put the data -- */
   return kdms_put_data(obj, segment, beg, end, data);   
}

/* 
 *    ================================================================== 
 *    Component Definition Methods
 *    ==================================================================
 */

/*-----------------------------------------------------------
|
|  Routine Name: (static) _kgeom_construct_component_defin
|
|       Purpose: This simple routine will construct a component
|                definition structure.  
|
|         Input: segment      - segment identifier
|		 required     - TRUE if the component is required
|		 presentation - presentation function
|		 get          - get data function
|		 put          - put data function
|
|        Output: none
|
|       Returns: the constructed component definition
|
|    Written By: Steve Kubica
|          Date: Feb 04, 1994 15:29
| Modifications:
|
------------------------------------------------------------*/
static kgeom_component_defin *
_kgeom_construct_component_defin(
   char       *segment,
   int         required,
   kfunc_int   presentation,
   kfunc_kaddr get,
   kfunc_int   put)  
{
    kgeom_component_defin *new_defin = NULL;

    new_defin = (kgeom_component_defin *) 
       kcalloc(sizeof(kgeom_component_defin),1);

    new_defin->segment      = kstrdup(segment);
    new_defin->required     = required;
    new_defin->presentation = presentation;
    new_defin->get          = get;
    new_defin->put          = put;

    return new_defin;
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _kgeom_free_component_defin
|
|       Purpose: This simple routine will free a component
|                definition structure.
|
|         Input: comp - pointer to the component definition 
|			to be freed
|
|        Output: none
|
|       Returns: none
|
|    Written By: Steve Kubica
|          Date: Feb 16, 1994 16:24
| Modifications:
|
------------------------------------------------------------*/
static void 
_kgeom_free_component_defin(kgeom_component_defin *comp)
{
    /* -- trivial free -- */
    if (comp == NULL) return;

    kfree(comp->segment);
    kfree(comp);
    return;
}

/* 
 *    ================================================================== 
 *    Primitive Definition Methods
 *    ==================================================================
 */

/*-----------------------------------------------------------
|
|  Routine Name: (static) _kgeom_construct_primitive_defin
|
|       Purpose: This simple routine will construct a primitive
|                definition structure.  The components will be 
|		 initialized to be NULL.
|
|         Input: primitive      - primitive identifier
|		 name           - primitive name
|		 num_components - number of empty components to 
|				  allocate
|
|        Output: none
|
|       Returns: the constructed primitive definition
|
|    Written By: Steve Kubica
|          Date: Feb 04, 1994 15:29
| Modifications:
|
------------------------------------------------------------*/
static kgeom_primitive_defin *
_kgeom_construct_primitive_defin(
   int   primitive,
   char *name,
   int   num_components)
{
    kgeom_primitive_defin *new_defin = NULL;

    new_defin = (kgeom_primitive_defin *)
       kmalloc(sizeof(kgeom_primitive_defin));

    /* -- assign the primitive type and type string -- */
    new_defin->primitive = primitive;
    new_defin->name      = kstrdup(name);

    /* -- allocate space for the given num_components -- */
    new_defin->num_components = num_components;
    new_defin->components = (kgeom_component_defin **)
       kcalloc(sizeof(kgeom_component_defin *),num_components);

    return new_defin;
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _kgeom_free_primitive_defin
|
|       Purpose: This simple routine will free a primitive 
|                definition structure along with all of its
|                component definition structures.
|
|         Input: prim - pointer to the primitive defintion structure 
|			to be freed
|        Output: none
|
|       Returns: none
|
|    Written By: Steve Kubica
|          Date: Feb 16, 1994 16:24
| Modifications:
|
------------------------------------------------------------*/
static void 
_kgeom_free_primitive_defin(kgeom_primitive_defin *prim)
{
    int i;

    /* -- trivial free -- */
    if (prim == NULL) return;

    for( i = 0; i < prim->num_components; i++)
       _kgeom_free_component_defin(prim->components[i]);

    kfree(prim->components);
    kfree(prim->name);
    kfree(prim);
    return;
}

/* 
 *    ================================================================== 
 *    Geometry Primitive Definition Methods
 *    ==================================================================
 */

/*-----------------------------------------------------------
|
|  Routine Name: kgeom_locate_primitive_defin
|
|       Purpose: This routine will search through the static list of
|		 primitive definitions for the defintion corresponding
|		 to the requested primitive.
|
|         Input: primitive - primitive to locate
|
|        Output: None
|
|       Returns: pointer to the desired primitive definition,
|                NULL if it has not been defined
|
|    Written By: Steve Kubica
|          Date: Feb 16, 1994 15:48
| Modifications:
|
------------------------------------------------------------*/
kgeom_primitive_defin *
kgeom_locate_primitive_defin(int primitive)
{
   int i;

   for (i = 0 ; i < kgeom_num_primitive_defins; i++)
      if (kgeom_primitive_defins[i]->primitive == primitive)
	return kgeom_primitive_defins[i];

   kgeom_error(KGEOM_EINVALID_PRIMITIVE);

   return FALSE;
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _kgeom_add_primitive_defin
|
|       Purpose: This routine adds a given primitive to the 
|                end of the static primitive definition list.  
|
|         Input: prim - primitive definition to add
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica
|          Date: Feb 16, 1994 15:48
| Modifications:
|
------------------------------------------------------------*/
static int 
_kgeom_add_primitive_defin(kgeom_primitive_defin *prim)
{
   kgeom_primitive_defin *old_prim = NULL;

   if (prim == NULL)
      return FALSE;

   /* -- see if this primitive has been defined before -- */
   old_prim = kgeom_locate_primitive_defin(prim->primitive);
   if (old_prim != NULL)
      kgeom_fail_error(KGEOM_EPRIMITIVE_REDEFINED);

   /* -- add the primitive to the list -- */
   kgeom_num_primitive_defins++;

   kgeom_primitive_defins = (kgeom_primitive_defin **)
      krealloc(kgeom_primitive_defins, 
	       kgeom_num_primitive_defins*sizeof(kgeom_primitive_defin *));

   if (kgeom_primitive_defins == NULL)
      kgeom_fail_error(KGEOM_EINTERNAL);

   kgeom_primitive_defins[kgeom_num_primitive_defins - 1] = prim; 
   
   return TRUE;
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _kgeom_delete_primitive_defin
|
|       Purpose: This routine will remove the primitive 
|		 definition corresponding to the given 
|		 primitive identifier from the static
|                primitive list.
|
|         Input: primitive - the primitive to delete
|
|        Output: None
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica
|          Date: Feb 16, 1994 15:48
| Modifications:
|
------------------------------------------------------------*/
static int 
_kgeom_delete_primitive_defin(int primitive)
{
   int i = 0;
   int j;   
   int found = FALSE;  

   /* -- see if it is on the list, and where -- */
   do {
      if (kgeom_primitive_defins[i]->primitive == primitive) 
	 found = TRUE; 
      else 
	 i++;

   } while ( i < kgeom_num_primitive_defins || !found);

   /* -- if it's on the list, then delete it -- */
   if (found)
   {
      _kgeom_free_primitive_defin(kgeom_primitive_defins[i]);
      
      /* -- shorten the list -- */
      kgeom_num_primitive_defins--;

      /* -- compress rest of list down - this works even at end of list -- */
      for (j = i ; j < kgeom_num_primitive_defins ; j++)
         kgeom_primitive_defins[j] = kgeom_primitive_defins[j+1];
      
      kgeom_primitive_defins = (kgeom_primitive_defin **)
	 krealloc(kgeom_primitive_defins,
		  kgeom_num_primitive_defins*sizeof(kgeom_primitive_defin *));
   }
   else
      kgeom_fail_error(KGEOM_EPRIMITIVE_NO_DEFINITION);
        
   return TRUE;
}

/* 
 *    ================================================================== 
 *    Public Geometry Primitive calls
 *    ==================================================================
 */

/************************************************************
*
*  Routine Name: kgeom_define_primitive - define a geometry primitive
*
*       Purpose: This routine will define a geometry primitive for a
*		 given session.  
*
*	  Input: primitive      - integer primitive identifier
*		 
*		 name           - string identifier for primitive
*
*                num_components - number of data components in primitive
*
*                kvalist        - the component descriptions; there should
*				  be one component description for each
*				  each compnent in the primitive.
*
*  	      Each component description should contain :
*                         
* 	  	 segment       - string specifying the segment name 
*			         in which the component will be stored.
*
*	         required      - TRUE if the component is required.
*
*        	 presentation  - presentation function for this 
*				 component.  This function will be 
*				 used to determine the size and data
*				 type of the component as determined
*				 by the current presentation attributes.
*
*       	   !The presentation handler declaration is of the form :
*       	   !
*       	   !   int presentation_handler(
*       	   !       kobject   object,
*		   !	   int       primitive,
*		   !	   char     *segment,
*		   !       int     **size,
*		   !	   int      *data_type)
*
*				 Since all components are all considered 
*				 to be at least two-dimensional, this routine 
*				 should return a pointer to a static
*				 array of size two via the size argument.   
*				 The second element of the size array is 
*				 considered to be the number of "points" 
*				 in the component, and the first element 
*				 in the array is considered to be the size 
*				 of the "point".  The data type should be
*				 returned via the data_type argument. The 
*				 object argument is passed from the calling 
*			 	 get or put routine. The primitive and 
*				 segment argument are carried down from 
*				 this definition.
*
*        	 get           - get function for this component.
*				 If NULL, a generic handler will be used
*				 which can cover most data handling.
*				 If special handling of the data is 
*				 required, a customized function can be
*				 provided here for storing the data.
*
*       	   !The get handler declaration is of the form :
*       	   !
*       	   !   kaddr get_handler(
*		   !       kobject obj,
*		   !       char   *segment,
*		   !       int     position,
*		   !       int     component,
*		   !       int    *pres_size, 
*		   !       int     data_type,
*		   !       int    *size,      
*		   !       int     begin_point,
*		   !       kaddr   data)        
*
*				 The object argument is passed from
*				 the calling get routine.  The segment
*				 argument refers to the segment from
*				 which to retrieve the component data.
*				 The position argument is the current
*				 primitive's position in the primitive
*				 list.  The component argument is the
*				 component number for this
*				 primitive. The pres_size and the
*				 data_type are the size and data type
*				 that was returned from the
*				 presentation function.  The size and
*				 begin point reflect the stored
*				 manifest information showing where
*				 the component data is actually
*				 located in the segment.  The begin
*				 point is the index marking the
*				 beginning a linear space along the
*				 dimension in which the points have
*				 been allocated.  The generic handlers
*				 will assume this to be the second
*				 dimension.  The data argument is an
*				 optionally pre-allocated array via
*				 which to return the data.  If this
*				 argument is NULL, then the
*				 get_handler is responsible for
*				 allocating sufficient space.  In
*				 either case, this handler should
*				 return the pointer to the retrieved
*				 data.
*
*        	 put           - put function for this component.
*				 If NULL, a generic handler will be used
*				 which can cover most data handling.
*				 If special handling of the data is 
*				 required, a customized function can be
*				 provided here for retrieving the data.
*				 This routine handles the allocation of
*			 	 the space in which to store the data.
*
*       	   !The put handler declaration is of the form :
*       	   !
*       	   !   int put_handler(
*       	   !       kobject   object,
*		   !       char     *segment,
*		   !       int       position,
*		   !       int       component,
*		   ! 	   int	    *size,
*		   !       int       data_type,
*		   !       int      *begin_point)
*		   !	   kaddr     data)
*
*				 The object argument is passed from
*				 the calling put routine.  The segment
*				 argument refers to the segment in
*				 which to perform the allocation.  The
*				 position argument is the primitive's
*				 position in the primitive list.  The
*				 component argument is the component
*				 number for this primitive. The size
*				 and the data_type are the size and
*				 data type that was returned from the
*				 presentation function.  The begin
*				 point returns the index marking the
*				 beginning a linear space along the
*				 dimension in which the points have
*				 been allocated.  The generic handlers
*				 will assume this to be the second
*				 dimension.  The data argument is the
*				 data being put.  If the segment does
*				 not yet exist in the object, a
*				 kdms_create_segment call will be
*				 used.
*
*        Output: none
*
*       Returns: none
*
*  Restrictions: 
*
*    Written By: Steven Kubica
*          Date: May 13, 1994 16:11
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kgeom_define_primitive(
*		 !	int   primitive,
*		 !	char *name,
*		 !	int   num_components,
*		 !	kvalist)
*
*************************************************************/
int kgeom_define_primitive(
   int   primitive,
   char *name,
   int   num_components,
   kvalist)
{
   kgeom_component_defin *comp = NULL;
   kgeom_primitive_defin *prim = NULL;
   int required;
   int i;
   char       *segment      = NULL;
   kfunc_int   presentation = NULL;
   kfunc_kaddr get          = NULL;
   kfunc_int   put          = NULL;
   kva_list    list;

   KGEOM_INIT;
   
   /* -- arbitrarily give verbose warning if number seems too large -- */
   if (num_components > 100)
      kinfo(KSYSLIB,"kgeom_define_primitive : Attempt to define primitive "
		     "%s with %d components. That is sure a lot of components",
		     name, num_components);

   /* -- verify that this primitive has not already been defined -- */
   if ((prim = kgeom_locate_primitive_defin(primitive)) != NULL)
      kgeom_fail_error(KGEOM_EPRIMITIVE_REDEFINED);

   /* -- construct an empty primitive definition -- */
   prim = _kgeom_construct_primitive_defin(primitive,name,num_components);
   if (prim == NULL)
      kgeom_fail_error(KGEOM_EINTERNAL);

   /* -- get component information off of the variable argument list -- */
   kva_start(list,num_components);

   for (i = 0; i < num_components; i++ )
   {
      /* -- pull all the component information for this component -- */
      segment  = (char *) kva_arg(list, char *);
      required = (int)    kva_arg(list, int); 

      presentation = (kfunc_int)   kva_arg(list, kfunc_int); 
      get          = (kfunc_kaddr) kva_arg(list, kfunc_kaddr); 
      put          = (kfunc_int)   kva_arg(list, kfunc_int); 

      if (presentation == NULL)
         kgeom_fail_error(KGEOM_EMUST_PROVIDE_PRESENTATION_HANDLER);

      if (get       == NULL) get       = _get_component;
      if (put       == NULL) put       = _put_component;

      comp = _kgeom_construct_component_defin(segment, required, presentation, 
					      get, put);

      /* -- put component in the appropriate place on the component list */
      prim->components[i] = comp;
   }

   /* -- end the variable argument list -- */
   kva_end(list);

   /* -- add this new primitive to the global primitive definition list -- */
   return _kgeom_add_primitive_defin(prim);
}

/************************************************************
*
*  Routine Name: kgeom_undefine_primitive - undefine a defined primitive
*
*       Purpose: This function will remove a primitive definition 
*                from the definition list.  
*
*         Input: primitive - the primitive to undefine. 
*
*        Output: none
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Steve Kubica
*          Date: Mar 25, 1994 13:37
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kgeom_undefine_primitive(
*                !      int primitive)
*
*************************************************************/
int 
kgeom_undefine_primitive(int primitive)
{
   KGEOM_INIT;

   return _kgeom_delete_primitive_defin(primitive);
}

/************************************************************
*
*  Routine Name: kgeom_primitive_name
*
*       Purpose: This routine will take a given geometry primitive
*                type and return a corresponding string.  
*                For example, kgeom_type_to_string(KGEOM_SPHERES)
*		 would return the string "spheres".
*
* 		 The corresponding string is defined via the geometry 
* 		 primitive definition call.  If the given primitive 
*                has not yet been defined, this routine will return NULL.
*
*		 The returned string is a pointer to the actual internal 
*		 string from the definition and should not be altered of
*		 freed.
*
*         Input: primitive - primitive to return the name for
*
*        Output: none
*
*       Returns: A pointer to the string corresponding to the given 
*                primitive if it has been defined.  NULL otherwise.
*
*  Restrictions:
*
*    Written By: Steven Kubica
*          Date: Mar 09, 1994 10:19
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration:
*
*************************************************************/
char *
kgeom_primitive_name(int primitive)
{
   kgeom_primitive_defin *prim;

   KGEOM_INIT;

   if ((prim = kgeom_locate_primitive_defin(primitive)) == NULL)
      return NULL;

   if (prim->name == NULL)
      return NULL;
   else
      return prim->name;
}
