 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>                Color Data Services
   >>>>
   >>>>   Static:       
   >>>>                 _map_autocolor_get()
   >>>>                 _map_autocolor_set()
   >>>>                 _map_autocolor_match()
   >>>>                 _map_autocolor_copy()
   >>>>                 _map_autocolor_query()
   >>>>                 _map_autocolor_print()
   >>>>			_map_autocolor_list_get()
   >>>>			_map_autocolor_list_query()
   >>>>			_map_autocolor_list_print()
   >>>>			_map_autocolor_num_get()
   >>>>			_map_autocolor_num_query()
   >>>>			_map_autocolor_num_print()
   >>>>
   >>>>                 _map_operation_get()
   >>>>                 _map_operation_set()
   >>>>                 _map_operation_match()
   >>>>                 _map_operation_copy()
   >>>>                 _map_operation_query()
   >>>>                 _map_operation_print()
   >>>>			_map_operation_num_get()
   >>>>			_map_operation_num_query()
   >>>>			_map_operation_num_print()
   >>>>
   >>>>                 _colorspace_get()
   >>>>                 _colorspace_set()
   >>>>                 _colorspace_match()
   >>>>                 _colorspace_copy()
   >>>>                 _colorspace_query()
   >>>>                 _colorspace_print()
   >>>>
   >>>>  Private:
   >>>>                 kcolor_init()
   >>>>   Public:
   >>>>                 kcolor_get_attribute()
   >>>>                 kcolor_get_attributes()
   >>>>                 kcolor_set_attribute()
   >>>>                 kcolor_set_attributes()
   >>>>                 kcolor_match_attribute()
   >>>>                 kcolor_match_attributes()
   >>>>                 kcolor_copy_attribute()
   >>>>                 kcolor_copy_attributes()
   >>>>                 kcolor_query_attribute()
   >>>>                 kcolor_print_attribute()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"

int _color_initialized = FALSE;

extern char *kcolor_map_autocolor_list[];
extern int   kcolor_map_autocolor_num;

extern char *kcolor_map_operation_list[];
extern int    kcolor_map_operation_num;


/*
 *    ================================================================== 
 *    KCOLOR Quasi Attribute Methods
 *    ==================================================================
 */

/*********** ---------------------------------------------------------------
 ***********  KCOLOR_MAP_AUTOCOLOR
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: (static) _map_autocolor_get
|       Purpose: get the map_autocolor attribute
|    Written By: John M. Salas & Jeremy Worley
|          Date: Oct 12, 1993
------------------------------------------------------------*/
/* ARGSUSED */
static int
_map_autocolor_get(kobject obj, int assoc, int attr, kaddr clientData,
	       kva_list * list)
{
   int *ret_typ;
   int  typ;


   if (!kdms_get_attribute(obj, NULL, _INTERNAL_MAP_AUTOCOLOR, &typ))
      return (FALSE);

   ret_typ = kva_arg(*list, int *);
   if (ret_typ != NULL) *ret_typ = typ;

   return (TRUE);
}

/*-----------------------------------------------------------
|  Routine Name: (static) _map_autocolor_set
|       Purpose: set the map_autocolor attribute
|    Written By: John M. Salas & Jeremy Worley
|          Date: Oct 12, 1993
------------------------------------------------------------*/
/* ARGSUSED */
static int
_map_autocolor_set(kobject obj, int assoc, int attr, kaddr clientData,
	       kva_list * list)
{
   int typ;


   typ = kva_arg(*list, int);
   kmap_map_autocolor(obj, typ);

   /* need to set this so we have a record of the map_autocolor type */
   if (!kdms_set_attribute(obj, NULL, _INTERNAL_MAP_AUTOCOLOR, typ))
      return (FALSE);

   return (TRUE);
}

/*-----------------------------------------------------------
|  Routine Name: (static) _map_autocolor_match
|       Purpose: match the map_autocolor attribute
|    Written By: Steve Kubica & John Salas
|          Date: Dec 23, 1993
------------------------------------------------------------*/
/* ARGSUSED */
static int
_map_autocolor_match(kobject obj1, kobject obj2, int assoc, int attrib,
		 kaddr clientData1, kaddr clientData2)
{
   return (kdms_match_attribute(obj1, obj2, NULL, _INTERNAL_MAP_AUTOCOLOR));
}

/*-----------------------------------------------------------
|  Routine Name: (static) _map_autocolor_copy
|       Purpose: copy the map_autocolor attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_map_autocolor_copy(kobject obj1, kobject obj2, int assoc, int attrib,
		kaddr clientData1, kaddr clientData2)
{
   int typ;


   if (!kdms_get_attribute(obj1, NULL, _INTERNAL_MAP_AUTOCOLOR, &typ))
      return (FALSE);

   if (!kdms_set_attribute(obj2, NULL, _INTERNAL_MAP_AUTOCOLOR, typ))
      return (FALSE);

   return (TRUE);
}


/*-----------------------------------------------------------
|  Routine Name: (static) _map_autocolor_query
|       Purpose: query the map_autocolor attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_map_autocolor_query(kobject obj, int assoc, int attrib, kaddr clientData,
		 int *num_args, int *arg_size, int *data_type,
		 int *permanent)
{
   if (num_args)
      *num_args = 1;
   if (arg_size)
      *arg_size = 1;
   if (data_type)
      *data_type = KINT;
   if (permanent)
      *permanent = FALSE;

   return (TRUE);
}

/*-----------------------------------------------------------
|  Routine Name: (static) _map_autocolor_print
|       Purpose: print the map_autocolor attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_map_autocolor_print(kobject obj, int assoc, int attrib, kaddr clientData,
		 kfile * outfile)
{
   int typ;


   if (!kdms_get_attribute(obj, NULL, _INTERNAL_MAP_AUTOCOLOR, &typ))
      return (FALSE);

   if (outfile)
   {
      switch (typ)
      {
	 case KRGB_CUBE:
	    kfprintf(outfile, "Autocolor : KRGB_CUBE");
	    break;
	 case KRGB_TRIANGLE:
	    kfprintf(outfile, "Autocolor : KRGB_TRIANGLE");
	    break;
	 case KRGB_SPIRAL:
	    kfprintf(outfile, "Autocolor : KRGB_SPIRAL");
	    break;
	 case KHLS_SPIRAL:
	    kfprintf(outfile, "Autocolor : KHLS_SPIRAL");
	    break;
	 case KHSV_RINGS:
	    kfprintf(outfile, "Autocolor : KHSV_RINGS");
	    break;
	 case KRGB_DISTANCE:
	    kfprintf(outfile, "Autocolor : KRGB_DISTANCE");
	    break;
	 case KCIE_DIAGRAM:
	    kfprintf(outfile, "Autocolor : KCIE_DIAGRAM");
	    break;
	 case KDENSITY_SLICE:
	    kfprintf(outfile, "Autocolor : KDENSITY_SLICE");
	    break;
	 case KGREYSCALE:
	    kfprintf(outfile, "Autocolor : KGREYSCALE");
	    break;
	 case KEQUALIZE:
	    kfprintf(outfile, "Autocolor : KEQUALIZE");
	    break;
	 case KSTRETCH:
	    kfprintf(outfile, "Autocolor : KSTRETCH");
	    break;
	 case KSTDDEV:
	    kfprintf(outfile, "Autocolor : KSTDDEV");
	    break;
	 case KSA_PSEUDO:
	    kfprintf(outfile, "Autocolor : KSA_PSEUDO");
	    break;
	 case KRAINBOW:
	    kfprintf(outfile, "Autocolor : KRAINBOW");
	    break;
	 case KDISJOINT:
	    kfprintf(outfile, "Autocolor : KDISJOINT");
	    break;
	 case KGREYCODE:
	    kfprintf(outfile, "Autocolor : KGREYCODE");
	    break;
	 case KORIGINAL:
	    kfprintf(outfile, "Autocolor : KORIGINAL");
	    break;
	 case KMAP332:
	    kfprintf(outfile, "Autocolor : KMAP332");
	    break;
	 default:
	    kfprintf(outfile, "Autocolor : %d (invalid)", typ);
      }
   }
   return (TRUE);
}

/*********** ---------------------------------------------------------------
 ***********  KCOLOR_MAP_AUTOCOLOR_LIST
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: (static) _map_autocolor_list_get
|       Purpose: get the colormap attribute
|    Written By: Jeremy Worley
|          Date: May 31, 1994 14:35
------------------------------------------------------------*/
/* ARGSUSED */
static int
_map_autocolor_list_get(kobject obj, int assoc, int attr, kaddr clientData,
		kva_list * list)
{
   char ***arg1;
   int    *arg2;

   
   arg1 = kva_arg(*list, char ***);
   arg2 = kva_arg(*list, int *);

   /*
    * get the address of the external map_autocolor list, so that we
    * can pass it back correctly.  Assigning kcolor_map_autocolor_list
    * directly to arg1 gives us the first element, instead of the
    * pointer to the list.  So we assign the address of the first
    * element.  
    */
   if (arg1 != NULL)
      *arg1 = &kcolor_map_autocolor_list[0];

   if (arg2 != NULL)
      *arg2 = kcolor_map_autocolor_num;
   
   return (TRUE);
}

/*-----------------------------------------------------------
|  Routine Name: (static) _map_autocolor_list_print
|       Purpose: print the map_autocolor_list attribute
|    Written By: Jeremy Worley
|          Date: May 31, 1994 14:35
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_map_autocolor_list_print(kobject obj, int assoc, int attrib, kaddr clientData,
		  kfile * outfile)
{
   int i;


   for (i = 0; i < kcolor_map_autocolor_num; i++)
      kfprintf(outfile, "%s\n", kcolor_map_autocolor_list[i]);

   return (TRUE);
}

/*********** ---------------------------------------------------------------
 ***********  KCOLOR_MAP_OPERATION
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: (static) _map_operation_get
|       Purpose: get the map_operation attribute
|    Written By: John M. Salas & Jeremy Worley
|          Date: Oct 12, 1993
------------------------------------------------------------*/
/* ARGSUSED */
static int
_map_operation_get(kobject obj, int assoc, int attr, kaddr clientData,
	       kva_list * list)
{
   int *ret_typ;
   int  typ;


   if (!kdms_get_attribute(obj, NULL, _INTERNAL_MAP_OPERATION, &typ))
      return (FALSE);

   ret_typ = kva_arg(*list, int *);
   if (ret_typ != NULL) *ret_typ = typ;

   return (TRUE);
}

/*-----------------------------------------------------------
|  Routine Name: (static) _map_operation_set
|       Purpose: set the map_operation attribute
|    Written By: John M. Salas & Jeremy Worley
|          Date: Oct 12, 1993
------------------------------------------------------------*/
/* ARGSUSED */
static int
_map_operation_set(kobject obj, int assoc, int attr, kaddr clientData,
	       kva_list * list)
{
   int typ;


   typ = kva_arg(*list, int);
   kmap_map_operation(obj, typ);

   /* need to set this so we have a record of the map_operation type */
   if (!kdms_set_attribute(obj, NULL, _INTERNAL_MAP_OPERATION, typ))
      return (FALSE);

   return (TRUE);
}

/*-----------------------------------------------------------
|  Routine Name: (static) _map_operation_match
|       Purpose: match the map_operation attribute
|    Written By: Steve Kubica & John Salas
|          Date: Dec 23, 1993
------------------------------------------------------------*/
/* ARGSUSED */
static int
_map_operation_match(kobject obj1, kobject obj2, int assoc, int attrib,
		 kaddr clientData1, kaddr clientData2)
{
   return (kdms_match_attribute(obj1, obj2, NULL, 
                                _INTERNAL_MAP_OPERATION));
}

/*-----------------------------------------------------------
|  Routine Name: (static) _map_operation_copy
|       Purpose: copy the map_operation attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_map_operation_copy(kobject obj1, kobject obj2, int assoc, int attrib,
		kaddr clientData1, kaddr clientData2)
{
   int typ;


   if (!kdms_get_attribute(obj1, NULL, _INTERNAL_MAP_OPERATION, &typ))
      return (FALSE);

   if (!kdms_set_attribute(obj2, NULL, _INTERNAL_MAP_OPERATION, typ))
      return (FALSE);

   return (TRUE);
}


/*-----------------------------------------------------------
|  Routine Name: (static) _map_operation_query
|       Purpose: query the map_operation attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_map_operation_query(kobject obj, int assoc, int attrib, kaddr clientData,
		 int *num_args, int *arg_size, int *data_type,
		 int *permanent)
{
   if (num_args)
      *num_args = 1;
   if (arg_size)
      *arg_size = 1;
   if (data_type)
      *data_type = KINT;
   if (permanent)
      *permanent = FALSE;

   return (TRUE);
}

/*-----------------------------------------------------------
|  Routine Name: (static) _map_operation_print
|       Purpose: print the map_operation attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_map_operation_print(kobject obj, int assoc, int attrib, kaddr clientData,
		 kfile * outfile)
{
   int typ;


   if (!kdms_get_attribute(obj, NULL, _INTERNAL_MAP_OPERATION, &typ))
      return (FALSE);

   if (outfile)
   {
      switch (typ)
      {
	 case KINVERT:
	    kfprintf(outfile, "Autocolor : KINVERT");
	    break;
	 case KINVERT_ORIG:
	    kfprintf(outfile, "Autocolor : KINVERT_ORIG");
	    break;
	 case KRANDOM:
	    kfprintf(outfile, "Autocolor : KRANDOM");
	    break;
	 case KREVERSE:
	    kfprintf(outfile, "Autocolor : KREVERSE");
	    break;
	 case KROW_ROTLEFT:
	    kfprintf(outfile, "Autocolor : KROW_ROTLEFT");
	    break;
	 case KROW_ROTRIGHT:
	    kfprintf(outfile, "Autocolor : KROW_ROTRIGHT");
	    break;
	 case KSWAP_REDGREEN:
	    kfprintf(outfile, "Autocolor : KSWAP_REDGREEN");
	    break;
	 case KSWAP_REDBLUE:
	    kfprintf(outfile, "Autocolor : KSWAP_REDBLUE");
	    break;
	 case KSWAP_GREENBLUE:
	    kfprintf(outfile, "Autocolor : KSWAP_GREENBLUE");
	    break;
	 case KRED_FILTER:
	    kfprintf(outfile, "Autocolor : KRED_FILTER");
	    break;
	 case KGREEN_FILTER:
	    kfprintf(outfile, "Autocolor : KGREEN_FILTER");
	    break;
	 case KBLUE_FILTER:
	    kfprintf(outfile, "Autocolor : KBLUE_FILTER");
	    break;
	 case KCHAIN_ROTLEFT:
	    kfprintf(outfile, "Autocolor : KCHAIN_ROTLEFT");
	    break;
	 case KCHAIN_ROTRIGHT:
	    kfprintf(outfile, "Autocolor : KCHAIN_ROTRIGHT");
	    break;
	 case KRED_ROTLEFT:
	    kfprintf(outfile, "Autocolor : KRED_ROTLEFT");
	    break;
	 case KRED_ROTRIGHT:
	    kfprintf(outfile, "Autocolor : KRED_ROTRIGHT");
	    break;
	 case KGREEN_ROTLEFT:
	    kfprintf(outfile, "Autocolor : KGREEN_ROTLEFT");
	    break;
	 case KGREEN_ROTRIGHT:
	    kfprintf(outfile, "Autocolor : KGREEN_ROTRIGHT");
	    break;
	 case KBLUE_ROTLEFT:
	    kfprintf(outfile, "Autocolor : KBLUE_ROTLEFT");
	    break;
	 case KBLUE_ROTRIGHT:
	    kfprintf(outfile, "Autocolor : KBLUE_ROTRIGHT");
	    break;
	 default:
	    kfprintf(outfile, "Autocolor : %d (invalid)", typ);
      }
   }
   return (TRUE);
}

/*********** ---------------------------------------------------------------
 ***********  KCOLOR_MAP_OPERATION_LIST
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: (static) _map_operation_list_get
|       Purpose: get the map operation attribute
|    Written By: Jeremy Worley
|          Date: May 31, 1994 14:35
------------------------------------------------------------*/
/* ARGSUSED */
static int
_map_operation_list_get(kobject obj, int assoc, int attr, kaddr clientData,
                kva_list * list)
{
   char ***arg1;
   int    *arg2;


   arg1 = kva_arg(*list, char ***);
   arg2 = kva_arg(*list, int *);

   /*
    * get the address of the external map_autocolor list, so that we can pass it
    * back correctly.  Assigning kcolor_map_autocolor_list directly to arg1 
    * gives us the first element, instead of the pointer to the list.  So we 
    * assign the address of the first element.
    */
   if (arg1 != NULL)
      *arg1 = &kcolor_map_operation_list[0];

   if (arg2 != NULL)
      *arg2 = kcolor_map_operation_num;

   return (TRUE);
}

/*-----------------------------------------------------------
|  Routine Name: (static) _map_operation_list_print
|       Purpose: print the map_operation_list attribute
|    Written By: Jeremy Worley
|          Date: May 31, 1994 14:35
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_map_operation_list_print(kobject obj, int assoc, int attrib, kaddr clientData,
                  kfile * outfile)
{
   int i;


   for (i = 0; i < kcolor_map_operation_num; i++)
      kfprintf(outfile, "%s\n", kcolor_map_operation_list[i]);

   return (TRUE);
}



/*********** ---------------------------------------------------------------
 ***********  KCOLOR_COLORSPACE
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: (static) _colorspace_get
|       Purpose: get the colorspace attribute
|    Written By: Steve Kubica
|          Date: Apr 28, 1994 21:07
------------------------------------------------------------*/
/* ARGSUSED */
static int
_colorspace_get(kobject obj, int assoc, int attr, kaddr clientData,
		kva_list * list)
{
   int *ret_typ;
   int  typ;


   if (!kdms_get_attribute(obj, NULL, _INTERNAL_COLORSPACE, &typ))
      return (FALSE);

   ret_typ = kva_arg(*list, int *);
   if (ret_typ != NULL) *ret_typ = typ;

   return (TRUE);
}

/*-----------------------------------------------------------
|  Routine Name: (static) _colorspace_set
|       Purpose: set the colorspace attribute
|    Written By: Steve Kubica
|          Date: Apr 28, 1994 21:09
------------------------------------------------------------*/
/* ARGSUSED */
static int
_colorspace_set(kobject obj, int assoc, int attr, kaddr clientData,
		kva_list * list)
{
   int typ;


   typ = kva_arg(*list, int);

   /* need to set this so we have an permanent record of the colorspace type */
   if (!kdms_set_attribute(obj, NULL, _INTERNAL_COLORSPACE, typ))
      return (FALSE);

   return (TRUE);
}

/*-----------------------------------------------------------
|  Routine Name: (static) _colorspace_match
|       Purpose: match the colorspace attribute
|    Written By: Steve Kubica
|          Date: Apr 28, 1994 21:09
------------------------------------------------------------*/
/* ARGSUSED */
static int
_colorspace_match(kobject obj1, kobject obj2, int assoc, int attrib,
		  kaddr clientData1, kaddr clientData2)
{
   return (kdms_match_attribute(obj1, obj2, NULL, _INTERNAL_COLORSPACE));
}

/*-----------------------------------------------------------
|  Routine Name: (static) _colorpace_copy
|       Purpose: copy the colorspace attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_colorspace_copy(kobject obj1, kobject obj2, int assoc, int attrib,
		 kaddr clientData1, kaddr clientData2)
{
   int typ;


   if (!kdms_get_attribute(obj1, NULL, _INTERNAL_COLORSPACE, &typ))
      return (FALSE);

   if (!kdms_set_attribute(obj2, NULL, _INTERNAL_COLORSPACE, typ))
      return (FALSE);

   return (TRUE);
}


/*-----------------------------------------------------------
|  Routine Name: (static) _map_autocolor_query
|       Purpose: query the map_autocolor attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_colorspace_query(kobject obj, int assoc, int attrib, kaddr clientData,
		  int *num_args, int *arg_size, int *data_type,
		  int *permanent)
{
   if (num_args)
      *num_args = 1;
   if (arg_size)
      *arg_size = 1;
   if (data_type)
      *data_type = KINT;
   if (permanent)
      *permanent = TRUE;

   return (TRUE);
}

/*-----------------------------------------------------------
|  Routine Name: (static) _colorspace_print
|       Purpose: print the colorspace attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_colorspace_print(kobject obj, int assoc, int attrib, kaddr clientData,
		  kfile * outfile)
{
   int typ;


   if (!kdms_get_attribute(obj, NULL, _INTERNAL_COLORSPACE, &typ))
      return (FALSE);

   if (outfile)
   {
      switch (typ)
      {
	 case KGREY:
	    kfprintf(outfile, "KGREY");
	    break;
	 case KRGB:
	    kfprintf(outfile, "KRGB");
	    break;
	 case KCMY:
	    kfprintf(outfile, "KCMY");
	    break;
	 case KYIQ:
	    kfprintf(outfile, "KYIQ");
	    break;
	 case KHSV:
	    kfprintf(outfile, "KHSV");
	    break;
	 case KHLS:
	    kfprintf(outfile, "KHLS");
	    break;
	 case KIHS:
	    kfprintf(outfile, "KIHS");
	    break;
	 case KXYZ:
	    kfprintf(outfile, "KXYZ");
	    break;
	 case KUVW:
	    kfprintf(outfile, "KUVW");
	    break;
	 case KUCSUVW:
	    kfprintf(outfile, "KUCSUVW");
	    break;
	 case KUCSSOW:
	    kfprintf(outfile, "KUCSSOW");
	    break;
	 case KUCSLab:
	    kfprintf(outfile, "KUCSLab");
	    break;
	 case KUCSLuv:
	    kfprintf(outfile, "KUCSLuv");
	    break;
	 case KUSERDEFINED:
	    kfprintf(outfile, "KUSERDEFINED");
	    break;
	 default:
	    kfprintf(outfile, "%d (invalid)", typ);
      }
   }
   return (TRUE);
}



/*
 *    ==================================================================
 *    Initialization Routine
 *    ==================================================================
 */

/*-----------------------------------------------------------
|
|  Routine Name: kcolor_init
|
|       Purpose: This routine initializes all the attributes required
|		 by color services.
|
|         Input: none
|
|        Output: none
|
|       Returns: none
|
|    Written By: Steve Kubica
|          Date: May 03, 1994 14:38
| Modifications:
|
------------------------------------------------------------*/
void
kcolor_init(void)
{
   if (_color_initialized)
      return;
   else
      _color_initialized = TRUE;


   /*
    *   map_autocolor and colormap attributes 
    */
   kdms_define_quasi_attribute(KDMS_OBJECT, KCOLOR_MAP_AUTOCOLOR, NULL,
		_map_autocolor_get, _map_autocolor_set, _map_autocolor_match,
		_map_autocolor_copy, _map_autocolor_query, _map_autocolor_print);

   kdms_define_attribute(KDMS_OBJECT, _INTERNAL_MAP_AUTOCOLOR, 1, 1, KINT,
			 FALSE, TRUE, KCOLOR_MAP_AUTOCOLOR_DEFAULT);

   kdms_define_quasi_attribute(KDMS_OBJECT, KCOLOR_MAP_OPERATION, NULL,
		_map_operation_get, _map_operation_set, 
                _map_operation_match, _map_operation_copy, 
                _map_operation_query, _map_operation_print);

   kdms_define_attribute(KDMS_OBJECT, _INTERNAL_MAP_OPERATION, 1, 1, KINT,
			 FALSE, TRUE, KCOLOR_MAP_OPERATION_DEFAULT);

   kdms_define_quasi_attribute(KDMS_OBJECT, KCOLOR_COLORSPACE, NULL,
		_colorspace_get, _colorspace_set, _colorspace_match,
		_colorspace_copy, _colorspace_query, _colorspace_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, KCOLOR_MAP_AUTOCOLOR_LIST, NULL,
		_map_autocolor_list_get, NULL, NULL, NULL, NULL,
		_map_autocolor_list_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, KCOLOR_MAP_OPERATION_LIST, NULL,
		_map_operation_list_get, NULL, NULL, NULL, NULL,
		_map_operation_list_print);

   kdms_define_attribute(KDMS_OBJECT, _INTERNAL_COLORSPACE, 1, 1, KINT,
			 TRUE, TRUE, KCOLOR_COLORSPACE_DEFAULT);

   kdms_define_attribute(KDMS_OBJECT, KCOLOR_HAS_ALPHA, 1, 1, KINT,
			 TRUE, TRUE, KCOLOR_HAS_ALPHA_DEFAULT);


   /*
    *   rgb mapping functions will one day be defined here ... 
    */



   /*
    * It is possible to generate errnos with duplicate attribute
    * definitions.  To prevent the user from seeing this, we'll
    * re-zero the errno.
    */
   errno = 0;

   return;
}


/*
 *    ==================================================================
 *    Public API Routines
 *    ==================================================================
 */

/************************************************************
*
*  Routine Name: kcolor_get_attribute - get the values of a color
*                      	                attribute from a data object.
*
*       Purpose: This function is used to retrieve the value of a
*		 a color attribute from a data object.
*
*		 This color service function should be used in
*		 conjunction with other application services such as
*		 polymorphic data services and geometry data services.
*		 This function will work on data objects opened or
*		 created with either of those services.
*
*		 Attributes are retrieved by passing in the address of
*		 a variable by which the attribute can be returned.
*		 Note that any array attributes, such as strings,
*		 which are retrieved should not be altered or freed.
*		 The pointer returned points to the actual internal
*		 storage array.  A copy should be made if the values
*		 need to be changed.
*
*                The following example illustrates the use of the get
*                attribute call to retrieve two different color
*                attributes.
*
* !   \s-2\f(CWchar **autocolor_list;\s+2\fP
* !   \s-2\f(CWint    num;\s+2\fP
* !   \s-2\f(CWint    colorspace;\s+2\fP
* !
* !   \s-2\f(CWkcolor_get_attribute(object, \s+2\fP
* !   \s-2\f(CW                     KCOLOR_AUTOCOLOR_LIST, &list, &num);\s+2\fP
* !   \s-2\f(CWkcolor_get_attribute(object, \s+2\fP
* !   \s-2\f(CW                     KCOLOR_COLORSPACE, &colorspace);\s+2\fP
*
*                A complete list of color attributes can be found in
*                Chapter 4 of Programming Services Volume II.
*
*         Input: object    - the data object from which the attribute's 
*                            value will be retrieved
*
*                attribute - the attribute to get
*
*                kvalist   - a variable argument list that contains
*                            the addresses of variables which will be 
*			     used to return the different components 
*			     associated with that attribute.
*
*                            The variable argument list takes the form:
*
*                  !  \s-2\f(CWATTRIBUTE_NAME, &value1 [, &value2, ...]\s+2\fP
*
*                            The number of value arguments in the variable 
*                            argument list corresponds to the number of
*			     arguments needed to retrieve the attribute.
*        Output: none
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: Calling this function with an incorrect number of arguments 
*                in the variable argument list will not cause any compiler 
*                errors, but will often generate a segmentation fault.
*    Written By: Steve Kubica 
*          Date: Sep 30, 1993 13:44
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: int kcolor_get_attribute(
*		!   kobject  object,
*		!   char    *attribute,
*		!   kvalist)
*
*************************************************************/
int
kcolor_get_attribute(
   kobject object, 
   char   *attribute, 
   kvalist)
{
   kva_list list;
   int status;

   kcolor_init();

   kva_start(list, attribute);

   status = kdms_vget_attribute(object, KDMS_OBJECT, attribute, &list);

   kva_end(list);

   return (status);
}

/************************************************************
*
*  Routine Name: kcolor_get_attributes - get multiple color attributes from a
*                                        data object.
*
*       Purpose: This function is used to retrieve the values of an 
*		 arbitrary number of attributes from a data object.
*
*		 This color service function should be used in
*		 conjunction with other application services such as
*		 polymorphic data services and geometry data services.
*		 This function will work on data objects opened or
*		 created with either of those services.
*		 
*		 Attributes are retrieved by passing in the address of
*		 a variable by which the attribute can be returned.
*		 Note that any array attributes, such as strings,
*		 which are retrieved should not be altered or freed.
*		 The pointer returned points to the actual internal
*		 storage array.  A copy should be made if the values
*		 need to be changed.
*
*                The following example illustrates the use of a 
*                single get attributes call to retrieve two different
*                color attributes.
*
* !   \s-2\f(CWchar **autocolor_list;\s+2\fP
* !   \s-2\f(CWint    num;\s+2\fP
* !   \s-2\f(CWint    colorspace;\s+2\fP
* !
* !   \s-2\f(CWkcolor_get_attributes(object, \s+2\fP
* !   \s-2\f(CW                      KCOLOR_AUTOCOLOR_LIST, &list, &num,\s+2\fP
* !   \s-2\f(CW                      KCOLOR_COLORSPACE, &colorspace,\s+2\fP
* !   \s-2\f(CW                      NULL);\s+2\fP
*
*                A complete list of color attributes can be found in
*                Chapter 4 of Programming Services Volume II.
*
*         Input: object    - the data object from which the values of the
*                            attributes will be retrieved
*
*        Output: kvalist   - NULL terminated variable argument list
*                            which contains a list of attributes,
*			     each attribute followed by the addresses of
*			     variables which will be used to return the
*                            different components associated with that
*                            attribute. 
*
*			     The variable argument list takes the form:
*
*                !  \s-2\f(CWATTRIBUTE_NAME1, &value1 [, &value2, ...],\s+2\fP
*                !  \s-2\f(CWATTRIBUTE_NAME2, &value1,[, &value2, ...],\s+2\fP
*                !  \s-2\f(CW..., NULL \s+2\fP
*
*                            The number of value arguments in the
*                            variable argument list for each attribute
*                            depends on the attribute. The NULL at the
*                            end of the variable argument list serves
*                            as a flag indicating the end of the list.
*
*			     Be careful not to forget the NULL at the
*			     end of the list.  This is a common programming
*			     error which unfortunately will not generate
*			     any compiler warnings. 
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: Calling this function with an incorrect number of arguments 
*                in the variable argument list will not cause any compiler 
*                errors, but will often generate a segmentation fault.
*    Written By: Steve Kubica
*          Date: May 03, 1994 14:38
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: int kcolor_get_attributes(
*		!   kobject  object,
*		!   kvalist)
*
*************************************************************/
int
kcolor_get_attributes(
   kobject object, 
   kvalist)
{
   kva_list list;
   int status;

   kcolor_init();

   kva_start(list, object);

   status = kdms_vget_attributes(object, KDMS_OBJECT, &list);

   kva_end(list);

   return (status);
}

/************************************************************
*
*  Routine Name: kcolor_set_attribute - set the value of a color
*                                       attribute in a data object.
*
*       Purpose: This function is used to assign the value of a color
*	 	 attribute to a data object.
*
*		 This color service function should be used in
*		 conjunction with other application services such as
*		 polymorphic data services and geometry data services.
*		 This function will work on data objects opened or
*		 created with either of those services.
*
*		 Attributes are set by passing in the attribute name
*		 along with the value or variable containing the value
*		 to assign to the attribute.
*
*                The following example illustrates the use of the set
*                attribute call to assign two different color
*                attributes.  The define \s-2\f(CWKRGB\s+2\fP could have been
*		 passed in directly to set the colorspace attribute.
*
* !   \s-2\f(CWint    colorspace = KRGB;\s+2\fP
* ! 
* !   \s-2\f(CWkcolor_set_attribute(object, \s+2\fP
* !   \s-2\f(CW                     KCOLOR_COLORSPACE, colorspace);,\s+2\fP
* !   \s-2\f(CWkcolor_set_attribute(object, \s+2\fP
* !   \s-2\f(CW                     KCOLOR_MAP_OPERATION, KROW_LEFT);\s+2\fP
*
*                A complete list of color attributes can be found in
*                Chapter 4 of Programming Services Volume II.
*
*         Input: object    - the data object into which the attribute's 
*                            value will be assigned
*
*                attribute - the attribute to set
*                
*                kvalist   - a variable argument list that contains the 
*			     values that will be assigned to the 
*			     different components associated with 
*			     that attribute.
*
*                            The variable argument list takes the form:
*
*                  !  \s-2\f(CWATTRIBUTE_NAME, value1 [, value2, ...]\s+2\fP
*
*                            The number of value arguments in the variable 
*                            argument list corresponds to the number of
*			     arguments needed to set the attribute.
*        Output: none 
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: Calling this function with an incorrect number of arguments 
*                in the variable argument list will not cause any compiler 
*                errors, but will often generate a segmentation fault.
*    Written By: Steve Kubica
*          Date: May 03, 1994 14:39
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: int kcolor_set_attribute(
*		!   kobject  object,
*		!   char    *attribute,
*		!   kvalist)
*
*************************************************************/
int
kcolor_set_attribute(
   kobject object, 
   char   *attribute, 
   kvalist)
{
   kva_list list;
   int status;

   kcolor_init();

   kva_start(list, attribute);

   status = kdms_vset_attribute(object, KDMS_OBJECT, attribute, &list);

   kva_end(list);

   return (status);
}

/************************************************************
*
*  Routine Name: kcolor_set_attributes - set multiple color attributes in a
*                                        data object.
*
*       Purpose: This function is used to assign the values of an 
*		 arbitrary number of color attributes to a data object.
*
*		 This color service function should be used in
*		 conjunction with other application services such as
*		 polymorphic data services and geometry data services.
*		 This function will work on data objects opened or
*		 created with either of those services.
*
*		 Attributes are set by passing in the attribute name
*		 along with the value or variable containing the value
*		 to assign to the attribute.
*
*                The following example illustrates the use of a single
*                set attributes call to assign two different color
*                attributes. The define \s-2\f(CWKRGB\s+2\fP could have been
*                passed in directly to set the colorspace attribute.
*
* !   \s-2\f(CWint    colorspace = KRGB;\s+2\fP
* ! 
* !   \s-2\f(CWkcolor_set_attributes(object, \s+2\fP
* !   \s-2\f(CW                      KCOLOR_COLORSPACE,    colorspace,\s+2\fP
* !   \s-2\f(CW                      KCOLOR_MAP_OPERATION, KROW_LEFT,\s+2\fP
* !   \s-2\f(CW                      NULL);\s+2\fP
*
*                A complete list of color attributes can be found in
*                Chapter 4 of Programming Services Volume II.
*
*         Input: object    - the data object into which the values of the
*                            attributes will be assigned
*
*                kvalist   - NULL terminated variable argument list
*                            which contains a list of attributes,
*                            each attribute followed by the values
*                            to assign to that attribute.  
* 
*      			     The variable argument list takes the form:
*
*              !  \s-2\f(CWATTRIBUTE_NAME1, value1 [, value2, ...],\s+2\fP
*              !  \s-2\f(CWATTRIBUTE_NAME2, value1,[, value2, ...],\s+2\fP
*              !  \s-2\f(CW..., NULL \s+2\fP
*
*                            The number of value arguments in the
*                            variable argument list for each attribute
*                            depends on the attribute. The NULL at the
*                            end of the variable argument list serves
*                            as a flag indicating the end of the list.
*
*			     Be careful not to forget the NULL at the
*			     end of the list.  This is a common programming
*			     error which unfortunately will not generate
*			     any compiler warnings. 
*        Output: none 
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: Calling this function with an incorrect number of arguments 
*                in the variable argument list will not cause any compiler 
*                errors, but will often generate a segmentation fault.
*    Written By: Steve Kubica
*          Date: May 03, 1994 14:40
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: int kcolor_set_attributes(
*		!   kobject  object,
*		!   kvalist)
*
*************************************************************/
int
kcolor_set_attributes(
   kobject object, 
   kvalist)
{
   kva_list list;
   int status;

   kcolor_init();

   kva_start(list, object);

   status = kdms_vset_attributes(object, KDMS_OBJECT, &list);

   kva_end(list);

   return (status);
}

/************************************************************
*
*  Routine Name: kcolor_match_attribute - compare a color attribute 
*					  between two data objects.
*
*       Purpose: This function is used to compare the value of a 
*		 color attribute between two data objects.
*
*		 This color service function should be used in
*		 conjunction with other application services such as
*		 polymorphic data services and geometry data services.
*		 This function will work on data objects opened or
*		 created with either of those services.
*
*		 If the value of the attribute in both objects is
*		 the same, then this function will return TRUE.
*		 If the values are different, then this function
*		 will return FALSE.
*		   
*                The following example illustrates the use of the match
*                attribute call to compare two different color
*                attributes.
*
* ! \s-2\f(CWif (kcolor_match_attribute(obj1, obj2, KCOLOR_COLORSPACE))\s+2\fP
* ! \s-2\f(CW   kprintf("colorspace is the same in both objects"); \s+2\fP
* ! \s-2\f(CWif (kcolor_match_attribute(obj1, obj2, KCOLOR_MAP_OPERATION))\s+2\fP
* ! \s-2\f(CW   kprintf("colormap operation is the same in both objects"); \s+2\fP
*
*                A complete list of color attributes can be found in
*                Chapter 4 of Programming Services Volume II.
*
*         Input: object1   - the first data object containing the
*		 	     attribute to be compared
*
*                object2   - the second data object containing the
*		 	     attribute to be compared
*
*		 attribute - the attribute to be compared
*
*        Output: none
*
*       Returns: TRUE (1) if the attribute matches, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Steve Kubica
*          Date: May 03, 1994 14:41
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: int kcolor_match_attribute(
*		!   kobject  object1,
*		!   kobject  object2,
*		!   char    *attribute)
*
*************************************************************/
int
kcolor_match_attribute(
   kobject object1, 
   kobject object2, 
   char   *attribute)
{
   kcolor_init();
   return (kdms_match_attribute(object1, object2, KDMS_OBJECT, attribute));
}

/************************************************************
*
*  Routine Name: kcolor_match_attributes - compare multiple attributes between 
*					   two objects.
*
*       Purpose: This function is used to compare the values of an 
*		 arbitrary number of color attributes between two
*		 data objects.
*
*		 This color service function should be used in
*		 conjunction with other application services such as
*		 polymorphic data services and geometry data services.
*		 This function will work on data objects opened or
*		 created with either of those services.
*
*		 If the value of all attributes in both objects are
*		 the same, then this function will return TRUE.
*		 If any of the values are different, then this function
*		 will return FALSE.
*
*                The following example illustrates the use of the match
*                attributes call to compare two different color
*                attributes.
*
* !   \s-2\f(CWif (kcolor_match_attributes(obj1, obj2, \s+2\fP
* !   \s-2\f(CW                            KCOLOR_COLORSPACE,   \s+2\fP
* !   \s-2\f(CW                            KCOLOR_MAP_OPERATION,\s+2\fP
* !   \s-2\f(CW                            NULL))\s+2\fP
* !   \s-2\f(CW   kprintf("colorspace and colormap operation are the same "\s+2\fP
* !   \s-2\f(CW           "in both objects");\s+2\fP
*
*                A complete list of color attributes can be found in
*                Chapter 4 of Programming Services Volume II.
*
*         Input: object1   - the first data object containing the
*		 	     attributes to be compared
*
*                object2   - the second data object containing the
*		 	     attributes to be compared
*
*                kvalist   - NULL terminated variable argument list
*                            which contains a list of attributes to 
*                            be compared.
*
*                            The variable argument list takes the form:
*
*                         !  \s-2\f(CWATTRIBUTE_NAME1,\s+2\fP
*                         !  \s-2\f(CWATTRIBUTE_NAME2,\s+2\fP
*                         !  \s-2\f(CW..., NULL \s+2\fP
*
*        Output: none
*
*       Returns: TRUE (1) if all listed attributes match, FALSE (0) otherwise
*
*  Restrictions: Calling this function and forgetting to NULL terminate
*		 the variable argument list will not cause any compiler 
*                errors, but will often generate a segmentation fault.
*    Written By: Steve Kubica
*          Date: May 03, 1994 14:41
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: int kcolor_match_attributes(
*		!   kobject  object1,
*		!   kobject  object2
*		!   kvalist)
*
*************************************************************/
int
kcolor_match_attributes(
   kobject object1, 
   kobject object2, 
   kvalist)
{
   kva_list list;
   int status;

   kcolor_init();

   kva_start(list, object2);

   status = kdms_vmatch_attributes(object1, object2, KDMS_OBJECT, &list);

   kva_end(list);

   return (status);
}

/************************************************************
*
*  Routine Name: kcolor_copy_attribute - copy a color attribute from 
*					 one data object to another.
*
*       Purpose: This function is used to copy the value of a
*		 color attribute from one data object to another.
*
*		 This color service function should be used in
*		 conjunction with other application services such as
*		 polymorphic data services and geometry data services.
*		 This function will work on data objects opened or
*		 created with either of those services.
*
*                The following example illustrates the use of the copy
*                attribute call to copy two different color
*                attributes.
*
* !   \s-2\f(CWkcolor_copy_attribute(obj1, ob2, KCOLOR_COLORSPACE);,\s+2\fP
* !   \s-2\f(CWkcolor_copy_attribute(obj1, obj2, KCOLOR_MAP_OPERATION);,\s+2\fP
*
*                A complete list of color attributes can be found in
*                Chapter 4 of Programming Services Volume II.
*
*         Input: object1   - the object to use as the source for the copy
*
*                object2   - the object to use as the destination for the copy
*
*		 attribute - the attribute to be compared
*
*        Output: none
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*
*    Written By: Steve Kubica
*          Date: Aug 26, 1994 12:11
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: int kcolor_copy_attribute(
*		!   kobject  object1,
*		!   kobject  object2,
*		!   char    *attribute)
*
*************************************************************/
int
kcolor_copy_attribute(
   kobject object1, 
   kobject object2, 
   char   *attribute)
{
   kcolor_init();
   return (kdms_copy_attribute(object1, object2, KDMS_OBJECT, attribute));
}

/************************************************************
*
*  Routine Name: kcolor_copy_attributes - copy multiple attributes from one
*                                         object to another.
*
*       Purpose: This function is used to copy the values of an 
*		 arbitrary number of color attributes from one
*		 data object to another.
*
*		 This color service function should be used in
*		 conjunction with other application services such as
*		 polymorphic data services and geometry data services.
*		 This function will work on data objects opened or
*		 created with either of those services.
*
*                The following example illustrates the use of the copy
*                attributes call to compare two different color
*                attributes.
*
* !   \s-2\f(CWkcolor_copy_attributes(obj1, obj2 \s+2\fP
* !   \s-2\f(CW                       KCOLOR_COLORSPACE,   \s+2\fP
* !   \s-2\f(CW                       KCOLOR_MAP_OPERATION,\s+2\fP
* !   \s-2\f(CW                       NULL);\s+2\fP
*
*                A complete list of color attributes can be found in
*                Chapter 4 of Programming Services Volume II.
*
*         Input: object1   - the object to use as the source for the copy
*
*                object2   - the object to use as the destination for the copy
*
*                kvalist   - NULL terminated variable argument list
*                            which contains a list of attributes to 
*                            be copied.
*
*                            The variable argument list takes the form:
*
*                         !  \s-2\f(CWATTRIBUTE_NAME1,\s+2\fP
*                         !  \s-2\f(CWATTRIBUTE_NAME2,\s+2\fP
*                         !  \s-2\f(CW..., NULL \s+2\fP
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: Calling this function and forgetting to NULL terminate
*		 the variable argument list will not cause any compiler 
*                errors, but will often generate a segmentation fault.
*
*    Written By: Steve Kubica
*          Date: Aug 26, 1994 12:11
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: int kcolor_copy_attributes(
*		!   kobject  object1,
*		!   kobject  object2
*		!   kvalist)
*
*************************************************************/
int
kcolor_copy_attributes(
   kobject object1, 
   kobject object2, 
   kvalist)
{
   kva_list list;
   int status;

   kcolor_init();

   kva_start(list, object2);

   status = kdms_vcopy_attributes(object1, object2, KDMS_OBJECT, &list);

   kva_end(list);

   return (status);
}

/************************************************************
*
*  Routine Name: kcolor_query_attribute - query characteristics of a 
*					  color attribute.
*
*       Purpose: This function is used to query characteristics of
*		 a color attribute from a data object.
*
*		 This color service function should be used in
*		 conjunction with other application services such as
*		 polymorphic data services and geometry data services.
*		 This function will work on data objects opened or
*		 created with either of those services.
*
*                The following example illustrates the use of the query
*                attribute call to determine the data type of the colorspace 
*                attribute.
*
* !   \s-2\f(CWint    data_type;\s+2\fP
* !
* !   \s-2\f(CWkcolor_query_attributes(object, KCOLOR_COLORSPACE,\s+2\fP
* !   \s-2\f(CW                        NULL, NULL, &data_type, NULL);\s+2\fP
*
*                A complete list of color attributes can be found in
*                Chapter 4 of Programming Services Volume II.
*
*         Input: object    - the data object to be queried for the
*			     existence of the named attribute
*
*		 attribute - the attribute to query
*
*        Output: num_args  - number of arguments
*		 arg_size  - size of the arguments, or NULL
*		 data_type - data type of the attribute
*		 permanent - TRUE if the attribute is stored
*			     with the object, FALSE if the
*			     attribute is transient
*
*       Returns: TRUE (1) if attribute exists, FALSE (0) otherwise
*
*  Restrictions: 
*
*    Written By: Steve Kubica
*          Date: Aug 26, 1994 12:10
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: int kcolor_query_attribute(
*		!   kobject  object,
*		!   char    *attribute,
*		!   int     *num_args,
*		!   int     *arg_size,
*		!   int     *data_type,
*		!   int     *permanent)
*
*************************************************************/
int
kcolor_query_attribute(
   kobject object,
   char   *attribute,
   int    *num_args,
   int    *arg_size,
   int    *data_type,
   int    *permanent)
{
   kcolor_init();
   return (kdms_query_attribute(object, NULL, attribute, num_args, arg_size,
				data_type, permanent));
}

/************************************************************
*
*  Routine Name: kcolor_print_attribute - print the value of a
*					  color attribute from a
*					  data object.
*
*       Purpose: This function is used to print the value of
*		 a color attribute from a data object to an output
*		 file.
*
*		 This color service function should be used in
*		 conjunction with other application services such as
*		 polymorphic data services and geometry data services.
*		 This function will work on data objects opened or
*		 created with either of those services.
*
*                The following example illustrates the use of the print
*                attribute call to print the colorspace attribute to
*		 the output file "outputfile".
*
* !   \s-2\f(CWkfile *outfile = kfopen("outputfile");\s+2\fP
* !
* !   \s-2\f(CWkcolor_print_attributes(object, KCOLOR_COLORSPACE, outfile);\s+2\fP
*
*                A complete list of color attributes can be found in
*                Chapter 4 of Programming Services Volume II.
*
*         Input: object    - the data object containing the attribute
*			     to be printed 
*
*		 attribute - the attribute to print
*
*		 printfile - a file or transport pointer opened
*			     with kfopen.   kstdout and kstderr 
*			     may be used to print to standard out
*			     and standard error.
*        Output: none
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*
*    Written By: Steve Kubica
*          Date: Aug 26, 1994 12:10
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: int kcolor_print_attribute(
*		!   kobject  object,
*		!   char    *attribute,
*		!   kfile   *printfile)
*
*************************************************************/
int
kcolor_print_attribute(
   kobject object,
   char   *attribute,
   kfile  *printfile)
{
   kcolor_init();
   return (kdms_print_attribute(object, NULL, attribute, printfile));
}
