 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>	        Automatic Position Incrmenting Routines
   >>>>            
   >>>>   Static:
   >>>>
   >>>>  Private:
   >>>>                 kaps_increment_position()
   >>>>                 kdut_advance_position()
   >>>>                 kaps_advance_segment_position()
   >>>>   Public:
   >>>>             
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/*-----------------------------------------------------------
|
|  Routine Name: kaps_increment_position
|
|       Purpose: This routine determines whether or not to 
|		 increment the position of each segment. 
|		 The routine also updates the increment state 
|		 of the presentation in question.
|
|         Input: a,b,c - three autoincrement state variables.
|		 rd    - TRUE if operation is a read, FALSE
|			 if operation is write
|
|        Output: a,b,c - updated in place
|
|       Returns: TRUE (1) if position should be incremented, 
|		 FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Jan 02, 1993 15:46
| Modifications:
|
------------------------------------------------------------*/
int
kaps_increment_position(int *a, int *b, int *c, int rd)
{
   char ta;
   char tb;
   char q;


   /*
    * execute one cycle of the state machine
    */
   q = (*a && !*b) || (!rd && *c) || (rd && *b && !*c);
   ta = (rd && *a) || (rd && *c) || (*b && !*c);
   tb = (rd && !*c) || (*a && !*b) || (!rd && *c);

   /*
    * update the state
    */
   *a = (int) ta;
   *b = (int) tb;
   *c = !rd;

   /*
    * return whether or not to increment
    */
   return ((int) q);
}

/*-----------------------------------------------------------
|
|  Routine Name: kdut_advance_position
|
|       Purpose: This routine advances a given position.
|
|		 The position will be incremented by the
|		 given increment unit.  It will optionally
|		 either stop at the end of the data set,
|		 or wrap back around to the beginning of
|		 the data set.
|
|         Input: begin - current begin point
|		 end   - current end point
|		 size  - size of data set
|		 unit  - unit size to increment by
|                dim   - the dimension of the position
|                wrap  - TRUE if position should wrap back 
|                        to the beginning
|
|        Output: begin and end are advanced in place
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Oct 07, 1993 13:18
| Modifications:
|
------------------------------------------------------------*/
int 
kdut_advance_position(
   int *begin, 
   int *end, 
   int *size, 
   int *unit, 
   int  dim, 
   int  wrap)
{
   int i;


   /*
    * increment to the beginning of the next unit of data... also
    * better check to see if we need to advance the pointer throughout
    * the data segment.  If we are at the end of plane then we need to
    * advance the data segment depth pointer, etc.  This scenario in
    * propagated all the way down to the element number. Where at last
    * the "eof" indicator is set.
    */

   for (i = 0; i < dim - 1; i++)
   {
      begin[i] += unit[i];
      end[i]   += unit[i];

      if (begin[i] < size[i])
	 return TRUE;
      else
      {
         begin[i] = 0;
	 end[i]   = unit[i] - 1;
      }
   }

   begin[dim - 1] += unit[dim - 1];
   end[dim - 1]   += unit[dim - 1];

   if (begin[dim - 1] >= size[dim - 1])
   {
      if (wrap)
      {
	 begin[dim - 1] = 0;
	 end[dim - 1]   = unit[i] - 1;
      }
      else
	 return FALSE;
   }

   return TRUE;
}

/*-----------------------------------------------------------
|
|  Routine Name: kaps_advance_segment_position
|
|       Purpose: This routine will auto-increment the position
|		 within a given segment.  The increment state
|		 and the position within the segment will be
|		 updated.  The amount of the increment is
|		 determined from the begin and end points
|		 of the region being accessed.
|			
|         Input: object  - object containing segment 
|		 segment - data segment whose position we will be
|			   incrementing
|
|		 begin   - beginning point of region being accessed
|		 end     - ending point of region being accessed
|
|		 The begin and end points will determine a region
|		 size - this will be the amount by which the position
|		 is incremented.
|
|                size    - the size of the data segment
|
|		 get_operation  - TRUE if operation is a read, FALSE
|			 	  if operation is write
|
|        Output: position and incrment state of object is updated
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica and Jeremy Worley
|          Date: Oct 07, 1993 13:18
| Modifications:
|
------------------------------------------------------------*/
int
kaps_advance_segment_position(
   kobject object,
   char   *segment,
   int    *begin,
   int    *end,
   int    *size,
   int     get_operation)
{
   int  a = 0;
   int  b = 0;
   int  c = 0;
   int  i;
   int  dim;
   int *p = NULL;
   int  interp;
   int  unit[KDMS_MAX_DIM] = {0, 0, 0, 0, 0};
   int  def_incsize[KDMS_MAX_DIM] = {0, 0, 0, 0, 0};
   int  newsize[KDMS_MAX_DIM];
   int  *incsize;
   int   e[KDMS_MAX_DIM] =  {0, 0, 0, 0, 0};  /* dummy array */
   
   /* get some important attributes */
   dim = kaps_get_segment_dimension(object, segment);
   p   = kaps_get_segment_position(object, segment);
   kaps_get_segment_inc_state(object, segment, &a, &b, &c);

   kaps_get_attributes(object, segment,
		       KDMS_INTERPOLATE, &interp, NULL);

   /* determine the unit size */
   for (i = 0; i < dim; i++)
      unit[i] = end[i] - begin[i] + 1;

   /* increment the position */
   if (kaps_increment_position(&a, &b, &c, get_operation))
   {
      /* -- only use this if it is a real physical segment -- */
      if (kdms_query_segment(object, segment))
	  kdms_get_attribute(object, segment, 
			     _INTERNAL_INCREMENT_SIZE, &incsize);
      else
	 incsize = &def_incsize[0];
      
      for (i = 0; i < dim; i++)
	 newsize[i] = incsize[i] > 0 ? incsize[i] : size[i];
      
      kdut_advance_position(p, e, newsize, unit, dim, interp != KNONE);
      
      kaps_set_segment_position(object, segment, p);
   }
   kaps_set_segment_inc_state(object, segment, a, b, c);
   return TRUE;
}
