#! /bin/sh

# kshot geenrates spikes at random points in the input object. Due to its
# random nature it is not possible to compare kshot output with an already
# stored object. The approach in this testsuite inserts a spike value that
# is greater than the maximum value of the existing object. The stats of
# the output object are then checked to see the maxval. For the case when 
# a mask segment is present, the problem gets trickier 'cos the random
# position geenrator for spike insertion might point to a masked location in
# which kshot does not spike the value. Due to this problem, when testing 
# the routine for objects in which the mask segment is present a sufficiently
# large percentage of spikes is requested to ensure that a non-masked value
# is spiked. There is always the chance that the random [position generator
# will generate all masked locations in which case the testsuite will fail.
# However, this occurence should be very rare.

echo "running test1.sh for kshot ... process ($$)"


# ------------------------------------------------------------------------
# report test_num test_obj cntrl_obj test_descr
#
# The report function uses ksegcmp to compare the test output data object
# ($2) against the correct result ($3), and reports success or failure of
# the test.  If the result of ksegcmp is false, status is set to 1.
# ------------------------------------------------------------------------
report()
{
test_num=$1
test_obj="$2"
cntrl_obj="$3"
test_descr="$4"

ksegcmp -i1 "$test_obj" -i2 "$cntrl_obj" -tol 1e-5 -sat 1 -s -rt 1 -all
if [ $? = 0 ]
then
   echo "Passed:  Test  $test_num --> "$test_descr""
else
   echo "FAILED:  Test  $test_num --> "$test_descr""
   status=1
fi
}

#
#  Initialize status to be success (ie. 0).  Upon failure, status is
#  initialized to 1.
#
status=0


# ------------------------------------------------------------------------
#  set path for test data, and path for temporary directory
# ------------------------------------------------------------------------
tbpath=`kecho -tb DATAMANIP -echo path`
dpath="$tbpath/data/testdata"
tmpdir=${TMPDIR:-/usr/tmp}

# ------------------------------------------------------------------------
# Test  1:  Single point float object
# ------------------------------------------------------------------------
test_num=1
descr="Single point float object"
kshot -o $tmpdir/out.$$ -i $dpath/1pt.float.xv -spikes 100 -real 123 -imag 0

kstats -maxval -i $tmpdir/out.$$  -o  $tmpdir/stats.$$
cat <<EOF > $tmpdir/ascii.$$
123
EOF
kasc2val -i1 $tmpdir/ascii.$$ -type 11 -o $tmpdir/answer.$$

report $test_num "$tmpdir/stats.$$" "$tmpdir/answer.$$" "$descr"

/bin/rm -f $tmpdir/out.$$ $tmpdir/ascii.$$ $tmpdir/answer.$$ $tmpdir/stats.$$

# ------------------------------------------------------------------------
# Test  2:  Byte xvimage
# ------------------------------------------------------------------------
test_num=2
descr="Shot noise in Byte xvimage"
kshot -o $tmpdir/out.$$ -spikes 20 -real 26 -imag 0 -i $dpath/3x4.byte.xv

set `kstats -maxval -i $tmpdir/out.$$ | tail +2`
max=$2

if [ \( "$max" -eq "26" \) ]; then
        echo "Passed:  Test  $test_num --> "$descr""
else
        echo "FAILED:  Test  $test_num --> "$descr""
        status=1
fi

/bin/rm -f $tmpdir/out.$$

# ------------------------------------------------------------------------
# Test  3: Double xvimage 
# ------------------------------------------------------------------------
test_num=3
descr="Shot noise in double xvimage"
kshot -o $tmpdir/out.$$ -i $dpath/3x4.double.viff -spikes 20 -real 255  -imag 0

set `kstats -maxval -i $tmpdir/out.$$ | tail +2`
max=$2

if [ \( "$max" -eq "255" \) ]; then
        echo "Passed:  Test  $test_num --> "$descr""
else
        echo "FAILED:  Test  $test_num --> "$descr""
        status=1
fi

/bin/rm -f $tmpdir/out.$$

# ------------------------------------------------------------------------
# Test  4: Value and map data
# ------------------------------------------------------------------------
test_num=4
descr="Shot noise in object containing value and map data"
kshot -o $tmpdir/out.$$  -i $dpath/2x3ival_6x3fmap.viff -spikes 20 -real 62 -imag 0

set `kstats -maxval -i $tmpdir/out.$$ | tail +2`
max=$2

if [ \( "$max" -eq "62" \) ]; then
        echo "Passed:  Test  $test_num --> "$descr""
else
        echo "FAILED:  Test  $test_num --> "$descr""
        status=1
fi

/bin/rm -f $tmpdir/out.$$

# ------------------------------------------------------------------------
# Test  5:  Value, map & mask
# ------------------------------------------------------------------------
test_num=5
descr="Shot noise in object containing value, map and mask data"
kshot -o $tmpdir/out.$$ -i $dpath/2x3ival_6x3fmap_2x3bmask.viff -spikes 75 -real 987 -imag 0

set `kstats -maxval -i $tmpdir/out.$$ | tail +2`
max=$2

if [ \( "$max" -eq "987" \) ]; then
        echo "Passed:  Test  $test_num --> "$descr""
else
        echo "FAILED:  Test  $test_num --> "$descr""
        status=1
fi

/bin/rm -f $tmpdir/out.$$

# ------------------------------------------------------------------------
# Test  6:  Value and mask data
# ------------------------------------------------------------------------
test_num=6
descr="Shot noise in object containing value and mask data"
kshot -o $tmpdir/out.$$  -i $dpath/3x4masked.double.viff -spikes 100 -imag 0 -real 444

set `kstats -maxval -i $tmpdir/out.$$ | tail +2`
max=$2

if [ \( "$max" -eq "444" \) ]; then
        echo "Passed:  Test  $test_num --> "$descr""
else
        echo "FAILED:  Test  $test_num --> "$descr""
        status=1
fi

/bin/rm -f $tmpdir/out.$$

exit $status
