 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for kshot
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 	lkshot
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/* -library_includes */
/* -library_includes_end */


/****************************************************************
* 
*  Routine Name: lkshot - corrupt data uniformly with spikes
* 
*       Purpose: Library routine for kshot. lkshot corrupts the input
*                data object uniformly with spikes. The value of the 
*                spike is specified by real_val and imag_val, which 
*                represent the real and imaginary component of the spike.
*                Imaginary value is used only if spikes are to be inserted  
*                in a complex object. Ratio specifies the percentage of
*                points in the input objects to be corrupted with shot
*                noise. The location of points to be corrupted by spikes
*                is generated by a uniform random number generator. If the
*                mask value at a point to be corrupted by a spike is 0, the
*                spike is not inserted at that point but it is counted in the
*                number of points corrupted.
*
*         Input: src_obj   - Source data object
*                ratio     - Percentage of spikes in the image
*                real_val  - Real value of spike
*                imag_val  - Imaginary value of spike
*
*        Output: dst_obj   - Destination object containing spiked data
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Ashish Malhotra
*          Date: Apr 08, 1995
*      Verified: 
*  Side Effects: Destroys the map segment on the destination object
*                if the source object contains both map and value data.
*                Also the size and datatype of the value segment is
*                set to the size and datatype of the the mapped
*                source object.
* Modifications: 
****************************************************************/
/* -library_def */
int lkshot(
   kobject src_obj,
   double  ratio,
   double  real_val,
   double  imag_val,
   kobject dst_obj)
/* -library_def_end */

/* -library_code */
{
	char *lib = "kdatamanip";       /* library name (for error reporting) */
        char *rtn = "lkshot";           /* routine name (for error reporting) */
        int  type;
        int  j;                     /* loop counters */
        int   error = FALSE;
        int   w,h,d,t,e;                /* object size */
        int   w_pos, h_pos, d_pos,
              t_pos, e_pos;
        int   obj_size, num_pts;
        int   map_exists,               /* TRUE if object has MAP data */
              val_exists,               /* TRUE if object has VALUE data */
              mask_exists;              /* TRUE if object has MASK data */
        double *rand_data = NULL;       /*                              */
        double dbl_data;
        kdcomplex cmplx_data;
        unsigned char *mask = NULL;
        char *prim;
        char *info;
        char *datasize;
        char *datatype;
        char *datapos;

        map_exists  = kpds_query_map(src_obj);
        val_exists  = kpds_query_value(src_obj);
        mask_exists = kpds_query_mask(src_obj);

        if (!map_exists && !val_exists)
        {
           kerror(lib, rtn,
                  "Input data object must have either Value or Map data.");
           return(FALSE);
        }

        if (map_exists && val_exists)
        {
           map_exists = FALSE;
           kpds_set_attribute(src_obj,  KPDS_MAPPING_MODE, TRUE);
           if (kpds_query_map(dst_obj))
           {
              kpds_destroy_map(dst_obj);
              kpds_get_attributes(src_obj,KPDS_VALUE_DATA_TYPE, &type,
                                  KPDS_VALUE_SIZE, &w, &h, &d, &t, &e, NULL);
              kpds_set_attributes(dst_obj,KPDS_VALUE_DATA_TYPE, type,
                                  KPDS_VALUE_SIZE, w, h, d, t, e, NULL);
           }
        }

        if ((dst_obj = kpds_reference_object(dst_obj)) == KOBJECT_INVALID)
        {
           kerror(lib, rtn, "Failed to duplicate output object");
           return(FALSE);
        }

        if (mask_exists)
        {
           if ((src_obj = kpds_reference_object(src_obj)) == KOBJECT_INVALID)
           {
              kpds_close_object(dst_obj);
              kerror(lib, rtn, "Failed to duplicate input object");
              return(FALSE);
           }
           kpds_set_attribute(src_obj,KPDS_MASK_DATA_TYPE, KUBYTE);
        }

        if (map_exists)
        {
           prim = KPDS_MAP_POINT;
           info = KPDS_MAP_LINE_INFO;
           datasize = KPDS_MAP_SIZE;
           datatype = KPDS_MAP_DATA_TYPE;
           datapos  = KPDS_MAP_POSITION;
        }
        else
        {
           prim = KPDS_VALUE_POINT;
           info = KPDS_VALUE_LINE_INFO;
           datasize = KPDS_VALUE_SIZE;
           datatype = KPDS_VALUE_DATA_TYPE;
           datapos  = KPDS_VALUE_POSITION;
        }

        kpds_get_attributes(src_obj,  datasize, &w, &h, &d, &t, &e,
                            datatype, &type, NULL);

        if ( (type == KCOMPLEX) || (type == KDCOMPLEX))
           type = KDCOMPLEX;
        else
           type = KDOUBLE;

        kpds_set_attribute(dst_obj, datatype, type);

        if (type == KDCOMPLEX)
           cmplx_data = kdccomp(real_val, imag_val);
        else
           dbl_data = real_val; 

       /*
        * Find the total number of points in the object and 
        * determine the number to be corrupted according to the 
        * ratio.
        */
        obj_size = w*h*d*t*e;
        num_pts = (int) ((ratio/100.0) * obj_size);

        rand_data = (double *)kmalloc(sizeof(double) * num_pts);
      
        /* Generate uniform random numbers between 0 and
         * number of points in the object.
         */
        kgen_unif(num_pts, 0.0 ,((double)(obj_size-1)),rand_data);
 
        for (j=0; j< num_pts; j++)
        {
          /*
           * Figure out the coordinates in the object from the random number.
           */
           w_pos = ((int) rand_data[j]) % w; 
           h_pos = (((int) rand_data[j])/w) % h;
           d_pos = (((int) rand_data[j])/(w*h)) %d;
           t_pos = (((int) rand_data[j])/(w*h*d)) %t;
           e_pos = (((int) rand_data[j])/(w*h*d*t)) %e;

           kpds_set_attribute(dst_obj, datapos, w_pos, h_pos,
                              d_pos, t_pos, e_pos);
           
           if (mask_exists)
           {
              kpds_set_attribute(src_obj, KPDS_MASK_POSITION, w_pos, h_pos,
                                 d_pos, t_pos, e_pos);
              mask = (unsigned char *) kpds_get_data(src_obj, KPDS_MASK_POINT,
                        (kaddr) mask);
           }
           
           if (!mask_exists || (mask[0] != 0) )
           {
              if (type == KDCOMPLEX)
              {
                 if (!kpds_put_data(dst_obj, prim,(kaddr) &cmplx_data))
                 {
                    error = TRUE;
                    kerror(lib,rtn,"Put data failed.");
                    break;
                 }
              }
              else
              {
                 if (!kpds_put_data(dst_obj, prim,(kaddr) &dbl_data))
                 {
                    error = TRUE;
                    kerror(lib,rtn,"Put data failed.");    
                    break;
                 }
              }
           }
        }

        if (mask_exists) kpds_close_object(src_obj);
        kpds_close_object(dst_obj);

        if (mask) kfree(mask);
        if (rand_data)  kfree(rand_data);

        if (error)
           return FALSE;
        else
	   return TRUE;
}
/* -library_code_end */
