 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for knormal
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 	lknormal
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/* -library_includes */
#include "internals.h"
/* -library_includes_end */


/****************************************************************
* 
*  Routine Name: lknormal - normalize regions using minimum & maximum of each region
* 
*       Purpose: lknormal normalizes the data within regions 
*		 to the lower_norm and upper_norm parameters.  
*		 Regions are defined by TRUE settings of w_flag, 
*		 h_flag, d_flag, t_flag, e_flag, and whole_flag.
*		 The minimum and maximum values in each region are 
*		 used to define the normalization over that region.
*		 If whole_flag is TRUE, the normalization region will
*		 be the entire data set, and w_flag, h_flag, d_flag, 
*		 t_flag, and e_flag settings are overridden.  If none 
*		 of the flags are true, the default is whole_flag.
*
*		 If the apriori flag is TRUE, the data will not be
*		 searched for minimum and maximum values but instead,
*		 the data_min and data_max parameters provided will 
*		 be used.  This eliminates a required pass through the
*		 data before normalization is performed.
*
*		 The data type of the output object is specified by
*		 the data_type_string parameter, and should be one of
*		 the following values:
*
*                .TS H
*                center tab(:) ;
*                lfB  lfB  lfB
*                l    l    lf(CW) .
*		 Data Type:Abbreviation:Data Type
*                =
*                .TH
*		 bit              :bi     : KBIT
*		 byte             :by     : KBYTE
*		 unsigned byte    :un by  : KUBYTE
*		 ubyte            :uby    : KUBYTE
*		 short            :sh     : KSHORT
*		 unsigned short   :un sh  : KUSHORT
*		 ushort           :ush    : KUSHORT
*		 integer          :in     : KINT
*		 unsigned integer :un in  : KUINT
*		 uint             :ui     : KUINT
*		 long             :lon    : KLONG
*		 unsigned long    :un lon : KULONG
*		 ulong            :ul     : KULONG
*		 float            :fl     : KFLOAT
*		 double           :do     : KDOUBLE
*		 complex          :co     : KCOMPLEX
*		 double complex   :do co  : KDCOMPLEX
*		 dcomplex         :dc     : KDCOMPLEX
*		 .T&
*		 l s s.
*		 Propagate Input Type - will not change the data type
*                .TE
*
*		 If the source object (src_obj) contains value data
*		 and no map, normalization is performed on the value 
*		 data.  Likewise, if the source object contains map
*		 data and no value data, normalization is performed 
*		 on the map.
*
*		 If the source object has both map and value data, 
*		 certain region conditions must be met for lknormal
*		 to succeed.
*		 !  1.  The first condition is that the normalization
*		 !      region must be defined by at least width and 
*		 !      height.
*		 !  2.  If the map depth, time, and elements dimensions
*		 !      are all one, the normalization region must be
*		 !      the entire data set.
*		 !  3.  If the map depth, time, or elements dimensions
*		 !      are greater than 1, the corresponding dimension
*		 !      does not have to be part of the normalization
*		 !      region.
*
*		 Masks: Currently, the validity mask is only transferred
*		 from the source to the destination.  However, if there 
*		 is a mask, the KPDS_MASKED_VALUE_PRESENTATION is used by 
*		 kpds.  As a future enhancement, invalid data will not be
*		 figured into the normalization calculations.
*
*		 lknormal ignores location, and time data.  
*
*         Input: *		 src_obj - the input data object to be normalized
*		 lower_norm - the upper normalization value		 
*		 upper_norm - the upper normalization value
*		 apriori_flag - if TRUE, uses specified data_min and data_max
*		 data_min - predetermined minimum data value
*		 data_max - predetermined maximum data value
*		 w_flag - if TRUE, include width in normalization unit
*		 h_flag - if TRUE, include height in normalization unit
*		 d_flag - if TRUE, include depth in normalization unit
*		 t_flag - if TRUE, include time in normalization unit
*		 e_flag - if TRUE, include elements in normalization unit
*		 whole_flag - if TRUE, normalize whole data set at one time
*                datatype - the data type of the output data object
*
*        Output: dst_obj - the output data object 
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: The validity mask is simply transferred from the 
*                source to the destination.  Invalid data will be
*                included in the normalization calculations.
*
*		 Does not support complex input types yet.
*
*    Written By: Donna Koechner
*          Date: Apr 08, 1995
*      Verified: 
*  Side Effects: 
* Modifications: 
****************************************************************/
/* -library_def */
int lknormal(
   kobject src_obj,
   double  norm_lower,
   double  norm_upper,
   int	   apriori_flag,
   double  data_min,
   double  data_max,
   int     w_flag,
   int     h_flag,
   int     d_flag,
   int     t_flag,
   int     e_flag,
   int     whole_flag,
   char    *data_type_string,
   kobject dst_obj)
/* -library_def_end */

/* -library_code */
{
	kaddr   data = NULL;
	double  *ddata=NULL;
	double  dmin, dmax, dfactor;
	int     dst_type, src_type;
	int	ws=1, hs=1, ds=1, ts=1, es=1;
	int	wp=0, hp=0, dp=0, tp=0, ep=0;
	int	reg_w=1, reg_h=1, reg_d=1, reg_t=1, reg_e=1;
	int	norm_rgns=1, num_subrgns=1, num_elems=1;
	int	i, j, k;
        char    *opt_region, *region_size, *region_prim;
        char    *data_size, *data_type;
        char    *position;
	int     has_value=FALSE, has_map=FALSE;
	char    *lib="kdatamanip", *prog="knormal";

	if (norm_upper < norm_lower)
        {
           kerror(lib, prog, 
		 "upper value (%g) must be greater than lower value (%g)",
		 norm_upper, norm_lower);
           return(FALSE);
        }
	
	if (apriori_flag && (data_min > data_max))
        {
           kerror(lib, prog, 
		 "data maximum (%g) must be greater than data minimum (%g)",
		 data_max, data_min);
           return(FALSE);
        }
	
	/* make sure that the objects passed in are valid */
	if ((src_obj == KOBJECT_INVALID) || (dst_obj == KOBJECT_INVALID))
        {
           kerror(lib, prog, "invalid object passed into library routine");
           return(FALSE);
        }
	
	/* if all of the dimension flags are false, normalize whole data set */
	if ((w_flag + h_flag + d_flag + t_flag + e_flag + whole_flag) == 0)
	   whole_flag = TRUE;

	/* source object must have value and/or map data */
	has_value = kpds_query_value(src_obj);
	has_map = kpds_query_map(src_obj);
	if ((has_value == FALSE) && (has_map == FALSE))
        {
           kerror(lib, prog, "source object must contain map or value data");
           return(FALSE);
        }
	
	/* reference the source object so the lknormal has no side effects */
        if ((src_obj = kpds_reference_object(src_obj)) == KOBJECT_INVALID)
        {
           kerror(lib, prog, "failed to reference source object");
           return(FALSE);
        }

	/* if there is a map, find out if we can process it */
	if (has_map)
	{
	   /* if there is value data, certain conditions must be met */
           if (has_value)
	   {
              if (!whole_flag && (!w_flag && !h_flag))
	      {
                 kerror(lib, prog, "Since input object has map data, %s",
		 "normalization region must contain Width and Height.");
                 kpds_close_object(src_obj);
                 return(FALSE);
	      }
   
	      if (!kpds_get_attribute(src_obj,
			KPDS_MAP_SIZE, &ws,&hs,&ds,&ts,&es))
	      {
                 kerror(lib, prog, "unable to get src_obj map size");
                 kpds_close_object(src_obj);
                 return(FALSE);
	      }
	      if (!whole_flag && (!d_flag && !t_flag && !e_flag))
	      {
	         if (ds==1 && !d_flag) 
	         {
                    kerror(lib, prog, 
		    "Map depth = 1: normalization region must contain Depth");
                    kpds_close_object(src_obj);
                    return(FALSE);
	         }
	         if (ts==1 && !t_flag) 
	         {
                    kerror(lib, prog, 
		    "Map time = 1: normalization region must contain Time");
                    kpds_close_object(src_obj);
                    return(FALSE);
	         }
	         if (es==1 && !e_flag)
	         {
                    kerror(lib, prog, "Map elements = 1: %s",
		    "normalization region must contain Elements");
                    kpds_close_object(src_obj);
                    return(FALSE);
	         }
	      }
	   }
	   /* if we managed to get this far, can directly process map data */
           region_prim  = KPDS_MAP_REGION;
           opt_region   = KPDS_MAP_OPTIMAL_REGION_SIZE;
           region_size  = KPDS_MAP_REGION_SIZE;
           data_size    = KPDS_MAP_SIZE;
           data_type    = KPDS_MAP_DATA_TYPE;
           position     = KPDS_MAP_POSITION;

	   /* create destination map segment if it does not exist */
	   if (!kpds_query_map(dst_obj))
           {
	      if (!kpds_create_map(dst_obj))
              {
	         kerror(lib,prog,"failed to create map for destination object");
                 kpds_close_object(src_obj);
                 return(FALSE);
	      }
           }

	   /* create destination value segment if source has value and 
	    * the destination does not */
	   if (has_value && (!kpds_query_value(dst_obj)))
           {
	      if (!kpds_create_value(dst_obj))
              {
	         kerror(lib,prog,"failed to create value for destination");
                 kpds_close_object(src_obj);
                 return(FALSE);
	      }
	      if (!kpds_get_attributes(src_obj, 
				KPDS_VALUE_SIZE, &ws, &hs, &ds, &ts, &es,
				KPDS_VALUE_DATA_TYPE, &src_type, NULL))
              {
	         kerror(lib,prog,"failed to get source value attributes");
                 kpds_close_object(src_obj);
                 return(FALSE);
	      }
	      if (!kpds_set_attributes(dst_obj, 
				KPDS_VALUE_SIZE, ws, hs, ds, ts, es,
				KPDS_VALUE_DATA_TYPE, src_type, NULL))
              {
	         kerror(lib,prog,"failed to get source value attributes");
                 kpds_close_object(src_obj);
                 return(FALSE);
	      }
           }
        }

	/* if no map, set up the value attributes */
        else
        {
           region_prim = KPDS_VALUE_REGION;
           opt_region  = KPDS_VALUE_OPTIMAL_REGION_SIZE;
           region_size = KPDS_VALUE_REGION_SIZE;
           data_size   = KPDS_VALUE_SIZE;
           data_type   = KPDS_VALUE_DATA_TYPE;
           position    = KPDS_VALUE_POSITION;

	   /* create destination value segment if it doesn't exist */
	   if (!kpds_query_value(dst_obj))
           {
	      if (!kpds_create_value(dst_obj))
              {
	         kerror(lib,prog,"failed to create destination value");
                 kpds_close_object(src_obj);
                 return(FALSE);
	      }
           }
   	   /* destoy destination map segment if it exists */
	   if (kpds_query_map(dst_obj) && (!kpds_destroy_map(dst_obj)))
           {
	      kerror(lib,prog,"failed to destroy destination map");
              kpds_close_object(src_obj);
              return(FALSE);
	   }
        }
	
	/* get the size and type of the source data */
	if (!kpds_get_attributes(src_obj, 
			data_size, &ws, &hs, &ds, &ts, &es,
			data_type, &src_type, NULL))
        {
	   kerror(lib,prog,"unable to get source object attributes");
           kpds_close_object(src_obj);
	   return(FALSE);
	}

	if (ws*hs*ds*ts*es < 2)
        {
	   kerror(lib,prog,"Input object must have more than 1 point");
           kpds_close_object(src_obj);
	   return(FALSE);
	}
	
	if ((src_type == KCOMPLEX) || (src_type == KDCOMPLEX))
	{
	   kerror(lib,prog,"Complex normalization not supported yet.");
           kpds_close_object(src_obj); kpds_close_object(dst_obj);
           return(FALSE);
	}

        /* get the destination data type from the data_type_string */
	if (kstrcasecmp(data_type_string, "Propagate Input Type") == 0)
	   dst_type = src_type;
        else if ((dst_type = kdatatype_to_define(data_type_string)) == 0)
	{
	   kerror(lib, prog, "Not a proper data type '%s'.", data_type_string);
	   return (FALSE);
	}
 
	/* set specified data type and size on destination object */
	if (!kpds_set_attributes(dst_obj, 
			data_size, ws, hs, ds, ts, es,
			data_type, dst_type, NULL))
        {
	   kerror(lib,prog,"unable to set attributes on destination object");
           kpds_close_object(src_obj);
	   return(FALSE);
	}

	/* reference destination object */
	if ((dst_obj = kpds_reference_object(dst_obj)) == KOBJECT_INVALID)
        {
	   kerror(lib,prog,"unable to reference destination object");
           kpds_close_object(src_obj);
	   return(FALSE);
	}

	/* set processing type to double or dcomplex, since the normalization
	   always required multiply by floating point value  */
/*
	src_type = kdatatype_cast_process(src_type, dst_type,
                        (KDOUBLE | KDCOMPLEX));
*/
src_type = KDOUBLE;
	
	/* set processing type on source object reference */
	if (!kpds_set_attribute(src_obj, data_type, src_type))
	{
	   kerror(lib,prog,"unable to set data type on source reference");
           kpds_close_object(src_obj); kpds_close_object(dst_obj);
           return(FALSE);
        }

	/* Figure out what the optimal region size is that will still let us
   	   process by the unit we want */

	norm_rgns = 1;
	if (!whole_flag)
	{
	   /* temporarily set constraints on dimensions if regions specified */
	   norm_rgns *= (w_flag) ?   1 : ws;
	   reg_w      = (w_flag) ?  ws :  1;
 
	   norm_rgns *= (h_flag) ?   1 : hs;
	   reg_h      = (h_flag) ?  hs :  1;
 
	   norm_rgns *= (d_flag) ?   1 : ds;
	   reg_d      = (d_flag) ?  ds :  1;
 
	   norm_rgns *= (t_flag) ?   1 : ts;
	   reg_t      = (t_flag) ?  ts :  1;
 
	   norm_rgns *= (e_flag) ?   1 : es;
	   reg_e      = (e_flag) ?  es :  1;
 
	   if (!kpds_set_attribute(src_obj, data_size,
				reg_w, reg_h, reg_d, reg_t, reg_e))
	   {
	      kerror(lib,prog,"unable to set size of source reference");
              kpds_close_object(src_obj); kpds_close_object(dst_obj);
              return(FALSE);
	   }
	}

	if (!kpds_get_attribute(src_obj, opt_region, 
		&reg_w, &reg_h, &reg_d, &reg_t, &reg_e, &num_subrgns))
	{
	   kerror(lib,prog,"unable to get source object optimal region size");
           kpds_close_object(src_obj); kpds_close_object(dst_obj);
           return(FALSE);
	}

	num_elems = reg_w * reg_h * reg_d * reg_t * reg_e;

	if (!whole_flag)
	{
	   /* set the source object back to its original size */
	   if (!kpds_set_attribute(src_obj, data_size, ws, hs, ds, ts, es))
	   {
	      kerror(lib,prog,"unable to set size of source reference");
              kpds_close_object(src_obj); kpds_close_object(dst_obj);
              return(FALSE);
	   }
	}

	/* set the region size on the source and destination objects */
	if (!kpds_set_attribute(src_obj, 
			region_size, reg_w, reg_h, reg_d, reg_t, reg_e))
	{
	   kerror(lib,prog,"unable to set source object region size");
           kpds_close_object(src_obj); kpds_close_object(dst_obj);
           return(FALSE);
	}
	if (!kpds_set_attributes(dst_obj, 
			region_size, reg_w, reg_h, reg_d, reg_t, reg_e,
			data_type, src_type,
			position, 0,0,0,0,0, NULL))
	{
	   kerror(lib,prog,"unable to set destination object attributes");
           kpds_close_object(src_obj); kpds_close_object(dst_obj);
           return(FALSE);
	}

	   for (i=0; i<norm_rgns; i++)
	   {
	      if (!apriori_flag)
	      {
	         /* first get min and max values in normalization region */
	         /* get first region separate, to correctly set min and max */
	         if ((ddata=kpds_get_data(src_obj,region_prim,ddata)) == NULL)
	         {
	            kerror(lib,prog,
			   "get region failed (norm_rgn %d, subrgn 0)",i);
                    kpds_close_object(src_obj); kpds_close_object(dst_obj);
                    return(FALSE);
	         }
	         dmin = dmax = ddata[0];
	         for (k=0; k<num_elems; k++)
	         {
	            dmin=kmin(dmin, ddata[k]);
	            dmax=kmax(dmax, ddata[k]);
	         }
   
	         /* record the source region position for resetting later 
	            need to do this after the first get because data 
	            preincrements */
	         if (!kpds_get_attribute(src_obj,position,
				      &wp,&hp,&dp,&tp,&ep))
	         {
	            kerror(lib,prog, "Unable to get source position");
                    kpds_close_object(src_obj); kpds_close_object(dst_obj);
                    return(FALSE);
	         }
		      
	         for (j=1; j<num_subrgns; j++)
	         {
	            if ((ddata=kpds_get_data(src_obj,region_prim,ddata))==NULL)
	            {
	               kerror(lib,prog,
			  "failed to get region (norm_rgn %d, subrgn %d)",i,j);
                       kpds_close_object(src_obj); kpds_close_object(dst_obj);
                       return(FALSE);
	            }
	            for (k=0; k<num_elems; k++)
	            {
	               dmin=kmin(dmin, ddata[k]);
	               dmax=kmax(dmax, ddata[k]);
	            }
	         }
	        
	         /* because get_data auto increments, need to set position 
	            on source back to region origin */
	         if (!kpds_set_attribute(src_obj, position, wp,hp,dp,tp,ep))
	         {
	            kerror(lib,prog, "Unable to set source position");
                    kpds_close_object(src_obj); kpds_close_object(dst_obj);
                    return(FALSE);
	         }
   	      }
	      else 
	      {
	         dmin = data_min;
	         dmax = data_max;
	      }

	      /* then normalize data in normalization region */
	      if (dmax == dmin)
	      {
	         kerror(lib,prog, "data minimum = data maximum = %d on region %d, so normalization factor cannot be computed (divide by zero)",norm_rgns,dmax);
                 kpds_close_object(src_obj); kpds_close_object(dst_obj);
                 return(FALSE);
	      }
	      dfactor = (norm_upper - norm_lower) / (dmax - dmin);

	      for (j=0; j<num_subrgns; j++)
	      {
	         if ((ddata=kpds_get_data(src_obj,region_prim,ddata)) == NULL)
	         {
	            kerror(lib,prog,
			   "failed to get region (norm_rgn %d, subrgn %d)",i,j);
                    kpds_close_object(src_obj); kpds_close_object(dst_obj);
                    return(FALSE);
	         }
	         for (k=0; k<num_elems; k++)
	         {
		    ddata[k]=(ddata[k] - dmin) * dfactor + norm_lower;
	         }
	         if (!kpds_put_data(dst_obj, region_prim, ddata))
	         {
	            kerror(lib,prog,
			   "failed to put region (norm_rgn %d, subrgn %d)",i,j);
                    kpds_close_object(src_obj); kpds_close_object(dst_obj);
                    return(FALSE);
	         }
	      }
	   }
	   kfree(ddata);
	
	/* if both value and map exist, copy value data (no conversion) */
	if (has_value && has_map)
	{
	   if (!kpds_copy_value(src_obj, dst_obj, TRUE))
	   {
	      kerror(lib,prog,"unable to copy value from source to dest");
              kpds_close_object(src_obj); kpds_close_object(dst_obj);
              return(FALSE);
	   }
	}

        kpds_close_object(src_obj); kpds_close_object(dst_obj);
	return TRUE;
}
/* -library_code_end */
