 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for klocxform
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 	lklocxform
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/* -library_includes */
static char *lib = "kdatamanip";
static char *rtn = "lklocxform";

static int _transform_uniform     PROTO((kobject, float, float, float, float, 
					 float, float, float, float, float,
					 kobject));

static int _transform_rectilinear PROTO((kobject, float, float, float,
					 float, float, float, kobject));

static int _transform_curvilinear PROTO((kobject, float, float, float, float, 
					 float, float, float, float, float,
					 kobject));
/* -library_includes_end */


/****************************************************************
* 
*  Routine Name: lklocxform - transform location data
* 
*       Purpose: This routine performs a homogeneous coordinate
*		 transformation on the location data in the source
*		 object, and saves the transformed location data into
*		 the destination object.
*
*		 The source object must have location data, and the
*		 destination object must not have location data for
*		 this routine to function properly.  This routine will
*		 create location data of the proper grid type in the
*		 destination.
*
*		 The coordinate transformation can be specfied by nine
*		 parameters which specify an x-y-z rotation, an x-y-z
*		 translation, and an x-y-z scaling.  The rotations are
*		 all performed with respect to the (0,0,0) origin.  A
*		 scaling matrix, a translation matrix, and three
*		 rotation matrices are constructed from these
*		 specified parameters and composited into a single
*		 transformation matrix in the following order :
*
*		 !  [1] Scaling
*		 !  [2] X Rotation
*		 !  [3] Y Rotation
*		 !  [4] Z Rotation
*		 !  [5] Translation
*
*		 A value of 1.0 should be used for any scaling
*		 parameters where no scaling is desired.  A value of
*		 0.0 should be used for any rotation parameters where
*		 no rotation is desired or for any translation
*		 parameters where no translation is desired.
*		
*		 The location points are interpreted to be in
*		 three-space for the transformations.  Explicit
*		 location data located in dimensions higher than three
*		 will be lost during the transformation.  The
*		 transformation will be performed using the datatype
*		 KFLOAT.
*
*		 In general, the grid type of the source object will
*		 be preserved in the destination object.  For example,
*		 uniform location data, when transformed and stored in
*		 the destination object, will still be uniform
*		 location data.  The exception to this rule is
*		 rectilinear data, which must be converted to
*		 curvilinear data in the event of a rotation.  In this
*		 case the destination location data will be stored as
*		 explicit curvilinear data points.
*
*		 Reference : Foley & VanDam "Computer Graphics :
*		 Principles and Practice", Second Edition, 1990,
*		 Chapter 5.
*
*         Input: src_obj - the input object with location data
*                xrot    - rotation about the x axis
*                yrot    - rotation about the y axis
*                zrot    - rotation about the z axis
*                xtrans  - translation in the x direction
*                ytrans  - translation in the y direction
*                ztrans  - translation in the z direction
*                xscale  - scaling in the x direction
*                yscale  - scaling in the y direction
*                zscale  - scaling in the z direction
*
*        Output: dst_obj - the output object
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Steve Kubica
*          Date: Apr 15, 1995
*      Verified: 
*  Side Effects: 
* Modifications: 
****************************************************************/
/* -library_def */
int lklocxform(
   kobject src_obj,
   float   xrot,
   float   yrot,
   float   zrot,
   float   xtrans,
   float   ytrans,
   float   ztrans,
   float   xscale,
   float   yscale,
   float   zscale,
   kobject dst_obj)
/* -library_def_end */

/* -library_code */
{
   int   grid;
   int   rstatus;
   

   /* -- make sure we have something to transform -- */
   if (!kpds_query_location(src_obj))
   {
      kerror(lib, rtn,"Input data object must have Location data.");
      return FALSE;
   }

   if (!kpds_get_attribute(src_obj, KPDS_LOCATION_GRID, &grid))
   {
      kerror(lib, rtn,"Failed to retrieve the source grid type.");
      return FALSE;
   }
   
   /* -- we can preserve grid type except when rotating rectilinear data -- */
   if (grid == KRECTILINEAR && (xrot != 0.0 || yrot != 0.0 || zrot != 0.0))
      grid = KCURVILINEAR;


   /* -- set the desired destination location grid type -- */
   if (!kpds_set_attribute(dst_obj, KPDS_LOCATION_GRID, grid))
   {
      kerror(lib, rtn,"Failed to set the destination grid type.");
      return FALSE;
   }

   /* -- create the destination location segment -- */
   if (!kpds_create_location(dst_obj))
   {
      kerror(lib, rtn,"Failed to create location segment in destination.");
      return FALSE;
   }

   /* -- set its physical size and data type -- */
   if (!kpds_copy_attributes(src_obj, dst_obj, 
			     KPDS_LOCATION_DATA_TYPE,
			     KPDS_LOCATION_SIZE, 
			     NULL))
   {
      kerror(lib, rtn,"Failed to set destination location size or data type.");
      return FALSE;
   }


   /* 
    *   destination now has a shiny new location segment waiting to be
    *   filled in and the grid variable dictates what grid type to
    *   work in.   in general, we can preserve the grid type, however
    *   rectilinar data can not be rotated without being mapped into
    *   a curvilinear space.
    */
   
   /* -- create a reference of source so we have no side effects -- */
   if ((src_obj = kpds_reference_object(src_obj)) == KOBJECT_INVALID)
   {
      kerror(lib, rtn, "Failed to reference input object");
      return FALSE;
   }

   /* -- create a reference of destination so we cast data okay -- */
   if ((dst_obj = kpds_reference_object(dst_obj)) == KOBJECT_INVALID)
   {
      kerror(lib, rtn, "Failed to reference output object");
      return FALSE;
   }

   /* -- transform locations in the appropriate grid type -- */
   switch (grid)
   {
      case KUNIFORM :
      {
	 rstatus = _transform_uniform(src_obj, 
				      xrot,   yrot,   zrot,
				      xtrans, ytrans, ztrans,
				      xscale, yscale, zscale,    
				      dst_obj);
      } break;
      
      case KRECTILINEAR :  /* -- we know there are no rotations -- */
      {
	 rstatus = _transform_rectilinear(src_obj, 
					  xtrans, ytrans, ztrans,
					  xscale, yscale, zscale,    
					  dst_obj);

      } break;

      case KCURVILINEAR :
      {
	 rstatus = _transform_curvilinear(src_obj, 
					  xrot,   yrot,   zrot,
					  xtrans, ytrans, ztrans,
					  xscale, yscale, zscale,    
					  dst_obj);
      } break;

      default :
	 kerror(lib, rtn, "Source location data has indeterminate grid type.");
	 rstatus = FALSE;
   }

   /* -- close reference objects -- */
   kpds_close_object(dst_obj);
   kpds_close_object(src_obj);
   
   return rstatus;
}


/*-----------------------------------------------------------
|  Routine Name: (static) _generate_transform_matrix
|       Purpose: generate a transformation matrix for given scalings
|		 translations and rotations
|       Returns: a static transformation matrix for the given 
|                translation  -- do not free this matrix
|    Written By: Wes Bethel and Steve Kubica
------------------------------------------------------------*/
static float *_generate_transform_matrix(
   float   xrot,
   float   yrot,
   float   zrot,
   float   xtrans, 
   float   ytrans, 
   float   ztrans,
   float   xscale, 
   float   yscale, 
   float   zscale)
{
   static float view[16];

   float scale[16];
   float trans[16];
   float rot_x[16];
   float rot_y[16];
   float rot_z[16];
   float xr;
   float yr;
   float zr;
   

   /* -- convert from degrees to radians -- */
   xr = kdegrees_radians(xrot);
   yr = kdegrees_radians(yrot);
   zr = kdegrees_radians(zrot);

   kfmatrix_identity(4, 4, view);

   kfmatrix_identity(4, 4, scale);
   scale[0]  = xscale;  /* (0,0) */
   scale[5]  = yscale;  /* (1,1) */
   scale[10] = zscale;  /* (2,2) */

   kfmatrix_identity(4, 4, trans);
   trans[12] = xtrans;  /* (3,1) */
   trans[13] = ytrans;  /* (3,2) */
   trans[14] = ztrans;  /* (3,3) */
   
   kfmatrix_identity(4, 4, rot_x);
   rot_x[5] = rot_x[10] = kcos(xr);   /* (1,1) (2,2) */
   rot_x[6] = ksin(xr);               /* (1,2) */
   rot_x[9] = -1.0 * rot_x[6];        /* (2,1) */

   kfmatrix_identity(4, 4, rot_y);
   rot_y[0] = rot_y[10] = kcos(yr);   /* (0,0) (2,2) */
   rot_y[8] = ksin(yr);               /* (2,0) */
   rot_y[2] = -1.0 * rot_y[8];        /* (0,2) */
   
   kfmatrix_identity(4, 4, rot_z);
   rot_z[0] = rot_y[5] = kcos(zr);    /* (0,0) (1,1) */
   rot_z[1] = ksin(zr);               /* (0,1) */
   rot_z[4] = -1.0 * rot_z[1];        /* (1,0) */

   kfmatrix_multiply(view,4,4, scale,4,4, view);
   kfmatrix_multiply(view,4,4, rot_x,4,4, view);
   kfmatrix_multiply(view,4,4, rot_y,4,4, view);
   kfmatrix_multiply(view,4,4, rot_z,4,4, view);
   kfmatrix_multiply(view,4,4, trans,4,4, view);

   return view;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _transform_uniform
|       Purpose: transform uniform location data
|       Returns: TRUE on success, FALSE otherwise
|    Written By: Steve Kubica
------------------------------------------------------------*/
static int _transform_uniform(
   kobject src_obj, 
   float   xrot,
   float   yrot,
   float   zrot,
   float   xtrans, 
   float   ytrans, 
   float   ztrans,
   float   xscale, 
   float   yscale, 
   float   zscale,    
   kobject dst_obj)
{
   float *view = _generate_transform_matrix(xrot,   yrot,   zrot,
					    xtrans, ytrans, ztrans,
					    xscale, yscale, zscale);
   double x;
   double y;
   double z;
   
   float vi[4];
   float vo[4];


   /* -- transform the location begin -- */
   if (!kpds_get_attribute(src_obj, KPDS_LOCATION_BEGIN, &x, &y, &z))
   {
      kerror(lib,rtn,"Unable to get uniform location begin from source.");
      return FALSE;
   }

   vi[0] = (float) x;
   vi[1] = (float) y;
   vi[2] = (float) z;
   vi[3] = 1.0;
   kfmatrix_vector_prod(view, vi, 4, 4, vo);

   if (!kpds_set_attribute(dst_obj, KPDS_LOCATION_BEGIN, 
			   (double) vo[0], (double) vo[1], (double) vo[2]))
   {
      kerror(lib,rtn,"Unable to store uniform location begin in destination.");
      return FALSE;
   }


   /* -- transform the location end -- */
   if (!kpds_get_attribute(src_obj, KPDS_LOCATION_END, &x, &y, &z))
   {
      kerror(lib,rtn,"Unable to get uniform location end from source.");
      return FALSE;
   }

   vi[0] = (float) x;
   vi[1] = (float) y;
   vi[2] = (float) z;
   vi[3] = 1.0;
   kfmatrix_vector_prod(view, vi, 4, 4, vo);

   if (!kpds_set_attribute(dst_obj, KPDS_LOCATION_END, 
			   (double) vo[0], (double) vo[1], (double) vo[2]))
   {
      kerror(lib,rtn,"Unable to store uniform location end in destination.");
      return FALSE;
   }

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _transform_rectilinear
|       Purpose: transform rectilinear location data
|       Returns: TRUE on success, FALSE otherwise
|    Written By: Steve Kubica
------------------------------------------------------------*/
static int _transform_rectilinear(
   kobject src_obj, 
   float   xtrans, 
   float   ytrans, 
   float   ztrans,
   float   xscale, 
   float   yscale, 
   float   zscale,    
   kobject dst_obj)
{
   float *data = NULL;
   float *v;

   int w;
   int h;
   int d;
   int i;


   /* -- we want to process data as (x,y,z) floating point triples -- */
   if (!kpds_set_attributes(src_obj, 
			    KPDS_LOCATION_DATA_TYPE, KFLOAT,
			    KPDS_LOCATION_SIZE, -1, -1, -1, 3,  /* 3 => xyz */
			    NULL))
   {
      kerror(lib, rtn,"Unable to set source presentation size or data type.");
      return FALSE;
   }
   
   /* -- we want to process data as (x,y,z) floating point triples -- */
   if (!kpds_set_attributes(dst_obj, 
			    KPDS_LOCATION_DATA_TYPE, KFLOAT,
			    KPDS_LOCATION_SIZE, -1, -1, -1, 3,  /* 3 => xyz */
			    NULL))
   {
      kerror(lib, rtn,"Unable to set destination presentation size or "
	              "data type."); 
      return FALSE;
   }


   /* -- get the size of the rectilinear data -- */
   if (!kpds_get_attribute(src_obj, KPDS_LOCATION_SIZE, &w, &h, &d, NULL))
   {
      kerror(lib, rtn,"Unable to get the source location size.");
      return FALSE;
   }

   /* -- just go for broke and get all the data in each direction -- */

      /*
       *   WIDTH data
       */

   /* -- get the width data -- */
   if ((data = kpds_get_data(src_obj, KPDS_LOCATION_WIDTH_ALL, NULL)) == NULL)
   {
      kerror(lib, rtn,"Unable to get location width data from source.");
      return FALSE;
   }

   /* -- scale and translate the width data -- */
   for (i = 0, v = data; i < w; i++)
      *v++ = *v * xscale + xtrans;

   /* -- store the width data -- */
   if (!kpds_put_data(dst_obj, KPDS_LOCATION_WIDTH_ALL, data))
   {
      kerror(lib, rtn,"Unable to store location width data in destination.");
      return FALSE;
   }
   kfree(data);

      /*
       *   HEIGHT data
       */

   /* -- get the height data -- */
   if ((data = kpds_get_data(src_obj, KPDS_LOCATION_HEIGHT_ALL, NULL)) == NULL)
   {
      kerror(lib, rtn,"Unable to get location height data from source.");
      return FALSE;
   }

   /* -- scale and translate the height data -- */
   for (i = 0, v = data; i < h; i++)
      *v++ = *v * yscale + ytrans;

   /* -- store the height data -- */
   if (!kpds_put_data(dst_obj, KPDS_LOCATION_HEIGHT_ALL, data))
   {
      kerror(lib, rtn,"Unable to store location height data in destination.");
      return FALSE;
   }
   kfree(data);
   
      /*
       *   DEPTH data
       */

   /* -- get the depth data -- */
   if ((data = kpds_get_data(src_obj, KPDS_LOCATION_DEPTH_ALL, NULL)) == NULL)
   {
      kerror(lib, rtn,"Unable to get location depth data from source.");
      return FALSE;
   }

   /* -- scale and translate the depth data -- */
   for (i = 0, v = data; i < d; i++)
      *v++ = *v * zscale + ztrans;

   /* -- store the depth data -- */
   if (!kpds_put_data(dst_obj, KPDS_LOCATION_DEPTH_ALL, data))
   {
      kerror(lib, rtn,"Unable to store location depth data in destination.");
      return FALSE;
   }
   kfree(data);

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _transform_curvilinear
|       Purpose: transform curvilinear location data
|       Returns: TRUE on success, FALSE otherwise
|    Written By: Steve Kubica and Wes Bethel
------------------------------------------------------------*/
static int _transform_curvilinear(
   kobject src_obj, 
   float   xrot,
   float   yrot,
   float   zrot,
   float   xtrans, 
   float   ytrans, 
   float   ztrans,
   float   xscale, 
   float   yscale, 
   float   zscale,    
   kobject dst_obj)
{
   float *data = NULL;
   float *view = _generate_transform_matrix(xrot,   yrot,   zrot,
					    xtrans, ytrans, ztrans,
					    xscale, yscale, zscale);
   float vi[4];
   float vo[4];

   float *xi;
   float *yi;
   float *zi;
   float *xo;
   float *yo;
   float *zo;

   int w;
   int h;
   int d;
   int e;
   int n;
   int num;

   int i;
   int j;
      

   /* -- we want to process data as (x,y,z) floating point triples -- */
   if (!kpds_set_attributes(src_obj, 
			    KPDS_LOCATION_DATA_TYPE, KFLOAT,
			    KPDS_LOCATION_SIZE, -1, -1, -1, 3,  /* 3 => xyz */
			    NULL))
   {
      kerror(lib, rtn,"Unable to set source presentation size or data type.");
      return FALSE;
   }
   
   /* -- we want to process data as (x,y,z) floating point triples -- */
   if (!kpds_set_attributes(dst_obj, 
			    KPDS_LOCATION_DATA_TYPE, KFLOAT,
			    KPDS_LOCATION_SIZE, -1, -1, -1, 3,  /* 3 => xyz */
			    NULL))
   {
      kerror(lib, rtn,"Unable to set destination presentation size or "
	              "data type."); 
      return FALSE;
   }
   
   /* -- want to work with the largest region of xyz points -- */
   if (!kpds_set_attributes(src_obj, 
			    KPDS_LOCATION_OPTIMAL_REGION_CONSTRAINT,1,1,1,3,
			    KPDS_LOCATION_OPTIMAL_REGION_FLAGS,     0,0,0,1, 
			    NULL))
   {
      kerror(lib, rtn,"Unable to set optimal region constraints.");
      return FALSE;
   }
      

   /* -- get the region size and number of regions we will be processing -- */
   if (!kpds_get_attribute(src_obj,
			   KPDS_LOCATION_OPTIMAL_REGION_SIZE,&w,&h,&d,&e,&num))
   {
      kerror(lib, rtn,"Unable to get the optimal region size.");
      return FALSE;
   }

   /* -- set the region size on the source -- */
   if (!kpds_set_attribute(src_obj,
			   KPDS_LOCATION_REGION_SIZE, w, h, d, e))
   {
      kerror(lib, rtn,"Unable to set the region size on the source.");
      return FALSE;
   }
      
   /* -- set the region size on the destination -- */
   if (!kpds_set_attribute(dst_obj,
			   KPDS_LOCATION_REGION_SIZE, w, h, d, e))
   {
      kerror(lib, rtn,"Unable to set the region size on the destination.");
      return FALSE;
   }
   

   /*
    *    Process the location data, region by region 
    */

   /* -- xyz points are indexically separated by this much -- */
   n = w*h*d;
   
   for (i = 0; i < num; i++)
   {
      /* -- get a region of data from the source -- */
      if ((data = kpds_get_data(src_obj, KPDS_LOCATION_REGION, data)) == NULL)
      {
	 kerror(lib, rtn,"Unable to get location data from source.");
	 return FALSE;
      }

      /* -- setup input pointers for easy data traversal -- */
      xi = data;
      yi = xi + n;
      zi = yi + n;

      /* -- setup output pointers for easy data traversal -- */
      xo = data;
      yo = xo + n;
      zo = yo + n;

      /* -- transform the data, point by point, in place -- */
      for (j = 0; j < n; j++)
      {
	 vi[0] = *xi++;
	 vi[1] = *yi++;
	 vi[2] = *zi++;
	 vi[3] = 1.0;

	 kfmatrix_vector_prod(view, vi, 4, 4, vo);
	 
	 *xo++ = vo[0];
	 *yo++ = vo[1];
	 *zo++ = vo[2];
      }
   
      /* -- store the region of data back into the destination -- */
      if (!kpds_put_data(dst_obj, KPDS_LOCATION_REGION, data))
      {
	 kerror(lib, rtn,"Unable to store location data in destination.");
	 return FALSE;
      }
   }
   kfree(data);
   
   return TRUE;
}
/* -library_code_end */
