 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for klinearop
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 	lklinearop
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/* -library_includes */
#include "internals.h"
/* -library_includes_end */


/****************************************************************
* 
*  Routine Name: lklinearop - perform linear operation (convolution/correlation) on data object
* 
*       Purpose: Library routine for klinearop.  Performs convolution or
*                correlation operations on one, all, or any combination 
*                of directions. The directions supported are: width, height,
*                depth, time, elements. Operates on sub-objects as well.
*                The sub-objects parameters suppored are:
*                    sub-object position and size in the supported 
*                    direction.
*
*         Input: i1 (src)       - source data object
*                i2 (kernel)    - kernel data object
*                filtering_type - linear (0) or circular (1)
*                real_pad_value - real value for padding
*                img_pad_value  - imaginary value for padding
*                offset         - array of kernel offset values (location of hot-spot), these 5 values (w,h,d,t,e) override those stored in the kernel object. If offset
is NULL, then the default values stored with the kernel object are used.
*                size_type      - truncate (0) or extend the result of the operation (output object) to include full range of support (1)
*                reflection     - reflect (for convolution) (1) or do not reflect (0) the kernel
*                num_dim        - number of directions to operate on
*                dim_list       - array of directions on which to operate (use KWIDTH,KHEIGHT,KDEPTH,KTIME,KELEMENTS)
*                sub_size       - array specifying the sub-object size, these 5 values (w,h,d,t,e) specify a subregion of the input object on which to operate 
*                upcast         - force upcast of input data type to KDOUBLE or KDCOMPLEX before processing (1), or allow upcast of data types to the best fitting type in {KUBYTE, KLONG, KDOUBLE, KDCOMPLEX} (0) (look out for loss of precision)
*                sub_pos        - array specifying the sub-object position, these 5 values specify the origin of the sub-object in (w,h,d,t,e)
*
*        Output: o (dest)    - destination data object 
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Jonio Roberto de Hollanda Cavalcanti, Ramiro Jordan
*          Date: Apr 13, 1995
*      Verified: Jan 26, 1994
*  Side Effects: 
* Modifications: 
****************************************************************/
/* -library_def */
int lklinearop( kobject src,
                kobject kernel,
                int filtering_type,
                double real_pad_value,
                double img_pad_value,
                int  *offset,
                int  size_type,
                int  reflection,
                int  num_dim, 
                long *dim_list,
                int  *sub_size,
                int  *sub_pos,
                int  upcast,
                kobject dest )
/* -library_def_end */

/* -library_code */
{

   klist *objlist=NULL;
   kobject src_ref=NULL;
   char *lib = "kdatamanip";       /* library name (for error reporting) */
   char *rtn = "lklinearop";       /* routine name (for error reporting) */

   int data_type;                /* source object data type */
   int kernel_data_type;         /* kernel object data type */
   int pdata_type;               /* data type for processing */
   int w, h, d, t, e;            /* No. of data elements in each direction */
			         /* of the source object */
   int wk, hk, dk, tk, ek;       /* No. of data elements in each direction */
                                 /* of the kernel object */
   int wt, ht, dt, tt, et;       /* No. of total data elements in each */
                                 /* direction for processing */ 
   int data_length;              /* No. of data points to process */
   int num_regions;              /* No. of regions to process */
   int i, j, loop_count, count;  /* loop counters */
   int temp;

   int region_size[5];
   int new_sub_size[5];

   unsigned char *bdata = NULL, *beval = NULL, *bkdata = NULL, tempb;
   long *ldata = NULL, *leval = NULL, *lkdata = NULL, templ;
   double *ddata = NULL, *deval = NULL, *dkdata = NULL, tempd;
   kdcomplex *cmplx_data = NULL, *cmplx_eval = NULL, *kcmplx_data = NULL;
   kdcomplex tempcmplx;
   
   int grid_type = KNONE, has_time = FALSE;
   int dim_size;
   int i0,i1,i2,i3,i4;
   int k0,k1,k2,k3,k4;
   int index0,index1,index2,index3,index4;
   int l_region0,l_region1,l_region2,l_region3,l_region4;
   int aux0,aux1,aux2,aux3,aux4;
   int pos1=0,pos2=0,pos3=0,pos4=0;
   int cnt1=0,cnt2=0,cnt3=0;
   int cntk1=0,cntk2=0,cntk3=0;

   kinfo(KVERBOSE,"lklinearop parameters:\n");
   kinfo(KVERBOSE,"Filtering type: %d\n",filtering_type);
   kinfo(KVERBOSE,"Pad value: %f %f\n",real_pad_value,img_pad_value);
   if (offset != NULL)
     kinfo(KVERBOSE,"Kernel offset: w:%d h:%d d:%d t:%d e:%d\n",
                     offset[0],offset[1],offset[2],offset[3],offset[4]);
   kinfo(KVERBOSE,"Sizing type: %d\n",size_type);
   kinfo(KVERBOSE,"Reflection type: %d\n",reflection);
   kinfo(KVERBOSE,"Number of dimensions: %d\n",num_dim);
   for (i=0; i<num_dim; i++) kinfo(KVERBOSE,"  Dim# %d is %d\n",i,dim_list[i]);
   kinfo(KVERBOSE,"Subobject size: w:%d h:%d d:%d t:%d e:%d\n",
                   sub_size[0],sub_size[1],sub_size[2],sub_size[3],sub_size[4]);
   kinfo(KVERBOSE,"Subobject posn: w:%d h:%d d:%d t:%d e:%d\n",
                   sub_pos[0],sub_pos[1],sub_pos[2],sub_pos[3],sub_pos[4]);
   kinfo(KVERBOSE,"Upcasting: %d\n",upcast);

  /*
   * Initialize parameters.
   */
   w = wk = wt = 0;
   h = hk = ht = 0;
   d = dk = dt = 0;
   t = tk = tt = 0;
   e = ek = et = 0;

  /*
   * Only one object may have time data.
   */
   if (!kdmanip_multi_input_time(&has_time, 2, src, kernel))
           return(FALSE);

  /*
   * Only one object may have location data.
   */
    if (!kdmanip_multi_input_loc(&grid_type, 2, src, kernel))
           return(FALSE);

  /*
   * If kernel object has MASK data exit lklinearop.
   */
   if ( kpds_query_mask(kernel) ) {
      kerror(lib, rtn, "Kernel object must have only VALUE data.");
      return(FALSE);
   }

  /*
   * If source object has MASK data exit lklinearop.
   */
   if ( kpds_query_mask(src) ) {
      kerror(lib, rtn, "Source object must have only VALUE data.");
      return(FALSE);
   }

  /*
   * Both source and destination objects must have VALUE data.
   */ 
   if (!kpds_query_value(src) && !kpds_query_value(kernel))
   {
    kerror(lib, rtn, "Source object must have VALUE data.");
    return(FALSE);
   }
   if (!kpds_query_value(dest) && !kpds_create_value(dest))
   {
    kerror(lib,rtn, "Unable to create value segment for dest object.");
    return(FALSE);
   }

  /*
   *  Set a reference for the source object so that the library does
   *  not have any side effects on the source object's attributes.
   */
   KCALL((src_ref = kpds_reference_object(src)));
   objlist = klist_add(objlist,src_ref,"KOBJECT");

  /*
   * Set up a reference to the kernel object to facilitate processing.
   * Do not want to introduce side effects on the kernel atributes.
   */
   KCALL((kernel = kpds_reference_object(kernel)));
   objlist = klist_add(objlist,kernel,"KOBJECT");


  /*
   * Copy the subobject size to an internal array - we may need to change
   * the elements dimension if the object has a map!
   */
   for (i=0; i<5; i++) new_sub_size[i] = sub_size[i];

  /*
   * If source object has MAP data un-map it. Be sure to fix up the new_sub_size
   * value for the elements direction by multiplying it by the number of 
   * columns in the map BEFORE turning mapping mode on (or else this information
   * becomes permanently unavailable!)
   */
   if (kpds_query_map(src_ref))
   {
    KCALL(kpds_get_attribute(src_ref, KPDS_MAP_SIZE, &w, &h, &d, &t, &e));
    KCALL(kpds_set_attribute(src_ref, KPDS_MAPPING_MODE, KMAPPED));
    new_sub_size[4] *= w;
   }

  /*
   * If kernel object has MAP data un-map it. 
   */
   if (kpds_query_map(kernel)) 
    KCALL(kpds_set_attribute(kernel, KPDS_MAPPING_MODE, KMAPPED));

  /*
   * If the destination object has map, remove it.
   */
   if (kpds_query_map(dest))
    KCALL(kpds_destroy_map(dest));

  /*
   * Now get the presentation attributes of the source reference object.
   *
   * The source object may contain a collection of data, thus we get
   * the KPDS_VALUE_SIZE attribute which provides the following information:
   *          - w = number of parameters in the width direction
   *          - h = number of parameters in the height direction
   *          - d = number of parameters in the depth direction
   *          - t = number of parameters in the time direction
   *          - e = number of parameters in the elements direction
   *
   * Determine the number of dimentions in the object.
   *
   */
   KCALL(kpds_get_attribute(src_ref, KPDS_VALUE_SIZE, &w, &h, &d, &t, &e));
   dim_size = 0;
   if (w>1) dim_size++; 
   if (h>1) dim_size++; 
   if (d>1) dim_size++; 
   if (t>1) dim_size++; 
   if (e>1) dim_size++; 

  /*
   * If offset == NULL get the default offset from the kernel by
   * using the KPDS_KERNEL_ORIGIN attribute.
   * If this attribute does not exist print out erro message.
   */
   if (offset == NULL)
   {
    offset = kmalloc(5*sizeof(int));
    if (offset == NULL)  
    {
     kerror(lib, rtn, "kmalloc failed for offset.");
     (void)klist_free(objlist, (kfunc_void)lkcall_free);
     return(FALSE);
    }
    KCALL(kpds_get_attribute(kernel,KPDS_KERNEL_ORIGIN,&offset[0],
                             &offset[1],&offset[2],&offset[3],&offset[4]));
   }

  /*
   * Now get the presentation attributes of the kernel reference object.
   *
   * The kernel object may contain a collection of data, thus we get
   * the KPDS_VALUE_SIZE attribute which provides the following information:
   *          - wk = number of parameters in the width direction
   *          - hk = number of parameters in the height direction
   *          - dk = number of parameters in the depth direction
   *          - tk = number of parameters in the time direction
   *          - ek = number of parameters in the elements direction
   *
   */
   KCALL(kpds_get_attribute(kernel,KPDS_VALUE_SIZE,&wk,&hk,&dk,&tk,&ek));

  /*
   * Verify whether the offset parameters are set accordingly.
   * It cannot generate a shift larger than input object size.
   * 
   */ 
   if (offset[0]<0 || offset[0]>(wk-1))
   {
    kerror(lib, rtn, "invalid offset for width direction.");
    (void)klist_free(objlist, (kfunc_void)lkcall_free);
    return(FALSE);
   }

   if (offset[1]<0 || offset[1]>(hk-1))
   {
    kerror(lib, rtn, "invalid offset for heigth direction.");
    (void)klist_free(objlist, (kfunc_void)lkcall_free);
    return(FALSE);
   }

   if (offset[2]<0 || offset[2]>(dk-1))  
   {
    kerror(lib, rtn, "invalid offset for depth direction.");
    (void)klist_free(objlist, (kfunc_void)lkcall_free);
    return(FALSE);
   }

   if (offset[3]<0 || offset[3]>(tk-1))
   {
    kerror(lib, rtn, "invalid offset for time direction.");
    (void)klist_free(objlist, (kfunc_void)lkcall_free);
    return(FALSE);
   }

   if (offset[4]<0 || offset[4]>(ek-1))
   {
    kerror(lib, rtn, "invalid offset for element direction.");
    (void)klist_free(objlist, (kfunc_void)lkcall_free);
    return(FALSE);
   }

  /*
   * Select REGIONS to process on.
   */
   region_size[0]=1; 
   region_size[1]=1;
   region_size[2]=1;
   region_size[3]=1;
   region_size[4]=1;
   for (loop_count = 0; loop_count<num_dim;loop_count++) {
      switch (dim_list[loop_count])
      {
          case KWIDTH:
                      region_size[0] = new_sub_size[0];
                      offset[0] += (-wk+1);
                      break;
          case KHEIGHT:
                      region_size[1] = new_sub_size[1];
                      offset[1] += (-hk+1);
                      break;
          case KDEPTH:
                      region_size[2] = new_sub_size[2];
                      offset[2] += (-dk+1);
                      break;
          case KTIME:
                      region_size[3] = new_sub_size[3];
                      offset[3] += (-tk+1);
                      break;
          case KELEMENTS:
                      region_size[4] = new_sub_size[4];
                      offset[4] += (-ek+1);
                      break;
      }
   }
   
  /*
   * Set up input object offset.
   */
   KCALL(kpds_set_attribute(src_ref, KPDS_VALUE_OFFSET,offset[0],offset[1],
                            offset[2],offset[3],offset[4]));

  /*
   * Determine the total size of the destination
   * object.
   */
   wt = (w + wk - 1);
   ht = (h + hk - 1);
   dt = (d + dk - 1);
   tt = (t + tk - 1);
   et = (e + ek - 1);

  /*
   * Verify whether the sub-object parameters are set accordingly.
   * It cannot generate a sub-object bigger than the original object.
   * 
   */ 
   if ((new_sub_size[0]+sub_pos[0])>wt)
   {
    kerror(lib, rtn, "invalid sub_object for width direction.");
    (void)klist_free(objlist, (kfunc_void)lkcall_free); 
    return(FALSE);
   }
   if ((new_sub_size[1]+sub_pos[1])>ht)
   {
    kerror(lib, rtn, "invalid sub_object for heigth direction.");
    (void)klist_free(objlist, (kfunc_void)lkcall_free);
    return(FALSE);
   }
   if ((new_sub_size[2]+sub_pos[2])>dt)
   {
    kerror(lib, rtn, "invalid sub_object for depth direction.");
    (void)klist_free(objlist, (kfunc_void)lkcall_free);
    return(FALSE);
   }
   if ((new_sub_size[3]+sub_pos[3])>tt)
   {
    kerror(lib, rtn, "invalid sub_object for time direction.");
    (void)klist_free(objlist, (kfunc_void)lkcall_free);
    return(FALSE);
   }
   if ((new_sub_size[4]+sub_pos[4])>et)
   {
    kerror(lib, rtn, "invalid sub_object for element direction.");
    (void)klist_free(objlist, (kfunc_void)lkcall_free);
    return(FALSE);
   }

  /*
   * Set up the object dimensions before processing.
   * There will be another set up after processing defined by  
   * the size_type variable.
   */
   KCALL(kpds_set_attribute(src_ref, KPDS_VALUE_SIZE, wt, ht, dt, tt, et));
   KCALL(kpds_set_attribute(dest, KPDS_VALUE_SIZE, wt, ht, dt, tt, et));

  /*
   * Pad the source object.
   * This pad value will be used to process the boundary points
   * in both cases truncated and expanded. 
   *
   * Based on the filtering type selected, perform zero (or constant)
   * padding or circular wrapping of data elements.  That is:
   * 0) linear filtering: zero or constant value pad
   * 1) circular filtering: wraparound of data elements
   */
   if ( filtering_type == 0 ) {
      KCALL(kpds_set_attribute(src_ref, KPDS_VALUE_INTERPOLATE, KPAD));
      KCALL(kpds_set_attribute(src_ref, KPDS_VALUE_PAD_VALUE,real_pad_value, 
                               img_pad_value));
   } else {
/*
 * The KWRAP option remains to be implemented in polymorphic services.
*/
/*
      KCALL(kpds_set_attribute(src_ref, KPDS_VALUE_INTERPOLATE, KWRAP));
*/
   }

  /*
   * If convolution type of operations are to be performed flip the
   * kernel object, else do nothing.
   */
   if (reflection == 1) {
      KCALL(lkflip(kernel, num_dim, dim_list, kernel));
   }

  /*
   * The upcast variable set to TRUE indicates that the data type
   * MUST be set to DOUBLE or KDCOMPLEX.
   * If upcast is FALSE, select the appropriate common data type for 
   * processing the data.
   * For that, obtain data type of kernel object to be processed.
   *     - kernel_data_type = kernel object data type
   *     - data_type = source  object data type
   *
   * Want to minimize any corruption on the data integrity.
   * Recommended data types are:
   *      - KUBYTE = unsigned char 
   *      - KLONG = long integer
   *      - KDOUBLE = double precision floating point
   *      - KDCOMPLEX = double precision complex
   */
   KCALL(kpds_get_attribute(kernel,KPDS_VALUE_DATA_TYPE,&kernel_data_type));
   KCALL(kpds_get_attribute(src_ref,KPDS_VALUE_DATA_TYPE,&data_type));
   if (!upcast)
     {
       pdata_type = kdatatype_cast_process(data_type,kernel_data_type,
                            (KUBYTE | KLONG | KDOUBLE | KDCOMPLEX) );
     }
   else 
     {
       pdata_type = kdatatype_cast_process(data_type,kernel_data_type,
                            (KDOUBLE | KDCOMPLEX) );
     }

  /*
   * If data is byte or ubyte representable then process it as ubyte (KUBYTE).
   * If data is ulong, long, ushort  or short representable then 
   * process it as long (KLONG).
   * If data is float representable then process it as double (KDOUBLE).
   * If data is COMPLEX representable then process it as double 
   * complex (KDCOMPLEX).  In the destination object, the data type 
   * will be automatically casted to the appropriate data type.
   */
   if ( (pdata_type == KUBYTE) || (pdata_type == KLONG) || 
        (pdata_type == KDOUBLE) || (pdata_type == KDCOMPLEX) ) {
      KCALL(kpds_set_attribute(src_ref, KPDS_VALUE_DATA_TYPE,pdata_type));
      KCALL(kpds_set_attribute(kernel, KPDS_VALUE_DATA_TYPE, pdata_type));
      KCALL(kpds_set_attribute(dest, KPDS_VALUE_DATA_TYPE, pdata_type));
   }
  /*
   * Set up the object size atttributes.
   */
   KCALL(kpds_set_attributes(src_ref,KPDS_VALUE_REGION_SIZE,
                          region_size[0],region_size[1],
                          region_size[2],region_size[3],
                          region_size[4],KPDS_VALUE_POSITION,
                          sub_pos[0],sub_pos[1],sub_pos[2],
                          sub_pos[3],sub_pos[4],NULL));
   KCALL(kpds_set_attributes(dest,KPDS_VALUE_REGION_SIZE,
                          region_size[0],region_size[1],
                          region_size[2],region_size[3],
                          region_size[4],KPDS_VALUE_POSITION,
                          sub_pos[0],sub_pos[1],sub_pos[2],
                          sub_pos[3],sub_pos[4],NULL));
   data_length = region_size[0]*region_size[1]*region_size[2]*
                 region_size[3]*region_size[4];

   num_regions = (new_sub_size[0]*new_sub_size[1]*new_sub_size[2]*
                  new_sub_size[3]*new_sub_size[4])/data_length;
  /*
   * Order the region_size variable to make the loops independent 
   * of the directions defined by the user.
   * For that pull up the positions of the region_size variable that
   * are different from 1 to the frist positions of the region_size variable. 
   * For examplo, if region_size is equal to (1,2,4,1,1) after the 
   * arrangement it  will be equal to (2,4,1,1,1). 
   */
   if(data_length != 1)
     for (i=0;i<4;i++)
       if (region_size[i] == 1)
       {
        j = i+1;
        for (j=i+1;j<4;j++)
          if (region_size[j] != 1) {
             temp = region_size[i];
             region_size[i] = region_size[j];
             region_size[j] = temp;
          }
       }
   l_region0 = l_region1 = l_region2 = 0;
   l_region3 = l_region4 = 0;
   offset[0] = offset[1] = offset[2] = 0;
   offset[3] = offset[4] = 0;

  /*
   * Set up the limits for the loops.
   */
   if (region_size[0] > 1) 
   {
    l_region0 = wk-1;
    offset[0] = l_region0;
   }
   if (region_size[1] > 1) 
   {
    l_region1 = hk-1;
    offset[1] = l_region1;
   }
   if (region_size[2] > 1) 
   {
    l_region2 = dk-1;
    offset[2] = l_region2;
   }
   if (region_size[3] > 1) 
   {
    l_region3 = tk-1;
    offset[3] = l_region3;
   }
   if (region_size[4] > 1) 
   {
    l_region4 = ek-1;
    offset[4] = l_region4;
   }

  /*
   * Set up constants to be used in
   * the loops.
   */
   cnt1 = region_size[0]*region_size[1]; 
   cnt2 = cnt1*region_size[2]; 
   cnt3 = cnt2*region_size[3]; 
   cntk1 = wk*hk;
   cntk2 = cntk1*dk;
   cntk3 = cntk2*tk;

  /*
   * Process KDOUBLE data.
   */
   if ( (pdata_type == KDOUBLE) ) {
     if ((dkdata = kpds_get_data(kernel,KPDS_VALUE_ALL,dkdata)) == NULL) {
        kerror(lib, rtn, "kpds_get_data failed for dkdata.");
        (void)klist_free(objlist, (kfunc_void)lkcall_free);
        return(FALSE);
     }
     deval = (double *)kmalloc(data_length*sizeof(double));
     if (deval == NULL) {
        kerror(lib, rtn, "kmalloc failed for deval.");
        (void)klist_free(objlist, (kfunc_void)lkcall_free);
        return(FALSE);
     }

    /*
     * Initialize deval (sub-object processng). 
     */
     for (i=0;i<data_length;i++) deval[i] = real_pad_value; 
     for (count=0;count<num_regions;count++) { 
        if ((ddata = kpds_get_data(src_ref, KPDS_VALUE_REGION,ddata))==NULL) 
        {
           kerror(lib, rtn, "kpds_get_data failed for ddata.");
           (void)klist_free(objlist, (kfunc_void)lkcall_free);
           return(FALSE);
        }
      if (dim_size>2)
      {
       for (i4=l_region4;i4<region_size[4];i4++){
        for (i3=l_region3;i3<region_size[3];i3++){
         for (i2=l_region2;i2<region_size[2];i2++){
          for (i1=l_region1;i1<region_size[1];i1++){
           for (i0=l_region0;i0<region_size[0];i0++){
            tempd = 0.0;
  
            index4 = i4;
            for (k4=0; k4<ek; k4++) {
             aux4 = index4*cnt3;
             pos4 = (i4-offset[4])*cnt3;
             index3 = i3;
             for (k3=0; k3<tk; k3++) {
              aux3 = index3*cnt2;
              pos3 = (i3-offset[3])*cnt2;
              index2 = i2;
              for (k2=0; k2<dk; k2++) {
               aux2 = index2*cnt1;
               pos2 = (i2-offset[2])*cnt1;
               index1 = i1;
               for (k1=0; k1<hk; k1++) {
                index0 = aux4 + aux3 + aux2 + index1*region_size[0] + i0;
                pos1 = (i1-offset[1])*region_size[0];
                for (k0=0; k0<wk; k0++) {
                 aux0  = k4*cntk3 +
                         k3*cntk2 +
                         k2*cntk1 +
                         k1*wk +
                         k0;
                 tempd += dkdata[aux0]*ddata[index0--];
                }
                index1--;
               }
               index2--;
              }
              index3--;
             }
             index4--;
            }
             aux0 =  pos4 + pos3 + pos2 + pos1 + (i0-offset[0]);
             deval[aux0]= tempd;
            } 
           }  
          } 
         }
        }
       } else {
               for (i1=l_region1;i1<region_size[1];i1++){
                aux0 = (i1-offset[1])*region_size[0];
                for (i0=l_region0;i0<region_size[0];i0++){
                 tempd = 0.0;
  
                 index1 = i1;
                 for (k1=0; k1<hk; k1++) {
                  index0 = index1*region_size[0] + i0;
                  for (k0=0; k0<wk; k0++) {
                   tempd += dkdata[k1*wk + k0]*ddata[index0--];
                  }
                  index1--;
                 }
                 aux1 = aux0 + (i0-offset[0]);
                 deval[aux1]= tempd;
                 }
               }
         }
         if (!kpds_put_data(dest, KPDS_VALUE_REGION, deval)) {
          kerror(lib, rtn, "kpds_put_data failed for deval.");
          (void)klist_free(objlist, (kfunc_void)lkcall_free);
          return(FALSE);
         }
         if (ddata) kfree(ddata);
       }
     }

   /*
    * Process KUBYTE data.
    */
    if ( (pdata_type == KUBYTE) ) {
    if ((bkdata = kpds_get_data(kernel, KPDS_VALUE_ALL,bkdata)) == NULL) {
     kerror(lib, rtn, "kpds_get_data failed for bkdata.");
     return(FALSE);
    }
    beval = (unsigned char *) kmalloc(data_length*sizeof(unsigned char));
    if (beval == NULL) {
     kerror(lib, rtn, "kmalloc failed for beval.");
     return(FALSE);
    }

   /*
    * Initialize beval (sub-object procesing).
    */
    for (i=0;i<data_length;i++) beval[i] = (unsigned char) real_pad_value;
    for (count=0;count<num_regions;count++) { 
     if ((bdata = kpds_get_data(src_ref,KPDS_VALUE_REGION,bdata)) == NULL) 
     {
      kerror(lib, rtn, "kpds_get_data failed for bdata.");
      return(FALSE);
     }
      if (dim_size>2)
      {
       for (i4=l_region4;i4<region_size[4];i4++){
        for (i3=l_region3;i3<region_size[3];i3++){
         for (i2=l_region2;i2<region_size[2];i2++){
          for (i1=l_region1;i1<region_size[1];i1++){
           for (i0=l_region0;i0<region_size[0];i0++){
            tempb = 0.0;
  
            index4 = i4;
            for (k4=0; k4<ek; k4++) {
             aux4 = index4*cnt3;
             pos4 = (i4-offset[4])*cnt3;
             index3 = i3;
             for (k3=0; k3<tk; k3++) {
              aux3 = index3*cnt2;
              pos3 = (i3-offset[3])*cnt2;
              index2 = i2;
              for (k2=0; k2<dk; k2++) {
               aux2 = index2*cnt1;
               pos2 = (i2-offset[2])*cnt1;
               index1 = i1;
               for (k1=0; k1<hk; k1++) {
                index0 = aux4 + aux3 + aux2 + index1*region_size[0] + i0;
                pos1 = (i1-offset[1])*region_size[0];
                for (k0=0; k0<wk; k0++) {
                 aux0  = k4*cntk3 +
                         k3*cntk2 +
                         k2*cntk1 +
                         k1*wk +
                         k0;
                 tempb += bkdata[aux0]*bdata[index0--];
                }
                index1--;
               }
               index2--;
              }
              index3--;
             }
             index4--;
            }
             aux0 =  pos4 + pos3 + pos2 + pos1 + (i0-offset[0]);
             beval[aux0]= tempb;
            } 
           }  
          } 
         }
        }
       } else {
               for (i1=l_region1;i1<region_size[1];i1++){
                aux0 = (i1-offset[1])*region_size[0];
                for (i0=l_region0;i0<region_size[0];i0++){
                 tempb = 0.0;
  
                 index1 = i1;
                 for (k1=0; k1<hk; k1++) {
                  index0 = index1*region_size[0] + i0;
                  for (k0=0; k0<wk; k0++) {
                   tempb += bkdata[k1*wk + k0]*bdata[index0--];
                  }
                  index1--;
                 }
                 aux1 = aux0 + (i0-offset[0]);
                 beval[aux1]= tempb;
                 }
               }
         }
      if (!kpds_put_data(dest, KPDS_VALUE_REGION, beval)) {
       kerror(lib, rtn, "kpds_put_data failed for beval.");
       (void)klist_free(objlist, (kfunc_void)lkcall_free);
       return(FALSE);
      }
     }
    }

   /*
    * Process KLONG data.
    */
    if ( (pdata_type == KLONG) ) {
    if ((lkdata = kpds_get_data(kernel, KPDS_VALUE_ALL, lkdata)) == NULL) {
     kerror(lib, rtn, "kpds_get_data failed for lkdata.");
     return(FALSE);
    }
    leval = (long *) kmalloc(data_length*sizeof(long));
    if (leval == NULL) {
     kerror(lib, rtn, "kmalloc failed for leval.");
     return(FALSE);
    }

   /*
    * Initialize leval (sub-object procesing).
    */
    for (i=0;i<data_length;i++) leval[i] = (long) real_pad_value;
    for (count=0;count<num_regions;count++) { 
     if ((ldata = kpds_get_data(src_ref, KPDS_VALUE_REGION, ldata)) == NULL) 
     {
      kerror(lib, rtn, "kpds_get_data failed for ldata.");
      return(FALSE);
     }
      if (dim_size>2)
      {
       for (i4=l_region4;i4<region_size[4];i4++){
        for (i3=l_region3;i3<region_size[3];i3++){
         for (i2=l_region2;i2<region_size[2];i2++){
          for (i1=l_region1;i1<region_size[1];i1++){
           for (i0=l_region0;i0<region_size[0];i0++){
            templ = 0.0;
  
            index4 = i4;
            for (k4=0; k4<ek; k4++) {
             aux4 = index4*cnt3;
             pos4 = (i4-offset[4])*cnt3;
             index3 = i3;
             for (k3=0; k3<tk; k3++) {
              aux3 = index3*cnt2;
              pos3 = (i3-offset[3])*cnt2;
              index2 = i2;
              for (k2=0; k2<dk; k2++) {
               aux2 = index2*cnt1;
               pos2 = (i2-offset[2])*cnt1;
               index1 = i1;
               for (k1=0; k1<hk; k1++) {
                index0 = aux4 + aux3 + aux2 + index1*region_size[0] + i0;
                pos1 = (i1-offset[1])*region_size[0];
                for (k0=0; k0<wk; k0++) {
                 aux0  = k4*cntk3 +
                         k3*cntk2 +
                         k2*cntk1 +
                         k1*wk +
                         k0;
                 templ += lkdata[aux0]*ldata[index0--];
                }
                index1--;
               }
               index2--;
              }
              index3--;
             }
             index4--;
            }
             aux0 =  pos4 + pos3 + pos2 + pos1 + (i0-offset[0]);
             leval[aux0]= templ;
            } 
           }  
          } 
         }
        }
       } else {
               for (i1=l_region1;i1<region_size[1];i1++){
                aux0 = (i1-offset[1])*region_size[0];
                for (i0=l_region0;i0<region_size[0];i0++){
                 templ = 0.0;
  
                 index1 = i1;
                 for (k1=0; k1<hk; k1++) {
                  index0 = index1*region_size[0] + i0;
                  for (k0=0; k0<wk; k0++) {
                   templ += lkdata[k1*wk + k0]*ldata[index0--];
                  }
                  index1--;
                 }
                 aux1 = aux0 + (i0-offset[0]);
                 leval[aux1]= templ;
                 }
               }
         }
      if (!kpds_put_data(dest, KPDS_VALUE_REGION, leval)) {
       kerror(lib, rtn, "kpds_put_data failed for leval.");
       (void)klist_free(objlist, (kfunc_void)lkcall_free);
       return(FALSE);
      }
     }
    }

   /*
    * Process KDCOMPLEX data.
    */
    if ( (pdata_type == KDCOMPLEX) ) {
    if ((kcmplx_data=kpds_get_data(kernel,KPDS_VALUE_ALL, kcmplx_data))==NULL) 
    {
     kerror(lib, rtn, "kpds_get_data failed for kcmplx_data.");
     return(FALSE);
    }
   /*
    * Initialize  cmplx_eval (sub-object procesing).
    */
    cmplx_eval = (kdcomplex *) kmalloc(data_length*sizeof(kdcomplex));
    if (cmplx_eval == NULL) {
     kerror(lib, rtn, "kmalloc failed for cmplx_eval.");
     return(FALSE);
    }
    for (i=0;i<data_length;i++) 
     cmplx_eval[i] = kdccomp(real_pad_value,img_pad_value);
    for (count=0;count<num_regions;count++) { 
     if ((cmplx_data=kpds_get_data(src_ref,KPDS_VALUE_REGION, cmplx_data))
           ==NULL) 
     {
      kerror(lib, rtn, "kpds_get_data failed for cmplx_data.");
      return(FALSE);
     }
     if (dim_size>2) 
     {
      for (i4=l_region4;i4<region_size[4];i4++){
       for (i3=l_region3;i3<region_size[3];i3++){
        for (i2=l_region2;i2<region_size[2];i2++){
         for (i1=l_region1;i1<region_size[1];i1++){
          for (i0=l_region0;i0<region_size[0];i0++){
           tempcmplx = kdccomp(0.0,0.0);
  
            index4 = i4;
            for (k4=0; k4<ek; k4++) {
             aux4 = index4*cnt3;
             pos4 = (i4-offset[4])*cnt3;
             index3 = i3;
             for (k3=0; k3<tk; k3++) {
              aux3 = index3*cnt2;
              pos3 = (i3-offset[3])*cnt2;
              index2 = i2;
              for (k2=0; k2<dk; k2++) {
               aux2 = index2*cnt1;
               pos2 = (i2-offset[2])*cnt1;
               index1 = i1;
               for (k1=0; k1<hk; k1++) {
                index0 = aux4 + aux3 + aux2 + index1*region_size[0] + i0;
                pos1 = (i1-offset[1])*region_size[0];
                for (k0=0; k0<wk; k0++) {
                 aux0  = k4*cntk3 +
                         k3*cntk2 +
                         k2*cntk1 +
                         k1*wk +
                         k0;
                 tempcmplx = kdcadd(tempcmplx,kdcmult(kcmplx_data[aux0],
                                                     cmplx_data[index0--]));
                }
                index1--;
               }
               index2--;
              }
              index3--;
             }
             index4--;
            }
            aux0 =  pos4 + pos3 + pos2 + pos1 + (i0-offset[0]);
            cmplx_eval[aux0] = tempcmplx;
           }
          }
         }
        }
       }
      } else {
               for (i1=l_region1;i1<region_size[1];i1++){
                aux0 = (i1-offset[1])*region_size[0];
                for (i0=l_region0;i0<region_size[0];i0++){
                 tempcmplx = kdccomp(0.0,0.0);
   
                 index1 = i1;
                 for (k1=0; k1<hk; k1++) {
                  index0 = index1*region_size[0] + i0;
                  for (k0=0; k0<wk; k0++) {
                   tempcmplx = kdcadd(tempcmplx,
                               kdcmult(kcmplx_data[k1*wk + k0], 
                                       cmplx_data[index0--]));
                  }
                  index1--;
                 }
                 aux1 = aux0 + (i0-offset[0]);
                 cmplx_eval[aux1] = tempcmplx;
                }
               }
        }
        if (!kpds_put_data(dest, KPDS_VALUE_REGION, cmplx_eval)) {
         kerror(lib, rtn, "kpds_put_data failed for cmplx_eval.");
         (void)klist_free(objlist, (kfunc_void)lkcall_free);
         return(FALSE);
        }
       }
      }

  /*
   * Set up size of filtering output based on user selection.
   * size_type = 0 = truncate: dimensions of output are the same as input
   * size_type = 1 = extend: new dim = source dim + kernel dim - 1
   * We copy the KPDS_TIME_SIZE and KPDS_LOCATION_SIZE from the
   * source object into the destination to keep the integrity of these 
   * attributes after processing.
   *
   */
   if (!size_type) 
   {
     KCALL(kpds_set_attribute(dest, KPDS_VALUE_SIZE, w,h,d,t,e));
     if (kpds_query_time(src))
      KCALL(kpds_copy_time(src,dest,FALSE));
     if (kpds_query_location(src))
      KCALL(kpds_copy_location(src,dest,FALSE));
   }

  /*
   * Free memory.
   */
   if (bdata) kfree(bdata);
   if (beval) kfree(beval);
   if (ldata) kfree(ldata);
   if (leval) kfree(leval);
   if (ddata) kfree(ddata);
   if (deval) kfree(deval);
   if (cmplx_data) kfree(cmplx_data);
   if (cmplx_eval) kfree(cmplx_eval);
   if (dkdata) kfree(dkdata);
   if (bkdata) kfree(bkdata);
   if (lkdata) kfree(lkdata);
   if (kcmplx_data) kfree(kcmplx_data);

  /*
   * Make a copy of the remaining data from source data object
   * into destination object. We do that to keep the integrity
   * of the source object (concerning location and time segments
   * for instance).
   */
   KCALL(kpds_copy_remaining_data(src_ref,dest));
  /*
   * Free objects.
   */
   (void)klist_free(objlist, (kfunc_void)lkcall_free);
   return (TRUE);
}
/* -library_code_end */
