 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for khisto
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 	lkhisto
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/* -library_includes */
/* -library_includes_end */


/****************************************************************
* 
*  Routine Name: lkhisto - compute histograms for data objects
* 
*       Purpose: lkisto computes histogram information for data objects.
*                The unit of data to be used to compute each histogram can be
*                selected to lie along any combination of the axes 5D data 
*                model.
*
*                For example, it is possible to compute a histogram for each
*                HxD plane in a data set comprised of a sequence of multiband
*                volumes.
*
*                Output histograms are stored as columns in the output object,
*                one column per unit of data, in WHDTE order.
*
*                The output data type is UNSIGNED LONG.
*
*                If a gating object is supplied, then only input data that has
*                a corresponding non-zero value in the gating object is
*                histogrammed.
*
*                The out-of-bounds flags can be used to catch any counts that
*                lie outside of the histogram array.
*
*         Input: in_obj - object to be histogrammed
*                gate_obj - optional gate object (ROI)
*		 w,h,d,t,e - flags that, if set, indicate
*			that the processing unit for independent
*			histograms should include the given axis
*	         min - minimum side of most negative bin
*	         binwidth - width of each bin
*	         bins - number of bins
*	         oob - out-of-bounds flag; append two extra bins
*			   to the histogram to count those values that
*			   fall out of the histogram bounds
*
*        Output: kobject out_obj - contains histograms for the
*                       processing units in the data
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Scott Wilson
*          Date: Apr 08, 1995
*      Verified: 
*  Side Effects: 
* Modifications: 
****************************************************************/
/* -library_def */
int lkhisto(
  kobject in_obj,
  kobject gate_obj,
  int w, int h, int d, int t, int e,
  double min,
  double binwidth,
  int bins,
  int oob,
  kobject out_obj)
/* -library_def_end */

/* -library_code */
{
        char *lib = "kdatamanip", *rtn = "lkhisto";
	unsigned int i;
	unsigned long *hist; /* Histogram array pointer */
	unsigned wc,hc,dc,tc,ec; /* Dimensions of data set */
	unsigned int wu,hu,du,tu,eu; /* Dimensions of operation unit */
	unsigned int wr,hr,dr,tr,er,junk; /* Dimensions of optimal region */
	double *b=NULL,*bmask=NULL,*bgate=NULL; /* Pointers for data */
        int n,type,hnum;
	int numpts,numptsunit,numunits,numptsreg,numrgns,numu,numr;
        int map_exists,mask_exists;
	klist *objlist=NULL;

        /* Make sure we have valid objects */
        if (in_obj == KOBJECT_INVALID || out_obj == KOBJECT_INVALID)
          {
            kerror(lib, rtn, "Bogus input or output");
            return(FALSE);
          }

        /* Check out the histogram params to make sure they mean something. */
	if (bins <= 1 || binwidth <= 0)
          {
            kerror(lib, rtn, "Bogus histogram parameters");
            return(FALSE);
          }

        /* Reference the input object to avoid side effects, then add the
           reference to the list of goodies to be autmatically free'd on
           error. */
        KCALL((in_obj = kpds_reference_object(in_obj)));
	objlist = klist_add(objlist,in_obj,"KOBJECT");

        mask_exists=kpds_query_mask(in_obj);
        map_exists =kpds_query_map(in_obj);
        /* If ap exists, pull all the data thru the map */
        if (map_exists)
          {
            KCALL(kpds_set_attribute(in_obj,KPDS_MAPPING_MODE,KMAPPED));
            KCALL(kpds_destroy_map(out_obj));
          }

        /* Constrain the input data to double real for reading */
	KCALL(kpds_set_attributes(in_obj,
                                 KPDS_VALUE_DATA_TYPE, KDOUBLE,
                                 KPDS_VALUE_COMPLEX_CONVERT, KREAL, NULL));
        if (mask_exists) KCALL(kpds_set_attribute(in_obj,
                                 KPDS_MASK_DATA_TYPE,KDOUBLE));
 
	/* If gating object is supplied, reference it and SET it for KMAPPED 
           and KDOUBLE presentation. Also add the reference to the autofree 
           list. */
        if (gate_obj)
          {
            KCALL((gate_obj = kpds_reference_object(gate_obj))); 
	    objlist = klist_add(objlist,gate_obj,"KOBJECT");
            if (kpds_query_mask(gate_obj))
              {
                kerror(lib, rtn, "Gating object cannot contain mask data.");
                (void) kpds_close_object(in_obj);
                (void) kpds_close_object(gate_obj);
                return(FALSE);
              }
            if (kpds_query_map(gate_obj)) 
                KCALL(kpds_set_attribute(gate_obj,KPDS_MAPPING_MODE,KMAPPED));
            KCALL(kpds_set_attributes(gate_obj,
                                 KPDS_VALUE_DATA_TYPE, KDOUBLE,
                                 KPDS_VALUE_COMPLEX_CONVERT, KREAL, NULL));
          }

        /* See how big the data is */
	KCALL(kpds_get_attributes(in_obj,
                                 KPDS_VALUE_SIZE,&wc,&hc,&dc,&tc,&ec,
                                 KPDS_VALUE_DATA_TYPE,&type,NULL));

        /* Figure out what the optimal region size is that will still let us
           process by the unit we want */
        KCALL(kpds_get_attributes(in_obj,
                       KPDS_VALUE_OPTIMAL_REGION_SIZE,&wr,&hr,&dr,&tr,&er,&junk,
                       NULL));
        numpts = wc*hc*dc*tc*ec;
        numptsunit = 1;
	/* If NO flags are set, set them to WxH for default unit */
        if (w+h+d+t+e == 0) w=h=1;
        if (w) numptsunit*= wc; if (h) numptsunit*= hc; if (d) numptsunit*= dc;
        if (t) numptsunit*= tc; if (e) numptsunit*= ec;
        numunits = numpts/numptsunit;
        /* Set the in and out region sizes */
        wu = wc; hu= hc; du = dc; tu = tc; eu = ec;
        if (!w) wu=1; if (!h) hu=1; if (!d) du=1;
        if (!t) tu=1; if (!e) eu=1;
        /* Now figure out what the REAL optimal region size is */
        if (wu == 1) wr = 1; if (hu == 1) hr = 1; if (du == 1) dr = 1;
        if (tu == 1) tr = 1; if (eu == 1) er = 1;
        numptsreg = wr*hr*dr*tr*er;
        numrgns = numptsunit/numptsreg;
 
        /* Allocate the histogramming array and add it to the autofree list */
        hnum = bins;			/* Just the histogram */
        if (oob) hnum += 2;		/* Add out-of-bounds bins */
        KCALL(!((hist = (unsigned long *)kmalloc(hnum*sizeof(unsigned long))) == NULL));
	objlist = klist_add(objlist,hist,"KMALLOC");

        /* Create the output value segment if it's not there already and
           set it's size and data type attributes.  */
        if (!kpds_query_value(out_obj)) KCALL(kpds_create_value(out_obj));
        KCALL(kpds_set_attributes(out_obj,
                                 KPDS_VALUE_SIZE,numunits,hnum,1,1,1,
                                 KPDS_VALUE_DATA_TYPE,KULONG,NULL));

        /* Kill the output object mask,map,location, time segs if they exist */
        if (kpds_query_mask(out_obj)) KCALL(kpds_destroy_mask(out_obj));
        if (kpds_query_map(out_obj))  KCALL(kpds_destroy_map(out_obj));
        if (kpds_query_location(out_obj)) KCALL(kpds_destroy_location(out_obj));
        if (kpds_query_time(out_obj)) KCALL(kpds_destroy_time(out_obj));
          
        /* Ref the output object to avoid side effects and add it to the
           autofree list */
        KCALL((out_obj = kpds_reference_object(out_obj)));
	objlist = klist_add(objlist,out_obj,"KOBJECT");

        /* Set the region sizes on the input, gate, and output objects */
        KCALL(kpds_set_attribute(in_obj,
                                  KPDS_VALUE_REGION_SIZE,wr,hr,dr,tr,er));
        KCALL(kpds_set_attribute(out_obj,
                                  KPDS_VALUE_REGION_SIZE,1,hnum,1,1,1));
        if (gate_obj) KCALL(kpds_set_attribute(gate_obj,
                                  KPDS_VALUE_REGION_SIZE,wr,hr,dr,tr,er));

        /* If a mask is in place set the region size attributes appropriately */
	if (mask_exists) KCALL(kpds_set_attribute(in_obj,
                                   KPDS_MASK_REGION_SIZE,wr,hr,dr,tr,er));

        for (numu=0; numu<numunits; numu++)
        {
           /* Zero the histo array */
	   (void)kbzero((kaddr)hist,hnum*(unsigned long)sizeof(unsigned long));

           for (numr=0; numr<numrgns; numr++)
           {
             /* Grab a region of goodies from the input object */
	     b = (double *) kpds_get_data(in_obj,KPDS_VALUE_REGION, (kaddr)b);
	     if (gate_obj)  bgate = (double *)kpds_get_data(gate_obj,
                                              KPDS_VALUE_REGION, (kaddr)bgate);
             if (mask_exists) bmask = (double *)kpds_get_data(in_obj,
                                              KPDS_MASK_REGION,(kaddr)bmask);

             /* Form the histogram */
             if (!mask_exists && gate_obj==NULL)
               {
                 for (i=0; i<numptsreg; i++)
                   {
                     n = (b[i]-min)/binwidth;
                     if (b[i] == min+binwidth*bins) n = bins-1;
                     if (n>=0 && n<bins) hist[n]++;
		     if (oob)
                       { /* If out-of-bounds bins requested */
                         if (n < 0) hist[hnum-2]++;
                         if (n >= bins) hist[hnum-1]++;
                       }
                   }
               }
             else if (mask_exists && gate_obj==NULL)
               {
                 for (i=0; i<numptsreg; i++)
                   {
                     if (bmask[i] != 0)
                       {
                         n = (b[i]-min)/binwidth;
                         if (b[i] == min+binwidth*bins) n = bins-1;
                         if (n >= 0 && n <bins) hist[n]++;
		         if (oob)
                           { /* If out-of-bounds bins requested */
                             if (n < 0) hist[hnum-2]++;
                             if (n >= bins) hist[hnum-1]++;
                           }
                       }
                   }
               }
             else if (!mask_exists && gate_obj)
               {
                 for (i=0; i<numptsreg; i++)
                   {
                     if (bgate[i] != 0)
                       {
                         n = (b[i]-min)/binwidth;
                         if (b[i] == min+binwidth*bins) n = bins-1;
                         if (n >= 0 && n <bins) hist[n]++;
                         if (oob)
                           { /* If out-of-bounds bins requested */
                             if (n < 0) hist[hnum-2]++;
                             if (n >= bins) hist[hnum-1]++;
                           }
                       }
                   }
               }
             else if (mask_exists && gate_obj)
               {
                 for (i=0; i<numptsreg; i++)
                   {
                     if (bmask[i] != 0 && bgate[i] != 0)
                       {
                         n = (b[i]-min)/binwidth;
                         if (b[i] == min+binwidth*bins) n = bins-1;
                         if (n >= 0 && n <bins) hist[n]++;
                         if (oob)
                           { /* If out-of-bounds bins requested */
                             if (n < 0) hist[hnum-2]++;
                             if (n >= bins) hist[hnum-1]++;
                           }
                       }
                   }
               }
           }

           KCALL(kpds_put_data(out_obj, KPDS_VALUE_REGION, (kaddr)hist));
         }

        /* Free up everything that has been allocated */
	kfree(b);
	if (mask_exists) kfree(bmask);
	if (gate_obj) kfree(bgate);
	(void)klist_free(objlist, (kfunc_void)lkcall_free);

	return TRUE;
}
/* -library_code_end */
