 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for kgenloc
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 	lkgenloc
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/* -library_includes */
static int _construct_unif PROTO((kobject, double, double, double,
				  double, double, double));

static int _construct_rect PROTO((kobject, double, double, double, 
 				  double, double, double, int, int,
				  int, int));

static int _construct_curv PROTO((kobject, double, double, double,
				  double, double, double, int, int,
				  int, int));
/* -library_includes_end */


/****************************************************************
* 
*  Routine Name: lkgenloc - generate location data
* 
*       Purpose: This routine will generate explicit location 
*		 data for a specified object.  

*		 This routine will create a location segment of the
*		 specified width, height, and depth using the
*		 specified data type.  The dimension size of the
*		 created data is assumed to be three to correspond to
*		 (x,y,z).  It is assumed that the object initially
*		 does not have an explicit location segment.  
*
*		 The explicit location data is specified by three sets
*		 of begin and end values, which together compose
*		 explicit corner markers for volume in three space
*		 (x,y,z).  Typically, the x-axis corresponds uniformly
*		 to the implicit width dimension on the polymorphic
*		 data model, with the y-axis and z-axis corresponding
*		 to implicit height, and depth, respectively.
*
*		 Data is generated uniformly from those specified
*		 corner markers, however the explicitness of the
*		 generated data can be specified by the grid
*		 parameter.  A KUNIFORM grid will dictate that the
*		 generated data consist simply of setting the uniform
*		 corner markers on the output data set.  A
*		 KRECTILINEAR grid will dictate that the generated
*		 data consist of three arrays of locations, an
*		 x-array, a y-array, and a z-array, with the data
*		 within those arrays changing uniformly from the
*		 specified (x,y,z) begin and end values.  A
*		 KCURVILINEAR grid will dictate that the generated
*		 data consist of a four dimensional array of location
*		 data.  Essentially, this data set is a three
*		 dimensional array of (x,y,z) points.
*
*		 The data is generated and stored as one block, so
*		 generating large location data sets are subject to
*		 the memory limitations of your machine.
*
*         Input: xbegin   - the x-axis begin point
*                ybegin   - the y-axis begin point
*                zbegin   - the z-axis begin point
*                xend     - the x-axis end point
*                yend     - the y-axis end point
*                zend     - the z-axis end point
*                wid      - width size to use for the constructed data
*                hgt      - height size to use for the constructed data
*                dep      - depth size to use for the constructed data
*                datatype - data type to use for the constructed data
*
*        Output: dst_obj - the output object
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: Destination object must not contain location data
*    Written By: Steve Kubica
*          Date: Apr 08, 1995
*      Verified: 
*  Side Effects: 
* Modifications: 
****************************************************************/
/* -library_def */
int lkgenloc(
   double  xbegin,
   double  ybegin,
   double  zbegin,
   double  xend,
   double  yend,
   double  zend,
   int     grid,
   int     wid,
   int     hgt,
   int     dep,
   int     datatype,
   kobject dst_obj)
/* -library_def_end */

/* -library_code */
{
   char *lib  = "kdatamanip";
   char *prog = "lkgenloc";
   int   status;
   
   /* -- set the grid type in the destination object -- */
   if (!kpds_set_attribute(dst_obj, KPDS_LOCATION_GRID, grid))
   {
      kerror(lib, prog, "Unable to set location grid on destination object.");
      return FALSE;
   }

   /* -- create the location segment in the destination -- */
   if (!kpds_create_location(dst_obj))
   {
      kerror(lib, prog, 
	     "Unable to create location segment in destination object.");
      return FALSE;
   }

   /* -- set the size and data type as specified -- */
   if (!kpds_set_attributes(dst_obj,
			    KPDS_LOCATION_SIZE, wid, hgt, dep, 3,
			    KPDS_LOCATION_DATA_TYPE, datatype,
			    NULL))
   {
      kerror(lib, prog, 
	     "Unable to set the size or data type of the location segment.");
      return FALSE;
   }
   
   /* -- create location data according to the appropriate grid type -- */
   switch (grid)
   {
      case KUNIFORM :
      {
	 status = _construct_unif(dst_obj, 
				  xbegin, ybegin, zbegin, xend, yend, zend);
      } break;
	 
      case KRECTILINEAR :
      {
	 status = _construct_rect(dst_obj, 
				  xbegin, ybegin, zbegin, xend, yend, zend,
				  wid, hgt, dep, datatype);
      } break;

      case KCURVILINEAR :
      {
	 status = _construct_curv(dst_obj, 
				  xbegin, ybegin, zbegin, xend, yend, zend,
				  wid, hgt, dep, datatype);
      } break;
	
      default :
	 kerror(lib, prog, 
		"Attempt to create data of invalid grid type %d", grid);
	 return FALSE;
   }
   
   return status;
}

/*-----------------------------------------------------------
|
|  Routine Name: _construct_unif - construct uniform location data
|
|       Purpose: This function sets the uniform location begin
|		 and end points.
|	
|		 This function assumes that the location segment
|		 has already been created and initialized to the
|		 correct size and data type.  
|
|         Input: obj      - object with location segment to store in 
|                xbegin   - x axis begin point
|                ybegin   - y axis begin point
|                zbegin   - z axis begin point
|                xend     - x axis end point
|                yend     - y axis end point
|                zend     - z axis end point
|
|        Output: obj - location attributes are set on obj
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica
|          Date: Wed Apr 5 1995
| Modifications:
|
------------------------------------------------------------*/
static int 
_construct_unif(
   kobject obj,
   double  xbegin,
   double  ybegin,
   double  zbegin,
   double  xend,
   double  yend,
   double  zend)
{
   char *lib  = "kdatamanip";
   char *prog = "_construct_unif";

   if (!kpds_set_attribute(obj, KPDS_LOCATION_BEGIN, xbegin, ybegin, zbegin))
   {
      kerror(lib, prog, 
	     "Unable to set destination uniform location begin point");
      return FALSE;
   }

   if (!kpds_set_attribute(obj, KPDS_LOCATION_END, xend, yend, zend))
   {
      kerror(lib, prog, 
	     "Unable to set destination uniform location end point");
      return FALSE;
   }

   return TRUE;
}

/*-----------------------------------------------------------
|
|  Routine Name: _construct_curv - construct rectilinear location data
|
|       Purpose: This function constructs axes of rectilinear
|		 data whose values range uniformly between the
|		 specified x, y, and z begin and end points.
|	
|		 This function assumes that the location segment
|		 has already been created and initialized to the
|		 correct size and data type.  The generated
|		 location data will be stored with width all, 
|		 height all, and depth all primitives.
|
|         Input: obj      - object with location segment to store in 
|                xbegin   - x axis begin point
|                ybegin   - y axis begin point
|                zbegin   - z axis begin point
|                xend     - x axis end point
|                yend     - y axis end point
|                zend     - z axis end point
|                wid      - width size for the constructed data
|                hgt      - height size for the constructed data
|                dep      - depth size for the constructed data
|                datatype - data type for the constructed data
|
|        Output: obj - constructed location is stored in obj
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica
|          Date: Wed Apr 5 1995
| Modifications:
|
------------------------------------------------------------*/
static int 
_construct_rect(
   kobject obj,
   double  xbegin,
   double  ybegin,
   double  zbegin,
   double  xend,
   double  yend,
   double  zend,
   int     wid,
   int     hgt,
   int     dep,
   int     datatype)
{
   kaddr  wdata = NULL;
   kaddr  hdata = NULL;
   kaddr  ddata = NULL;
   double del[3];
   int    i;
   char  *lib  = "kdatamanip";
   char  *prog = "_construct_rect";
   
   
   /* -- allocate sufficient data space -- */
   wdata = kmalloc(wid * kdata_size(datatype));
   hdata = kmalloc(hgt * kdata_size(datatype));
   ddata = kmalloc(dep * kdata_size(datatype));

   /* -- determine the deltas between adjacent points (no 0 deltas!) -- */
   del[0] = (xend - xbegin)/(double) ((wid-1 == 0) ? 1 : (wid-1));
   del[1] = (yend - ybegin)/(double) ((hgt-1 == 0) ? 1 : (hgt-1));
   del[2] = (zend - zbegin)/(double) ((dep-1 == 0) ? 1 : (dep-1));

   switch (datatype)
   {
      case KBYTE :
      {
	 char *wd = (char *) wdata;
	 char *hd = (char *) hdata;
	 char *dd = (char *) ddata;

	 for (i = 0; i < wid; i++)
	    wd[i] = (char) xbegin + del[0]*i;
	 for (i = 0; i < hgt; i++)
	    hd[i] = (char) ybegin + del[1]*i;
	 for (i = 0; i < dep; i++)
	    dd[i] = (char) zbegin + del[2]*i;
      } break;

      case KUBYTE :
      {
	 unsigned char *wd = (unsigned char *) wdata;
	 unsigned char *hd = (unsigned char *) hdata;
	 unsigned char *dd = (unsigned char *) ddata;

	 for (i = 0; i < wid; i++)
	    wd[i] = (unsigned char) xbegin + del[0]*i;
	 for (i = 0; i < hgt; i++)
	    hd[i] = (unsigned char) ybegin + del[1]*i;
	 for (i = 0; i < dep; i++)
	    dd[i] = (unsigned char) zbegin + del[2]*i;
      } break;

      case KSHORT :
      {
	 short *wd = (short *) wdata;
	 short *hd = (short *) hdata;
	 short *dd = (short *) ddata;

	 for (i = 0; i < wid; i++)
	    wd[i] = (short) xbegin + del[0]*i;
	 for (i = 0; i < hgt; i++)
	    hd[i] = (short) ybegin + del[1]*i;
	 for (i = 0; i < dep; i++)
	    dd[i] = (short) zbegin + del[2]*i;
      } break;

      case KUSHORT :
      {
	 unsigned short *wd = (unsigned short *) wdata;
	 unsigned short *hd = (unsigned short *) hdata;
	 unsigned short *dd = (unsigned short *) ddata;

	 for (i = 0; i < wid; i++)
	    wd[i] = (unsigned short) xbegin + del[0]*i;
	 for (i = 0; i < hgt; i++)
	    hd[i] = (unsigned short) ybegin + del[1]*i;
	 for (i = 0; i < dep; i++)
	    dd[i] = (unsigned short) zbegin + del[2]*i;
      } break;

      case KINT :
      {
	 int *wd = (int *) wdata;
	 int *hd = (int *) hdata;
	 int *dd = (int *) ddata;

	 for (i = 0; i < wid; i++)
	    wd[i] = (int) xbegin + del[0]*i;
	 for (i = 0; i < hgt; i++)
	    hd[i] = (int) ybegin + del[1]*i;
	 for (i = 0; i < dep; i++)
	    dd[i] = (int) zbegin + del[2]*i;
      } break;

      case KUINT :
      {
	 unsigned int *wd = (unsigned int *) wdata;
	 unsigned int *hd = (unsigned int *) hdata;
	 unsigned int *dd = (unsigned int *) ddata;

	 for (i = 0; i < wid; i++)
	    wd[i] = (unsigned int) xbegin + del[0]*i;
	 for (i = 0; i < hgt; i++)
	    hd[i] = (unsigned int) ybegin + del[1]*i;
	 for (i = 0; i < dep; i++)
	    dd[i] = (unsigned int) zbegin + del[2]*i;
      } break;

      case KLONG :
      {
	 long *wd = (long *) wdata;
	 long *hd = (long *) hdata;
	 long *dd = (long *) ddata;

	 for (i = 0; i < wid; i++)
	    wd[i] = (long) xbegin + del[0]*i;
	 for (i = 0; i < hgt; i++)
	    hd[i] = (long) ybegin + del[1]*i;
	 for (i = 0; i < dep; i++)
	    dd[i] = (long) zbegin + del[2]*i;
      } break;

      case KULONG :
      {
	 unsigned long *wd = (unsigned long *) wdata;
	 unsigned long *hd = (unsigned long *) hdata;
	 unsigned long *dd = (unsigned long *) ddata;

	 for (i = 0; i < wid; i++)
	    wd[i] = (unsigned long) xbegin + del[0]*i;
	 for (i = 0; i < hgt; i++)
	    hd[i] = (unsigned long) ybegin + del[1]*i;
	 for (i = 0; i < dep; i++)
	    dd[i] = (unsigned long) zbegin + del[2]*i;
      } break;

      case KFLOAT :
      {
	 float *wd = (float *) wdata;
	 float *hd = (float *) hdata;
	 float *dd = (float *) ddata;

	 for (i = 0; i < wid; i++)
	    wd[i] = (float) xbegin + del[0]*i;
	 for (i = 0; i < hgt; i++)
	    hd[i] = (float) ybegin + del[1]*i;
	 for (i = 0; i < dep; i++)
	    dd[i] = (float) zbegin + del[2]*i;
      } break;

      case KDOUBLE :
      {
	 double *wd = (double *) wdata;
	 double *hd = (double *) hdata;
	 double *dd = (double *) ddata;

	 for (i = 0; i < wid; i++)
	    wd[i] = (double) xbegin + del[0]*i;
	 for (i = 0; i < hgt; i++)
	    hd[i] = (double) ybegin + del[1]*i;
	 for (i = 0; i < dep; i++)
	    dd[i] = (double) zbegin + del[2]*i;
      } break;

      case KCOMPLEX :
      {
	 kcomplex *wd = (kcomplex *) wdata;
	 kcomplex *hd = (kcomplex *) hdata;
	 kcomplex *dd = (kcomplex *) ddata;

	 for (i = 0; i < wid; i++)
	 {
	    wd[i].r = (float) xbegin + del[0]*i;
	    wd[i].i = (float) 0.0;
	 }
	 for (i = 0; i < hgt; i++)
	 {
	    hd[i].r = (float) ybegin + del[1]*i;
	    hd[i].i = (float) 0.0;
	 }
	 for (i = 0; i < dep; i++)
	 {
	    dd[i].r = (float) zbegin + del[2]*i;
	    dd[i].i = (float) 0.0;
	 }
      }

      case KDCOMPLEX :
      {
	 kdcomplex *wd = (kdcomplex *) wdata;
	 kdcomplex *hd = (kdcomplex *) hdata;
	 kdcomplex *dd = (kdcomplex *) ddata;

	 for (i = 0; i < wid; i++)
	 {
	    wd[i].r = (double) xbegin + del[0]*i;
	    wd[i].i = (double) 0.0;
	 }
	 for (i = 0; i < hgt; i++)
	 {
	    hd[i].r = (double) ybegin + del[1]*i;
	    hd[i].i = (double) 0.0;
	 }
	 for (i = 0; i < dep; i++)
	 {
	    dd[i].r = (double) zbegin + del[2]*i;
	    dd[i].i = (double) 0.0;
	 }
      } break;

      default :
	 kerror(lib, prog, "Invalid data type (%d)", datatype);
	 return FALSE;
   }
   
   if (!kpds_put_data(obj, KPDS_LOCATION_WIDTH_ALL, wdata))
   {
      kerror(lib, prog, 
	     "Unable to store destination rectilinear location width data");
      return FALSE;
   }

   if (!kpds_put_data(obj, KPDS_LOCATION_HEIGHT_ALL, hdata))
   {
      kerror(lib, prog, 
	     "Unable to store destination rectilinear location height data");
      return FALSE;
   }

   if (!kpds_put_data(obj, KPDS_LOCATION_DEPTH_ALL, ddata))
   {
      kerror(lib, prog, 
	     "Unable to store destination rectilinear location depth data");
      return FALSE;
   }

   return TRUE;
}

/*-----------------------------------------------------------
|
|  Routine Name: _construct_curv - construct curvilinear location data
|
|       Purpose: This function constructs a block of curvilinear 
|		 data whose values range uniformly between the
|		 specified x, y, and z begin and end points.
|	
|		 This function assumes that the location segment
|		 has already been created and initialized to the
|		 correct size and data type.  The generated
|		 location data will be put with a location all 
|		 primitive.
|
|         Input: obj      - object with location segment to store in 
|                xbegin   - x axis begin point
|                ybegin   - y axis begin point
|                zbegin   - z axis begin point
|                xend     - x axis end point
|                yend     - y axis end point
|                zend     - z axis end point
|                wid      - width size for the constructed data
|                hgt      - height size for the constructed data
|                dep      - depth size for the constructed data
|                datatype - data type for the constructed data
|
|        Output: obj - constructed location is stored in obj
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica
|          Date: Wed Apr 5 1995
| Modifications:
|
------------------------------------------------------------*/
static int 
_construct_curv(
   kobject obj,
   double  xbegin,
   double  ybegin,
   double  zbegin,
   double  xend,
   double  yend,
   double  zend,
   int     wid,
   int     hgt,
   int     dep,
   int     datatype)
{
   kaddr  ldata = NULL;
   double del[3];
   double ubeg[3];
   int    k[3];
   int    n;
   char  *lib  = "kdatamanip";
   char  *prog = "_construct_curv";


   /* -- allocate sufficient data space -- */
   ldata = kmalloc(wid*hgt*dep*3 * kdata_size(datatype));

   /* -- determine the deltas between adjacent points (no 0 deltas!) -- */
   del[0] = (xend - xbegin)/(double) ((wid-1 == 0) ? 1 : (wid-1));
   del[1] = (yend - ybegin)/(double) ((hgt-1 == 0) ? 1 : (hgt-1));
   del[2] = (zend - zbegin)/(double) ((dep-1 == 0) ? 1 : (dep-1));

   /* -- this information will be much easier to manage in arrays -- */
   ubeg[0] = xbegin;
   ubeg[1] = ybegin;
   ubeg[2] = zbegin;

   switch (datatype)
   {
      case KBYTE :
      {
	 char *data = (char *) ldata;

         for (n = 0; n < 3; n++)
            for (k[2] = 0; k[2] < dep; k[2]++)
               for (k[1] = 0; k[1] < hgt; k[1]++)
                  for (k[0] = 0; k[0] < wid; k[0]++)
                     *data++ = (char) ubeg[n] + del[n] * k[n];
      } break;

      case KUBYTE :
      {
	 unsigned char *data = (unsigned char *) ldata;

         for (n = 0; n < 3; n++)
            for (k[2] = 0; k[2] < dep; k[2]++)
               for (k[1] = 0; k[1] < hgt; k[1]++)
                  for (k[0] = 0; k[0] < wid; k[0]++)
                     *data++ = (unsigned char) ubeg[n] + del[n] * k[n];
      } break;

      case KSHORT :
      {
	 short *data = (short *) ldata;

         for (n = 0; n < 3; n++)
            for (k[2] = 0; k[2] < dep; k[2]++)
               for (k[1] = 0; k[1] < hgt; k[1]++)
                  for (k[0] = 0; k[0] < wid; k[0]++)
                     *data++ = (short) ubeg[n] + del[n] * k[n];
      } break;

      case KUSHORT :
      {
	 unsigned short *data = (unsigned short *) ldata;

         for (n = 0; n < 3; n++)
            for (k[2] = 0; k[2] < dep; k[2]++)
               for (k[1] = 0; k[1] < hgt; k[1]++)
                  for (k[0] = 0; k[0] < wid; k[0]++)
                     *data++ = (unsigned short) ubeg[n] + del[n] * k[n];
      } break;

      case KINT :
      {
	 int *data = (int *) ldata;

         for (n = 0; n < 3; n++)
            for (k[2] = 0; k[2] < dep; k[2]++)
               for (k[1] = 0; k[1] < hgt; k[1]++)
                  for (k[0] = 0; k[0] < wid; k[0]++)
                     *data++ = (int) ubeg[n] + del[n] * k[n];
      } break;

      case KUINT :
      {
	 unsigned int *data = (unsigned int *) ldata;

         for (n = 0; n < 3; n++)
            for (k[2] = 0; k[2] < dep; k[2]++)
               for (k[1] = 0; k[1] < hgt; k[1]++)
                  for (k[0] = 0; k[0] < wid; k[0]++)
                     *data++ = (unsigned int) ubeg[n] + del[n] * k[n];
      } break;

      case KLONG :
      {
	 long *data = (long *) ldata;

         for (n = 0; n < 3; n++)
            for (k[2] = 0; k[2] < dep; k[2]++)
               for (k[1] = 0; k[1] < hgt; k[1]++)
                  for (k[0] = 0; k[0] < wid; k[0]++)
                     *data++ = (long) ubeg[n] + del[n] * k[n];
      } break;

      case KULONG :
      {
	 unsigned long *data = (unsigned long *) ldata;

         for (n = 0; n < 3; n++)
            for (k[2] = 0; k[2] < dep; k[2]++)
               for (k[1] = 0; k[1] < hgt; k[1]++)
                  for (k[0] = 0; k[0] < wid; k[0]++)
                     *data++ = (unsigned long) ubeg[n] + del[n] * k[n];
      } break;

      case KFLOAT :
      {
	 float *data = (float *) ldata;

         for (n = 0; n < 3; n++)
            for (k[2] = 0; k[2] < dep; k[2]++)
               for (k[1] = 0; k[1] < hgt; k[1]++)
                  for (k[0] = 0; k[0] < wid; k[0]++)
                     *data++ = (float) ubeg[n] + del[n] * k[n];
      } break;

      case KDOUBLE :
      {
	 double *data = (double *) ldata;

         for (n = 0; n < 3; n++)
            for (k[2] = 0; k[2] < dep; k[2]++)
               for (k[1] = 0; k[1] < hgt; k[1]++)
                  for (k[0] = 0; k[0] < wid; k[0]++)
                     *data++ = (double) ubeg[n] + del[n] * k[n];
      } break;

      case KCOMPLEX :
      {
	 kcomplex *data = (kcomplex *) ldata;

         for (n = 0; n < 3; n++)
            for (k[2] = 0; k[2] < dep; k[2]++)
               for (k[1] = 0; k[1] < hgt; k[1]++)
                  for (k[0] = 0; k[0] < wid; k[0]++)
		  {
                     (*data).r = (float) ubeg[n] + del[n] * k[n];
                     (*data).i = (float) 0.0;
		     data++;
		  }
      } break;

      case KDCOMPLEX :
      {
	 kdcomplex *data = (kdcomplex *) ldata;

         for (n = 0; n < 3; n++)
            for (k[2] = 0; k[2] < dep; k[2]++)
               for (k[1] = 0; k[1] < hgt; k[1]++)
                  for (k[0] = 0; k[0] < wid; k[0]++)
		  {
                     (*data).r = (double) ubeg[n] + del[n] * k[n];
                     (*data).i = (double) 0.0;
		     data++;
		  }
      } break;

      default:
	 kerror(lib, prog, "Invalid data type (%d)", datatype);
	 return FALSE;
   }

   return kpds_put_data(obj, KPDS_LOCATION_ALL, ldata);
}
/* -library_code_end */
