 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for kextract
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 	lkextract
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/* -library_includes */
/* -library_includes_end */


/****************************************************************
* 
*  Routine Name: lkextract - extract rectangular region from data object
* 
*       Purpose: Library Routine for kextract.
*
*		 Lkextract is based on implicit indexing.  This means
*		 that if location or time data exist, the extract 
*		 operation is not done in terms an interpretation of 
*		 the location/time data values, but in terms of the 
*		 implicit indexing of these data (which is specified 
*		 by the width, height, depth, time, elements indices 
*		 of the polymorphic data model).
*
*		 Lkextract extracts from the source data object (src_obj) 
*		 a region specified by the w_size, h_size, d_size, t_size, 
*		 and e_size parameters and places the extracted data 
*		 into the corresponding segment(s) in the destination 
*		 object (dst_obj).  The region will be extracted starting 
*		 at the point defined by (w_off, h_off, d_off, t_off, 
*		 e_off).  This point will be the origin (0,0,0,0,0) 
*		 of the destination object.  The size attributes of the 
*		 destination object's segment(s) are set according to 
*		 the <dimension>_size input parameters. 
*
*		 If the source data object has mask, location, time, 
*		 and/or value data segments, the specified region is 
*		 extracted from each existing segment in the source and 
*		 placed in the corresponding segment of the destination 
*		 object.  If the segment does not already exist in the 
*		 destination object, it will be created.
*
*		 If the offset (defined by w_off, h_off, d_off, t_off, 
*		 e_off) and/or the extracted region size, fall outside 
*		 the indexing bounds, or size, of the data segments, the 
*		 object segments will be padded with the values pad value 
*		 attribute of the source object, which can be set prior
*		 to calling lkextract.
*
*		 If the destination object contains segments not present 
*		 in the source object, they will automatically be resized 
*		 according to the size input parameters (to maintain the 
*		 integrety of the data with respect to the polymorphic data 
*		 model).  The offset will be the origin (0,0,0,0,0), not the.  
*		 offset values passed in.  If padding of these segments is 
*		 necessary, the padding values will be determined from the 
*		 pad attribute of the destination object, which can be set 
*		 prior to calling lkextract.
*
*		 If the subpos parameter is TRUE, lkextract sets the 
*		 KPDS_SUBOBJECT_POSITION attribute of the destination 
*		 object to the subregion origin coordinates 
*		 (w_off, h_off, d_off, t_off, e_off).  If subpos is FALSE, 
*		 the KPDS_SUBOBJECT_POSITION is set to (0,0,0,0,0).
*
*		 Error Checks: The size parameters passed in must be 
*		 greater than zero, lkextract will fail if they are
*		 not.  Lkextract will also fail if the source and 
*		 destination objects passed in are not valid.  
*		 Any offset value is legal.
*
*
*         Input: src_obj - the source data object to be processed
*		 w_off   - origin of region on width dimension
*		 h_off   - origin of region on height dimension
*		 d_off   - origin of region on depth dimension
*		 t_off   - origin of region on time dimension
*		 e_off   - origin of region on elemens dimension
*		 w_size  - region size on width dimension
*		 h_size  - region size on height dimension
*		 d_size  - region size on depth dimension
*		 t_size  - region size on time dimension
*		 e_size  - region size on elements dimension
*		 subpos  - if TRUE, set dst_obj KPDS_SUBOBJECT_POSITION
*			   attribute to (w_off,h_off,d_off,t_off,e_off).  
*			   Otherwise set it to (0,0,0,0,0).
*
*        Output: dst_obj - the destination data object
*
*       Returns: TRUE (1) on success, FALSE (0) on failure
*  Restrictions: Only operates on value, mask, location, and time segments.
*    Written By: Donna Koechner and Mark Young
*          Date: Apr 08, 1995
*      Verified: 
*  Side Effects: 
* Modifications: 
****************************************************************/
/* -library_def */
int lkextract(
   kobject src_obj,
   int    w_off,
   int    h_off,
   int    d_off,
   int    t_off,
   int    e_off,
   int    w_size,
   int    h_size,
   int    d_size,
   int    t_size,
   int    e_size,
   int    subpos,
   kobject dst_obj)
/* -library_def_end */

/* -library_code */
{
	/*
	 *  The variable "data" is declared as a khoros generic data pointer, 
	 *  kaddr.  It is declared this way because we do not know the data 
	 *  type of our data, and since this routine will not directly 
	 *  manipulate the data, we don't really care.  
	 */
	kaddr 	data = NULL;

        int     src_has_mask, src_has_value, src_has_location, src_has_time;
        int     dst_has_mask, dst_has_value, dst_has_location, dst_has_time;
	int	rw=1, rh=1, rd=1, rt=1, re=1, numrgns=1;
	int	temp=1, loc_dim=1, i;
	int     grid;
	char 	*funct = "lkextract";
	char 	*lib = "kdatamanip";

        src_has_mask = src_has_value = src_has_location = src_has_time = FALSE;
        dst_has_mask = dst_has_value = dst_has_location = dst_has_time = FALSE;

	
	/* 
	 *  Check to ensure that valid data objects were passed in.  
	 */
	if (src_obj == KOBJECT_INVALID) 
	{
           kerror(lib,funct,"src_obj not a valid object");
	   return(FALSE);
	}
	if (dst_obj == KOBJECT_INVALID) 
	{
           kerror(lib,funct,"dst_obj not a valid object");
	   return(FALSE);
	}


	/* 
	 *  Check to ensure that the region size parameters are valid.
	 */
	if (w_size <= 0 || h_size <= 0 || d_size <= 0 || t_size <= 0 || 
	   t_size <= 0 ) 
	{
           kerror(lib,funct, "All size parameters must be greater than zero.");
	   return(FALSE);
	}


	/*
	 *  Find out which segments are available in source object.
	 */
	if (kpds_query_value(src_obj))     src_has_value    = TRUE;
	if (kpds_query_mask(src_obj))      src_has_mask     = TRUE;
	if (kpds_query_location(src_obj))  src_has_location = TRUE;
	if (kpds_query_time(src_obj))      src_has_time     = TRUE;

	/*
	 *  Find out which segments are available in destination object.
	 */
	if (kpds_query_value(dst_obj))     dst_has_value    = TRUE;
	if (kpds_query_mask(dst_obj))      dst_has_mask     = TRUE;
	if (kpds_query_location(dst_obj))  dst_has_location = TRUE;
	if (kpds_query_time(dst_obj))      dst_has_time     = TRUE;


	/*
	 *  Reference the source object so that the library does not 
	 *  have any side effects on the source object.
	 */
	if ((src_obj = kpds_reference_object(src_obj)) == KOBJECT_INVALID) 
	{
	   kerror(lib,funct,"Failed to create source reference object.");
	   return(FALSE);
	}

	/*
	 *  Set the size attribute on all segments of the destination 
	 *  (dst_obj) object.  If segment does not exist, it must be 
	 *  created first.  
	 */

	/* VALUE SEGMENT */
	if (src_has_value) 
	{
	   if (!dst_has_value)
	   {
	      if (!kpds_create_value(dst_obj))
	      {
	         kerror(lib,funct,"Failed to create destination value seg.");
		 kpds_close_object(src_obj);
	         return(FALSE);
	      }
	      dst_has_value = TRUE;
	   }
	   if (!kpds_set_attribute(dst_obj, 
		KPDS_VALUE_SIZE, w_size, h_size, d_size, t_size, e_size))
	   {
	      kerror(lib,funct,"Unable to set destination object value size.");
	      kpds_close_object(src_obj);
	      return(FALSE);
	   }
	}

	/* MASK SEGMENT */
	if (src_has_mask) 
	{
	   if (!dst_has_mask)
	   {
	      if (!kpds_create_mask(dst_obj))
	      {
	         kerror(lib,funct,"Failed to create destination mask.");
		 kpds_close_object(src_obj);
	         return(FALSE);
	      }
	      dst_has_mask = TRUE;
	   }
	   if (!kpds_set_attribute(dst_obj,
		KPDS_MASK_SIZE, w_size, h_size, d_size, t_size, e_size))
	   {
	      kerror(lib,funct,"Unable to set destination object mask size.");
	      kpds_close_object(src_obj);
	      return(FALSE);
	   }
	}

	/* LOCATION SEGMENT */
	if (src_has_location) 
	{
	   /* Get the grid type from the source object */
	   if (!kpds_get_attribute(src_obj, KPDS_LOCATION_GRID, &grid))
	   {
	      kerror(lib,funct,
		     "Failed to retreive source location grid type.");
	      kpds_close_object(src_obj);
	      return(FALSE);
	   }

	   if (!dst_has_location)
	   {
	      if (!kpds_set_attribute(dst_obj, KPDS_LOCATION_GRID, grid) ||
		  !kpds_create_location(dst_obj))
	      {
	         kerror(lib,funct,"Failed to create destination location.");
		 kpds_close_object(src_obj);
	         return(FALSE);
	      }
	      dst_has_location = TRUE;
	   }
	   /*
	    *  Since the location dimension is not one of the specified
	    *  sizes, need to get that attribute from the source so that 
	    *  we can use it when setting the size.
	    */
	   if (!kpds_get_attribute(src_obj,
		KPDS_LOCATION_SIZE, &temp, &temp, &temp, &loc_dim))
	   {
	      kerror(lib,funct,"Unable to get source object location size.");
	      kpds_close_object(src_obj);
	      return(FALSE);
	   }
	   if (!kpds_set_attribute(dst_obj,
		KPDS_LOCATION_SIZE, w_size, h_size, d_size, loc_dim))
	   {
	      kerror(lib,funct,"Unable to set destination location size.");
	      kpds_close_object(src_obj);
	      return(FALSE);
	   }
	}

	/* TIME SEGMENT */
	if (src_has_time) 
	{
	   if (!dst_has_time)
	   {
	      if (!kpds_create_time(dst_obj))
	      {
	         kerror(lib,funct,"Failed to create destination time.");
		 kpds_close_object(src_obj);
	         return(FALSE);
	      }
	      dst_has_time = TRUE;
	   }
	   if (!kpds_set_attribute(dst_obj, KPDS_TIME_SIZE, t_size))
	   {
	      kerror(lib,funct,"Unable to set destination time size.");
	      kpds_close_object(src_obj);
	      return(FALSE);
	   }
	}


	/*
	 *  Except for the SIZE attribute, which was just set, we do not wish 
	 *  to change any attributes of the destination object.  This ensures 
	 *  that the library will not have any side effect on the destination 
	 *  object).  To accomplish this, reference the destination object, 
	 *  and set any other attributes on that object.
	 */
	if ((dst_obj = kpds_reference_object(dst_obj)) == KOBJECT_INVALID) 
	{
           kerror(lib,funct,
		"Failed to create reference for destination object.");
	   kpds_close_object(src_obj);
           return(FALSE);
        }

	/*
	 *  Now set attributes on segments of the source reference 
	 *  (src_obj) and the destination reference (dst_obj) objects.  
	 *
	 *  OFFSET & POSITION ATTRIBUTES:  Set the offset on the source 
	 *  reference object to be the starting point defined by the *_off 
	 *  parameters, and set position to the origin.  Set the destination 
	 *  reference object position and offset to the origin.
	 *  
	 *  MOVING DATA:  Use optimal regions to enable processing of very
	 *  When copying the data, the KPDS_<segment>_OPTIMAL_REGION_SIZE 
	 *  attribute and the KPDS_<segment>_REGION primitive are used to 
	 *  support processing of very large objects.  This requires a get 
	 *  attribute call to get the number of optimal regions, and a set 
	 *  attributes call to set region size and the increment size.  
	 *  Increment size will define the bounds of the total region being 
	 *  extracted.  Setting up constrained optimal regions such as this 
	 *  will probably change in the future - hopefully it will become 
	 *  easier to program to.  
	 *
	 *  Next, get the region to be extracted from the source object 
	 *  data, and put it in the destination object using the 
	 *  kpds_get_data and kpds_put_data function calls.  
	 */

	/*
	 *  VALUE SEGMENT
	 */
	if (src_has_value && dst_has_value) 
	{
	   /* 
	    *  Set the reference destination object datatype to be the same
	    *  as the source object type.  Then get the optimal region size
	    *  from the destination object.
	    */
	   if (!kpds_copy_attribute(src_obj, dst_obj, KPDS_VALUE_DATA_TYPE))
           {
              kerror(lib,funct,"Unable to copy data type from src to dst.");
              kpds_close_object(src_obj); kpds_close_object(dst_obj);
              return(FALSE);
           }

	   if (!kpds_get_attribute(dst_obj, KPDS_VALUE_OPTIMAL_REGION_SIZE,
				   &rw, &rh, &rd, &rt, &re, &numrgns))
           {
              kerror(lib,funct,"Failed to get value optimal region size.");
              kpds_close_object(src_obj); kpds_close_object(dst_obj);
              return(FALSE);
           }

	   /*  
	    *  Set the optimal region size and offsets on the source and 
	    *  destination objects.
	    */
	   if (!kpds_set_attributes(src_obj,
		KPDS_VALUE_REGION_SIZE, rw, rh, rd, rt, re,
		KPDS_VALUE_INTERPOLATE, KPAD,
		KPDS_VALUE_POSITION, 0, 0, 0, 0, 0,
		KPDS_VALUE_INCREMENT_SIZE, w_size,h_size,d_size,t_size,e_size,
		KPDS_VALUE_OFFSET, w_off, h_off, d_off, t_off, e_off, NULL))
           {
              kerror(lib,funct,"Failed to set source value attributes.");
              kpds_close_object(src_obj); kpds_close_object(dst_obj);
              return(FALSE);
           }
/* uncomment for debugging
kpds_print_attribute(src_obj, KPDS_VALUE_POSITION, kstdout);
kinfo(KSTANDARD, " <-- KPDS_VALUE_POSITION");
*/

	   if (!kpds_set_attributes(dst_obj,
		KPDS_VALUE_REGION_SIZE, rw, rh, rd, rt, re,
		KPDS_VALUE_POSITION, 0, 0, 0, 0, 0,
		KPDS_VALUE_OFFSET, 0, 0, 0, 0, 0, NULL))
           {
              kerror(lib,funct,"Failed to set destination value attributes.");
              kpds_close_object(src_obj); kpds_close_object(dst_obj);
              return(FALSE);
           }
	   for (i = 0; i<numrgns; i++) 
	   {
	      if ((data=kpds_get_data(src_obj,KPDS_VALUE_REGION,data)) == NULL)
              {
                 kerror(lib,funct,"get_data failed for value on region %d.",i);
                 kpds_close_object(src_obj); kpds_close_object(dst_obj);
                 return(FALSE);
              }
	      if (!kpds_put_data(dst_obj, KPDS_VALUE_REGION, data))
              {
                 kerror(lib,funct,"put_data failed for value on region %d.",i);
                 kpds_close_object(src_obj); kpds_close_object(dst_obj);
                 return(FALSE);
              }
	   }
           kfree(data);
	   data = NULL;
	}

	/*
	 *  MASK SEGMENT
	 */
	if (src_has_mask && dst_has_mask) 
	{
           /*
            *  Set the reference destination object datatype to be the same
            *  as the source object type.  Then get the optimal region size
            *  from the destination object.
            */
           if (!kpds_copy_attribute(src_obj, dst_obj, KPDS_MASK_DATA_TYPE))
           {
              kerror(lib,funct,"Unable to copy mask data type from src to dst");
              kpds_close_object(src_obj); kpds_close_object(dst_obj);
              return(FALSE);
           }
 
           if (!kpds_get_attribute(dst_obj, KPDS_MASK_OPTIMAL_REGION_SIZE,
                                   &rw, &rh, &rd, &rt, &re, &numrgns))
           {
              kerror(lib,funct,"Failed to get mask optimal region size.");
              kpds_close_object(src_obj); kpds_close_object(dst_obj);
              return(FALSE);
           }
 
           /*
            *  Set the optimal region size and offsets on the source and
            *  destination objects.
            */
/* commented out until KPDS_MASK_INCREMENT_SIZE is implemented (see test3.sh)
           if (!kpds_set_attributes(src_obj,
                KPDS_MASK_REGION_SIZE, rw, rh, rd, rt, re,
		KPDS_MASK_INTERPOLATE, KPAD,
                KPDS_MASK_POSITION, 0, 0, 0, 0, 0,
                KPDS_MASK_INCREMENT_SIZE, w_size,h_size,d_size,t_size,e_size,
                KPDS_MASK_OFFSET, w_off, h_off, d_off, t_off, e_off, NULL))
*/
           if (!kpds_set_attributes(src_obj,
                KPDS_MASK_REGION_SIZE, rw, rh, rd, rt, re,
		KPDS_MASK_INTERPOLATE, KPAD,
                KPDS_MASK_POSITION, 0, 0, 0, 0, 0,
                KPDS_MASK_OFFSET, w_off, h_off, d_off, t_off, e_off, NULL))
           {
              kerror(lib,funct,"Failed to set source mask attributes.");
              kpds_close_object(src_obj); kpds_close_object(dst_obj);
              return(FALSE);
           }
 
           if (!kpds_set_attributes(dst_obj,
                KPDS_MASK_REGION_SIZE, rw, rh, rd, rt, re,
                KPDS_MASK_POSITION, 0, 0, 0, 0, 0, 
                KPDS_MASK_OFFSET, 0, 0, 0, 0, 0, NULL))
           {
              kerror(lib,funct,"Failed to set destination mask attributes.");
              kpds_close_object(src_obj); kpds_close_object(dst_obj);
              return(FALSE);
           }
           for (i = 0; i<numrgns; i++)
           {
              if ((data=kpds_get_data(src_obj,KPDS_MASK_REGION,data)) == NULL)
              {
                 kerror(lib,funct,"get_data failed for mask on region %d.",i);
                 kpds_close_object(src_obj); kpds_close_object(dst_obj);
                 return(FALSE);
              }
              if (!kpds_put_data(dst_obj, KPDS_MASK_REGION, data))
              {
                 kerror(lib,funct,"put_data failed for mask on region %d.",i);
                 kpds_close_object(src_obj); kpds_close_object(dst_obj);
                 return(FALSE);
              }
           }
           kfree(data);
           data = NULL;
	}

	/*
	 *  LOCATION SEGMENT
	 */
	if (src_has_location && dst_has_location) 
	{
	   /*  Extract location differently depending on the grid type */
	   switch (grid)
	   {
	      case KUNIFORM :
		 {
		    double w_beg, h_beg, d_beg, t_beg, e_beg;
		    double w_end, h_end, d_end, t_end, e_end;
		    double wd, hd, dd, ed, td;
		    int    ws, hs, ds, es, ts;
		    
		    if (!kpds_get_attributes(src_obj,
				   KPDS_LOCATION_BEGIN, 
				   &w_beg, &h_beg, &d_beg, /* &t_beg, &e_beg, */
				   KPDS_LOCATION_END, 
				   &w_end, &h_end, &d_end, /* &t_end, &e_end, */
				   NULL))
		    {
		       kerror(lib,funct,
			      "unable to retrieve location begin and end");
		       return(FALSE);
		    }

		    /* use the location size for the width height and depth */
		    if (!kpds_get_attribute(src_obj, KPDS_LOCATION_SIZE, 
					    &ws, &hs, &ds, NULL))
		    {
		       kerror(lib,funct,
			      "unable to retrieve location size");
		       return(FALSE);
		    }
		    
		    /* if possible, use the value size for time and elements */
		    if (!kpds_get_attribute(src_obj, KPDS_VALUE_SIZE, 
					    NULL, NULL, NULL, &ts, &es))
		    {
		       /* no way to determine the time size, so just use 1 */
		       ts = 1;
		       es = 1;
		    }
		       
		    /* Determine the extracted begin and end points */ 
		    wd = (w_end - w_beg)/ ws;
		    hd = (h_end - h_beg)/ hs;
		    dd = (d_end - d_beg)/ ds;
		    td = (t_end - t_beg)/ ts;
		    ed = (e_end - e_beg)/ es;

		    /* Compute the offseted begin points */
		    w_beg = w_beg + w_off*wd;
		    h_beg = h_beg + h_off*hd;
		    d_beg = d_beg + d_off*dd;
		    t_beg = t_beg + t_off*td;
		    e_beg = e_beg + e_off*ed;
		    
		    /* Compute the extracted end points */
		    w_end = w_beg + w_size*wd;
		    h_end = h_beg + h_size*hd;
		    d_end = d_beg + d_size*dd;
		    e_end = e_beg + e_size*dd;

		    if (!kpds_set_attributes(dst_obj,
					     KPDS_LOCATION_BEGIN, 
					     w_beg, h_beg, d_beg, /* t_beg, e_beg, */
					     KPDS_LOCATION_END, 
					     w_end, h_end, d_end, /* t_end, e_end, */
					     NULL))
		    {
		       kerror(lib,funct,
			      "unable to set location begin and end");
		       return(FALSE);
		    }

		 } break;

	      case KRECTILINEAR :
		 {
		    /*
		     *  Set the region size and offsets on the source
		     *  and destination objects. 
		     */
		    if (!kpds_set_attributes(src_obj, 
			      KPDS_LOCATION_REGION_SIZE,
					     w_size,h_size,d_size,loc_dim,
			      KPDS_LOCATION_INTERPOLATE, KPAD,
			      KPDS_LOCATION_OFFSET, 
					     w_off, h_off, d_off, 0, NULL))
		    {
		       kerror(lib,funct,
			      "Failed to set source location attributes.");
		       kpds_close_object(src_obj); kpds_close_object(dst_obj);
		       return(FALSE);
		    }
 
		    if (!kpds_set_attributes(dst_obj, 
			      KPDS_LOCATION_REGION_SIZE, 
					     w_size, h_size, d_size, loc_dim,
			      KPDS_LOCATION_OFFSET, 0, 0, 0, 0, NULL))
		    {
		       kerror(lib,funct,
			      "Failed to set destination location attributes");
		       kpds_close_object(src_obj); kpds_close_object(dst_obj);
		       return(FALSE);
		    }

		    /* Extract along rectilinear width */
		    if ((data=kpds_get_data(src_obj,
				   KPDS_LOCATION_WIDTH_REGION,data)) == NULL)
		    {
		       kerror(lib,funct,"get_data failed for location width.");
		       kpds_close_object(src_obj); kpds_close_object(dst_obj);
		       return(FALSE);
		    }
		    if (!kpds_put_data(dst_obj, KPDS_LOCATION_WIDTH_REGION, 
				       data))
		    {
		       kerror(lib,funct,
			      "put_data failed for location width.");
		       kpds_close_object(src_obj); kpds_close_object(dst_obj);
		       return(FALSE);
		    }
		    kfree(data);
		    data = NULL;

		    /* Extract along rectilinear height */
		    if ((data=kpds_get_data(src_obj,
				   KPDS_LOCATION_HEIGHT_REGION,data)) == NULL)
		    {
		       kerror(lib,funct,
			      "get_data failed for location height.");
		       kpds_close_object(src_obj); kpds_close_object(dst_obj);
		       return(FALSE);
		    }
		    if (!kpds_put_data(dst_obj, KPDS_LOCATION_HEIGHT_REGION, 
				       data))
		    {
		       kerror(lib,funct,
			      "put_data failed for location height.");
		       kpds_close_object(src_obj); kpds_close_object(dst_obj);
		       return(FALSE);
		    }
		    kfree(data);
		    data = NULL;

		    /* Extract along rectilinear depth */
		    if ((data=kpds_get_data(src_obj,
				   KPDS_LOCATION_DEPTH_REGION,data)) == NULL)
		    {
		       kerror(lib,funct,
			      "get_data failed for location depth.");
		       kpds_close_object(src_obj); kpds_close_object(dst_obj);
		       return(FALSE);
		    }
		    if (!kpds_put_data(dst_obj, KPDS_LOCATION_DEPTH_REGION, 
				       data))
		    {
		       kerror(lib,funct,
			      "put_data failed for location depth.");
		       kpds_close_object(src_obj); kpds_close_object(dst_obj);
		       return(FALSE);
		    }
		    kfree(data);
		    data = NULL;


		 } break;
		 
	      case KCURVILINEAR :
		 {
		    /*
		     *  Set the region size and offsets on the source
		     *  and destination objects. 
		     */
		    if (!kpds_set_attributes(src_obj, 
			      KPDS_LOCATION_REGION_SIZE,
					     w_size,h_size,d_size,loc_dim,
			      KPDS_LOCATION_INTERPOLATE, KPAD,
			      KPDS_LOCATION_OFFSET, 
					     w_off, h_off, d_off, 0, NULL))
		    {
		       kerror(lib,funct,
			      "Failed to set source location attributes.");
		       kpds_close_object(src_obj); kpds_close_object(dst_obj);
		       return(FALSE);
		    }
 
		    if (!kpds_set_attributes(dst_obj, 
			      KPDS_LOCATION_REGION_SIZE, 
					     w_size, h_size, d_size, loc_dim,
			      KPDS_LOCATION_OFFSET, 0, 0, 0, 0, NULL))
		    {
		       kerror(lib,funct,
			      "Failed to set destination location attributes");
		       kpds_close_object(src_obj); kpds_close_object(dst_obj);
		       return(FALSE);
		    }

		    if ((data=kpds_get_data(src_obj,
				   KPDS_LOCATION_REGION,data)) == NULL)
		    {
		       kerror(lib,funct,"get_data failed for location.");
		       kpds_close_object(src_obj); kpds_close_object(dst_obj);
		       return(FALSE);
		    }
		    if (!kpds_put_data(dst_obj, KPDS_LOCATION_REGION, 
				       data))
		    {
		       kerror(lib,funct,
			      "put_data failed for location.");
		       kpds_close_object(src_obj); kpds_close_object(dst_obj);
		       return(FALSE);
		    }
		    kfree(data);
		    data = NULL;

 		 } break;
	   }
        }

	/*
	 *  TIME SEGMENT
	 */
	if (src_has_time && dst_has_time) 
	{
           /*
            *  Set the region size and offsets on the source and
            *  destination objects.  There is no optimal region size
            *  attribute for time, so get the whole chunk.
            */
           if (!kpds_set_attributes(src_obj, KPDS_TIME_REGION_SIZE, t_size,
		KPDS_TIME_INTERPOLATE, KPAD,
                KPDS_TIME_OFFSET, t_off, NULL))
           {
              kerror(lib,funct,"Failed to set source time attributes.");
              kpds_close_object(src_obj); kpds_close_object(dst_obj);
              return(FALSE);
           }
 
           if (!kpds_set_attributes(dst_obj, KPDS_TIME_REGION_SIZE, t_size, 
                KPDS_TIME_OFFSET, 0, NULL))
           {
              kerror(lib,funct,"Failed to set destination time attributes.");
              kpds_close_object(src_obj); kpds_close_object(dst_obj);
              return(FALSE);
           }

           if ((data = kpds_get_data(src_obj, KPDS_TIME_REGION, data)) == NULL)
           {
              kerror(lib,funct,"get_data failed for time.");
              kpds_close_object(src_obj); kpds_close_object(dst_obj);
              return(FALSE);
           }
           if (!kpds_put_data(dst_obj, KPDS_TIME_REGION, data))
           {
              kerror(lib,funct,"put_data failed for time.");
              kpds_close_object(src_obj); kpds_close_object(dst_obj);
              return(FALSE);
           }
           kfree(data);
           data = NULL;
	}

	if (subpos)
	{
	   if (!kpds_set_attribute(dst_obj, 
		   KPDS_SUBOBJECT_POSITION, w_off, h_off, d_off, t_off, e_off))
           {
              kerror(lib,funct,"Unable to set destination subobject position.");
              kpds_close_object(src_obj); kpds_close_object(dst_obj);
              return(FALSE);
           }
	}
	else 
	{
	   if (!kpds_set_attribute(dst_obj, 
		   KPDS_SUBOBJECT_POSITION, 0, 0, 0, 0, 0))
           {
              kerror(lib,funct,"Unable to set destination subobject position.");
              kpds_close_object(src_obj); kpds_close_object(dst_obj);
              return(FALSE);
           }
	}
		
/* uncomment for debugging
kpds_print_attribute(dst_obj,KPDS_SUBOBJECT_POSITION,kstdout);
kinfo(KSTANDARD, " <-- KPDS_SUBOBJECT_POSITION");
*/

	/* 
	 * Close the source and destination reference objects.
	 */
	kpds_close_object(src_obj);
	kpds_close_object(dst_obj);
	return(TRUE);
}
/* -library_code_end */
