/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * kstrftime.c
 *
 * Public-domain implementation of ANSI C library routine.
 *
 * If you want stuff in the System V ascftime routine, add the SYSV_EXT define.
 * For extensions from SunOS, add SUNOS_EXT.
 * For stuff needed to implement the P1003.2 date command, add POSIX2_DATE.
 * For VMS dates, add VMS_EXT.
 * For complete POSIX semantics, add POSIX_SEMANTICS.
 *
 * The code for %c, %x, and %X is my best guess as to what's "appropriate".
 * This version ignores LOCALE information.
 * It also doesn't worry about multi-byte characters.
 * So there.
 *
 * Arnold Robbins
 * January, February, March, 1991
 * Updated March, April 1992
 * Updated April, 1993
 * Updated February, 1994
 *
 * Fixes from ado@elsie.nci.nih.gov
 * February 1991, May 1992
 * Fixes from Tor Lillqvist tml@tik.vtt.fi
 * May, 1993
 * Further fixes from ado@elsie.nci.nih.gov
 * February 1994
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	String Time Formatting Routines for kutils
   >>>> 
   >>>>	Private: 
   >>>>	Static: 
   >>>>	Public: 
   >>>>		kstrftime()
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"


/* defaults: season to taste */
#define SYSV_EXT	1	/* stuff in System V ascftime routine */
#define SUNOS_EXT	1	/* stuff in SunOS strftime routine */
#define POSIX2_DATE	1	/* stuff in Posix 1003.2 date command */
#define VMS_EXT		1	/* include %v for VMS date format */
#define POSIX_SEMANTICS	1	/* call tzset() if TZ changes */

/*#define HAVE_TZNAME 1*/
#if !defined(OS2) && !defined(MSDOS) && defined(HAVE_TZNAME)
extern char *tzname[2];
extern int daylight;
#endif


/*-----------------------------------------------------------
|
|  Routine Name: weeknumber - figure how many weeks into the year
|
|       Purpose:
|         Input: timeptr
|		 firstweekday
|        Output: 
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Arnold Robbins
|          Date: February 1994
| Modifications: With thanks and tip of the hatlo to ado@elsie.nci.nih.gov
|
------------------------------------------------------------*/

static int
weeknumber(const struct tm *timeptr, int firstweekday)
{
	int wday = timeptr->tm_wday;
	int ret;

	if (firstweekday == 1) {
		if (wday == 0)	/* sunday */
			wday = 6;
		else
			wday--;
	}
	ret = ((timeptr->tm_yday + 7 - wday) / 7);
	if (ret < 0)
		ret = 0;
	return ret;
}

/*-----------------------------------------------------------
|
|  Routine Name: iso8601wknum --- compute week number according to ISO 8601
|
|       Purpose: This routine computes the week number according to ISO 8601.
|         Input: timeptr - 
|        Output: 
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Arnold Robbins (arnold@skeeve.atl.ga.us)
|          Date: February 1994
| Modifications:
|
------------------------------------------------------------*/
static int
iso8601wknum(const struct tm *timeptr)
{
	/*
	 * From 1003.2:
	 *	If the week (Monday to Sunday) containing January 1
	 *	has four or more days in the new year, then it is week 1;
	 *	otherwise it is week 53 of the previous year, and the
	 *	next week is week 1.
	 *
	 * ADR: This means if Jan 1 was Monday through Thursday,
	 *	it was week 1, otherwise week 53.
	 */
	int weeknum, jan1day;

	/* get week number, Monday as first day of the week */
	weeknum = weeknumber(timeptr, 1);

	/*
	 * With thanks and tip of the hatlo to tml@tik.vtt.fi
	 *
	 * What day of the week does January 1 fall on?
	 * We know that
	 *	(timeptr->tm_yday - jan1.tm_yday) MOD 7 ==
	 *		(timeptr->tm_wday - jan1.tm_wday) MOD 7
	 * and that
	 * 	jan1.tm_yday == 0
	 * and that
	 * 	timeptr->tm_wday MOD 7 == timeptr->tm_wday
	 * from which it follows that. . .
 	 */
	jan1day = timeptr->tm_wday - (timeptr->tm_yday % 7);
	if (jan1day < 0)
		jan1day += 7;

	/*
	 * If Jan 1 was a Monday through Thursday, it was in
	 * week 1.  Otherwise it was last year's week 53, which is
	 * this year's week 0.
	 *
	 * What does that mean?
	 * If Jan 1 was Monday, the week number is exactly right, it can
	 *	never be 0.
	 * If it was Tuesday through Thursday, the weeknumber is one
	 *	less than it should be, so we add one.
	 * Otherwise, Friday, Saturday or Sunday, the week number is
	 * OK, but if it is 0, it needs to be 53.
	 */
	switch (jan1day) {
	case 1:		/* Monday */
		break;
	case 2:		/* Tuesday */
	case 3:		/* Wednedsday */
	case 4:		/* Thursday */
		weeknum++;
		break;
	case 5:		/* Friday */
	case 6:		/* Saturday */
	case 0:		/* Sunday */
		if (weeknum == 0)
			weeknum = 53;
		break;
	}
	return weeknum;
}


/************************************************************
*
*  Routine Name: kstrftime - generate formatted time information
*
*       Purpose: The following description is transcribed verbatim from the
*		 December 7, 1988 draft standard for ANSI C.  This draft is
*		 essentially identical in technical content to the final
*		 version of the standard.
*
*	   The kstrftime function places characters into the array
*	   pointed to by s as controlled by the string pointed to by
*	   format. The format shall be a multibyte character sequence,
*	   beginning and ending in its initial shift state. The format
*	   string consists of zero or more conversion specifiers and
*	   ordinary multibyte characters. A conversion specifier consists
*	   of a % character followed by a character that determines the
*	   behavior of the conversion specifier. All ordinary multibyte
*	   characters (including the terminating null
*	   character) are copied unchanged into the array. If copying
*	   takes place between objects that overlap the behavior is
*	   undefined.  No more than maxsize characters are placed into
*	   the array.  Each conversion specifier is replaced  by
*	   appropriate characters as described in the following list.
*	   The appropriate characters are determined by the LC_TIME
*	   category of the current locale and by the values contained
*	   in the structure pointed to by timeptr.
*	 
*	   %a  is replaced by the locale's abbreviated weekday name.
*	 
*	   %A  is replaced by the locale's full weekday name.
*	 
*	   %b  is replaced by the locale's abbreviated month name.
*	 
*	   %B  is replaced by the locale's full month name.
*	 
*	   %c  is replaced by the locale's appropriate date and time
*	     representation.
*	 
*	   %d  is replaced by the day of the month as a decimal number
*	     (01-31).
*	 
*	   %H  is replaced by the hour (24-hour clock) as a decimal
*	     number (00-23).
*	 
*	   %I  is replaced by the hour (12-hour clock) as a decimal
*	     number (01-12).
*	 
*	   %j  is replaced by the day of the year as a decimal number
*	     (001-366).
*	 
*	   %m  is replaced by the month as a decimal number (01-12).
*	 
*	   %M  is replaced by the minute as a decimal number (00-59).
*	 
*	   %p  is replaced by the locale's equivalent of the AM/PM
*	     designations associated with a 12-hour clock.
*	 
*	   %S  is replaced by the second as a decimal number (00-61).
*	 
*	   %U  is replaced by the week number of the year (the first
*	     Sunday as the first day of week 1) as a decimal number
*	     (00-53).
*	 
*	   %w  is replaced by the weekday as a decimal number [0
*	     (Sunday)-6].
*	 
*	   %W  is replaced by the week number of the year (the first
*	     Monday as the first day of week 1) as a decimal number
*	     (00-53).
*	 
*	   %x  is replaced by the locale's appropriate date represen-
*	     tation.
*	 
*	   %X  is replaced by the locale's appropriate time represen-
*	     tation.
*	 
*	   %y  is replaced by the year without century as a decimal
*	     number (00-99).
*	 
*	   %Y  is replaced by the year with century as a decimal
*	     number.
*	 
*	   %Z  is replaced by the time zone name or abbreviation, or
*	     by no characters if no time zone is determinable.
*	 
*	   %%  is replaced by %.
*	 
*	   If a conversion specifier is not one of the above, the
*	   behavior is undefined.
*	
*	EXTENSIONS
*	 
*	NON-ANSI EXTENSIONS
*	   If SYSV_EXT is defined when the routine is compiled, then
*	   the following additional conversions will be available.
*	   These are borrowed from the System V cftime(3) and ascf-
*	   time(3) routines.
*	 
*	   %D  is equivalent to specifying %m/%d/%y.
*	 
*	   %e  is replaced by the day of the month, padded with a
*	     blank if it is only one digit.
*	 
*	   %h  is equivalent to %b, above.
*	 
*	   %n  is replaced with a newline character (ASCII LF).
*	 
*	   %r  is equivalent to specifying %I:%M:%S %p.
*	 
*	   %R  is equivalent to specifying %H:%M.
*	 
*	   %T  is equivalent to specifying %H:%M:%S.
*	 
*	   %t  is replaced with a TAB character.
*	 
*	   If SUNOS_EXT is defined when the routine is compiled, then
*	   the following additional conversions will be available.
*	   These are borrowed from the SunOS version of kstrftime.
*	 
*	   %k  is replaced by the hour (24-hour clock) as a decimal
*	     number (0-23). Single digit numbers are padded with a
*	     blank.
*	 
*	   %l  is replaced by the hour (12-hour clock) as a decimal
*	     number (1-12). Single digit numbers are padded with a
*	     blank.
*	 
*	POSIX 1003.2 EXTENSIONS
*	   If POSIX2_DATE is defined, then all of the conversions
*	   available with SYSV_EXT and SUNOS_EXT are available, as well
*	   as the following additional conversions:
*	 
*	   %C  The century, as a number between 00 and 99.
*	 
*	   %u  is replaced by the weekday as a decimal number [1
*	     (Monday)-7].
*	 
*	   %V  is replaced by the week number of the year (the first
*	     Monday as the first day of week 1) as a decimal number
*	     (01-53). The method for determining the week number is
*	     as specified by ISO 8601 (to wit: if the week contain-
*	     ing January 1 has four or more days in the new year,
*	     then it is week 1, otherwise it is week 53 of the pre-
*	     vious year and the next week is week 1).
*	   The text of the POSIX standard for the date utility
*	   describes %U and %W this way:
*	 
*	   %U  is replaced by the week number of the year (the first
*	     Sunday as the first day of week 1) as a decimal number
*	     (00-53). All days in a new year preceding the first
*	     Sunday are considered to be in week 0.
*	 
*	   %W  is replaced by the week number of the year (the first
*	     Monday as the first day of week 1) as a decimal number
*	     (00-53). All days in a new year preceding the first
*	     Monday are considered to be in week 0.
*	 
*	   In addition, the alternate representations %Ec, %EC, %Ex,
*	   %Ey, %EY, %Od, %Oe, %OH, %OI, %Om, %OM, %OS, %Ou, %OU, %OV,
*	   %Ow, %OW, and %Oy are recognized, but their normal represen-
*	   tations are used.
*	 
*	VMS EXTENSIONS
*	   If VMS_EXT is defined, then the following additional conver-
*	   sion is available:
*	 
*	   %v  The date in VMS format (e.g. 20-JUN-1991).
*
*         Input: maxsize - size of s output array
*		 format  - string to describe how the time information should be
*			   formatted.
*		 timeptr - pointer to a time structure with the time you want
*			   to format.
*        Output: s       - formatted output time string
*       Returns: If the total number of resulting characters including the
*		 terminating null character is not more than maxsize, the
*		 kstrftime function returns the number of characters placed
*		 into the array pointed to by s not including the terminating
*		 null character. Otherwise, zero is returned and the
*		 contents of the array are indeterminate.
*
*  Restrictions:
*    Written By: Arnold Robbins (arnold@skeeve.atl.ga.us)
*          Date: February 1994 
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int kstrftime(
    char  *s,
    int   maxsize,
    char  *format,
    struct tm *timeptr)
{
	char *endp = s + maxsize;
	char *start = s;
	auto char tbuf[100];
	int i;
	static short first = 1;
#ifdef POSIX_SEMANTICS
	static char *savetz = NULL;
	static int savetzlen = 0;
	char *tz;
#endif /* POSIX_SEMANTICS */

	/* various tables, useful in North America */
	static char *days_a[] = {
		"Sun", "Mon", "Tue", "Wed",
		"Thu", "Fri", "Sat",
	};
	static char *days_l[] = {
		"Sunday", "Monday", "Tuesday", "Wednesday",
		"Thursday", "Friday", "Saturday",
	};
	static char *months_a[] = {
		"Jan", "Feb", "Mar", "Apr", "May", "Jun",
		"Jul", "Aug", "Sep", "Oct", "Nov", "Dec",
	};
	static char *months_l[] = {
		"January", "February", "March", "April",
		"May", "June", "July", "August", "September",
		"October", "November", "December",
	};
	static char *ampm[] = { "AM", "PM", };

	if (s == NULL || format == NULL || timeptr == NULL || maxsize == 0)
		return 0;

	/* quick check if we even need to bother */
	if (kstrchr(format, '%') == NULL && kstrlen(format) + 1 >= maxsize)
		return 0;

#ifndef POSIX_SEMANTICS
	if (first) {
		tzset();
		first = 0;
	}
#else	/* POSIX_SEMANTICS */
	tz = getenv("TZ");
	if (first) {
		if (tz != NULL) {
			int tzlen = kstrlen(tz);

			savetz = (char *) kmalloc(tzlen + 1);
			if (savetz != NULL) {
				savetzlen = tzlen + 1;
				kstrcpy(savetz, tz);
			}
		}
		tzset();
		first = 0;
	}
	/* if we have a saved TZ, and it is different, recapture and reset */
	if (tz && savetz && (tz[0] != savetz[0] || kstrcmp(tz, savetz) != 0)) {
		i = kstrlen(tz) + 1;
		if (i > savetzlen) {
			savetz = (char *) krealloc(savetz, i);
			if (savetz) {
				savetzlen = i;
				kstrcpy(savetz, tz);
			}
		} else
			kstrcpy(savetz, tz);
		tzset();
	}
#endif	/* POSIX_SEMANTICS */

	for (; *format && s < endp - 1; format++) {
		tbuf[0] = '\0';
		if (*format != '%') {
			*s++ = *format;
			continue;
		}
	again:
		switch (*++format) {
		case '\0':
			*s++ = '%';
			goto out;

		case '%':
			*s++ = '%';
			continue;

		case 'a':	/* abbreviated weekday name */
			if (timeptr->tm_wday < 0 || timeptr->tm_wday > 6)
				kstrcpy(tbuf, "?");
			else
				kstrcpy(tbuf, days_a[timeptr->tm_wday]);
			break;

		case 'A':	/* full weekday name */
			if (timeptr->tm_wday < 0 || timeptr->tm_wday > 6)
				kstrcpy(tbuf, "?");
			else
				kstrcpy(tbuf, days_l[timeptr->tm_wday]);
			break;

#ifdef SYSV_EXT
		case 'h':	/* abbreviated month name */
#endif
		case 'b':	/* abbreviated month name */
			if (timeptr->tm_mon < 0 || timeptr->tm_mon > 11)
				kstrcpy(tbuf, "?");
			else
				kstrcpy(tbuf, months_a[timeptr->tm_mon]);
			break;

		case 'B':	/* full month name */
			if (timeptr->tm_mon < 0 || timeptr->tm_mon > 11)
				kstrcpy(tbuf, "?");
			else
				kstrcpy(tbuf, months_l[timeptr->tm_mon]);
			break;

		case 'c':	/* appropriate date and time representation */
			ksprintf(tbuf, "%s %s %2d %02d:%02d:%02d %d",
				days_a[krange(0, timeptr->tm_wday, 6)],
				months_a[krange(0, timeptr->tm_mon, 11)],
				krange(1, timeptr->tm_mday, 31),
				krange(0, timeptr->tm_hour, 23),
				krange(0, timeptr->tm_min, 59),
				krange(0, timeptr->tm_sec, 61),
				timeptr->tm_year + 1900);
			break;

		case 'd':	/* day of the month, 01 - 31 */
			i = krange(1, timeptr->tm_mday, 31);
			ksprintf(tbuf, "%02d", i);
			break;

		case 'H':	/* hour, 24-hour clock, 00 - 23 */
			i = krange(0, timeptr->tm_hour, 23);
			ksprintf(tbuf, "%02d", i);
			break;

		case 'I':	/* hour, 12-hour clock, 01 - 12 */
			i = krange(0, timeptr->tm_hour, 23);
			if (i == 0)
				i = 12;
			else if (i > 12)
				i -= 12;
			ksprintf(tbuf, "%02d", i);
			break;

		case 'j':	/* day of the year, 001 - 366 */
			ksprintf(tbuf, "%03d", timeptr->tm_yday + 1);
			break;

		case 'm':	/* month, 01 - 12 */
			i = krange(0, timeptr->tm_mon, 11);
			ksprintf(tbuf, "%02d", i + 1);
			break;

		case 'M':	/* minute, 00 - 59 */
			i = krange(0, timeptr->tm_min, 59);
			ksprintf(tbuf, "%02d", i);
			break;

		case 'p':	/* am or pm based on 12-hour clock */
			i = krange(0, timeptr->tm_hour, 23);
			if (i < 12)
				kstrcpy(tbuf, ampm[0]);
			else
				kstrcpy(tbuf, ampm[1]);
			break;

		case 'S':	/* second, 00 - 61 */
			i = krange(0, timeptr->tm_sec, 61);
			ksprintf(tbuf, "%02d", i);
			break;

		case 'U':	/* week of year, Sunday is first day of week */
			ksprintf(tbuf, "%02d", weeknumber(timeptr, 0));
			break;

		case 'w':	/* weekday, Sunday == 0, 0 - 6 */
			i = krange(0, timeptr->tm_wday, 6);
			ksprintf(tbuf, "%d", i);
			break;

		case 'W':	/* week of year, Monday is first day of week */
			ksprintf(tbuf, "%02d", weeknumber(timeptr, 1));
			break;

		case 'x':	/* appropriate date representation */
			ksprintf(tbuf, "%s %s %2d %d",
				days_a[krange(0, timeptr->tm_wday, 6)],
				months_a[krange(0, timeptr->tm_mon, 11)],
				krange(1, timeptr->tm_mday, 31),
				timeptr->tm_year + 1900);
			break;

		case 'X':	/* appropriate time representation */
			ksprintf(tbuf, "%02d:%02d:%02d",
				krange(0, timeptr->tm_hour, 23),
				krange(0, timeptr->tm_min, 59),
				krange(0, timeptr->tm_sec, 61));
			break;

		case 'y':	/* year without a century, 00 - 99 */
			i = timeptr->tm_year % 100;
			ksprintf(tbuf, "%02d", i);
			break;

		case 'Y':	/* year with century */
			ksprintf(tbuf, "%d", 1900 + timeptr->tm_year);
			break;

		case 'Z':	/* time zone name or abbrevation */
#if defined(HAVE_TZNAME)
#ifdef HAVE_TZNAME
			i = (daylight && timeptr->tm_isdst); /* 0 or 1 */
			kstrcpy(tbuf, tzname[i]);
#else
			kstrcpy(tbuf, timeptr->tm_zone);
#endif
#endif
			break;

#ifdef SYSV_EXT
		case 'n':	/* same as \n */
			tbuf[0] = '\n';
			tbuf[1] = '\0';
			break;

		case 't':	/* same as \t */
			tbuf[0] = '\t';
			tbuf[1] = '\0';
			break;

		case 'D':	/* date as %m/%d/%y */
			kstrftime(tbuf, sizeof tbuf, "%m/%d/%y", timeptr);
			break;

		case 'e':	/* day of month, blank padded */
			ksprintf(tbuf, "%2d", krange(1, timeptr->tm_mday, 31));
			break;

		case 'r':	/* time as %I:%M:%S %p */
			kstrftime(tbuf, sizeof tbuf, "%I:%M:%S %p", timeptr);
			break;

		case 'R':	/* time as %H:%M */
			kstrftime(tbuf, sizeof tbuf, "%H:%M", timeptr);
			break;

		case 'T':	/* time as %H:%M:%S */
			kstrftime(tbuf, sizeof tbuf, "%H:%M:%S", timeptr);
			break;
#endif

#ifdef SUNOS_EXT
		case 'k':	/* hour, 24-hour clock, blank pad */
			ksprintf(tbuf, "%2d", krange(0, timeptr->tm_hour, 23));
			break;

		case 'l':	/* hour, 12-hour clock, 1 - 12, blank pad */
			i = krange(0, timeptr->tm_hour, 23);
			if (i == 0)
				i = 12;
			else if (i > 12)
				i -= 12;
			ksprintf(tbuf, "%2d", i);
			break;
#endif


#ifdef VMS_EXT
		case 'v':	/* date as dd-bbb-YYYY */
			ksprintf(tbuf, "%02d-%3.3s-%4d",
				krange(1, timeptr->tm_mday, 31),
				months_a[krange(0, timeptr->tm_mon, 11)],
				timeptr->tm_year + 1900);
			for (i = 3; i < 6; i++)
				if (islower(tbuf[i]))
					tbuf[i] = toupper(tbuf[i]);
			break;
#endif


		case 'C':
			ksprintf(tbuf, "%02d", (timeptr->tm_year + 1900) / 100);
			break;


		case 'E':
		case 'O':
			/* POSIX locale extensions, ignored for now */
			goto again;

		case 'V':	/* week of year according ISO 8601 */
			ksprintf(tbuf, "%02d", iso8601wknum(timeptr));
			break;

		case 'u':
		/* ISO 8601: Weekday as a decimal number [1 (Monday) - 7] */
			ksprintf(tbuf, "%d", timeptr->tm_wday == 0 ? 7 :
					timeptr->tm_wday);
			break;
		default:
			tbuf[0] = '%';
			tbuf[1] = *format;
			tbuf[2] = '\0';
			break;
		}
		i = kstrlen(tbuf);
		if (i) {
			if (s + i < endp - 1) {
				kstrcpy(s, tbuf);
				s += i;
			} else
				return 0;
		}
	}
out:
	if (s < endp && *format == '\0') {
		*s = '\0';
		return (s - start);
	} else
		return 0;
}
