/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Transport Machine Independent Writing Utilities
   >>>>
   >>>>  Private:
   >>>>             write_convert
   >>>>   Public:
   >>>>             kwrite_bit()
   >>>>             kwrite_byte()
   >>>>             kwrite_short()
   >>>>             kwrite_int()
   >>>>             kwrite_long()
   >>>>             kwrite_float()
   >>>>             kwrite_double()
   >>>>             kwrite_ubyte()
   >>>>             kwrite_ushort()
   >>>>             kwrite_uint()
   >>>>             kwrite_ulong()
   >>>>             kwrite_complex()
   >>>>             kwrite_dcomplex()
   >>>>             kwrite_string()
   >>>>             kwrite_generic()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"	


/*-----------------------------------------------------------
|
|  Routine Name: write_convert - private write and conversion routine
|
|       Purpose: The write_convert routine is the grunt routine that
|		 does the actual writing of data as well calls the
|		 appropriate conversion routine. 
|
|         Input: kfile - the device in which we will write the data to
|                ptr   - the data array in which we will be storing the
|			 converted data
|                src_esize  - the source element size
|                dest_esize - the destination element size
|                num        - the number of elements to be converted and written
|                converter  - the routine that converts the data for that
|			      routine.
|
|        Output: None
|         Returns: number of converted elements on success, -1 otherwise
|
|    Written By: Mark Young  
|          Date: Jun 14, 1992
| Modifications:
|
------------------------------------------------------------*/
/*ARGSUSED*/
static int write_convert(
   kfile         *file,
   kaddr         ptr,
   int           src_esize,
   int           dest_esize,
   int           num,
   unsigned char *(*converter)())
{
	int	 count = 0;
	long	 src_mach, dest_mach;
	
	static   int size = 0;
	static   kaddr data = NULL;

	/*
	 *  Retrieve the source and destination machine types.
	 */
	src_mach  = kmach_type(NULL);
	dest_mach = file->machtype;
	if (size < num*dest_esize)
	{
	   size = num*dest_esize;
	   data = (kaddr) krealloc(data, size);
	}

	/*
	 *  Convert and write the data
	 */
	if (num > 0 && src_mach != dest_mach)
	{
	   if (!converter(ptr, data, src_mach, dest_mach, num))
	      return(0);

	   ptr = data;
	}
	count = kfwrite(ptr, dest_esize, num, file);
	return(count);
}


/************************************************************
*  
*    Routine Name: kwrite_bit - write an array of bits
*         Purpose: This module is used to write an array of bits.
*  
*		   (note: bits are machine independent and therefore
*		   require no conversion.  This routine is included to
*		   complete the set of write utilities)
*           Input: id   - the file id to be written which was opened earlier
*		          with kopen().
*		   data - the unsigned char array which holds the data.
*		   num  - the number of bits to be written.
*         Returns: the number of bits written
*      Written By: Mark Young  
*            Date: Jun 14, 1992
*************************************************************/

int kwrite_bit(
   int  id,
   unsigned char *data,
   int  num)
{
	kfile   *file;
	int	val, src, dest;


        /*
         *  Get the file entry associated with the id
         */
        if ((file = kfile_checkid(id)) == NULL)
        {
           errno = EBADF;
           return(-1);
        }

	/*
	 *  Get the source and destination element sizes for signed bytes for
	 *  the local (destination) and remote (source) machine types.
	 */
	src  = kmach_sizeof(file->machtype, KUBYTE);
	dest = kmach_sizeof(kmach_type(NULL), KUBYTE);

	/*
	 *  Call write_convert to do the actual work of writing and converting
	 *  the data to the local machine type.
	 */
	val = (num+7)/8;
	val = write_convert(file, (kaddr) data, src, dest, val, convert_byte);
	return(kmin(val*8, num));
}

/************************************************************
*  
*    Routine Name: kwrite_byte - write an array of signed bytes
*         Purpose: This module is used to write an array of signed
*		   bytes.
*  
*		   (note: bytes are machine independent and therefore
*		    require no conversion.  This routine is included
*		    to complete the set of write utilities)
*           Input: id   - the file id to be written to which was opened earlier
*		          with kopen().
*		   data - the char array which holds the data.
*		   num  - the number of signed bytes to be written.
*         Returns: the number of bytes written
*      Written By: Mark Young  
*            Date: Jun 14, 1992
*************************************************************/

int kwrite_byte(
   int  id,
   char *data,
   int  num)
{
	kfile   *file;
	int	src, dest;


        /*
         *  Get the file entry associated with the id
         */
        if ((file = kfile_checkid(id)) == NULL)
        {
           errno = EBADF;
           return(-1);
        }

	/*
	 *  Get the source and destination element sizes for signed bytes for
	 *  the local (destination) and remote (source) machine types.
	 */
	src  = kmach_sizeof(kmach_type(NULL), KBYTE);
	dest = kmach_sizeof(file->machtype, KBYTE);

	/*
	 *  Call write_convert to do the actual work of writing and converting
	 *  the data to the remote machine type.
	 */
	return(write_convert(file,(kaddr) data, src, dest, num, convert_byte));
}

/************************************************************
*  
*    Routine Name: kwrite_short - write an array of signed shorts
*         Purpose: This module is used to write an array of signed
*		   shorts.
*           Input: id   - the file id to be written to which was opened earlier
*		          with kopen().
*		   data - the short array which holds the data.
*		   num  - the number of shorts to be written.
*         Returns: the number of shorts written
*      Written By: Mark Young  
*            Date: Jun 11, 1992
*************************************************************/

int kwrite_short(
   int  id,
   short *data,
   int  num)
{
	kfile   *file;
	int	src, dest;


        /*
         *  Get the file entry associated with the id
         */
        if ((file = kfile_checkid(id)) == NULL)
        {
           errno = EBADF;
           return(-1);
        }

	/*
	 *  Get the source and destination element sizes for signed bytes for
	 *  the local (destination) and remote (source) machine types.
	 */
	src  = kmach_sizeof(kmach_type(NULL), KSHORT);
	dest = kmach_sizeof(file->machtype, KSHORT);

	/*
	 *  Call write_convert to do the actual work of writing and converting
	 *  the data to the local machine type.
	 */
	return(write_convert(file,(kaddr) data, src, dest, num, convert_short));
}

/************************************************************
*  
*    Routine Name: kwrite_int - write an array of signed ints
*         Purpose: This module is used to write an array of signed
*		   ints.
*           Input: id   - the file id to be written to which was opened earlier
*		          with kopen().
*		   data - the integer array which holds the data.
*		   num  - the number of integers to be written.
*         Returns: the number of ints written
*      Written By: Mark Young  
*            Date: Jun 11, 1992
*************************************************************/

int kwrite_int(
   int  id,
   int *data,
   int  num)
{
	kfile   *file;
	int	src, dest;


        /*
         *  Get the file entry associated with the id
         */
        if ((file = kfile_checkid(id)) == NULL)
        {
           errno = EBADF;
           return(-1);
        }

	/*
	 *  Get the source and destination element sizes for signed bytes for
	 *  the local (destination) and remote (source) machine types.
	 */
	src  = kmach_sizeof(kmach_type(NULL), KINT);
	dest = kmach_sizeof(file->machtype, KINT);

	/*
	 *  Call write_convert to do the actual work of writing and converting
	 *  the data to the local machine type.
	 */
	return(write_convert(file,(kaddr) data, src, dest, num, convert_int));
}

/************************************************************
*  
*    Routine Name: kwrite_long - write an array of signed longs
*         Purpose: This module is used to write an array of signed
*		   longs.  The data will need to contain enough
*		   memory to store "num" signed longs.
*           Input: id   - the file id to be written to which was opened earlier
*		        with kopen().
*		   data - the long array in which holds the data.
*		   num  - the number of longs to be written.
*         Returns: the number of longs written
*      Written By: Mark Young  
*            Date: Jun 11, 1992
*************************************************************/

int kwrite_long(
   int  id,
   long *data,
   int  num)
{
	kfile   *file;
	int	src, dest;


        /*
         *  Get the file entry associated with the id
         */
        if ((file = kfile_checkid(id)) == NULL)
        {
           errno = EBADF;
           return(-1);
        }

	/*
	 *  Get the source and destination element sizes for signed bytes for
	 *  the local (destination) and remote (source) machine types.
	 */
	src  = kmach_sizeof(kmach_type(NULL), KLONG);
	dest = kmach_sizeof(file->machtype, KLONG);

	/*
	 *  Call write_convert to do the actual work of writing and converting
	 *  the data to the local machine type.
	 */
	return(write_convert(file,(kaddr) data, src, dest, num, convert_long));
}

/************************************************************
*  
*    Routine Name: kwrite_float - write an array of floats
*         Purpose: This module is used to write an array of floats.
*           Input: id   - the file id to be written to which was opened earlier
*		          with kopen().
*		   data - the float array which holds the data.
*		   num  - the number of floats to be written.
*         Returns: the number of floats written
*      Written By: Mark Young  
*            Date: Jun 11, 1992
*************************************************************/

int kwrite_float(
   int  id,
   float *data,
   int  num)
{
	kfile   *file;
	int	src, dest;


        /*
         *  Get the file entry associated with the id
         */
        if ((file = kfile_checkid(id)) == NULL)
        {
           errno = EBADF;
           return(-1);
        }

	/*
	 *  Get the source and destination element sizes for floats for
	 *  the local (destination) and remote (source) machine types.
	 */
	src  = kmach_sizeof(kmach_type(NULL), KFLOAT);
	dest = kmach_sizeof(file->machtype, KFLOAT);

	/*
	 *  Call write_convert to do the actual work of writing and converting
	 *  the data to the local machine type.
	 */
	return(write_convert(file,(kaddr) data, src, dest, num, convert_float));
}

/************************************************************
*  
*    Routine Name: kwrite_double - write an array of doubles
*         Purpose: This module is used to write an array of doubles.
*           Input: id   - the file id to be written to which was opened earlier
*		          with kopen().
*		   data - the double array which holds the data.
*		   num  - the number of doubles to be written.
*         Returns: the number of doubles written
*      Written By: Mark Young  
*            Date: Jun 11, 1992
*************************************************************/

int kwrite_double(
   int  id,
   double *data,
   int  num)
{
	kfile   *file;
	int	src, dest;


        /*
         *  Get the file entry associated with the id
         */
        if ((file = kfile_checkid(id)) == NULL)
        {
           errno = EBADF;
           return(-1);
        }

	/*
	 *  Get the source and destination element sizes for doubles for
	 *  the local (destination) and remote (source) machine types.
	 */
	src  = kmach_sizeof(kmach_type(NULL), KDOUBLE);
	dest = kmach_sizeof(file->machtype, KDOUBLE);

	/*
	 *  Call write_convert to do the actual work of writing and converting
	 *  the data to the local machine type.
	 */
	return(write_convert(file,(kaddr) data, src, dest, num,convert_double));
}

/************************************************************
*  
*    Routine Name: kwrite_ubyte - write an array of unsigned bytes
*         Purpose: This module is used to write an array of unsigned bytes.
*  
*		   (note: bytes are machine independent and therefore
*		    require no conversion.  This routine is included to
*		    complete the set of write utilities)
*           Input: id   - the file id to be written to which was opened earlier
*		          with kopen().
*		   data - the unsigned char array which holds the data.
*		   num  - the number of unsigned bytes to be written.
*         Returns: the number of bytes written
*      Written By: Mark Young  
*            Date: Jun 11, 1992
*************************************************************/

int kwrite_ubyte(
   int  id,
   unsigned char *data,
   int  num)
{
	kfile   *file;
	int	src, dest;


        /*
         *  Get the file entry associated with the id
         */
        if ((file = kfile_checkid(id)) == NULL)
        {
           errno = EBADF;
           return(-1);
        }

	/*
	 *  Get the source and destination element sizes for unsigned bytes
	 *  for the local (destination) and remote (source) machine types.
	 */
	src  = kmach_sizeof(kmach_type(NULL), KUBYTE);
	dest = kmach_sizeof(file->machtype, KUBYTE);

	/*
	 *  Call write_convert to do the actual work of writing and converting
	 *  the data to the local machine type.
	 */
	return(write_convert(file,(kaddr) data, src, dest, num, convert_ubyte));
}

/************************************************************
*  
*    Routine Name: kwrite_ushort - write an array of unsigned shorts
*  
*         Purpose: This module is used to write an array of unsigned shorts.
*  
*           Input: id   - the file id to be written to which was opened earlier
*		          with kopen().
*		   data - the unsigned short array which holds the data.
*		   num  - the number of unsigned shorts to be written.
*         Returns: the number of shorts written
*      Written By: Mark Young  
*            Date: Jun 11, 1992
*************************************************************/

int kwrite_ushort(
   int  id,
   unsigned short *data,
   int  num)
{
	kfile   *file;
	int	src, dest;


        /*
         *  Get the file entry associated with the id
         */
        if ((file = kfile_checkid(id)) == NULL)
        {
           errno = EBADF;
           return(-1);
        }

	/*
	 *  Get the source and destination element sizes for unsigned shorts
	 *  for the local (destination) and remote (source) machine types.
	 */
	src  = kmach_sizeof(kmach_type(NULL), KUSHORT);
	dest = kmach_sizeof(file->machtype, KUSHORT);

	/*
	 *  Call write_convert to do the actual work of writing and converting
	 *  the data to the local machine type.
	 */
	return(write_convert(file,(kaddr) data, src, dest, num,convert_ushort));
}

/************************************************************
*  
*    Routine Name: kwrite_uint - write an array of unsigned ints
*         Purpose: This module is used to write an array of unsigned ints.
*           Input: id   - the file id to be written to which was opened earlier
*		          with kopen().
*		   data - the unsigned integer array which holds the data.
*		   num  - the number of unsigned integers to be written.
*         Returns: the number of ints written
*      Written By: Mark Young  
*            Date: Jun 11, 1992
*************************************************************/

int kwrite_uint(
   int  id,
   unsigned int *data,
   int  num)
{
	kfile   *file;
	int	src, dest;


        /*
         *  Get the file entry associated with the id
         */
        if ((file = kfile_checkid(id)) == NULL)
        {
           errno = EBADF;
           return(-1);
        }

	/*
	 *  Get the source and destination element sizes for unsigned ints
	 *  for the local (destination) and remote (source) machine types.
	 */
	src  = kmach_sizeof(kmach_type(NULL), KUINT);
	dest = kmach_sizeof(file->machtype, KUINT);

	/*
	 *  Call write_convert to do the actual work of writing and converting
	 *  the data to the local machine type.
	 */
	return(write_convert(file,(kaddr) data, src, dest, num, convert_uint));
}

/************************************************************
*  
*    Routine Name: kwrite_ulong - write an array of unsigned longs
*         Purpose: This module is used to write an array of unsigned longs.
*           Input: id   - the file id to be written to which was opened earlier
*		          with kopen().
*		   data - the unsigned long array which holds the data.
*		   num  - the number of unsigned longs to be written.
*         Returns: the number of longs written
*      Written By: Mark Young  
*            Date: Jun 11, 1992
*************************************************************/

int kwrite_ulong(
   int  id,
   unsigned long *data,
   int  num)
{
	kfile   *file;
	int	src, dest;


        /*
         *  Get the file entry associated with the id
         */
        if ((file = kfile_checkid(id)) == NULL)
        {
           errno = EBADF;
           return(-1);
        }

	/*
	 *  Get the source and destination element sizes for unsigned longs
	 *  for the local (destination) and remote (source) machine types.
	 */
	src  = kmach_sizeof(kmach_type(NULL), KULONG);
	dest = kmach_sizeof(file->machtype, KULONG);

	/*
	 *  Call write_convert to do the actual work of writing and converting
	 *  the data to the local machine type.
	 */
	return(write_convert(file,(kaddr) data, src, dest, num, convert_ulong));
}

/************************************************************
*  
*    Routine Name: kwrite_complex - write an array of complex
*         Purpose: This module is used to write an array of complex numbers.
*           Input: id   - the file id to be written to which was opened earlier
*		          with kopen().
*		   data - the kcomplex array which holds the data.
*		   num  - the number of complex pairs to be written.
*         Returns: the number of complexs written
*      Written By: Mark Young  
*            Date: Jul 20, 1992
*************************************************************/

int kwrite_complex(
   int  id,
   kcomplex *data,
   int  num)
{
	kfile   *file;
	int	src, dest;


        /*
         *  Get the file entry associated with the id
         */
        if ((file = kfile_checkid(id)) == NULL)
        {
           errno = EBADF;
           return(-1);
        }

	/*
	 *  Get the source and destination element sizes for complex numbers
	 *  for the local (destination) and remote (source) machine types.
	 */
	src  = kmach_sizeof(kmach_type(NULL), KCOMPLEX);
	dest = kmach_sizeof(file->machtype, KCOMPLEX);

	/*
	 *  Call write_convert to do the actual work of writing and converting
	 *  the data to the local machine type.
	 */
	return(write_convert(file,(kaddr) data, src, dest,num,convert_complex));
}

/************************************************************
*  
*    Routine Name: kwrite_dcomplex - write an array of double complex
*         Purpose: This module is used to write an array of double complex
*		   numbers.
*           Input: id   - the file id to be written to which was opened earlier
*		          with kopen().
*		   data - the kdcomplex array which holds the data.
*		   num  - the number of double complex pairs to be written.
*         Returns: the number of double complexs written
*      Written By: Mark Young  
*            Date: Jul 20, 1992
*************************************************************/

int kwrite_dcomplex(
   int  id,
   kdcomplex *data,
   int  num)
{
	kfile   *file;
	int	src, dest;


        /*
         *  Get the file entry associated with the id
         */
        if ((file = kfile_checkid(id)) == NULL)
        {
           errno = EBADF;
           return(-1);
        }

	/*
	 *  Get the source and destination element sizes for dcomplex numbers
	 *  for the local (destination) and remote (source) machine types.
	 */
	src  = kmach_sizeof(kmach_type(NULL), KDCOMPLEX);
	dest = kmach_sizeof(file->machtype, KDCOMPLEX);

	/*
	 *  Call write_convert to do the actual work of writing and converting
	 *  the data to the local machine type.
	 */
	return(write_convert(file,(kaddr) data, src,dest,num,convert_dcomplex));
}

/************************************************************
*  
*    Routine Name: kwrite_string - write an array of strings
*         Purpose: This module is used to write an array of strings.
*           Input: id   - the file id to be written to which was opened earlier
*		          with kopen().
*		   data - the string array in which the data will be stored.
*		   num  - the number of strings to be written.
*         Returns: the number of strings written
*      Written By: Mark Young
*            Date: Jul 20, 1992
*************************************************************/

int kwrite_string(
   int  id,
   kstring *data,
   int  num)
{
	int	i;
	char	*tmp;
	kfile   *file;


        /*
         *  Get the file entry associated with the id
         */
        if ((file = kfile_checkid(id)) == NULL)
        {
           errno = EBADF;
           return(-1);
        }

	for (i = 0; i < num; i++)
	{
	   tmp = (data[i] != NULL) ? data[i] : "";
	   if (kwrite(id, tmp, kstrlen(tmp) + 1) == -1)
	      break;
	}
	return(i);
}

/************************************************************
*  
*    Routine Name: kwrite_generic - write an array in any data type.
*         Purpose: This module is used to write an array of 
*      	       	   data to a transport.
*           Input: id   - the file id to be written to which was
*			  opened earlier with kopen().
*		   data - the kaddr array which holds the data.
*		   num  - the number of data points to be written.
*      	       	   type - data type.
*         Returns: the number of points written
*      Written By: Jeremy Worley, Mark Young
*            Date: Sep 04, 1992
*************************************************************/

int kwrite_generic(
   int  id,
   kaddr data,
   int  num,
   int  type)
{
	kfile   *file;
	int	src, dest;
        unsigned char *(*converter)();

        /*
         *  Get the file entry associated with the id
         */
        if ((file = kfile_checkid(id)) == NULL)
        {
           errno = EBADF;
           return(-1);
        }

	/*
	 *  Get the source and destination element sizes for unsigned longs
	 *  for the local (source) and remote (destination) machine types.
	 */
	src  = kmach_sizeof(kmach_type(NULL), type);
	dest = kmach_sizeof(file->machtype, type);

	/*
	 *  Set the correct converter according to the specified data
	 *  type.
	 */
        switch(type)
	{
       	   case KBYTE: 	   converter = convert_byte;     break;
       	   case KUBYTE:	   converter = convert_ubyte;    break;
       	   case KSHORT:	   converter = convert_short;    break;
       	   case KUSHORT:   converter = convert_ushort;   break;
       	   case KLONG: 	   converter = convert_long;     break;
       	   case KULONG:	   converter = convert_ulong;    break;
       	   case KINT:  	   converter = convert_int;      break;
       	   case KUINT: 	   converter = convert_uint;     break;
       	   case KFLOAT:	   converter = convert_float;    break;
       	   case KDOUBLE:   converter = convert_double;   break;
       	   case KCOMPLEX:  converter = convert_complex;  break;
       	   case KDCOMPLEX: converter = convert_dcomplex; break;
       	   case KBIT: 	   return(kwrite_bit(id, data, num));
       	   case KSTRING:   return(kwrite_string(id, data, num));
       	   default:    	   return(-1);
       	}

	/*
	 *  Call write_convert to do the actual work of writing and converting
	 *  the data to the remote machine type.
	 */
	return(write_convert(file,(kaddr) data, src, dest, num, converter));
}

/************************************************************
*  
*    Routine Name: kfwrite_generic - write an array in any data type.
*         Purpose: This module is used to write an array of 
*      	       	   data to a transport.
*           Input: file - the file to be written to which was
*			  opened earlier with kfopen().
*		   data - the kaddr array which holds the data.
*		   num  - the number of data points to be written.
*      	       	   type - data type.
*         Returns: the number of points written
*      Written By: Jeremy Worley, Mark Young
*            Date: Sep 04, 1992
*************************************************************/

int kfwrite_generic(
   kfile *file,
   kaddr data,
   int  num,
   int  type)
{
	return(kwrite_generic(kfileno(file), data, num, type));
}
