/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Unix Sytem Utilities
   >>>>
   >>>>  Private:
   >>>>   Public:
   >>>>             kwait3()
   >>>>             ksignal()
   >>>>             ksignal_print()
   >>>>             
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"	
#include <kutils/ksignal.h>


/************************************************************
*
*  Routine Name: kwait3 - replacement for the system wait3()
*
*       Purpose: This return emulates the bsd wait3() command.
*		 The wait3 process waits until any child has
*		 completed, without hanging (WNOHANG).  It
*		 should be called rather than the system wait3,
*		 because not all systems include a wait3 call.
*
*         Input: options - options used for wait3 (WUNTRACED, WNOHANG)
*
*        Output: status  - the return status of the process
*		 rusage  - the resource usage (for systems that do not
*			   have a wait3 call, this output is not set.)
*                RETURNS: the status of the process.
*
*  Restrictions: Restrictions on data or input as applicable
*    Written By: Bob Weaver
*          Date: Jul 03, 1992 14:06
*      Verified:
*  Side Effects:
* Modifications: Converted from vwait3 in Khoros 1.0 (SJ)
*
*************************************************************/

int kwait3(
   kstatus *status,
   int     options,
   kaddr   rusage)
{
    return(waitpid(-1, status, options));
}

/************************************************************
*
*  Routine Name: ksignal - khoros signal handler
*
*       Purpose: This routine is a replacement for the UNIX signal()
*		 handler.  The problem with UNIX signal() is that
*		 under SVR4 signal() provedes the older, unreliable
*		 signal semantics.
*
*		 The 
*
*         Input: signo  -  the signal number that you wish to install the
*			   handler for.
*                sigfunc - the signal function to be called when the signal
*			   is encountered.
*
*        Output: 
*       Returns: the previously installed signal handler on success, SIG_ERR
*		 on failure
*  Restrictions:
*    Written By: Mark Young & John Salas
*          Date: Mar 24, 1994
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

kfunc_void ksignal(
   int signo,
   kfunc_void sigfunc)
{
	struct sigaction act, oact;

	act.sa_handler = sigfunc;
	sigemptyset(&act.sa_mask);
	act.sa_flags = 0;
	if (signo == SIGALRM)
	{
#ifdef SA_INTERRUPT
	   act.sa_flags |= SA_INTERRUPT;
#endif
	}
	else
	{
#ifdef SA_RESTART
	   act.sa_flags |= SA_RESTART;
#endif
	}

	if (sigaction(signo, &act, &oact) < 0)
	   return(SIG_ERR);

	return(oact.sa_handler);
}

/************************************************************
*
*  Routine Name: ksignal_print - prints the actual signal error
*
*       Purpose: This routine actually prints to the file the signal
*		 that occured.
*
*         Input: file  - the structure in which to print the message to.
*                signo - the signal to print.
*        Output: 
*       Returns: 
*  Restrictions: 
*    Written By: Mark Young & John Salas
*          Date: Nov 29, 1993
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

void ksignal_print(
   kfile *file,
   int   signo)
{
	switch (signo)
	{
	   case SIGALRM:
		kfprintf(file,"Alarm has timed out...\n");
		break;

	   case SIGCHLD:
		kfprintf(file,"Child process terminated or stopped...\n");
		break;

	   case SIGCONT:
		kfprintf(file,"Continuing process...\n");
		break;

	   case SIGHUP:
		kfprintf(file,"Hangup encountered...\n");
		break;

	   case SIGKILL:
		kfprintf(file,"Routine killed...\n");
		break;

	   case SIGPIPE:
		kfprintf(file,"Write on a pipe but no process to read it...\n");
		break;

	   case SIGQUIT:
		kfprintf(file,"Quit character typed on controlling \
keyboard...\n");
		break;

	   case SIGSTOP:
		kfprintf(file,"Routine stopped...\n");
		break;

	   case SIGTSTP:
		kfprintf(file,"Stop special character typed on controlling \
keyboard...\n");
		break;

	   case SIGTTIN:
		kfprintf(file,"Read from controlling terminal attempted by a \
member of a background process group...\n");
		break;

	   case SIGTTOU:
		kfprintf(file,"Write to controlling terminal attempted by a \
member of a background process group...\n");
		break;

	   case SIGUSR1:
		kfprintf(file,"Application-defined signal 1...\n");
		break;

	   case SIGUSR2:
		kfprintf(file,"Application-defined signal 2...\n");
		break;

	   case SIGABRT:
		kfprintf(file,"Routine aborted...\n");
		break;

	   case SIGFPE:
		kfprintf(file,"Arithmetic exception...\n");
		break;

	   case SIGILL:
		kfprintf(file,"Illegal instruction...\n");
		break;

	   case SIGINT:
		kfprintf(file,"Routine interupted...\n");
		break;

	   case SIGSEGV:
		kfprintf(file,"Segmentation violation...\n");
		break;

	   case SIGTERM:
		kfprintf(file,"Software terminated from signal...\n");
		break;

	   default:
		kfprintf(file,"Unknown signal number (%d).... Check signal.3\n",
			 signo);
        }
}
