/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Inter Process Communication Support Utility
   >>>>
   >>>>   Static:
   >>>>		    kipc_getdescriptors
   >>>>		    kipc_setdescriptors
   >>>>  Private:
   >>>>             kipc_start()
   >>>>             kipc_stop()
   >>>>             kipc_sendmsg()
   >>>>             kipc_recvmsg()
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"


/*
 *  The following are private routines that the user should not call but
 *  should be private to the phantom library that enables a negotiation
 *  method between processes thru some user defined negotiator.
 *
 *	routines:	kipc_start
 *			kipc_stop
 *			kipc_sendmsg
 *			kipc_recvmsg
 */

static klist *ipc_list = NULL;

static char  *ipc_transports[] =
{
	"pipe",
	"socket",
	"stream",
};
#define NumList knumber(ipc_transports)


/*-----------------------------------------------------------
|
|  Routine Name: kipc_getdescriptors - gets the ipc transport
|
|       Purpose: This function gets the kfile structure associated
|		 with the ipc used in communicating with some
|		 other process.  This used by more complex routines
|		 such as cantata to communicate with sub-process
|		 so that special considerations can be made when
|		 a process needs to open, close, or a special
|		 notification.
|
|         Input: identifier - the ipc identifier
|		        this is really the environment variable
|			in which the ipc transport is stored.
|			For simplisticty we hardwire the
|			environment variable to be:
|
|				identifier_Request
|				identifier_Reply
|
|        Output: ifile - the input or reply transport
|		 ofile - the output or request transport
|       Returns: 0 on success, -1 upon failure
|
|    Written By: Mark Young
|          Date: Oct 14, 1992
| Modifications:
|
------------------------------------------------------------*/

int kipc_getdescriptors(
   char  *identifier,
   kfile **ifile,
   kfile **ofile)
{
	int   token;
	klist *list;
	char  *path, temp[KLENGTH], env[KLENGTH];
	kfile **files = NULL, *reply = NULL, *request = NULL;

	static int busy = FALSE;


	if (busy == TRUE)
	{
	   if (ifile != NULL) *ifile = reply;
	   if (ofile != NULL) *ofile = request;
	   return(-1);
	}
	else
	   busy = TRUE;

	token = kstring_to_token(identifier);
	if ((list = klist_locate(ipc_list, (kaddr) token)) == NULL)
	{
	   ksprintf(env, "%s_Reply", identifier);
	   if ((path = kgetenv(env)) != NULL)
	   {
	      /*
	       *  Make sure that the transport is supported before
	       *  attempting to make a connection.
	       */
	      if (ktransport_routines(ktransport_identifier(path,temp)) != NULL)
	         reply = kfopen(path, "r+");
	   }

	   ksprintf(env, "%s_Request", identifier);
	   if ((path = kgetenv(env)) != NULL)
	   {
	      /*
	       *  Make sure that the transport is supported before
	       *  attempting to make a connection.
	       */
	      if (ktransport_routines(ktransport_identifier(path,temp)) != NULL)
	         request = kfopen(path, "r+");
	   }

	   if (reply != NULL && request != NULL)
	   {
	      files = (kfile **) karray_add((char **) files, reply, 0);
	      files = (kfile **) karray_add((char **) files, request, 1);
	      ipc_list = klist_add(ipc_list, (kaddr) token, files);
	   }
	}
	else
	{
	   files = (kfile **) klist_clientdata(list);
	   reply = files[0]; request = files[1];
	}
	busy = FALSE;

	if (ifile != NULL) *ifile = reply;
	if (ofile != NULL) *ofile = request;
	return(0);
}

/*-----------------------------------------------------------
|
|  Routine Name: kipc_setdescriptors - sets the ipc transport
|				   
|
|       Purpose: This function sets the kfile structure associated
|                with the ipc used in communicating with some
|                other process.  This used by more complex routines
|                such as cantata to communicate with sub-process
|                so that special considerations can be made when
|                a process needs to open, close, or a special
|                notification.
|
|         Input: identifier - the ipc identifier
|		 ipath - the input or reply transport path
|		 opath - the output or request transport path
|
|       Returns: 0 on success, -1 upon failure
|
|    Written By: Mark Young
|          Date: Oct 14, 1992
| Modifications:
|
------------------------------------------------------------*/

int kipc_setdescriptors(
   char *identifier,
   char *ipath,
   char *opath)
{
	int   token;
	char  buffer[KLENGTH];
	kfile **files = NULL, *itemp = NULL, *otemp = NULL;


	/*
	 *  Open the new input ipc transport.  If we suceed then go
	 *  ahead and close the old transport and update this transport
	 *  as the new one. 
	 */
	if ((itemp = kfopen(ipath, "R+")) == NULL ||
	    (otemp = kfopen(opath, "R+")) == NULL)
	{
	   if (itemp != NULL) kfclose(itemp);
	   else if (otemp != NULL) kfclose(otemp);
	   return(-1);
	}
	kfile_setstate(itemp, KFILE_TEMP);
	kfile_setstate(otemp, KFILE_TEMP);

	/*
	 *  Stop the old ipc transport and add our new ones if one
	 *  exists.
	 */
	(void) kipc_stop(identifier);

	/*
	 *  Now set the environment variables so that sub-processes
	 *  will talk with us, but first remember to switch the
	 *  input and output so that we write output to the sub-processes
	 *  input and read input from the sub-processes output.
	 */
	files = (kfile **) karray_add((char **) files, otemp, 0);
	(void) ksprintf(buffer, "%s_Reply=%s", identifier, ipath);
	kputenv(kstrdup(buffer));

	files = (kfile **) karray_add((char **) files, itemp, 1);
	(void) ksprintf(buffer, "%s_Request=%s", identifier, opath);
	kputenv(kstrdup(buffer));

	token = kstring_to_token(identifier);
	ipc_list = klist_add(ipc_list, (kaddr) token, files);
	return(0);
}


/*-----------------------------------------------------------
|
|  Routine Name: kipc_start - initialize the ipc for the supplied identifier
|
|       Purpose: This routine is used to initialize the ipc
|                so that we will be able to communicate sub-
|                processes.
|
|         Input: identifier - the ipc identifier
|       Returns: return the result from kipc_setdescriptors() or -1 upon failure
|
|    Written By: Mark Young
|          Date: Oct 14, 1992
| Modifications:
|
------------------------------------------------------------*/

int kipc_start(
   char *identifier)
{
	int	i;
	TransportInformation *routines;
	char	template[KLENGTH], *itemp = NULL, *otemp = NULL;


	/*
	 *  This strange bit of code simply generates two unique transports
	 *  one for the input (or reply) process and one for the output (or
	 *  request) process.
	 */
	for (i = 0; i < NumList; i++)
	{
	   if ((routines = ktransport_routines(ipc_transports[i])) == NULL ||
	       (routines->tempnam == NULL))
	   {
	      continue;
	   }
	   else
	   {
	      (void) ksprintf(template,"%s=i%s", ipc_transports[i], identifier);
	      itemp = ktempnam(NULL, template);
	      (void) ksprintf(template,"%s=o%s", ipc_transports[i], identifier);
	      otemp = ktempnam(NULL, template);
	   }

	   if (itemp != NULL && otemp != NULL)
	      break;
	   else if (itemp != NULL)
	      kunlink(itemp);
	   else if (otemp != NULL)
	      kunlink(otemp);
	   else
	      break;
	}

	if (itemp == NULL || otemp == NULL)
	   return(-1);

	return(kipc_setdescriptors(identifier, itemp, otemp));
}

/*-----------------------------------------------------------
|
|  Routine Name: kipc_stop - stops the ipc for the supplied identifier
|
|       Purpose: This routine is used to stop or discontinue an ipc
|                so that processes will no longer communicate with
|                each other.  This is done by actually closing the
|		 ipc transports and removing it from the user's
|		 environment variables.
|
|         Input: identifier - the ipc identifier
|       Returns: 0 on success, -1 upon failure
|
|    Written By: Mark Young
|          Date: Oct 14, 1992
| Modifications:
|
------------------------------------------------------------*/

int kipc_stop(
   char *identifier)
{
	kfile *ifile, *ofile;
	char  *filename, env[KLENGTH];


	if (kipc_getdescriptors(identifier, &ifile, &ofile) == -1)
	   return(-1);

	if (ifile != NULL)
	{
	   kfclose(ifile);
	   (void) ksprintf(env, "%s_Reply", identifier);
	   if ((filename = kgetenv(env)) != NULL)
	      kunlink(filename);
	   kremenv(env);
	}

	if (ofile != NULL)
	{
	   kfclose(ofile);
	   (void) ksprintf(env, "%s_Request", identifier);
	   if ((filename = kgetenv(env)) != NULL)
	      kunlink(filename);
	   kremenv(env);
	}
	return(0);
}
