/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Array manipulation utilities
   >>>>
   >>>>  Private:
   >>>>             none
   >>>>   Public:
   >>>>             karray_insert()
   >>>>             karray_add()
   >>>>             karray_copy()
   >>>>             karray_delete()
   >>>>		    karray_free()
   >>>>             karray_locate()
   >>>>             karray_sort()
   >>>>		    karray_merge()
   >>>>		    karray_split()
   >>>>		    karray_to_list()
   >>>>		    karray_to_string()
   >>>>
   >>>>		    karray_filelist()
   >>>>		    karray_dirlist()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"


/************************************************************
*
*  Routine Name: karray_insert - insert an entry into the array list
*
*       Purpose: Inserts an entry into the array list.  This is done by
*		 adding the entry to the end of the array list
*		 (if the list currently exists).  The new array list
*		 is then passed back to the calling routine.
*		 The routine first scans the array list to make sure the
*		 identifier is not already on the list, if so then we
*		 don't change original list.
*
*		 The position field is used to indicate where in the
*		 list an entry should be inserted:
*
*		 !	KLIST_HEAD - insert at the head of the array
*		 !	KLIST_PREV - insert previous to the current position
*		 !	KLIST_NEXT - insert next from the current position
*		 !	KLIST_TAIL - insert at the tail of the array
*
*	  Input: array      - The current array in which we will be adding the
*			      entry to
*		 identifier - The entry identifier to be added to the array
*		 num	    - the number of entries in the current array
*		 position   - where in the array to add the entry
*		 duplicate_entries - whether to be allowed multiple occurences
*				     of an identifier in the array
*
*        Output: none
*       Returns: returns the head of the modified array
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Dec 09, 1992 17:55
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

char **karray_insert(
   char  **array,
   kaddr identifier,
   int   num,
   int   position,
   int   duplicate_entries)
{
	int i;


	/*
	 *  If duplicate entries aren't allowed then make sure that the
	 *  identifier doesn't currently exist anywhere in the list
	 */
	if (!duplicate_entries && karray_locate(array, identifier, num) != -1)
	   return(array);

	if (array == NULL)
	{
	   num = 0;
	   if ((array = (char **) kmalloc(KMAXENTRIES*sizeof(char *))) == NULL)
	   {
              kerror("kutils", "karray_insert", "Unable to \
kmalloc enough memory for an array list of size %d", num);
              return(NULL);
	   }
	}
	else if (num > 0 && (num % KMAXENTRIES) == 0)
	{
	   int size = num + KMAXENTRIES;

	   if ((array = (char **) krealloc(array, size*sizeof(char *))) == NULL)
	   {
              kerror("kutils", "karray_insert", "Unable to \
krealloc enough memory for an array list of size %d", num);
              return(NULL);
	   }
	}

	/*
	 *  indicates that the programmer wants the entry inserted at
	 *  the head or tail of the list.
	 */
	if (position == KLIST_HEAD || position == KLIST_PREV)
	{
	   for (i = num; i > 0; i--) array[i] = array[i-1];
	   array[0] = identifier; num++;
	}
	else if (position == KLIST_TAIL || position == KLIST_NEXT)
	{
	   array[num++] = identifier;
	}
	return(array);
}


/************************************************************
*
*  Routine Name: karray_add - add an entry into the array list
*
*       Purpose: Adds an entry to the array list.  This is done by
*		 adding the entry to the end of the array list
*		 (if the array currently exists).  If the array is
*		 currently NULL then the new item is returned as the
*		 first entryof the array list.  If the array list is
*		 not NULL then the entry is added to the end of the
*		 list and the original array is passed back to the calling
*		 routine.
*
*		 The routine first scans the array to make sure the
*		 identifier is not already on the list, if so then we
*		 don't change original array.
*
*	  Input: array      - The current array in which we will be adding the
*			      entry to (if NULL then return the newly
*			      malloc'ed array list).
*		 identifier - The entry identifier to be added to the array
*			      list
*		 num	    - The number of entry in the list; ignored if
*			      list is NULL
*
*        Output: none
*       Returns: returns the modified array list.
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Nov 28, 1992 15:36
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

char **karray_add(
   char  **array,
   kaddr identifier,
   int   num)
{
	/*
	 *  Call karray_insert() to do the actually work
	 */
	return(karray_insert(array, identifier, num, KLIST_TAIL, FALSE));
}


/************************************************************
*
*  Routine Name: karray_copy - copy an array of strings
*
*       Purpose: This module is used to copy the input array
*		 of strings into a new array.  If the "copy_entries"
*		 parameter is TRUE, then each entry will also be
*		 copied rather than just copying the pointers.
*
*         Input: array         - the array that is to be sorted
*		 num          - the number of entries in the array
*		 copy_entries - whether to kmalloc and copy each entry or not
*
*        Output: none
*       Returns: a pointer to the new array, or NULL on error
*
*  Restrictions: Restrictions on data or input as applicable
*    Written By: Mark Young
*          Date: Jul 03, 1992 14:07
*      Verified:
*  Side Effects: kmallocs the space for the new array
* Modifications: Converted from vcopylist in Khoros 1.0 (SJ)
*
*************************************************************/

char **karray_copy(
   char **array,
   int  num,
   int  copy_entries)
{
	char **temp;

	temp = karray_merge(array, NULL, num, 0, copy_entries);
	return(temp);
}


/************************************************************
*
*  Routine Name: karray_delete - delete an entry from the array list
*       Purpose: Delete an entry from the array list.  This is done by
*		 deleting the entry from the array list.  The new
*		 list is then passed back to the calling routine.
*	  Input: array      - The current array list in which the entry will be
*			      deleted.
*		 identifier - the identifier which identifies the entry to
*			      be deleted.
*		 num        - the number of elements in the array
*       Returns: the modified array, NULL when the array list becomes empty.
*    Written By: Mark Young
*          Date: Nov 28, 1992
*************************************************************/

char **karray_delete(
   char  **array,
   kaddr identifier,
   int   num)
{
	int entry, numleft;


	/*
	 *  Sanity check...
	 */
	if (array == NULL)
	   return(NULL);

	/*
	 *  Make sure that the entry exists
	 */
	if ((entry = karray_locate(array, identifier, num)) == -1)
	   return(array);

	/*
	 *  If the number left over is 0 and the entry we just delete is 0
	 *  this means that the array is empty and we need to free it.
	 */
	if ((numleft = num - (entry+1)) == 0)
	{
	   /*
	    * entry == 0 means we can finally delete the list.
            * note: never reallocate the list, since a pointer to it obtained
            *       by the caller may be forced to be prematurely freed.
	    */
	   if (entry == 0)
	   {
	      kfree(array);
	      return(NULL);
	   }
	}
	else
	   kmemcpy(&array[entry], &array[entry+1], numleft*sizeof(char *));

	return(array);
}


/************************************************************
*
*  Routine Name: karray_free - reclaim memory for an array of objects
*
*       Purpose: This module is used to reclaim all the memory associated
*		 with an array of objects.  One of the parameters
*		 is an element routine pointer that will allow
*		 the user to specify a routine to be used to destroy
*		 each element of the array.  This routine must
*		 have a return type of int, and return TRUE on success
*		 or FALSE on fail.  It must also take a single argument
*		 as a parameter; the pointer to the element to reclaim, cast
*		 as a (char *).  If the routine parameter is NULL,
*		 the routine uses kfree.
*
*         Input: array       - the array that is to be reclaimed
*		 num         - the number of entries in the array
*		 routine     - the routine to destroy each element
*
*        Output: none
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions:
*    Written By: Mark Young & Charlie Gage
*          Date: Jul 03, 1992 13:57
*      Verified:
*  Side Effects:
* Modifications: Converted from vfreelist in Khoros 1.0 (SJ)
*
*************************************************************/

void karray_free(
   char **array,
   int  num,
   kfunc_void routine)
{
	int  i;

	if (!array)
	   return;

        for (i = 0; i < num; i++)
	   (!routine) ? kfree(array[i]) : (*routine)(array[i]);

	kfree(array);
}


/************************************************************
*
*  Routine Name: karray_locate - locate an entry in the array list
*
*       Purpose: Tries to locate an entry in the array list.  This
*		 is done by racing thru the array list until the
*		 desired identifier is found or the end of the
*		 array is encountered.  If the identifier exists
*		 then the array index is returned.  If the identifier
*		 doesn't exist then -1 is returned.
*
*
*	  Input: array      - The current array list in which the entry will be
*			      located.
*		 identifier - the identifier which identifies the entry to
*			      be located.
*		 num	    - the number of entries in the array
*
*        Output: none
*
*       Returns: returns the item's index or -1 if the identifier is not found.
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Nov 28, 1992 13:32
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int karray_locate(
   char  **array,
   kaddr identifier,
   int   num)
{
	int  i;

	/*
	 *  Sanity check to make sure that the linked list is not NULL
	 */
	if (array == NULL)
	   return(-1);

	for (i = 0; i < num; i++)
	{
	   if (((unsigned *) array[i]) == ((unsigned *) identifier))
	      return(i);
	}
	return(-1);
}


/************************************************************
*
*  Routine Name: karray_sort - sort an array of strings
*
*       Purpose: This module sorts an array of strings into
*		 ascending alphabetical order.  It uses
*		 kstrcmp to determine the lexigraphical order.
*
*         Input: array - the array to sort
*		 num  - the number of entries in the array
*		 duplicate_entries - toggle where TRUE allows duplicate entries
*				     and FALSE removes duplicate entries.
*
*        Output: none
*       Returns: a pointer to a sorted array, or NULL on error
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Jul 03, 1992 14:07
*      Verified:
*  Side Effects: The sorted array passed back is the same as the original
*		 array passed in.  So original input array should not be
*		 sorted.
* Modifications:
*   Declaration: char **karray_sort(
*		 !   char **array,
*		 !   int  num,
*		 !   int duplicate_entries)
*
*************************************************************/

#ifndef NEED_QSORT
static int compare_entry(
   const void *str1,
   const void *str2)
{
	return(kstrcmp(*((char **) str1), *((char **) str2)));
}
#endif

char **karray_sort(
   char **array,
   int  num,
   int  duplicate_entries)
{
#ifdef NEED_QSORT 
	int	 i, j;
	register char  *temp;
	register int   sorted, compare;

	if (array == NULL)
	   return(NULL);

	for (i = 0; i < num; i++)
	{
	   sorted = TRUE;
	   for (j = 1; j < num - i; j++)
	   {
	      compare = kstrcmp(array[j-1], array[j]);
	      if (compare > 0)
	      {
		 temp = array[j-1];
		 array[j-1] = array[j];
		 array[j] = temp;
		 sorted = FALSE;
	      }
	      else if (compare == 0 && duplicate_entries == FALSE)
	      /*EMPTY*/
	      {
	      }
	   }
	   if (sorted == TRUE) break;
	}
#else
	if (array == NULL)
	   return(NULL);

	qsort((void *) array, num, sizeof(char *), compare_entry);
#endif
	return(array);
}


/************************************************************
*
*  Routine Name: karray_merge - merge two arrays of strings into one
*
*       Purpose: This module is used to merge two arrays into a
*		 single array.  Note that the entries in each array
*		 are only copied if the "copy_entries" parameter
*		 is TRUE.  Otherwise, it just points to the old string.
*		 Note:  The arrays are merged by concatenating array2 onto
*		 the end of the first.
*
*         Input: array1	      - first array of strings
*		 array2       - second array of strings
*		 num1         - number of entries in array1
*		 num2         - number of entries in array2
*		 copy_entries - toggle to determine whether to copy pointers
*				or the string
*
*        Output: none
*       Returns: a pointer to the merged array, or NULL on error.
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Jul 03, 1992 14:07
*      Verified:
*  Side Effects: kmalloc's space for new array
* Modifications: Converted from vmergelist in Khoros 1.0 (SJ)
*
************************************************************/

char **karray_merge(
   char **array1,
   char **array2,
   int  num1,
   int  num2,
   int  copy_entries)
{
	int  i, num, indx = 0;
	char **temp;


	if (array1 == NULL && array2 == NULL)
	   return(NULL);

	num = num1+num2;
	if ((num % KMAXENTRIES) != 0)
	   num += KMAXENTRIES - (num % KMAXENTRIES);
	temp = (char **) kcalloc(num, sizeof(char *));

	if (array1 != NULL)
	{
	   for (i = 0; i < num1; i++, indx++)
	   {
	      if (copy_entries == TRUE)
	         temp[indx] = kstrdup(array1[i]);
	      else
	         temp[indx] = array1[i];
	   }
	}

	if (array2 != NULL)
	{
	   for (i = 0; i < num2; i++, indx++)
	   {
	      if (copy_entries == TRUE)
	         temp[indx] = kstrdup(array2[i]);
	      else
	         temp[indx] = array2[i];
	   }
	}
	return(temp);
}


/************************************************************
*
*  Routine Name: karray_to_list - convert the array list into a linked list
*       Purpose: Creates a linked list from an array list.  This is
*		 done by calling klist_insert() with each entry of the
*		 the array.
*	  Input: array - The current linked list in which the array
*			 of identifiers will be created.
*		 num   - The number of entries in the array.
*       Returns: The newly created link list or NULL upon failure
*    Written By: Mark Young
*          Date: Nov 28, 1992 14:28
*************************************************************/

klist *karray_to_list(
   char **array,
   int  num)
{
	int   i;
	klist *list = NULL;


	/*
	 *  Copy each list entry identifier into the array
	 */
	for (i = 0; i < num; i++)
	   list = klist_insert(list, array[i], NULL, KLIST_TAIL, TRUE);

	return(list);
}


/************************************************************
*
*  Routine Name: karray_to_string - convert the array list into a string
*       Purpose: Creates a string from an array list.  This is
*		 done by calling kstrcat() with each entry of the
*		 the array.  The whitespace entry is used to insert
*		 the string between each entry of the array. Therefore
*		 if you an array with:
*
*		 !  array[0] = "ls"
*		 !  array[1] = "-a"
*		 !  array[2] = "/tmp"
*
*		 and a delimter of " ", then you will get a returning
*		 string of:
*
*		 !  "ls -a /tmp"
*
*		 If the delimter is NULL then the resulting string would
*		 be:
*
*		 !  "ls-a/tmp"
*	  Input: array    - The current linked list in which the array
*			  of identifiers will be created.
*		 num      - The number of entries in the array.
*		 delimter - Inserts the string delimiter between
*			    each entry of the array list.
*       Returns: The newly created string or NULL upon failure
*    Written By: Mark Young, Tom Sauer, John Salas
*          Date: Nov 28, 1992 14:28
*************************************************************/

char *karray_to_string(
   char **array,
   int  num,
   char *delimter)
{
	int   i;
	char  *temp, *string = NULL;


	/*
	 *  sanity check
	 */
	if (num <= 0)
	   return(NULL);

	/*
	 *  Copy each list entry identifier into the array
	 */
	for (i = 0; i < num - 1; i++)
	{
	   temp = kstring_3cat(string, array[i], delimter, NULL);
	   kfree(string); string = temp;
	}
	temp = kstring_cat(string, array[i], NULL);
	kfree(string); string = temp;
	return(string);
}


/************************************************************
*
*  Routine Name: karray_filelist - create an array of strings from a file
*
*       Purpose: This module is used to create an array from the
*		 supplied file.  It reads the lines out of the
*		 specified file, and returns them in an array of
*		 strings.  The programmer also has the option of
*		 specifying if they want the resulting string sorted
*		 or not.
*
*         Input: filename    - base name of the files to be arrayed.
*		 global_dir  - the global directory is used as a prefix
*			       to the filename
*		 sort_flag   - a flag indicating whether we are to sort
*			       the array before returning
*		 num         - the number of entries in the array that are
*			       returned.
*
*        Output: none
*       Returns: returns an array of strings to the lines in the file
*		 upon successfull completion of this routine, otherwise
*		 NULL is returned.
*
*  Restrictions: 
*    Written By: Mark Young
*          Date: Mar 20, 1995
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

char **karray_filelist(
   char *filename,
   char *global_dir,
   int  sort_flag,
   int  *num)
{
	kfile   *file;
	int	inum = 0;
	char    path[KLENGTH], temp[KLENGTH], **array = NULL;


	if (num) *num= 0;
	if (global_dir == NULL && filename == NULL)
	{
	   errno = KNULL_PARAMETER;
	   kerror("kutils", "kfilearray",
		  "NULL input filename and directory encountered");
	   return(NULL);
	}

	/*
	 *  Expand the path to get the full path to the directory.
	 */
	if (kstrlen(filename) == 0)
	   kfullpath(global_dir, NULL, path);
	else
	   kfullpath(filename, global_dir, path);

	/*
	 *  Open the file
	 */
	if ((file = kfopen(path, "r")) == NULL)
	{
	   kerror("kutils", "karray_filelist", "Unable to open file '%s'",path);
	   return(NULL);
	}

	while (kfgets(temp, KLENGTH, file))
	   array = karray_insert(array,kstrdup(temp), inum++, KLIST_TAIL, TRUE);

	/*
	 *  Need to see if the user wants to sort the entries into
	 *  ascending order.
	 */
	if (inum > 0 && sort_flag == TRUE)
	   array = karray_sort(array, inum, TRUE);

	if (num) *num= inum;
	return(array);
}

/************************************************************
*
*  Routine Name: karray_dirlist - create an array of file names
*
*       Purpose: This module is used to create an array of file
*		 names according to a user supplied basename and an
*		 initial global directory.  The list mode is used
*		 to indicate what we are going to list in the
*		 directory.  The possible defines are listed in
*		 $BOOTSTRAP/include/kutils/karray.h.  The following
*		 symbols are the current list mode:
*
*		 !	KPATH		- prepend the path to each file
*		 !	KFILE		- list plain text files
*		 !	KDIR			- list directories
*		 !	KDOT		- list dot files
*		 !	KLINK		- list symbolic files
*		 !	KSOCK		- list socket files
*		 !	KREAD		- file is readable by caller
*		 !	KWRITE		- file is writable by caller
*		 !	KEXEC		- file is executable by caller
*		 !	KRECURSE	- recursively list all subdirectories
*
*		The selections are or'ed together in order to choose
*		the set of attributes that are desired.  (e.g KFILE | KDIR)
*		will list only files and directories that match the basename.
*
*         Input: basename    - base name of the files to be listed.
*		 global_dir  - the global directory is used as a prefix to
*			       basename
*		 filter	     - the filter is used in matching the directory
*			       entries.  The syntax used is the same as the
*			       "kparse" utilities.  If filter is NULL then
*			       all entries are accepted.
*		 list_mode   - a flag indicating what we are to list
*			       (ie. files, directories, dot files, symbolic
*				links, etc.)
*		 format      - whether the files should be prepended with
*			       type of file. (ie. "@" for sym links, "/" for
*			       directories)
*
*        Output: num         - the number of entries in the array that
*			       are returned
*
*       Returns: an array of strings to filenames that match the basename
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Jul 03, 1992 13:58
*      Verified:
*  Side Effects:
* Modifications: Converted from vlistdir in Khoros 1.0 (SJ)
*
*************************************************************/

char **karray_dirlist(
   char *basename,
   char *global_dir,
   char *filter,
   int  list_mode,
   int  format,
   int  *num)
{
	DIR	*dir;
	struct  dirent *file;

	struct  stat buf;
	int	i, check_perm, base_len, status, inum = 0, dirnum = 0;
	char    *dirpath, *fullpath, entry[KLENGTH], **array = NULL,
		**dirlist = NULL;

	*num = 0;
	if (global_dir == NULL && basename == NULL)
	{
	   errno = KNULL_PARAMETER;
	   kerror("kutils", "kdirarray",
		  "NULL input basename and directory encountered");
	   return(NULL);
	}

	/*
	 *  Expand the path to get the full path to the directory.
	 */
	if (kstrlen(basename) == 0)
	{
	   kstring_cat(global_dir, "/", entry);
	   fullpath = kfullpath(entry, NULL,  NULL);
	}
	else
	   fullpath = kfullpath(basename, global_dir, NULL);

	/*
	 *  Get the directory path to the file
	 */
	if (stat(fullpath, &buf) == -1 || !S_ISDIR(buf.st_mode))
	{
	   if ((dirpath = kdirname(fullpath,NULL)) == NULL)
	   {
	      kfree(fullpath);
	      return(NULL);
	   }
	   basename = kbasename(fullpath,NULL);
	}
	else
	{
	   basename = NULL;
	   dirpath  = kstrdup(fullpath);
	}

	/*
	 *  Open the directory
	 */
	if ((dir = opendir(dirpath)) == NULL)
	{
	   kfree(fullpath); kfree(dirpath); kfree(basename);
	   return(NULL);
	}
	base_len = kstrlen(basename);

	/*
	 *  Read the entries and see if the file compares to that of the
	 *  basename we are looking for.  If the basename is NULL then
	 *  we add all regular files.
	 */
	check_perm = 0;
	if (list_mode & KREAD)  check_perm |= R_OK;
	if (list_mode & KWRITE) check_perm |= W_OK;
	if (list_mode & KEXEC)  check_perm |= X_OK;
	while ((file = readdir(dir)) != NULL)
	{
	   if (basename != NULL)
	   {
	      if (kstrncmp(file->d_name, basename, base_len) != 0)
	         continue;
	   }
	   else if ((list_mode & KDOT) == 0 && file->d_name[0] == '.')
	   {
	      if (((list_mode & KDIR) && kstrcmp(file->d_name,"..") != 0) ||
		  ((list_mode & KDIR) == 0))
	         continue;
	   }

	   /*
	    *  Check to see if the user wants to look at file permissions.
	    *  If so then we check the access of the "check_perm" mode(s).
	    */
	   if (dirpath[kstrlen(dirpath)-1] == '/')
	      (void) ksprintf(entry, "%s%s", dirpath, file->d_name);
	   else
	      (void) ksprintf(entry, "%s/%s", dirpath, file->d_name);
	   if (check_perm)
	   {
	      if (kaccess(entry, check_perm) == -1)
	         continue;
	   }

	   /*
	    *  Need to get infomation (status) about the file.  If the user
	    *  wants to know about symbolic links then we'll call lstat()
	    *  otherwise we will call stat().
	    */
#ifndef SYSV
	   if (list_mode & KLINK)
	   {
	      if (lstat(entry, &buf) == -1)  /*  get stats for file */
	         continue;
	   }
	   else
#endif
	   {
	      if (stat(entry, &buf) == -1)  /*  get stats for file */
	         continue;
	   }

	   /*
	    *  If we are suppose to recurse then we better save the directories
	    *  so that we don't have to rewind and stat all the files again.
	    */
	   if ((list_mode & KRECURSE) && S_ISDIR(buf.st_mode) &&
	       kstrcmp(file->d_name,"..") != 0 &&
	       kstrcmp(file->d_name,".")  != 0)
	   {
	      dirlist = karray_add(dirlist, kstrdup(entry), dirnum++);
	   }

	   /*
	    *  Unless the want the entire path, get rid of it.
	    */
	   if ((list_mode & KPATH) == 0)
	      (void) ksprintf(entry, "%s", file->d_name);


	   /*
	    *  If the filter is not NULL then see if the entry matches
	    *  what we are looking for.
	    */
	   if (filter != NULL)
	   {
	      (void) kparse_string_search(entry,filter,KLITERAL,NULL,&status);
	      if (status != KPARSE_OK)
		 continue;
	   }

	   /*
	    *  Check the kind of file we have.  Make sure it part of the
	    *  array of files to be be arrayed.
	    */
	   if (S_ISDIR(buf.st_mode) && (list_mode & KDIR))
	   {
	      if (format) kstrcat(entry,"/");
	   }

#if defined(S_ISLNK)
	   else if (S_ISLNK(buf.st_mode) && (list_mode & KLINK))
	   {
	      if (format) kstrcat(entry,"@");
	   }
#endif

#if defined(S_ISSOCK)
	   else if (S_ISSOCK(buf.st_mode) && (list_mode & KSOCK))
	   {
	      if (format) kstrcat(entry,"=");
	   }
#endif
	   else if (S_ISREG(buf.st_mode) && (list_mode & KFILE))
	   {
	      if (format) kstrcat(entry," ");
	   }
	   else
	      continue;

	   array = karray_add(array, kstrdup(entry), inum); inum++;
	}

	/*
	 *  We need to make sure that the array of filenames is greater than 0.
	 *  If not then kfree the array and set it to NULL, otherwise we need
	 *  to sort the entries into descending order.
	 */
	if (inum == 0)
	{
	   kfree(array);
	   array = NULL;
	}
	else
	{
	   array = karray_sort(array, inum, TRUE);
	}

	if (list_mode & KRECURSE)
	{
	   char **list;
	   char **tmp;
	   int  lnum = 0;

	   for (i = 0; i < dirnum; i++)
	   {
	      list = karray_dirlist(dirlist[i], NULL, filter, list_mode,
					format, &lnum);
	      tmp = karray_merge(array, list, inum, lnum, FALSE);
	      kfree(array);
	      kfree(list);
	      array = tmp;
	      inum += lnum;
	   }
	   karray_free(dirlist, dirnum, NULL);
	}
	*num= inum;

	/*
	 *  Cleanup time.  Free the fullpath and dirpath, and the basename if
	 *  currently kallocated.
	 */
	kfree(fullpath); kfree(dirpath);
	if (basename != NULL) kfree(basename);
	closedir(dir);
	return(array);
}
