/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>		   Form Tree Creation Routines                <<<<
   >>>>                                                       <<<<
   >>>>  Private:                                             <<<<
   >>>>                kvf_build_form()                       <<<<
   >>>>   Static:                                             <<<<
   >>>>                kvf_fill_in_subform()                  <<<<
   >>>>                kvf_fill_in_master()                   <<<<
   >>>>                kvf_fill_in_guide()                    <<<<
   >>>>                kvf_fill_in_pane()                     <<<<
   >>>>                kvf_fill_in_control()                  <<<<
   >>>>                kvf_fill_in_toggle()                   <<<<
   >>>>                kvf_link_submenu_group()               <<<<
   >>>>                kvf_fill_in_group()                    <<<<
   >>>>   Public:                                             <<<<
   >>>>                kvf_create_form()                      <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<  */

#include "internals.h"


/* createtree.c */
static int kvf_fill_in_master  PROTO((char **, kform *, kcontrol *,
                                      int *, int *, int *, int *, char **));
static int kvf_fill_in_subform PROTO((char **, kform *, ksubform *, 
                                      int *, int *, int *, char **));
static int kvf_fill_in_guide   PROTO((char **, kform *, 
                                      ksubform *, kcontrol *, int *, 
                                      int *, int *, int *, char **));
static int kvf_fill_in_control PROTO((char **, kform *, 
                                      ksubform *, kguide *, 
                                      kcontrol *, int *, int *, int *, 
                                      int *, char **));
static int kvf_fill_in_toggle  PROTO((char **, kselection *, int *, 
                                      kform *, ksubform *, kguide *, 
                                      kcontrol *, int *, int *, char **));
static int kvf_fill_in_group   PROTO((char **, kselection *, int *, 
                                      kform *, ksubform *, kguide *, 
                                      kcontrol *, int *, int *, char **, 
                                      int *));
static int kvf_link_submenu_group PROTO((char **database, kcontrol *, 
                                         int *));

static kcontrol *kvf_fill_in_pane PROTO((kform *, ksubform *, 
                                            kguide *, char **, int *, 
                                            int *, int *, char **, char ***, 
                                            int *, int *));
#define NO_GROUP_DEFAULT 1
#define GROUP_DEFAULT    2


/************************************************************
*
*  Routine Name: kvf_create_form - create a form tree from a UIS file
*
*       Purpose: Takes a UIS file, and constructs the internal representation
*                of the associated user interface (the form tree is used with
*                both the GUI and the CLUI).
*
*        Input: filename         - UIS file specifying user interface.
*               glyph_type       - Type of glyph to use w/ form: SIMPLE or NONE
*                                  (ignored by CLUI; pass NONE)
*               callback_routine - callback routine for when the GUI is used 
*                                  (ignored by CLUI; pass NULL)
*               client_data      - client data for callback routine (if applic)
*                                  (ignored by CLUI; pass NULL)
*               x, y             - location at which to place the newly created
*                                  GUI. If values of x and y are both -1,
*                                  placement of the GUI must be done manually.
*                                  (ignored by CLUI; pass 0,0)
*        Output: 
*       Returns: The pointer to the form tree representing the user interface
*                specified by the UIS file passed in on success, NULL on 
*                failure.
*  Restrictions: 
*    Written By: Danielle Argiro
*          Date: March 20, 1994
*      Verified:
*  Side Effects: 
* Modifications:
*
*************************************************************/
kform *kvf_create_form(
   char       *filename,
   int        glyph_type,
   kfunc_void callback_routine,
   kaddr      client_data)
{
        kfile   *file;
        kform   *form;
        char    **database, **filenames;
        int     i, line_num, db_size, filecount = 0;
        int     *filename_lookup, *linenum_lookup;

	/* open the UIS file */
        if ((file = kfopen(filename, "r")) == NULL)
        {
            kerror("kforms", "xvf_create_form",
                    "Unable to open '%s' to read UIS", filename);
            return(NULL);
        }
        if ((filenames = (char **) kcalloc(1,sizeof(char *))) == NULL)
        {
            kerror("kforms", "xvf_create_form",
                   "Unable to allocate internal filenames array");
            return(NULL);
        }
        filenames[0] = kstrdup(filename);

	/* read UIS file into internal database */
        database = kvf_read_database(file, &line_num, &db_size,
                                     &filename_lookup, &linenum_lookup,
                                     &filenames, &filecount);
        kfclose(file);

        if (database == NULL)
            return(NULL);

        /*
         * build the form tree
         */
        form = kvf_build_form(database, glyph_type, callback_routine,
                              client_data, filename_lookup,
                              linenum_lookup, filenames);

	/*
	 *  save the location of the UIS file defining this form
	 */
	if (form != NULL) form->uis_location = kstrdup(filename);

        /*
         * create the objects associated w/ the master form or main subform
         */
        if (filecount == 0) kfree(filenames[0]);
        for (i = 0; i < filecount; i++)
            kfree(filenames[i]);
        kfree(filenames);
        kfree(filename_lookup); kfree(linenum_lookup);

        for (i = 0; i < line_num; i++)
           kfree(database[i]);
        kfree(database);


	return(form);
}


/*-------------------------------------------------------------
|
| Routine Name:  kvf_build_form
|
|      Purpose: This is the main driver for the routines that create
|		the form tree (the internal abstract data structure
|               the represents the GUI of an application).  In some very
|               special cases (conductor and ghostwriter), this routine may 
|               be called directly from the application in order to create 
|               the form tree but NOT create any physical GUI.  A previous
|               call to kvf_read_database must be made to acquire the array 
|               of strings that internally represents the UIS;  this array is
|               passed in as the 'database' parameter.  kvf_read_database
|               also returns the 'filename_lookup', 'linenum_lookup', and
|               'filenames' arrays that must be passed in for correct printing
|               of UIS syntax & logic error messages.
|
|               IMPORTANT NOTE: Use this routine at your own risk!  
|               In general, all applications should call kvf_create_form(),
|               which calls this routine at the right time.
|
|        Input: database         - Array of strings holding UIS; returned
|                                  by kvf_read_database().
|		glyph_type       - Type of glyph to use w/ form: SIMPLE or NONE
|		callback_routine - optional callback routine for this form 
|	        x, y             - location at which to place the newly created 
|                                  GUI. If values of x and y are both -1,
|                                  placement of the GUI must be done manually.
|               filename_lookup  - indexes into 'filenames' array 
|               linenum_lookup   - holds actual line numbers associated with
|                                  UIS lines, as opposed to the indices into
|                                  the internal UIS 'database'
|               filenames        - holds names of all files read in as part of
|                                  UIS (used with -p and -k UIS lines).
|
|               Note: The latter three arguments are used solely to print out
|                     correct information when reporting syntax and logic 
|                     errors in the UIS. 
|
|        Output: none
|       Returns: Returns a pointer to the form tree on success,
|                NULL on failure
|          Date: Jul 16, 1992 09:56
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

kform *kvf_build_form(
   char  **database,
   int   glyph_type,
   kfunc_void callback_routine,
   kaddr client_data,
   int   *filename_lookup,
   int   *linenum_lookup,
   char  **filenames)
{
	kform 	  *form;
	ksubform  *subform;
	Line_Info line_info;
	int 	  done, flag, has_submenu; 
	int	  subform_num = 0, subformbutton_num = 0;
	int	  i, kvf_index = 0, save_index, already_selected;
	char 	  *comment;
	char	  **subform_names;

	done = FALSE;  has_submenu = FALSE;

	if ((subform_names = (char **) kcalloc(1,sizeof(char *))) == NULL)
	{
	    kerror("kforms", "kvf_build_form",
		   "Unable to allocate internal subform_names array");
	    return(NULL);
	}

	/*
	 *  Initialize the forms structures
	 */
	kvf_initialize();

	kvf_clear_line_info(&line_info);

	if (database == NULL)
	{
	    errno = KINVALID_UISFILE;
            kerror("kforms", "kvf_create_form", 
		   "Apparently empty UIS found.  Aborting form creation");
	    return(NULL);
	}

	/* allocate room for the form structure & return on failure */
	if ((form = (kform *) kcalloc(1,sizeof (kform))) == NULL)
	{
            kerror("kforms", "kvf_build_form", 
		   "Unable to allocate internal form structure");
	    return(NULL);
    	}

/*
	form->back_kformstruct = kvf_create_struct((kaddr) form,
                                                   KUIS_STARTFORM, KFORM);
        kvf_link_formstruct(form->back_kformstruct);
*/

	/* set miscellaneous form pointer fields */
	form->quit  = FALSE;	     


	/**** start with (-F) flag: create form pointer & set fields  ****/

	form->control_comment = kvf_get_comments(database, &kvf_index);
        flag = kvf_get_line_type(database[kvf_index]);
	if (flag != KUIS_STARTFORM)
	{
	    errno = KUIS_SYNTAX;
	    kerror("kforms", "kvf_create_form", 
		   "UIS definition must begin with [-F] line");
            return(NULL);
	}

	/* parse the Start Form line */
        if (!(kvf_parse_startform_line(database[kvf_index],&line_info)))
	{
	    kvf_parsing_error_mesg(kvf_index, 
                              filename_lookup, linenum_lookup, filenames);
	    return(NULL);
	}
	/* using "master" as the variable name on the -F line will cause
           conductor to generate code that will produce warnings on a
           strict compiler */
        if (kstrcmp(line_info.variable, "master") == 0)
        {
	    errno = KUIS_SYNTAX;
            kerror("kforms", "kvf_create_form",
                   "The variable name 'master' may not be used on the -F line, as this is a word reserved for use by conductor;  please use another variable name.");
            return(NULL);
        }
	form->var_token = kstring_to_token(line_info.variable);
	form->control_line = kstrdup(database[kvf_index]);

	/* set callback for form */
	kvf_add_entry((kaddr) form, KFORM, TRUE, callback_routine, client_data);

	/* set remaining formptr fields except subform struct */
        kvf_index++; 
	comment = kvf_get_comments(database, &kvf_index);
	flag = kvf_get_line_type(database[kvf_index]);
	if (flag == -1) return(NULL);

	/*
	 *  Check to see if this is an empty form.  If so then we are done,
	 *  since there is nothing left to fill in or check.
	 */
	if (flag == KUIS_END) return(form);

	flag = kvf_get_line_type(database[kvf_index]);

	/**** next is the (-S) flag (if there is a master) or (-M) flag ****/

	if ((flag != KUIS_STARTMASTER) && (flag != KUIS_STARTSUBFORM))
	{
	    errno = KUIS_SYNTAX;
	    kerror("kforms", "kvf_create_form", "StartForm [-F] line must be followed by either a StartMaster [-S] line or StartSubform [-M] line;  Unable to create forms.");
            return(NULL);
	}


	/**** (-S) flag: they have defined a Master Form *****/

	if (flag == KUIS_STARTMASTER)
	{
	    /* allocate room for the form structure & return on failure */
	    if ((form->master = (kcontrol *) 
				kcalloc(1, sizeof (kcontrol))) == NULL)
	    {
                kerror("kforms", "kvf_build_form", 
		       "Unable to allocate internal master form structure");
	        return(NULL);
    	    }

	    form->master->var_token      = form->var_token;
	    form->master->type           = KMASTER;
	    form->master->control_comment= kstrdup(comment);
	    kfree(comment);
	    form->master->back_form      = form;

	    form->master->back_kformstruct = 
			kvf_create_struct((kaddr) form->master,
                                          KUIS_STARTMASTER, KMASTER);
            kvf_link_formstruct(form->master->back_kformstruct);

	    /* parse the Start Master line */
	    if (!(kvf_parse_startmaster_line(database[kvf_index], &line_info)))
	    {
		kvf_parsing_error_mesg(kvf_index, 
                              filename_lookup, linenum_lookup, filenames);
		return(NULL);
	    }
	    if (!(kvf_check_for_subform_buttons(database, kvf_index)))
	    {
	        errno = KUIS_LOGIC;
                kerror("kforms", "kvf_create_form", "Unnecessary Master Form [-S to -E] definition in file %s;  Master Form definitions may only be used with 2 or more subform buttons (-d or -u lines).  Either (1) move any selections in the Master Form definition into a Subform [-M to -E] definition, or (2) add at least two subform buttons to the Master Form definition.", filenames[filename_lookup[kvf_index]]);
                return(NULL);
	    }

	    form->master->excl_subforms = line_info.logical_val;
	    form->master->control_line = kstrdup(database[kvf_index]);
	    kvf_index++;

	    /* fill in the master form w/ form buttons, etc */
	    save_index = kvf_index;
	    if (!(kvf_fill_in_control(database, form, NULL, NULL, 
				      form->master, &kvf_index, &has_submenu,
				      filename_lookup, linenum_lookup, 
				      filenames)))
		return(NULL);
	    kvf_index = save_index;
	    if (!(kvf_fill_in_master(database, form, form->master, &kvf_index,  
			             &subformbutton_num, filename_lookup, 
				     linenum_lookup, filenames)))
                return(NULL);
	    if (has_submenu)
	    {
	        kvf_index = save_index;
	        if (!(kvf_link_submenu_group(database, form->master, 
					     &kvf_index)))
	    	    return(NULL);
	    }

	    
	}

	else
	{
	     form->master = NULL;
        }


	/*****  now a series of (-M) subform definitions should follow *****/

	while (!done)
	{
	     if (!((subform_num == 0) && (flag != KUIS_STARTMASTER)))
	     {
		 kfree(comment);
	         comment = kvf_get_comments(database, &kvf_index);
	     }
             flag = kvf_get_line_type(database[kvf_index]);
	     if (flag == -1) return(NULL);

	     if (flag == KUIS_STARTSUBFORM)
	     {
		if (!(kvf_parse_startsubform_line(database[kvf_index], 
				&line_info))) 
		{
		    kvf_parsing_error_mesg(kvf_index, 
                              filename_lookup, linenum_lookup, filenames);
		    return(NULL);
		}


		/*
	         *  check to be sure that subform variable name is not 
		 *  the same as the one on the -F line
	         */
		if (kstrcmp(line_info.variable, 
			    ktoken_to_string(form->var_token)) == 0)
		{
	            errno = KUIS_LOGIC;
                    kerror("kforms", "kvf_create_form", "variable on line %d of %s cannot be the same as that on the -F line", linenum_lookup[kvf_index], filenames[filename_lookup[kvf_index]]);
                    return(NULL);
		}

		/*
	         *  check to be sure that subform variable name is not 
		 *  the same as the one on any other -M line
	         */
		for (i = 0; i < subform_num; i++)
		{
		    if (kstrcmp(line_info.variable, subform_names[i]) == 0)
		    {
	                errno = KUIS_LOGIC;
                        kerror("kforms", "kvf_create_form", "variable on line %d of %s cannot be the same as the variable name on any other -M line in the UIS", linenum_lookup[kvf_index], filenames[filename_lookup[kvf_index]]);
                        return(NULL);
		    }
		}

		if ((subform_names = (char **) krealloc(subform_names,
				(subform_num+1) * sizeof(char*))) == NULL)
		{
            	    kerror("kforms", "kvf_build_form", 
		           "Unable to re-allocate subform_names array");
	            return(NULL);
		}
		subform_names[subform_num++] = kstrdup(line_info.variable);

		/* 
		 * if there is no KUIS_STARTMASTER line, the form->subform 
		 * struct will not have been allocated in kvf_fill_in_master.
		 * There will be only a single subform on the form--
		 * Create an "empty subform" for the sake of bookkeeping
		 */
		if (form->master == NULL)
		{
		   /* allocate room for the subform structure */
		   if ((form->subform = (ksubform *) 
		           kcalloc(1,sizeof (ksubform))) == NULL)
		   {
			kvf_allocation_error_mesg();
	    	        return(NULL);
		   }
		   form->subform->button = NULL;
		   form->subform->control_comment = kstrdup(comment);

		   form->subform->line  = NULL;
		   form->subform->quit  = FALSE;
		   form->subform->guide = NULL;
		   form->subform->type  = KUIS_SUBFORMBUTTON;
		   form->subform->next  = NULL;
		   form->subform->selected = TRUE;
                   form->subform->control_line = kstrdup(database[kvf_index]);
		   form->subform->back_form = form;

		   form->subform->back_kformstruct = 
				kvf_create_struct((kaddr) form->subform,
                                                   KUIS_STARTSUBFORM, KSUBFORM);
                   kvf_link_formstruct(form->subform->back_kformstruct);

                   subform = form->subform;
		   /* still need to parse the KUIS_STARTSUBFORM (-M) line:
	              don't increment kvf_index before creating subform */
		   subform->control_comment = kstrdup(comment);
		   if (!(kvf_fill_in_subform(database, form, subform, 
			                     &kvf_index, filename_lookup, 
			                     linenum_lookup, filenames)))
		       return(NULL);
		}
	        else
		{
		   /*
		    *  run thru subform list, creating subform branches
		    */
		    subform = form->master->subform_list;
		    already_selected = FALSE;
		    while (subform != NULL)
		    {
			/*
                         * a little trick - used selected field to store
                         * the index of the -M line temporarily. now,
                         * restore proper value of selected.
                         */
                        kvf_index = subform->selected;

			kvf_clear_line_info(&line_info);
			kvf_gen_parse(subform->line, &line_info);
			if ((form->master->excl_subforms) &&
			    (already_selected) &&
			    (line_info.selected))
			{
	                    kerror("kforms", "kvf_create_form", "Only one (or none) subform buttons may have their selected field set to TRUE (indicating automatic mapping) when the master form has Mutual Exclusion of Subforms set to TRUE (-S 1 1)."); 
                            return(NULL);
			}
			subform->selected = line_info.selected;
			if (subform->selected) already_selected = TRUE;
			kvf_free_line_info_strings(&line_info);

                        /*
                         * but now, need to backtrack to find comment (if one)
                         */
                        flag = kvf_get_line_type(database[kvf_index-1]);
                        while (flag == KUIS_COMMENT)
                        {
                            kvf_index--;
                            flag = kvf_get_line_type(database[kvf_index-1]);
                        }
                        comment = kvf_get_comments(database, &kvf_index);
                        subform->control_comment = kstrdup(comment);

			/* 
			 * still need to parse the KUIS_STARTSUBFORM (-M) line:
			 * don't increment kvf_index before creating subform 
			 */
			 if (!(kvf_fill_in_subform(database, form, subform, 
				 &kvf_index, filename_lookup, linenum_lookup, 
				 filenames)))
			      return(NULL);

			 kfree(comment);
			 comment = kvf_get_comments(database, &kvf_index); 
			 subform = subform->next;
		    }
		    done = TRUE;
		}
		kvf_free_line_info_strings(&line_info);
		 
	     }  /* end if */
	
	     /*** flag (-E)  should be the end of the form definiton ***/

	     else if (flag == KUIS_END)  
	     {
		/* if (subformbutton_num == 1)
		{
		    errno = KUIS_SYNTAX;
	            kerror("kforms", "kvf_create_form", "UIS must have more than one subform button to qualify for use of a Master Form.  Eliminate unneeded master form [-S to -E] definition, or add additional [-d] subform button(s) and their associated subform [-M to -E] definitions."); 
                    return(NULL);
		}
		else */ if ((subform_num == 0) && (subformbutton_num > 0))
		{
		    errno = KUIS_SYNTAX;
	            kerror("kforms", "kvf_create_form", "There must be one Subform [-M to -E] definition in the UIS file for each (-d) SubformButton UIS line."); 
                    return(NULL);
	 	}
		else if ((subformbutton_num == 0) && (subform_num > 1))
		{
		    errno = KUIS_SYNTAX;
	            kerror("kforms", "kvf_create_form", "When more than one Subform [-M to -E] definition is used, there must be a Master Form [-S to -E] definition, with one (-d) SubformButton UIS line for each subform definition in the UIS file."); 
                    return(NULL);
	 	}
		else if (subformbutton_num > subform_num) 
		{
		    errno = KUIS_SYNTAX;
	            kerror("kforms", "kvf_create_form", "There are %d (-d) SubformButton UIS lines, but only %d Subform [-M to -E] definition(s) in the UIS file;  there must be one subform definition for each subform button.", subformbutton_num, subform_num); 
                    return(NULL);
	 	}
		else if ((subform_num > subformbutton_num) && (subform_num > 1))
		{
		    errno = KUIS_SYNTAX;
	            kerror("kforms", "kvf_create_form", "There are %d Subform [-M to -E] definition(s) in the UIS file, but only %d (-d) SubformButton UIS lines; there must be one subform button for each subform definition.", subform_num, subformbutton_num); 
                    return(NULL);
	 	}

                done = TRUE;
		
	     }  /* end else */

	     /* database line out of place */
	     else 
	     {
		errno = KUIS_SYNTAX;
	        kerror("kforms", "kvf_create_form","UIS line out of place on line %d in %s.", linenum_lookup[kvf_index], filenames[filename_lookup[kvf_index]]);
		return(NULL);
	     }

	}  /* end while */

	/* if only one subform, make sure it is selected */
	if (form->subform != NULL)
	{
	    form->subform->selected = TRUE;
	    kvf_parse_startsubform_line(form->subform->control_line,
					&line_info);
	    line_info.selected = TRUE;
	    kvf_deparse_startsubform_line(&line_info, 
					  &form->subform->control_line);
	    kvf_free_line_info_strings(&line_info);
	}

	for (i = 0; i < subform_num; i++) kfree(subform_names[i]);
	kfree(subform_names);

	return(form);

}  /* end kvf_build_form */




/*-----------------------------------------------------------------------------
|
|  Routine Name: kvf_fill_in_master
|
|       Purpose: Creates the list of subforms.
|
|         Input: database    - pointer to internal representation of UIS
|		 form	    - pointer to the form tree
|                master      - pointer to master control structure
|		 kvf_index   - index into the current line of UIS
|                filename_lookup - indexes into 'filenames' array
|                linenum_lookup  - holds actual line number of file for UIS line
|                filenames       - array of filenames comprising UIS
|
|        Output: subformbutton_num - number of buttons specified on the master
|				      that will bring up sub-forms
|       Returns: Returns TRUE on success, FALSE on failure
|          Date: Apr 10, 1992
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|		
----------------------------------------------------------------------------*/


static int kvf_fill_in_master(
   char     **database,
   kform    *form,
   kcontrol *master,
   int      *kvf_index,
   int      *subformbutton_num,
   int      *filename_lookup,
   int      *linenum_lookup,
   char     **filenames)
{
	Line_Info  line_info;
	ksubform   *subform, *last_subform = NULL;
	int	   done, flag, submenu, toggle, group; 
	char       *tmpline, *comment = NULL;
	kselection *tmpsel, *save_submenu_selptr;
	int        ok = TRUE;

	/* initialization */
	done = submenu = toggle = group = FALSE;
	save_submenu_selptr = NULL;
             
	*subformbutton_num = 0;
	kvf_clear_line_info(&line_info);

	while (!done) 
	{
	    /* 
	     *	see what type of line we have
 	     */
	    
	    comment = kvf_get_comments(database, kvf_index);
	    flag = kvf_get_line_type(database[*kvf_index]);
	    if (flag == -1) return(FALSE);
	    

	    switch (flag)
	    {
		case KUIS_STARTSUBMENU:
		     submenu = TRUE;
		     tmpsel = master->sel_list;
		     while (tmpsel != NULL) 
		     {
			tmpline = kstrdup(database[*kvf_index]);
		        kvf_parse_startsubmenu_line(tmpline, &line_info);
		        kvf_deparse_startsubmenu_line(&line_info, &tmpline);
			if (kstrcmp(tmpsel->line, tmpline) == 0)
			   save_submenu_selptr = tmpsel;
			kfree(tmpline);
		        tmpsel = tmpsel->next;
		     }
		     if (save_submenu_selptr == NULL)
		     {
			 errno = KINTERNAL;
		         kerror("kforms", "kvf_fill_in_master", 
		                "Can't find associated submenu, aborting form creation");
			 return(FALSE);
		     }
		     (*kvf_index)++;
		     break;
		     
		     /*
		      *  (-u)  pseudo subform button is obsolete in K2
		      */
                 case KUIS_PSEUDOSUBFORM:
                      kerror("kforms", "kvf_read_database",
                            "The (-u) UIS line specifying a pseudosubform button on line %s of %s is obsolete; pseudosubform buttons are not supported in Khoros 2.0.  Please change the pseudosubform button to a subform button, and verify that it is being used correctly in the UIS file before going on.", linenum_lookup[*kvf_index], filenames[filename_lookup[*kvf_index]]);
		      ok = FALSE;
                      break;

		     /*
		      *  (-d)  indicates a subform button on the master 
		      */
		case KUIS_SUBFORMBUTTON:

	     	     /* allocate room for the subform structure */
	     	     if ((subform = (ksubform *) 
					kcalloc(1, sizeof (ksubform))) == NULL)
	             {
		         kvf_allocation_error_mesg();
	                 return(FALSE);
	             }

		     /* parse the line */
		     if (!(kvf_gen_parse(database[*kvf_index], &line_info))) 
		     {
			kvf_parsing_error_mesg(*kvf_index, 
                              filename_lookup, linenum_lookup, filenames);
			return(FALSE);
		     }
		     subform->comment   = kstrdup(comment);
		     subform->var_token = kstring_to_token(line_info.variable);
	             subform->control_line = kvf_find_matching_M_line(database,
                                                *kvf_index, filename_lookup,
                                                linenum_lookup,  filenames,
                                                &subform->selected);
                     if (subform->control_line == NULL) return(FALSE);

		     /* set fields of subform that can be set now */
                     if (!(kvf_set_subform_values(master, subform, 
			      database[*kvf_index], KUIS_SUBFORMBUTTON, form, 
			      save_submenu_selptr, *kvf_index,
                              filename_lookup, linenum_lookup, filenames)))
			return(FALSE);

		     kvf_link_subform(master, subform, &last_subform);

		     if (submenu == TRUE)
		         subform->back_submenu = save_submenu_selptr;

		     (*subformbutton_num)++; 
		     (*kvf_index)++;
		     break; 

		case KUIS_TOGGLE:
		     toggle = TRUE;
		     (*kvf_index)++;
		     break;

		case KUIS_MUTEXCL:
		case KUIS_MUTINCL:
		case KUIS_GROUP:
		     group++;
	             (*kvf_index)++;
                     break;

		case KUIS_GUIDEBUTTON:
		case KUIS_STARTMASTER:
		case KUIS_STARTGUIDE:
		case KUIS_STARTFORM:
		case KUIS_STARTPANE:
		     errno = KUIS_SYNTAX;
		     kerror("kforms", "kvf_create_form",
			    "UIS line out of place on line %d of %s", 
                             linenum_lookup[*kvf_index], 
                             filenames[filename_lookup[*kvf_index]]);
		     ok = FALSE;
		     break;

		case KUIS_END:

		     /*
		      * (-E) At the end of this master form description
		      */ 
		    if (submenu == TRUE)
		    {
		       submenu = FALSE;
		       save_submenu_selptr = NULL;
		    }
		    else if (toggle == TRUE)
			toggle = FALSE;
		    else if (group > 0)
			group--;
		    else
		       done = TRUE;

		    (*kvf_index)++;
		    break;

                default:
		     (*kvf_index)++;
		    break;
	    }
	    if (!ok) return(FALSE);

	    kvf_free_line_info_strings(&line_info);
	    kfree(comment);

	} /* end while */
	return(TRUE);

} /* end kvf_fill_in_master() */


/*------------------------------------------------------------
|
|  Routine Name: kvf_fill_in_subform()
|
|       Purpose: Fills in the subform structure
|
|         Input: database - pointer to UIS description
|		 form     - pointer to the form tree
|		 subform  - pointer to this subform branch
|		 kvf_index- index to current line of the database
|                filename_lookup - indexes into 'filenames' array
|                linenum_lookup  - holds actual line number of file for UIS line
|                filenames       - array of filenames comprising UIS
|
|        Output: none
|       Returns: TRUE on success, FALSE on failure
|          Date: Apr 10, 1992
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
-------------------------------------------------------------*/

static int kvf_fill_in_subform(
   char     **database,
   kform    *form,
   ksubform *subform,
   int      *kvf_index,
   int      *filename_lookup,
   int      *linenum_lookup,
   char     **filenames)
{
	int 	  i, done, flag, has_submenu; 
	int       save_index, highest_index = 0;
	int       pane_num = 0, guidebutton_num = 0; 
	char 	  temp[KLENGTH], item_name[KLENGTH];
	char	  *comment = NULL;
	char	  **pane_names;
	Line_Info line_info;
	kguide    *guide;

	done = FALSE; has_submenu = FALSE;

	if ((pane_names = (char **) kcalloc(1,sizeof(char *))) == NULL)
	{
            kerror("kforms", "kvf_fill_in_subform", 
		   "Unable to allocate internal pane_names array");
	    return(FALSE);
	}

	kvf_clear_line_info(&line_info);

	/**** make sure we're starting with a (-M) KUIS_STARTSUBFORM line *****/

	flag = kvf_get_line_type(database[*kvf_index]);
	if (flag != KUIS_STARTSUBFORM)
	{
	    ksprintf(temp, "Error on line %d of %s;  UIS subform definition must begin with [-M] line;  Unable to create forms.", linenum_lookup[*kvf_index], filenames[filename_lookup[*kvf_index]]);
	    errno = KUIS_SYNTAX;
	    kerror("kforms", "kvf_create_form",  temp);
	    return(FALSE);
	}

	/* parse the KUIS_STARTSUBFORM line */
	if (!(kvf_parse_startsubform_line(database[*kvf_index], &line_info))) 
	{
	    kvf_parsing_error_mesg(*kvf_index, 
                              filename_lookup, linenum_lookup, filenames);
	    return(FALSE);
	}

	/* set the subforms's database to be the same as the form's */
	subform->var_token = kstring_to_token(line_info.variable);
	ksprintf(item_name, "%s.%s", 
		ktoken_to_string(form->var_token), line_info.variable);
        subform->name_token = kstring_to_token(item_name);

	if (subform->control_line == NULL)
	    subform->control_line = kstrdup(database[*kvf_index]);

	if ((!line_info.selected) && (form->master == NULL))
	{
	    line_info.selected = TRUE;
	    kvf_deparse_startsubform_line(&line_info, &subform->control_line);
	}
	kvf_free_line_info_strings(&line_info);
	(*kvf_index)++;


	/**** next is the (-G) flag (if a guide pane) or (-P) flag ***/

	comment = kvf_get_comments(database, kvf_index);
	flag = kvf_get_line_type(database[*kvf_index]);
	if (flag == -1) return(FALSE);

	if ((flag != KUIS_STARTGUIDE) && (flag != KUIS_STARTPANE))
	{
	     ksprintf(temp, "syntax error on line %d of %s - KUIS_STARTSUBFORM [-M] line must be followed by either a KUIS_STARTGUIDE [-G] line or KUIS_STARTPANE [-P] line;  Unable to create forms.", linenum_lookup[*kvf_index], filenames[filename_lookup[*kvf_index]]);
	     errno = KUIS_SYNTAX;
	     kerror("kforms", "kvf_create_form",  temp);
	     return(FALSE);
	}

	/**** (-G) flag: they have defined a Guide Pane on the Subform *****/

	if (flag == KUIS_STARTGUIDE)
	{
	    /* allocate room for the form structure & return on failure */
            if ((subform->guidepane = (kcontrol *)
                                kcalloc(1, sizeof (kcontrol))) == NULL)
            {
	        kerror("kforms", "kvf_fill_in_subform",  
			"Unable to allocate internal guidepane structure");
                return(FALSE);
            }

            subform->guidepane->type = KGUIDEPANE;
            subform->guidepane->control_line = kstrdup(database[*kvf_index]);
            subform->guidepane->control_comment = kstrdup(comment);
	    kfree(comment);
            subform->guidepane->back_subform = subform;
            subform->guidepane->back_form = form;

	    subform->guidepane->back_kformstruct =
			kvf_create_struct((kaddr) subform->guidepane,
                                           KUIS_STARTGUIDE, KGUIDEPANE);
            kvf_link_formstruct(subform->guidepane->back_kformstruct);

	    save_index = *kvf_index;
	    (*kvf_index)++;
	    if (!(kvf_fill_in_control(database, form, subform, NULL, 
				      subform->guidepane, kvf_index, 
				      &has_submenu, filename_lookup, 
				      linenum_lookup, filenames)))
	        return(FALSE);

	    *kvf_index = save_index;
	    if (!(kvf_fill_in_guide(database, form, subform, 
				    subform->guidepane, kvf_index, 
				    &guidebutton_num, filename_lookup, 
				    linenum_lookup, filenames))) 
		return(FALSE);
	    if (has_submenu)
	    {
	        *kvf_index = save_index;
	        if (!(kvf_link_submenu_group(database, subform->guidepane, 
					     kvf_index)))
		    return(FALSE);
 	    }

	}

	/*****  now a series of (-P) pane definitions should follow *****/

	while (!done) 
	{

	   if (!((pane_num == 0) && (flag != KUIS_STARTGUIDE)))
	      comment = kvf_get_comments(database, kvf_index);
	   flag = kvf_get_line_type(database[*kvf_index]);
	   if (flag == -1) return(FALSE);

	   if (flag == KUIS_STARTPANE)
	   {
		   /* 
		    *	if there is no KUIS_STARTGUIDE line, this implies that
		    *	there will be only a single pane on the form.
		    *	Create an "empty guidebutton" for the sake of 
		    *	bookkeeping 
		    */

		   if (subform->guidepane == NULL) 
		   {
			if ((subform->guide = (kguide *)
					kcalloc(1,sizeof(kguide))) == NULL)
			{
			    kvf_allocation_error_mesg();
			    return(FALSE);
			}
			guide = subform->guide;
			guide->button = NULL;
			guide->selected = TRUE;
			guide->next = NULL;
			guide->line = NULL;
	                guide->control_line = kstrdup(database[*kvf_index]);
			guide->back_subform = subform;
			guide->back_form = form;
		        guide->back_kformstruct 
				= kvf_create_struct((kaddr) guide,
                                                  KUIS_GUIDEBUTTON, KGUIDE);
                        kvf_link_formstruct(guide->back_kformstruct);
			guide->pane = kvf_fill_in_pane(form, subform,
                                        guide,  database, kvf_index,
                                        filename_lookup, linenum_lookup,
                                        filenames, &pane_names, &pane_num,
					&highest_index);
			if (guide->pane == NULL) return(FALSE);
	                guide->pane->control_comment = kstrdup(comment);
		   }
		   else
		   {
		       /* 
		        *  run thru guide list, creating guide branches
		        */
		        guide = subform->guidepane->guide_list;
			while (guide != NULL)
			{
                            /*
                             * a little trick - used selected field to store
                             * the index of the -P line temporarily
                             */
                             *kvf_index = guide->selected;

                            /*
                             * but now, need to backtrack to find comment
                             */
                             flag = kvf_get_line_type(database[*kvf_index-1]);
                             while (flag == KUIS_COMMENT)
                             {
                                 (*kvf_index)--;
                                 flag =
                                    kvf_get_line_type(database[(*kvf_index)-1]);
                             }
			     /* kfree(comment); */
                             comment = kvf_get_comments(database, kvf_index);

			     kvf_clear_line_info(&line_info);
			     kvf_parse_guide_line(guide->line, &line_info);
			     guide->selected = line_info.selected;
	                     kvf_free_line_info_strings(&line_info);
			     guide->pane = kvf_fill_in_pane(form, subform,
					   guide,  database, kvf_index,
                                           filename_lookup, linenum_lookup, 
					   filenames, &pane_names, &pane_num,
					   &highest_index);
			     if (guide->pane == NULL) return(FALSE);
			     guide->pane->control_comment = kstrdup(comment);
			     /* kfree(comment); */
			     comment = kvf_get_comments(database, kvf_index);
		             guide = guide->next;
		        }
			done = TRUE;
			*kvf_index = highest_index;
		   }
	   }  /* end if */


	   /*** flag (-E)  should be the end of the pane definiton ***/

	   else if (flag == KUIS_END)
	   {
/*
		if (guidebutton_num == 1)
                {
                    ksprintf(temp, "UIS must have more than one guide button to qualify for use of a Guidepane.  Eliminate unneeded master form [-S to -E] definition, or add additional [-g] guide button(s) and their associated pane [-P to -E] definitions.");
	            errno = KUIS_SYNTAX;
                    kerror("kforms", "kvf_create_subform", temp);
                    return(FALSE);
                }
                else */ if ((pane_num == 0) && (guidebutton_num > 0))
                {
                    ksprintf(temp, "There must be one Pane [-P to -E] definition in the UIS file for each (-g) GuideButton UIS line.");
	            errno = KUIS_SYNTAX;
                    kerror("kforms", "kvf_create_subform", temp);
                    return(FALSE);
                }
                else if ((guidebutton_num == 0) && (pane_num > 1))
                {
                    ksprintf(temp, "When there is more than one Pane [-P to -E] definition in the UIS file, there must be a GuidePane [-G to -E] definition within the subform, with one (-g) GuideButton UIS line for each pane definition.");
	            errno = KUIS_SYNTAX;
                    kerror("kforms", "kvf_create_subform", temp);
                    return(FALSE);
                }
                else if (guidebutton_num > pane_num) 
                {
                    ksprintf(temp, "There are %d (-g) GuideButton UIS lines, but only %d Pane [-P to -E] definition(s) in the UIS file;  there must be one pane definition for each guide button.", guidebutton_num, pane_num);
	            errno = KUIS_SYNTAX;
                    kerror("kforms", "kvf_create_subform", temp);
                    return(FALSE);
                }
                else if ((pane_num > guidebutton_num) && (pane_num > 1))
                {
                    ksprintf(temp, "There are %d Pane [-P to -E] definition(s) in the UIS file, but only %d (-g) GuideButton UIS lines; there must be one guide button for each pane definition.", pane_num, guidebutton_num);
	            errno = KUIS_SYNTAX;
                    kerror("kforms", "kvf_create_form", temp);
                    return(FALSE);
                }
		done = TRUE;
	   }


	   /* database line out of place */
	   else
	   {
	      ksprintf(temp, "UIS line out of place on line %d of %s.", linenum_lookup[*kvf_index], filenames[filename_lookup[*kvf_index]]);
	      errno = KUIS_SYNTAX;
	      kerror("kforms", "kvf_create_form",  temp);
	      return(FALSE);
	   }

	   kfree(comment);


	} /* end while */

	for (i = 0; i < pane_num; i++)
	    kfree(pane_names[i]);
	kfree(pane_names);
	kvf_free_line_info_strings(&line_info);

	(*kvf_index)++;

    	return(TRUE);

} /* end kvf_fill_in_subform() */
 
/*-----------------------------------------------------------
|
|  Routine Name: kvf_fill_in_pane
|
|       Purpose: Creates a pane and returns it.
|
|         Input: database    - pointer to UIS description
|                group_start - the pointer to the hdr of me group
|                kvf_index   - kvf_index to the current UIS line
|                form        - pointer to the form tree
|                subform     - pointer to this subform branch
|                guide       - pointer to this guide branch
|                filename_lookup - indexes into 'filenames' array
|                linenum_lookup  - holds actual line number of file for UIS line
|                filenames       - array of filenames comprising UIS
|                pane_names   - array of names from -P lines (for errorchecking)
|                pane_num   - size of pane_names array
|                highest_index - greatest database index achieved by this 
|                                routine (for restoration later)
|        Output: none
|       Returns: 0                on failure
|                NO_GROUP_DEFAULT if no default is set in group
|                GROUP_DEFAULT    if default is set for group
|
|          Date: Jun 10, 1992
|    Written By: Danielle Argiro
| Modifications:
|
------------------------------------------------------------*/
static kcontrol *kvf_fill_in_pane(
   kform    *form,
   ksubform *subform,
   kguide   *guide,
   char     **database,
   int      *kvf_index,
   int      *filename_lookup,
   int      *linenum_lookup,
   char     **filenames,
   char     ***pane_names,
   int      *pane_num,
   int      *highest_index)
{
	int         has_submenu = FALSE;
	int         i, save_index;
	char        temp[KLENGTH], pane_name[KLENGTH];
	Line_Info   line_info;
	kcontrol *pane;

	/* Allocate room for the guide structure */
	kvf_clear_line_info(&line_info);
	if ((pane = (kcontrol *) kcalloc(1, sizeof(kcontrol))) == NULL)
	{
	    kvf_allocation_error_mesg();
	    return(NULL);
	}
	pane->type         = KPANE;
	pane->back_form    = form;
	pane->back_subform = subform;
	pane->back_guide   = guide;
	pane->control_line = kstrdup(database[*kvf_index]);

	pane->back_kformstruct = kvf_create_struct((kaddr) pane,
                                           KUIS_STARTPANE, KPANE);
	kvf_link_formstruct(pane->back_kformstruct);

	/* parse the StartPane (-P) Line line */
	if (!(kvf_parse_startpane_line(database[*kvf_index], &line_info))) 
	{
	    kvf_parsing_error_mesg(*kvf_index, filename_lookup, 
				   linenum_lookup, filenames);
		return(NULL);
	}
	if ((line_info.selected) && (pane->back_subform->guidepane != NULL))
	{
	     errno = KUIS_LOGIC;
	     kerror("kforms", "kvf_parse_startpane_line",
		    "Please initialize 'selected' field on (-P) StartPane on line %d of %s to 0", linenum_lookup[*kvf_index], filenames[filename_lookup[*kvf_index]]); 
	     return(NULL);
	}

	/*
	 *  check to be sure that guide variable name is not 
	 *  the same as the one on the -M line
	 */
	if (kstrcmp(line_info.variable, 
		    ktoken_to_string(subform->var_token)) == 0)
	{
	    ksprintf(temp, "variable on line %d of %s cannot be the same as that on the -M line", linenum_lookup[*kvf_index], filenames[filename_lookup[*kvf_index]]);
	    errno = KUIS_LOGIC;
	    kerror("kforms", "kvf_create_form",  temp);
            return(NULL);
 	}

	for (i = 0; i < *pane_num; i++)
	{
	    if (kstrcmp(line_info.variable, (*pane_names)[i]) == 0)
	    {
		ksprintf(temp, "variable on line %d of %s cannot be the same as the variable on any other -P line", linenum_lookup[*kvf_index], filenames[filename_lookup[*kvf_index]]);
	        errno = KUIS_LOGIC;
	        kerror("kforms", "kvf_create_form",  temp);
                return(NULL);
	    }
	}

	if ((*pane_names = (char **) krealloc(*pane_names,
			                (*pane_num+1)*sizeof(char*))) == NULL)
	{
	     kerror("kforms", "kvf_fill_in_pane",
		     "Unable to re-allocate internal pane_names array");
	     return(NULL);
	}

	(*pane_names)[*pane_num] = kstrdup(line_info.variable);
	ksprintf(pane_name, "%s.%s.%s", 
		ktoken_to_string(form->var_token), 
		ktoken_to_string(subform->var_token), 
	        line_info.variable);
	pane->name_token  = kstring_to_token(pane_name);
	pane->var_token   = kstring_to_token(line_info.variable);

	pane->sel_list = NULL;

	if (subform->guidepane == NULL)
	    line_info.selected = TRUE;
	else line_info.selected = pane->back_guide->selected;

	kvf_deparse_startpane_line(&line_info, &pane->control_line);

	kvf_free_line_info_strings(&line_info);

	(*kvf_index)++; 
	(*pane_num)++;
	save_index = *kvf_index;
	if (!(kvf_fill_in_control(database, form, subform, guide, pane, 
				  kvf_index, &has_submenu, filename_lookup, 
				  linenum_lookup, filenames)))
	    return(NULL);

	if (*kvf_index > *highest_index) 
	    *highest_index = *kvf_index;

	if (has_submenu)
	{
	    *kvf_index = save_index;
	    if (!(kvf_link_submenu_group(database, pane, kvf_index)))
		 return(NULL);
 	}

	return(pane);
}





/*------------------------------------------------------------
|
|  Routine Name: kvf_fill_in_guide
|
|       Purpose: Creates the list of guides.
|
|         Input: database  - pointer to internal UIS
|		 form      - pointer to the form tree
|		 subform   - pointer to this subform branch
|                guidepane - pointer to this guidepane struct
|		 kvf_index - kvf_index into the database of current line
|                filename_lookup - indexes into 'filenames' array
|                linenum_lookup  - holds actual line number of file for UIS line
|                filenames       - array of filenames comprising UIS
|
|	 Output: guidebutton_num - passes back # of guide buttons in UIS 
|       Returns: TRUE on success, FALSE on failure
|
|          Date: Apr 10, 1992
|    Written By: Danielle Argiro and Stephanie Hallett
| Modifications: Converted from Khoros 1.0 (DA)
|		
------------------------------------------------------------*/


static int kvf_fill_in_guide(
   char     **database,
   kform    *form,
   ksubform *subform,
   kcontrol *guidepane,
   int      *kvf_index,
   int      *guidebutton_num,
   int      *filename_lookup,
   int      *linenum_lookup,
   char     **filenames)
{
	int        toggle, submenu, group, done, flag;
        int	   ok = TRUE, guide_def = 0;
	Line_Info  line_info;
	kguide     *guide, *last_guide = NULL;
	char       temp[2*KLENGTH];
	char       *comment = NULL, *tmpline = NULL;
	kselection *tmpsel, *save_submenu_selptr;

	done = toggle = submenu = group = FALSE;
	save_submenu_selptr = NULL;
	kvf_clear_line_info(&line_info);

	/* parse the KUIS_STARTGUIDE Line (-G) line */
	if (!(kvf_parse_startguide_line(database[*kvf_index], &line_info))) 
	{
	    kvf_parsing_error_mesg(*kvf_index, 
                              filename_lookup, linenum_lookup, filenames);
	    return(FALSE);
	}
	kvf_free_line_info_strings(&line_info);
	(*kvf_index)++;

	if (!(kvf_check_for_guide_buttons(database, *kvf_index)))
	{
	    ksprintf(temp, "Unnecessary GuidePane [-G to -E] definition in file %s;  GuidePane definitions may only be used with 2 or more guide buttons (-g lines).  Either (1) move any selections in the GuidePane definition into a Pane [-P to -E] definition, or (2) add at least two guide buttons to the GuidePane definition.", filenames[filename_lookup[*kvf_index]]);
	    errno = KUIS_LOGIC;
            kerror("kforms", "kvf_create_form", temp);
            return(FALSE);
	}

	while (!done) 
	{
	    /* 
	     *  really looking only for guide buttons
	     */
	     comment = kvf_get_comments(database, kvf_index);

	     flag = kvf_get_line_type(database[*kvf_index]);
	     if (flag == -1) return(FALSE);
		
	     switch (flag) {

		case KUIS_STARTSUBMENU:
		     submenu = TRUE;
		     tmpsel  = guidepane->sel_list;
		     while (tmpsel != NULL) 
		     {
			tmpline = kstrdup(database[*kvf_index]);
		        kvf_parse_startsubmenu_line(tmpline, &line_info);
		        kvf_deparse_startsubmenu_line(&line_info, &tmpline);
			if (kstrcmp(tmpsel->line, tmpline) == 0)
			   save_submenu_selptr = tmpsel;
			tmpsel = tmpsel->next;
		     }
		     if (save_submenu_selptr == NULL)
		     {
			errno = KINTERNAL;
			kerror("kforms", "kvf_fill_in_guide",
			       "can't find associated submenu, \
borting form creation");
			 return(FALSE);
		     }
		     (*kvf_index)++;
		     break;

		case KUIS_GUIDEBUTTON:
		    /* 
		     * A Guide Button allows the user to control which 
		     * pane is placed within the form
		     */
	             if ((guide = (kguide *) kcalloc(1,sizeof(kguide))) == NULL)
	             {
		         kvf_allocation_error_mesg();
		         return(FALSE);
	             }
		     kvf_clear_line_info(&line_info);

		     /* 
		      * fill in initial guide struct values 
                      */
		     if (!(kvf_set_guide_values(guidepane, guide, 
				 database[*kvf_index], form, 
				 subform, save_submenu_selptr, *kvf_index,
                                 filename_lookup, linenum_lookup, filenames)))
			return(FALSE);
		     guide->comment = kstrdup(comment);

		     /*
                      *  find matching -P line
                      */
                     guide->control_line = kvf_find_matching_P_line(database,
                                             *kvf_index, filename_lookup,
                                             linenum_lookup, filenames,
                                             &guide->selected);
                     if (guide->control_line == NULL) return(FALSE);

		     /*
		      *  parse -g line 
		      */
		     if (!(kvf_parse_guide_line(database[*kvf_index],
		           &line_info))) 
		     {
			kvf_parsing_error_mesg(*kvf_index, 
                              filename_lookup, linenum_lookup, filenames);
			return(FALSE);
		     }
		     guide->var_token = kstring_to_token(line_info.variable);

		     /*
		      *  make sure only one -g line is selected 
		      */
		     if (line_info.selected == TRUE)
		     {
			if (guide_def == TRUE)
			{
			    ksprintf(temp, "On line %d of %s, the 'selected' (2nd) field may not be set to TRUE; only one guide button may be selected as the default on a subform.", linenum_lookup[*kvf_index], filenames[filename_lookup[*kvf_index]]);	
			    errno = KUIS_SYNTAX;
	                    kerror("kforms", "kvf_create_form", 
                                        temp);
			    return(FALSE);
			}
			guide_def = TRUE;
		     }

		     /*
		      * guide button might be on pulldown menu
		      */
		     if (submenu == TRUE)
		         guide->back_submenu = save_submenu_selptr;

		     kvf_link_guide(subform->guidepane, guide, &last_guide);
		     kvf_free_line_info_strings(&line_info);
		     (*guidebutton_num)++; 
		     (*kvf_index)++;
		     break;

		case KUIS_END:
		     /* 
		      * at the end of the description of the guiding pane
		      */
		     if (submenu == TRUE)
		     {
			submenu = FALSE;
			save_submenu_selptr = NULL;
		     }
		     else if (group > 0)
	                group--;
		     else if (toggle == TRUE)
			toggle = FALSE;
		     else
		        done = TRUE;
		     (*kvf_index)++;
		     break;

		case KUIS_SUBFORMBUTTON:
		case KUIS_STARTGUIDE:
		case KUIS_STARTMASTER:
		case KUIS_STARTFORM:
		case KUIS_STARTPANE:
		     ksprintf(temp, "UIS line out of place on line %d of %s", 
                             linenum_lookup[*kvf_index], 
                             filenames[filename_lookup[*kvf_index]]);
	             errno = KUIS_SYNTAX;
		     kerror("kforms", "kvf_create_form", temp);
                     ok = FALSE;
		     break;

		case KUIS_TOGGLE:
		     toggle = TRUE;
		     (*kvf_index)++;
		     break;

		case KUIS_MUTEXCL:
		case KUIS_MUTINCL:
		case KUIS_GROUP:
		     group++;
	             (*kvf_index)++;
                     break;

                default:
		     (*kvf_index)++;
		     break;

	    }  /* end switch  */

	    if (!ok) return(FALSE); 
	    kfree(comment);

	} /* end while */

/*
	if (*guidebutton_num < 2)
	{
	    ksprintf(temp, "UIS must have more than one guide button to qualify for use of a Guide Pane.  Eliminate unneeded guide pane [-G to -E] definition, or add additional [-g] guide button(s) and their associated pane [-P to -E] definitions."); 
	    errno = KUIS_SYNTAX;
            kerror("kforms", "kvf_create_form",  temp);
            return(FALSE);
	}

*/
	return(TRUE);

}  /* end kvf_fill_in_guide() */
	    

	
 
/*------------------------------------------------------------
|
|  Routine Name: kvf_fill_in_control
|
|       Purpose: fills in the master, guidepane, or pane with selections
|
|         Input: database    - ptr to UIS description
| 		 form        - ptr to the form tree
|		 subform     - ptr to this subform branch (if applicable)
|		 guide       - ptr to this guide branch (if applicable)
|		 control     - ptr to this master, guidepane, or pane
|		 kvf_index   - index into the UIS of current line
|                has_submenu - TRUE if there is a submenu on this control
|                filename_lookup - indexes into 'filenames' array
|                linenum_lookup  - holds actual line number of file for UIS line
|                filenames       - array of filenames comprising UIS
|
|        Output: none
|       Returns: Returns TRUE on success, FALSE on failure
|          Date: Apr 10, 1992
|    Written By: Danielle Argiro 
| Modifications:
|		
------------------------------------------------------------*/

static int kvf_fill_in_control(
   char     *database[],
   kform    *form,
   ksubform *subform,
   kguide   *guide,
   kcontrol *control,
   int      *kvf_index,
   int      *has_submenu,
   int      *filename_lookup,
   int      *linenum_lookup,
   char     **filenames)
{
	int       i, done, flag, ok = TRUE; 
	int       submenu_item = FALSE;
        int       routine_num = 0; 
        Line_Info line_info;
        char      temp[KLENGTH];
        char      *comment = NULL;
	int       group_nestcount = 0;
        kselection *selection, *last, *save_submenu_selptr;

	done         = FALSE;
	submenu_item = FALSE;
	*has_submenu = FALSE;
	save_submenu_selptr = NULL;

	while (!done)
	{

	    if (control->sel_names == NULL)
	    {
               if ((control->sel_names = (char **) 
			kcalloc(1,sizeof(char *))) == NULL)
	       {
                    kerror("kforms", "kvf_fill_in_control",
	    	           "Unable to allocate internal sel_names array");
	            return(FALSE);
	        }
	    }

            kvf_clear_line_info(&line_info);

	    /* save comments before selection UIS line */
	    comment = kvf_get_comments(database, kvf_index);
	    flag = kvf_get_line_type(database[*kvf_index]);
	    if (flag == -1) return(FALSE);

	    /* allocate selection structure */
	    if ((selection = (kselection *) 
				kcalloc(1,sizeof(kselection))) == NULL)
	    {
	        kvf_allocation_error_mesg();
	        return(FALSE);
	    }
    
	    /* parse & error check UIS line */
    	    kvf_clear_line_info(&line_info);
	    if (!(kvf_gen_parse(database[*kvf_index], &line_info)))
	    {
	        kvf_parsing_error_mesg(*kvf_index, filename_lookup, 
			               linenum_lookup, filenames);
	        kvf_free_line_info_strings(&line_info);
	        kfree(selection);
	        return(FALSE);
	    }
	
	    /* make sure variable name != any other sel's variable on control */
	    for (i = 0; i < control->sel_count; i++)
            {
                if (kstrcmp(line_info.variable, control->sel_names[i]) == 0)
                {
                    ksprintf(temp, "Variable on line %d of %s cannot be the same as the variable on any other selection's UIS line", linenum_lookup[*kvf_index], filenames[filename_lookup[*kvf_index]]); 
	    	    errno = KUIS_LOGIC;
	            kerror("kforms", "kvf_create_form",  temp);
		    kvf_free_line_info_strings(&line_info);
		    kfree(selection);
	    	    return(FALSE);
                }
            }

	    /* add variable name to list of selection variable names */
            if (line_info.variable != NULL)
            {
                control->sel_names = (char **) krealloc(control->sel_names,
                                (control->sel_count+1)*sizeof(char*));
	        if (control->sel_names == NULL)
                {
                    kerror("kforms", "kvf_fill_in_control",
                           "Unable to re-allocate internal sel_names array");
                	return(FALSE);
                }

                control->sel_names[(control->sel_count)++]
		   	= kstrdup(line_info.variable);

            }

	
	    switch (flag) 
	    {

		     /*
		      *  (-D)  indicates a submenu button on the control panel 
		      */
		case KUIS_STARTSUBMENU:

		     /* set the selection values */
		     if (!(kvf_set_selection_values(selection, 
                           database[*kvf_index], flag, form, 
                           subform, guide, control, NULL,
			   line_info.variable, comment))) 
		     {
			kvf_parsing_error_mesg(*kvf_index, filename_lookup,
                                               linenum_lookup, filenames);
			return(FALSE);
		     }

                     kvf_link_sel(control, selection, &last);
		     save_submenu_selptr = selection;
		     submenu_item = TRUE;
		     *has_submenu = TRUE;
		     (*kvf_index)++;
		     break;

		     /*
 		      *  (-d, -u, -g) : ignore any control buttons 
		      */
		case KUIS_SUBFORMBUTTON:
		case KUIS_GUIDEBUTTON:
		     (*kvf_index)++;
		     kfree(selection);
		     break;

		     /* 
		      *  basic selections 
		      */
		case KUIS_STDIN:
		case KUIS_STDOUT:
		     if (kvf_gui_item_exists(control->sel_list, flag))
                     {
		         ksprintf(temp, "Error on line %d of %s - only one GUI answer input file or GUI answer output file allowed on a pane.", linenum_lookup[*kvf_index], filenames[filename_lookup[*kvf_index]]);
	    	        errno = KUIS_SYNTAX;
	                kerror("kforms", "kvf_create_form",  temp);
                        return(FALSE);
		     }
		     /* illegal within submenu */
	             if (kvf_error_on_submenu(submenu_item, *kvf_index, 
			   filename_lookup, linenum_lookup, filenames))
                         return(FALSE);

		     /* set the selection values */
		     if (!(kvf_set_selection_values(selection, 
			   database[*kvf_index], flag, form,
			   subform, guide, control, save_submenu_selptr, 
                           line_info.variable, comment))) 
		     {
			kvf_parsing_error_mesg(*kvf_index, filename_lookup,
                                               linenum_lookup, filenames);
			return(FALSE);
		     }

                     kvf_link_sel(control, selection, &last);
		     (*kvf_index)++;
		     break;

		case KUIS_INPUTFILE:
		case KUIS_OUTPUTFILE:
	 	case KUIS_INTEGER:
		case KUIS_FLOAT:
		case KUIS_DOUBLE:
		case KUIS_STRING:
		case KUIS_STRINGLIST:
		case KUIS_LOGICAL:
		case KUIS_FLAG:
		case KUIS_CYCLE:
		case KUIS_LIST:
		case KUIS_DISPLAYLIST:
		     /* illegal within submenu */
	             if (kvf_error_on_submenu(submenu_item, *kvf_index, 
			   filename_lookup, linenum_lookup, filenames))
                         return(FALSE);

		     /* set the selection values */
		     if (!(kvf_set_selection_values(selection, 
			   database[*kvf_index], flag, form,
			   subform, guide, control, save_submenu_selptr, 
                           line_info.variable, comment))) 
		     {
			kvf_parsing_error_mesg(*kvf_index, filename_lookup,
                                               linenum_lookup, filenames);
			return(FALSE);
		     }

                     kvf_link_sel(control, selection, &last);
		     (*kvf_index)++;
		     break;

		     /* 
		      *  (-w) workspace objects
		      */
		case KUIS_WORKSPACE:
		     /* illegal within submenu */
	             if (kvf_error_on_submenu(submenu_item, *kvf_index, 
			   filename_lookup, linenum_lookup, filenames))
			 return(FALSE);

		     /* set the selection values */
		     if (!(kvf_set_selection_values(selection, 
                           database[*kvf_index], flag, form, 
                           subform, guide, control, NULL, 
                           line_info.variable, comment))) 
		     {
			kvf_parsing_error_mesg(*kvf_index, filename_lookup,
                                               linenum_lookup, filenames);
			return(FALSE);
		     }
                     kvf_link_sel(control, selection, &last);
		     (*kvf_index)++;
		     break;

		     /* 
		      *  (-n, -m, -a) Action buttons
		      */
		case KUIS_MASTERACTION:
		case KUIS_SUBFORMACTION:
		case KUIS_PANEACTION:
		case KUIS_BLANK:

		     /* set the selection values */
		     if (!(kvf_set_selection_values(selection, 
                           database[*kvf_index], flag, form, 
                           subform, guide, control, save_submenu_selptr, 
                           line_info.variable, comment))) 
		     {
			kvf_parsing_error_mesg(*kvf_index, filename_lookup,
                                               linenum_lookup, filenames);
			return(FALSE);
		     }
                     kvf_link_sel(control, selection, &last);
		     if (submenu_item) 
			   selection->back_submenu = save_submenu_selptr;
		     (*kvf_index)++;
		     break;

		     /* 
		      *  (-R) Routine buttons
		      */
		case KUIS_ROUTINE:

		     if (kvf_gui_item_exists(control->sel_list, KUIS_ROUTINE))
                     {
		         ksprintf(temp, "Error on line %d of %s - only one Routine button allowed on a pane.", linenum_lookup[*kvf_index], filenames[filename_lookup[*kvf_index]]);
	    	        errno = KUIS_SYNTAX;
	                kerror("kforms", "kvf_create_form",  temp);
                        return(FALSE);
		     }

		     /* set the selection values, generating variable name */
		     if (!(kvf_set_selection_values(selection, 
                           database[*kvf_index], flag, form, 
                           subform, guide, control, save_submenu_selptr, 
                           line_info.variable, comment))) 
		     {
			kvf_parsing_error_mesg(*kvf_index, filename_lookup,
                                               linenum_lookup, filenames);
			return(FALSE);
		     }
		     routine_num++;
                     kvf_link_sel(control, selection, &last);
		     if (submenu_item) 
			   selection->back_submenu = save_submenu_selptr;
		     (*kvf_index)++;
		     break;

		     /* 
		      *  (-H) Help buttons
		      */
		case KUIS_HELP:

		     /* set the selection values, generating variable name */
		     if (!(kvf_set_selection_values(selection, 
			   database[*kvf_index], flag, form, 
			   subform, guide, control,  save_submenu_selptr, 
			   line_info.variable, comment))) 
		     {
			kvf_parsing_error_mesg(*kvf_index, filename_lookup,
                                               linenum_lookup, filenames);
			return(FALSE);
		     }
                     kvf_link_sel(control, selection, &last);
		     if (submenu_item) 
			   selection->back_submenu = save_submenu_selptr;
		     (*kvf_index)++;
		     break;

		     /* 
		      *  (-T) KUIS_TOGGLE groups 
		      */
		case KUIS_TOGGLE:
		     /* illegal on submenu */
	             if (kvf_error_on_submenu(submenu_item, *kvf_index, 
			   filename_lookup, linenum_lookup, filenames))
			 return(FALSE);

		     if (!(kvf_set_selection_values(selection, 
			   database[*kvf_index], flag, form, 
			   subform, guide, control, NULL, 
			   line_info.variable, comment))) 
		     {
			kvf_parsing_error_mesg(*kvf_index, filename_lookup,
                                               linenum_lookup, filenames);
			return(FALSE);
		     }
		     (*kvf_index)++;

		     /* fill in rest of toggle options */
		     if (!(kvf_fill_in_toggle(database, selection, 
				      kvf_index, form, subform, guide, 
				      control, filename_lookup, linenum_lookup, 
			              filenames)))
			  return(FALSE);

                     kvf_link_sel(control, selection, &last);
		     break;

		case KUIS_MUTEXCL:
		case KUIS_MUTINCL:
		case KUIS_GROUP:
		     /* illegal on submenu */
	             if (kvf_error_on_submenu(submenu_item, *kvf_index, 
			   filename_lookup, linenum_lookup, filenames))
			 return(FALSE);

		     /* set the selection values */
		     if (!(kvf_set_selection_values(selection, 
			   database[*kvf_index], flag, form, 
			   subform, guide, control, NULL, 
			   line_info.variable, comment))) 
		     {
			kvf_parsing_error_mesg(*kvf_index, filename_lookup,
                                               linenum_lookup, filenames);
			return(FALSE);
		     }
		     (*kvf_index)++;

		     if ((flag == KUIS_MUTINCL) && (line_info.logical_val))
		     {
			ksprintf(temp, "Error on line %d of %s - Mutually Inclusive groups must be optional; otherwise, the user will always be forced to specify all the parameters. If this was the intent, simply make all the selections required; no mutually inclusive group is needed. Alternatively, give the user the option of NOT specifying the values in the group, by making the Mutually Inclusive Group optional (ie, by setting the 'required' field on the -B line to 0).", linenum_lookup[*kvf_index], filenames[filename_lookup[*kvf_index]]);
	    	        errno = KUIS_LOGIC;
			kerror("kforms", "kvf_fill_in_control", temp);
			return(FALSE);
		     }
		     selection->selected = line_info.logical_val;

		     /* fill in rest of mutually exclusive group members */
		     if (!(kvf_fill_in_group(database, selection,  
				     kvf_index, form, subform, guide, control, 
                                     filename_lookup, linenum_lookup, filenames,
			             &group_nestcount)))
			  return(FALSE);

		     /* selected field used for saying if its required */
		     selection->selected = line_info.logical_val;

                     kvf_link_sel(control, selection, &last);
                     break;


		     /* 
		      *  (-Q) Quit buttons
		      */
		case KUIS_QUIT:

		     /* set the selection values, generating variable name */
		     if (!(kvf_set_selection_values(selection, 
			   database[*kvf_index], flag, form, 
			   subform, guide, control, save_submenu_selptr, 
			   line_info.variable, comment))) 
		     {
			kvf_parsing_error_mesg(*kvf_index, filename_lookup,
                                               linenum_lookup, filenames);
			return(FALSE);
		     }
		     if (kvf_gui_item_exists(control->sel_list, KUIS_QUIT))
                     {
		         ksprintf(temp, "Error on line %d of %s - only one Quit button allowed on a pane.", linenum_lookup[*kvf_index], filenames[filename_lookup[*kvf_index]]);
	    	        errno = KUIS_SYNTAX;
	                kerror("kforms", "kvf_create_form",  temp);
                        return(FALSE);
		     }

                     kvf_link_sel(control, selection, &last);
		     if (submenu_item) 
			   selection->back_submenu = save_submenu_selptr;
		     (*kvf_index)++;
		     break;

		case KUIS_END:
		     if (submenu_item)
		     {
			submenu_item = FALSE;
			save_submenu_selptr = NULL;
		     }
                     else done = TRUE;
		     kfree(selection);
                     (*kvf_index)++;
                     break;

                default:
	    	     ksprintf(temp, "UIS line out of place on line %d of %s.", 
			     linenum_lookup[*kvf_index], 
			     filenames[filename_lookup[*kvf_index]]);
	    	     errno = KUIS_SYNTAX;
	             kerror("kforms", "kvf_create_form",  temp);
		     kfree(selection);
		     ok = FALSE;
		     break;

 	    }  /* end switch */

	    if (!ok) return(FALSE);
	    kvf_free_line_info_strings(&line_info);
	    kfree(comment);
	}

        return(TRUE);

} /* end kvf_fill_in_control */

 
/*------------------------------------------------------------
|
|  Routine Name: kvf_fill_in_toggle
|
|       Purpose: Creates the list of toggle members.
|	         Note that toggle_start is the kselection containing 
|                the [-T] line, NOT the first toggle item.  toggle_start->next 
|	         is the next item following the entire toggle list specified 
|	         by [-T to -E], while toggle_start->toggle_next starts the 
|	         list of selections in the toggle.  
|
|         Input: database     - pointer to UIS description
|		 toggle_start - the pointer to the list of toggle items
|		 kvf_index    - kvf_index into the database of current line
|		 form 	      - pointer to the form tree
|		 subform      - pointer to this subform branch
|		 guide        - pointer to this guide branch
|		 control      - pointer to this pane
|                filename_lookup - indexes into 'filenames' array
|                linenum_lookup  - holds actual line number of file for UIS line
|                filenames       - array of filenames comprising UIS
|
|        Output: none
|       Returns: Returns TRUE on success, FALSE on failure
|          Date: Apr 10, 1992
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

static int kvf_fill_in_toggle(
   char       *database[],
   kselection *toggle_start,
   int        *kvf_index,
   kform      *form,
   ksubform   *subform,
   kguide     *guide,
   kcontrol   *control,
   int        *filename_lookup,
   int        *linenum_lookup,
   char       **filenames)
{
	int        count = 1, done = FALSE, flag = -1, ok = TRUE; 
	int        toggle_type = -1, T_index = *kvf_index -1;
	char       temp[KLENGTH];
	char       scale[KLENGTH];
	kselection *last, *toggle;
	Line_Info  line_info, tmp_line_info;
	char       *comment = NULL;

	last = toggle_start;
	kvf_clear_line_info(&line_info);
	kvf_clear_line_info(&tmp_line_info);
	kvf_parse_toggle_line(toggle_start->line, &line_info);

	while (!done)
	{
	    comment = kvf_get_comments(database, kvf_index);
	    flag = kvf_get_line_type(database[*kvf_index]);
	    if (flag == -1) return(FALSE);
	    if (count == 1) toggle_type = flag;

	    if ((toggle = (kselection *) kcalloc(1,sizeof(char)*
			sizeof(kselection))) == NULL)
	    {
                kerror("kforms", "kvf_fill_in_toggle", 
		       "Unable to allocate internal toggle structure");
	        return(FALSE);
	    }
	    toggle->modified = FALSE;


	    if ((flag != toggle_type) && (flag != KUIS_END))
	    {
		 ksprintf(temp, "Error on line %d of %s;  Members of %s toggle group must be all of type %s; incorrect UIS line type of %s", linenum_lookup[*kvf_index], filenames[filename_lookup[*kvf_index]], kvf_ascii_typeflag(toggle_type), kvf_ascii_typeflag(toggle_type), kvf_ascii_typeflag(flag));
	    	 errno = KUIS_SYNTAX;
	         kerror("kforms", "kvf_create_form",  temp);
		 return(FALSE);
	    }

	    switch (flag) {

		case KUIS_INPUTFILE:
		case KUIS_OUTPUTFILE:
	 	case KUIS_INTEGER:
		case KUIS_FLOAT:
		case KUIS_DOUBLE:
		case KUIS_STRING:
		case KUIS_LOGICAL:
		case KUIS_FLAG:
		     if (!(kvf_gen_parse(database[*kvf_index], &tmp_line_info)))
		     {
	    	         kvf_parsing_error_mesg(*kvf_index, 
                              filename_lookup, linenum_lookup, filenames);
	                 return(FALSE);
	             }
		     if (tmp_line_info.optional != TRUE)
		     {
			 ksprintf(temp, "Error on line %d of %s;  Members of toggle group must all have the optional field (3rd field) set to 1 (TRUE)", linenum_lookup[*kvf_index], filenames[filename_lookup[*kvf_index]]);
	    	         errno = KUIS_SYNTAX;
	                 kerror("kforms", "kvf_create_form",  temp);
			 return(FALSE);
		     }
		     if (tmp_line_info.opt_sel == 2)
		     {
			 ksprintf(temp, "Error on line %d of %s;  Members of toggle group may not have their 'opt_sel' field (4th field) set to 2", linenum_lookup[*kvf_index], filenames[filename_lookup[*kvf_index]]);
	    	         errno = KUIS_SYNTAX;
	                 kerror("kforms", "kvf_create_form",  temp);
			 return(FALSE);
		     }
		     if (!(kvf_set_selection_values(toggle, 
			   database[*kvf_index], flag, form, 
			   subform, guide, control, NULL, 
			   tmp_line_info.variable, comment))) 
		     {
			kvf_parsing_error_mesg(*kvf_index, filename_lookup,
                                               linenum_lookup, filenames);
			return(FALSE);
		     }
		     toggle->toggle_num = count;
		     toggle->back_toggle = toggle_start;
		     if (toggle->toggle_num == line_info.toggle_def)
		     {
			toggle->opt_selected = TRUE;
			if ((toggle->type == KUIS_FLAG) ||
                            (toggle->type == KUIS_LOGICAL))
                        {
                            ksprintf(temp, "%d", count);
                        }
                        else if (toggle->type == KUIS_INTEGER)
                        {
                            ksprintf(temp, "%d", tmp_line_info.int_def);
                        }
                        else if (toggle->type == KUIS_FLOAT) 
                        {
			    if (line_info.precision == 0)
                                ksprintf(scale, "%%g");
			    else if (line_info.precision == -1)
                                ksprintf(scale, "%%f");
			    else ksprintf(scale, "%%.%df", line_info.precision);
                            ksprintf(temp, scale, tmp_line_info.float_def);
                        }
                        else if (toggle->type == KUIS_DOUBLE) 
                        {
			    if (line_info.precision == 0)
			        ksprintf(scale, "%%g");
			    else if (line_info.precision == -1)
                                ksprintf(scale, "%%f");
			    else ksprintf(scale, "%%.%df", line_info.precision);
	                    ksprintf(temp, scale, tmp_line_info.double_def);
                        }
                        else if ((toggle->type == KUIS_INPUTFILE) ||
                                 (toggle->type == KUIS_OUTPUTFILE))
                        {
                            ksprintf(temp, "%s", tmp_line_info.file_def);
                        }
                        else if (toggle->type == KUIS_STRING)
                        {
                            ksprintf(temp, "%s", tmp_line_info.string_def);
                        }
		        if (!tmp_line_info.opt_sel)
			{
			    kerror("kforms", "kvf_fill_in_toggle",
				   "Error on line %d of %s; the opt_sel field of toggle member %d must be set to 1, in accordance with toggle default %d as specified on the [-T] line", linenum_lookup[*kvf_index], filenames[filename_lookup[*kvf_index]], toggle->toggle_num, line_info.toggle_def);
			    return(FALSE);
			}
		     }
		     else if (tmp_line_info.opt_sel)
		     {
			    kerror("kforms", "kvf_fill_in_toggle",
				   "Error on line %d of %s; the opt_sel field of toggle member %d must be set to 0; in accordance with toggle default %d as specified on the [-T] line, only the opt_sel field of toggle member %d should be set to 1", linenum_lookup[*kvf_index], filenames[filename_lookup[*kvf_index]], toggle->toggle_num, line_info.toggle_def, line_info.toggle_def);
			    return(FALSE);
	             }
		     else toggle->opt_selected = FALSE;
		     (*kvf_index)++;
	             kvf_free_line_info_strings(&tmp_line_info);
		     break;

		case KUIS_BLANK:
		     if (toggle != NULL) kfree(toggle);
		     toggle = NULL;
		     (*kvf_index)++;
		     break;

		case KUIS_END:
	
		     if (toggle != NULL) kfree(toggle);
		     toggle = NULL;
		     done = TRUE;
		     (*kvf_index)++;
		     break;

                default:
		     ksprintf(temp, "Error on line %d of %s;  Illegal UIS line type within toggle definition.", linenum_lookup[*kvf_index], filenames[filename_lookup[*kvf_index]]);
	    	     errno = KUIS_SYNTAX;
	             kerror("kforms", "kvf_create_form",  temp);
		     ok = FALSE;
		     break;

	    }  /* end switch */

	    if (!ok) return(FALSE);

	    if (count == 1)
	    {
	       	toggle_start->toggle_next = toggle;
	    }
  	    else
	    {
		last->next = toggle;
 	    }
	    last = toggle;
	    count++;
	    kfree(comment);
	}
	if (line_info.toggle_def > count-2)
	{
	    ksprintf(temp, "Error on line %d of %s;  Toggle default specified exceeds number of members in the toggle.", linenum_lookup[T_index], filenames[filename_lookup[T_index]]);
	    errno = KUIS_LOGIC;
	    kerror("kforms", "kvf_create_form",  temp);
	    return(FALSE);
	}
	kvf_deparse_toggle_line(&line_info, &toggle_start->line);
	kvf_free_line_info_strings(&line_info);


	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: kvf_fill_in_group
|
|       Purpose: Creates the list of mutually exclusive (-C), 
|                mutually inclusive (-B), or loose group (-K) members.
|	         group_start is the kselection containing the (-C, -B, or -K) 
|	         line, NOT the first group item.  group_start->next 
|	         is the next item following the entire ME group specified 
|	         by [-C to -E], while group_start->group_next starts the 
|	         list of selections in the toggle.  
|
|         Input: database    - pointer to UIS description
|		 group_start - the pointer to the hdr of me group
|		 kvf_index   - kvf_index to the current UIS line
|		 form 	      - pointer to the form tree
|		 subform     - pointer to this subform branch
|		 guide       - pointer to this guide branch
|		 control     - pointer to this control
|                filename_lookup - indexes into 'filenames' array
|                linenum_lookup  - holds actual line number of file for UIS line
|                filenames       - array of filenames comprising UIS
|
|        Output: none
|       Returns: 0                on failure
|                NO_GROUP_DEFAULT if no default is set in group
|                GROUP_DEFAULT    if default is set for group
|
|          Date: Jun 10, 1992
|    Written By: Danielle Argiro 
| Modifications:
|		
------------------------------------------------------------*/
#define NO_GROUP_DEFAULT_SET 1
#define GROUP_DEFAULT_SET    2

static int kvf_fill_in_group(
   char       *database[],
   kselection *group_start,
   int        *kvf_index,
   kform      *form,
   ksubform   *subform,
   kguide     *guide,
   kcontrol   *control,
   int        *filename_lookup,
   int        *linenum_lookup,
   char       **filenames,
   int        *group_nestcount)
{
	int        count = 1, first = TRUE, done = FALSE, ok = TRUE;
	kselection *last = NULL, *group; 
	int        flag = -1; 
	char       temp[KLENGTH];
	Line_Info  line_info;
	char       *comment = NULL;
	int        group_status;
	int        mi_default_set = -1;
	int        me_default_set = FALSE;
	int        gp_default_set = FALSE;
	int        group_required = group_start->selected;
	int        save_group_index, save_subgroup_index;

	kvf_clear_line_info(&line_info);

	save_group_index = *kvf_index - 1;
	(*group_nestcount)++;

	while (!done)
	{
	    comment = kvf_get_comments(database, kvf_index);
	    flag = kvf_get_line_type(database[*kvf_index]);
	    if (flag == -1) return(-1);

	    if ((group = (kselection *) kcalloc(1,sizeof(char) *
			sizeof(kselection))) == NULL)
	    {
                kerror("kforms", "kvf_fill_in_group", 
		       "Unable to allocate internal group structure");
	        return(FALSE);
	    }
	    group->modified = FALSE;

	    switch (flag) {

		     /* 
		      *  allow mutually exclusive groups within
                      *  mutually exclusive groups via recursion
	              */
		case KUIS_MUTEXCL:
		case KUIS_MUTINCL:
		case KUIS_GROUP:
		     if (!(kvf_gen_parse(database[*kvf_index], &line_info)))
		     {
	    	         kvf_parsing_error_mesg(*kvf_index, 
                              filename_lookup, linenum_lookup, filenames);
	                 return(FALSE);
	             }

		     /* only 2 levels of nesting supported */
		     if (*group_nestcount > 1)
		     {
			 ksprintf(temp, "Error on line %d of %s;  Mutually Exclusive/Inclusive groups may only contain one level of nesting due to internal limitations...", linenum_lookup[*kvf_index], filenames[filename_lookup[*kvf_index]]);
			 errno = KUIS_SYNTAX;
	                 kerror("kforms", "kvf_create_form", temp);
			 return(FALSE);
		     }

		     /*  No ME groups within ME groups */
		     if ((group_start->type == KUIS_MUTEXCL) &&
			 (flag == KUIS_MUTEXCL))
		     {
                         ksprintf(temp, "Error on line %d of %s;  There is no need to nest a mutually exclusive group within another mutually exclusive group;  simply including all the group members in a single mutually exclusive group will achieve the same effect.", linenum_lookup[*kvf_index], filenames[filename_lookup[*kvf_index]]);
			 errno = KUIS_LOGIC;
                         kerror("kforms", "kvf_create_form",  temp);
                         return(FALSE);
		     }

		     /*  No MI groups within MI groups */
		     if ((group_start->type == KUIS_MUTINCL) &&
                         (flag == KUIS_MUTINCL))
                     {
                         ksprintf(temp, "Error on line %d of %s;  There is no need to nest a mutually inclusive group within another mutually inclusive group;  simply including all the group members in a single mutually inclusive group will achieve the same effect.", linenum_lookup[*kvf_index], filenames[filename_lookup [*kvf_index]]);
			 errno = KUIS_LOGIC;
                         kerror("kforms", "kvf_create_form",  temp);
                         return(FALSE);
                     }

		     /*  No loose groups within loose groups */
		     if ((group_start->type == KUIS_GROUP) &&
                         (flag == KUIS_GROUP))
                     {
                         ksprintf(temp, "Error on line %d of %s;  There is no need to nest a loose group within another loose group;  simply including all the group members in a single loose group will achieve the same effect.", linenum_lookup[*kvf_index], filenames[filename_lookup [*kvf_index]]);
			 errno = KUIS_LOGIC;
                         kerror("kforms", "kvf_create_form",  temp);
                         return(FALSE);
                     }

		     /* nested ME groups must have 'required' field set to 0 */
		     if ((line_info.logical_val == 1) &&
			 (flag == KUIS_MUTEXCL))
		     {
			ksprintf(temp, "Error on line %d of %s; nested mutually exclusive groups may not be required (ie, the 'required' field on the (-C) line must be set to 0.", linenum_lookup[*kvf_index], filenames[filename_lookup [*kvf_index]]);
			 errno = KUIS_LOGIC;
                         kerror("kforms", "kvf_create_form",  temp);
                         return(FALSE);
		     }

		     /* nested MI groups must have 'required' field set to 0 */
		     if ((line_info.logical_val == 1) &&
			 (flag == KUIS_MUTINCL))
		     {
			ksprintf(temp, "Error on line %d of %s; nested mutually inclusive groups may not be required (ie, the 'required' field on the (-B) line mut be set to 0.", linenum_lookup[*kvf_index-1], filenames[filename_lookup [*kvf_index]]);
			 errno = KUIS_LOGIC;
                         kerror("kforms", "kvf_create_form",  temp);
                         return(FALSE);
		     }

		     /* set the selection values */
		     if (!(kvf_set_selection_values(group, database[*kvf_index],
			   		flag, form, subform, guide, control, 
					NULL, line_info.variable, comment))) 
		     {
			kvf_parsing_error_mesg(*kvf_index, filename_lookup,
                                               linenum_lookup, filenames);
			return(FALSE);
		     }

		     group->back_group = group_start;

		     /* selected field used for saying if its required */
		     group->selected = group_required = line_info.logical_val;

		     /* process subgroup */
		     save_subgroup_index = *kvf_index;
                     (*kvf_index)++;
		     group_status = kvf_fill_in_group(database, group, 
				     kvf_index, form, subform, guide, control, 
				     filename_lookup, linenum_lookup, filenames,
				     group_nestcount);
		     if (group_status == FALSE)
			  return(FALSE);
		     else if (group_status == GROUP_DEFAULT_SET)
		     {
		         if (group_start->type == KUIS_MUTEXCL) 
			 {
			    if (me_default_set)
			    {
				ksprintf(temp, "Error in Mutually Inclusive/Exclusive group beginning on line %d of %s; members of nested group must have their 'opt_sel' (4th) fields set to 0 since default for outer group has already been set.", linenum_lookup[save_subgroup_index], filenames[filename_lookup[*kvf_index]]); 
			        errno = KUIS_LOGIC;
				kerror("kforms", "kvf_create_form", temp);
				return(FALSE);
			    }
			    else me_default_set = TRUE;
			 }
			 else
			 {
			    if (mi_default_set == 0)
			    {
				ksprintf(temp, "Error in Mutually Inclusive/Exclusive group beginning on line %d of %s; members of nested group must have their 'opt_sel' (4th) fields set to 0 because outer Mutually Inclusive group has 'opt_sel' fields set to 0.", linenum_lookup[save_subgroup_index], filenames[filename_lookup[*kvf_index]]); 
			        errno = KUIS_LOGIC;
				kerror("kforms", "kvf_create_form", temp);
				return(FALSE);
			    }
			    else mi_default_set = 1;
			 }
		     }
		     else 
		     {
			if ((group_start->type == KUIS_MUTINCL) && 
			    (mi_default_set == 1))
			{
			    if (group->type == KUIS_MUTINCL)
			    {
			        ksprintf(temp, "Error in Mutually Inclusive group beginning on line %d of %s; members of nested group must have their 'opt_sel' (4th) fields set to 1 because outer Mutually Inclusive group has 'opt_sel' fields set to 1.", linenum_lookup[save_subgroup_index], filenames[filename_lookup[*kvf_index]]); 
				errno = KUIS_LOGIC;
			        kerror("kforms", "kvf_create_form", temp);
				return(FALSE);
			    }
			    else if (group->type == KUIS_MUTEXCL)
			    {
                                ksprintf(temp, "Error in Mutually Exclusive group beginning on line %d of %s; one member of nested group must have its 'opt_sel' (4th) field set to 1 indicating default, because outer Mutually Inclusive group has 'opt_sel' fields set to 1.", linenum_lookup[save_subgroup_index], filenames[filename_lookup[*kvf_index]]); 
			        errno = KUIS_LOGIC;
                                kerror("kforms", "kvf_create_form", temp);
				return(FALSE);
                            }
			    else 
			    {
                                ksprintf(temp, "Error in Loose group beginning on line %d of %s; one member of nested group must have its 'opt_sel' (4th) field set to 1 indicating default, because outer Mutually Inclusive group has 'opt_sel' fields set to 1.", linenum_lookup[save_subgroup_index], filenames[filename_lookup[*kvf_index]]); 
			        errno = KUIS_LOGIC;
                                kerror("kforms", "kvf_create_form", temp);
				return(FALSE);
                            }

			}
		     }
                     break;

		case KUIS_INPUTFILE:
		case KUIS_OUTPUTFILE:
		case KUIS_STDIN:
		case KUIS_STDOUT:
	 	case KUIS_INTEGER:
		case KUIS_FLOAT:
		case KUIS_DOUBLE:
		case KUIS_STRING:
		case KUIS_STRINGLIST:
		case KUIS_LOGICAL:
		case KUIS_FLAG:
		case KUIS_TOGGLE:
		case KUIS_CYCLE:
		case KUIS_LIST:
		case KUIS_DISPLAYLIST:
		case KUIS_BLANK:
		     if (!(kvf_gen_parse(database[*kvf_index], &line_info)))
		     {
	    	         kvf_parsing_error_mesg(*kvf_index, 
                              filename_lookup, linenum_lookup, filenames);
	                 return(FALSE);
	             }
		     if (line_info.optional != TRUE)
		     {
			 ksprintf(temp, "Error on line %d of %s;  Members of mutually exclusive, mutually inclusive, and loose groups must all have the optional field (3rd field) set to 1 (TRUE)", linenum_lookup[*kvf_index], filenames[filename_lookup[*kvf_index]]);
			 errno = KUIS_SYNTAX;
	                 kerror("kforms", "kvf_create_form", temp);
			 return(FALSE);
		     }
		     if (group_start->type == KUIS_MUTINCL)
		     {
			if (flag != KUIS_BLANK)
			{
			    if (first) 
			    {
			        mi_default_set = line_info.opt_sel;
		                first = FALSE;
			    }
			    if (!(kvf_check_mi_member_default(&mi_default_set, 
			          line_info.opt_sel, *kvf_index, 
			          filename_lookup, linenum_lookup, filenames)))
			   return(FALSE);
			}

		     }
		     else if (group_start->type == KUIS_MUTEXCL)
		     {
			if (flag != KUIS_BLANK)
                        {
			    if (!(kvf_check_me_member_default(&me_default_set,
                                  line_info.opt_sel, *kvf_index,
                                  filename_lookup, linenum_lookup, filenames)))
                               return(FALSE);
			}
	             }
		     if (!(kvf_set_selection_values(group, database[*kvf_index],
			   		flag, form, subform, guide, control, 
					NULL, line_info.variable, comment))) 
		     {
			kvf_parsing_error_mesg(*kvf_index, filename_lookup,
                                               linenum_lookup, filenames);
			return(FALSE);
		     }

		     if (flag == KUIS_TOGGLE)
		     {
		         (*kvf_index)++;
		         if (!(kvf_fill_in_toggle(database, group, 
					     kvf_index, form, subform, 
					     guide, control, filename_lookup, 
					     linenum_lookup, filenames)))
                              return(FALSE);
		         (*kvf_index)--;
		     }

		     if ((group_start->type == KUIS_MUTEXCL) &&
			 (line_info.opt_sel))
		         group->prev_selected = TRUE;
		     if ((group_start->type == KUIS_GROUP) &&
                         (line_info.opt_sel))
			gp_default_set = TRUE;
		     (*kvf_index)++;
		     group->back_group = group_start;
		     kvf_free_line_info_strings(&line_info);
		     break;

		case KUIS_END:
	
		     if (group != NULL) kfree(group);
		     group = NULL;
		     done = TRUE;
		     if ((group_required) && (group_start->type == KUIS_MUTEXCL))
		     {
			if (!(me_default_set))
			{
			    ksprintf(temp, "Error in ME group beginning on line %d;  No default set for group, please indicate default by setting the 'opt_sel' (4th) field of the desired UIS line to 1.", linenum_lookup[save_group_index]);
			    errno = KUIS_LOGIC;
			    kerror("kforms", "kvf_create_form", temp);
			    return(FALSE);
			}
		     }
		     (*kvf_index)++;
		     break;

                default:
		     ksprintf(temp, "Error on line %d of %s;  Illegal UIS line type within mutually exclusive group definition.", linenum_lookup[*kvf_index], filenames[filename_lookup[*kvf_index]]);
		     errno = KUIS_SYNTAX;
	             kerror("kforms", "kvf_create_form",  temp);
		     ok = FALSE;
		     break;

	    }  /* end switch */

	    if (!ok) return(FALSE);

	    if (count == 1)
	    {
	       	group_start->group_next = group;
	    }
  	    else
	    {
	        if (last != NULL)
		    last->next = group;
		else
		{
		    kerror("kforms", "kvf_fill_in_group",
		           "Error in linking group");
		}
 	    }
	    last = group;
	    count++;
	    kfree(comment);
	}
	(*group_nestcount)--;

	if ((me_default_set == 1) && (group_start->type == KUIS_MUTEXCL))
	    return(GROUP_DEFAULT_SET);
	else if ((mi_default_set == 1) && (group_start->type == KUIS_MUTINCL))
	    return(GROUP_DEFAULT_SET);
	else if ((gp_default_set == 1) && (group_start->type == KUIS_GROUP))
	    return(GROUP_DEFAULT_SET);
	else return(NO_GROUP_DEFAULT_SET);
}



/*-----------------------------------------------------------
|
|  Routine Name: kvf_link_submenu_group
|
|       Purpose: links members of submenu together in order
|
|         Input: database    - ptr to UIS description
|		 form        - ptr to the form tree
|		 subform     - ptr to this subform branch (if applicable)
|		 guide       - ptr to this guide branch (if applicable)
|		 control     - ptr to this master, guidepane, or pane
|		 kvf_index   - index into the UIS of current line
|                filename_lookup - indexes into 'filenames' array
|                linenum_lookup  - holds actual line number of file for UIS line
|                filenames       - array of filenames comprising UIS
|
|        Output: none
|       Returns: TRUE on success, FALSE on failure
|          Date: May 10, 1992
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/

static int kvf_link_submenu_group(
   char     *database[],
   kcontrol *control,
   int      *kvf_index)
{
	kguide       *guide = NULL;
	ksubform     *subform = NULL;
	kselection   *selection = NULL, *last_selection = NULL;
	kform_struct *member_ptr, *last_member = NULL;
	char         temp[KLENGTH];
	int          first, done, flag, submenu, toggle; 
	int          group, var_token;
	Line_Info    line_info;

	kvf_clear_line_info(&line_info);
	done = submenu = toggle = group = FALSE;  first = TRUE;
	if (control->sel_list != NULL)
	    selection = control->sel_list;

	if (control->subform_list != NULL)
	   subform = control->subform_list;

	if (control->guide_list != NULL)
	   guide = control->guide_list;

	while (!done)
	{
	    flag = kvf_get_line_type(database[*kvf_index]);
    
	    switch (flag) {

		case KUIS_STARTSUBMENU:

		     /*
		      *  (-D)  indicates a submenu button on the control panel 
		      */

		     submenu = TRUE;
		     kvf_parse_startsubmenu_line(database[*kvf_index], 
						 &line_info);
		     var_token = kstring_to_token(line_info.variable);
		     kvf_free_line_info_strings(&line_info);

		     last_selection = kvf_find_selection(selection, 
							 var_token);
		     if (last_selection == NULL)
		     {
		         ksprintf(temp, "can't find selection %s", 
				 database[*kvf_index]);
			 errno = KINTERNAL;
			 kerror("kforms", "kvf_link_submenu_group", temp);
			 return(FALSE);
		     }
		     first = TRUE;
		     break;

		case KUIS_SUBFORMBUTTON:
	             if (submenu)
		     {
		         kvf_gen_parse(database[*kvf_index], &line_info);
		         var_token = kstring_to_token(line_info.variable);
			 kvf_free_line_info_strings(&line_info);

		         subform = kvf_find_subform(subform, var_token);
		         if (subform == NULL)
		         {
		            ksprintf(temp, "can't find subform %s", 
				    database[*kvf_index]);
			    errno = KINTERNAL;
			    kerror("kforms", "kvf_link_submenu_group", temp);
			    return(FALSE);
		         }
	                 member_ptr = kvf_create_struct((kaddr) subform, 
                                                        flag, KSUBFORM);
		         if (first)
			 {
			     if (last_selection == NULL)
		             {
				errno = KINTERNAL;
				kerror("kforms", "kvf_link_submenu_group",
				        "Cannot find -D line for submenu member");
				return(FALSE);
			     }
		             last_selection->submenu_next = member_ptr;
			 }
		         else last_member->next = member_ptr;
    
		         first = FALSE;
		         last_member = member_ptr;
		     }
		     break;

		case KUIS_GUIDEBUTTON:
		     if (submenu)
		     {
		         kvf_parse_guide_line(database[*kvf_index], 
					      &line_info);
		         var_token = kstring_to_token(line_info.variable);
			 kvf_free_line_info_strings(&line_info);

		         guide = kvf_find_guide(guide, var_token);
		         if (guide == NULL)
		         {
		            ksprintf(temp, "can't find guide %s", 
					        database[*kvf_index]);
			    errno = KINTERNAL;
			    kerror("kforms", "kvf_link_submenu_group", 
			            temp);
			    return(FALSE);
		         }
	                 member_ptr = kvf_create_struct((kaddr) guide, 
							 flag, KGUIDE);
		         if (first)
			 {
			     if (last_selection == NULL)
		             {
				errno = KINTERNAL;
				kerror("kforms", "kvf_link_submenu_group",
				        "Cannot find -D line for submenu member");
				return(FALSE);
			     }
		             last_selection->submenu_next = member_ptr;
			 }
		         else last_member->next = member_ptr;
    
		         first = FALSE;
		         last_member = member_ptr;
		     }
		     break;

		case KUIS_MASTERACTION:
		case KUIS_SUBFORMACTION:
		case KUIS_PANEACTION:
		case KUIS_BLANK:
		case KUIS_ROUTINE:
		case KUIS_HELP:
		case KUIS_QUIT:
		     if (submenu)
		     {
		         kvf_gen_parse(database[*kvf_index], &line_info);
		         var_token = kstring_to_token(line_info.variable);
		         kvf_free_line_info_strings(&line_info);

		         selection = kvf_find_selection(selection, var_token);
		         if (selection == NULL)
		         {
			    errno = KINTERNAL;
			    kerror("kforms", "kvf_link_submenu_group", 
		            	   "can't find selection %s", 
					        database[*kvf_index]);
			    return(FALSE);
		         }
                         member_ptr = kvf_create_struct((kaddr) selection, 
							flag, KSELECTION);
		         if (first)
			 {
			     if (last_selection == NULL)
		             {
				errno = KINTERNAL;
				kerror("kforms", "kvf_link_submenu_group",
				      "Cannot find -D line for submenu member");
				return(FALSE);
			     }
		             last_selection->submenu_next = member_ptr;
			 }
		         else last_member->next = member_ptr;
    
		         first = FALSE;
		         last_member = member_ptr;
		     }
                     break;

		case KUIS_MUTEXCL:
		case KUIS_MUTINCL:
		case KUIS_GROUP:
		     group++;
		     break;

		case KUIS_TOGGLE:
		     toggle = TRUE;
		     break;

		case KUIS_END:
		     if (group > 0)
                        group--;
		     else if (toggle == TRUE)
			toggle = FALSE;
		     else if (submenu == TRUE)
		     {
		        if (last_member != NULL)
	                    last_member->next = NULL;
			submenu = FALSE;
		     }
                     else
		     {
                        done = TRUE;
		     }
                     break;

                default:
		     break;

	    }  /* end switch */

            (*kvf_index)++;
	}
	return(TRUE);
	
}
