/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>            General Utilities for Form Tree Creation   <<<<
   >>>>                                                       <<<<
   >>>>  Private:                                             <<<<
   >>>>               kvf_find_selection()                    <<<<
   >>>>               kvf_find_subform()                      <<<<
   >>>>               kvf_find_guide()                        <<<<
   >>>>               kvf_set_subform_values()                <<<<
   >>>>               kvf_set_guide_values()                  <<<<
   >>>>               kvf_set_selection_values()              <<<<
   >>>>               kvf_link_subform()                      <<<<
   >>>>               kvf_link_guide()                        <<<<
   >>>>               kvf_link_sel()                          <<<<
   >>>>               kvf_link_member()                       <<<<
   >>>>               kvf_get_comments()                      <<<<
   >>>>               kvf_error_on_submenu()                  <<<<
   >>>>               kvf_find_matching_P_line()              <<<<
   >>>>   Static:                                             <<<<
   >>>>   Public:                                             <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"


/*------------------------------------------------------------
|
|  Routine Name: kvf_find_selection
|
|       Purpose: Given a UIS line, finds matching selection in 
|                a selection list
|
|         Input: selection - header of selection list
|                var_token - variable name of selection we're looking for
|        Output: none
|
|       Returns: Returns selection on success, NULL on failure.
|          Date: Jun 05, 1992
|    Written By: Danielle Argiro
| Modifications:
|
------------------------------------------------------------*/

kselection *kvf_find_selection(
   kselection *selection,
   int        var_token)
{
	while (selection != NULL)
	{
	    if (selection->var_token == var_token) break;
	    selection = selection->next;
	}
	return(selection);
}

/*------------------------------------------------------------
|
|  Routine Name: kvf_find_subform
|
|       Purpose: Given a (-d or -u) UIS line, finds matching subform in 
|                a subform list
|
|         Input: subform   - header of subform list
|                var_token - variable name of subform we're looking for
|        Output: none
|
|       Returns: Returns subform on success, NULL on failure.
|          Date: Jun 05, 1992
|    Written By: Danielle Argiro
| Modifications:
|
------------------------------------------------------------*/

ksubform *kvf_find_subform(
   ksubform *subform,
   int      var_token)
{
        while (subform != NULL)
        {
            if (subform->var_token == var_token) break;
            subform = subform->next;
        }
        return(subform);
}

/*------------------------------------------------------------
|
|  Routine Name: kvf_find_guide
|
|       Purpose: Given a (-g) UIS line, finds matching guide in 
|                a guide list
|
|         Input: guide     - header of guide list
|                var_token - variable name of guide we're looking for
|        Output: none
|
|       Returns: Returns guide on success, NULL on failure.
|          Date: Jun 05, 1992
|    Written By: Danielle Argiro
| Modifications:
|
------------------------------------------------------------*/
kguide *kvf_find_guide(
   kguide *guide,
   int    var_token)
{
        while (guide != NULL)
        {
            if (guide->var_token  == var_token) break;
            guide = guide->next;
        }
        return(guide);
}


/*------------------------------------------------------------
|
|  Routine Name: kvf_set_subform_values
|
|       Purpose: Sets values in the subform structure to defaults
|		 or according to values passed in
|
|         Input: master    - pointer to control structure rep. master
|                subform   - pointer to subform structure
|	  	 line      - UIS line defining -d, -n, etc. subform item
|		 type      - type of subform button
|		 form      - backlink to the form on which subform button is
|		 submenu   - TRUE if subform button is part of a submenu
|                index     - index of UIS line
|                filename_lookup - indexes into 'filenames' array
|                linenum_lookup  - holds actual line number of file for UIS line
|                filenames       - array of filenames comprising UIS
|
|        Output: none
|       Returns: TRUE on success, FALSE on failure
|
|          Date: May 11, 1992
|    Written By: Danielle Argiro 
| Modifications:
|
---------------------------------------------------------------*/

int kvf_set_subform_values(
   kcontrol   *master,
   ksubform   *subform,
   char       *line,
   int        type,
   kform      *form,
   kselection *submenu,
   int        index,
   int        *filename_lookup,
   int        *linenum_lookup,
   char       **filenames)
{
	Line_Info line_info;

	subform->line         = kstrdup(line); 	    
	subform->quit         = FALSE; 	    
	subform->guide        = NULL; 
	subform->type         = type;
	subform->next         = NULL;
	subform->back_submenu = submenu;
	subform->back_form    = form;
	subform->back_control = master;
	subform->back_kformstruct = kvf_create_struct((kaddr) subform,
                                                       type, KSUBFORM);
        kvf_link_formstruct(subform->back_kformstruct);

	kvf_clear_line_info(&line_info);
	if (!(kvf_gen_parse(line, &line_info)))
	{
	    kvf_parsing_error_mesg(index, filename_lookup, 
				   linenum_lookup, filenames);
            return(FALSE);
	}
	kvf_free_line_info_strings(&line_info);
	return(TRUE);
}


/*------------------------------------------------------------
|
|  Routine Name: kvf_set_guide_values
|
|       Purpose: sets values in the guide structure to defaults
|		 or according to values passed in
|
|         Input: guide    - pointer to guide structure to set
|		 line     - current UIS line
|		 form     - backlink to the form on which guide button is
|		 subform  - backlink to the subform on which guide button is
|		 index   - index into the UIS of current line
|                filename_lookup - indexes into 'filenames' array
|                linenum_lookup  - holds actual line number of file for UIS line
|                filenames       - array of filenames comprising UIS
|        Output: none
|       Returns: TRUE on success, FALSE on failure
|
|          Date: Apr 11, 1992
|    Written By: Danielle Argiro 
| Modifications:
|
---------------------------------------------------------------*/

int kvf_set_guide_values(
   kcontrol   *guidepane,
   kguide     *guide,
   char       *line,
   kform      *form,
   ksubform   *subform,
   kselection *submenu,
   int        index,
   int        *filename_lookup,
   int        *linenum_lookup,
   char       **filenames)
{
	Line_Info line_info;
	char      item_name[KLENGTH], guide_name[KLENGTH];

	kvf_clear_line_info(&line_info);
	if (!(kvf_gen_parse(line, &line_info)))
	{
	      kvf_parsing_error_mesg(index, filename_lookup, 
				     linenum_lookup, filenames);
	      kvf_free_line_info_strings(&line_info);
              return(FALSE);
	}

	guide->button       = NULL;
	guide->line         = kstrdup(line);
	guide->selected     = FALSE; 
	guide->pane         = NULL;
	guide->back_submenu = submenu;
	guide->back_form    = form;
	guide->back_subform = subform;
	guide->back_control = guidepane;
	guide->next         = NULL;

	guide->back_kformstruct = kvf_create_struct((kaddr) guide,
                                       KUIS_GUIDEBUTTON, KGUIDE);
        kvf_link_formstruct(guide->back_kformstruct);

	ksprintf(guide_name, "%s_button", line_info.variable);
	kvf_free_line_info_strings(&line_info);

	ksprintf(item_name, "%s.%s.%s", 
		ktoken_to_string(form->var_token), 
		ktoken_to_string(subform->var_token), 
		guide_name);
	guide->name_token = kstring_to_token(item_name);
	guide->var_token  = kstring_to_token(line_info.variable);

	kvf_free_line_info_strings(&line_info);
	return(TRUE);
}



/*------------------------------------------------------------
|
|  Routine Name: kvf_set_selection_values
|
|       Purpose: Sets values in the selection structure to defaults
|		 or according to values passed in
|
|         Input: selection     - pointer to selection structure to set
|                line          - current UIS line producing this selection
|                type          - type of UIS line
|                modified      - for cantata, starting modified or not
|                subform       - backlink to subform on which selection is
|                guide         - backlink to guide on which selection is
|                control       - backlink to control panel on which selection is
|                back_submenu  - backlink to submenu, if selection on submenu
|                variable      - variable name on UIS line
|                comment       - comment before UIS line
|
|        Output: none
|       Returns: Returns TRUE on success, FALSE on failure
|          Date: Apr 10, 1992
|    Written By: Danielle Argiro
| Modifications:
|
---------------------------------------------------------------*/

int kvf_set_selection_values(
   kselection *selection,
   char       *line,
   int        type,
   kform      *form,
   ksubform   *subform,
   kguide     *guide,
   kcontrol   *control,
   kselection *back_submenu,
   char       *variable,
   char       *comment)
{
        char      item_name[KLENGTH];
        char      sel_name[KLENGTH];
	Line_Info line_info;

        selection->type          = type;
        selection->line          = kstrdup(line);
        selection->comment       = kstrdup(comment);
        selection->modified      = FALSE;
        selection->back_submenu  = back_submenu;
        selection->toggle_next   = NULL;
        selection->toggle_num    = 0;
        selection->back_form     = form;
        selection->back_subform  = subform;
        selection->back_guide    = guide;
        selection->back_control  = control;
        selection->next          = NULL;
        selection->opt_selected  = 0;

	selection->back_kformstruct = kvf_create_struct((kaddr) selection, type,
                                                        KSELECTION);
	kvf_link_formstruct(selection->back_kformstruct);

	if (type == KUIS_QUIT)
	   ksprintf(sel_name, "quit");
	else if (type == KUIS_ROUTINE)
	{
	   kvf_clear_line_info(&line_info);
	   kvf_parse_routine_line(line, &line_info);
	   ksprintf(sel_name, "%s", line_info.routine);
	   kvf_free_line_info_strings(&line_info);
	}
	else if (type == KUIS_MUTEXCL)
	   ksprintf(sel_name, "mut_excl", variable);
	else if (type == KUIS_MUTINCL)
	   ksprintf(sel_name, "mut_incl", variable);
	else if (type == KUIS_GROUP)
	   ksprintf(sel_name, "group", variable);
	else ksprintf(sel_name, "%s", variable);

	if (control->type == KPANE)
	    ksprintf(item_name, "%s.%s.%s.%s", 
		    ktoken_to_string(form->var_token), 
		    ktoken_to_string(subform->var_token), 
		    ktoken_to_string(control->var_token), 
		    sel_name);

	else if (control->type == KGUIDEPANE)
	    ksprintf(item_name, "%s.%s.%s", 
		    ktoken_to_string(form->var_token), 
		    ktoken_to_string(subform->var_token), 
		    sel_name);

	else if (control->type == KMASTER)
	    ksprintf(item_name, "%s.%s", 
		    ktoken_to_string(form->var_token), 
		    sel_name);

	selection->name_token = kstring_to_token(item_name);
        selection->var_token  = kstring_to_token(sel_name);

	kvf_clear_line_info(&line_info);
	if (!(kvf_gen_parse(selection->line, &line_info)))
	    return(FALSE);

        selection->opt_selected  = line_info.opt_sel;
	kvf_gen_deparse(&line_info, &selection->line);
	kvf_free_line_info_strings(&line_info);

	return(TRUE);
}

/*-------------------------------------------------------------
|
|  Routine Name: kvf_link_subform
|       Purpose: links subform into linked list of subforms
|         Input: master  - pointer to the master
|                subform - subform to link in
|                last    - calling routine declares last
|        Output: none
|       Returns: nothing
|          Date: May 13, 1992
|    Written By: Danielle Argiro 
| Modifications:
|
---------------------------------------------------------------*/
void kvf_link_subform(
   kcontrol *master,
   ksubform *subform,
   ksubform **last_subform)
{
	if (master->subform_list == NULL)
	{
	    master->subform_list = subform;
	    *last_subform = subform;
	}
	else
	{
	    (*last_subform)->next = subform;
	    *last_subform = subform;
	}
}

/*-------------------------------------------------------------
|
|  Routine Name: kvf_link_guide
|       Purpose: links guide into linked list of guides
|         Input: guidepane - pointer to the guidepane
|                guide     - guide to link in
|                last      - calling routine declares last
|        Output: none
|       Returns: nothing
|          Date: May 13, 1992
|    Written By: Danielle Argiro 
| Modifications:
|
---------------------------------------------------------------*/
void kvf_link_guide(
   kcontrol *guidepane,
   kguide   *guide,
   kguide   **last)
{
	if (guidepane->guide_list == NULL) 
	{ 
	    guidepane->guide_list = guide;
	    *last = guide;
	}
	else
	{
	    (*last)->next = guide;
	    *last = guide;
	}
}

/*-------------------------------------------------------------
|
|  Routine Name: kvf_link_sel
|       Purpose: Links selection into linked list of selections.
|                This routine is used by createtree.c routines;
|                it is more efficient because it passes back the *last
|                pointer so we don't have to search for the last selection
|                each time (since the sel_list is not doubly linked)
|
|         Input: control   - pointer to the master, guidepane, or pane
|                selection - selection to link in
|                last      - calling routine declares last
|        Output: 
|       Returns:
|          Date: May 13, 1992
|    Written By: Danielle Argiro 
| Modifications:
|
---------------------------------------------------------------*/
void kvf_link_sel(
   kcontrol   *control,
   kselection *selection,
   kselection **last)
{
	if (control->sel_list == NULL) 
	{ 
	    control->sel_list = selection;
	    *last = selection;
	}
	else
	{
	    (*last)->next = selection;
	    *last = selection;
	}
}

/*-------------------------------------------------------------
|
|  Routine Name: kvf_link_selection
|       Purpose: Links selection into linked list of selections.
|                This routine is used creation.c routines;
|                it is less efficient because it has to search for 
|                the last selection in the list each time since the 
|                sel_list is not doubly linked) but guise and xvlang
|                are not in a position to save the *last pointer like
|                the routines in createtree.c are.
|
|         Input: control   - pointer to the master, guidepane, or pane
|                selection - selection to link in
|        Output:
|       Returns:
|          Date: June 5, 1994
|    Written By: Danielle Argiro
| Modifications:
|
---------------------------------------------------------------*/
void kvf_link_selection(
   kcontrol   *control,
   kselection *selection)
{
        kselection *last;

        if (control->sel_list == NULL)
            control->sel_list = selection;
        else
        {
            last = control->sel_list;
            while(last->next != NULL)
                last = last->next;

            last->next = selection;
        }
}


/*-------------------------------------------------------------
|
|  Routine Name: kvf_link_member
|       Purpose: links a group member into linked list of members
|         Input: group_start  - pointer to the group header
|                member       - group member to link in
|                last         - calling routine declares last
|        Output: none
|       Returns: nothing
|          Date: Oct 07, 1992
|    Written By: Danielle Argiro
| Modifications:
|
---------------------------------------------------------------*/
void kvf_link_member(
   kselection *group_start,
   kselection *member,
   kselection **last)
{
	if (group_start->group_next == NULL)
        {
            group_start->group_next = member;
            *last = member;
        }
        else
        {
            (*last)->next = member;
            *last = member;
        }
}

/*-------------------------------------------------------------
|
|  Routine Name: kvf_link_toggle
|       Purpose: links a toggle member into linked list of toggle members
|         Input: group_start  - pointer to the toggle header
|                member       - toggle member to link in
|                last         - calling routine declares last
|        Output: none
|       Returns: nothing
|          Date: Oct 07, 1992
|    Written By: Danielle Argiro
| Modifications:
|
---------------------------------------------------------------*/
void kvf_link_toggle(
   kselection *toggle_start,
   kselection *member,
   kselection **last)
{
	if (toggle_start->toggle_next == NULL)
        {
            toggle_start->toggle_next = member;
            *last = member;
        }
        else
        {
            (*last)->next = member;
            (*last)->next = member;
            *last = member;
        }
}


/*-------------------------------------------------------------
|
|  Routine Name: kvf_link_submenu
|       Purpose: links kformstruct into linked list of submenu items
|         Input: submenu_start - pointer to the submenu (-D) 
|                kformstruct     - submenu member to link in
|                last          - calling routine declares last
|        Output: none
|       Returns: nothing
|          Date: May 13, 1992
|    Written By: Danielle Argiro 
| Modifications:
|
---------------------------------------------------------------*/
void kvf_link_submenu(
   kselection   *submenu_start,
   kform_struct *member,
   kform_struct **last_member)
{
	if (submenu_start->submenu_next == NULL)
	{
	    submenu_start->submenu_next = member;
	    *last_member = member;
	}
	else
	{
	    (*last_member)->next = member;
	    *last_member = member;
	}
}

/*------------------------------------------------------------
|
|  Routine Name: kvf_error_on_submenu
|
|       Purpose: Prints appropriate error message on encountering
|                illegal line within SubMenu [-D to -E] definition
|
|         Input: submenu         - TRUE if item on submenu
|                index           - index into the UIS of current line
|                filename_lookup - indexes into 'filenames' array
|                linenum_lookup  - holds actual line number of file for UIS line
|                filenames       - array of filenames comprising UIS
|
|        Output: none
|       Returns: TRUE if item on submenu, FALSE otherwise
|          Date: May 5, 1992
|    Written By: Danielle Argiro
| Modifications:
|
---------------------------------------------------------------*/
int kvf_error_on_submenu(
   int  submenu,
   int  index,
   int  *filename_lookup,
   int  *linenum_lookup,
   char **filenames)
{
        if (submenu == TRUE)
        {
	    errno = KUIS_SYNTAX;
            kerror("kforms", "kvf_create_form", "Illegal UIS line in SubMenu definition on line %d of %s;  aborting form creation.", linenum_lookup[index], filenames[filename_lookup[index]]);
        }
        return(submenu);
}

/*----------------------------------------------------------
|
| Routine Name: kvf_find_matching_M_line
|
|      Purpose: A problem results when we must name the (-d) subform button
|               according to the name on the [-M] line, which hasn't
|               been read in yet.  This routine finds the (-M) line with a
|               variable name to match the (-d) line, to solve that problem.
|
|         Input: database           - ptr to internal UIS struct
|                sfb_index          - index into the UIS of current -d line
|                filename_lookup    - indexes into 'filenames' array
|                linenum_lookup     - holds actual line # of file for UIS line
|                filenames          - array of filenames comprising UIS
|
|        Output: Returns the -M line on success, NULL on failure.
|                M_index - returns the database index of the -M line
|
|          Date: Nov 3, 1992
|    Written By: Danielle Argiro
| Modifications:
|
---------------------------------------------------------------*/

char *kvf_find_matching_M_line(
   char **database,
   int  sfb_index,
   int  *filename_lookup,
   int  *linenum_lookup,
   char **filenames,
   int  *M_index)
{
        int  flag, found = FALSE;
        char *variable, *control_line = NULL;
        int  index;
        Line_Info line_info;

        kvf_clear_line_info(&line_info);
        flag = kvf_get_line_type(database[sfb_index]);
        if (flag != KUIS_SUBFORMBUTTON)
        {
	    errno = KINTERNAL;
            kerror("kvforms", "kvf_find_matching_M_line",
                   "Incorrect UIS line passed into kvf_find_matching_M_line");
            return(NULL);
        }
        kvf_gen_parse(database[sfb_index], &line_info);
        variable = kstrdup(line_info.variable);
        kvf_free_line_info_strings(&line_info);

        index = sfb_index;

        while ((!found) && (database[index] != NULL))
        {
            flag = kvf_get_line_type(database[index]);
            if (flag == KUIS_STARTSUBFORM)
            {
                kvf_parse_startsubform_line(database[index], &line_info);
                if (kstrcmp(variable, line_info.variable) == 0)
                {
                    found = TRUE;
                    control_line = kstrdup(database[index]);
                    *M_index = index;
                }
                else index++;
                kvf_free_line_info_strings(&line_info);
            }
            else index++;
        }

        if (!found)
        {
	    errno = KUIS_SYNTAX;
            kerror("kvforms", "kvf_create_form",  "Cannot find [-M] line with variable that matches the variable '%s' of the SubformButton [-d] on line %d of %s;  aborting form creation.", variable, linenum_lookup[sfb_index], filenames[filename_lookup[sfb_index]]);
        }

        kfree(variable);
        return(control_line);
}


/*------------------------------------------------------------
|
| Routine Name: kvf_find_matching_P_line
|
|      Purpose: A problem results when we must name the (-g) guide button
|               according to the name on the [-P] line, which hasn't
|               been read in yet.  This routine finds the (-P) line with a
|               variable name to match the (-g) line, to solve that problem.
|
|         Input: database           - ptr to internal UIS
|                gb_index           - index into the UIS of (-g) line
|                filename_lookup    - indexes into 'filenames' array
|                linenum_lookup     - holds actual line # of file for UIS line
|                filenames          - array of filenames comprising UIS
|
|        Output: Returns the -P line on success, NULL on failure.
|                P_index - returns the database index of the -P line
|          Date: Nov 3, 1992
|    Written By: Danielle Argiro
| Modifications:
|
---------------------------------------------------------------*/

char *kvf_find_matching_P_line(
   char **database,
   int  gb_index,
   int  *filename_lookup,
   int  *linenum_lookup,
   char **filenames,
   int  *P_index)
{
        int  flag, found = FALSE;
        char *variable, *control_line = NULL;
        int  index;
        Line_Info line_info;

        kvf_clear_line_info(&line_info);
        flag = kvf_get_line_type(database[gb_index]);
        if (flag != KUIS_GUIDEBUTTON)
        {
	    errno = KINTERNAL;
            kerror("kvforms", "kvf_find_matching_P_line",
                   "Incorrect UIS line passed into kvf_find_matching_P_line");
            return(NULL);
        }
        kvf_parse_guide_line(database[gb_index], &line_info);
        variable = kstrdup(line_info.variable);
        kvf_free_line_info_strings(&line_info);

        index = gb_index;

        while ((!found) && (database[index] != NULL))
        {
            flag = kvf_get_line_type(database[index]);
            if (flag == KUIS_STARTPANE)
            {
                if (!(kvf_parse_startpane_line(database[index], &line_info)))
                {
                    kvf_parsing_error_mesg(index, filename_lookup,
                                           linenum_lookup, filenames);
                    return(NULL);
                }
                if (kstrcmp(variable, line_info.variable) == 0)
                {
                    found = TRUE;
                    control_line = kstrdup(database[index]);
                    *P_index = index;
                }
                else index++;
                kvf_free_line_info_strings(&line_info);
            }
            else index++;
        }

        if (!found)
        {
	    errno = KUIS_SYNTAX;
            kerror("kvforms", "kvf_create_form",  "Cannot find [-P] line with variable that matches the variable '%s' of the GuideButton [-g] on line %d of %s;  aborting form creation.", variable, linenum_lookup[gb_index], filenames[filename_lookup[gb_index]]);
        }

        kfree(variable);
        return(control_line);
}


