/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>        Miscellaneous Attribute Actions                 <<<<
   >>>>                                                       <<<<
   >>>>  Private:                                             <<<<
   >>>>	     		kvf_get_logic_val()		      <<<<
   >>>>	     		kvf_set_logic_val()		      <<<<
   >>>>	     		kvf_get_logic_def()		      <<<<
   >>>>	     		kvf_set_logic_def()		      <<<<
   >>>>	     		kvf_get_logic_label()		      <<<<
   >>>>	     		kvf_set_logic_label()		      <<<<
   >>>>                                                       <<<<
   >>>>	     		kvf_set_list_index()		      <<<<
   >>>>	     		kvf_get_list_index()		      <<<<
   >>>>	     		kvf_set_list_label()		      <<<<
   >>>>	     		kvf_get_list_label()		      <<<<
   >>>>	     		kvf_get_list_size()		      <<<<
   >>>>	     		kvf_set_list_size()		      <<<<
   >>>>	     		kvf_get_list_val()		      <<<<
   >>>>	     		kvf_set_list_val()		      <<<<
   >>>>	     		kvf_get_list_start()		      <<<<
   >>>>	     		kvf_set_list_start()		      <<<<
   >>>>	     		kvf_get_list_contents()		      <<<<
   >>>>	     		kvf_set_list_contents()		      <<<<
   >>>>	     		kvf_get_list_doubleclick()	      <<<<
   >>>>	     		kvf_set_list_doubleclick()	      <<<<
   >>>>                                                       <<<<
   >>>>	     		kvf_get_exectype()		      <<<<
   >>>>	     		kvf_set_exectype()		      <<<<
   >>>>	     		kvf_get_routine()		      <<<<
   >>>>	     		kvf_set_routine()		      <<<<
   >>>>	     		kvf_get_routine()		      <<<<
   >>>>	     		kvf_set_routine()		      <<<<
   >>>>	     		kvf_get_me_subforms()		      <<<<
   >>>>	     		kvf_set_me_subforms()		      <<<<
   >>>>                                                       <<<<
   >>>>	     		kvf_set_list_add()	              <<<<
   >>>>	     		kvf_set_list_delete()	              <<<<
   >>>>	     		kvf_set_list_deleteall()	      <<<<
   >>>>   Static:                                             <<<<
   >>>>   Public:                                             <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<  */

#include "internals.h"


/*-----------------------------------------------------------
|
|       Routine: kvf_get_logic_val
|       Purpose: Gets the value of the logical from 
|                the (-l) UIS line of a logical selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_LOGIC_VAL ("kvf_logic_val")
|        Output: calldata    - passes back logical value (TRUE or FALSE)
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_logic_val(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	Line_Info    line_info;
	int          *logic_val;

        /* set the pointer to be returned */
        logic_val = (int *) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

	if (line_info.logical_val)
	    *logic_val = TRUE;
	else *logic_val = FALSE;

	kvf_free_line_info_strings(&line_info);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_logic_val
|       Purpose: Sets the value of the logical on 
|                the (-l) UIS line of a logical selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_LOGIC_VAL ("kvf_logic_val")
|                calldata    - provides boolean value of TRUE or FALSE
|        Output: none
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_logic_val(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int          *logic_val;
	Line_Info    line_info;

	/* the value to which KVF_LOGIC_VAL is to be set */
        logic_val  = (int *) calldata;
	if (!(kvf_check_boolean(*logic_val)))
        {
	    errno = KINVALID_PARAMETER;
            kerror("kforms", "kvf_set_logic_val", 
		   "'kvf_logic_val' may only be set to 0 or 1");
	    return(FALSE);
        }

	/*	
	 * change default of logical
	 */
	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

	line_info.logical_val = *logic_val;
	kvf_gen_deparse_kformstruct(kformstruct, &line_info);

	return(TRUE);
}


/*-----------------------------------------------------------
|
|       Routine: kvf_get_logic_def
|       Purpose: Gets the value of the logical default field
|                from the (-l) UIS line of a logical selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_LOGIC_DEF ("kvf_logic_def")
|        Output: calldata    - passes back logical default (TRUE or FALSE)
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_logic_def(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	Line_Info    line_info;
	int          *logic_def;

        /* set the pointer to be returned */
        logic_def = (int *) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

	if (line_info.logical_def)
	    *logic_def = TRUE;
	else *logic_def = FALSE;

	kvf_free_line_info_strings(&line_info);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_logic_def
|       Purpose: Sets the value of the logical default field
|                on the (-l) UIS line of a logical selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_LOGIC_DEF ("kvf_logic_def")
|                calldata    - provides boolean value of TRUE or FALSE
|        Output: none
|
|          Date: Nov 12, 1992
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_logic_def(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int          *logic_def;
	Line_Info    line_info;

	/* the value to which KVF_LOGIC_DEF is to be set */
        logic_def  = (int *) calldata;
	if (!(kvf_check_boolean(*logic_def)))
        {
	    errno = KINVALID_PARAMETER;
            kerror("kforms", "kvf_set_logic_def", 
		   "'kvf_logic_def' may only be set to 0 or 1");
	    return(FALSE);
        }

	/*	
	 * change default of logical
	 */
	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

	line_info.logical_def = *logic_def;
	kvf_gen_deparse_kformstruct(kformstruct, &line_info);

	kvf_set_attribute(kformstruct, KVF_LOGIC_VAL, *logic_def);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_get_logic_label
|       Purpose: Gets the label associated with a value of 0 or 1
|                from the (-l) UIS line of a logical selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_LOGIC_1LABEL ("kvf_logic_1label") or 
|                              KVF_LOGIC_0LABEL ("kvf_logic_0label")
|                              KVF_LOGIC_LABEL  ("kvf_logic_label")
|        Output: calldata    - passes back title for value of 0 or 1 on logical
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_logic_label(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	Line_Info    line_info;
	char         **val_label;
	int          logic_val;

        /* set the pointer to be returned */
        val_label = (char **) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);


	if (kstrcmp(attribute, KVF_LOGIC_1LABEL) == 0)
	{
	    if (line_info.val_labels[1] == NULL)
	 	*val_label = NULL;
	    else *val_label = kstrdup(line_info.val_labels[1]);
	}

	else if (kstrcmp(attribute, KVF_LOGIC_0LABEL) == 0)
	{
	    if (line_info.val_labels[0] == NULL)
		*val_label = NULL;
	    else *val_label = kstrdup(line_info.val_labels[0]);
	}

	else if (kstrcmp(attribute, KVF_LOGIC_LABEL) == 0)
	{
	     kvf_get_attribute(kformstruct, KVF_LOGIC_VAL, &logic_val);
	     if (line_info.val_labels[logic_val] == NULL)
                *val_label = NULL;
             else *val_label = kstrdup(line_info.val_labels[logic_val]);
	}

	kvf_free_line_info_strings(&line_info);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_logic_label
|       Purpose: Sets the label associated with a value of 0 or 1
|                on the (-l) UIS line of a logical selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_LOGIC_1LABEL ("kvf_logic_1label") or 
|                              KVF_LOGIC_0LABEL ("kvf_logic_0label")
|                              KVF_LOGIC_LABEL  ("kvf_logic_label")
|                calldata    - provides string for value of 0 or 1 for logical
|        Output: none
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_logic_label(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	char         **val_label;
	Line_Info    line_info;
	int          logic_val;

	/* value to which KVF_LOGIC_1LABEL or KVF_LOGIC_0LABEL is to be set */
        val_label  = (char **) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);


	if (kstrcmp(attribute, KVF_LOGIC_1LABEL) == 0)
	{
	    kfree(line_info.val_labels[1]);
	    if (kstrlen(*val_label) > 0)
	        line_info.val_labels[1] = kstrdup(*val_label);
	    else line_info.val_labels[1] = NULL;
	}
	else if (kstrcmp(attribute, KVF_LOGIC_0LABEL) == 0)
	{
	    kfree(line_info.val_labels[0]);
	    if (kstrlen(*val_label) > 0)
	        line_info.val_labels[0] = kstrdup(*val_label);
	    else line_info.val_labels[0] = NULL;
	}
	else if (kstrcmp(attribute, KVF_LOGIC_LABEL) == 0)
        {
	    kvf_get_attribute(kformstruct, KVF_LOGIC_VAL, &logic_val);
            kfree(line_info.val_labels[logic_val]);
            if (kstrlen(*val_label) > 0)
                line_info.val_labels[logic_val] = kstrdup(*val_label);
            else line_info.val_labels[logic_val] = NULL;
        }

	else return(FALSE);

	kvf_gen_deparse_kformstruct(kformstruct, &line_info);

	return(TRUE);
}
/*-----------------------------------------------------------
|
|       Routine: kvf_get_list_index
|       Purpose: Gets the value of the value index field
|                from the (-l), (-x), (-z), or (-c) UIS line 
|                of a logical, list, displaylist, or cycle selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_LIST_INDEX ("kvf_list_index")
|        Output: calldata    - passes back integer value of the list index
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_list_index(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	Line_Info    line_info;
	int          *list_index;

        /* set the pointer to be returned */
        list_index = (int *) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

	if (kformstruct->Selptr->type == KUIS_STRINGLIST)
	    *list_index = line_info.list_val - 1;
	else *list_index = line_info.list_val - line_info.int_val;

	kvf_free_line_info_strings(&line_info);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_list_index
|       Purpose: Sets the value of the value index field
|                on the (-l), (-x), (-z), or (-c) UIS line of a 
|                logical, list, displaylist, or cycle selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_LIST_INDEX ("kvf_list_index")
|                calldata    - provides integer value of list_index
|        Output: none
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_list_index(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int          *list_index;
	Line_Info    line_info;

	/* the value to which KVF_VAL_INDEX is to be set */
        list_index  = (int *) calldata;

	if (*list_index == -1)
	{
	    if (kformstruct->flag != KUIS_DISPLAYLIST)
	    {
		kerror("kforms", "kvf_set_list_index",
		       "Cannot set list index to -1, except on displaylist selections");
		return(FALSE);
	    }
	}
	else if (*list_index < 0)
	{
	    kerror("kforms", "kvf_set_list_index",
		   "Attempt to set list index to invalid negative value");
	    return(FALSE);
	}

	/*	
	 * change val index
	 */
	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

	if (kformstruct->Selptr->type == KUIS_STRINGLIST)
	    line_info.list_val = *list_index + 1;
	else line_info.list_val = *list_index + line_info.int_val;
	kvf_gen_deparse_kformstruct(kformstruct, &line_info);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_get_list_label
|       Purpose: Gets the label associated with a value of n from the
|                (-l), (-x), (-z), or (-c) UIS line of a 
|                logical, list, displaylist, or cycle selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_LIST_LABEL ("kvf_list_label")
|        Output: calldata    - passes back title for value of n on logical
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_list_label(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	Line_Info    line_info;
	char         **list_label;
	int          list_index;

        /* set the pointer to be returned */
        list_label = (char **) calldata;

	if (!(kvf_get_attribute(kformstruct, KVF_LIST_INDEX, &list_index)))
	    return(FALSE);

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

	if (kformstruct->Selptr->type == KUIS_STRINGLIST)
            list_index = line_info.list_val - 1;
        else list_index = line_info.list_val - line_info.int_val;

	if (list_index == -1)
	    *list_label = NULL;
	else if (line_info.val_labels == NULL)
	    *list_label = NULL;
	else if (line_info.val_labels[list_index] == NULL)
	    *list_label = NULL;
	else *list_label = kstrdup(line_info.val_labels[list_index]);

	kvf_free_line_info_strings(&line_info);
	return(TRUE);
 
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_list_label
|       Purpose: Sets the label associated with a value of N on the
|                (-l), (-x), (-z), or (-c) UIS line of a 
|                logical, list, displaylist, or cycle selection.
|                Note that the list value must be set to N first,
|                before the label for that value can be assigned.
|
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_LIST_LABEL ("kvf_list_label")
|                calldata    - provides string value of title for 
|                            value of n on logical
|        Output: none
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_list_label(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	char         **list_label;
	int          list_index;
	Line_Info    line_info;

	/* the value to which KVF_VAL_LABEL is to be set */
        list_label  = (char **) calldata;

	/* can't set list label to NULL */
	if (*list_label == NULL)
	{
	    kerror("kforms", "kvf_set_list_label",
		   "Attempt to set list label to invalid NULL value");
	    return(FALSE);
	}

	/* get the current index, to see which label we are setting */
	if (!(kvf_get_attribute(kformstruct, KVF_LIST_INDEX, &list_index)))
            return(FALSE);

	/* on displaylists, index can be set to -1, indicating no value.
           if this is the case, cannot change the label */
	if (list_index == -1)
	{
	    kerror("kforms", "kvf_set_list_label",
		   "Invalid attempt to set label on displaylist selection when index is set to -1 (indicating NO current value)");
	    return(FALSE);
	}

	/* set the label */
	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

        kfree(line_info.val_labels[list_index]);
	if (kstrlen(*list_label) > 0)
            line_info.val_labels[list_index] = kstrdup(*list_label);
	else line_info.val_labels[list_index] = NULL;
        kvf_gen_deparse_kformstruct(kformstruct, &line_info);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_get_list_size
|       Purpose: Gets the number of list choices from the 
|                (-l), (-x), (-z), or (-c) UIS line of a 
|                logical, list, displaylist, or cycle selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_LIST_SIZE ("kvf_list_size")
|        Output: calldata    - passes back number of choices for list
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_list_size(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	Line_Info    line_info;
        int          *list_size;

        /* set the pointer to be returned */
        list_size = (int *) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

        *list_size = line_info.list_num;

        kvf_free_line_info_strings(&line_info);
	return(TRUE);

}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_list_size
|       Purpose: Sets the number of list choices on the 
|                (-l), (-x), (-z), or (-c) UIS line of a 
|                logical, list, displaylist, or cycle selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_LIST_SIZE ("kvf_list_size")
|                calldata    - provides number of choices for list
|        Output: none
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_list_size(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
        int           i, *list_size, old_list_size;
	Line_Info     line_info;
	kselection    *selection;
	char          **old_val_labels = NULL;

        /* the value to which KVF_LIST_SIZE is to be set */
        list_size = (int *) calldata;

	selection = kformstruct->Selptr;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);


	/*
	 *  save old val labels
	 */
	old_list_size = line_info.list_num;
	if (old_list_size > 0)
	{
	    old_val_labels = (char **) kcalloc(1, old_list_size*sizeof(char *));
	    if (old_val_labels == NULL)
	    {
	        kerror("kforms", "kvf_set_list_size",
	    	   "Unable to allocate array for labels");
	        return(FALSE);
	    }
	    for (i = 0; i < old_list_size; i++)
	        old_val_labels[i] = kstrdup(line_info.val_labels[i]);
	}

	/*
	 *  update list num, and reallocate val labels
	 */
	line_info.list_num = *list_size;
	kvf_realloc_val_labelnum(&line_info, *list_size);

	/*
	 *  transfer as many old labels as possible to 
         *  act as defaults for new labels
	 */
	for (i = 0; i < *list_size; i++)
	{
	    if (i < old_list_size)
	    {
	        if (old_val_labels[i] != NULL)
		    line_info.val_labels[i] = 
			kstrdup(old_val_labels[i]);
	        else line_info.val_labels[i] = kstrdup(" ");
	    }
	    else line_info.val_labels[i] = kstrdup(" ");
	}
	
	/*
	 *  recalculate the current list value
	 */
	if (line_info.list_val > (line_info.int_val + line_info.list_num - 1))
	    line_info.list_val = line_info.int_val + line_info.list_num - 1;
	else if (line_info.list_val < line_info.int_val)
	    line_info.list_val = line_info.int_val;

	if ((line_info.list_val - line_info.int_val == -1) &&
	    (selection->type != KUIS_DISPLAYLIST))
	    line_info.list_val =  line_info.int_val;

        kvf_gen_deparse_kformstruct(kformstruct, &line_info);

	/*
	 *  free the old val labels
	 */
	if (old_list_size > 0)
	{
	    for (i = 0; i < old_list_size; i++)
	        kfree(old_val_labels[i]);
	    kfree(old_val_labels);
	}

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_get_list_val
|       Purpose: Gets the current list choice (by integer value) from the 
|                (-l), (-x), (-z), or (-c) UIS line of a 
|                logical, list, displaylist, or cycle selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_LIST_VAL ("kvf_list_val")
|        Output: calldata    - passes back default choice for list
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_list_val(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
        Line_Info  line_info;
        int        *list_val;

        /* set the pointer to be returned */
        list_val = (int *) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

        *list_val = line_info.list_val;

        kvf_free_line_info_strings(&line_info);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_list_val
|       Purpose: Sets the current list choice (by integer value) on the 
|                (-l), (-x), (-z), or (-c) UIS line of a 
|                logical, list, displaylist, or cycle selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_LIST_VAL ("kvf_list_val")
|                calldata    - provides default choice for list
|        Output: none
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_list_val(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	Line_Info    line_info;
        int          i, *list_val; 
        kselection   *selection;
	char         temp[KLENGTH];
	char         mesg[KLENGTH];

        /* set the pointer to be returned */
        list_val = (int *) calldata;

	selection = kformstruct->Selptr;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

	if ((*list_val < line_info.int_val) &&
	    (selection->type != KUIS_DISPLAYLIST))
	{
	    mesg[0] = '\0';
	    ksprintf(mesg, "list value for %s (%s) must be >= start (%d); legal values include:\n",
                   line_info.title, line_info.variable, line_info.int_val);
	    for (i = 0; i < line_info.list_num; i++)
            {
                if (i < line_info.list_num-1)
		{
		     ksprintf(temp, "%d (\"%s\"), ", 
			      i+line_info.int_val, line_info.val_labels[i]);
                     kstring_cat(mesg, temp, mesg);
		}
                else 
		{
		    ksprintf(temp, "or %d (\"%s\")\n", 
			     i+line_info.int_val, line_info.val_labels[i]);
                    kstring_cat(mesg, temp, mesg);
		}
            }
	    kerror("kforms", "kvf_set_list_val", mesg);
	    kvf_free_line_info_strings(&line_info);
	    return(FALSE);
	}
	else if ((*list_val < line_info.int_val - 1) &&
            (selection->type == KUIS_DISPLAYLIST))
	{
	    kerror("kforms", "kvf_set_list_val",
		   "list default must be >= start (%d) to indicate selected item, or (%d) to indicate that NO item is selected.", line_info.int_val, line_info.int_val - 1);
	    kvf_free_line_info_strings(&line_info);
	    return(FALSE);
	}

	else if (*list_val > (line_info.int_val + line_info.list_num - 1))
	{
	    mesg[0] = '\0';
            ksprintf(mesg, "list value for %s (%s) must be <= %d; legal values include:\n",
		     line_info.title, line_info.variable, 
		     line_info.int_val + line_info.list_num - 1);
            for (i = 0; i < line_info.list_num; i++)
            {
                if (i < line_info.list_num-1)
                {
                     ksprintf(temp, "%d (\"%s\"), ",
                              i+line_info.int_val, line_info.val_labels[i]);
                     kstring_cat(mesg, temp, mesg);
                }
                else
                {
                    ksprintf(temp, "or %d (\"%s\")",
                             i+line_info.int_val, line_info.val_labels[i]);
                    kstring_cat(mesg, temp, mesg);
                }
            }
	    kerror("kforms", "kvf_set_list_val", mesg);
	    kvf_free_line_info_strings(&line_info);
	    return(FALSE);
	}
	else line_info.list_val = *list_val;

        kvf_gen_deparse_kformstruct(kformstruct, &line_info);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_get_list_item
|       Purpose: Gets the current list choice (by string value) from the 
|                (-l), (-x), (-z), or (-c) UIS line of a 
|                logical, list, displaylist, or cycle selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_LIST_VAL ("kvf_list_val")
|        Output: calldata    - passes back default choice for list
|
|          Date: March 31, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_list_item(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
        Line_Info  line_info;
	char      **list_item;
	int        indx;

        /* set the pointer to be returned */
        list_item = (char **) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

	indx =  line_info.list_val - line_info.int_val;

	if (line_info.val_labels == NULL)
	    *list_item = NULL;
	else if (indx < 0)
	    *list_item = NULL;
	else if (line_info.val_labels[indx] == NULL)
	    *list_item = NULL;
	else *list_item = kstrdup(line_info.val_labels[indx]);

        kvf_free_line_info_strings(&line_info);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_list_item
|       Purpose: Sets the current list choice (by string value) on the 
|                (-l), (-x), (-z), or (-c) UIS line of a 
|                logical, list, displaylist, or cycle selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_LIST_ITEM ("kvf_list_item")
|                calldata    - provides default choice for list
|        Output: none
|
|          Date: March 31, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_list_item(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
        int          i; 
	Line_Info    line_info;
	char         **list_item;

        /* set the pointer to be returned */
        list_item = (char **) calldata;

	if (*list_item == NULL)
	{	
	    kerror("kforms", "kvf_set_list_item",
		   "Cannot set value to NULL");
	    return(FALSE);
	}

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

	if (line_info.val_labels == NULL)
	    return(FALSE);

	/*
	 *  they want to set the list value to the item with the 
         *  string they specify.  first, find the string in the list.
         *  if it's not there, return false.
	 */
	i = 0;
        while ((i < line_info.list_num) &&
               (kstrncasecmp(line_info.val_labels[i], 
			     *list_item, kstrlen(*list_item)) != 0))
             i++;
	if (i >= line_info.list_num)
             return(FALSE);

        kvf_free_line_info_strings(&line_info);

	kvf_set_attribute(kformstruct, KVF_LIST_INDEX, i);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_get_list_start
|       Purpose: Gets the integer start value from the 
|                (-l), (-x), (-z), or (-c) UIS line of a 
|                logical, list, displaylist, or cycle selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_LIST_START ("kvf_list_start")
|        Output: calldata    - passes back starting integer for list
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_list_start(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	Line_Info    line_info;
        int          *list_start;

        /* set the pointer to be returned */
        list_start = (int *) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

        *list_start = line_info.int_val;

        kvf_free_line_info_strings(&line_info);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_list_start
|       Purpose: Sets the integer start value on the 
|                (-l), (-x), (-z), or (-c) UIS line of a 
|                logical, list, displaylist, or cycle selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_LIST_START ("kvf_list_start")
|                calldata    - provides starting integer for list
|        Output: none
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_list_start(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
        int          old_index, list_val;
	int          *list_start;
	Line_Info    line_info;

        /* the value to which KVF_LIST_START is to be set */
        list_start = (int *) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

	old_index = line_info.list_val - line_info.int_val;

	if (!(kvf_get_attribute(kformstruct, KVF_LIST_VAL, &list_val)))
	{
	    kvf_free_line_info_strings(&line_info);
	    return(FALSE);
	}

        /*
         * current list val less than new list start, putting the
         * list value L of range.  scoot list value up to list start.
         */
        if (list_val < *list_start)
            line_info.list_val = *list_start;

        /*
         * current list value greater than list_start + item num, putting the
         * list value R of range.  scoot list value back to highest value.
         */
        else if (list_val > (*list_start + line_info.list_num - 1))
            line_info.list_val = *list_start + line_info.list_num - 1;

	line_info.int_val = *list_start;
        kvf_gen_deparse_kformstruct(kformstruct, &line_info);

	
	/*
	 *  current list value is within range now, but relative offset probably
         *  changed.  (ie, if the 2nd item was selected before, want the
         *  second item selected now, even though the start value changed.
	 */
	if (line_info.list_num > 0)
	    kvf_set_attribute(kformstruct, KVF_LIST_INDEX, old_index);


	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_get_list_contents
|       Purpose: Gets the list of strings from the 
|                (-l), (-x), (-z), or (-c) UIS line of a
|                logical, list, displaylist, or cycle selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_LIST_CONTENTS ("kvf_list_contents")
|        Output: calldata    - passes back array of strings making up list
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_list_contents(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int          i;
	Line_Info    line_info;
        char         ***val_labels;

        /* set the pointer to be returned */
        val_labels = (char ***) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

	*val_labels = (char **) kcalloc(1, line_info.list_num*sizeof(char *));
	if ((*val_labels == NULL) && (line_info.list_num > 0))
	{
	    kerror("kforms", "kvf_get_list_contents",
		   "Unable to allocate string array in which to return list contents");
	    return(FALSE);
	}
	for (i = 0; i < line_info.list_num; i++)
	    (*val_labels)[i] = kstrdup(line_info.val_labels[i]);

        kvf_free_line_info_strings(&line_info);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_list_contents
|       Purpose: Sets the list of string on the 
|                (-l), (-x), (-z), or (-c) UIS line of a
|                logical, list, displaylist, or cycle selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_LIST_CONTENTS ("kvf_list_contents")
|                calldata    - provides array of strings for list
|        Output: none
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications: 
| ------------------------------------------------------------*/

/* ARGSUSED */
int kvf_set_list_contents(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int          i;
	kselection   *selection;
	char         ***val_labels;
	Line_Info    line_info;

        /* the value to which KVF_LIST_CONTENTS is to be set */
        val_labels = (char ***) calldata;

	selection = kformstruct->Selptr;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);


        /*
         *  move new list contents into line_info.val_labels array
         */
        for (i = 0; i < line_info.list_num; i++)
        {
            kfree(line_info.val_labels[i]);
            line_info.val_labels[i] = kstrdup((*val_labels)[i]);
        }

	if (selection->type == KUIS_DISPLAYLIST)
        {
	   if ((line_info.list_num > 0) || (*val_labels != NULL))
		line_info.list_val = line_info.int_val - 1;
	}
        kvf_gen_deparse_kformstruct(kformstruct, &line_info);
	return(TRUE);
}


/*-----------------------------------------------------------
|
|       Routine: kvf_get_list_doubleclick
|       Purpose: Gets the value of the doubleclick field
|                from the (-z) DISPLAYLIST UIS line 
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_LIST_DOUBLECLICK ("kvf_list_doubleclick")
|        Output: calldata    - passes back boolean value of the list index
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_list_doubleclick(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	Line_Info    line_info;
	int          *doubleclick;

        /* set the pointer to be returned */
        doubleclick = (int *) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

	*doubleclick = line_info.special;

	kvf_free_line_info_strings(&line_info);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_list_doubleclick
|       Purpose: Sets the value of the doubleclick field
|                from the (-z) DISPLAYLIST UIS line
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_LIST_DOUBLECLICK ("kvf_list_doubleclick")
|                calldata    - provides integer value of list_index
|        Output: none
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_list_doubleclick(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int          *doubleclick;
	Line_Info    line_info;

	/* the value to which KVF_VAL_INDEX is to be set */
        doubleclick  = (int *) calldata;

        if (!(kvf_check_boolean(*doubleclick)))
        {
            errno = KINVALID_PARAMETER;
            kerror("kforms", "kvf_set_list_doubleclick",
                   "KVF_LIST_DOUBLECLICK may only be set to 0 or 1");
            return(FALSE);
        }

	/*	
	 * change doubleclick flag
	 */
	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

	line_info.special = *doubleclick;
	kvf_gen_deparse_kformstruct(kformstruct, &line_info);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_get_exectype
|       Purpose: Gets the execution type from the (-R) UIS line 
|                of a routine selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_EXECTYPE ("kvf_exectype")
|        Output: calldata    - passes back execution type for Routine button
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_exectype(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	Line_Info    line_info;
        int          *exectype;

        /* set the pointer to be returned */
        exectype = (int *) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

        *exectype = line_info.exec_type + 1;

        kvf_free_line_info_strings(&line_info);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_exectype
|       Purpose: Sets the execution type on the (-R) UIS line 
|                of a routine selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_EXECTYPE ("kvf_exectype")
|                calldata    - provides execution type for Routine button
|        Output: none
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications: 
| ------------------------------------------------------------*/

/* ARGSUSED */
int kvf_set_exectype(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int        *exectype;
	Line_Info  line_info;

        /* the value to which KVF_EXECTYPE is to be set */
        exectype = (int *) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

	line_info.exec_type = *exectype -1;
        kvf_gen_deparse_kformstruct(kformstruct, &line_info);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_get_routine
|       Purpose: Gets the value of the routine field from 
|                the (-R) UIS line of a routine selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute - KVF_ROUTINE ("kvf_routine")
|        Output: calldata  - passes back string value of routine
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_routine(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	Line_Info    line_info;
	char         **routine;

        /* set the pointer to be returned */
        routine = (char **) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

	if (line_info.routine == NULL)
	{
	    *routine = NULL;
	    return(FALSE);
	}
	else *routine = kstrdup(line_info.routine);

	kvf_free_line_info_strings(&line_info);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_routine
|       Purpose: Sets the value of the routine field on 
|                the (-R) UIS line of a routine selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_ROUTINE ("kvf_routine")
|                calldata    - provides string value of routine
|        Output: none
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
-------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_routine(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	char         **routine;
	Line_Info    line_info;

	/* the value to which KVF_ROUTINE is to be set */
        routine  = (char **) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);


        kfree(line_info.routine);
        if (*routine == NULL)
            *routine = kstrdup(" ");
        else line_info.routine = kstrdup(*routine);

        kvf_gen_deparse_kformstruct(kformstruct, &line_info);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_get_help_path
|       Purpose: Gets the value of the help path from 
|                the (-H) UIS line of a Help selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_HELPPATH ("kvf_help_path")
|        Output: calldata    - passes back string value of help_path
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_help_path(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	Line_Info    line_info;
	char         **help_path;

        /* set the pointer to be returned */
        help_path = (char **) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

	if (line_info.help_file == NULL)
	{
	    *help_path = NULL;
	    return(FALSE);
	}
	else *help_path = kstrdup(line_info.help_file);

	kvf_free_line_info_strings(&line_info);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_help_path
|       Purpose: Sets the value of the help path on 
|                the (-H) UIS line of a Help selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_HELPPATH ("kvf_help_path")
|                calldata    - provides string value of help_path
|        Output: none
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
-------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_help_path(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	char         **help_path;
	Line_Info    line_info;

	/* the value to which KVF_HELPPATH is to be set */
        help_path  = (char **) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

	kfree(line_info.help_file);
	line_info.help_file = kstrdup(*help_path);

	kvf_gen_deparse_kformstruct(kformstruct, &line_info);

	return(TRUE);
}


/*-----------------------------------------------------------
|
|       Routine: kvf_get_me_subforms
|       Purpose: Gets the TRUE/FALSE value of whether or not multiple
|                subforms on a master are supposed to be mutually 
|                exclusive (ie, only one allowed to map at any one time)
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_ME_SUBFORMS ("kvf_me_subforms")
|        Output: calldata    - passes back boolean value of me_subform:
|                              TRUE  - subforms are mutually exclusive
|                              FALSE - multiple subforms may be mapped
|          Date: April 7, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_me_subforms(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	Line_Info    line_info;
	int          *me_subforms;

        /* set the pointer to be returned */
        me_subforms = (int *) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

	*me_subforms = line_info.logical_val;

	kvf_free_line_info_strings(&line_info);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_me_subforms
|       Purpose: Sets the TRUE/FALSE value of whether or not multiple
|                subforms on a master are supposed to be mutually 
|                exclusive (ie, only one allowed to map at any one time)
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_ME_SUBFORMS ("kvf_me_subforms")
|                calldata    - passes in integer value of scope: 
|                              TRUE  - subforms are mutually exclusive
|                              FALSE - multiple subforms may be mapped
|        Output: none
|          Date: April 7, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
-------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_me_subforms(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int          *me_subforms;
	Line_Info    line_info;

	/* the value to which KVF_ME_SUBFORMS is to be set */
        me_subforms  = (int *) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

	line_info.logical_val = *me_subforms;

	kvf_gen_deparse_kformstruct(kformstruct, &line_info);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_list_add
|       Purpose: Adds an item to a list of choices on the
|                (-l), (-x), (-z), or (-c) UIS line of a
|                logical, list, displaylist, or cycle selection
|                This is an action attribute routine.
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_LIST_ADD ("kvf_list_add")
|                calldata    - provides label for new item in list
|                              (index is determined by length of list;
|                               new item is always added at the end>)
|        Output: none
|
|          Date: Jan 31, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_list_add(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
        int          indx;
	char         **new_choice;
        Line_Info    line_info;
	kselection   *selection;

	new_choice = (char **) calldata;
	
        kvf_clear_line_info(&line_info);
        selection = kformstruct->Selptr;
	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

	/*
	 *  update number of choices; save choice label
	 */
	indx = line_info.list_num;
	(line_info.list_num)++;

	line_info.val_labels = (char **) 
			       krealloc(line_info.val_labels, 
					line_info.list_num*sizeof(char *));
	if (line_info.val_labels == NULL)
	{
	    kerror("kforms", "kvf_change_listadd",
		   "Unable to allocate string(s) for new item label(s)");
	    return(FALSE);
	}
	line_info.val_labelnum = line_info.list_num;
	if (kstrlen(*new_choice) > 0)
	    line_info.val_labels[indx] = kstrdup(*new_choice);
	else line_info.val_labels[indx] = NULL;

	kvf_gen_deparse(&line_info, &selection->line);
	kvf_free_line_info_strings(&line_info);
	return(TRUE);
}


/*-----------------------------------------------------------
|
|       Routine: kvf_set_list_delete
|       Purpose: Deletes an item from the list of choices on the
|                (-l), (-x), (-z), or (-c) UIS line of a
|                logical, list, displaylist, or cycle selection
|                This is an action attribute routine.
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_LIST_DELETE ("kvf_list_delete")
|                calldata    - provides index of choice to be deleted
|        Output: none
|
|          Date: Jan 31, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_list_delete(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
        kselection   *selection;
        Line_Info    line_info;
        char         **tmp_labels;
        int          i, j = 0, reset = FALSE, old_index = -1; 
	char         **old_choice;

	old_choice = (char **) calldata;

        selection = kformstruct->Selptr;
	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);


	if (line_info.list_num == 0)
	{
	    kerror(NULL, "kvf_change_listdel",
		   "list selection has no choices; cannot delete choice '%s'",
		    *old_choice); 
	    return(FALSE);
	}
	else if (line_info.list_num == 1)
	{
	    if ((atoi(*old_choice) != line_info.int_val) &&
	        (kstrcmp(*old_choice, line_info.val_labels[0]) != 0))
	    {
	        kerror(NULL, "kvf_change_listdel",
	  	      "list selection has only one choice, '%s'; cannot delete choice '%s'", line_info.val_labels[0], *old_choice); 
	        return(FALSE);
	    }
	    kfree(line_info.string_val);
	    kfree(line_info.val_labels[0]);
	    kfree(line_info.val_labels);
	    old_index = 0;
	    line_info.list_num = 0;
	    line_info.list_val = line_info.int_val;
	    line_info.val_labelnum = 0;
	}
	else
	{
	    for (i = 0; i < line_info.list_num; i++)
	    {
	        if (atoi(*old_choice) == (line_info.int_val + i))
		    old_index = i;
		else if (kstrcmp(*old_choice, line_info.val_labels[i]) == 0)
		    old_index = i;
	    }
	    if (old_index == -1)
	    {
	        kerror(NULL, "kvf_change_listdel",
	  	      "Cannot find choice '%s' in list selection '%s' for deletion", *old_choice, line_info.title); 
	        return(FALSE);
	    }

	    tmp_labels = (char **) kcalloc(1, (line_info.list_num - 1) *
					   sizeof(char *));
	    if (tmp_labels == NULL)
	    {
		kerror("kforms", "kvf_change_listdel",
		       "Unable to allocate temp storage for labels");
		return(FALSE);
	    }
	    for (i = 0; i < line_info.list_num; i++)
	    {
	        if (i != old_index) 
		    tmp_labels[j++] = kstrdup(line_info.val_labels[i]);
		kfree(line_info.val_labels[i]);
	    }
	    kfree(line_info.val_labels);
	    line_info.val_labels = tmp_labels;
	    line_info.list_num--;
	    line_info.val_labelnum = line_info.list_num;
	    if (line_info.list_val > line_info.int_val+line_info.list_num - 1)
	    {
		line_info.list_val = line_info.int_val + line_info.list_num - 1;
	        reset = TRUE;
	    }
	}

	kvf_gen_deparse(&line_info, &selection->line);

	if ((line_info.list_num > 0) && (reset))
	    kvf_set_attribute(kformstruct, KVF_LIST_VAL, 
	    	 	      line_info.int_val + line_info.list_num - 1);

	kvf_free_line_info_strings(&line_info);
	return(TRUE);
}


/*-----------------------------------------------------------
|
|       Routine: kvf_set_list_deleteall
|       Purpose: Deletes all items from the list of choices on the
|                (-l), (-x), (-z), or (-c) UIS line of a
|                logical, list, displaylist, or cycle selection
|                This is an action attribute routine.
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_LIST_DELETEALL ("kvf_list_deleteall")
|                calldata    - provides integer value of TRUE (ignored)
|        Output: none
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_list_deleteall(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	Line_Info    line_info;
	kselection   *selection;

        kvf_clear_line_info(&line_info);
        selection = kformstruct->Selptr;
	kvf_gen_parse(selection->line, &line_info);

	kfree(line_info.string_val);

	line_info.list_num = 0;
	line_info.list_val = line_info.int_val;
	line_info.val_labelnum = 0;

        kvf_gen_deparse(&line_info, &selection->line);
	kvf_free_line_info_strings(&line_info);

	return(TRUE);
}


/*-----------------------------------------------------------
|
|       Routine: kvf_get_val_labels
|       Purpose: Gets the val_labels array of strings from the
|                (-l), (-x), (-z), or (-c) UIS line of a
|                logical, list, displaylist, or cycle selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_VAL_LABELS ("kvf_val_labels")
|        Output: calldata    - passes back array of strings making up list
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_val_labels(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	Line_Info line_info;
	char      ***val_labels;

        /* the value to which KVF_VAL_LABELS is to be set */
        val_labels = (char ***) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

	/* DON'T call kvf_free_line_info_strings after this...
	   it is the only get routine that doesn't calloc */
	*val_labels = line_info.val_labels;

	kfree(line_info.title);
        kfree(line_info.description);
        kfree(line_info.variable);
        kfree(line_info.routine);
        kfree(line_info.help_file);
        kfree(line_info.filename);
        kfree(line_info.string_val);
        kfree(line_info.string_def);
        kfree(line_info.file_def);
        kfree(line_info.literal);
        kfree(line_info.lib_call);

	return(TRUE);
}
