/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>		Attribute Actions For Double Selections       <<<<
   >>>>                                                       <<<<
   >>>>  Private:                                             <<<<
   >>>>	     		kvf_get_double_gt0_sel()	      <<<<
   >>>>	     		kvf_set_double_gt0_sel()	      <<<<
   >>>>	     		kvf_get_double_ge0_sel()	      <<<<
   >>>>	     		kvf_set_double_ge0_sel()	      <<<<
   >>>>	     		kvf_get_double_lt0_sel()	      <<<<
   >>>>	     		kvf_set_double_lt0_sel()	      <<<<
   >>>>	     		kvf_get_double_le0_sel()	      <<<<
   >>>>	     		kvf_set_double_le0_sel()	      <<<<
   >>>>	     		kvf_get_double_upper()		      <<<<
   >>>>	     		kvf_set_double_upper()		      <<<<
   >>>>	     		kvf_get_double_lower()		      <<<<
   >>>>	     		kvf_set_double_lower()		      <<<<
   >>>>	     		kvf_get_double_def()		      <<<<
   >>>>	     		kvf_set_double_def()		      <<<<
   >>>>	     		kvf_get_double_val()		      <<<<
   >>>>	     		kvf_set_double_val()		      <<<<
   >>>>	     		kvf_get_precision()		      <<<<
   >>>>	     		kvf_set_precision()		      <<<<
   >>>>	     		kvf_change_doublebounds()	      <<<<
   >>>>                                                       <<<<
   >>>>   Static:                                             <<<<
   >>>>   Public:                                             <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<  */

#include "internals.h"


/*-----------------------------------------------------------
|
|       Routine: kvf_get_double_gt0_sel
|       Purpose: Gets the boolean TRUE or FALSE depending on whether
|                or not the bounds of an double selection are > 0
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_DOUBLE_GT0_SEL ("kvf_double_gt0_sel")
|        Output: calldata    - passes back TRUE if bounds > 0 set
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_double_gt0_sel(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int          *gt0;
	Line_Info    line_info;

        /* set the pointer to be returned */
        gt0 = (int *) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

	if ((line_info.upper_double == line_info.lower_double) &&
	    (line_info.lower_double == 2.0))
  	    *gt0 = TRUE;
	else *gt0 = FALSE;

	kvf_free_line_info_strings(&line_info);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_double_gt0_sel
|       Purpose: Sets the bounds of an double selection to > 0
|                according to the value of the "double_gt0" 
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_DOUBLE_GT0_SEL ("kvf_double_gt0_sel")
|                calldata    - provides value of TRUE if bounds > 0 are set
|        Output: none
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_double_gt0_sel(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int          *gt0; 
	double       double_def;

	/* value to which KVF_DOUBLE_GT0_SEL is to be set */
        gt0  = (int *) calldata;

	if (*gt0 != TRUE) return(FALSE);

	if (!(kvf_change_doublebounds(kformstruct, "gt0")))
	    return(FALSE);

	kvf_get_attribute(kformstruct, KVF_DOUBLE_DEF, &double_def);
	if (double_def <= 0)
	    kvf_set_attribute(kformstruct, KVF_DOUBLE_DEF, 1.0);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_get_double_ge0_sel
|       Purpose: Gets the boolean TRUE or FALSE depending on whether
|                or not the bounds of an double selection are >= 0
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_DOUBLE_GE0_SEL ("kvf_double_ge0_sel")
|        Output: calldata    - passes back TRUE if bounds are set >= 0
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_double_ge0_sel(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int          *ge0;
	Line_Info    line_info;

        /* set the pointer to be returned */
        ge0 = (int *) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

	if ((line_info.lower_double == line_info.upper_double) &&
	    (line_info.lower_double == 1))
  	    *ge0 = TRUE;
	else *ge0 = FALSE;

	kvf_free_line_info_strings(&line_info);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_double_ge0_sel
|       Purpose: Sets the bounds of an double selection to >= 0
|                according to the value of the "double_ge0" 
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_DOUBLE_GE0_SEL ("kvf_double_ge0_sel")
|                calldata    - provides boolean value of TRUE if bounds are >= 0
|        Output: none
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_double_ge0_sel(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int          *ge0; 
	double       double_def;

	/* value to which KVF_DOUBLE_GE0_SEL is to be set */
        ge0  = (int *) calldata;

	if (*ge0 != TRUE) return(FALSE);

	if (!(kvf_change_doublebounds(kformstruct, "ge0")))
	    return(FALSE);

	kvf_get_attribute(kformstruct, KVF_DOUBLE_DEF, &double_def);
	if (double_def < 0)
	    kvf_set_attribute(kformstruct, KVF_DOUBLE_DEF, 0.0);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_get_double_lt0_sel
|       Purpose: Gets the boolean TRUE or FALSE depending on whether
|                or not the bounds of an double selection are < 0
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_DOUBLE_LT0_SEL ("kvf_double_lt0_sel")
|        Output: calldata    - passes back TRUE if bounds are set to < 0
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_double_lt0_sel(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int          *lt0;
	Line_Info    line_info;

        /* set the pointer to be returned */
        lt0 = (int *) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

	if ((line_info.lower_double == line_info.lower_double) &&
	    (line_info.lower_double == -2.0))
  	    *lt0 = TRUE;
	else *lt0 = FALSE;

	kvf_free_line_info_strings(&line_info);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_double_lt0_sel
|       Purpose: Sets the bounds of an double selection to < 0
|                according to the value of the "double_lt0" 
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_DOUBLE_LT0_SEL ("kvf_double_lt0_sel")
|                calldata    - provides boolean value of TRUE if bounds are < 0
|        Output: none
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_double_lt0_sel(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int    *lt0; 
	double double_def;

	/* value to which KVF_DOUBLE_LT0_SEL is to be set */
        lt0  = (int *) calldata;

	if (*lt0 != TRUE) return(FALSE);
 
	if (!(kvf_change_doublebounds(kformstruct, "lt0")))
	    return(FALSE);

	kvf_get_attribute(kformstruct, KVF_DOUBLE_DEF, &double_def);
	if (double_def >= 0)
	    kvf_set_attribute(kformstruct, KVF_DOUBLE_DEF, -1.0);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_get_double_le0_sel
|       Purpose: Gets the boolean TRUE or FALSE depending on whether
|                or not the bounds of an double selection are <= 0
|
|         Input: kformstruct    - backplane of the selection
|                attribute - KVF_DOUBLE_LE0_SEL ("kvf_double_le0_sel")
|        Output: calldata  - passes back TRUE if bounds are set to <= 0
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_double_le0_sel(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int       *le0;
	Line_Info line_info;

        /* set the pointer to be returned */
        le0 = (int *) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

	if ((line_info.lower_double == line_info.lower_double) &&
	    (line_info.lower_double == -1.0))
  	    *le0 = TRUE;
	else *le0 = FALSE;

	kvf_free_line_info_strings(&line_info);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_double_le0_sel
|       Purpose: Sets the bounds of an double selection to <= 0
|                according to the value of the "double_le0" 
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_DOUBLE_LE0_SEL ("kvf_double_le0_sel")
|                calldata    - provides boolean value of TRUE if bounds are <= 0
|        Output: none
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_double_le0_sel(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int    *le0; 
	double double_def;

	/* value to which KVF_DOUBLE_LE0_SEL is to be set */
        le0  = (int *) calldata;

	if (*le0 != TRUE) return(FALSE);

	if (!(kvf_change_doublebounds(kformstruct, "le0")))
	    return(FALSE);

	kvf_get_attribute(kformstruct, KVF_DOUBLE_DEF, &double_def);
	if (double_def > 0)
	    kvf_set_attribute(kformstruct, KVF_DOUBLE_DEF, 0.0);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_get_double_upper
|       Purpose: Gets the value of the upper bound for an double,
|                assuming that "kvf_double_upper_sel" is TRUE
|
|         Input: kformstruct    - backplane of the selection
|                attribute - KVF_DOUBLE_UPPER ("kvf_double_upper")
|        Output: calldata  - passes back double upper bound value
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_double_upper(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	double    *upper;
	Line_Info line_info;

        /* set the pointer to be returned */
        upper = (double *) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

  	*upper = line_info.upper_double;

	kvf_free_line_info_strings(&line_info);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_double_upper
|       Purpose: Sets the value of the upper bound for an double,
|                assuming that "kvf_double_upper_sel" is TRUE
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_DOUBLE_UPPER ("kvf_double_upper")
|                calldata    - provides double upper bound value
|        Output: none
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_double_upper(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int     prec;
	double  *upper; 
	double  double_def;
	char    temp[KLENGTH];
	char    scale[KLENGTH];

	/* value to which KVF_DOUBLE_UPPER is to be set */
        upper  = (double *) calldata;

	kvf_get_attribute(kformstruct, KVF_DOUBLE_PREC, &prec);
	if (prec == 0)
	    sprintf(scale, "upper = %%g");
	else if (prec == -1)
	    sprintf(scale, "upper = %%f");
	else sprintf(scale, "upper = %%.%df", prec);
	sprintf(temp, scale, *upper);

	if (!(kvf_change_doublebounds(kformstruct, temp)))
	    return(FALSE);

	if (!(kvf_get_attribute(kformstruct, KVF_DOUBLE_DEF, &double_def)))
            return(FALSE);

	if (double_def > *upper)
	{
	     double_def = *upper;
	     if (!(kvf_set_attribute(kformstruct, KVF_DOUBLE_DEF, double_def)))
            	return(FALSE);
	}
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_get_double_lower
|       Purpose: Gets the value of the lower bound for an double,
|                assuming that "kvf_double_lower_sel" is TRUE
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_DOUBLE_LOWER ("kvf_double_lower")
|        Output: calldata    - passes back double lower bound value
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_double_lower(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	double    *lower;
	Line_Info line_info;

        /* set the pointer to be returned */
        lower = (double *) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

  	*lower = line_info.lower_double;

	kvf_free_line_info_strings(&line_info);
	return(TRUE);

}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_double_lower
|       Purpose: Sets the value of the lower bound for an double,
|                assuming that "kvf_double_lower_sel" is TRUE
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_DOUBLE_LOWER ("kvf_double_lower")
|                calldata    - provides double lower bound value
|        Output: none
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_double_lower(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int    prec;
	double *lower; 
	double double_def;
	char   temp[KLENGTH];
	char   scale[KLENGTH];

	/* value to which KVF_DOUBLE_GE0_SEL is to be set */
        lower  = (double *) calldata;

	kvf_get_attribute(kformstruct, KVF_DOUBLE_PREC, &prec);
	if (prec == 0)
	    sprintf(scale, "lower = %%g");
	else if (prec == -1)
	    sprintf(scale, "lower = %%f");
	else sprintf(scale, "lower = %%.%df", prec);
	sprintf(temp, scale, *lower);

	if (!(kvf_change_doublebounds(kformstruct, temp)))
	    return(FALSE);

	if (!(kvf_get_attribute(kformstruct, KVF_DOUBLE_DEF, &double_def)))
	    return(FALSE);

	if (double_def < *lower)
	{
	     double_def = *lower;
	     if (!(kvf_set_attribute(kformstruct, KVF_DOUBLE_DEF, double_def)))
            	return(FALSE);
	}
	return(TRUE);
}


/*-----------------------------------------------------------
|
|       Routine: kvf_get_double_def
|       Purpose: Gets the value of the default for an double
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_DOUBLE_DEF ("kvf_double_def")
|        Output: calldata    - passes back double default value
|
|          Date: March 15, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_double_def(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	double       *double_def;
	Line_Info    line_info;

        /* set the pointer to be returned */
        double_def = (double *) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

  	*double_def = line_info.double_def;

	kvf_free_line_info_strings(&line_info);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_double_def
|       Purpose: Sets the value of the default for an double
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_DOUBLE_DEF ("kvf_double_def")
|                calldata    - provides double default value
|        Output: none
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_double_def(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	char       temp[KLENGTH];
	double     *double_def;
	Line_Info  line_info;
	kselection *selection;

	/* value to which KVF_DOUBLE_DEF is to be set */
        double_def  = (double *) calldata;

	if (!(kvf_set_attribute(kformstruct, KVF_DOUBLE_VAL, *double_def)))
	    return(FALSE);

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

	line_info.double_def = *double_def;
	kvf_gen_deparse_kformstruct(kformstruct, &line_info);

	if (!(kvf_set_attribute(kformstruct, KVF_DOUBLE_VAL, *double_def)))
            return(FALSE);

	selection = kformstruct->Selptr;
	if (selection->back_toggle != NULL)
	{
	    sprintf(temp, "%f", *double_def);
	    kvf_set_attribute(kformstruct, KVF_TITLE, temp);
	}

	return(TRUE);
}


/*-----------------------------------------------------------
|
|       Routine: kvf_get_double_val
|       Purpose: Gets the value of the double
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_DOUBLE_VAL ("kvf_double_val")
|        Output: calldata    - passes back double value
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_double_val(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	double    *double_val;
	Line_Info line_info;

        /* set the pointer to be returned */
        double_val = (double *) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

  	*double_val = line_info.double_val;

	kvf_free_line_info_strings(&line_info);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_double_val
|       Purpose: Sets the value of an double
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_DOUBLE_VAL ("kvf_double_val")
|                calldata    - provides double value
|        Output: none
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_double_val(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	double     *double_val;
	Line_Info  line_info;

	/* value to which KVF_DOUBLE_VAL is to be set */
        double_val  = (double *) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);


        if (!(kvf_check_double(*double_val, line_info.upper_double,
                               line_info.lower_double, line_info.variable,
			line_info.title)))
        {
            kvf_free_line_info_strings(&line_info);
            return(FALSE);
        }
	line_info.double_val = *double_val;

	kvf_gen_deparse_kformstruct(kformstruct, &line_info);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_get_precision
|       Purpose: Gets the number of decimal places for a float or double
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_DOUBLE_PREC ("kvf_double_prec") or 
|                              KVF_FLOAT_PREC  ("kvf_float_prec")
|        Output: calldata    - passes back int value giving # decimal places
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_precision(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	Line_Info line_info;
	int       *precision;

        /* set the pointer to be returned */
        precision = (int *) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

  	*precision = line_info.precision;

	kvf_free_line_info_strings(&line_info);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_precision
|       Purpose: Sets the number of decimal places for a float or double
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_DOUBLE_PREC ("kvf_double_prec") or 
|                              KVF_FLOAT_PREC  ("kvf_float_prec")
|                calldata  - provides number of decimal places
|        Output: none
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_precision(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int       *precision; 
	Line_Info line_info;

	/* value to which KVF_DOUBLE_PREC or KVF_FLOAT_PREC is to be set */
        precision  = (int *) calldata;

	if ((kstrcmp(attribute, KVF_DOUBLE_PREC) == 0) && (*precision > 14))
	{
	    kerror("kforms", "kvf_set_precision",
		   "max precision for a double selection is 14 decimal places");
	    return(FALSE);
	}
	else if ((kstrcmp(attribute, KVF_FLOAT_PREC) == 0) && 
		 (*precision > 7))
	{
	    kerror("kforms", "kvf_set_precision",
		   "max precision for a float selection is 7 decimal places");
	    return(FALSE);
	}
	else if (*precision < -1)
        {
            kerror("kforms", "kvf_set_precision",
                   "precision may not be set to a negative number");
            return(FALSE);
        }

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

	line_info.precision = *precision;
	kvf_gen_deparse_kformstruct(kformstruct, &line_info);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: kvf_change_doublebounds
|
|       Purpose: Changes the bounds for a double selection
|
|         Input: kformstruct - generic kform_struct pointer
|		 string    - string rep of double bounds, may be one of:
|			     "gt0"        : bounds >  0
|			     "ge0"        : bounds >= 0
|			     "lt0"        : bounds <  0
|			     "le0"        : bounds <= 0
|                            "lower = %f" : lower bound 
|                            "upper = %f" : upper bound 
|			     "nobounds"   : no bounds 
|
|        Output: Returns TRUE on success, FALSE on Failure
|          Date: March 20, 1994 
|    Written By: Danielle Argiro
| Modifications:
|
-------------------------------------------------------------*/

int kvf_change_doublebounds(
   kform_struct *kformstruct,
   char         *string)
{
	kselection *selection;
	Line_Info  line_info;
	double     lower, upper;
	int        status, bounds = FALSE;

	kvf_clear_line_info(&line_info);
	selection = kformstruct->Selptr;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

	if (kstrcmp(string, "gt0") == 0)
	{
	   line_info.lower_double = 2.0;
	   line_info.upper_double = 2.0;
	}
	else if (kstrcmp(string, "ge0") == 0)
        {
           line_info.lower_double = 1.0;
           line_info.upper_double = 1.0;
        }
	else if (kstrcmp(string, "lt0") == 0)
        {
           line_info.lower_double = -2.0;
           line_info.upper_double = -2.0;
        }
	else if (kstrcmp(string, "le0") == 0)
        {
           line_info.lower_double = -1.0;
           line_info.upper_double = -1.0;
        }
	else if (kstrcmp(string, "nobounds") == 0)
        {
           line_info.lower_double = 0.0;
           line_info.upper_double = 0.0;
        }
	else if (kstrcmp(string, "bounds") == 0)
        {
	   if (line_info.lower_double == line_info.upper_double)
	   {
               line_info.lower_double = -100.0;
               line_info.upper_double =  100.0;
	   }
	   bounds = TRUE;
        }
	else 
	{
	   status = ksscanf(string, "lower = %lg", &lower);
	   if (status != 1)
	   {
	       status = ksscanf(string, "upper = %lg", &upper);
	       if (status != 1)
	       {
	 	   kerror("kforms", "change_double_bounds",
		          "Invalid string provided for KVF_DOUBLEBOUNDS_CHNG");
		   return(FALSE);
	       }
	       else line_info.upper_double = upper;
	   }
	   else line_info.lower_double = lower;

           if (line_info.lower_double != line_info.upper_double)
	       bounds = TRUE;
	}

	if (line_info.upper_double < line_info.lower_double)
	{
	    kerror("kforms", "change_double_bounds",
	  	   "Attempt to set double upper bound < lower bound");
	    return(FALSE);
	}
	kvf_gen_deparse(&line_info, &selection->line);

	if (bounds)
	{
	    if (line_info.double_val < line_info.lower_double)
	        line_info.double_val = line_info.lower_double;

    	    if (line_info.double_val > line_info.upper_double)
	        line_info.double_val = line_info.upper_double;

	    kvf_set_attribute(kformstruct, KVF_DOUBLE_DEF,
			      line_info.double_val);
	}

	kvf_free_line_info_strings(&line_info);

	return(TRUE);
}

