 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>      Various Private Print Routines For the kexpr Library
   >>>>
   >>>>  Private:
   >>>>		kexpr_add_number()
   >>>>		kexpr_add_operator()
   >>>>		kexpr_add_instruction()
   >>>>		kexpr_add_string()
   >>>>		kexpr_add_symlist()
   >>>>		kexpr_copy_symlist()
   >>>>		kexpr_error()
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"
#include "y.tab.h"


/*-----------------------------------------------------------
|
|  Routine Name: kexpr_add_number()
|
|       Purpose: kexpr_add_number() is called when we want to add a
|		 number to the symbol list.  The current symbol
|		 list is passed in and the number is then tacked
|		 onto the end of the symbol list.  If the
|		 symbol list is currently NULL then we will
|		 create a new list and add the number to it.
|
|        Input:  symlist  - the current symbol list
|		 number   - the number to be added to the list
|
|       Output:  returns the new symbol list.
|
|    Written By: Mark Young  
|          Date: Thu Jun 25 1992
| Modifications:
|
|----------------------------------------------------------*/

SymbolList *kexpr_add_number(
   SymbolList *symlist,
   double     number)
{
	Symbol	   *symbol;
	SymbolList *list, *entry;
	char	   error[KLENGTH];


	/*
	 *  Allocate the entry for our new number
	 */
	if ((entry = (SymbolList *) kmalloc(sizeof(SymbolList))) == NULL)
	{
	   (void) sprintf(error, "Error! Ran out of memory trying to allocate \
reference list for number '%g'", number);
	   kexpr_error(error);
	}

	/*
	 *  Allocate the symbol to store our number
	 */
	if ((symbol = (Symbol *) kcalloc(1, sizeof(Symbol))) == NULL)
	{
	   (void) sprintf(error, "Error! Ran out of memory trying to allocate \
symbol for number '%g'", number); kfree(entry);
	   kexpr_error(error);
	}
	symbol->type  = NUMBER;
	symbol->Value = number;
	entry->symbol = symbol;
	entry->next   = NULL;

	/*
	 *  If the current symlist is NULL then return entry as our symlist.
	 */
	if (symlist == NULL)
	{
	   return(entry);
	}

	/*
	 *  The current symbol list is not empty so add the entry to the
	 *  end of the current list.
	 */
	list = symlist;
	while (list->next != NULL)
	   list = list->next;

	list->next = entry;
	return(symlist);
}

/*-----------------------------------------------------------
|
|  Routine Name: kexpr_add_operator()
|
|       Purpose: kexpr_add_operator() is called when we want to add a
|		 operator to the symbol list.  The current symbol
|		 list is passed in and the operator is then tacked
|		 onto the end of the symbol list.  If the
|		 symbol list is currently NULL then we will
|		 create a new list and add the operator to it.
|
|        Input:  symlist  - the current symbol list
|		 operator - the operator to be added to the list
|		 name     - the string that represents that operator
|		 type     - the type of operator, can be OPERATOR or
|			    OPERATOR.  The operator type takes two
|			    arguments, but the unary operator takes one
|			    argument.
|
|       Output:  returns the new symbol list.
|
|    Written By: Mark Young  
|          Date: Thu Jun 25 1992
| Modifications:
|
|----------------------------------------------------------*/

SymbolList *kexpr_add_operator(
   SymbolList *symlist,
   int	  operator,
   char	  *name,
   int    type)
{
	Symbol	   *symbol;
	SymbolList *list, *entry;
	char	   error[KLENGTH];


	/*
	 *  Allocate the entry for our new operator
	 */
	if ((entry = (SymbolList *) kmalloc(sizeof(SymbolList))) == NULL)
	{
	   (void) sprintf(error, "Error! Ran out of memory trying to allocate \
reference list for operator '%c' ", operator);
	   kexpr_error(error);
	}

	/*
	 *  Allocate the symbol to store our operator
	 */
	if ((symbol = (Symbol *) kcalloc(1,sizeof(Symbol))) == NULL)
	{
	   (void) sprintf(error, "Error! Ran out of memory trying to allocate \
symbol for operator '%c' ", operator);
	   kexpr_error(error);
	}
	symbol->type     = type;
	symbol->Operator = operator;
	symbol->name	 = kstring_copy(name, NULL);
	entry->symbol    = symbol;
	entry->next	 = NULL;

	/*
	 *  If the current symlist is NULL then return entry as our symlist.
	 */
	if (symlist == NULL)
	{
	   return(entry);
	}

	/*
	 *  The current symbol list is not empty so add the entry to the
	 *  end of the current list.
	 */
	list = symlist;
	while (list->next != NULL)
	   list = list->next;

	list->next = entry;
	return(symlist);
}

/*-----------------------------------------------------------
|
|  Routine Name: kexpr_add_instruction()
|
|       Purpose: kexpr_add_instruction() is called when we want to add a
|		 instruction to the symbol list.  The current symbol
|		 list is passed in and the instruction is then tacked
|		 onto the end of the symbol list.  If the
|		 symbol list is currently NULL then we will
|		 create a new list and add the instruction to it.
|
|        Input:  symlist     - the current symbol list
|		 instruction - the operator to be added to the list
|		 name     - the string that represents that instruction
|		 num      - the number of expression used by the
|			    instruction.
|
|       Output:  returns the new symbol list.
|
|    Written By: Mark Young  
|          Date: Thu Jun 25 1992
| Modifications:
|
|----------------------------------------------------------*/

SymbolList *kexpr_add_instruction(
    SymbolList *symlist,
    int	   instruction,
    char   *name,
    int    num)
{
	Symbol	   *symbol;
	SymbolList *list, *entry;
	char	   error[KLENGTH];


	/*
	 *  Allocate the entry for our new operator
	 */
	if ((entry = (SymbolList *) kmalloc(sizeof(SymbolList))) == NULL)
	{
	   (void) sprintf(error, "Error! Ran out of memory trying to allocate \
reference list for instruction '%s' ", name);
	   kexpr_error(error);
	}

	/*
	 *  Allocate the symbol to store our operator
	 */
	if ((symbol = (Symbol *) kcalloc(1,sizeof(Symbol))) == NULL)
	{
	   (void) sprintf(error, "Error! Ran out of memory trying to allocate \
symbol for instruction '%s' ", name);
	   kexpr_error(error);
	}
	symbol->num	    = num;
	symbol->type	    = INSTRUCTION;
	symbol->Instruction = instruction;
	symbol->name	    = kstring_copy(name, NULL);
	entry->symbol       = symbol;
	entry->next	    = NULL;

	/*
	 *  If the current symlist is NULL then return entry as our symlist.
	 */
	if (symlist == NULL)
	{
	   return(entry);
	}

	/*
	 *  The current symbol list is not empty so add the entry to the
	 *  end of the current list.
	 */
	list = symlist;
	while (list->next != NULL)
	   list = list->next;

	list->next = entry;
	return(symlist);
}

/*-----------------------------------------------------------
|
|  Routine Name: kexpr_add_string()
|
|       Purpose: kexpr_add_string() is called when we want to add a
|		 string to the symbol list.  The current symbol
|		 list is passed in and the string is then tacked
|		 onto the end of the symbol list.  If the
|		 symbol list is currently NULL then we will
|		 create a new list and add the string to it.
|
|        Input:  symlist  - the current symbol list
|		 string   - the string to be added to the list
|
|       Output:  returns the new symbol list.
|
|    Written By: Mark Young  
|          Date: Thu Jun 25 1992
| Modifications:
|
|----------------------------------------------------------*/

SymbolList *kexpr_add_string(
   SymbolList *symlist,
   char	   *string)
{
	Symbol	   *symbol;
	SymbolList *list, *entry;
	char	   error[KLENGTH];


	/*
	 *  Allocate the entry for our new string
	 */
	if ((entry = (SymbolList *) kmalloc(sizeof(SymbolList))) == NULL)
	{
	   (void) sprintf(error, "Error! Ran out of memory trying to allocate \
reference list for string '%s' ", string);
	   kexpr_error(error);
	}

	/*
	 *  Allocate the symbol to store our string
	 */
	if ((symbol = (Symbol *) kcalloc(1,sizeof(Symbol))) == NULL)
	{
	   (void) sprintf(error, "Error! Ran out of memory trying to allocate \
symbol for string '%s' ", string);
	   kexpr_error(error);
	}
	symbol->type     = STRING;
	symbol->String   = kstring_copy(string, NULL);
	entry->symbol    = symbol;
	entry->next	 = NULL;

	/*
	 *  If the current symlist is NULL then return entry as our symlist.
	 */
	if (symlist == NULL)
	{
	   return(entry);
	}

	/*
	 *  The current symbol list is not empty so add the entry to the
	 *  end of the current list.
	 */
	list = symlist;
	while (list->next != NULL)
	   list = list->next;

	list->next = entry;
	return(symlist);
}

/*-----------------------------------------------------------
|
|  Routine Name: kexpr_add_symlist()
|
|       Purpose: kexpr_add_string() is called when we want to add a
|		 string to the symbol list.  The current symbol
|		 list is passed in and the symlist is then tacked
|		 onto the end of the symbol list.  If the
|		 symbol list is currently NULL then we will
|		 return the other list.
|
|        Input:  symlist  - the current symbol list
|		 endlist  - the symlist to be added to the current list
|
|       Output:  returns the new symbol list.
|
|    Written By: Mark Young  
|          Date: Thu Jun 25 1992
| Modifications:
|
|----------------------------------------------------------*/

SymbolList *kexpr_add_symlist(
   SymbolList *symlist,
   SymbolList *endlist)
{
	SymbolList *list;


	/*
	 *  If the current symlist is NULL then the endlist will become
	 *  the beginning of the symbol list, so we just return it.
	 */
	if (symlist == NULL)
	   return(endlist);

	/*
	 *  The current symbol list is not empty so add the endlist to the
	 *  end of the current list.
	 */
	list = symlist;
	while (list->next != NULL)
	   list = list->next;

	list->next = endlist;
	return(symlist);
}

/*-----------------------------------------------------------
|
|  Routine Name: kexpr_copy_symlist()
|
|       Purpose: kexpr_copy_symlist() is called when we want to copy
|		 the contents of a symbol list.
|
|        Input:  symlist - the current symbol to be copied
|
|       Output:  None.
|
|    Written By: Mark Young  
|          Date: Thu Jun 25 1992
| Modifications:
|
|----------------------------------------------------------*/

SymbolList *kexpr_copy_symlist(
    SymbolList   *symlist)
{
	Symbol	     *symbol;
	SymbolList   *list, *temp;
	char	     error[KLENGTH];


	/*
	 *  If the current symlist is NULL then just return
	 */
	if (symlist == NULL)
	   return(NULL);

	/*
	 *  copy all the resources that are allocated with the symbol,
	 *  ie). the symlist and the string (if the symbol is of type
	 *  string).
	 */
	list = NULL;
	while (symlist != NULL)
	{
	   symbol = symlist->symbol;
	   switch (symbol->type)
	   {
	      case STRING:
		   list = kexpr_add_string(list, symbol->String);
		   break;

	      case NUMBER:
		   list = kexpr_add_number(list, symbol->Value);
		   break;

	      case OPERATOR:
	      case UOPERATOR:
		   list = kexpr_add_operator(list, symbol->Operator, symbol->name,
				symbol->type);
		   break;

	      case VARIABLE:
	      case EXPRESSION:
	      case FUNCTION:
	      case UFUNCTION:
		   if ((temp = (SymbolList *) kmalloc(sizeof(SymbolList)))==NULL)
		   {
		      (void) sprintf(error, "Error! Ran out of memory trying \
to allocate reference list for symbol '%s'", symbol->name);
		      kexpr_error(error);
		   }
		   temp->symbol = symbol;
		   temp->next   = list;
		   list	= temp;
		   break;

	      default:
		   (void) sprintf(error, "Error! Unknown symbol type '%d' to \
be copied.\n", symbol->type);
		   kexpr_error(error);
		   break;
	   }
	   symlist = symlist->next;
	}
	return(list);
}

/*-----------------------------------------------------------
|
|  Routine Name: kexpr_error()
|
|       Purpose: called when an error has been detected on the
|		 expression string.
|
|        Input:  error  - the error that occured
|
|       Output:  formats an error message into the *kexpr_error
|		 array and long jumps back to the beginning
|		 of kexpr_eval_expression().
|
|    Written By: Mark Young  
|          Date: Thu Jun 25 1992
| Modifications:
|
|----------------------------------------------------------*/

void kexpr_error(
   char *error)
{
	int	i;
	char	temp[KLENGTH];


	if (string_array != NULL)
	{
	   ksprintf(temp, "\nfunction string is = \"%s\"\n", string_array);
	   kstrcat(temp, "                     ");
	   for (i = 0; i < (sptr - string_array); i++)
	       kstrcat(temp ," ");

	   kstrcat(temp, "^\n");
	}
	else temp[0] = '\0';

	kstrcat(temp, error);
	kstrcat(temp, "...\n");

	if (kexpr_error_string != NULL)
	   kstrcpy(kexpr_error_string, temp);
	else
	   kerror("kexpr", "kexpr_error", temp);

	/*
	 * error out to calling routine kexpr_get_function().
	 */
	error_found = TRUE;
	if (jump_dest_enabled)
	   longjmp(jump_dest, 1);
}
