/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>    Routines to generate form information extraction   <<<<
   >>>>           code for xvroutines in form_info.c          <<<<
   >>>>                                                       <<<<
   >>>>  Private:                                             <<<<
   >>>>              kgen_gui_get_forminfo()                  <<<<
   >>>>   Static:                                             <<<<
   >>>>              gen_get_subforminfo()                    <<<<
   >>>>              gen_get_paneinfo()                       <<<<
   >>>>              gen_get_selections()                     <<<<
   >>>>   Public:                                             <<<<
   >>>>                                                       <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"  

static int gen_get_subforminfo PROTO((kobject, ksubform *, char *, char *,
                                          char *, int, char ***, int *));
static int gen_get_paneinfo    PROTO((kcontrol *, char *, char *, char *,
					  char *, int, char ***, int *));
static int gen_get_selections  PROTO((kfile *, int, kselection *));

static void print_get_sel_and_optsel PROTO((kfile *, int, int, char *, char *));


/*--------------------------------------------------------------
|
|  Routine Name: kgen_gui_get_forminfo
|
|      Purpose:  This routine creates a file called "form_info.c" 
|		 in the local directory, and automatically generates in
|		 this file the routines "_xvf_get_*",  to take the
|		 UIS and pull information out of it into the
|		 FormInfo structure defined in "form_info.h"
|
|         Input: form         - pointer to the form tree
|                object       - the software object being generated.
|                program_name - name of program being generated
|		 levels       - level of extraction [-l]
|
|        Output: Returns TRUE if form_info.c was created successfully,
|                FALSE otherwise
|          Date: May 12, 1992
|    Written By: Danielle Argiro 
| Modifications:
|
-------------------------------------------------------------*/

int kgen_gui_get_forminfo(
   kform    *form,
   kobject  program,
   char     *program_name,
   int      levels)
{
        kstring    routine = "kgen_gui_get_forminfo()";
	kstring    oname   = NULL;
	char       *form_var;
	char       *subform_var;
	char       *copyright; 
	kobject    toolbox;
	char       **stat_routines;
	char       **pub_routines;
	char       temp[KLENGTH], func_name[KLENGTH], 
		   input[KLENGTH]; 
	int        i, user_refused, stat_num;
	kfile      *file;
	ksubform   *subform;
	kselection *selection;
	

	if (!kcms_get_attributes(program,
				KCMS_NAME,   &oname,
				KCMS_PARENT, &toolbox,
				KCMS_END))
	   return(FALSE);

        /*
	 *  first of all, if 'form_info.c' exists, see if they want to over-
	 *  write it.  If not, return immediately.
	 */
	file = kgen_open_file(program, "form_info.c", &user_refused);
        if (user_refused) return(TRUE);
        else if (file == NULL) 
        {
	    kerror(KCODEGEN, "kgen_gui_get_forminfo", 
		   "Could not create file 'form_info.c'");
            return(FALSE);
        }
        kfclose(file);

	/* 
	 * create & open the "subform_info" file which will have the 
	 * "form_info" structure definition 
	 */
	ksprintf(temp, "%s/form_info", kgen_tmpdirname);
	if ((file = kfopen(temp, "w"))== NULL)
	{
	    kerror(KCODEGEN, "kgen_gui_get_forminfo", 
		   "Could not create tmp file '%s/form_info'", kgen_tmpdirname);
	    return(FALSE);
	}


	/* read the StartForm definition & get FormInfo variable name */
       	form_var = ktoken_to_string(form->var_token);
        
       	/* 
	 * generate private header 
	 */
	ksprintf(func_name, "_xvf_get_%s", form_var);
	ksprintf(input, "forminfo - pointer to gui_info_struct");
	ksprintf(temp, "conductor -name %s -l %d", program_name, levels);
	kgen_privsrc_hdr(file, func_name, "Extracts information from GUI", 
			 input, "none", temp, NULL);
       
	stat_routines = (char **) kcalloc(1, sizeof(char *));
	stat_num = 0;
	

       	/* write out program program_name & argument list */
       	kfprintf(file, "void %s(\n", func_name, form_var);
        kfprintf(file, "     gui_info_struct *forminfo)\n");
       
       	/* write out beginning bracket & declarations */
       	kfprintf(file, "{\n"); 
       
	if (form->master != NULL)
        {
            subform = form->master->subform_list;
            selection = form->master->sel_list;
        }
        else
        {
            subform = form->subform;
            selection = NULL;
        }

	if (selection != NULL)
       	    kfprintf(file, "\tkselection *selection;\n");
       	kfprintf(file, "\tksubform   *subform;\n");

        if (!(gen_get_selections(file, KMASTER, selection)))
	    return(FALSE);

	while (subform != NULL)
       	{
	    if (subform->type == KUIS_SUBFORMBUTTON) 
	    {
                subform_var = ktoken_to_string(subform->var_token);
		kfprintf(file, "\tif (forminfo->%s_struct != NULL)\n\t{\n",
				   subform_var);
		kfprintf(file, "\t    subform = forminfo->%s_struct->Subformptr;\n", 
				subform_var);
		kfprintf(file, "\t    kvf_get_attribute(subform->back_kformstruct, KVF_SELECTED,\n");
                kfprintf(file, "\t                    &(forminfo->%s_selected));\n", subform_var);
	
		if (levels == 1)
		{
		    kfprintf(file, "\t    _xvf_get_%s(forminfo->%s);\n\n",
				subform_var, subform_var);
		}
		kfprintf(file, "\t}\n");

                if (!(gen_get_subforminfo(program, subform, form_var, 
				subform_var, program_name, levels, 
				&stat_routines, &stat_num)))
                    return(FALSE);

	    } 
	    subform = subform->next;
	} 

       	kfprintf(file, "\n}\n\n");

        kannounce(KCODEGEN, routine, "done generating 'form_info.c'");
       
        kfclose(file);

	/* 
	 * create & open the "form_info_hdr" file which will have the 
	 * RCS headers, copyright, file header (doing it here because
         * we don't have program_names of all routines until the end)
	 */
	ksprintf(temp, "%s/form_info_hdr", kgen_tmpdirname);
	if ((file = kfopen(temp, "w"))== NULL)
	{
	    kerror(KCODEGEN, "kgen_gui_get_forminfo", 
		   "Could not create file tmp file '%s/form_info_hdr'",
		   kgen_tmpdirname);
	    return(FALSE);
	}

	/* begin file with RCS header */
	kgen_rcs_chdr(file);

	/*
	 * generate the copyright
	 */
	if ((copyright = kcms_get_copyright(toolbox, KCMS_TB_COPYRIGHT_LONG,
					  KCMS_LANG_C)) != NULL)
	{
	   kfprintf(file, "%s\n", copyright);
	   kfree(copyright);
	}
		
	pub_routines = (char **) kcalloc(1, sizeof(char *));
        pub_routines[0] = kstrdup(func_name);
	kgen_srcfile_hdr(file, "FormInfo Extraction Routines", pub_routines, 
			 1, stat_routines, stat_num, NULL, 0);

	kfree(pub_routines[0]);
	kfree(pub_routines);
	for (i = 0; i < stat_num; i++)
	    kfree(stat_routines[i]);
	kfree(stat_routines);

	/* write to file the necessary #includes */
	kfprintf(file,"#include \"%s.h\"\n\n\n", oname);

	kfclose(file);
    
	ksprintf(temp,  "%s/form_info.c", kgen_fulltopsrc);

#if 0
	/*-- seems to be a no-op - neilb 22-nov-94 --*/
	if (kaccess(temp, R_OK) != 0) kgen_newfile = TRUE;
#endif

	ksprintf(temp,  "cat %s/form_info_hdr %s/form_info > %s/form_info.c",
		kgen_tmpdirname, kgen_tmpdirname, kgen_fulltopsrc);
	ksystem(temp);

	ksprintf(temp, "%s/subform_info", kgen_tmpdirname);
	if (kaccess(temp, R_OK) == 0)
	{
	    ksprintf(temp, "cat %s/subform_info >> %s/form_info.c",
		    kgen_tmpdirname, kgen_fulltopsrc);
	    ksystem(temp);
	}
	
        return(TRUE);

}  /* end  kgen_gui_get_forminfo */
        


/*--------------------------------------------------------------
|
|  Routine Name: gen_get_subforminfo
|
|       Purpose: Generates the code to extract the values from 
|                each subform from the UIS & put them into the 
|                appropriate place in the FormInfo structure.
|
|         Input: object        - the software object in question.
|                subform       - pointer to the subform
|                form_var      - program_name of the form variable
|                subform_var   - program_name of the subform variable
|                program_name  - program_name of program being generated
|		 levels        - level of extraction [-l]
|                stat_routines - keeps track of all routine program_names
|                stat_num      - number of private routines
|
|        Output: Returns TRUE if subform extraction code was 
|                created successfully, FALSE otherwise
|          Date: May 12, 1992
|    Written By: Danielle Argiro
| Modifications:
|
-------------------------------------------------------------*/

static int gen_get_subforminfo(
   kobject  object,
   ksubform *subform,
   char     *form_var,
   char     *subform_var,
   char     *program_name,
   int      levels,
   char     ***stat_routines,
   int      *stat_num)
        		
{
	kfile         *file;
        char          *pane_var;
	char          func_name[KLENGTH], input[KLENGTH];
        char          purpose[KLENGTH], temp[KLENGTH];
	kguide        *guide;
	kselection    *selection;
	unsigned long update_flag;

	/* 
	 * create & open the "subform_info" file which will have the 
	 * "form_info" structure definition 
	 */
	ksprintf(temp, "%s/subform_info", kgen_tmpdirname);
	if ((file = kfopen(temp, "a"))== NULL)
	{
	    kerror(KCODEGEN, "gen_get_subforminfo", 
		   "Could not create tmp file subform_info");
	    return(FALSE);
	}

	if (!kcms_get_attribute(object, KCMS_CMOBJ_UPDATE_DB, &update_flag))
	    update_flag = KCMS_UPDATE_NONE;

       	/* 
	 * generate private header 
	 */
	ksprintf(func_name, "_xvf_get_%s", subform_var);
	ksprintf(input, "subforminfo - pointer to %s_%s structure",
		form_var, subform_var);
	ksprintf(purpose, "Extracts information from %s subform", subform_var);
	ksprintf(temp, "conductor -name %s -l %d", program_name, levels);
	kgen_privsrc_hdr(file, func_name, purpose, 
			 input, "none", temp, NULL);
       
	*stat_routines = (char **) krealloc(*stat_routines,(*stat_num+1)
						      *sizeof(char *));
	(*stat_routines)[*stat_num] = kstrdup(func_name);
	(*stat_num)++;

       	/* write out program program_name & argument list */

        kfprintf(file, "void _xvf_get_%s(\n", subform_var);
        kfprintf(file, "     %s_%s *subforminfo)\n", form_var, subform_var);
       
       	/* write out beginning bracket & declarations */
       	kfprintf(file, "{\n"); 

	if (subform->guidepane != NULL)
        {
            guide = subform->guidepane->guide_list;
            selection = subform->guidepane->sel_list;
        }
        else
        {
            guide = subform->guide;
            selection = NULL;
        }

	if (selection != NULL)
       	    kfprintf(file, "\tkselection *selection;\n");
       	kfprintf(file, "\tkcontrol   *pane;\n");

        if (!(gen_get_selections(file, KGUIDEPANE, selection)))
	    return(FALSE);
	while (guide != NULL)
	{
	    if ((guide->pane != NULL) &&
		(guide->pane->sel_list != NULL) &&
		(kgen_valid_selections(guide->pane->sel_list)))
	    {
                pane_var = ktoken_to_string(guide->pane->var_token);
	        kfprintf(file, "\tif (subforminfo->%s_struct != NULL)\n\t{\n",
		         pane_var);
                kfprintf(file, "\t    pane = subforminfo->%s_struct->Controlptr;\n",
		         pane_var);
	        kfprintf(file, "\t    kvf_get_attribute(pane->back_kformstruct, KVF_SELECTED,\n");
                kfprintf(file, "\t                    &(subforminfo->%s_selected));\n", pane_var);

		if ((!kgen_selection_codegen_warranted(guide->pane->sel_list)) 
		     && (levels == 3)
		     && (update_flag & KCMS_UPDATE_NEW) != KCMS_UPDATE_NEW)
		{
		        kwarn(NULL, "gen_get_subforminfo", "Warning: using a level 3 extraction on a form containing pane '%s', having no live selections or action buttons; there will be no way to get the current information off this pane unless a level 1 or level 2 extraction is used.", pane_var);
		}
	        if ((levels == 1) || (levels == 2))
	        {
	                kfprintf(file, "\t    _xvf_get_%s(subforminfo->%s);\n",
		        	     pane_var, pane_var);
	        }
	        if (!(gen_get_paneinfo(guide->pane, form_var, subform_var, 
			           pane_var, program_name, levels, 
				   stat_routines, stat_num)))
                        return(FALSE);
                kfprintf(file, "\t}\n\n");
	    }
	    guide = guide->next;
	} 
        kfprintf(file, "\n}\n\n\n\n");
	kfclose(file);

	ksprintf(temp, "%s/pane_info", kgen_tmpdirname);
	if (kaccess(temp, R_OK) == 0)
	{
	    ksprintf(temp, "cat %s/pane_info >> %s/subform_info",
	    	     kgen_tmpdirname, kgen_tmpdirname);
            ksystem(temp);
	}

	ksprintf(temp, "%s/pane_info", kgen_tmpdirname);
	kunlink(temp);

	return(TRUE);

} /* end gen_get_subforminfo */
        		
/*--------------------------------------------------------------
|
|  Routine Name: gen_get_paneinfo
|
|       Purpose: Generates the code to extract values from a pane 
|                & put them into the appropriate place in the 
|                Form Info structure
|
|         Input: pane        - control panel struct for pane
|		 form_var    - program_name of the form variable
|                subform_var - program_name of the subform variable
|                pane_var    - program_name of the pane variable
|                program_name        - program_name of program being generated
|                levels      - levels input by user
|                stat_routines - keeps track of all routine program_names
|                stat_num    - number of private routines
|
|        Output: Returns TRUE if pane extraction code was created 
|                successfully, FALSE otherwise
|          Date: May 12, 1992
| Modifications:
|
-------------------------------------------------------------*/

static int gen_get_paneinfo(
   kcontrol *pane,
   char     *form_var,
   char     *subform_var,
   char     *pane_var,
   char     *program_name,
   int      levels,
   char     ***stat_routines,
   int      *stat_num)
{
	kfile *file;
	char   func_name[KLENGTH], input[KLENGTH];
        char   purpose[KLENGTH], temp[KLENGTH];

	/* 
	 * create & open the "pane_info" file which will have the 
	 * routines which get "pane_info" information 
	 */
	ksprintf(temp, "%s/pane_info", kgen_tmpdirname);
	if ((file = kfopen(temp, "a"))== NULL)
	{
	    kerror(KCODEGEN, "gen_get_paneinfo", 
		   "Could not create tmp file 'pane_info'");
	    return(FALSE);
	}

       	/* 
	 * generate private header 
	 */
	ksprintf(func_name, "_xvf_get_%s", pane_var);
	ksprintf(input, "paneinfo - pointer to %s_%s structure",
		subform_var, pane_var);
	ksprintf(purpose, "Extracts information from %s pane", pane_var);
	ksprintf(temp, "conductor -name %s -l %d", program_name, levels);
	kgen_privsrc_hdr(file, func_name, purpose, 
			 input, "none", temp, NULL);
       
	*stat_routines = (char **) krealloc(*stat_routines,(*stat_num+1)
						      *sizeof(char *));
	(*stat_routines)[*stat_num] = kstrdup(func_name);
	(*stat_num)++;


       	/* write out program program_name & argument list */

        kfprintf(file, "void _xvf_get_%s(\n", pane_var);
       	kfprintf(file, "     %s_%s *paneinfo)\n", subform_var, pane_var); 
       
       	/* write out beginning bracket & declarations */
       	kfprintf(file, "{\n"); 

	if (pane->sel_list != NULL)
       	    kfprintf(file, "\tkselection *selection;\n");
        
	kfprintf(file, "\n\t/*\n\t * extracting '%s->%s->%s'",form_var, 
			subform_var, pane_var); 
	kfprintf(file, " Pane Info structure\n\t */\n\n");

	if (!(gen_get_selections(file, KPANE, pane->sel_list)))
 	    return(FALSE);

       	kfprintf(file, "}\n\n\n"); 

	kfclose(file);
	return(TRUE);
}


/*--------------------------------------------------------------
|
|  Routine Name: gen_get_selections
|
|       Purpose: Generates the code to extract the value of each selection
|		 from a master, guidepane, or pane of the UIS & put them 
|                into the appropriate place in the Form Info structure
|
|         Input: file         - open stream to "form_info.c" file
|                location    - Master, GuidePane, or Pane
|                selection   - pointer to head of selection list
|
|        Output: Returns TRUE on success, FALSE on failure
|          Date: May 12, 1992
|    Written By: Danielle Argiro
| Modifications:
|
-------------------------------------------------------------*/

static int gen_get_selections(
   kfile      *file,
   int        location,
   kselection *selection)
{
	char      *variable;
	char      *info_var = NULL;
	int       live, optional, doubleclick, toggle_type;
 	
	if (location == KMASTER)
	   info_var = kstrdup("forminfo");
	else if (location == KGUIDEPANE)
	   info_var = kstrdup("subforminfo");
	else if (location == KPANE)
	   info_var = kstrdup("paneinfo");

	while (selection != NULL)
       	{
       	      switch(selection->type) {
        
	      case KUIS_INPUTFILE:
                   variable = ktoken_to_string(selection->var_token);
		   kvf_get_attributes(selection->back_kformstruct,
				      KVF_LIVE,     &live,
				      KVF_OPTIONAL, &optional,
				      NULL);
		   kfprintf(file, "\tif (%s->%s_struct != NULL)\n\t{\n",
			    info_var, variable);
		   kfprintf(file, "\t    selection = %s->%s_struct->Selptr;\n", 
			    info_var, variable);

		   if (selection->back_group == NULL)
  		       kfprintf(file, "\t    if (selection->modified)\n");
		   else kfprintf(file, "\t    if (selection->modified || selection->back_group->modified)\n");
		   kfprintf(file, "\t    {\n");
		   kfprintf(file, "\t\tif (%s->%s != NULL) kfree(%s->%s);\n",
			    info_var, variable, info_var, variable);
	           kfprintf(file, "\t\tkvf_get_attribute(selection->back_kformstruct, KVF_FILE_NAME,\n");
                   kfprintf(file, "\t\t                  &(%s->%s));\n", info_var, variable);

		   print_get_sel_and_optsel(file, live, optional, 
				            info_var, variable);
		   kfprintf(file, "\t    }\n");
		   kfprintf(file, "\t}\n\n");
        	   break;
       
       	      case KUIS_OUTPUTFILE:
                   variable = ktoken_to_string(selection->var_token);
		   kvf_get_attributes(selection->back_kformstruct,
				      KVF_LIVE,     &live,
				      KVF_OPTIONAL, &optional,
				      NULL);
		   kfprintf(file, "\tif (%s->%s_struct != NULL)\n\t{\n",
				info_var, variable);
                   kfprintf(file, "\t    selection = %s->%s_struct->Selptr;\n", 
				info_var, variable);
		   if (selection->back_group == NULL)
		       kfprintf(file, "\t    if (selection->modified)\n");
		   else kfprintf(file, "\t    if (selection->modified || selection->back_group->modified)\n");
		   kfprintf(file, "\t    {\n");

		   kfprintf(file, "\t    if (%s->%s != NULL) kfree(%s->%s);\n",
			    info_var, variable, info_var, variable);
	           kfprintf(file, "\t    kvf_get_attribute(selection->back_kformstruct, KVF_FILE_NAME,\n");
                   kfprintf(file, "\t                    &(%s->%s));\n", info_var, variable);
		   print_get_sel_and_optsel(file, live, optional, 
				            info_var, variable);
		   kfprintf(file, "\t    }\n");
		   kfprintf(file, "\t}\n\n");
       		   break;
       
      	      case KUIS_INTEGER:
                   variable = ktoken_to_string(selection->var_token);
		   kvf_get_attributes(selection->back_kformstruct,
				      KVF_LIVE,     &live,
				      KVF_OPTIONAL, &optional,
				      NULL);
		   kfprintf(file, "\tif (%s->%s_struct != NULL)\n\t{\n",
				info_var, variable);
                   kfprintf(file, "\t    selection = %s->%s_struct->Selptr;\n", 
				info_var, variable);

		   if (selection->back_group == NULL)
		       kfprintf(file, "\t    if (selection->modified)\n");
		   else kfprintf(file, "\t    if (selection->modified || selection->back_group->modified)\n");
		   kfprintf(file, "\t    {\n");
	           kfprintf(file, "\t    kvf_get_attribute(selection->back_kformstruct, KVF_INT_VAL,\n");
                   kfprintf(file, "\t                    &(%s->%s));\n", info_var, variable);
		   print_get_sel_and_optsel(file, live, optional, 
				            info_var, variable);
		   kfprintf(file, "\t    }\n");
		   kfprintf(file, "\t}\n\n");
       		   break;
       
       	      case KUIS_FLOAT:
                   variable = ktoken_to_string(selection->var_token);
		   kvf_get_attributes(selection->back_kformstruct,
				      KVF_LIVE,     &live,
				      KVF_OPTIONAL, &optional,
				      NULL);
		   kfprintf(file, "\tif (%s->%s_struct != NULL)\n\t{\n",
				info_var, variable);
                   kfprintf(file, "\t    selection = %s->%s_struct->Selptr;\n", 
				info_var, variable);
		   if (selection->back_group == NULL)
		       kfprintf(file, "\t    if (selection->modified)\n");
		   else kfprintf(file, "\t    if (selection->modified || selection->back_group->modified)\n");
		   kfprintf(file, "\t    {\n");
	           kfprintf(file, "\t    kvf_get_attribute(selection->back_kformstruct, KVF_FLOAT_VAL,\n");
                   kfprintf(file, "\t                    &(%s->%s));\n", info_var, variable);
		   print_get_sel_and_optsel(file, live, optional, 
				            info_var, variable);
		   kfprintf(file, "\t    }\n");
		   kfprintf(file, "\n\t}\n\n");
       		   break;
      
       	      case KUIS_DOUBLE:
                   variable = ktoken_to_string(selection->var_token);
		   kvf_get_attributes(selection->back_kformstruct,
				      KVF_LIVE,     &live,
				      KVF_OPTIONAL, &optional,
				      NULL);
		   kfprintf(file, "\tif (%s->%s_struct != NULL)\n\t{\n",
				info_var, variable);
                   kfprintf(file, "\t    selection = %s->%s_struct->Selptr;\n", 
				info_var, variable);
		   if (selection->back_group == NULL)
		       kfprintf(file, "\t    if (selection->modified)\n");
		   else kfprintf(file, "\t    if (selection->modified || selection->back_group->modified)\n");
		   kfprintf(file, "\t    {\n");
	           kfprintf(file, "\t    kvf_get_attribute(selection->back_kformstruct, KVF_DOUBLE_VAL,\n");
                   kfprintf(file, "\t                    &(%s->%s));\n", info_var, variable);
		   print_get_sel_and_optsel(file, live, optional, 
				            info_var, variable);
		   kfprintf(file, "\t    }\n");
		   kfprintf(file, "\n\t}\n\n");
       		   break;
      
      		
       	      case KUIS_STRING:
       	      case KUIS_STRINGLIST:
                   variable = ktoken_to_string(selection->var_token);
		   kvf_get_attributes(selection->back_kformstruct,
				      KVF_LIVE,     &live,
				      KVF_OPTIONAL, &optional,
				      NULL);
		   kfprintf(file, "\tif (%s->%s_struct != NULL)\n\t{\n",
				info_var, variable);
                   kfprintf(file, "\t    selection = %s->%s_struct->Selptr;\n", 
				info_var, variable);

		   if (selection->back_group == NULL)
		       kfprintf(file, "\t    if (selection->modified)\n");
		   else kfprintf(file, "\t    if (selection->modified || selection->back_group->modified)\n");
		   kfprintf(file, "\t    {\n");
		   kfprintf(file, "\t    if (%s->%s != NULL) kfree(%s->%s);\n",
			    info_var, variable, info_var, variable);
	           kfprintf(file, "\t    kvf_get_attribute(selection->back_kformstruct, KVF_STRING_VAL,\n");
                   kfprintf(file, "\t                    &(%s->%s));\n", info_var, variable);
		   print_get_sel_and_optsel(file, live, optional, 
				            info_var, variable);
		   kfprintf(file, "\t    }\n");
		   kfprintf(file, "\n\t}\n\n");
       		   break;
       
       	      case KUIS_LOGICAL:
                   variable = ktoken_to_string(selection->var_token);
		   kvf_get_attributes(selection->back_kformstruct,
				      KVF_LIVE,     &live,
				      KVF_OPTIONAL, &optional,
				      NULL);
		   kfprintf(file, "\tif (%s->%s_struct != NULL)\n\t{\n",
				info_var, variable);
                   kfprintf(file, "\t   selection = %s->%s_struct->Selptr;\n", 
				info_var, variable);
		   if (selection->back_group == NULL)
		       kfprintf(file, "\t    if (selection->modified)\n");
		   else kfprintf(file, "\t    if (selection->modified || selection->back_group->modified)\n");
		   kfprintf(file, "\t    {\n");
	           kfprintf(file, "\t    kvf_get_attribute(selection->back_kformstruct, KVF_LOGIC_VAL,\n");
                   kfprintf(file, "\t                    &(%s->%s));\n", info_var, variable);
	           kfprintf(file, "\t    kvf_get_attribute(selection->back_kformstruct, KVF_LOGIC_LABEL,\n");
                   kfprintf(file, "\t                    &(%s->%s_label));\n", info_var, variable);
		   print_get_sel_and_optsel(file, live, optional, 
				            info_var, variable);
		   kfprintf(file, "\t    }\n");
		   kfprintf(file, "\n\t}\n\n");
       		   break;
       
       	      case KUIS_FLAG:
                   variable = ktoken_to_string(selection->var_token);
		   kvf_get_attributes(selection->back_kformstruct,
                                      KVF_LIVE,     &live,
                                      NULL);
		   kfprintf(file, "\tif (%s->%s_struct != NULL)\n\t{\n",
				info_var, variable);
                   kfprintf(file, "\t   selection = %s->%s_struct->Selptr;\n", 
				info_var, variable);
		   if (selection->back_group == NULL)
		       kfprintf(file, "\t    if (selection->modified)\n");
		   else kfprintf(file, "\t    if (selection->modified || selection->back_group->modified)\n");
		   kfprintf(file, "\t    {\n");
	           kfprintf(file, "\t    kvf_get_attribute(selection->back_kformstruct, KVF_OPTSEL,\n");
                   kfprintf(file, "\t                    &(%s->%s));\n", info_var, variable);
		   if (live)
		   {
	              kfprintf(file, "\t    kvf_get_attribute(selection->back_kformstruct, KVF_SELECTED,\n");
                      kfprintf(file, "\t                    &(%s->%s_selected));\n", info_var, variable);
		   }
		   kfprintf(file, "\t    }\n");
		   kfprintf(file, "\n\t}\n\n");
       		   break;

       	      case KUIS_CYCLE:
                   variable = ktoken_to_string(selection->var_token);
		   kvf_get_attributes(selection->back_kformstruct,
				      KVF_LIVE,     &live,
				      KVF_OPTIONAL, &optional,
				      NULL);
		   kfprintf(file, "\tif (%s->%s_struct != NULL)\n\t{\n",
				info_var, variable);
                   kfprintf(file, "\t    selection = %s->%s_struct->Selptr;\n", 
				info_var, variable);
		   if (selection->back_group == NULL)
		       kfprintf(file, "\t    if (selection->modified)\n");
		   else kfprintf(file, "\t    if (selection->modified || selection->back_group->modified)\n");
		   kfprintf(file, "\t    {\n");
	           kfprintf(file, "\t    kvf_get_attribute(selection->back_kformstruct, KVF_CYCLE_VAL,\n");
                   kfprintf(file, "\t                    &(%s->%s));\n", info_var, variable);
		   kfprintf(file, "\t    if (%s->%s_label != NULL) kfree(%s->%s_label);\n",
			    info_var, variable, info_var, variable);
	           kfprintf(file, "\t    kvf_get_attribute(selection->back_kformstruct, KVF_CYCLE_LABEL,\n");
                   kfprintf(file, "\t                    &(%s->%s_label));\n", info_var, variable);
		   print_get_sel_and_optsel(file, live, optional, 
				            info_var, variable);
		   kfprintf(file, "\t    }\n");
		   kfprintf(file, "\n\t}\n\n");
       		   break;
       
       	      case KUIS_LIST:
       	      case KUIS_DISPLAYLIST:
                   variable = ktoken_to_string(selection->var_token);
		   kvf_get_attributes(selection->back_kformstruct,
				      KVF_LIVE,     &live,
				      KVF_OPTIONAL, &optional,
				      NULL);
		   if (selection->type == KUIS_DISPLAYLIST)
		        kvf_get_attribute(selection->back_kformstruct,
					  KVF_LIST_DOUBLECLICK, &doubleclick);
		   else doubleclick = FALSE;

		   kfprintf(file, "\tif (%s->%s_struct != NULL)\n\t{\n",
				info_var, variable);
                   kfprintf(file, "\t    selection = %s->%s_struct->Selptr;\n", 
				info_var, variable);
		   if (selection->back_group == NULL)
		       kfprintf(file, "\t    if (selection->modified)\n");
		   else kfprintf(file, "\t    if (selection->modified || selection->back_group->modified)\n");
		   kfprintf(file, "\t    {\n");
	           kfprintf(file, "\t    kvf_get_attribute(selection->back_kformstruct, KVF_LIST_VAL,\n");
                   kfprintf(file, "\t                    &(%s->%s));\n", info_var, variable);
		   kfprintf(file, "\t    if (%s->%s_label != NULL) kfree(%s->%s_label);\n",
			    info_var, variable, info_var, variable);
	           kfprintf(file, "\t    kvf_get_attribute(selection->back_kformstruct, KVF_LIST_LABEL,\n");
                   kfprintf(file, "\t                    &(%s->%s_label));\n", info_var, variable);
		   print_get_sel_and_optsel(file, live, optional, 
				            info_var, variable);
		   if (doubleclick)
		      kfprintf(file,
			 "\t    %s->%s_dblclick = selection->prev_selected;\n",
                               info_var, variable);
		   kfprintf(file, "\t    }\n");
		   kfprintf(file, "\n\t}\n\n");
       		   break;
       
	      case KUIS_TOGGLE:
                   variable = ktoken_to_string(selection->var_token);
		   kvf_get_attributes(selection->back_kformstruct,
				      KVF_LIVE,        &live,
				      KVF_OPTIONAL,    &optional,
				      KVF_TOGGLE_TYPE, &toggle_type,
				      NULL);
		   kfprintf(file, "\tif (%s->%s_struct != NULL)\n\t{\n",
				info_var, variable);
		   kfprintf(file, "\t    char *value_string;\n");
                   kfprintf(file, "\t    selection = %s->%s_struct->Selptr;\n", 
			info_var, variable);
		   if (selection->back_group == NULL)
		       kfprintf(file, "\t    if (selection->modified)\n");
		   else kfprintf(file, "\t    if (selection->modified || selection->back_group->modified)\n");
		   kfprintf(file, "\t    {\n");
	           kfprintf(file, "\t    kvf_get_attribute(selection->back_kformstruct, KVF_TOGGLE_NUM,\n");
                   kfprintf(file, "\t                    &(%s->%s_num));\n", info_var, variable);
	           kfprintf(file, "\t    kvf_get_attribute(selection->back_kformstruct, KVF_TOGGLE_VAL,\n");
                   kfprintf(file, "\t                    &(value_string));\n");
		   if ((toggle_type == KUIS_FLAG) ||
		       (toggle_type == KUIS_INTEGER) ||
		       (toggle_type == KUIS_LOGICAL))
		   {
		        kfprintf(file, "\t    %s->%s_val =",
				info_var, variable);
		        kfprintf(file, " atoi(value_string);\n");
		        kfprintf(file, "\t    kfree(value_string);\n");
		   }
		   else if ((toggle_type == KUIS_FLOAT) ||
			    (toggle_type == KUIS_DOUBLE))
		   {
		        kfprintf(file, "\t    %s->%s_val =",
				info_var, variable);
		        kfprintf(file, " atof(value_string);\n");
		        kfprintf(file, "\t    kfree(value_string);\n");
		   }
		   else if ((toggle_type == KUIS_STRING) ||
			    (toggle_type == KUIS_INPUTFILE) ||
			    (toggle_type == KUIS_OUTPUTFILE))
		   {
			kfprintf(file, "\t    kfree(%s->%s_val);\n",
				 info_var, variable);
		        kfprintf(file, "\t    %s->%s_val =",
				 info_var, variable);
		        kfprintf(file, " kstrdup(value_string);\n");
		        kfprintf(file, "\t    kfree(value_string);\n");
		   }
		   else
		   {
			errno = KINTERNAL;
			kerror(KCODEGEN, "gen_get_selections",
			       "Toggle member of unsupported type");
			return(FALSE);
		   }
	
		   print_get_sel_and_optsel(file, live, optional, 
				            info_var, variable);
		   kfprintf(file, "\t    }\n");
		   kfprintf(file, "\n\t}\n\n");
		   break;
			
			
       	      case KUIS_ROUTINE:
		   kfprintf(file, "\tif (%s->routine_struct != NULL)\n\t{\n",
				info_var);
                   kfprintf(file, "\t    selection = %s->routine_struct->Selptr;\n", 
				info_var);
	           kfprintf(file, "\t    kvf_get_attribute(selection->back_kformstruct, KVF_SELECTED,\n");
                   kfprintf(file, "\t                    &(%s->routine));\n", info_var);
		   kfprintf(file, "\n\t}\n\n");
       		   break;
       
       	      case KUIS_MASTERACTION:
                   variable = ktoken_to_string(selection->var_token);
		   kfprintf(file, "\tif (%s->%s_struct != NULL)\n\t{\n",
				info_var, variable);
                   kfprintf(file, "\t    selection = %s->%s_struct->Selptr;\n", 
				info_var, variable);
	           kfprintf(file, "\t    kvf_get_attribute(selection->back_kformstruct, KVF_SELECTED,\n");
                   kfprintf(file, "\t                    &(%s->%s));\n", info_var, variable);
		   kfprintf(file, "\n\t}\n\n");
       		   break;

       	      case KUIS_SUBFORMACTION:
                   variable = ktoken_to_string(selection->var_token);
		   kfprintf(file, "\tif (%s->%s_struct != NULL)\n\t{\n",
				info_var, variable);
                   kfprintf(file, "\t    selection = %s->%s_struct->Selptr;\n", 
				info_var, variable);
	           kfprintf(file, "\t    kvf_get_attribute(selection->back_kformstruct, KVF_SELECTED,\n");
                   kfprintf(file, "\t                    &(%s->%s));\n", info_var, variable);
		   kfprintf(file, "\n\t}\n\n");
       		   break;

       	      case KUIS_PANEACTION:
                   variable = ktoken_to_string(selection->var_token);
		   kfprintf(file, "\tif (%s->%s_struct != NULL)\n\t{\n",
				info_var, variable);
                   kfprintf(file, "\t    selection = %s->%s_struct->Selptr;\n", 
				info_var, variable);
	           kfprintf(file, "\t    kvf_get_attribute(selection->back_kformstruct, KVF_SELECTED,\n");
                   kfprintf(file, "\t                    &(%s->%s));\n", info_var, variable);
		   kfprintf(file, "\n\t}\n\n");
       		   break;
      
       	      case KUIS_WORKSPACE:
                   variable = ktoken_to_string(selection->var_token);
		   kfprintf(file, "\tif (%s->%s_struct != NULL)\n\t{\n",
				info_var, variable);
                   kfprintf(file, "\t    selection = %s->%s_struct->Selptr;\n", 
				info_var, variable);
		   kfprintf(file, "\t    if (selection->modified)\n");
		   kfprintf(file, "\t    {\n");
	           kfprintf(file, "\t    kvf_get_attribute(selection->back_kformstruct, KVF_WORKSPACE,\n");
                   kfprintf(file, "\t                    &(%s->%s));\n", info_var, variable);
		   kfprintf(file, "\t    }\n");
		   kfprintf(file, "\n\t}\n\n");
       		   break;
      
       	      case KUIS_BLANK:
       	      case KUIS_STDIN:
       	      case KUIS_STDOUT:
	      case KUIS_HELP:
       		   break;
       
	      case KUIS_QUIT:
		   kfprintf(file, "\tif (%s->quit_struct != NULL)\n\t{\n", 
				info_var);
                   kfprintf(file, "\t    selection = %s->quit_struct->Selptr;\n",
				info_var);
	           kfprintf(file, "\t    kvf_get_attribute(selection->back_kformstruct, KVF_SELECTED,\n");
                   kfprintf(file, "\t                    &(%s->quit));\n", info_var);
		   kfprintf(file, "\n\t}\n\n");
       		   break;

	      case KUIS_MUTEXCL:
	      case KUIS_MUTINCL:
                   if (!(gen_get_selections(file, location, 
					        selection->group_next)))
			return(FALSE);
		   break;

      	      default:
       		   break;
       
       		   }  /* end switch */
       
	    selection = selection->next;
	}
       
	kfree(info_var);
	return(TRUE);
	
}  /* gen_get_selections */

static void print_get_sel_and_optsel(
    kfile *file,
    int   live,
    int   optional,
    char  *info_var,
    char  *variable)
{
	if (live)
	{
	    kfprintf(file, "\t    kvf_get_attribute(selection->back_kformstruct, KVF_SELECTED,\n");
            kfprintf(file, "\t                    &(%s->%s_selected));\n", info_var, variable);
	}
        if (optional)
	{
	    kfprintf(file, "\t    kvf_get_attribute(selection->back_kformstruct, KVF_OPTSEL,\n");
            kfprintf(file, "\t                    &(%s->%s_optsel));\n", info_var, variable);
	}
}
