/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>         Routines To Print CLUI Program Syntax         <<<<
   >>>>                                                       <<<<
   >>>>  Private:                                             <<<<
   >>>>                kgen_clui_print_usage()                <<<<
   >>>>   Static:                                             <<<<
   >>>>   Public:                                             <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"

static void print_arg_usage        PROTO((kfile *, kselection *, 
					  int, int, int, int));
static void print_toggle_usage     PROTO((kfile *, kselection *, 
					  int, int, int, int));
static void print_list_usage       PROTO((kfile *, kselection *, 
					  int, int, int, int));
static void print_group_usage      PROTO((kfile *, kselection *, 
					  int, int));
static void print_spaces           PROTO((kfile *, int, int));

static int  max_variable_length    PROTO((kselection *));
static int  max_datatype_length    PROTO((kselection *));

/*-----------------------------------------------------------
|
|  Routine Name: kgen_clui_print_usage
|
|       Purpose: Prints the usage of the program.
|
|         Input: file - file to which to write usage (may be kstderr)
|	 Output: none
|       Returns: none
|    Written By: Danielle Argiro
|          Date: April 19, 1993
| Modifications: 
|
------------------------------------------------------------*/


void kgen_clui_print_usage(
    kfile *file)
{
	int        indx;
	int        variable_spacenum = 0;
	int        datatype_spacenum = 0;
	kselection *selection;

	/* 
	 * print out usage for required arguments 
	 */
	if (kgen_req_num > 0) 
	{
	    selection = kgen_req_sels[0];
	    variable_spacenum = 
		max_variable_length(selection->back_control->sel_list);
	    datatype_spacenum = 
		max_datatype_length(selection->back_control->sel_list);
	}
	else if (kgen_opt_num > 0) 
        {
            selection = kgen_opt_sels[0];
            variable_spacenum = 
                max_variable_length(selection->back_control->sel_list);
	    datatype_spacenum = 
		max_datatype_length(selection->back_control->sel_list);
        }

        if (kgen_req_num > 0)
        {
            indx = 0;
	    while (indx < kgen_req_num)
	    {
		selection = kgen_req_sels[indx++];

		if (selection->type == KUIS_TOGGLE)
		    print_toggle_usage(file, selection, TRUE, FALSE, 
				       variable_spacenum, datatype_spacenum);

		else if ((selection->type == KUIS_LIST)        ||
		         (selection->type == KUIS_DISPLAYLIST) ||
		         (selection->type == KUIS_CYCLE))
		    print_list_usage(file, selection, TRUE, FALSE,
			             variable_spacenum, datatype_spacenum);

		else if ((selection->type == KUIS_MUTEXCL) ||
		         (selection->type == KUIS_MUTINCL) ||
		         (selection->type == KUIS_GROUP))
		    print_group_usage(file, selection, TRUE, FALSE);

		else print_arg_usage(file, selection, TRUE, FALSE,
				     variable_spacenum, datatype_spacenum);
	    }
            kfprintf(file, "\n");
        }

	/* 
	 * print out usage for optional arguments 
	 */
	if (kgen_opt_num > 0)
	{
	    indx = 0;
	    while (indx < kgen_opt_num)
	    {
                selection = kgen_opt_sels[indx++];

                if (selection->type == KUIS_TOGGLE)
                    print_toggle_usage(file, selection, FALSE, FALSE,
			               variable_spacenum, datatype_spacenum);

                else if ((selection->type == KUIS_LIST)        ||
                         (selection->type == KUIS_DISPLAYLIST) ||
                         (selection->type == KUIS_CYCLE))
                    print_list_usage(file, selection, FALSE, FALSE,
			             variable_spacenum, datatype_spacenum);

                else if ((selection->type == KUIS_MUTEXCL) ||
                         (selection->type == KUIS_MUTINCL) ||
                         (selection->type == KUIS_GROUP))
                    print_group_usage(file, selection, FALSE, FALSE);

                else print_arg_usage(file, selection, FALSE, FALSE,
			             variable_spacenum, datatype_spacenum);
	    }
	}
}


/*-----------------------------------------------------------
|
|  Routine Name: print_arg_usage
|
|       Purpose: Prints the usage for the "regular" arguments
|                of a program (everything but toggles, lists,
|                groups, etc).
|
|         Input: file      - file to which to write usage (may be kstderr)
|                selection - selection representing argument
|                required  - pass TRUE if argument is required
|                nested    - pass FALSE if argument is not nested,
|                            pass 1 if argument is nested once,
|                            pass 2 if argument is nested twice
|                variable_spacenum - number of spaces required by longest 
|                                    variable in selection list
|                datatype_spacenum - number of spaces required by longest 
|                                    datatype in selection list
|        Output: none
|       Returns: none
|    Written By: Danielle Argiro
|          Date: April 19, 1993
| Modifications:
|
------------------------------------------------------------*/

static void print_arg_usage(
    kfile      *file,
    kselection *selection,
    int        required, 
    int        nested,
    int        variable_spacenum,
    int        datatype_spacenum)
{
	int    i;
	char   *title;
	char   *variable;
	char   *description;
	char   *ascii_default = NULL;
	char   *datatype = NULL;
	char   desc_buffer[KLENGTH];
	char   title_buffer[KLENGTH];

	kvf_get_attributes(selection->back_kformstruct,
			   KVF_DESCRIPTION, &description,
			   KVF_TITLE,       &title,
			   NULL);

	if (description == NULL) 
	    description = kstrdup("no description");
	variable = ktoken_to_string(selection->var_token);

        if (description != NULL)
	    sprintf(desc_buffer, "%s", description);
	else sprintf(desc_buffer, " ");

	if (title != NULL)
	    sprintf(title_buffer, "%s", title);
	else sprintf(title_buffer, " ");

	/*
	 *  indent appropriate for nesting (if any)
	 */
	for (i = 0; i < 2+2*nested; i++)
	   kfprintf(file, " ");

	/*
	 *  required argument
	 */
	if (required)
	{
	    kfprintf(file,"-%s", variable);
	    print_spaces(file, kstrlen(variable) + 1, variable_spacenum);
	    datatype = kgen_ascii_datatype(selection->type);
	    kfprintf(file, "(%s)", datatype);
	    print_spaces(file, kstrlen(datatype) + 2, datatype_spacenum);
	    kfprintf(file, "%s", desc_buffer);

            if ((selection->type == KUIS_DOUBLE) ||
                (selection->type == KUIS_FLOAT)  ||
                (selection->type == KUIS_INTEGER))
	    {
		kfprintf(file, "\n");
	        print_spaces(file, 0, variable_spacenum+3);
                kgen_clui_print_bounds(file, selection, KGEN_CLUI_PRINT_USAGE);
	    }
	    kfprintf(file, "\n");
	}

        /*
         *  optional argument
         */
	else
	{
	    kfprintf(file, "[-%s]", variable);
	    print_spaces(file, kstrlen(variable) + 3, variable_spacenum);
	    datatype = kgen_ascii_datatype(selection->type);
	    kfprintf(file, "(%s)", datatype);
	    print_spaces(file, kstrlen(datatype) + 2, datatype_spacenum);
	    kfprintf(file, "%s", desc_buffer);

            ascii_default = kgen_ascii_def_val(selection);

            if ((selection->type == KUIS_FLOAT)  ||
                (selection->type == KUIS_DOUBLE) ||
                (selection->type == KUIS_INTEGER))
	    {
		kfprintf(file, "\n");
	        print_spaces(file, 0, variable_spacenum+2+2*nested);
		kfprintf(file, "(");
                kgen_clui_print_bounds(file, selection, KGEN_CLUI_PRINT_USAGE);
	    }
	    if (ascii_default != NULL)
	    {
		if ((selection->type != KUIS_FLOAT)  &&
                    (selection->type != KUIS_DOUBLE) &&
                    (selection->type != KUIS_INTEGER))
		{
		    kfprintf(file, "\n");
	            print_spaces(file, 0, variable_spacenum+2+2*nested);
		    kfprintf(file, "(");
	        }
		else kfprintf(file, ", ");
                kfprintf(file, "default = %s)", ascii_default);
	    }
	    kfprintf(file, "\n");
	}

	kfree(description);
	kfree(title);
}

/*-----------------------------------------------------------
|
|  Routine Name: print_toggle_usage
|
|       Purpose: Prints the usage for the optional toggle arguments
|                of a program; variable formatting for usage.
|
|         Input: file      - file to which to write usage (may be kstderr)
|                selection - selection representing argument
|                required  - pass TRUE if argument is required
|                nested    - pass FALSE if argument is not nested,
|                            pass 1 if argument is nested once,
|                            pass 2 if argument is nested twice
|                variable_spacenum - number of spaces required by longest 
|                                    variable in selection list
|                datatype_spacenum - number of spaces required by longest 
|                                    datatype in selection list
|        Output: none
|       Returns: none
|    Written By: Danielle Argiro
|          Date: April 19, 1993
| Modifications:
|
------------------------------------------------------------*/
static void print_toggle_usage(
    kfile      *file,
    kselection *selection,
    int        required,
    int        nested,
    int        variable_spacenum,
    int        datatype_spacenum)
{
	int    i, toggle_type;
	char  *variable;
	char  *description;
	char  *title;
	char  *datatype      = NULL;
	char  *ascii_default = NULL;
	char  desc_buffer[KLENGTH];
        char  title_buffer[KLENGTH];

        kvf_get_attributes(selection->back_kformstruct,
                           KVF_DESCRIPTION, &description,
                           KVF_TITLE,       &title,
                           KVF_TOGGLE_TYPE, &toggle_type,
                           NULL);

	if (description == NULL) 
	    description = kstrdup("no description");
	variable = ktoken_to_string(selection->var_token);

        if (description != NULL)
            sprintf(desc_buffer, "%s", description);
        else sprintf(desc_buffer, " ");

        if (title != NULL)
            sprintf(title_buffer, "%s", title);
        else sprintf(title_buffer, " ");

	/*
	 *  indent appropriate for nesting (if any)
	 */
	for (i = 0; i < 2+2*nested; i++)
	   kfprintf(file, " ");

	/*
	 *  required toggle argument
	 */
	if (required)
	{
            kfprintf(file, "-%s", variable);
	    print_spaces(file, kstrlen(variable) + 1, variable_spacenum);
	    datatype = kgen_ascii_datatype(toggle_type);
            kfprintf(file, "(%s)", datatype);
            print_spaces(file, kstrlen(datatype) + 2, datatype_spacenum);
            kfprintf(file, "%s\n", desc_buffer);

            kgen_clui_print_toggle_vals(file, selection, KGEN_CLUI_PRINT_USAGE, 
	    		               FALSE);
	    kfprintf(file, "\n");
	}

        /*
         *  optional toggle argument
         */
	else
	{
            kfprintf(file, "[-%s]", variable);
	    print_spaces(file, kstrlen(variable) + 3, variable_spacenum);
            datatype = kgen_ascii_datatype(toggle_type);
            kfprintf(file, "(%s)", datatype);
            print_spaces(file, kstrlen(datatype) + 2, datatype_spacenum);
            kfprintf(file, "%s\n", desc_buffer);

            kgen_clui_print_toggle_vals(file, selection,
                                        KGEN_CLUI_PRINT_USAGE, nested);
            ascii_default = kgen_ascii_def_val(selection);
	    if (ascii_default != NULL)
                kfprintf(file, "[default: %s]\n", ascii_default);
	}
}



/*-----------------------------------------------------------
|
|  Routine Name: print_list_usage
|
|       Purpose: Prints the usage for the optional toggle arguments
|                of a program; variable formatting for usage.
|
|         Input: file      - file to which to write usage (may be kstderr)
|                selection - selection representing argument
|                required  - pass TRUE if argument is required
|                nested    - pass FALSE if argument is not nested,
|                            pass 1 if argument is nested once,
|                            pass 2 if argument is nested twice
|                variable_spacenum - number of spaces required by longest 
|                                    variable in selection list
|                datatype_spacenum - number of spaces required by longest 
|                                    datatype in selection list
|        Output: none
|       Returns: none
|    Written By: Danielle Argiro
|          Date: April 19, 1993
| Modifications:
|
------------------------------------------------------------*/
static void print_list_usage(
    kfile      *file,
    kselection *selection,
    int        required,
    int        nested,
    int        variable_spacenum,
    int        datatype_spacenum)
{
	int    i;
        char  *variable;
        char  *description;
        char  *title;
        char  *ascii_default = NULL;
        char  *datatype      = NULL;
        char  desc_buffer[KLENGTH];
        char  title_buffer[KLENGTH];

        kvf_get_attributes(selection->back_kformstruct,
                           KVF_DESCRIPTION, &description,
                           KVF_TITLE,       &title,
                           NULL);

	if (description == NULL) 
	    description = kstrdup("no description");
	variable = ktoken_to_string(selection->var_token);

        if (description != NULL)
            sprintf(desc_buffer, "%s", description);
        else sprintf(desc_buffer, " ");

        if (title != NULL)
            sprintf(title_buffer, "%s", title);
        else sprintf(title_buffer, " ");


	/*
	 *  indent appropriate for nesting (if any)
	 */
	for (i = 0; i < 2+2*nested; i++)
	   kfprintf(file, " ");

	/* 
	 * required list argument 
         */
	if (required)
	{
            kfprintf(file, "-%s", variable);
	    print_spaces(file, kstrlen(variable) + 1, variable_spacenum);
	    datatype = kgen_ascii_datatype(selection->type);
	    kfprintf(file, "(%s)", datatype);
	    print_spaces(file, kstrlen(datatype) + 2, datatype_spacenum);
	    kfprintf(file, "%s\n", desc_buffer);
            kgen_clui_print_list_vals(file, selection, KGEN_CLUI_PRINT_USAGE, 
			              FALSE);
            kfprintf(file, "\n");
	}
	
	/* 
	 * optional list argument 
         */
	else
	{
            kfprintf(file, "[-%s]", variable);
	    print_spaces(file, kstrlen(variable) + 3, variable_spacenum);
	    datatype = kgen_ascii_datatype(selection->type);
	    kfprintf(file, "(%s)", datatype);
	    print_spaces(file, kstrlen(datatype) + 2, datatype_spacenum);
	    kfprintf(file, "%s\n", desc_buffer);

            kgen_clui_print_list_vals(file, selection,
                                      KGEN_CLUI_PRINT_USAGE, nested);

            if (nested)
                kfprintf(file, "\t\t\t");
            kfprintf(file, "\t\t");

            ascii_default = kgen_ascii_def_val(selection);
	    if (ascii_default != NULL)
                kfprintf(file, "[default: %s]\n", ascii_default);
	}

}

/*-----------------------------------------------------------
|
|  Routine Name: print_group_usage
|
|       Purpose: Prints the usage for an ME/MI group; variable
|                formatting for usage.
|
|         Input: file      - file to which to write usage (may be kstderr)
|                group_start - header of group list
|                required    - pass TRUE if group is a required ME group
|                nested      - pass FALSE if argument is not nested,
|                              pass 1 if argument is nested once,
|                              pass 2 if argument is nested twice
|        Output: none
|       Returns: none
|    Written By: Danielle Argiro
|          Date: April 19, 1993
| Modifications: 
|
------------------------------------------------------------*/

static void print_group_usage(
    kfile      *file,
    kselection *group_start,
    int        required,
    int        nested)
{
	int        i, group_size; 
	int        group_variable_spacenum; 
	int        group_datatype_spacenum;
	kselection *groupmember;

	kfprintf(file, "\n");

	/*
	 *  indent appropriate for nesting (if any)
	 */
	for (i = 0; i < 2+2*nested; i++)
	   kfprintf(file, " ");

	/*
	 *  print introductory stmt about group
	 */
	if (required)
	    kfprintf(file, "Required ");
	else kfprintf(file, "[Optional ");

	group_size = kgen_get_group_size(group_start);
	if (group_start->type == KUIS_MUTEXCL)
	{
	    kfprintf(file, "Mutually Exclusive Group - Specify ");
	    if (required)
	       kfprintf(file, "exactly one of the %d options:", group_size);
	    else kfprintf(file, "one or none of the %d options:", group_size);
	}
	else if (group_start->type == KUIS_MUTINCL)
	{
	    kfprintf(file, "Mutually Inclusive Group - "
		     "Specify all or none of the %d options:", group_size);
	}
	else if (group_start->type == KUIS_GROUP)
        {
            kfprintf(file, "Loose Group - "
		     "Specify at least one of the %d options:", group_size);
        }
	if (!required)
	    kfprintf(file, "]\n");
	else kfprintf(file, "\n");

	/*
	 *  print usage for all members in the group
	 */
	group_variable_spacenum = max_variable_length(group_start->group_next);
	group_datatype_spacenum = max_datatype_length(group_start->group_next);
	groupmember = group_start->group_next;
	while (groupmember != NULL)
	{
	    /* ignore blanks */
	    if (groupmember->type == KUIS_BLANK)
	    {
		groupmember= groupmember->next;
		continue;
	    }

	    /* toggle in group */
	    if (groupmember->type == KUIS_TOGGLE)
	 	print_toggle_usage(file, groupmember, FALSE, nested + 1,
			           group_variable_spacenum, 
			           group_datatype_spacenum);

	    /* list in group */
            else if ((groupmember->type == KUIS_LIST)        ||
                     (groupmember->type == KUIS_DISPLAYLIST) ||
                     (groupmember->type == KUIS_CYCLE))
                print_list_usage(file, groupmember, FALSE, nested + 1,
                                 group_variable_spacenum, 
                                 group_datatype_spacenum);
	  
	    /* group nested in group */
            else if ((groupmember->type == KUIS_MUTEXCL) ||
                     (groupmember->type == KUIS_MUTINCL) ||
                     (groupmember->type == KUIS_GROUP))
		print_group_usage(file, groupmember, FALSE, nested + 1);

	    /* "regular" argument in group */
	    else print_arg_usage(file, groupmember, FALSE, nested + 1,
                                 group_variable_spacenum, 
                                 group_datatype_spacenum);

	    groupmember = groupmember->next;
	}
}


/*-----------------------------------------------------------
|
|  Routine Name: max_variable_length
|
|       Purpose: Returns the maximum length of the variables
|                of the selections in a selection list (plus one)
|                so that when printing the variables in the usage statement,
|                they can all be nicely lined up in a column.
|
|         Input: sel_list - header of selection list
|        Output: none
|       Returns: The length of the longest variable name in selection list
|    Written By: Danielle Argiro
|          Date: April 19, 1993
| Modifications:
|
------------------------------------------------------------*/

static int max_variable_length(
    kselection *sel_list)
{
	char       *variable;
	int        spacenum = 0;
	kselection *selection;

	selection = sel_list;
        while (selection != NULL)
        {
	    /* don't want to take routines, blanks into account */
	    if ((selection->type == KUIS_ROUTINE) ||
	        (selection->type == KUIS_QUIT)    ||
	        (selection->type == KUIS_BLANK)   ||
	        (selection->type == KUIS_MUTEXCL) ||
	        (selection->type == KUIS_MUTINCL) ||
	        (selection->type == KUIS_GROUP))
	    {
	        selection = selection->next;
		continue;
	    }
	    variable = ktoken_to_string(selection->var_token);

	    /* don't want to take standardized selections into account */
	    if ((selection->var_token == kstring_to_token("copyright"))    ||
	        (selection->var_token == kstring_to_token("_gui_options")) ||
	        (selection->var_token == kstring_to_token("help")))
	    {
	        selection = selection->next;
		continue;
	    }
	    if (kstrlen(variable) > spacenum)
		spacenum = kstrlen(variable);
	    selection = selection->next;
        }
	spacenum += 4;

	return(spacenum);
}

/*-----------------------------------------------------------
|
|  Routine Name: max_datatype_length
|
|       Purpose: Returns the maximum length of the variables
|                of the selections in a selection list (plus one)
|                so that when printing the variables in the usage statement,
|                they can all be nicely lined up in a column.
|
|         Input: sel_list - header of selection list
|        Output: none
|       Returns: The length of the longest variable name in selection list
|    Written By: Danielle Argiro
|          Date: April 19, 1993
| Modifications:
|
------------------------------------------------------------*/

static int max_datatype_length(
    kselection *sel_list)
{
        char       *datatype;
        int        spacenum = 0;
        kselection *selection;

        selection = sel_list;
        while (selection != NULL)
        {
            datatype = kgen_ascii_datatype(selection->type);
            if (kstrlen(datatype) > spacenum)
                spacenum = kstrlen(datatype);
            selection = selection->next;
        }
        spacenum += 3;

        return(spacenum);
}


/*-----------------------------------------------------------
|
|  Routine Name: print_spaces
|
|       Purpose: Used to make the usage statement pretty and lined
|                up, this routine takes the number of spaces that have
|                already been used in the column, and then the number
|                of spaces in the column, and pads out to the end of
|                the column with spaces.
|
|         Input: file        - file to which to write usage (may be kstderr)
|                used_spaces - number of the spaces in the column already used
|                spacenum    - total number of the spaces in the column
|        Output: none
|       Returns: The length of the longest variable name in selection list
|    Written By: Danielle Argiro
|          Date: April 19, 1993
| Modifications:
|
------------------------------------------------------------*/
static void print_spaces(
    kfile *file,
    int   used_spaces,
    int   spacenum)
{
	int i;

        for (i = 0; i < (spacenum - used_spaces); i++)
            kfprintf(file, " ");
}

