/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>            Routines to Generate *.c file              <<<<
   >>>>                                                       <<<<
   >>>>   Private:                                            <<<<
   >>>>                kgen_clui_generate_cfile()             <<<<
   >>>>                                                       <<<<
   >>>>    Static:                                            <<<<
   >>>>                print_main()                           <<<<
   >>>>                                                       <<<<
   >>>>                print_usage_additions()                <<<<
   >>>>                                                       <<<<
   >>>>                print_debug_stmts()	  	      <<<<
   >>>>                print_group_debug_stmt()  	      <<<<
   >>>>                print_debug_stmt()	  	      <<<<
   >>>>                                                       <<<<
   >>>>                print_mainhdr()	              	      <<<<
   >>>>                print_mainhdr_args()	  	      <<<<
   >>>>                print_mainhdr_single_arg()	      <<<<
   >>>>    Public:                                            <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"

static void print_main               PROTO((kfile *, char **, kobject,
					   int, int, char *));
static void print_usage_additions    PROTO((kfile *, kobject, char **,
					    char *));
static void print_debug_stmts        PROTO((kfile *, char *));
static void print_debug_stmt         PROTO((kfile *, kselection *));
static void print_group_debug_stmt   PROTO((kfile *, char *, kselection *,
					    int));
static void print_mainhdr            PROTO((kfile *, kobject, char **,
					    char *));
static void print_mainhdr_args       PROTO((kfile *));
static void print_mainhdr_single_arg PROTO((kfile *, kselection *, int));
static void print_mainhdr_group_arg  PROTO((kfile *, kselection *, int));
static void print_mainhdr_toggle_arg PROTO((kfile *, kselection *, int));
static void set_prefix               PROTO((int, char *));
static kstring my_kstring_format     PROTO((kstring, kstring, kstring,
					    int, int, kstring));


/*-----------------------------------------------------------
|
|  Routine Name: kgen_clui_generate_cfile
|
|       Purpose: This routine creates the main program file *.c
|		 containing the main() program
|
|         Input: program   - program object being created
|                prog_spec -  pointer to the internal PS struct
|                opath     -  path to program object being created
|                oname     -  name of program object being generated
|                hybrid    - TRUE if it is a hybrid xvroutine
|
|        Output: Returns TRUE on success, FALSE on failure
|    Written By: Danielle Argiro
|          Date: Jul 20, 1992
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

int  kgen_clui_generate_cfile(
   kobject   program,
   char    **prog_spec,
   char     *opath,
   char     *oname,
   int       hybrid,
   char     *form_var)
{
        kstring   routine     = "kgen_clui_generate_cfile()";
	kobject   file_object = NULL;
	kobject   lib_object  = NULL;
	int       force_flag  = FALSE;
	int       isnewfile  = FALSE;
	int       object_type;
	kfile    *file;
	kstring   filepath;
	kstring   lpath;
	char      temp[KLENGTH];
	int       libcall;

        /*
         * get file object representing *.c file
         */
	if (!kcms_get_attributes(program,
				 KCMS_CMOBJ_GEN_MAIN,  &file_object,
				 KCMS_CMOBJ_TYPE,      &object_type,
				 KCMS_END)
	    || !kcms_query_bit(program, KCMS_CMOBJ_FLAGS, KCMS_BIT_CMOBJ_FORCE,
			       &force_flag))
	    return(FALSE);
		
	/*
         * 1st time *.c file has been created - create file object, so that 
         * kcms registers the *.c file in the kcms database
         */
	if (file_object == NULL)
        {
            ksprintf(temp, "%s/src/%s.c", opath, oname);
            file_object = kcms_create_fileobj(program, temp, NULL,
                                               KCMS_FOBJ_TYPE_SRC,
                                               KCMS_FOBJ_SUBTYPE_C,
                                               KCMS_FOBJ_GEN_MAIN,
                                               KCMS_FOBJ_ACCESS_RDWR);
            if (file_object == NULL)
                return(FALSE);
	    isnewfile = TRUE;
            filepath = temp;
	}

	/*
         * the *.c file has already been created & added to kcms database;
         * simply get the path to the man1 page.
         */
	else if (!kcms_get_attribute(file_object, KCMS_PATH, &filepath))
                return(FALSE);

        /*
         *  if somehow the database GHOST_SRC key got deleted from the kcms
         *  database, but the *.c file really is there, don't want to
         *  clobber it!
         */
        if ((isnewfile) && (kaccess(filepath, R_OK) == 0))
        {
            kerror(KCODEGEN, routine,
                   "WARNING!  CMS database appears to have been corrupted; "
		   "cms dbm key GHOST_SRC is missing, but the %s.c file does "
		   "exist.  Regeneration of *.c file will NOT take place.  "
		   "Adding GHOST_SRC cms key to database; next run of "
		   "ghostwriter should be OK.", oname);
	    return(FALSE);
        }


	/*
	 *  see if there is a library routine associated with this program obj
	 */
	if (!(kcms_get_attribute(program, KCMS_CMOBJ_GEN_LFILE, &lib_object)))
	    return(FALSE);
        if ((lib_object == NULL) || 
            !kcms_get_attribute(lib_object, KCMS_PATH, &lpath) || 
	     (lpath == NULL))
	    libcall = FALSE;
	else libcall = TRUE;

	/*
	 *  see if *.c file already exists.  If so, prompt to over-write,
	 *  return TRUE if the answer is NO
	 */
	if (!force_flag &&
	   koverwrite(kget_notify(), filepath) == FALSE)
	   return(TRUE);

	/*
	 *  create & open *.c file
	 */
	if (kaccess(filepath, R_OK) != 0)
	   isnewfile = TRUE;
	if ((file = kfopen(filepath, "w"))== NULL)
	{
	    kerror(KCODEGEN, "kgen_clui_generate_cfile", 
		   "Could not create file '%s'", filepath);
	    return(FALSE);
	}

	/*
	 * begin *.c file with RCS header
	 */
	kgen_rcs_chdr(file);
      
	/*
	 *  generate the main program
	 */
	print_main(file, prog_spec, program, libcall, hybrid, form_var);
	kfprintf(file, "\n\n");

	/*
	 *  generate the usage_additions routine
 	 */
	print_usage_additions(file, program, prog_spec, oname);

	/*
	 *  generate the routine to free command line args
	 */
	kgen_clui_print_free_args(program, file, prog_spec);


	/*
	 *  that was all..
	 */
 	kfclose(file);
	kannounce(KCODEGEN, routine, "done generating %s.c", oname);
	kinfo(KHOSTILE, "your %s.c file is simply wasting space on disk", 
	      oname);
	return(TRUE);
}


/*-----------------------------------------------------------
|
|  Routine Name: print_main
|
|       Purpose: This routine prints out the main program in file *.c
|
|         Input: file      - open stream to .c file
|                prog_spec - pointer to the internal PS struct
|		 object    - the software object we are generating main()
|			     for.
|                libcall   - should we generate a lprogram() call?
|                hybrid    - TRUE if this is a hybrid xvroutine
|                form_var  - variable from -F line of UIS file
|
|        Output: none
|       Returns: none
|    Written By: Danielle Argiro
|          Date: March 30, 1994
| Modifications: 
|
------------------------------------------------------------*/

static void print_main(
   kfile   *file,
   char    **prog_spec,
   kobject object,
   int     libcall,
   int     hybrid,
   char    *form_var)
{
        char      temp[KLENGTH];
	int       gen_debug;
	kobject   toolbox;
	int       object_type;
	kstring   tb_name;
	kstring   oname;
	kstring   bname;
        kstring   copyright;
        char    **priv_routines; 
   

	if (!kcms_get_attributes(object,
				 KCMS_PARENT,         &toolbox,
				 KCMS_NAME,           &oname,
				 KCMS_CMOBJ_BNAME,    &bname,
				 KCMS_CMOBJ_TYPE,     &object_type,
				 KCMS_END)
	    || !kcms_get_attribute(toolbox, KCMS_NAME, &tb_name))
	   return;

        /*
         * begin .c file with Khoros copyright
         */
        copyright = kcms_get_copyright(toolbox, KCMS_TB_COPYRIGHT_LONG,
				       KCMS_LANG_C);
        if (copyright != NULL) 
            kfprintf(file, "%s\n", copyright);
	kfree(copyright);
   
       /*
        *  print file header 
        */
        priv_routines = (char **) kcalloc(1, 1 * sizeof(char *));
        priv_routines[0] = kstrdup("main");
        ksprintf(temp, "Main program for %s", bname);
        kgen_srcfile_hdr(file, temp, priv_routines, 1, NULL, 0, NULL, 0);
	kfree(priv_routines[0]);
	kfree(priv_routines);
    
       /*
        *  add in #includes
        */
        kfprintf(file, "#include \"%s.h\"\n\n", oname);
	if ((kgen_arg_total > 0) || object_type == KCMS_XVROUTINE)
 	    kfprintf(file, "clui_info_struct *clui_info = NULL;\n\n");
	if (object_type == KCMS_XVROUTINE && !hybrid)
	    kfprintf(file, "gui_info_struct *gui_info;\n\n");

       /*
        *   put in header for main program
        */
	print_mainhdr(file, object, prog_spec, oname);
   
       /*
        *   put in beginning of main program
        */
	kfprintf(file, "void main(\n");
        kfprintf(file, "   int  argc,\n");
        kfprintf(file, "   char **argv,\n");
        kfprintf(file, "   char **envp)\n{\n\n");

       /*
 	*  if this is an xvroutine, need to generate some variables
	*/
	if (object_type == KCMS_XVROUTINE && !hybrid)
	{
            kfprintf(file,
                    "\tkform  *form;            /* form tree representing *.form file */\n");
            kfprintf(file,
                    "\tchar   *form_filestring; /* *.form file for %s    */ \n",
                     oname);
            kfprintf(file,
                    "\tchar   *form_pathname;   /* location of %s.form   */\n",
                     oname);
            kfprintf(file,
                    "\tchar   *form_fullpath;   /* expanded %s.form path */\n",
                     oname);

	}
        kfprintf(file, "\tkform  *pane;         /* form tree representing *.pane file */\n");
   
       /*
        *   add in main_variable_list from .prog file
        */
        kfprintf(file,  "/* -main_variable_list */\n");
	if (kstrlen(prog_spec[MAIN_VARIABLES]) > 0) 
           kfprintf(file,  "%s", prog_spec[MAIN_VARIABLES]);
        kfprintf(file,  "/* -main_variable_list_end */\n\n");
   
	kstring_upper(tb_name, temp);
        kfprintf(file, "\tkhoros_initialize(argc, argv, envp, \"%s\");\n",
		 temp);
	if ((kgen_arg_total > 0) || object_type == KCMS_XVROUTINE)
	    kfprintf(file, "\tkexit_handler(%s_free_args, NULL);\n\n", 
	         oname);

	/*
	 *  put in get_args section so that users can specify other 
         *  routines to get the arguments if desired
	 */
	kfprintf(file, "/* -main_get_args_call */\n");
	if (kstrlen(prog_spec[MAIN_GETARGS]) > 0)
	    kfprintf(file, "%s", prog_spec[MAIN_GETARGS]);
	else 
	{
	    if (object_type == KCMS_KROUTINE ||
	       (object_type == KCMS_XVROUTINE && hybrid))
            {
                if (kstrcmp(oname, "kexec") == 0)
                    kfprintf(file, "\tpane = kgen_initialize(PANEPATH, KGEN_NONE, \"%s\", \"%s\",\n", tb_name, oname);
                else kfprintf(file, "\tpane = kgen_initialize(PANEPATH, KGEN_KROUTINE, \"%s\", \"%s\",\n", tb_name, oname);
            }
            else if (object_type == KCMS_XVROUTINE)
                kfprintf(file, "\tpane = kgen_initialize(PANEPATH, KGEN_XVROUTINE, \"%s\", \"%s\",\n", tb_name, oname);
            else if (object_type == KCMS_PANE)
                kfprintf(file, "\tpane = kgen_initialize(PANEPATH, KGEN_PANE, \"%s\", \"%s\",\n", tb_name, oname);
            else
            {
                kerror(KCODEGEN, "print_get_args_routine",
                       "Only supported software object types are kroutines, xvroutines, and pane objects");
                return;
            }
            kfprintf(file, "\t\t%s_usage_additions);\n\n", oname);
	    if (object_type != KCMS_XVROUTINE || hybrid)
	         kfprintf(file, "\tif (!(kclui_check_args()))\n\t    kexit(KEXIT_FAILURE);\n");
	    kfprintf(file, "\t%s_get_args(pane);\n", oname);
	    kfprintf(file, "\tkvf_destroy_form(pane);\n");
	}

	kfprintf(file, "/* -main_get_args_call_end */\n\n");
   
       /*
        *   if -debug specified, write out print stmts
        */
        if (kcms_query_bit(object, KCMS_CMOBJ_FLAGS, KCMS_BIT_CMOBJ_GENDEBUG,
	                   &gen_debug) && gen_debug)
   	   print_debug_stmts(file, oname);
   
       /*
        * if this is an xvroutine, need to:
	*    1 - initialize xvwidgets lib
        *    2 - set form_fullpath (path to *.form file)
        *    3 - allocate the GUI info structure
        *    4 - create form tree
        *    5 - initialize the GUI info structure
        */
        if (object_type == KCMS_XVROUTINE && !hybrid)
        {
	    /* gen code to initialize xvwidgets lib */
	    kfprintf(file, "\t/*\n\t * Initialize xvwidgets lib\n\t */\n");
            kfprintf(file, "\tif (!(xvw_initialize(XVW_MENUS_XVFORMS)))\n");
            kfprintf(file, "\t{\n");
            kfprintf(file, "\t     kinfo(KSTANDARD, \"Could not open connection to server\");\n");
            kfprintf(file, "\t     kexit(KEXIT_FAILURE);\n");
            kfprintf(file, "\t}\n\n");

	    /* gen code to set form_fullpath (path to *.form file) */
            kfprintf(file, "\t/*\n\t * Set location of %s.form\n\t */\n",
		     oname);
            kfprintf(file, "\tif (clui_info->form_flag)\n\t{\n");
            kfprintf(file, "\t    form_filestring = kstrdup(clui_info->form_file);\n");
            kfprintf(file, "\t    form_pathname   = NULL;\n");
            kfprintf(file, "\t}\n\telse\n\t{\n");
            kfprintf(file, "\t    form_filestring = kstrdup(\"%s.form\");\n", oname);
            kfprintf(file, "\t    form_pathname   = kstrdup(\"$TOOLBOX/objects/xvroutine/%s/uis\");\n", oname);
            kfprintf(file, "\t}\n");
            kfprintf(file, "\tform_fullpath   = kfullpath(form_filestring, form_pathname, NULL);\n\n");

	    /* gen code to allocate GUI info structure */
	    kfprintf(file, "\t/*\n\t * Allocate the GUI Info structure \n\t */\n");
	    kfprintf(file, "\tgui_info = (gui_info_struct *)\n");
            kfprintf(file, "\t\t\t\tkcalloc (1, sizeof (gui_info_struct));\n");

	    /* gen code to create form tree */
	    kfprintf(file, "\t/*\n\t * Create the forms for %s \n\t */\n", 
		     oname);
            kfprintf(file, "\tform = xvf_create_form(form_fullpath, NONE, run_%s, gui_info,\n", form_var);
            kfprintf(file, "\t\t\t\tclui_info->x_int, clui_info->y_int,\n");
	    kfprintf(file, "\t\t\t\tXVF_PARTIAL_EDIT);\n");
	    kfprintf(file, "\tif (form == NULL)\n");
            kfprintf(file, "\t   kexit(KEXIT_FAILURE);\n\n");

	    /* gen code to initialize GUI info structure */
	    kfprintf(file, "\t/*\n\t * Initialize the GUI Info structure \n\t */\n");
	    kfprintf(file, "\tif (!(_xvf_init_%s(form, gui_info)))\n",
                     form_var);
	    kfprintf(file, "\t    kexit(KEXIT_FAILURE);\n\n");

	}
	else if (hybrid)
	{
	    kfprintf(file, "\t/*\n\t * Initialize xvwidgets lib\n\t */\n");
            kfprintf(file, "\tif (!(xvw_initialize(XVW_MENUS_XVFORMS)))\n");
            kfprintf(file, "\t{\n");
            kfprintf(file, "\t     kinfo(KSTANDARD, \"Could not open connection to server\");\n");
            kfprintf(file, "\t     kexit(KEXIT_FAILURE);\n");
            kfprintf(file, "\t}\n\n");
	}

       /*
        *   add in main_before_lib_call from .prog file
        */
        kfprintf(file,  "/* -main_before_lib_call */\n");
	if (kstrlen(prog_spec[MAIN_BEFORELIB]) > 0) 
           kfprintf(file,  "%s", prog_spec[MAIN_BEFORELIB]);
        kfprintf(file,  "/* -main_before_lib_call_end */\n\n");
   
       /*
        *   add in main_library_call from .prog file
        */
        kfprintf(file,  "/* -main_library_call */\n");
	if (kstrlen(prog_spec[MAIN_LIBCALL]) > 0) 
           kfprintf(file,  "%s", prog_spec[MAIN_LIBCALL]);
	else if (object_type == KCMS_XVROUTINE)
	   kfprintf(file,  "\txvf_run_form();\n");
	else if (libcall) 
	   kfprintf(file,  "\tl%s();\n", oname);
        kfprintf(file,  "/* -main_library_call_end */\n\n");
   
       /*
        *   add in main_after_lib_call from .prog file
        */
        kfprintf(file,  "/* -main_after_lib_call */\n");
	if (kstrlen(prog_spec[MAIN_AFTERLIB]) > 0) 
           kfprintf(file,  "%s", prog_spec[MAIN_AFTERLIB]);
        kfprintf(file,  "/* -main_after_lib_call_end */\n\n");
   
       /*
        *  if this is an xvroutine, need to free GUI info structure
        *  and free the string variables used for fullpath of *.form file
        */
        if (object_type == KCMS_XVROUTINE && !hybrid)
        {
            kfprintf(file, "\tkfree(form_filestring);\n");
            kfprintf(file, "\tkfree(form_pathname);\n");
            kfprintf(file, "\tkfree(form_fullpath);\n");
	}
   
        kfprintf(file, "\n\tkexit(KEXIT_SUCCESS);\n");
        kfprintf(file, "}\n");   
}


/*---------------------------------------------------------- 
|
|  Routine Name: print_usage_additions
|
|       Purpose: This routine prints the usage_additions() routine in file *.c
|
|         Input: file      - open stream to .c file
|                prog_spec - pointer to the internal specification struct
|                name      - name of the program being generated
|        Output: none
|       Returns: none
|    Written By: Danielle Argiro
|          Date: Jul 20, 1992
| Modifications: 
|
------------------------------------------------------------*/

static void print_usage_additions(
   kfile    *file,
   kobject   object,
   char    **prog_spec,
   char     *name)
{
	int    i, count;
        char  **textlines = NULL, *tmpptr;
	char   temp[KLENGTH], purpose[KLENGTH], routine_name[KLENGTH];
	kstring  short_description = NULL;


	ksprintf(routine_name, "%s_usage_additions", name);
	ksprintf(purpose, "Prints usage additions in %s_usage routine", name);
        ksprintf(temp, "ghostwriter -oname %s", name);
        kgen_privsrc_hdr(file, routine_name, purpose, 
			 "None", "None", temp, NULL);

	kfprintf(file, "void %s_usage_additions(void)\n{\n", name);

	if (!kcms_get_attribute(object,
				KCMS_CMOBJ_SHORT_DESC, &short_description))
	   return;

	if (kstrlen(short_description) > 0)
	{
	   tmpptr = my_kstring_format(short_description, NULL, NULL, 60,
				      FALSE, NULL);

	   textlines = kparse_string_delimit(tmpptr, "\n", KDELIM_CLEAN,
				              &count);
	    kfree(tmpptr);
	    if (textlines != NULL && count > 0)
	    {
		for (i = 0; i < count; i++)
		    kfprintf(file, "\tkfprintf(kstderr, \"\\t%s\\n\");\n",
			     textlines[i]);
		karray_free(textlines, count, NULL);
	    }
	}

	/*
	 *   add in usage_additions from .prog file
	 */
	kfprintf(file,  "\n/* -usage_additions */\n");
	if (kstrlen(prog_spec[USAGEADD]) > 0)
	    kfprintf(file,  "%s", prog_spec[USAGEADD]);
	kfprintf(file,  "/* -usage_additions_end */\n\n");

	kfprintf(file, "}\n");
}

/*-----------------------------------------------------------
|
|  Routine Name: print_debug_stmts
|
|       Purpose: For debugging only, this routine prints kfprintf() statements
| 		 into the main program, to print out values of the generated
|		 structure in the .h file, as extracted from the command line
|		 by the generated kgen_get_args() routine.
|
|         Input: file - open stream to .c file
|		 name - name of program being generated.
|        Output: none
|       Returns: none
|    Written By: Danielle Argiro
|          Date: March 30, 1994
| Modifications:
|
------------------------------------------------------------*/

static void print_debug_stmts(
   kfile *file,
   char  *name)
{
    	int        indx;
	kselection *selection;

    	/*
      	 * print debugging stmts for required arguments
     	 */

	if (kgen_req_num > 0)
	{
            kfprintf(file, "\t/*\n\t * print values of required args for %s\n\t */\n", name);
	    kfprintf(file, "\tkfprintf(kstderr,\"\\nRequired Arguments\\n\");\n");

            indx = 0;
	    while (indx < kgen_req_num)
	    {
		selection = kgen_req_sels[indx];
		if (selection->type == KUIS_TOGGLE)
		    kgen_clui_print_toggle_debug_stmt(file, selection);

		else if ((selection->type == KUIS_MUTEXCL) ||
		         (selection->type == KUIS_MUTINCL) ||
			 (selection->type == KUIS_GROUP))
		{
                     kfprintf(file,"\n\t/*\n\t * print values of optional ME/MI groups for %s\n\t */\n", name);
		     print_group_debug_stmt(file, name, selection, TRUE);
		     kfprintf(file, "\n");
		}

		else print_debug_stmt(file, selection);
		indx++;
	    }
	}

    	/*
      	 * print debugging stmts for optional arguments
     	 */

	if (kgen_opt_num > 0)
	{
            kfprintf(file,"\t/*\n\t * print values of optional args for %s\n\t */\n",
		     name);
	    kfprintf(file, "\tkfprintf(kstderr,\"\\nOptional Arguments\\n\");\n");

            indx = 0;
	    while (indx < kgen_opt_num)
	    {
		selection = kgen_opt_sels[indx];
                if (selection->type == KUIS_TOGGLE)
                    kgen_clui_print_toggle_debug_stmt(file, selection);

		else if ((selection->type == KUIS_MUTEXCL) ||
		         (selection->type == KUIS_MUTINCL) ||
			 (selection->type == KUIS_GROUP))
		{
                     kfprintf(file,"\n\t/*\n\t * print values of optional ME/MI groups for %s\n\t */\n", name);
		     print_group_debug_stmt(file, name, selection, FALSE);
		     kfprintf(file, "\n");
		}
		else print_debug_stmt(file, selection);
		indx++;
	    }
	}

	kfprintf(file, "\n");
    	
}  /* end print_debug_stmts */

/*-----------------------------------------------------------
|
|  Routine Name: print_group_debug_stmt
|
|       Purpose: prints a single debug statement for an ME or MI group
|
|         Input: file        - open stream to *.c file
|                name        - name of program being created
|                group_start - start of group member list
|                optional    - TRUE if ME group is optional
|        Output: none
|       Returns: none
|    Written By: Danielle Argiro
|          Date: March 30, 1994
| Modifications: 
|
------------------------------------------------------------*/
static void print_group_debug_stmt(
    kfile      *file,
    char       *name,
    kselection *group_start,
    int        optional)
{
	kselection *groupmember;

        if (group_start->type == KUIS_MUTEXCL) 
	{
            if (!optional) 
                kfprintf(file, 
                         "\tkfprintf(kstderr,\"\\nRequired ME Group\\n\");\n");
	    else kfprintf(file, 
                         "\tkfprintf(kstderr,\"\\nOptional ME Group\\n\");\n");
	}
        else if (group_start->type == KUIS_MUTINCL)
	{
	    if (!optional)
		 kfprintf(file, 
		        "\tkfprintf(kstderr,\"\\nRequired MI Group\\n\");\n");
	    else kfprintf(file, 
	   	        "\tkfprintf(kstderr,\"\\nOptional MI Group\\n\");\n");
	}
	else if (group_start->type == KUIS_MUTINCL)
        {
            if (!optional)
                 kfprintf(file,
                       "\tkfprintf(kstderr,\"\\nRequired Loose Group\\n\");\n");
            else kfprintf(file,
                       "\tkfprintf(kstderr,\"\\nOptional Loose Group\\n\");\n");
        }

	groupmember = group_start->group_next;
        while (groupmember != NULL)
        {
	    if (groupmember->type == KUIS_BLANK)
	    {
		groupmember = groupmember->next;
		continue;
	    }
	    if (groupmember->type == KUIS_TOGGLE)
		kgen_clui_print_toggle_debug_stmt(file, groupmember);

	    else if (groupmember->group_next != NULL)
	    {
		kvf_get_attribute(groupmember->back_kformstruct, 
                                  KVF_OPTIONAL, &optional);
	 	print_group_debug_stmt(file, name, groupmember, 
				       optional);
	    }
	    else
	    {
                print_debug_stmt(file, groupmember);
	    }
            groupmember = groupmember->next;
        }
}

/*-----------------------------------------------------------
|
|  Routine Name: print_debug_stmt
|
|       Purpose: prints a single debug statement
|
|         Input: file      - open stream to *.c file
|                selection - selection corresponding to argument
|                            for which debug stmt is being generated.
|        Output: none
|       Returns: none
|    Written By: Danielle Argiro
|          Date: March 30, 1994
| Modifications: 
|
------------------------------------------------------------*/

static void print_debug_stmt(
   kfile      *file,
   kselection *selection)
{
	char *variable;
	variable = ktoken_to_string(selection->var_token);
	switch (selection->type)
	{
	    case KUIS_INPUTFILE:
	    case KUIS_OUTPUTFILE:
		 kfprintf(file, 
                 "\tif (clui_info->%s_file != NULL)\n", variable);
	     	 kfprintf(file, 
		 "\t    kfprintf(kstderr, \"%s_file = %%s\\n\", clui_info->%s_file);\n", 
		            variable, variable);
		 kfprintf(file, 
                 "\telse kfprintf(kstderr, \"%s_file = NULL\\n\");\n", variable);
		 break;

	    case KUIS_STRING:
	    case KUIS_STRINGLIST:
		 kfprintf(file, 
                 "\tif (clui_info->%s_string != NULL)\n", variable);
	     	 kfprintf(file, 
		 "\t    kfprintf(kstderr, \"%s_string = %%s\\n\", clui_info->%s_string);\n",
		            variable, variable);
		 kfprintf(file, 
                 "\telse kfprintf(kstderr, \"%s_string = NULL\\n\");\n", variable);
		 break;

	    case KUIS_LOGICAL:
	         kfprintf(file, 
		    "\tkfprintf(kstderr, \"%s_logic = %%d\\n\", clui_info->%s_logic);\n", 
		            variable, variable);
		 break;

	    case KUIS_INTEGER:
	         kfprintf(file, 
		    "\tkfprintf(kstderr, \"%s_int = %%d\\n\", clui_info->%s_int);\n", 
		            variable, variable);
		 break;

	    case KUIS_FLAG:
	         kfprintf(file, 
		    "\tkfprintf(kstderr, \"%s_flag = %%d\\n\", clui_info->%s_flag);\n", 
		            variable, variable);
		 break;

	    case KUIS_FLOAT:
	         kfprintf(file, 
		    "\tkfprintf(kstderr, \"%s_float = %%#g\\n\", clui_info->%s_float);\n", 
	                    variable, variable);
		 break;

	    case KUIS_DOUBLE:
	         kfprintf(file, 
		    "\tkfprintf(kstderr, \"%s_double = %%#g\\n\", clui_info->%s_double);\n", 
	                    variable, variable);
		 break;

	    case KUIS_LIST:
	    case KUIS_DISPLAYLIST:
	         kfprintf(file, 
		    "\tkfprintf(kstderr, \"%s_list = %%d\\n\", clui_info->%s_list);\n", 
	                    variable, variable);
		 kfprintf(file, 
                 "\tif (clui_info->%s_label != NULL)\n", variable);
	         kfprintf(file, 
		    "    \tkfprintf(kstderr, \"%s_label = %%s\\n\", clui_info->%s_label);\n", 
	                   variable, variable);
		 kfprintf(file, 
                 "\telse kfprintf(kstderr, \"%s_label = NULL\\n\");\n", variable);
		 break;

	    case KUIS_CYCLE:
	         kfprintf(file, 
		    "\tkfprintf(kstderr, \"%s_cycle = %%d\\n\", clui_info->%s_cycle);\n", 
	                    variable, variable);
		 kfprintf(file, 
                 "\tif (clui_info->%s_label != NULL)\n", variable);
	         kfprintf(file, 
		    "\tkfprintf(kstderr, \"%s_label = %%s\\n\", clui_info->%s_label);\n", 
	                   variable, variable);
		 kfprintf(file, 
                 "\telse kfprintf(kstderr, \"%s_label = NULL\\n\");\n", variable);
		 break;
	}

}  /* end print_debug_stmt */


/*-----------------------------------------------------------
|
|  Routine Name: print_mainhdr
|
|       Purpose: prints the header of the main program
|
|         Input: file        - open stream to *.c file
|                prog_spec   - pointer to program specification
|                name        - name of program being generated
|        Output: none
|
|    Written By: Danielle Argiro
|          Date: Dec 22, 1993
| Modifications:
|
------------------------------------------------------------*/

static void print_mainhdr(
    kfile    *file,
    kobject   object,
    char    **prog_spec,
    char     *name)
{
	char        date[KLENGTH];
        time_t      clck;
        struct tm  *localclock;
	kstring     short_description = NULL;


	if (!kcms_get_attribute(object,
				KCMS_CMOBJ_SHORT_DESC, &short_description)
	    || short_description == NULL)
	{
	    short_description = "<short description not found>";
	}

	/* 
	 * print routine name 
         */
	kfprintf(file, 
	"/*-----------------------------------------------------------\n|\n");
	kfprintf(file, "|  Routine Name: main() - %s\n", short_description);
	kfprintf(file, "|\n");

	/* 
	 * print purpose, input, output, returns
	 */
	kfprintf(file, "|       Purpose: main program for %s\n|\n", name);
	kfprintf(file, "|         Input:\n");
	print_mainhdr_args(file);

	kfprintf(file, "|        Output:\n");
	kfprintf(file, "|       Returns:\n|\n");

	/* 
	 * print author(s)
	 */
	kfprintf(file, "|    Written By: ");
	if (kstrlen(prog_spec[AUTHORS]) > 0) 
	  kfprintf(file, "%s", prog_spec[AUTHORS]);
        kfprintf(file, "\n");

	/* 
	 * print date
	 */
	kfprintf(file, "|          Date: ");
        clck = (time_t) time(NULL);
        localclock = (struct tm *) localtime((const time_t *) &clck);
        strftime(date, KLENGTH, "%h %d, %Y", localclock);
        kfprintf(file, "%s\n", date);

	/* 
	 * print modifications, end header
	 */
	kfprintf(file, "| Modifications:\n|\n");
	kfprintf(file, 
	 "------------------------------------------------------------*/\n\n");

}

/*-----------------------------------------------------------
|
|  Routine Name: print_mainhdr_args
|
|       Purpose: prints the command line arguments to the program
|                into the header of the main program
|
|         Input: file        - open stream to *.c file
|        Output: none
|
|    Written By: Danielle Argiro
|          Date: March 30, 1994
| Modifications:
|
------------------------------------------------------------*/

static void print_mainhdr_args(
    kfile *file)
{
        int        indx;
	kselection *selection;

	/* print out required command line arguments */
        indx = 0;
        while (indx < kgen_req_num)
	{
	    selection = kgen_req_sels[indx];
	    if (selection->type == KUIS_TOGGLE)
		 print_mainhdr_toggle_arg(file, selection, FALSE);
	    else if ((selection->type == KUIS_MUTEXCL) ||
		     (selection->type == KUIS_MUTINCL) ||
		     (selection->type == KUIS_GROUP))
		 print_mainhdr_group_arg(file, selection, FALSE);
	    else print_mainhdr_single_arg(file, selection, FALSE);
	    indx++;
	}

	/* print out optional command line arguments */
	indx = 0;
	while (indx < kgen_opt_num)
	{
	    selection = kgen_opt_sels[indx];
	    if (selection->type == KUIS_TOGGLE)
		 print_mainhdr_toggle_arg(file, selection, FALSE);
	    else if ((selection->type == KUIS_MUTEXCL) ||
		     (selection->type == KUIS_MUTINCL) ||
		     (selection->type == KUIS_GROUP))
		 print_mainhdr_group_arg(file, selection, FALSE);
	    else print_mainhdr_single_arg(file, selection, FALSE);
	    indx++;
	}
	
}

/*-----------------------------------------------------------
|
|  Routine Name: print_mainhdr_single_arg
|
|       Purpose: prints a single command line argument
|                in the input field of the main header
|
|         Input: file      - open stream to *.c file
|                selection - selection corresponding to argument
|                nested    - TRUE if group is nested inside another group
|        Output: none
|
|    Written By: Danielle Argiro
|          Date: March 30, 1994
| Modifications: 
|
------------------------------------------------------------*/

static void print_mainhdr_single_arg(
    kfile      *file,
    kselection *selection,
    int        nested)
{
	
	char *variable;
	char *description;
	char prefix[KLENGTH];

	set_prefix(nested, prefix);
        switch(selection->type)
        {
	    case KUIS_INPUTFILE:
            case KUIS_OUTPUTFILE:
	         variable = ktoken_to_string(selection->var_token);
	         kvf_get_attribute(selection->back_kformstruct,
			           KVF_DESCRIPTION, &description);
	         if (description == NULL) 
		    description = kstrdup("no description");
	         kfprintf(file, "%schar *clui_info->%s_", prefix, variable);
                 kfprintf(file, "file; {%s}\n", description);
	         kfprintf(file, "%sint   clui_info->%s_", prefix, variable);
                 kfprintf(file, "flag; {TRUE if -%s specified}\n|\n",
                                 variable);
                 break;
                          
            case KUIS_FLOAT:
	         variable = ktoken_to_string(selection->var_token);
	         kvf_get_attribute(selection->back_kformstruct,
			           KVF_DESCRIPTION, &description);
	         if (description == NULL) 
		    description = kstrdup("no description");
	         kfprintf(file, "%sfloat clui_info->%s_", prefix, variable);
                 kfprintf(file, "float; {%s}\n", description);
	         kfprintf(file, "%sint   clui_info->%s_", prefix, variable);
                 kfprintf(file, "flag; {TRUE if -%s specified}\n|\n",
                                 variable);
                 break;

            case KUIS_DOUBLE:
	         variable = ktoken_to_string(selection->var_token);
	         kvf_get_attribute(selection->back_kformstruct,
			           KVF_DESCRIPTION, &description);
	         if (description == NULL) 
		    description = kstrdup("no description");
	         kfprintf(file, "%sdouble clui_info->%s_", prefix, variable);
                 kfprintf(file, "double; {%s}\n", description);
	         kfprintf(file, "%sint    clui_info->%s_", prefix, variable);
                 kfprintf(file, "flag; {TRUE if -%s specified}\n|\n",
                                 variable);
                 break;

            case KUIS_INTEGER:
	         variable = ktoken_to_string(selection->var_token);
	         kvf_get_attribute(selection->back_kformstruct,
			           KVF_DESCRIPTION, &description);
	         if (description == NULL) 
		    description = kstrdup("no description");
	         kfprintf(file, "%sint clui_info->%s_", prefix, variable);
                 kfprintf(file, "int; {%s}\n", description);
	         kfprintf(file, "%sint clui_info->%s_", prefix, variable);
                 kfprintf(file, "flag; {TRUE if -%s specified}\n|\n",
                                 variable);
                 break;

            case KUIS_STRING:
            case KUIS_STRINGLIST:
	         variable = ktoken_to_string(selection->var_token);
	         kvf_get_attribute(selection->back_kformstruct,
			           KVF_DESCRIPTION, &description);
	         if (description == NULL) 
		    description = kstrdup("no description");
	         kfprintf(file, "%schar *clui_info->%s_", prefix, variable);
                 kfprintf(file, "string; {%s}\n", description);
	         kfprintf(file, "%sint   clui_info->%s_", prefix, variable);
                 kfprintf(file, "flag; {TRUE if -%s specified}\n|\n",
                                 variable);
                 break;

            case KUIS_LOGICAL:
	         variable = ktoken_to_string(selection->var_token);
	         kvf_get_attribute(selection->back_kformstruct,
			           KVF_DESCRIPTION, &description);
	         if (description == NULL) 
		    description = kstrdup("no description");
	         kfprintf(file, "%sint clui_info->%s_", prefix, variable);
                 kfprintf(file, "logic; {%s}\n", description);
	         kfprintf(file, "%sint clui_info->%s_", prefix, variable);
                 kfprintf(file, "flag; {TRUE if -%s specified}\n|\n",
                                 variable);
                 break;

            case KUIS_FLAG:
	         variable = ktoken_to_string(selection->var_token);
	         kvf_get_attribute(selection->back_kformstruct,
			           KVF_DESCRIPTION, &description);
	         if (description == NULL) 
		    description = kstrdup("no description");
	         kfprintf(file, "%sint clui_info->%s_", prefix, variable);
                 kfprintf(file, "flag; {TRUE if -%s specified}\n|\n",
                                 variable);
                 break;

            case KUIS_LIST:
            case KUIS_DISPLAYLIST:
	         variable = ktoken_to_string(selection->var_token);
	         kvf_get_attribute(selection->back_kformstruct,
			           KVF_DESCRIPTION, &description);
	         if (description == NULL) 
		    description = kstrdup("no description");
	         kfprintf(file, "%sint   clui_info->%s_list;", 
			  prefix, variable);
                 kfprintf(file, " {%s}\n", description);
                 kfprintf(file, "%schar *clui_info->%s_label;\n",
                          prefix, variable);
                 kfprintf(file, "%sint   clui_info->%s_flag;", 
			  prefix, variable);
                 kfprintf(file, " {TRUE if -%s specified}\n|\n",
                          variable);
		 break;

            case KUIS_CYCLE:
	         variable = ktoken_to_string(selection->var_token);
	         kvf_get_attribute(selection->back_kformstruct,
			           KVF_DESCRIPTION, &description);
	         if (description == NULL) 
		    description = kstrdup("no description");
                 kfprintf(file, "%sint   clui_info->%s_cycle;",
                          prefix, variable);
                 kfprintf(file, " {%s}\n", description);
                 kfprintf(file, "%schar *clui_info->%s_label;\n",
                          prefix, variable);
                 kfprintf(file, "%sint   clui_info->%s_flag;", 
			  prefix, variable);
                 kfprintf(file, " {TRUE if -%s specified}\n|\n",
                          variable);
		 break;

	    default:
		 break;
        }
}

/*-----------------------------------------------------------
|
|  Routine Name: print_mainhdr_group_arg
|
|       Purpose: prints an ME, MI, or loose group of command line arguments
|                in the input field of the main header
|
|         Input: file      - open stream to *.c file
|                selection - selection corresponding to toggle argument
|                nested    - TRUE if group is nested inside another group
|        Output: none
|
|    Written By: Danielle Argiro
|          Date: April 5, 1994
| Modifications:
|
------------------------------------------------------------*/

static void print_mainhdr_group_arg(
    kfile      *file,
    kselection *selection,
    int        nested)
{
	int        optional;
	kselection *group;
	char       prefix[KLENGTH];

	set_prefix(nested, prefix);

	if (selection->type == KUIS_MUTEXCL)
	{
	    kvf_get_attribute(selection->back_kformstruct,
			      KVF_OPTIONAL, &optional);
	    if (optional)
	        kfprintf(file, "%sOptional M.E. group:\n", prefix);
	    else kfprintf(file, "%sRequired M.E. group:\n", prefix);
	}
	else if (selection->type == KUIS_MUTINCL)
	    kfprintf(file, "%sM.I. group:\n", prefix);
	else if (selection->type == KUIS_GROUP)
	    kfprintf(file, "%sLoose group:\n", prefix);

	group = selection->group_next;
	while (group != NULL)
	{
	    if (group->type == KUIS_TOGGLE)
	        print_mainhdr_toggle_arg(file, group, nested+1);
	    else if ((group->type == KUIS_MUTEXCL) ||
		     (group->type == KUIS_MUTINCL) ||
		     (group->type == KUIS_GROUP))
		print_mainhdr_group_arg(file, group, nested+1);
	    else print_mainhdr_single_arg(file, group, nested+1);
	    group = group->next;
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: print_mainhdr_toggle_arg
|
|       Purpose: prints a toggle command line argument
|                in the input field of the main header
|
|         Input: file      - open stream to *.c file
|                selection - selection corresponding to toggle argument
|                nested    - TRUE if group is nested inside another group
|        Output: none
|
|    Written By: Danielle Argiro
|          Date: March 30, 1994
| Modifications:
|
------------------------------------------------------------*/

static void print_mainhdr_toggle_arg(
    kfile      *file,
    kselection *selection,
    int        nested)
{
	char *variable;
	char *description;
	int  toggle_type;
	char prefix[KLENGTH];

	set_prefix(nested, prefix);

	kvf_get_attributes(selection->back_kformstruct,
			   KVF_TOGGLE_TYPE, &toggle_type,
			   KVF_DESCRIPTION, &description,
			   NULL);

	if (description == NULL) 
	    description = kstrdup("no description");
	variable = ktoken_to_string(selection->var_token);
	switch(toggle_type) {

            case KUIS_INPUTFILE:
            case KUIS_OUTPUTFILE:
            case KUIS_STRING:
	         kfprintf(file, "%schar *clui_info->%s_toggle;", 
			  prefix, variable);
                 kfprintf(file, " {%s}\n", description);
                 kfprintf(file, "%sint   clui_info->%s_flag;", 
			  prefix, variable);
                 kfprintf(file, " {TRUE if -%s specified}\n|\n",
                          variable);
		 break;

            case KUIS_LOGICAL:
            case KUIS_FLAG:
            case KUIS_INTEGER:
	         kfprintf(file, "%sint clui_info->%s_toggle;", 
			  prefix, variable);
                 kfprintf(file, " {%s}\n", description);
                 kfprintf(file, "%sint clui_info->%s_flag;",
			  prefix, variable);
                 kfprintf(file, " {TRUE if -%s specified}\n|\n",
                          variable);
		 break;

            case KUIS_FLOAT:
	         kfprintf(file, "%sfloat clui_info->%s_toggle;", 
			  prefix, variable);
                 kfprintf(file, " {%s}\n", description);
                 kfprintf(file, "%sint   clui_info->%s_flag;", 
			  prefix, variable);
                 kfprintf(file, " {TRUE if -%s specified}\n|\n",
                          variable);
		 break;

            case KUIS_DOUBLE:
	         kfprintf(file, "%sdouble clui_info->%s_toggle;", 
			  prefix, variable);
                 kfprintf(file, " {%s}\n", description);
                 kfprintf(file, "%sint    clui_info->%s_flag;", 
			  prefix, variable);
                 kfprintf(file, " {TRUE if -%s specified}\n|\n",
                          variable);
		 break;

	}
}

/*-----------------------------------------------------------
|
|  Routine Name: set_prefix
|
|       Purpose: Simply fills out a buffer with the correct
|                number of leading spaces for the header of
|                the main routine, depending on nestig.
|         Input: nested - level of nesting
|        Output: buffer - buffer w/ correct # spaces
|       Returns: 
|    Written By: Danielle Argiro
|          Date: April 4, 1994
| Modifications:
|
------------------------------------------------------------*/

static void set_prefix(
    int  nested,
    char *buffer)
{
	buffer[0] = '\0';
	if (nested == 0)
	    ksprintf(buffer, "|\t\t");
	else if (nested == 1)
	    ksprintf(buffer, "|\t\t    ");
	else if (nested == 2)
	    ksprintf(buffer, "|\t\t\t");
	else kerror(KCODEGEN, "set_prefix", "bogus nested value passed in");
}

/*-----------------------------------------------------------
|  Routine Name: kstring_format - format a string
|
|       Purpose: This routine will take a string,
|		 and re-format it according to the specified line-length.
|
|         Input: istr         - The string to be formatted.
|                prepend      - A string which should be prepended to every
|                               line after formatting.  This can be used to
|	                        apply a standard indentation after formatting.
|                append       - A string which should be appended to every
|                               line, after formatting.
|                format_width - The resulting line width of the string.
|                               This width includes the `prepend' and `append'
|				strings.
|                ignore_cr    - A boolean flag which specifies whether
|				newline characters in the input string should
|				be ignored (treated as space characters), or
|				treated as hard newlines.
|        Output: ostr         - An optional output string.  If non-NULL, then
|				the formatted result will be stored in the
|				referenced string.
|				If this parameter is NULL, then the resulting
|				string will be allocated and returned.
|       Returns: A formatted message string, or NULL on error.
|
|    Written By: Steven Jorgensen
|          Date: Jan 14, 1993 21:18
------------------------------------------------------------*/
static kstring
my_kstring_format(
   kstring  istr,
   kstring  prepend,
   kstring  append,
   int      format_width,
   int      ignore_cr,
   kstring  ostr)
{
   kstring  new_message = NULL;
   kstring  ptr         = istr;
   kstring  last        = ptr;
   int      cnt         = 0;
   int      width       = format_width;
   kstring  tmp1;
   kstring  tmp2;


   if (kstrlen(istr) <= format_width)
      return kstrdup(istr);

   if (prepend != NULL)
      width -= kstrlen(prepend);
   if (append != NULL)
      width -= kstrlen(append);

   while (*ptr != '\0')
   {
      if (*ptr == '\n')
      {
	 tmp1 = kstring_ncat(new_message, last, -1, ptr - last + 1,NULL);
	 kfree(new_message);
	 new_message = tmp1;
	 tmp1 = NULL;
	 last = ptr +1;
	 cnt = -1;
      }
      else if (cnt > format_width)
      {
	 tmp2 = ptr;
	 while (tmp2 > last && !isspace(*tmp2))
	    tmp2--;
	 if (tmp2 != last)
	 {
	    tmp1 = kstring_ncat(new_message, last, -1, tmp2-last, NULL);
	    last = tmp2 + 1;
	 }
	 else
	 {
	    tmp1 = kstring_ncat(new_message, last, -1, cnt, NULL);
	    last = ptr+1;
	 }
	 tmp2 = kstring_cat(tmp1, "\n", NULL);
	 kfree(tmp1);
	 kfree(new_message);
	 new_message = tmp2;
	 cnt = -1;
      }
      ptr++;
      cnt++;
   }
   if (kstrlen(last))
   {
      tmp1 = kstring_cat(new_message, last, NULL);
      kfree(new_message);
      new_message = tmp1;
   }
   return new_message;
}
