/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>         Utility routines
   >>>>
   >>>>  Private:
   >>>>             none
   >>>>   Public:
   >>>>             kstrstr()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"

static char *reserved_tbnames[] =
{
   "toolbox", "object", "gis", "home", "user", "term", "termcap", "pwd",
   "mail", "host", "shell", "path", "display"
};
 
static char *reserved_onames[] =
{
   "toolbox", "object", "usage"
};

typedef struct
{
   int id;
   kstring path;
}
Template;

static Template templates[] =
{
   {KCMS_TEMPLATE_TODO,            "info/todo"},
   {KCMS_TEMPLATE_BUGS,            "info/bugs"},
   {KCMS_TEMPLATE_DONE,            "info/done"},
   {KCMS_TEMPLATE_CHANGELOG,       "info/changelog"},
   {KCMS_TEMPLATE_UIS_PANE,        "uis/template.pane"},
   {KCMS_TEMPLATE_UIS_FORM,        "uis/template.form"},
   {KCMS_TEMPLATE_SRC_LIBRARY,     "src/newlib.c"},
   {KCMS_TEMPLATE_SRC_PROGRAM,     "src/program.c"},
   {KCMS_TEMPLATE_SRC_INCLUDE,     "src/template.h"},
   {KCMS_TEMPLATE_FORTRAN_LIBRARY, "src/newlib.f"},
   {KCMS_TEMPLATE_FORTRAN_PROGRAM, "src/program.f"},
   {KCMS_TEMPLATE_MISC,            "misc/template.misc"},
   {KCMS_TEMPLATE_DOC,             "doc/template.doc"},
   {KCMS_TEMPLATE_HLP,             "doc/template.hlp"},
   {KCMS_TEMPLATE_MANUAL,          "doc/manual"},
   {KCMS_TEMPLATE_INDEX,           "doc/index"},
   {KCMS_TEMPLATE_GLOSSARY,        "doc/glossary"},
   {KCMS_TEMPLATE_INFO_FILE,       "toolbox/info-file"},
   {KCMS_TEMPLATE_CONFIG_FILE,     "toolbox/toolbox.def"},
   {KCMS_TEMPLATE_TB_INCLUDE_FILE, "toolbox/toolbox.h"},
   {KCMS_TEMPLATE_TB_ALIAS_FILE,   "toolbox/aliases"},
   {KCMS_TEMPLATE_TB_TODO,         "toolbox/todo"},
   {KCMS_TEMPLATE_TB_CHANGELOG,    "toolbox/changelog"}
};

#define TEMPLATE_DIRECTORY	"$BOOTSTRAP/objects/library/kcms/templates"

/************************************************************
* Routine Name:	kcms_add_keyword - add word to an object's keyword list
*
* Purpose:	This function is used to add a word to the keyword list
*		associated with a toolbox or software object.
*		This is a convenience function for manipulating the
*		\f(CW\s-2KCMS_TB_KEYWORDS\s+2\fP and
*		\f(CW\s-2KCMS_CMOBJ_KEYWORDS\s+2\fP attributes.
*
* Input:	object  - A toolbox or software object.
*		keyword - The word to be added to the object's keyword list.
*
* Returns:	TRUE (1) if the keyword was successfully added,
*		FALSE (0) otherwise.
* Written By:	Neil Bowers
* Date:		22-jun-94
*************************************************************/
int
kcms_add_keyword(
   kobject  object,
   kstring  keyword)
{
   kstring   routine      = "kcms_add_keyword()";
   int       result       = TRUE;
   int       object_type;
   klist    *list;
   klist    *newlist;
   kstring   word         = kstrdup(keyword);
   int       token;


   if (!kcms_get_attribute(object, KCMS_TYPE, &object_type) || word == NULL)
      return FALSE;

   switch (object_type)
   {
      case KOBJ_CMSTB:
	 if (!kcms_get_attribute(object, KCMS_TB_KEYWORDS, &list))
	    result = FALSE;
	 else
	 {
	    token = kstring_to_token(word);
	    newlist = klist_add(list, (kaddr)token, (kaddr)word);
	    if (!kcms_set_attribute(object, KCMS_TB_KEYWORDS, newlist))
	       result = FALSE;
	 }
	 break;

      case KOBJ_CMSOBJ:
	 if (!kcms_get_attribute(object, KCMS_CMOBJ_KEYWORDS, &list))
	    result = FALSE;
	 else
	 {
	    token = kstring_to_token(word);
	    newlist = klist_add(list, (kaddr)token, (kaddr)word);
	    if (!kcms_set_attribute(object, KCMS_CMOBJ_KEYWORDS, newlist))
	       result = FALSE;
	 }
	 break;

      case KOBJ_CMSFILE:
	 kerror(NULL, routine,
		"Keywords cannot be specified for a file object.");
	 result = FALSE;
	 break;

      default:
	 kerror(NULL, routine, "Invalid object typed passed.");
	 result = FALSE;
	 break;
   }

   return result;
}

/************************************************************
* Routine Name:	kcms_remove_keyword - remove word from an object's keyword list
*
* Purpose:	This function is used to add a word to the keyword list
*		associated with a toolbox or software object.
*		This is a convenience function for manipulating the
*		\f(CW\s-2KCMS_TB_KEYWORDS\s+2\fP and
*		\f(CW\s-2KCMS_CMOBJ_KEYWORDS\s+2\fP attributes.
*
* Input:	object  - A toolbox or software object.
*		keyword - The word to be removed from the object's
*			  keyword list.
*
* Returns:	TRUE (1) if the keyword was successfully added,
*		FALSE (0) otherwise.
* Written By:	Neil Bowers
* Date:		22-jun-94
*************************************************************/
int
kcms_remove_keyword(
   kobject  object,
   kstring  keyword)
{
   kstring   routine      = "kcms_remove_keyword()";
   int       result       = TRUE;
   int       object_type;
   klist     *list;
   klist     *newlist;
   int       token;


   if (!kcms_get_attribute(object, KCMS_TYPE, &object_type))
      return FALSE;

   switch (object_type)
   {
      case KOBJ_CMSTB:
	 if (!kcms_get_attribute(object, KCMS_TB_KEYWORDS, &list))
	    result = FALSE;
	 else
	 {
	    token = kstring_to_token(keyword);
	    newlist = klist_delete(list, (kaddr)token);
	    if (!kcms_set_attribute(object, KCMS_TB_KEYWORDS, newlist))
	       result = FALSE;
	 }
	 break;

      case KOBJ_CMSOBJ:
	 if (!kcms_get_attribute(object, KCMS_CMOBJ_KEYWORDS, &list))
	    result = FALSE;
	 else
	 {
	    token = kstring_to_token(keyword);
	    newlist = klist_delete(list, (kaddr)token);
	    if (!kcms_set_attribute(object, KCMS_CMOBJ_KEYWORDS, newlist))
	       result = FALSE;
	 }
	 break;

      case KOBJ_CMSFILE:
	 kerror(NULL, routine,
		"File objects cannot have keywords associated with them.");
	 result = FALSE;
	 break;

      default:
	 kerror(NULL, routine, "Invalid object typed passed.");
	 result = FALSE;
	 break;
   }

   return result;
}

/************************************************************
* Routine Name:	kcms_set_bit - set a bit in a flags attribute
*
* Purpose:	This function is used to set a single bit in an object
*		flag attribute.
*
* Input:	object    - A toolbox or software object.
*		attribute - The word to be removed from the object's
*			    keyword list.
*		bitid     - An identifier for the bit to set.
*
* Returns:	TRUE (1) if the keyword was successfully added,
*		FALSE (0) otherwise.
* Written By:	Neil Bowers
* Date:		28-jan-1995
*************************************************************/
int
kcms_set_bit(
   kobject        object,
   int            attribute,
   unsigned long  bitid)
{
   unsigned long  ulValue;


   if (!kcms_get_attribute(object, attribute, &ulValue))
      return FALSE;

   return kcms_set_attribute(object, attribute, (ulValue | bitid));
}

/************************************************************
* Routine Name:	kcms_clear_bit - set a bit in a flags attribute
*
* Purpose:	This function is used to set a single bit in an object
*		flag attribute.
*
* Input:	object    - A toolbox or software object.
*		attribute - The word to be removed from the object's
*			    keyword list.
*		bitid     - An identifier for the bit to set.
*
* Returns:	TRUE (1) if the keyword was successfully added,
*		FALSE (0) otherwise.
* Written By:	Neil Bowers
* Date:		28-jan-1995
*************************************************************/
int
kcms_clear_bit(
   kobject        object,
   int            attribute,
   unsigned long  bitid)
{
   unsigned long  ulValue;


   if (!kcms_get_attribute(object, attribute, &ulValue))
      return FALSE;

   return kcms_set_attribute(object, attribute, (ulValue & ~bitid));
}

/************************************************************
* Routine Name:	kcms_query_bit - set a bit in a flags attribute
*
* Purpose:	This function is used to set a single bit in an object
*		flag attribute.
*
* Input:	object    - A toolbox or software object.
*		attribute - The word to be removed from the object's
*			    keyword list.
*		bitid     - An identifier for the bit to set.
*               bitvalue  - A pointer to an integer.  If the query
*			    is successful, then the referenced integer
*			    will be set to TRUE if the bit is set,
*			    FALSE if not.
*
* Returns:	TRUE (1) if the keyword was successfully added,
*		FALSE (0) otherwise.
* Written By:	Neil Bowers
* Date:		28-jan-1995
*************************************************************/
int
kcms_query_bit(
   kobject         object,
   int             attribute,
   unsigned long   bitid,
   int            *bitvalue)
{
   unsigned long  ulValue;


   if (!kcms_get_attribute(object, attribute, &ulValue))
      return FALSE;

   *bitvalue = ((ulValue & bitid) == bitid);
   return TRUE;
}


/************************************************************
* Routine Name:	kcms_get_date - get the current time and date in a string
*
* Purpose:	This function determines the current time and date,
*		and returns a fixed format string.
*
* Input:	object    - A toolbox or software object.
*		attribute - The word to be removed from the object's
*			    keyword list.
*		bitid     - An identifier for the bit to set.
*               bitvalue  - A pointer to an integer.  If the query
*			    is successful, then the referenced integer
*			    will be set to TRUE if the bit is set,
*			    FALSE if not.
*
* Returns:	A string containing the current time and date,
*		in the format used by all the tools.
* Written By:	Neil Bowers
* Date:		23-mar-1995
*************************************************************/
kstring
kcms_get_date(
   void)
{
   char        buffer[KLENGTH];
   struct tm  *curtime;
   time_t      timeval;


   timeval = time(NULL);
   if (timeval == (time_t)-1)
      return NULL;

   curtime = localtime(&timeval);
   if (curtime == NULL)
      return NULL;

   if (kstrftime(buffer, KLENGTH, KCMS_DATE_FORMAT, curtime) == 0)
      return NULL;

   return kstrdup(buffer);
}

/*-----------------------------------------------------------
| Routine Name:	kcms_decode_yesno - decode a yes or no string
|
| Purpose:	This routine translates a "yes" into the logical
|		value TRUE, and a "no" into the logical FALSE.
|
| Input:	yesnostr - the string containing yes or no
|
| Returns:	TRUE  - if the string is "yes"
|		FALSE - if the string is "no"
|		-1 on error
|
| Written By:	Steven Jorgensen
| Date:		29-oct-92
------------------------------------------------------------*/
int
kcms_decode_yesno(
   kstring  yesnostr)
{
   if (kstrcasecmp(yesnostr, "yes") == 0)
      return TRUE;
   if (kstrcasecmp(yesnostr, "no") == 0)
      return FALSE;
   return -1;
}

/*-----------------------------------------------------------
| Routine Name:	kstrip_header - string source code '*' garbage out of
|		a string
|
| Purpose:	This routine removes the '[ \t]*[|*][ \t]*' that follows the
|		'\n' in text that was retrieved from a source code header.
|
| Input:	str - string to clean up
|
| Output:	
|
| Returns:	The cleaned up string
|
| Written By:	Steven Jorgensen
| Date:		Oct 22, 1992 22:44
------------------------------------------------------------*/
char *
kstrip_header(
   kstring  str)
{
   kstring  new;
   kstring  sstr;
   kstring  ret;


   if (str == NULL)
      return NULL;

   sstr = kregex_replace(str, "\n[ \t]*[|*][ \t]*", "\n", NULL);
   new = kstring_replace(sstr, "\n!", "\n ", NULL);
   ret = kstring_cleanup(new, NULL);
   kfree(sstr);
   kfree(new);
   return (ret);
}

/*-----------------------------------------------------------
| Routine Name:	close_n_free - calls kcms_close on the client data before kfree
|
| Purpose:	This routine calls kcms_close on the client data
|		field before freeing the list structure.
|
| Input:	list - a list entry to be free'd
| Output:	
| Returns:	none
|
| Written By:	Steven Jorgensen
| Date:		May 31, 1993 15:47
------------------------------------------------------------*/
void
close_n_free(
   klist  *list)
{
   kcms_close((kobject) klist_clientdata(list));
   list->client_data = NULL;
}

/*-----------------------------------------------------------
| Routine Name:	partclose_n_free - calls kcms_cmobj_partial_close on the
|		client data before kfree
|
| Purpose:	This routine calls kcms_cmobj_partial_close on the client data
|		field before freeing the list structure.
|
| Input:	list - a list entry to be free'd
| Output:	
| Returns:	none
|
| Written By:	Steven Jorgensen
| Date:		May 31, 1993 15:47
------------------------------------------------------------*/
void
partclose_n_free(
   klist  *list)
{
   kcms_cmobj_partial_close((kobject) klist_clientdata(list));
   list->client_data = NULL;
}

/*-----------------------------------------------------------
| Routine Name:	obj_list_to_string - convert a klist of objects into
|		a string separated by '\n's
|
| Purpose:	This routine will take a single string piece of information
|		specified by the attr paramter from a list of objects, and
|		create a single string of these pieces separated by '\n's.
|
| Input:	list   - the list of kobjects
|		attr   - the attribute to get from the object
| Output:	string - a pre-alloced string to store the result in
| Written By:	Steven Jorgensen
| Date:		Aug 11, 1993
------------------------------------------------------------*/
void
obj_list_to_string(
   klist    *list,
   int       attr,
   kstring   string)
{
   char     *tmp;
   kobject   object;


   for (; list != NULL; list = klist_next(list))
   {
      object = (kobject) klist_clientdata(list);
      if (!kcms_get_attribute(object, attr, &tmp))
	 return;
      kstring_3cat(string, tmp, "\n", string);
   }
}

/*-----------------------------------------------------------
| Routine Name:	include_list_to_string - convert a klist of objects into
|		a string separated by '\n's
|
| Purpose:	This routine is a kludge to get round a current
|		restriction in kcms, whereby public include files are
|		`interpretive', being sucked into the object structure at
|		runtime, and are not stored in the database.
|
| Input:	list   - the list of kobjects
|		attr   - the attribute to get from the object
| Output:	string - a pre-alloced string to store the result in
| Returns:	
| Written By:	Steven Jorgensen
| Date:		Aug 11, 1993
------------------------------------------------------------*/
void
include_list_to_string(
   kobject   object,
   klist    *list,
   char     *string)
{
   kstring  tmp;
   kobject  fobj;
   kstring  opath;


   kcms_get_attribute(object, KCMS_PATH, &opath);
   for (; list != NULL; list = klist_next(list))
   {
      fobj = (kobject) klist_clientdata(list);
      kcms_get_attribute(fobj, KCMS_PATH, &tmp);
      if (!kstrncmp(tmp, opath, kstrlen(opath)))
	 kstring_3cat(string, tmp, "\n", string);
   }
}


/*-----------------------------------------------------------
| Routine Name:	kcms_create_toplevel - change a relative path to a direct path
|
| Purpose:	This routine changes any relative path (i.e starting with
|		'./' or '../') to a direct path.  It also changes
|		direct paths to $TBNAME/path if possible.
| Input:	toolbox - the toolbox object
|		path    - the path to convert
| Output:	outpath - the outpath to store the result.. if NULL, it is
|		malloc'ed and returned
| Returns:	a pointer to the output path
| Written By:	Steven Jorgensen
| Date:		20-nov-92
------------------------------------------------------------*/
kstring
kcms_create_toplevel(
   kobject  toolbox,
   kstring  path,
   kstring  outpath)
{
   kstring  routine = "kcms_create_toplevel()";
   char     fullpath[KLENGTH];
   char     upper[KLENGTH];
   char     temp[KLENGTH];
   kstring  epath;
   kstring  tpath;
   kstring  ptr;


   if (path == NULL)
      return NULL;
   if (!kcms_get_attribute(toolbox, KCMS_PATH, &tpath) || tpath == NULL)
   {
      kerror(KCMS, routine, "Cannot determine toolbox path.\n");
      return NULL;
   }
   if (tpath[0] == '~' || tpath[0] == '.')
      tpath = kexpandpath(tpath, NULL, NULL);
   else
      tpath = kfullpath(tpath, NULL, NULL);
   if (tpath == NULL)
      return NULL;

   ptr = NULL;
   if (((unsigned int)tpath[kstrlen(tpath) - 1]) == '/')
      tpath[kstrlen(tpath) - 1] = '\0';

   if (path[0] == '~' || path[0] == '.')
      epath = kexpandpath(path, NULL, NULL);
   else
      epath = kfullpath(path, NULL, NULL);
   if (epath == NULL)
      return NULL;

   /* replace fullpath with $NAME if possible */
   if (kstrncmp(epath, tpath, kstrlen(tpath)) == 0)
   {
      (void)kcms_get_attribute(toolbox, KCMS_NAME, &ptr);
      (void)kstring_upper(ptr, temp);
      (void)kstring_cat("$", temp, upper);
      kstring_replace(epath, tpath, upper, fullpath);
   }
   else if (((unsigned int)path[0]) == '.')
      kstrcpy(fullpath, epath);
   else
      kstrcpy(fullpath, path);
   kfree(tpath);
   kfree(epath);
   if (outpath == NULL)
      return kstrdup(fullpath);
   kstrcpy(outpath, fullpath);
   return (outpath);
}

/************************************************************
* Routine Name:	kcms_legal_identifier - check string for legality as an id
*
* Purpose:	This function takes a string and checks to see whether
*		it contains any illegal characters with respect to using
*		the string as an identifier in kcms, such as an object name.
*
* Input:	string - The string to be checked for validity.
*		kotype - The type of kcms object that the identifier will
*			 be used for.  Legal values are
*		.TS H
*		center tab(:) ;
*		lfB | lfB
*		lf(CW) | l .
*		kotype:Object type
*		=
*		.TH
*		KOBJ_CMSTB:Toolbox object
*		KOBJ_CMSOBJ:Software object
*		KOBJ_CMSOBJ:File object
*		.TE
*
* Returns:	TRUE (1) if the string is acceptable, FALSE (0) otherwise.
*
* Written By:	Neil Bowers
* Date:		3-dec-93
*************************************************************/
int
kcms_legal_identifier(
   kstring  string,
   int      kotype)
{
   register char   *pc        = string;
   char           **reserved  = NULL;
   int              nreserved = 0;
   int              i;
   int              islegal   = TRUE;
 

   if (string == NULL || kstrlen(string) == 0)
      return FALSE;

   if (!isalpha(*string))
      islegal = FALSE;
   else
   {
      while (*pc != '\0' && islegal)
      {
	 if (isalnum((int)*pc) || *pc == '_')
	    pc++;
	 else
	    islegal = FALSE;
      }
   }
 
   /*-- check whether it is a reserved word ---------------------*/
 
   if (islegal)
   {
      if (kotype == KOBJ_CMSTB)
      {
	 reserved        = reserved_tbnames;
	 nreserved       = knumber(reserved_tbnames);
      }
      else if (kotype == KOBJ_CMSOBJ)
      {
	 reserved        = reserved_onames;
	 nreserved       = knumber(reserved_onames);
      }
 
      for (i=0; i<nreserved && islegal; i++)
	 if (!kstrcasecmp(reserved[i],string))
	    islegal = FALSE;
   }
 
   return islegal;
}

/************************************************************
* Routine Name:	kcms_query_template - get path to a specified template file
*
* Purpose:	This function takes an object and a `template identifier',
*		and returns a string which contains the path to the
*		required template.
*
*		This function is a private function, and is likely to
*		be replaced with a more general template mechanism.
*
* Input:	object	- The object which the template is required for.
*		id	- A template identifier.
*			  The following templates are currently available:
*		.TS H
*		tab(:) ;
*		lfB | lfB
*		lp7f(CW) | l .
*		Template Identifier:Template file
*		=
*		.TH
*		KCMS_TEMPLATE_TODO:to do file
*		KCMS_TEMPLATE_BUGS:bugs file
*		KCMS_TEMPLATE_DONE:done file
*		KCMS_TEMPLATE_CHANGELOG:changelog
*		KCMS_TEMPLATE_UIS_PANE:pane UIS file
*		KCMS_TEMPLATE_UIS_FORM:form UIS file
*		KCMS_TEMPLATE_SRC_LIBRARY:library source file
*		KCMS_TEMPLATE_SRC_PROGRAM:program source file
*		KCMS_TEMPLATE_SRC_INCLUDE:include file
*		KCMS_TEMPLATE_FORTRAN_LIBRARY:fortran library source
*		KCMS_TEMPLATE_FORTRAN_PROGRAM:fortran program source
*		KCMS_TEMPLATE_MISC:miscellaneous file
*		KCMS_TEMPLATE_DOC:online documentation
*		KCMS_TEMPLATE_HLP:help page
*		KCMS_TEMPLATE_MANUAL:manual chapter toplevel
*		KCMS_TEMPLATE_INDEX:index chapter toplevel
*		KCMS_TEMPLATE_GLOSSARY:glossary chapter toplevel
*		KCMS_TEMPLATE_INFO_FILE:toolbox info file
*		KCMS_TEMPLATE_CONFIG_FILE:imake config file
*		.TE
*
* Returns:	A string containing the path to the template,
*		or NULL on failure.
*
* Written By:	Neil Bowers
* Date:		9-may-94
*************************************************************/
kstring
kcms_query_template(
   kobject  object,
   int      id)
{
   int  cmstype;
   int  tnum;
   int  otype;


   if (!kcms_get_attribute(object, KCMS_TYPE, &cmstype))
      return NULL;

   if (cmstype == KOBJ_CMSOBJ)
   {
      if (!kcms_get_attribute(object, KCMS_CMOBJ_PROGTYPE, &otype))
	 return NULL;

      if (id == KCMS_TEMPLATE_TODO && otype == KCMS_PANE)
	 return kstring_cat(TEMPLATE_DIRECTORY, "/info/pane.todo", NULL);
   }

   for (tnum = 0; tnum < knumber(templates); tnum++)
   {
      if (templates[tnum].id == id)
	 return kstring_3cat(TEMPLATE_DIRECTORY, "/",
			     templates[tnum].path, NULL);
   }

   return NULL;
}

/*-----------------------------------------------------------
| Routine Name:	kcms_valid_object_path - check whether path is in an object
| Purpose:	This function is used to check whether a given full path is
|		a valid path to within an object's directory structure.
|
| Input:	path - the full path to be checked
| Returns:	TRUE (1) if the path does correspond to an object,
|		FALSE (0) otherwise.
| Written By:	Neil Bowers
| Date:		22-jun-94
------------------------------------------------------------*/
kbool
kcms_valid_object_path(
   kstring  path)
{
   kstring    tbpath;
   char     **items;
   int        nitems;
   kobject    toolbox;
   kobject    object;


   if ((tbpath = ktbpath(path, NULL)) == NULL)
      return FALSE;

   items = kparse_string_delimit(tbpath, "/", KDELIM_CLEAN, &nitems);
   if (items == NULL)
      return FALSE;

   if (nitems < 4 || *items[0] != '$' || kstrcmp(items[1],"objects"))
      return FALSE;

   if (kcms_attr_string2int(KCMS_CMOBJ_PROGTYPE, items[2]) < 1)
      return FALSE;

   if ((toolbox = kcms_open_toolbox(items[0]+1)) == NULL)
      return FALSE;

   if ((object = kcms_open_cmobj(toolbox, items[3])) == NULL)
   {
      kcms_close(toolbox);
      return FALSE;
   }

   kcms_close(object);
   kcms_close(toolbox);
   return TRUE;
}
