/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            The Public API for KCMS
   >>>>
   >>>>  Private:
   >>>>
   >>>>   Static:
   >>>>
   >>>>   Public:
   >>>>         kcms_open_cmobj
   >>>>         kcms_open_toolbox
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"
#include <kcms/private.h>


struct _kobject
{
   int         type;
   int         attribute_cnt;
   AttrTbl    *attributes;
   int        (*add_sub_obj) PROTO((kobject, kobject));
   int        (*del_sub_obj) PROTO((kobject, kobject));
   int        (*mod_sub_obj) PROTO((kobject, kobject, kobject *));
   kobject     parent;

   klist      *callbacks;
};

typedef struct _kobject ko_inter;

struct rout_tbl
{
   int      type;
   int      (*close)     PROTO((kobject));
   int      (*sync)      PROTO((kobject));
   kobject  (*duplicate) PROTO((kobject, kobject));
   int      (*destroy)   PROTO((kobject));
   kbool    (*rename)    PROTO((kobject, kstring));
};

static struct rout_tbl methods[] =
{
   {0, NULL, NULL, NULL, NULL, NULL},

   {KOBJ_DATA, NULL, NULL, NULL, NULL, NULL},

   {KOBJ_TRANSPORT, NULL, NULL, NULL, NULL, NULL},

   {KOBJ_CMSTB,
    kcms_tb_close,
    kcms_tb_sync,
    NULL,
    kcms_tb_destroy,
    NULL                   /*-- rename --*/
   },

   {KOBJ_CMSOBJ,
    kcms_cmobj_close,
    kcms_cmobj_sync,
    kcms_cmobj_duplicate,
    kcms_cmobj_destroy,
    kcms_cmobj_rename
   },

   {KOBJ_CMSFILE,
    kcms_fobj_free,
    NULL,			/* sync */
    kcms_fobj_duplicate,
    kcms_fobj_destroy,
    NULL
   },
};

static int _kcms_check_method PROTO((kfunc_void, char *, char *));
static void kcms_call_callbacks PROTO((kobject, int));

/************************************************************
* Routine Name:	kcms_create_fileobj - create a new kcms file object
*
* Purpose:	This routine allocates and initializes a kcms file object
*		structure for a new file on disk.  If the file already
*		exists, this routine will act the same as kcms_open_fileobj()
*
* Input:	parent	 - a kobject with the parent object in it
*		filename - A relative or fully specified filename
*		path	 - The $TOOLBOX/... part of the fullname path;
*			   set to NULL if name uses the fullname path.
*		ftype	 - The file type.  Preprocessor symbols for
*			   legal file types can be found in the include
*			   file "kcms/fileobj.h"
*
*			   Note, if type is set to -1, this routine will
*			   try to figure out the type and subtype given
*			   the path and relative path.
*
*		subtype	 - The subtype of the file depending on type.
*			   Preprocessor symbols for legal file subtypes
*			   can be found in the include
*			   file "kcms/fileobj.h"
*
*				Note, if type is set to -1, this value is
*				ignored.
*
*		source   - A flag indicating which tool generated it.
*			   Preprocessor symbols for legal source flags
*			   can be can be found in the include
*			   file "kcms/fileobj.h"
*
*		perms    - a flag indicating the user permissions on
*			   the file it can be one of:
*		! KCMS_FACCESS_RDWR   : User can read and write the file
*		! KCMS_FACCESS_RDONLY : User can only read the file
* Output:	
* Returns:	The newly created file object, or NULL on failure.
*
* Written By:	Steven Jorgensen and Neil Bowers
* Date:		16-may-93
*************************************************************/
kobject
kcms_create_fileobj(
   kobject  parent,
   kstring  filename,
   kstring  path,
   int      ftype,
   int      subtype,
   int      source,
   int      perms)
{
   kobject        fobj;
   int            val;
   unsigned long  update;
   int            cmstype;


   fobj = kcms_open_fileobj(parent, filename, path, ftype, subtype,
			    source, perms);
   if (fobj != NULL && !fobj->parent->add_sub_obj(fobj->parent, fobj))
   {
      val = errno;
      (void)kcms_close(fobj);
      fobj = NULL;
      errno = val;
   }
   if (kcms_get_attribute(parent, KCMS_TYPE, &cmstype))
   {
      if (cmstype == KOBJ_CMSOBJ)
      {
	 if (kcms_get_attribute(parent, KCMS_CMOBJ_UPDATE_DB, &update))
	 {
	    update |= KCMS_UPDATE_SYNC;
	    if (ftype == KCMS_FOBJ_TYPE_SRC)
	       update |= KCMS_UPDATE_REGEN;
	    kcms_set_attribute(parent, KCMS_CMOBJ_UPDATE_DB, update);
	 }
      }
      else
      {
	 if (kcms_get_attribute(parent, KCMS_TB_UPDATE_FLAG, &update))
	 {
	    update |= KCMS_UPDATE_SYNC;
	    kcms_set_attribute(parent, KCMS_TB_UPDATE_FLAG, update);
	 }
      }
   }

   return fobj;
}

/************************************************************
* Routine Name:	kcms_vset_attribute - set attribute from varargs list
* Purpose:	This function is used to set a single attribute when the
*		attribute arguments are already available in a varargs list.
* Input:	object    - The kcms object the attribute is associated with.
*		attribute - The identifier for the attribute we want to set.
*		valist    - The varargs list which contains the new value
*			    for the specified attribute.
* Returns:	TRUE (1) if the attribute was successfully modified,
*		FALSE (0) otherwise.
* Written By:	Neil Bowers
* Date:		29-jun-94
*************************************************************/
int
kcms_vset_attribute(
   kobject    object,
   int        attribute,
   kva_list  *valist)
{
   int  result;

   if (object->attributes[attribute].set_attr != NULL)
   {
      result = object->attributes[attribute].set_attr(object,attribute,valist);
      if (result && object->callbacks != NULL)
	 kcms_call_callbacks(object, attribute);
      return result;
   }

   return FALSE;
}

/************************************************************
* Routine Name:	kcms_vget_attribute - get attribute from varargs list
* Purpose:	This function is used to get a single attribute value when the
*		attribute arguments are already available in a varargs list.
* Input:	object    - The kcms object the attribute is associated with.
*		attribute - The identifier for the attribute we want to get.
*		valist    - The varargs list which location into which we
*			    want to store the attribute value.
* Returns:	TRUE (1) if the attribute was successfully modified,
*		FALSE (0) otherwise.
* Written By:	Neil Bowers
* Date:		29-jun-94
*************************************************************/
int
kcms_vget_attribute(
   kobject    object,
   int        attribute,
   kva_list  *valist)
{
   return (object->attributes[attribute].get_attr != NULL
	   ? object->attributes[attribute].get_attr(object,attribute, valist)
	   : FALSE);
}

/************************************************************
* Routine Name:	kcms_get_attribute - get a single attribute from kcms object
*
* Purpose:	This function is used to retrieve the value of an attribute
*		from a kcms object.
*		All kcms objects have a \f(CW\s-1KCMS_NAME\s+2\fP attribute,
*		which is of type \f(CW\s-1kstring\s+2\fP.
*		Given any kcms object, the following code will determine
*		the object's name:
*		!\f(CW\s-2
*		!   kobject  object;
*		!   kstring  name;
*		!
*		!   kcms_get_attribute(object, KCMS_NAME, &name);
*		!   kprintf("Object's name is %s.\n", name);
*		!\s+2\fP
*
* Input:	object  - the object on which to get an attribute
*		kvalist - variable argument list, in the form:
*		! <\fIattribute-identifier\fP\^>, &arg1, [ &arg2, ... ]
* Output:	
* Returns:	TRUE (1) on success, FALSE (0) otherwise
* Written By:	Steven Jorgensen
* Date:		26-apr-93
*************************************************************/
int 
kcms_get_attribute(
   kobject  object,
   kvalist)
{
   kstring    routine    = "Get Attribute";
   kva_list   valist;
   int	      attribute;
   int	      status;


   errno = KCMS_OK;
   if (!kcms_legal_object(object,routine))
      return FALSE;

   kva_start(valist,object);

   attribute = kva_arg(valist, int);
   if (attribute < 0 || attribute > object->attribute_cnt)
   {
      errno = KCMS_INVALIDATTRIBUTE;
      kerror(KCMS,routine,"Invalid attribute specified");
      kva_end(valist);
      return FALSE;
   }
   status = kcms_vget_attribute(object, attribute, &valist);
   kva_end(valist);

   return status;
}

/************************************************************
* Routine Name:	kcms_get_attributes - get multiple attributes from kcms object
*
* Purpose:	This function is used to get one or more attributes from
*		a kcms object.
*
*		Given any kcms object, the following code will determine
*		the object's type, name, and path:
*		!\f(CW\s-2
*		!   kobject  object;
*		!   int      type;
*		!   kstring  path;
*		!   kstring  name;
*		!
*		!   kcms_get_attributes(object,
*		!                       KCMS_TYPE, &type,
*		!                       KCMS_NAME, &name,
*		!                       KCMS_PATH, &path,
*		!                       KCMS_END);
*		!\s+2\fP
*		\fBNote:\fP the argument list must be terminated
*		with the symbol \f(CW\s-1KCMS_END\s+1\fP.
*
* Input:	object  - A valid kcms object.
*		kvalist - A variable argument list, in the form:
*		! ATTRIBUTE_NAME, &value,
*		! ATTRIBUTE_NAME, &value,
*		The attribute list must be terminated with the symbol
*		! KCMS_END
*		to signify the end of the variable argument list.
*
* Returns:	TRUE (1) on success, FALSE (0) otherwise
* Written By:	Neil Bowers
* Date:		29-jun-94
*************************************************************/
int
kcms_get_attributes(
   kobject  object,
   kvalist)
{
   kstring   routine     = "Get Attribute";
   int       status	 = TRUE;
   kva_list  valist;
   int       attribute;


   errno = KCMS_OK;
   if (!kcms_legal_object(object,routine))
      return FALSE;

   kva_start(valist,object);

   attribute = kva_arg(valist, int);

   while (attribute != 0 && status)
   {
      if (attribute < 0 || attribute > object->attribute_cnt)
      {
	 errno = KCMS_INVALIDATTRIBUTE;
	 kerror(KCMS,routine,"Invalid attribute specified (%d)", attribute);
	 kva_end(valist);
	 return FALSE;
      }
      status = kcms_vget_attribute(object, attribute, &valist);
      if (status)
	 attribute = kva_arg(valist, int);
   }
   kva_end(valist);

   return status;
}

/************************************************************
* Routine Name:	kcms_set_attributes - set multiple attributes of kcms object
*
* Purpose:	This function is used to set one or more attributes of
*		a kcms object.
*
*		The following code will set the author's name and email
*		address for a software object;
*		!\f(CW\s-2
*		!   kobject  object;
*		!
*		!   kcms_set_attributes(object,
*		!                       KCMS_AUTHOR,       "Fred Bloggs",
*		!                       KCMS_AUTHOR_EMAIL, "fred@foobar.org",
*		!                       KCMS_END);
*		!\s+2\fP
*		\fBNote:\fP the argument list must be terminated
*		with the symbol \f(CW\s-1KCMS_END\s+1\fP.
*
* Input:	object  - The kcms object to set attributes of.
*		kvalist - A variable argument list, in the form:
*		! ATTRIBUTE_NAME, &value,
*		The attribute list must be terminated with the symbol
*		! KCMS_END
*		to signify the end of the variable argument list.
*
* Returns:	TRUE (1) on success, FALSE (0) otherwise
* Written By:	Neil Bowers
* Date:		29-jun-94
*************************************************************/
int
kcms_set_attributes(
   kobject  object,
   kvalist)
{
   kstring   routine     = "Get Attribute";
   int	     status	 = TRUE;
   kva_list  valist;
   int	     attribute;


   errno = KCMS_OK;
   if (!kcms_legal_object(object,routine))
      return FALSE;

   kva_start(valist,object);

   attribute = kva_arg(valist, int);

   while (attribute != 0 && status)
   {
      if (attribute < 0 || attribute > object->attribute_cnt)
      {
	 errno = KCMS_INVALIDATTRIBUTE;
	 kerror(KCMS,routine,"Invalid attribute specified (%d)", attribute);
	 kva_end(valist);
	 return FALSE;
      }
      status = kcms_vset_attribute(object, attribute, &valist);
      if (status)
	 attribute = kva_arg(valist, int);
   }
   kva_end(valist);

   return status;
}

/************************************************************
* Routine Name:	kcms_get_indirect_attribute - get indirect attribute from
*					      a kcms object
*
* Purpose:	This function is used to get an attribute from an object
*		which is itself an attribute of an object.
*		A common example is getting the KCMS_PARENT attribute
*		of an object, then getting an attribute from the parent.
*
*		The following code will determine the name of the toolbox
*		containing a software object:
*		!\f(CW\s-2
*		!   kobject  object;
*		!   kstring  tbname;
*		!
*		!   kcms_get_indirect_attribute(object, KCMS_PARENT, KCMS_NAME, &tbname);
*		!\s+2\fP
*
* Input:	object  - The kcms object to set attributes of.
*		kvalist - A variable argument list, in the form:
*		          ! ATTRIBUTE_NAME, INDIRECT_ATTRIBUTE_NAME, value
*
* Returns:	TRUE (1) on success, FALSE (0) otherwise
* Written By:	Neil Bowers
* Date:		29-jun-94
*************************************************************/
int 
kcms_get_indirect_attribute(
   kobject   object,
   kvalist)
{
   kstring   routine    = "Get Attribute";
   kobject   indobject	= NULL;
   kva_list  valist;
   int	     indattr;
   int	     attribute;
   int	     result     = FALSE;


   if (!kcms_legal_object(object,routine))
      return FALSE;

   kva_start(valist,object);

   attribute = kva_arg(valist, int);
   if (attribute < 0 || attribute > object->attribute_cnt)
   {
      errno = KCMS_INVALIDATTRIBUTE;
      kerror(KCMS, routine, "Invalid attribute specified");
      kva_end(valist);
      return FALSE;
   }
   if (kcms_get_attribute(object, attribute, &indobject) && indobject != NULL)
   {
      indattr = kva_arg(valist, int);
      result = kcms_vget_attribute(indobject, indattr, &valist);
   }
   kva_end(valist);

   return result;
}

/************************************************************
* Routine Name:	kcms_set_indirect_attribute - set indirect attribute of
*					      a kcms object
*
* Purpose:	This function is used to set an attribute of an object
*		which is itself an attribute of an object.
*		A common example is getting the KCMS_PARENT attribute
*		of an object, then setting an attribute of the parent.
*
* Input:	object  - The kcms object to set attributes of.
*		kvalist - A variable argument list, in the form:
*		! ATTRIBUTE_NAME, INDIRECT_ATTRIBUTE_NAME, value
*
* Returns:	TRUE (1) on success, FALSE (0) otherwise
* Written By:	Neil Bowers
* Date:		29-jun-94
*************************************************************/
int 
kcms_set_indirect_attribute(
   kobject   object,
   kvalist)
{
   kstring   routine    = "kcms_set_indirect_attribute()";
   kobject   indobject	= NULL;
   kva_list  valist;
   int	     indattr;
   int	     attribute;
   int	     result     = FALSE;


   if (!kcms_legal_object(object, routine))
      return FALSE;

   kva_start(valist,object);

   attribute = kva_arg(valist, int);
   if (attribute < 0 || attribute > object->attribute_cnt)
   {
      errno = KCMS_INVALIDATTRIBUTE;
      kerror(KCMS,routine,"Invalid attribute specified");
      kva_end(valist);
      return FALSE;
   }
   if (kcms_get_attribute(object, attribute, &indobject) && indobject != NULL)
   {
      indattr = kva_arg(valist, int);
      result = kcms_vset_attribute(indobject,indattr,&valist);
   }
   kva_end(valist);

   return result;
}

/************************************************************
* Routine Name:	kcms_set_attribute - set attribute of a kcms object
*
* Purpose:	This function is used to set a single attribute of
*		a kcms object.
*		The attributes supported by KCMS can be found in the public
*		include files, and are also listed in the KCMS manual.
*
*		The following code will set the \fIversion\fP attribute
*		of a previously opened toolbox:
*		!\f(CW\s-2
*		!   kobject  toolbox;
*		!
*		!   kcms_set_attribute(toolbox, KCMS_TB_VERSION, "2.0.0");
*		!\s+2\fP
*
* Input:	object   - the object on which to set an attribute
*		kvalist  - variable argument list, in the form:
*		! ATTRIBUTE_NAME, value
*		! (Allowable Attributes found in the description
*		portion of the document.)
* Output:	
* Returns:	TRUE (1) on success, FALSE (0) otherwise
* Written By:	Steven Jorgensen & Neil Bowers
* Date:		22-apr-93
*************************************************************/
int 
kcms_set_attribute(
   kobject  object,
   kvalist)
{
   kstring   routine	= "kcms_set_attribute()";
   kva_list  valist;
   int	     attribute;
   int	     status;


   if (!kcms_legal_object(object,routine))
      return FALSE;

   kva_start(valist, object);

   attribute = kva_arg(valist, int);
   if (attribute < 0 || attribute > object->attribute_cnt)
   {
      errno = KCMS_INVALIDATTRIBUTE;
      kerror(KCMS,routine,"Invalid attribute specified");
      kva_end(valist);
      return FALSE;
   }
   status = kcms_vset_attribute(object,attribute,&valist);
   kva_end(valist);

   return status;
}

/************************************************************
* Routine Name:	kcms_close - close a kcms object
*
* Purpose:	This routine calls the appropriate synchronization
*		methods to close and free the memory associated with
*		toolbox, program, library.  The configuration information
*		database is synchronized with the information held in
*		the object.
*
* Input:	object - the object to be closed
*
* Returns:	TRUE (1) on success, FALSE (0) otherwise
*
* Written By:	Neil Bowers
* Date:		27-apr-93
*************************************************************/
int
kcms_close(
   kobject  object)
{
   kstring  routine = "Close Object";


   if (kcms_legal_object(object, routine) && _kcms_check_method(
		(kfunc_void) methods[object->type].close, routine, "close"))
      return methods[object->type].close(object);
   return FALSE;
}

/************************************************************
* Routine Name:	kcms_sync - synchronize a kcms object (flush to database)
*
* Purpose:	This function synchronizes an object with any permanent
*		store maintained for the object.  If you have changed
*		something in an object, and want the changes to be reflected
*		in the object database immediately, then you should use
*		this function.
*
*		Note that an object is always sync'ed on close if anything
*		in the object has changed.
*
* Input:	object		- the object to be synchronized.
*
* Returns:	TRUE (1) on success, FALSE (0) otherwise
*
* Written By:	Neil Bowers
* Date:		5-dec-93
*************************************************************/
int
kcms_sync(
   kobject  object)
{
   kstring  routine = "kcms_sync()";


   if (kcms_legal_object(object, routine) && _kcms_check_method(
		  (kfunc_void) methods[object->type].sync, routine, "sync"))
      return methods[object->type].sync(object);
   return FALSE;
}

/************************************************************
* Routine Name:	kcms_rename - rename a kcms object
*
* Purpose:	This function is used to rename a kcms object.
*		At the present time this function can only be used to
*		rename a software object.
*		In the future it will be possible to rename toolbox
*		and file objects with this function.
*
* Input:	object - The object to be renamed.
*		name   - The new name for the object.
*
* Returns:	TRUE (1) if the object was successfully renamed,
*		FALSE (0) otherwise.
*
* Written By:	Neil Bowers
* Date:		7-jul-94
*************************************************************/
int
kcms_rename(
   kobject  object,
   kstring  name)
{
   kstring  routine = "kcms_rename()";


   if (kcms_legal_object(object, routine)
       && _kcms_check_method((kfunc_void)methods[object->type].rename,
			     routine, "rename"))
      return methods[object->type].rename(object, name);
   return FALSE;
}

/************************************************************
* Routine Name:	kcms_duplicate - duplicate a kcms object
*
* Purpose:	This routine calls the appropriate copy routine to
*		copy an object to another object.  The following
*		combinations are legal:
*
*		!  1.  All objects in one toolbox to another toolbox
*		!  2.  A program or library object to another toolbox
*		!  3.  A CM file object to another program or library object
*
* Input:	src  - object to be copied
*		dest - parent object where new object will reside
*
* Returns:	An object reference (kobject) for the newly created duplicate,
*		or NULL on failure.
* Written By:	Steven Jorgensen
* Date:		27-apr-93
*************************************************************/
kobject
kcms_duplicate(
   kobject  src,
   kobject  dest)
{
   kstring  routine = "Duplicate Object";


   if (kcms_legal_object(src, routine) && _kcms_check_method(
	   (kfunc_void) methods[src->type].duplicate, routine, "duplicate"))
      return methods[src->type].duplicate(src, dest);
   return NULL;
}

/************************************************************
* Routine Name:	kcms_destroy - destroy a kcms object
*
* Purpose:	This routine destroy's the object.  It recursively calls
*		destroy on sub-objects and removes all files on disk
*		associated with the object.  Finally, it informs its
*		parent object that it is gone so the parent can remove
*		the reference from its database.
*
* Input:	object - object to be destroyed
*
* Returns:	TRUE (1) if the object was successfully destroyed,
*		FALSE (0) otherwise.
* Written By:	Steven Jorgensen
* Date:		27-apr-93
*************************************************************/
int
kcms_destroy(
   kobject  object)
{
   kstring  routine = "Destroy Object";


   errno = KCMS_OK;
   if (kcms_legal_object(object, routine) && _kcms_check_method(
	    (kfunc_void) methods[object->type].destroy, routine, "destroy"))
      return methods[object->type].destroy(object);
   return FALSE;
}

/************************************************************
* Routine Name:	kcms_list_get_attribute - get attribute from list
*					  of kcms objects
*
* Purpose:	This routine takes a list of kcms objects,
*		and gets a string attribute from each,
*		building a string of the values file.
*
*		The following code will get a list of all \fIlibrary\fP
*		objects in the \fBdesign\fP toolbox,
*		then build an array of strings
*		which contains the names of the library objects.
*		!\f(CW\s-1
*		!   kobject   toolbox;
*		!   klist    *libraries;
*		!   kstring  *libnames;
*		!   int       nlibs;
*		!
*		!   toolbox   = kcms_open_toolbox("design");
*		!   libraries = kcms_get_objects(toolbox, KCMS_LIBRARY);
*		!   libnames  = kcms_list_get_attribute(libraries, KCMS_NAME, &nlibs);
*		!\s+1\fP
*
* Input:	list	  - A list (type \f(CW\s-1klist *\s+1\fP)
*			    of kcms objects.
*		attribute - The string attribute to get from each object.
*		count     - A pointer to an integer.  The number of objects in
*			    the list will be stored in the integer referenced.
*
* Written By:	Neil Bowers
* Date:		23-sep-93
*************************************************************/
kstring *
kcms_list_get_attribute(
   klist  *list,
   int     attribute,
   int    *count)
{
   kobject   object;
   kstring  *result = NULL;
   kstring   tmp;


   for (*count = 0; list != NULL; list = klist_next(list))
   {
      object = (kobject) klist_clientdata(list);
      if (!kcms_get_attribute(object, attribute, &tmp))
      {
	 *count = 0;
	 return NULL;
      }
      result = karray_insert(result, tmp, *count, KLIST_TAIL, TRUE);
      (*count)++;
   }
   return result;
}

/************************************************************
* Routine Name:	kcms_foreach - invoke a function on a set of kcms objects
*
* Purpose:	This function is used to invoke a function over some set
*		(or subset) of kcms objects.
*		The following handler function is used in the examples below:
*		!\f(CW\s-2
*		!   int print_name(kobject object)
*		!   {
*		!      kstring  name;
*		!
*		!      kcms_get_attribute(object, KCMS_NAME, &name);
*		!      kprintf("object name: %s\n", name);
*		!      return TRUE;
*		!   }
*		!\s+2\fP
*		To print out the names of all software objects in a toolbox,
*		you could use the following:
*		!\f(CW\s-2
*		!   kcms_foreach(toolbox, print_name, KOBJ_CMSOBJ, TRUE);
*		!\s+2\fP
*		The following call will print out the name of all available
*		toolboxes:
*		!\f(CW\s-2
*		!   kcms_foreach(NULL, print_name, KOBJ_CMSTB, TRUE);
*		!\s+2\fP
*		The following call will print out the name of all software
*		objects in all available toolboxes:
*		!\f(CW\s-2
*		!   kcms_foreach(NULL, print_name, KOBJ_CMSOBJ, TRUE);
*		!\s+2\fP
*
* Input:	parent	    - A kcms kobject, or NULL.
*		              This is used to seed the foreach.
*		handler	    - The handler function.
*			      This should be declared as an integer function,
*			      which takes a single parameter,
*			      of type \f(CW\s-1kobject\s+1\fP:
*			      !\f(CW\s-2
*			      !   int
*			      !   my_handler(
*			      !      kobject   object)
*			      !\s+2\fP
*			      The function should return \f(CW\s-1TRUE\s+1\fP,
*			      to signify success, and \f(CW\s-1FALSE\s+1\fP
*			      to signify failure.
*		object_type - The type of kcms object we're interested in,
*			      which can be one of the following:
*		.TS H
*		center tab(:) ;
*		lfB | lfB
*		lf(CW) | l .
*		object_type:Kcms object type
*		=
*		.TH
*		KOBJ_CMSTB:Toolbox object
*		KOBJ_CMSOBJ:Software object
*		KOBJ_CMSOBJ:File object
*		.TE
*		persevere   - A boolean flag which specifies whether
*			      \f(CW\s-1kcms_foreach()\s+1\fP should continue
*			      when the handler function
*			      returns \f(CW\s-1FALSE\s+1\fP.
*
* Returns:	TRUE (1) on success, FALSE (0) otherwise
*
* Written By:	Neil Bowers
* Date:		7-jul-94
*************************************************************/
int
kcms_foreach(
   kobject      parent,
   kcms_method  handler,
   int          object_type,
   int          persevere)
{
   kstring    routine   = "kcms_foreach()";
   int        result    = FALSE;
   int        attribute = -1;
   int        cmstype;
   kobject    toolbox;
   kobject    object;
   klist     *list;
   kstring   *tblist;
   int        ntbs;
   int        i;


   /*UPDATE--------------------------------------------------------------
   |	The structure of this routine needs cleaning up, once the
   |	functionality has settled down.
   --------------------------------------------------------------------*/

   /*-- want all of the specified object type -------------------------*/
   if (parent == NULL)
   {
      if (object_type == KOBJ_CMSFILE)
      {
	 kerror(KCMS, routine, "File object not currently supported for "
		"the case where the parent parameter is NULL.");
	 return FALSE;
      }

      if (object_type != KOBJ_CMSTB && object_type != KOBJ_CMSOBJ)
      {
	 kerror(KCMS, routine, "Illegal object type requested.");
	 return FALSE;
      }

      tblist = kcms_query_toolboxes(&ntbs);
      for (i = 0; i<ntbs; i++)
      {
	 if ((toolbox = kcms_open_toolbox(tblist[i])) == NULL)
	    continue;

	 if (object_type == KOBJ_CMSTB)
	 {
	    result = handler(toolbox);
	    kcms_close(toolbox);
	    if (!result && !persevere)
	       return FALSE;
	 }
	 else
	 {
	    if (!kcms_get_attribute(toolbox, KCMS_TB_SOFTWARE_OBJECTS, &list))
	    {
	       kcms_close(toolbox);
	       return FALSE;
	    }

	    for (; list != NULL; list = klist_next(list))
	    {
	       if (object_type == KOBJ_CMSOBJ)
	       {
		  object = (kobject)klist_clientdata(list);
		  if (object == NULL)
		     continue;

		  result = handler(object);
		  if (!result && !persevere)
		  {
		     kcms_close(toolbox);
		     return FALSE;
		  }
	       }
	    }
	    kcms_close(toolbox);
	 }
      }

      return TRUE;
   }

   if (!kcms_get_attribute(parent, KCMS_TYPE, &cmstype))
      return FALSE;

   /*-- determine attribute, based on parent & requested object types -*/
   /*UPDATE: this should be table driven, most likely -----------------*/
   switch (cmstype)
   {
      case KOBJ_CMSTB:
	 if (object_type == KOBJ_CMSOBJ)
	    attribute = KCMS_TB_SOFTWARE_OBJECTS;
	 else if (object_type == KOBJ_CMSFILE)
	    attribute = KCMS_TB_MISC_FILES;
	 else
	    kerror(KCMS, routine,
		   "Illegal object type specified for toolbox.");
	 break;

      case KOBJ_CMSOBJ:
	 if (object_type == KOBJ_CMSFILE)
	    attribute = KCMS_CMOBJ_ALL_FILES;
	 else
	    kerror(KCMS, routine,
		   "Illegal object type specified for software object.");
	 break;

      case KOBJ_CMSFILE:
	 kerror(KCMS, routine, "Cannot use kcms_foreach() with file objects.");
	 break;

      default:
	 kerror(KCMS, routine, "Unrecognized object type passed for parent.");
	 break;
   }

   if (attribute == -1)
      return FALSE;

   if (!kcms_get_attribute(parent, attribute, &list))
      return FALSE;

   for (; list != NULL; list = klist_next(list))
   {
      object = (kobject)klist_clientdata(list);

      if (object == NULL)
	 continue;

      result = handler(object);

      /*-- if handler function returns FALSE, and persevere==FALSE ----*/
      if (!result && !persevere)
	 return FALSE;
   }

   return TRUE;
}

/************************************************************
* Routine Name:	kcms_add_callback - add an attribute callback
*
* Purpose:	This routine is used to associate a callback function with
*		a particular attribute of a software object.  Whenever the
*		attribute is modified on the given software object, the
*		callback function will be invoked.
*
* Input:	object      - The object to hook the callback into.
*		attribute   - The attribute to associate the callback with.
*		callback    - The callback function to be invoked when the
*			      attribute is modified.
*		client_data - An optional client pointer, which can be used
*			      to associated some data with the callback.
*			      This will be passed as the third argument to
*			      the callback function.
*
* Returns:	TRUE (1) if the callback was added, FALSE (0) otherwise.
* Written By:	Neil Bowers
* Date:		16-nov-94
*************************************************************/
int
kcms_add_callback(
   kobject   object,
   int       attribute,
   void      (*callback) PROTO((kobject, int, void *)),
   void     *client_data)
{
   CallbackInfo  *info;


   info = kcalloc(1, sizeof(CallbackInfo));
   if (info == NULL)
      return FALSE;

   info->attribute   = attribute;
   info->callback    = callback;
   info->client_data = client_data;

   object->callbacks = klist_add(object->callbacks, (kaddr)attribute,
				 (kaddr)info);
   return TRUE;
}

/************************************************************
* Routine Name:	kcms_remove_callback - remove an attribute callback
*
* Purpose:	This routine is used to remove a callback function from
*		a particular attribute of a software object.
*
* Input:	object      - The object to hook the callback into.
*		attribute   - The attribute which the callback is associated
*			      with.
*		callback    - The callback function to be invoked when the
*			      attribute is modified.
*		client_data - An optional client pointer, which can be used
*			      to associated some data with the callback.
*			      This will be passed as the third argument to
*			      the callback function.
*
* Returns:	TRUE (1) if the callback was added, FALSE (0) otherwise.
* Written By:	Neil Bowers
* Date:		16-nov-94
*************************************************************/
int
kcms_remove_callback(
   kobject   object,
   int       attribute,
   void      (*callback) PROTO((kobject, int, void *)),
   void     *client_data)
{
   CallbackInfo  *info;
   klist         *item;


   item = klist_locate(object->callbacks, (kaddr)attribute);
   if (item == NULL)
      return FALSE;

   info = (CallbackInfo *)klist_clientdata(item);
   if (info == NULL)
      return FALSE;

   object->callbacks = klist_delete(object->callbacks, (kaddr)attribute);

   kfree(info);

   return TRUE;
}

/*-----------------------------------------------------------
| Routine Name:	kcms_legal_object - check object pointer for kcms-ness
|
| Purpose:	This function takes a kobject reference and performs
|		some simple integrity checking.
|		The first check guards against a NULL reference being passed.
|		The object is then checked to see whether it is a KCMS
|		object: a toolbox, cmobj or file object.
|
| Input:	object	- The kobject to be checked.
|		routine	- A short string identifying the routine which
|                         called this routine.  This is the routine name
|                         given in any error message.
|
| Returns:	TRUE (1) if the object is ok, FALSE (0) otherwise.
|
| Written By:	Neil Bowers
| Date:		9-nov-93
------------------------------------------------------------*/
int
kcms_legal_object(
   kobject  object,
   kstring  routine)
{
   if (object == NULL)
      kerror(KCMS, routine, "NULL object passed to %s.\n", routine);
   else if (object->type != KOBJ_CMSTB && object->type != KOBJ_CMSOBJ &&
	    object->type != KOBJ_CMSFILE)
      kerror(KCMS, routine, "Non-kcms object passed to %s.\n", routine);
   else
      return TRUE;
   return FALSE;
}

/*-----------------------------------------------------------
| Routine Name:	_kcms_check_method - check whether method defined
|
| Purpose:	This function takes a kobject reference and performs
|		some simple integrity checking.
|		The first check guards against a NULL reference being passed.
|		The object is then checked to see whether it is a KCMS
|		object --- a toolbox, cmobj or file object.
|
| Input:	object	- The kobject to be checked.
|		routine	- A short string identifying the routine which
|                         called this routine.  This is the routine name
|                         given in any error message.
|		method	- the name of the method.
|
| Returns:	TRUE (1) if the object is ok, FALSE (0) otherwise.
|
| Written By:	Neil Bowers
| Date:		9-nov-93
------------------------------------------------------------*/
static int
_kcms_check_method(
   kfunc_void  func,
   kstring     routine,
   kstring     method)
{
   if (func == NULL)
   {
      errno = KCMS_NOMETHOD;
      kerror(KCMS, routine, "No %s method defined for object.", method);
      return FALSE;
   }
   return TRUE;
}

static void
kcms_call_callbacks(
   kobject  object,
   int      attribute)
{
   klist         *list;
   CallbackInfo  *info;


   for (list=object->callbacks; list != NULL; list = klist_next(list))
   {
      info = (CallbackInfo *)klist_clientdata(list);
      if (info->attribute == attribute)
	 info->callback(object, attribute, info->client_data);
   }
}
