/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Re-Generate Imakefile and Makefile
   >>>>
   >>>>   Static:
   >>>>  Private:
   >>>>             kcms_dir_imakeandmake()
   >>>>   Public:
   >>>>             kcms_gen_imakeandmake()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"

/*-----------------------------------------------------------
| Routine Name:	kcms_dir_imakeandmake - generate an Imakefile and a Makefile
|		in the parent dir of the program.
|
| Purpose:	This routine makes system calls to kgenimake and
|		kgenmake to create an Imakefile and a Makefile in
|		the directory specified by iteratively dirnaming the
|		path provided until $TOOLBOX is reached, and updating
|		all the files for each dirname.
|
| Input:	path     - directory where we generate the files
|		tbname   - the toolbox name to be used by the Imakefile
| Output:	
| Returns:	TRUE (1) on success, FALSE (0) otherwise
| Written By:	Steven Jorgensen
| Date:		10-jun-93
------------------------------------------------------------*/
int
kcms_dir_imakeandmake(
   kstring  path,
   kstring  tbname,
   int      dircount)
{
   kstring  routine = "kcms_dir_imakeandmake";
   char     buff[KLENGTH];
   char     tbupper[KLENGTH];
   char     topsrc[KLENGTH];
   char     tbpath[KLENGTH];


   errno = KCMS_OK;
   if (kstrlen(path) == 0)
   {
      kerror(KCMS, routine, "\nPath parameter was NULL");
      errno = KCMS_NULLPATH;
      return FALSE;
   }
   if (kstrlen(tbname) == 0)
   {
      kerror(KCMS, routine, "\nToolbox name was NULL");
      errno = KCMS_NULLPARENT;
      return FALSE;
   }

   kannounce(KCMS, routine, "Generating directory Imakefile & Makefile");

   kstring_upper(tbname, tbupper);
   ksprintf(topsrc, "$%s", tbupper);
   kstrcpy(tbpath, path);
   while (dircount > 0 && kstrcmp(topsrc, tbpath) != 0 && kstrlen(tbpath) > 0)
   {
      ksprintf(buff, "cd %s ; kgenimake -dir -tb %s", tbpath, tbname);
      ksystem(buff);
      if (--dircount)
	 kdirname(tbpath, tbpath);
   }

   return TRUE;
}


/*-----------------------------------------------------------
| Routine Name:	kcms_gen_imakeandmake - generate Imakefile and Makefile
|
| Purpose:	This routine makes system calls to kgenimake and
|		kgenmake to create an Imakefile and a Makefile in
|		the directory specified by the calling routine
|
| Input:	object   - binary name of the program or library
|		path     - directory where we generate the files,
|		           if NULL then uses the object's top src path.
| Output:	
| Returns:	TRUE (1) on success, FALSE (0) otherwise
|
| Written By:	Danielle Argiro & Steven Jorgensen
| Date:		28-aug-94
------------------------------------------------------------*/
int
kcms_gen_imakeandmake(
   kobject  object,
   kstring  path)
{
   kstring        routine  = "kcms_gen_imakeandmake()";
   kobject        toolbox;
   kstring        oname;
   kstring        tbname;
   kstring        topsrc;
   char           buff[KLENGTH];
   char           dirpath[KLENGTH];
   char           fullpath[KLENGTH];
   unsigned long  update;


   if (!kcms_get_attributes(object,
			    KCMS_NAME,            &oname,
			    KCMS_CMOBJ_TOPSRC,    &topsrc,
			    KCMS_PARENT,          &toolbox,
			    KCMS_CMOBJ_UPDATE_DB, &update,
			    KCMS_END)
       || !kcms_get_attribute(toolbox, KCMS_NAME, &tbname))
   {
      return FALSE;
   }

   if (path == NULL)
      path = topsrc;

   if (kfullpath(path, NULL, fullpath) == NULL)
      return FALSE;

   ksprintf(buff,
	    "cd %s ; "
	    "kgenimake -oname %s -tb %s ",
	    fullpath, oname, tbname);
   kannounce(KCMS, routine, "Generating Imakefile & Makefile in src");
   ksystem(buff);
   kdirname(fullpath, dirpath);
   if (update & KCMS_UPDATE_NEW == KCMS_UPDATE_NEW)
      kcms_dir_imakeandmake(dirpath, tbname, 3);

   return TRUE;
}

/************************************************************
* Routine Name:	kcms_get_imake_symbols - parse symbols from Imakefile
*
* Purpose:	This routine will try to parse a local Imakefile for
*		the toolbox name, object name, binary name, and
*		object type.
*
* Output:	tbname - a pointer to a string.  If the pointer is non-NULL,
*			 and the referenced pointer is NULL, then we will
*			 parse the TOOLBOX_NAME symbol out of the Imakefile
*			 and set the pointer to string containing the
*			 toolbox name.
*		oname  - A pointer to a string for the object name.
*		bname  - A pointer to a string for the binary name.
*		otype  - A pointer to an integer value.  If non-NULL,
*			 then the OBJECT_TYPE symbol will be parsed from
*			 the Imakefile, and converted to its integer
*			 representation.
*
* Returns:	TRUE (1) on success, and FALSE (0) on failure.
*
* Written By:	Neil Bowers
* Date:		26-feb-1995
*************************************************************/
int
kcms_get_imake_symbols(
   kstring  *tbname,
   kstring  *oname,
   kstring  *bname,
   int      *otype)
{
   kstring   routine = "kcms_get_imake_symbols()";
   kstring   data;
   kstring  *list;
   int       nbytes;
   int       cnt;
   int       err;
   char      continuation[KLENGTH];    /*-- continuation chars --*/


   ksprintf(continuation, "%c", '\\');

   if ((tbname == NULL || *tbname != NULL)
       && (oname == NULL || *oname != NULL)
       && (bname == NULL || *bname != NULL)
       && otype == NULL)
      return TRUE;

   data = (kstring)kreadfile("./Imakefile", &nbytes);
   if (data == NULL || nbytes < 1)
      return FALSE;

   /*-- get the toolbox name if we need to ----------------------------*/
   if (tbname != NULL && *tbname == NULL)
   {
      list = kparse_string_search_delimit(data, "TOOLBOX_NAME =",
					KDELIM_CLEAN, " \t\n",
					continuation,
					NULL, &cnt, &err);
      if (cnt != 1 || err != KPARSE_OK)
      {
	 kerror(KCMS, routine, "Unable to parse TOOLBOX NAME from Imakefile");
	 kfree(data);
	 return FALSE;
      }

      *tbname = kstrdup(list[0]);
      karray_free(list, cnt, NULL);
   }

   /*-- get the object name if we need to -----------------------------*/
   if (oname != NULL && *oname == NULL)
   {
      list = kparse_string_search_delimit(data, "OBJECT_NAME =",
					KDELIM_CLEAN, " \t\n",
					continuation,
					NULL, &cnt, &err);
      if (err != KPARSE_OK)
      {
	 kerror(KCMS, routine, "Unable to parse OBJECT NAME from Imakefile");
	 kfree(data);
	 return FALSE;
      }

      if (cnt == 1)
	 *oname = kstrdup(list[0]);
      karray_free(list, cnt, NULL);
   }

   /*-- get the object binary name if we need to ----------------------*/
   if (bname != NULL && *bname == NULL)
   {
      list = kparse_string_search_delimit(data, "(BINARY_NAME|CHAPTER_NAME) =",
					KDELIM_CLEAN, " \t\n",
					continuation,
					NULL, &cnt, &err);
      if (err != KPARSE_OK)
      {
	 kerror(KCMS, routine, "Unable to parse BINARY NAME from Imakefile");
	 kfree(data);
	 return FALSE;
      }

      if (cnt == 1)
	 *bname = kstrdup(list[0]);
      karray_free(list, cnt, NULL);
   }

   /*-- get the object type -------------------------------------------*/
   if (otype != NULL)
   {
      list = kparse_string_search_delimit(data, "OBJECT_TYPE =",
					KDELIM_CLEAN, " \t\n",
					continuation,
					NULL, &cnt, &err);
      if (cnt != 1 || err != KPARSE_OK)
      {
	 kerror(KCMS, routine, "Unable to parse OBJECT TYPE from Imakefile");
	 kfree(data);
	 return FALSE;
      }

      *otype = kcms_attr_string2int(KCMS_CMOBJ_PROGTYPE, list[0]);
      karray_free(list, cnt, NULL);
   }

   kfree(data);
   return TRUE;
}
