/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Image Display Utility Routines
   >>>>
   >>>>   Static:
   >>>>		_kcms_rename_file_objects()
   >>>>		_kcms_sed_files()
   >>>>
   >>>>  Private:
   >>>>		kcms_cmobj_rename()
   >>>>
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"
#include "cmobjP.h"
#include "databaseP.h"

typedef struct
{
   int      attribute;
   kstring  format_string;
} AttributeString;

static AttributeString formats[] =
{
   {KCMS_CMOBJ_GEN_MAIN,   "%s.c"},
   {KCMS_CMOBJ_GEN_INCL,   "%s.h"},
   {KCMS_CMOBJ_GEN_LFILE,  "l%s.c"},
   {KCMS_CMOBJ_GEN_MAN1,   "%s.1"},
   {KCMS_CMOBJ_GEN_MAN3,   "%s.3"},
   {KCMS_CMOBJ_GEN_HELP,   "%s.hlp"},
   {KCMS_CMOBJ_GEN_SCRIPT, "%s.sh"},
   {KCMS_CMOBJ_UIS_PANE,   "%s.pane"},
   {KCMS_CMOBJ_UIS_FORM,   "%s.form"},
   {KCMS_CMOBJ_TODO,       "%s.todo"},
   {KCMS_CMOBJ_BUGS,       "%s.bugs"},
   {KCMS_CMOBJ_DONE,       "%s.done"},
   {KCMS_CMOBJ_CHANGELOG,  "%s.changelog"}
};

void kcms_tb_rename_cmobj PROTO((kobject, kstring));
static void _kcms_rename_file_objects PROTO((kobject, kstring, kstring));
static void _kcms_sed_files PROTO((kobject, kstring, kstring));

typedef struct
{
   int      id;
   kstring  string;
} IntString;

static IntString lang_extensions[] =
{
   {KCMS_LANG_SH, "sh"},
   {KCMS_LANG_KSH, "ksh"},
   {KCMS_LANG_CSH, "csh"},
   {KCMS_LANG_PERL, "pl"}
};

static kstring
kcms_lang_extension(
   int  lang_type)
{
   int  i;


   for (i=0; i<knumber(lang_extensions); i++)
      if (lang_extensions[i].id == lang_type)
	 return lang_extensions[i].string;

   return NULL;
}

/*-----------------------------------------------------------
| Routine Name:	kcms_cmobj_rename - rename a software object
| Purpose:	This function is the private method for kcms_rename(),
|		and implements the functionality for software objects.
|
| Input:	object  - The object to rename.
|		newname - The new name for the software object.
|
| Returns:	TRUE (1) if the object was successfully renamed,
|		FALSE (0) otherwise
| Written By:	Neil Bowers
| Date:		7-jul-94
------------------------------------------------------------*/
kbool
kcms_cmobj_rename(
   kobject  object,
   kstring  newname)
{
   kstring  routine  = "kcms_cmobj_rename()";
   kobject  toolbox;
   kobject  existing;
   kdbm    *database;
   kstring  oname;
   kstring  tbname;
   kstring  tbpath;
   int      otype;
   klist   *list;
   kobject  fobj;
   kstring  typestring;
   kstring  opath;
   char     dbpath[KLENGTH];
   char     oldname[KLENGTH];
   char     oldpath[KLENGTH];
   char     newpath[KLENGTH];
   char     oldincpath[KLENGTH];
   char     newincpath[KLENGTH];


   if (!kcms_legal_identifier(newname, KOBJ_CMSOBJ))
   {
      kerror(KCMS, routine, "`%s' is not a valid name for a software object.",
	     newname);
      return FALSE;
   }

   if (!kcms_get_attributes(object,
			    KCMS_NAME,             &oname,
			    KCMS_PATH,             &opath,
			    KCMS_PARENT,           &toolbox,
			    KCMS_CMOBJ_PROGTYPE,   &otype,
			    KCMS_CMOBJ_ALL_FILES,  &list,
			    KCMS_END)
       || !kcms_get_attributes(toolbox,
			       KCMS_NAME, &tbname,
			       KCMS_PATH, &tbpath,
			       KCMS_END))
      return FALSE;

   typestring = kcms_attr_int2string(KCMS_CMOBJ_TYPE, otype);
   kstrcpy(oldpath, opath);
   kstrcpy(oldname, oname);

   /*-- is there already an object of the specified name? ------------*/
   if ((existing = kcms_open_cmobj(toolbox, newname)) != NULL)
   {
      kerror(KCMS, routine, "There is already an object called \"%s\" "
	     "in toolbox \"%s\".", newname, tbname);
      kcms_close(existing);
      return FALSE;
   }

   /*-- is the new name a valid object name? -------------------------*/

   /*-- rename the toplevel directory of the object ------------------*/
   ksprintf(newpath, "$%s/objects/%s/%s", tbname, typestring, newname);
   if (krename(oldpath, newpath) == -1)
   {
      kerror(KCMS, routine, "Could not rename toplevel directory of object.\n"
	     "\tFrom: %s\n"
             "To    : %s\n", oldpath, newpath);
      return FALSE;
   }
   ksprintf(oldincpath, "$%s/include/%s/", tbname, oldname);
   ksprintf(newincpath, "$%s/include/%s/", tbname, newname);

   /*-------------------------------------------------------------------
   |   update internal object state
   +------------------------------------------------------------------*/

   kfree(object->oname);
   kfree(object->opath);
   kfree(object->topsrc);

   object->oname     = kstrdup(newname);
   object->oname_tkn = kstring_to_token(newname);
   object->opath     = kstrdup(newpath);
   object->topsrc    = kstring_cat(newpath, "/src", NULL);

   if (object->bname != NULL)
   {
      kfree(object->bname);
      object->bname = NULL;
   }

   /*-- if a library object, rename public include dir & file ---------*/
   if (object->prog_type == KCMS_LIBRARY)
   {
      char  liboldpath[KLENGTH];
      char  libnewpath[KLENGTH];

      /*-- rename public include directory: $TOOLBOX/include/<object> --*/
      ksprintf(liboldpath, "$%s/include/%s", tbname, oldname);
      ksprintf(libnewpath, "$%s/include/%s", tbname, newname);
      if (krename(liboldpath, libnewpath) == -1)
      {
	 kwarn(KCMS, routine,
		"Could not rename public include directory for object.\n"
		"\tFrom : %s\n"
		"\tTo   : %s\n", liboldpath, libnewpath);
	 return FALSE;
      }

      /*-- rename public include: $TOOLBOX/include/<object>/<object>.h --*/
      ksprintf(liboldpath, "$%s/include/%s/%s.h", tbname, newname, oldname);
      ksprintf(libnewpath, "$%s/include/%s/%s.h", tbname, newname, newname);
      if (krename(liboldpath, libnewpath) == -1)
      {
	 kwarn(KCMS, routine,
		"Could not rename public include file for object.\n"
		"\tFrom : %s\n"
		"\tTo   : %s\n", liboldpath, libnewpath);
	 return FALSE;
      }
   }

   /*-------------------------------------------------------------------
   |	update every file object's idea of it's path
   +------------------------------------------------------------------*/
   for (; list != NULL; list = klist_next(list))
   {
      if ((fobj = (kobject)klist_clientdata(list)) != NULL)
	 kcms_fobj_shift_parent(fobj, oldname, newname,
				oldpath, newpath,
				oldincpath, newincpath);
   }

   kannounce(KCMS, routine, "Renaming files in %s::%s", tbname, newname);
   _kcms_rename_file_objects(object, oldname, newname);

   kannounce(KCMS, routine, "Substituting for `%s' in %s::%s", oldname,
	     tbname, newname);
   _kcms_sed_files(object, oldname, newname);

   if (object->prog_type == KCMS_KROUTINE && object->library != NULL)
   {
      kobject  libobj;


      if ((libobj = kcms_open_cmobj(object->parent, object->library)) != NULL)
      {
	 kcms_set_attribute(libobj, KCMS_CMOBJ_UPDATE_DB,
			    (KCMS_UPDATE_SYNC | KCMS_UPDATE_REGEN));
	 kcms_close(libobj);
      }
   }

   kcms_tb_rename_cmobj(toolbox, newname);

   /*-- remove the old name out of the objects cache ------------------*/
   ksprintf(dbpath, "%s/repos/db/cache/objects", tbpath);
   if ((database = kdbm_open(dbpath, O_WRONLY | O_CREAT, 0666)) != NULL)
   {
      kcms_db_remove_key(database, oldname);
      kdbm_close(database);
   }

   /*-- do we need to modify the update flag at all? ------------------*/
   kcms_set_attribute(object, KCMS_CMOBJ_UPDATE_DB,
		      (KCMS_UPDATE_SYNC | KCMS_UPDATE_REGEN
		       | KCMS_UPDATE_CACHE));

   kcms_set_attribute(toolbox, KCMS_TB_UPDATE_FLAG, KCMS_UPDATE_SYNC);
   kcms_sync(toolbox);

   return TRUE;
}

/*-----------------------------------------------------------
| Routine Name:	_kcms_rename_file_objects - rename file objects in object
| Purpose:	This function renames all files in the software object
|		which have the object name as part of the file name.
|
| Input:	object  - The object to rename.
|		newname - The new name for the software object.
|
| Written By:	Neil Bowers
| Date:		10-aug-94
------------------------------------------------------------*/
static void
_kcms_rename_file_objects(
   kobject  object,
   kstring  oldname,
   kstring  newname)
{
   int      i;
   kstring  oldfilename;
   kstring  extension;
   kstring  tbname;
   char     scriptfile[KLENGTH];
   char     newfilename[KLENGTH];
   kobject  file_object;
   int      otype;
   int      langtype;
   klist   *files;


   if (!kcms_get_indirect_attribute(object, KCMS_PARENT, KCMS_NAME, &tbname))
      return;

   /*-- update files which have object name in filename ---------------*/
   for (i = 0; i < knumber(formats); i++)
   {
      if (!kcms_get_attribute(object, formats[i].attribute, &file_object)
	  || file_object == NULL
	  || !kcms_get_attribute(file_object, KCMS_NAME, &oldfilename))
      {
	 continue;
      }

      ksprintf(newfilename, formats[i].format_string, newname);

      /*-- set the basename attribute of the file object --------------*/
      kcms_set_attribute(file_object, KCMS_NAME, newfilename);
   }


   /*-------------------------------------------------------------------
   |   file renames which are specific to software object type
   +------------------------------------------------------------------*/

   if (!kcms_get_attributes(object,
			    KCMS_CMOBJ_TYPE, &otype,
			    KCMS_CMOBJ_LANGTYPE, &langtype,
			    KCMS_END))
      return;

   if (otype == KCMS_SCRIPT
       && kcms_get_attribute(object, KCMS_CMOBJ_ALL_SCRIPT, &files))
   {
      for (; files != NULL; files = klist_next(files))
      {
	 if ((file_object = (kobject) klist_clientdata(files)) == NULL
	     || !kcms_get_attribute(file_object, KCMS_NAME, &oldfilename))
	    continue;

	 extension = kcms_lang_extension(langtype);
	 ksprintf(scriptfile, "%s.%s", oldname, extension);

	 /*-- if file is <oname>.<script extension>, rename it --------*/
	 if (!kstrcmp(scriptfile, oldfilename))
	 {
	    ksprintf(newfilename, "%s.%s", newname, extension);
	    kcms_set_attribute(file_object, KCMS_NAME, newfilename);
	 }
      }
   }
}

/*-----------------------------------------------------------
| Routine Name:	_kcms_sed_files - perform substitutions in all file objects
| Purpose:	Perform any and all substitutions necessary on the file
|		objects within the passed software object.
|
| Input:	object  - The object to rename.
|		newname - The new name for the software object.
|
| Written By:	Neil Bowers
| Date:		10-aug--94
------------------------------------------------------------*/
static void
_kcms_sed_files(
   kobject  object,
   kstring  oldname,
   kstring  newname)
{
   kstring   routine      = "_kcms_sed_files()";
   char      newpath[KLENGTH];
   char      bakpath[KLENGTH];
   char      oldstring[27][KLENGTH];
   char      newstring[27][KLENGTH];
   kstring   tbname;
   kstring   typestring;
   klist    *file_list;
   klist    *list;
   kobject   toolbox;
   kobject   file_object;
   kstring   path;
   int       otype;
   int       ftype;
   int       fsubtype;


   if (!kcms_get_attributes(object,
			    KCMS_CMOBJ_ALL_FILES, &file_list,
			    KCMS_PARENT,          &toolbox,
			    KCMS_CMOBJ_PROGTYPE,  &otype,
			    KCMS_END)
       || !kcms_get_attribute(toolbox, KCMS_NAME, &tbname))
      return;

   typestring = kcms_attr_int2string(KCMS_CMOBJ_TYPE, otype);

   /*-- build substitution strings ---------------------------------*/

   ksprintf(oldstring[0], "$%s/objects/%s/%s/help/%s.hlp",
	    tbname, typestring, oldname, oldname);
   ksprintf(newstring[0], "$%s/objects/%s/%s/help/%s.hlp",
	    tbname, typestring, newname, newname);

   ksprintf(oldstring[1], "$%s/objects/%s/%s/help",
	    tbname, typestring, oldname);
   ksprintf(newstring[1], "$%s/objects/%s/%s/help",
	    tbname, typestring, newname);

   ksprintf(oldstring[2], "_%s_%s_h_", oldname, oldname);
   ksprintf(newstring[2], "_%s_%s_h_", newname, newname);

   ksprintf(oldstring[3], "<%s/%s.h>", oldname, oldname);
   ksprintf(newstring[3], "<%s/%s.h>", newname, newname);

   ksprintf(oldstring[4], "<%s/", oldname);
   ksprintf(newstring[4], "<%s/", newname);

   ksprintf(oldstring[5], "%s_get_args", oldname);
   ksprintf(newstring[5], "%s_get_args", newname);

   ksprintf(oldstring[6], "%s_free_args", oldname);
   ksprintf(newstring[6], "%s_free_args", newname);

   ksprintf(oldstring[7], "%s_usage_additions", oldname);
   ksprintf(newstring[7], "%s_usage_additions", newname);

   ksprintf(oldstring[8], "#include \"%s.h\"", oldname);
   ksprintf(newstring[8], "#include \"%s.h\"", newname);

   ksprintf(oldstring[9], "#ifndef _%s_", oldname);
   ksprintf(newstring[9], "#ifndef _%s_", newname);

   ksprintf(oldstring[10], "#define _%s_", oldname);
   ksprintf(newstring[10], "#define _%s_", newname);

   ksprintf(oldstring[11], "#endif /\\* _%s_", oldname);
   ksprintf(newstring[11], "#endif /\\* _%s", newname);

   ksprintf(oldstring[12], "%s_clui_struct", oldname);
   ksprintf(newstring[12], "%s_clui_struct", newname);

   ksprintf(oldstring[13], "Main program for %s", oldname);
   ksprintf(newstring[13], "Main program for %s", newname);

   ksprintf(oldstring[14], "\"%s\"", oldname);
   ksprintf(newstring[14], "\"%s\"", newname);

   ksprintf(oldstring[15], "%s \\\\-", oldname);
   ksprintf(newstring[15], "%s \\-", newname);

   ksprintf(oldstring[16], "$%sBIN/%s", tbname, oldname);
   ksprintf(newstring[16], "$%sBIN/%s", tbname, newname);

   ksprintf(oldstring[17], "$%s/objects/%s/%s", tbname, typestring, oldname);
   ksprintf(newstring[17], "$%s/objects/%s/%s", tbname, typestring, newname);

   ksprintf(oldstring[18], "%s_gui_struct", oldname);
   ksprintf(newstring[18], "%s_gui_struct", newname);

   ksprintf(oldstring[19], "^Object Name:[	 ]*%s$", oldname);
   ksprintf(newstring[19], "Object Name:\t%s\n", newname);

   ksprintf(oldstring[20], "^File:[	 ]*%s\\.", oldname);
   ksprintf(newstring[20], "File:\t\t%s.", newname);

   ksprintf(oldstring[21], "^\\*[\t ]*Routine Name:[\t ]*l%s[\t ]", oldname);
   ksprintf(newstring[21], "*  Routine Name: l%s ", newname);

   ksprintf(oldstring[22], "^\\.syntax %s %s$", tbname, oldname);
   ksprintf(newstring[22], ".syntax %s %s\n", tbname, newname);

   /*UPDATE: the next two could be done with one regex ----------------*/
   ksprintf(oldstring[23], "\\\\fI%s\\\\fP", oldname);
   ksprintf(newstring[23], "\\\\fI%s\\\\fP", newname);

   ksprintf(oldstring[24], "\\\\fI%s\\\\fR", oldname);
   ksprintf(newstring[24], "\\\\fI%s\\\\fR", newname);

   /*-- following are for lkroutine renames ---------------------------*/
   ksprintf(oldstring[25], "\\bl%s(\\s*)\\(", oldname);
   ksprintf(newstring[25], "l%s\\1(", newname);
   ksprintf(oldstring[26], "\\bl%s(\\s+)PROTO", oldname);
   ksprintf(newstring[26], "l%s\\1PROTO", newname);


   for (list = file_list; list != NULL; list = klist_next(list))
   {
      file_object = (kobject)klist_clientdata(list);

      /*-- we don't want to make any substitutions in database files --*/
      if (!kcms_get_attributes(file_object,
			      KCMS_PATH,         &path,
			      KCMS_FOBJ_TYPE,    &ftype,
			      KCMS_FOBJ_SUBTYPE, &fsubtype,
			      KCMS_END))
	 continue;

      /*-- we don't want to make any substitutions in database files --*/
      if (ftype == KCMS_FOBJ_TYPE_DBM)
	 continue;

      /*-- new path for the file --------------------------------------*/
      ksprintf(newpath, "%s.new", path);

      if (!ksedfile(path, newpath,
		    TRUE,                       /* regular expression? */
		    KFILE_UPDATE,
		    NULL,                        /* optional status (int *) */
		    oldstring[0],  newstring[0],
		    oldstring[1],  newstring[1],
		    oldstring[2],  newstring[2],
		    oldstring[3],  newstring[3],
		    oldstring[4],  newstring[4],
		    oldstring[5],  newstring[5],
		    oldstring[6],  newstring[6],
		    oldstring[7],  newstring[7],
		    oldstring[8],  newstring[8],
		    oldstring[9],  newstring[9],
		    oldstring[10], newstring[10],
		    oldstring[11], newstring[11],
		    oldstring[12], newstring[12],
		    oldstring[13], newstring[13],
		    oldstring[14], newstring[14],
		    oldstring[15], newstring[15],
		    oldstring[16], newstring[16],
		    oldstring[17], newstring[17],
		    oldstring[18], newstring[18],
		    oldstring[19], newstring[19],
		    oldstring[20], newstring[20],
		    oldstring[21], newstring[21],
		    oldstring[22], newstring[22],
		    oldstring[23], newstring[23],
		    oldstring[24], newstring[24],
		    oldstring[25], newstring[25],
		    oldstring[26], newstring[26],
		    NULL))
      {
	 kwarn(KCMS, routine,
	       "Could not substitute `%s' for `%s' in file %s.",
	       newstring, oldstring, path);
      }
      else
      {
	 ksprintf(bakpath, "%s.bak", path);
	 if (krename(path, bakpath) == -1)
	 {
	    kerror(KCMS, routine, "Could not rename %s to %s", path, bakpath);
	 }
	 else if (krename(newpath, path) == -1)
	 {
	    kerror(KCMS, routine, "Could not rename %s to %s", newpath, path);
	 }
      }
   }
}
