/*
 * Khoros: $Id$
 */

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>       Purpose: Khoros Transport Declarations
   >>>>
   >>>>    Written By: Mark Young
   >>>>
   >>>>          Date: Jul 08, 1992
   >>>>
   >>>> Modifications: Converted from kfile.h in Khoros 1.0 (MY)
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#ifndef _kutils_kfile_h_
#define _kutils_kfile_h_


	/*-------------------------------------*
	|       #include 
	---------------------------------------*/


	/*-------------------------------------*
	|       #defines 
	---------------------------------------*/

/*
 *  Define the different open options for kopen()
#define KOPEN_RDONLY	0x00
#define KOPEN_WRONLY	0x01
#define KOPEN_RDWR	0x02
#define KOPEN_NONBLOCK	0x04
#define KOPEN_APPEND	0x08
#define KOPEN_CREAT	0x10
#define KOPEN_TRUNC	0x20
#define KOPEN_EXCL	0x40
#define KOPEN_NOCTTY	0x80
#define KOPEN_STREAM    0x100
 */
#define KOPEN_RDONLY	O_RDONLY
#define KOPEN_WRONLY	O_WRONLY
#define KOPEN_RDWR	O_RDWR
#define KOPEN_NONBLOCK	O_NONBLOCK
#define KOPEN_APPEND	O_APPEND
#define KOPEN_CREAT	O_CREAT
#define KOPEN_TRUNC	O_TRUNC
#define KOPEN_EXCL	O_EXCL
#define KOPEN_NOCTTY	O_NOCTTY
#define KOPEN_STREAM    0x100000
#define KOPEN_LOCK	0x400000

/*
 *  Define the access mode KOPEN_ACCMODE
 */
#define KOPEN_ACCMODE       (KOPEN_RDONLY|KOPEN_WRONLY|KOPEN_RDWR)

/*
 *  Define the different seek options for klseek()
 */
#define KSEEK_SET 0
#define KSEEK_CUR 1
#define KSEEK_END 2

/*
 *  Need to see if we are on a system V machine.  If so then we need to define
 *  LOCK_EX, LOCK_UN, lockf
 */
#define KLOCK_SH         1       /* shared lock */
#define KLOCK_EX         2       /* exclusive lock */
#define KLOCK_NB         4       /* don't block when locking */
#define KLOCK_UN         8       /* unlock */

/*
 *  kfile stderr, stdout, and stdin...
 */
#define kstderr ((kfile *) stderr)
#define kstdin  ((kfile *) stdin)
#define kstdout ((kfile *) stdout)

/*
 *  fileno for kstdin, kstdout, kstderr
 */
#define KSTDIN_FILENO    0
#define KSTDOUT_FILENO   1
#define KSTDERR_FILENO   2

/*
 *  kfile state flags...
 */
#define KFILE_READ	(1L << 0)
#define KFILE_RDWR	(1L << 1)
#define KFILE_WRITE	(1L << 2)
#define KFILE_MYBUF	(1L << 3)
#define KFILE_EOF	(1L << 4)
#define KFILE_ERR	(1L << 5)
#define KFILE_PERM	(1L << 6)
#define KFILE_OPEN	(1L << 7)
#define KFILE_DUP	(1L << 8)
#define KFILE_TEMP	(1L << 9)
#define KFILE_LOCK	(1L << 11)

/*
 *  Standard Inter-Process Communication Channel Identifiers
 */
#define KIPC_PROCESS	"Process"
#define KIPC_TRANSPORT  "Transport"


	/*-------------------------------------*
	|       typedefs
	---------------------------------------*/

/*
 *  typedef structure for kfile.  The internal structure definition of a
 *  Khoros File.  The kfile strucutre is used by the different transport
 *  mechanisms as internal information for the different operations such
 *  as reading, writing, opening, closing, etc.
 *
 *  The actual kfile defintion is located in:
 *
 *		 $BOOTSTRAP/include/kutils/kfileP.h
 */
typedef struct _kobject kfile;


	/*-------------------------------------*
	|       global variable declarations
	---------------------------------------*/


	/*-------------------------------------*
	|       macros
	---------------------------------------*/

/************************************************************
*
*  Routine Name: kgetc - get character from transport stream
*
*       Purpose: This function is a replacement for the system getc()
*		 routine.  The only difference is that kgetc() supports
*		 all available transport mechanisms.  The routine will get
*		 a single character from the specified transport.  If the
*		 character could not be read, EOF is returned; otherwise,
*		 the character that is read is returned.
*
*         Input: file - the kfile transport to be read which was opened earlier
*			with kfopen().
*        Output:
*       Returns: returns the character read or EOF upon error.
*
*  Restrictions: 
*    Written By: Mark Young
*          Date: Sep 12, 1992
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kgetc(
*		 !  kfile *file)
*
*************************************************************/

#define kgetc(file) kfgetc(file)


/************************************************************
*
*  Routine Name: kgets - reads from kstdin until a newline or EOF
*
*       Purpose: This function is a replacement for the system gets()
*		 routine.  The only difference is that kgets() uses
*		 the khoros transport mechanisms.  The routine will get
*		 a single line from kstdin.  It reads until a newline or
*		 end of file is encountered.
*
*         Input: buffer - the buffer to read the data into
*        Output:
*       Returns: returns the string read from kstdin or NULL upon error
*
*  Restrictions: 
*    Written By: Mark Young
*          Date: Feb 08, 1993
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: char *kgets(
*		 !  char *buffer)
*
*************************************************************/

#define kgets(buffer) kfgets(buffer, -1, kstdin)


/************************************************************
*
*  Routine Name: kputc - put a character onto the khoros transport stream
*
*       Purpose: This function is a replacement for the system "fputc()"
*		 routine.  The only difference is that kfputc() supports
*		 more than just files, it supports other data transports
*		 such as shared memory, pipes, files, etc.
*
*		 The routine will put a single character onto the
*		 specified transport.  If the character could not
*		 be written, EOF is returned; otherwise, the character
*		 that is written is returned.
*
*         Input: character - the character to be written
*		 file      - the kfile transport to be read which was
*			     opened earlier with kfopen().
*        Output:
*       Returns: returns the character written or EOF upon error.
*
*  Restrictions: 
*    Written By: Mark Young
*          Date: Sep 12, 1992
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kputc(
*		 !  int   character,
*		 !  kfile *file)
*
*************************************************************/

#define kputc(character, file) kfputc(character, file)


/************************************************************
*
*  Routine Name: kinput - opens a file for reading using kopen()
*
*       Purpose: This function is just a simplified interface to kopen()
*		 that opens a file for reading.  The macros calls
*		 kopen using the following syntax:
*
*		 !    kopen(filename, KOPEN_RDONLY, 0666)
*
*
*         Input: filename - the filename to be opened for reading using
*			     the kopen().
*        Output:
*       Returns: returns the kfile id on success, -1 otherwise
*
*  Restrictions: 
*    Written By: Mark Young
*          Date: Feb 01, 1993
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kinput(
*		 !  char *filename)
*
*************************************************************/

#define kinput(filename) kopen(filename, KOPEN_RDONLY, 0666)


/************************************************************
*
*  Routine Name: koutput - opens/creates a file for writing using kopen()
*
*       Purpose: This function is just a simplified interface to kopen()
*		 that creates and opens a file for writing.  The file is
*		 truncated upon a successful kopen call.  The macros calls
*		 kopen using the following syntax:
*
*		 ! kopen(filename, KOPEN_WRONLY|KOPEN_CREAT|KOPEN_TRUNC, 0666)
*
*
*         Input: filename - the filename to be opened for writing using
*			     the kopen().
*        Output:
*       Returns: returns the kfile id on success, -1 otherwise
*
*  Restrictions: 
*    Written By: Mark Young
*          Date: Feb 01, 1993
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int koutput(
*		 !  char *filename)
*
*************************************************************/

#define koutput(filename) kopen(filename, KOPEN_WRONLY|KOPEN_CREAT|KOPEN_TRUNC, 0666)


/************************************************************
*
*  Routine Name: kfinput - opens a file for reading using kfopen()
*
*       Purpose: This function is just a simplified interface to kfopen()
*		 that opens a file for reading.  The macros calls
*		 kfopen using the following syntax:
*
*		 !      kfopen(filename, "r")
*
*
*         Input: filename - the filename to be opened for reading using
*			     the kfopen().
*        Output:
*       Returns: returns the kfile structure on success, NULL upon failure
*
*  Restrictions: 
*    Written By: Mark Young
*          Date: Feb 01, 1993
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: kfile *kfinput(
*		 !  char *filename)
*
*************************************************************/

#define kfinput(filename) kfopen(filename, "r")


/************************************************************
*
*  Routine Name: kfoutput - opens/creates a file for writing using kfopen()
*
*       Purpose: This function is just a simplified interface to kfopen()
*		 that creates and opens a file for writing.  The file is
*		 truncated upon a successful kfopen call.  The macros calls
*		 kfopen using the following syntax:
*
*			! kfopen(filename, "w")
*
*
*         Input: filename - the filename to be opened for writing using
*			     the kfopen().
*        Output:
*       Returns: returns the kfile structure on success, NULL upon failure
*
*  Restrictions: 
*    Written By: Mark Young
*          Date: Feb 01, 1993
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration:
*   Declaration: kfile *kfoutput(
*		 !  char *filename)
*
*************************************************************/

#define kfoutput(filename) kfopen(filename, "w")


/************************************************************
*
*  Routine Name: kfile_isread - khoros stream transport is readable
*
*       Purpose: Whether a khoros stream transport is readable.
*
*         Input: file - the khoros stream transport to test whether readable
*        Output: 
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Apr 29, 1993
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kfile_isread(
*		 !  kfile *file)
*
*************************************************************/

#define kfile_isread(file)  ((kfile_state(file) & KFILE_READ || \
			     kfile_state(file) & KFILE_RDWR))


/************************************************************
*
*  Routine Name: kfile_iswrite - khoros stream transport is writeable
*
*       Purpose: Whether a khoros stream transport is writeable.
*
*         Input: file - the khoros stream transport to test whether writeable
*        Output: 
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Apr 29, 1993
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kfile_iswrite(
*		 !  kfile *file)
*
*************************************************************/

#define kfile_iswrite(file)  ((kfile_state(file) & KFILE_WRITE || \
			        kfile_state(file) & KFILE_RDWR))


/************************************************************
*
*  Routine Name: kfile_isrdwr - khoros stream transport is both readable and
*				writeable
*
*       Purpose: Whether a khoros stream transport is readable and writeable.
*
*         Input: file - the khoros stream transport to test whether it's
*			readable and writeable
*        Output:
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Apr 29, 1993
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kfile_isrdwr(
*		 !  kfile *file)
*
*************************************************************/

#define kfile_isrdwr(file)  ((kfile_state(file) & KFILE_RDWR))


/************************************************************
*
*  Routine Name: kfile_ismybuf - khoros stream transport stream buffer was set
*				 by the application
*
*       Purpose: Whether a khoros stream transport's read/write buffer was
*		 set by the programmer or was internally allocated by the
*		 open transport process.
*
*         Input: file - the khoros stream transport to test whether the
*			stream buffer was set by the programmer.
*        Output:
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Apr 29, 1993
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kfile_ismybuf(
*		 !  kfile *file)
*
*************************************************************/

#define kfile_ismybuf(file)  ((kfile_state(file) & KFILE_MYBUF))


/************************************************************
*
*  Routine Name: kfile_ispermanent - khoros stream transport has permanence 
*
*       Purpose: Whether a khoros stream transport has data permanence.  If
*		 a transport has permanence then the data can be accessed
*		 multiple times.  In the case of files, shared memory, memory
*		 mapped files, these transports have permanence.  Where as
*		 for pipes and sockets the data can only be accessed (read or
*		 written) to once.
*
*         Input: file - the khoros stream transport to test whether the
*			transport has data permanence.
*        Output:
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Apr 29, 1993
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kfile_ispermanent(
*		 !  kfile *file)
*
*************************************************************/

#define kfile_ispermanent(file)  ((kfile_state(file) & KFILE_PERM))


/************************************************************
*
*  Routine Name: kfile_istemp - khoros stream transport is temporary
*
*       Purpose: Whether a khoros stream transport has been created as
*		 a temporary.  A transport created via ktempnam() or
*		 kfile_tempnam() is flagged as a temporary transport.
*
*         Input: file - the khoros stream transport to test whether the
*			transport is a temporary
*        Output:
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions:
*    Written By: Mark Young
*          Date: May 11, 1993
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kfile_istemp(
*		 !  kfile *file)
*
*************************************************************/

#define kfile_istemp(file)  ((kfile_state(file) & KFILE_TEMP))


/************************************************************
*
*  Routine Name: kfile_isopen - khoros stream transport stream has been 
*				properly opened
*
*       Purpose: Whether a khoros stream transport has been opened.
*
*         Input: file - the khoros stream transport to test whether the
*			transport has been opened.
*        Output:
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Apr 29, 1993
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kfile_isopen(
*		 !  kfile *file)
*
*************************************************************/

#define kfile_isopen(file)  ((kfile_state(file) & KFILE_OPEN))


/************************************************************
*
*  Routine Name: kfile_isdup - khoros stream transport stream has been 
*			       has been/is a dupped transport
*
*       Purpose: Whether a khoros stream transport has been or is a
*		 dupped transport.  A transport can be dupped by
*		 the routines kdup(), kdup2(), kfdup(), kfdup2().
*
*         Input: file - the khoros stream transport to test whether the
*			transport has been dupped.
*        Output:
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Apr 29, 1993
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kfile_isdup(
*		 !  kfile *file)
*
*************************************************************/

#define kfile_isdup(file)  ((kfile_state(file) & KFILE_DUP))


/************************************************************
*
*  Routine Name: kfile_iseof - khoros stream transport stream is at end of
*			       file (eof)
*
*       Purpose: Whether a khoros stream transport is currently at end of
*		 file.  A transport end of file flag can be cleared using
*		 the kclearerr() routine.
*
*         Input: file - the khoros stream transport to test whether the
*			transport is at end of file (eof).
*        Output:
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Apr 29, 1993
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kfile_iseof(
*		 !  kfile *file)
*
*************************************************************/

#define kfile_iseof(file)   ((kfile_state(file) & KFILE_EOF))


/************************************************************
*
*  Routine Name: kfile_islock - khoros stream transport stream has been 
*			        has been/is locked transport
*
*       Purpose: Whether a khoros stream transport has been or is a
*		 locked transport.  A transport can be locked using
*		 the KFILE_LOCK flag on open.
*
*         Input: file - the khoros stream transport to test whether the
*			transport has been opened lock.
*        Output:
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Jan 13, 1995
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kfile_islock(
*		 !  kfile *file)
*
*************************************************************/

#define kfile_islock(file)  ((kfile_state(file) & KFILE_LOCK))


#define kfile_state(file) kfile_getstate(file)
	/*-------------------------------------*
	|       routine definitions
	---------------------------------------*/

/*
 *  Declaration for public routines.  These routines should used instead of
 *  their system equivalents.  For the most part they have the same parameters
 *  as their system counterparts, but a few are different so you should consult
 *  the appropriate cgparm documentation.  For online doc consult the man3
 *  vgparm man page or for offline doc the section on vgparm in chapter 8 of
 *  the "Khoros Programmer's Manual".
 */

/* ktransport.c */
kfile *kpopen		PROTO((char *, char *));
char  *ktempnam		PROTO((char *, char *));
char  *kmakecommand	PROTO((char *, char *, char *));
int   krename		PROTO((char *, char *));
int   kopen		PROTO((char *, int, int));
int   kclose		PROTO((int));
int   kread		PROTO((int, char *, int));
int   kwrite		PROTO((int, char *, int));
int   kaccess		PROTO((char *, int));
int   kunlink		PROTO((char *));
int   kcreat		PROTO((char *, int));
int   ktouch		PROTO((char *, int));
int   kflock		PROTO((int, int));
int   klseek		PROTO((int, int, int));
int   ktell		PROTO((int));
int   kdup		PROTO((int));
int   kdup2		PROTO((int, int));
int   kpclose		PROTO((kfile *));
int   kpinfo		PROTO((kfile *));
int   kspawn		PROTO((char *));
int   ksystem		PROTO((char *));
int   kexecvp		PROTO((char *, char **));
int   kfileno		PROTO((kfile *));
int   kgetdescriptors	PROTO((kfile *, int *, int *));
int   kgetarchitecture	PROTO((char *, char *));
int   kgethostname	PROTO((char *, char *, int));
void  kexit		PROTO((kexit_status));
int   kexit_handler	PROTO((kfunc_void, kaddr));


/* kread.c */
int kread_bit	   PROTO((int, unsigned char *, int));
int kread_byte	   PROTO((int, char *, int));
int kread_short	   PROTO((int, short *, int));
int kread_int	   PROTO((int, int *, int));
int kread_long	   PROTO((int, long *, int));
int kread_float	   PROTO((int, float *, int));
int kread_double   PROTO((int, double *, int));
int kread_ubyte    PROTO((int, unsigned char *, int));
int kread_ushort   PROTO((int, unsigned short *, int));
int kread_uint     PROTO((int, unsigned int *, int));
int kread_ulong    PROTO((int, unsigned long *, int));
int kread_complex  PROTO((int, kcomplex *, int));
int kread_dcomplex PROTO((int, kdcomplex *, int));
int kread_generic  PROTO((int, kaddr, int, int));
int kfread_generic PROTO((kfile *, kaddr, int, int));


/* kwrite.c */
int kwrite_bit	    PROTO((int, unsigned char *, int));
int kwrite_byte     PROTO((int, char *, int));
int kwrite_short    PROTO((int, short *, int));
int kwrite_int      PROTO((int, int *, int));
int kwrite_long     PROTO((int, long *, int));
int kwrite_float    PROTO((int, float *, int));
int kwrite_double   PROTO((int, double *, int));
int kwrite_ubyte    PROTO((int, unsigned char *, int));
int kwrite_ushort   PROTO((int, unsigned short *, int));
int kwrite_uint     PROTO((int, unsigned int *, int));
int kwrite_ulong    PROTO((int, unsigned long *, int));
int kwrite_complex  PROTO((int, kcomplex *, int));
int kwrite_dcomplex PROTO((int, kdcomplex *, int));
int kwrite_generic  PROTO((int, kaddr, int, int));
int kfwrite_generic PROTO((kfile *, kaddr, int, int));


/* kstdio.c */
kfile *kfopen		PROTO((char *, char *));
kfile *kfdopen		PROTO((int, char *));
kfile *kfreopen		PROTO((char *, char *, kfile *));
kfile *kfdup		PROTO((kfile *));
kfile *kfdup2		PROTO((kfile *, kfile *));
kfile *ktmpfile		PROTO((void));
char  *kfgets		PROTO((char *, int, kfile *));
char  *kgetbuffer	PROTO((kfile *, int *));
void  krewind		PROTO((kfile *));
void  kclearerr		PROTO((kfile *));
void  ksetbuffer	PROTO((kfile *, char *, int));
int   kscanf		PROTO((char *, kvalist));
int   kfscanf		PROTO((kfile *, char *, kvalist));
int   ksscanf		PROTO((char *, char *, kvalist));
int   kprintf		PROTO((char *, kvalist));
int   kfprintf		PROTO((kfile *, char *, kvalist));
int   ksprintf		PROTO((char *, char *, kvalist));
int   kvfprintf		PROTO((kfile *, char *, kva_list));
int   kvsprintf		PROTO((char *, char *, kva_list));
int   kvscanf		PROTO((char *, kva_list));
int   kvfscanf		PROTO((kfile *, char *, kva_list));
int   kvsscanf		PROTO((char *, char *, kva_list));
int   kvprintf		PROTO((char *, kva_list));
int   kfclose		PROTO((kfile *));
int   kfread		PROTO((kaddr, int, int, kfile *));
int   kfwrite		PROTO((kaddr, int, int, kfile *));
int   kfputc		PROTO((int, kfile *));
int   kfputs		PROTO((char *, kfile *));
int   kfgetc		PROTO((kfile *));
int   kungetc		PROTO((int, kfile *));
int   kfseek		PROTO((kfile *, long, int));
int   kftell		PROTO((kfile *));
int   kfflush		PROTO((kfile *));
int   kfeof		PROTO((kfile *));

/*
 *  Other utilities that maybe handy but are usually needed only by more
 *  complex applications that wish to do more than the basic transport/
 *  remote execution.
 */
/* kroutines.c */
kaddr kfile_readdata	   PROTO((int, kaddr, int *));
char  **kfile_listhost	   PROTO((int *));
char  **ktransport_list	   PROTO((int, int, int, int *));
char  *kfile_filename	   PROTO((int));
char  *kfile_type	   PROTO((kfile *));
kfile *kfile_reopen	   PROTO((char *, char *, kfile *));
int   kfile_flags	   PROTO((int));
int   kfile_getstate	   PROTO((kfile *));
int   kfile_setstate	   PROTO((kfile *, int));
int   kfile_clrstate	   PROTO((kfile *, int));
int   kfile_mode	   PROTO((int));
int   kfile_closeall	   PROTO((void));
int   kfile_getpermanence  PROTO((char *));
int   kfile_getmachtype    PROTO((int));
int   kfile_setmachtype	   PROTO((int, int));
int   kfile_remotehost	   PROTO((char *));
int   kfile_writedata	   PROTO((int, kaddr, int));
int   kfile_seddata	   PROTO((int, int, int, kvalist));
int   kfile_copydata	   PROTO((int, int, int));
int   kfile_comparedata	   PROTO((int, int, int, int *));
void  kfile_listall	   PROTO((void));

char  *kflags_to_type	   PROTO((int, char *));
int   ktype_to_flags	   PROTO((char *));
void  kcommand_to_args	   PROTO((char *, char **));

/*
 *  Transport negotiator which can be used by processes that wishes to
 *  arbitrate or simply recieve information about what transport state
 *  a process maybe in.  (developed with cantata in mind)
 */
/* knegotiator.c */
void knegotiate_transport PROTO((int));
int  knegotiate_open	  PROTO((char *, int));
int  knegotiate_close	  PROTO((char *, int));

/* kipc.c */
int kipc_start		 PROTO((char *));
int kipc_stop		 PROTO((char *));
int kipc_sendmsg	 PROTO((char *, char *));
int kipc_recvmsg	 PROTO((char *, char *));

int kipc_setdescriptors	 PROTO((char *, char *,   char *));
int kipc_getdescriptors	 PROTO((char *, kfile **, kfile **));

#endif /* _kutils_kfile_h_ */
/* don`t add after the endif */
