/*
 * Khoros: $Id$
 */

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>       Purpose: Definitions and prototypes for library klibm
   >>>>
   >>>>       Toolbox: bootstrap
   >>>>
   >>>>    Written By: Jeremy Worley & Mark Young
   >>>>
   >>>>          Date: Apr 04, 1994
   >>>>
   >>>> Modifications:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#ifndef _klibm_klibm_h_
#define _klibm_klibm_h_


	/*-------------------------------------*
	|       #include
	---------------------------------------*/

#include <math.h>
#include <klibm/kdata.h>


	/*-------------------------------------*
	|       #defines
	---------------------------------------*/

/*
 * Standard data type defines
 *	KANYTYPE   all ones
 *	KBIT	   1
 *	KBYTE      2
 *	KUBYTE     4
 *	KSHORT     8
 *	KUSHORT    16
 *	KINT       32
 *	KUINT      64
 *	KLONG      128
 *	KULONG     256
 *	KFLOAT     512
 *	KDOUBLE    1024
 *	KCOMPLEX   2048
 *	KDCOMPLEX  4096
 *	KSTRING    8192
 *	KSTRUCT    16384
 *	KLOGICAL   32768
 *	KBIT_LSB   65536
 */
#define KANYTYPE	(~0L)
#define KBIT		(1L<<0)
#define KBYTE		(1L<<1)
#define KUBYTE		(1L<<2)
#define KSHORT		(1L<<3)
#define KUSHORT		(1L<<4)
#define KINT		(1L<<5)
#define KUINT		(1L<<6)
#define KLONG		(1L<<7)
#define KULONG		(1L<<8)
#define KFLOAT		(1L<<9)
#define KDOUBLE		(1L<<10)
#define KCOMPLEX	(1L<<11)
#define KDCOMPLEX	(1L<<12)
#define KSTRING		(1L<<13)
#define KSTRUCT		(1L<<14)
#define KBIT_LSB	(1L<<15)
#define KLOGICAL	(1L<<16)

        /*
         *  Definitions for some possibly machine dependent constants, more
         *  dependable than math.h since math.h is often missing (Way to go
         *  SYS V!).  Some installations may want to put values in here that
         *  more closely match the machine.
         */
#define KMAXFLOAT 1.0e+38       /* Good enough for most applications */
#define KMAXLINT  2147483647    /* Assuming 32 bit 2's complement */
#define KMAXSINT  32767         /* Assuming 16 bit 2's complement */
#define KGAMMA    0.57721566490153286060
#define KDEG      57.29577951308232087680
#define KPHI      1.61803398874989484820
#define KE        2.71828182845904523536
#define KLOG2E    1.4426950408889634074
#define KLOG10E   0.43429448190325182765
#define KLN2      0.69314718055994530942
#define KLN10     2.30258509299404568402
#define KPI       3.14159265358979323846
#define K2PI      (KPI * 2)
#define KPI_2     1.57079632679489661923
#define KPI_4     0.78539816339744830962
#define K1_PI     0.31830988618379067154
#define K2_PI     0.63661977236758134308
#define KSQRTPI   1.77245385090551602730
#define KSQRT2    1.41421356237309504880
#define KSQRT1_2  0.70710678118654752440
#define KEPSILON  0.00001

#define KDATAOP_NOP       0
#define KDATAOP_SCALE     1
#define KDATAOP_NORMALIZE 2

#define KLIN_DETERMINATE  0x1
#define KLIN_INVERSE      0x2

	/*-------------------------------------*
	|       typedefs
	---------------------------------------*/

	/*-------------------------------------*
	|       global variable declarations
	---------------------------------------*/

	/*-------------------------------------*
	|       macros
	---------------------------------------*/

/************************************************************
*
*  Routine Name: kmax - return the greater of two values.
*       Purpose: The kmax function obtains the larger of the two
*		 input arguments.
*		 This is a macro, so any data type is supported.
*         Input: x - a variable of any base data type.
*	         y - a variable of any base data type.
*        Output: none
*       Returns: the larger value of the two input arguments.
*  Restrictions:
*    Written By: The Mystery Machine
*          Date: Aug 10, 1994 11:58
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: kmax(x, y)
*
*************************************************************/

#define kmax(x,y) (((x) > (y)) ? (x) : (y))

/************************************************************
*
*  Routine Name: kmin - return the lessor of two values.
*       Purpose: The kmin function obtains the smaller of the two input
*		 arguments.
*		 This is a macro, so any data type is supported.
*         Input: x - a variable of any base data type.
*	         y - a variable of any base data type.
*        Output: none
*       Returns: the smaller value of the two input arguments.
*  Restrictions:
*    Written By: The Mystery Machine
*          Date: Aug 10, 1994 11:58
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: kmin(x, y)
*
*************************************************************/

#define kmin(x,y) (((x) < (y)) ? (x) : (y))

#define krange(x, w, y) (kmax(x, kmin(w, y)))
#define kmin3(x0, x1, x2) ((kmin(x0,x1) < x2) ? kmin(x0,x1) : x2)
#define kmax3(x0, x1, x2) ((kmax(x0,x1) > x2) ? kmax(x0,x1) : x2)
#define kmin4(x0, x1, x2, x3) ((kmin(x0,x1) < kmin(x2,x3)) \
                                          ? kmin(x0,x1) : kmin(x2,x3))
#define kmax4(x0, x1, x2, x3) ((kmax(x0,x1) > kmax(x2,x3)) \
                                          ? kmax(x0,x1) : kmax(x2,x3))
#define kabs(x) (((x) < 0) ? (-(x)) : (x))
#define ksqr(x) ((x)*(x))
#define kdegrees_radians(degrees) ((KPI/180.0)*(degrees))
#define kradians_degrees(radians) ((180.0/KPI)*(radians))
 
/*
 *   The following macro is used to interpolate the intercept from a vector
 *   described by the two points (x0,y0) & (x1,y1) and a value along the
 *   vector, y2, the macro will return the corresponding x2 value.
 */
#define kintercept(x0,x1,y0,y1,y2)   ((y1) != (y0) \
        ? ((x0) + (((x1) - (x0))*((y2) - (y0)))/((y1) - (y0))) \
        : (((x0) + (x1)) / 2.0))


/*
 * interfaces for the posix libm.a
 */

/************************************************************
*
*  Routine Name: kacos - compute the arc cosine of the argument.
*
*       Purpose: The kacos function computes the arc cosine
*		 of the double precision input argument.
*
*		 This is provided in all POSIX and X/OPEN 
*		 compliant C math libraries.  Currently
*		 kacos is simply a macro to acos because
*		 Khoros does not run on any machine whose
*		 math library does not include this function.
*		 If Khoros is ported to a machine without this
*		 function then a replacement will be written.
*
*         Input: x - input argument to take arc cosine of.
*
*        Output:
*
*	Returns: the double precision arc cosine of the
*		 double precision input argument.
*               
*  Restrictions: 
*    Written By: Jeremy Worley 
*          Date: Mar 31, 1993 09:36
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: double kacos(double x)
*
*************************************************************/

#define kacos acos

/************************************************************
*
*  Routine Name: kasin - compute the arc sine of the 
*			 argument.
*
*       Purpose: The kasin function computes arc sine of
*		 the double precision input argument, whose
*		 value lies between -1 and 1.
*
*		 This is provided in all POSIX and X/OPEN 
*		 compliant C math libraries.  Currently
*		 kasin is simply a macro to asin because
*		 Khoros does not run on any machine whose
*		 math library does not include this function.
*		 If Khoros is ported to a machine without this
*		 function then a replacement will be written.
*
*         Input: x - input argument to take arc sine of.
*
*        Output:
*
*	Returns: the double precision arc sine of the
*		 input argument.
*               
*  Restrictions: 
*    Written By: Jeremy Worley 
*          Date: Mar 31, 1993 09:36
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: double kasin(double x)
*
*************************************************************/

#define kasin asin

/************************************************************
*
*  Routine Name: katan2 - compute the arc tangent of y/x.
*
*       Purpose: The katan2 function converts the rectangular
*		 coordinates (x,y) into polar coordinates 
*		 and returns the phase component in the range
*		 (-pi/2,pi/2)
*
*		 This is provided in all POSIX and X/OPEN 
*		 compliant C math libraries.  Currently
*		 katan2 is simply a macro to atan2 because
*		 Khoros does not run on any machine whose
*		 math library does not include this function.
*		 If Khoros is ported to a machine without this
*		 function then a replacement will be written.
*
*         Input: y - numerator component of the tangent given
*		     y/x.
*		 x - the non-zero denominator component of the tangent given
*		     by y/x.  
*
*        Output: 
*
*	Returns: the double precision arc tangent of the
*		 input arguments.
*               
*  Restrictions: The second argument x must be non-zero.
*    Written By: Jeremy Worley 
*          Date: Mar 31, 1993 09:36
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: double katan2(
*		   double y,
*		   double x)
*
*************************************************************/

#define katan2 atan2

/************************************************************
*
*  Routine Name: katan - compute the arc tangent of the argument.
*
*       Purpose: The katan function computes arc tangent of
*		 the double precison input argument.  The result
*		 is an angle with the value between -pi/2 and pi/2
*		 radians.
*
*		 This is provided in all POSIX and X/OPEN 
*		 compliant C math libraries.  Currently
*		 katan is simply a macro to atan because
*		 Khoros does not run on any machine whose
*		 math library does not include this function.
*		 If Khoros is ported to a machine without this
*		 function then a replacement will be written.
*
*         Input: x - input argument to take arc tangent of.
*
*        Output:
*
*	Returns: the double precision arc tangent of the
*		 input argument.
*               
*  Restrictions: 
*    Written By: Jeremy Worley 
*          Date: Mar 31, 1993 09:36
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: double katan(double x)
*
*************************************************************/

#define katan atan

/************************************************************
*
*  Routine Name: ksinh - compute the hyperbolic sine of the argument.
*
*       Purpose: The ksinh function computes the hyperbolic
*		 sine of the input argument
*
*		 This is provided in all POSIX and X/OPEN 
*		 compliant C math libraries.  Currently
*		 ksinh is simply a macro to sinh because
*		 Khoros does not run on any machine whose
*		 math library does not include this function.
*		 If Khoros is ported to a machine without this
*		 function then a replacement will be written.
*
*         Input: x - input argument to compute the hyperbolic
*		     sine on.
*
*        Output:
*
*	Returns: the double precision hyperbolic sine of the
*		 input argument
*               
*  Restrictions: 
*    Written By: Jeremy Worley 
*          Date: Mar 31, 1993 09:36
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: double ksinh(double x)
*
*************************************************************/

#define ksinh sinh

/************************************************************
*
*  Routine Name: kcosh - compute the hyperbolic cosine of the argument.
*
*       Purpose: The kcosh function computes the hyperbolic
*		 cosine of the double precison input argument.
*
*		 This is provided in all POSIX and X/OPEN compliant C
*		 math libraries.  Currently kcosh is simply a macro to
*		 cosh because Khoros does not run on any machine whose
*		 math library does not include this function.  If
*		 Khoros is ported to a machine without this function
*		 then a replacement will be written.
*
*         Input: x - input argument to compute the double precision hyperbolic
*		     cosine on.
*
*        Output:
*
*	Returns: the double precision hyperbolic cosine of the
*		 input argument or KMAXFLOAT if the
*		 input argument is less than one.
*               
*  Restrictions: 
*    Written By: Jeremy Worley 
*          Date: Mar 31, 1993 09:36
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: double kcosh(double x)
*
*************************************************************/

#define kcosh cosh

/************************************************************
*
*  Routine Name: ktanh - compute the hyperbolic tangent of the argument.
*
*       Purpose: The ktanh function computes the hyperbolic
*		 tangent of the double precision input argument
*
*		 This is provided in all POSIX and X/OPEN 
*		 compliant C math libraries.  Currently
*		 ktanh is simply a macro to tanh because
*		 Khoros does not run on any machine whose
*		 math library does not include this function.
*		 If Khoros is ported to a machine without this
*		 function then a replacement will be written.
*
*         Input: x - input argument to compute the hyperbolic
*		     tangent on.
*
*        Output:
*
*	Returns: the double precision hyperbolic tangent of the
*		 input argument
*               
*  Restrictions: 
*    Written By: Jeremy Worley 
*          Date: Mar 31, 1993 09:36
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: double ktanh(double x)
*
*************************************************************/

#define ktanh tanh

/************************************************************
*
*  Routine Name: ksin - compute the double precision sine of the argument.
*
*       Purpose: The ksin function computes the sine of the
*		 double precision input argument, which represents
*		 an angle in radians.
*
*		 This is provided in all POSIX and X/OPEN 
*		 compliant C math libraries.  Currently
*		 ksin is simply a macro to sin because
*		 Khoros does not run on any machine whose
*		 math library does not include this function.
*		 If Khoros is ported to a machine without this
*		 function then a replacement will be written.
*
*         Input: x - radian input argument to compute the sine
*		     on.
*
*        Output:
*
*	Returns: the double precision sine of the input argument
*               
*  Restrictions: 
*    Written By: Jeremy Worley 
*          Date: Mar 31, 1993 09:36
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: double ksin(double x)
*
*************************************************************/

#define ksin sin

/************************************************************
*
*  Routine Name: kcos - compute the double precision cosine of the argument.
*
*       Purpose: The kcos function computes the cosine of the
*		 double precision input argument.
*
*		 This is provided in all POSIX and X/OPEN 
*		 compliant C math libraries.  Currently
*		 kcos is simply a macro to cos because
*		 Khoros does not run on any machine whose
*		 math library does not include this function.
*		 If Khoros is ported to a machine without this
*		 function then a replacement will be written.
*
*         Input: x - radian input argument to compute the cosine
*		     on.
*
*        Output:
*
*	Returns: the double precision cosine of the input argument
*               
*  Restrictions: 
*    Written By: Jeremy Worley 
*          Date: Mar 31, 1993 09:36
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: double kcos(double x)
*
*************************************************************/

#define kcos cos

/************************************************************
*
*  Routine Name: ktan - compute the double precision tangent of the argument.
*
*       Purpose: The ktan function computes the tangent of the
*		 input argument.
*
*		 This is provided in all POSIX and X/OPEN 
*		 compliant C math libraries.  Currently
*		 ktan is simply a macro to tan because
*		 Khoros does not run on any machine whose
*		 math library does not include this function.
*		 If Khoros is ported to a machine without this
*		 function then a replacement will be written.
*
*         Input: x - radian input argument to compute the tangent
*		     on.
*
*        Output:
*
*	Returns: the double precision tangent of the input argument
*               
*  Restrictions: 
*    Written By: Jeremy Worley 
*          Date: Mar 31, 1993 09:36
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: double ktan(double x)
*
*************************************************************/

#define ktan tan

/************************************************************
*
*  Routine Name: kceil - compute the ceiling of the argument.
*
*       Purpose: The kceil function computes the least integral
*		 value (smallest integer value) greater than or equal 
*		 to the input argument and returns it as a double.
*
*		 This is provided in all POSIX and X/OPEN 
*		 compliant C math libraries.  Currently
*		 kceil is simply a macro to ceil because
*		 Khoros does not run on any machine whose
*		 math library does not include this function.
*		 If Khoros is ported to a machine without this
*		 function then a replacement will be written.
*
*         Input: x - number to calculate ceiling of.
*
*        Output:
*
*	Returns: the double precision ceiling of the input argument.
*               
*  Restrictions: 
*    Written By: Jeremy Worley 
*          Date: Mar 31, 1993 09:36
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: double kceil(double x)
*
*************************************************************/

#define kceil ceil

/************************************************************
*
*  Routine Name: kfloor - compute the floor of the argument.
*
*       Purpose: The kfloor function computes the largest integral
*		 value (largest integer) less than or equal to the 
*		 input argument and returns it as a double.
*
*		 This is provided in all POSIX and X/OPEN 
*		 compliant C math libraries.  Currently
*		 kfloor is simply a macro to floor because
*		 Khoros does not run on any machine whose
*		 math library does not include this function.
*		 If Khoros is ported to a machine without this
*		 function then a replacement will be written.
*
*         Input: x - number to calculate floor of.
*
*        Output:
*
*	Returns: the double precision floor of the input argument.
*               
*  Restrictions: 
*    Written By: Jeremy Worley 
*          Date: Mar 31, 1993 09:36
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: double kfloor(double x)
*
*************************************************************/

#define kfloor floor

/************************************************************
*
*  Routine Name: kexp - compute the exponential of the argument.
*
*       Purpose: The kexp function computes the value of
*		 the exponential of the double precision input
*		 argument (e**x).  
*
*		 This is provided in all POSIX and X/OPEN 
*		 compliant C math libraries.  Currently
*		 kexp is simply a macro to exp because
*		 Khoros does not run on any machine whose
*		 math library does not include this function.
*		 If Khoros is ported to a machine without this
*		 function then a replacement will be written.
*
*         Input: x - variable whose exponential is to be computed.
*
*        Output:
*
*	Returns: the double precision result of the exponential of the 
*		 input argument.
*               
*  Restrictions: 
*    Written By: Jeremy Worley 
*          Date: Mar 31, 1993 09:36
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: double kexp(double x)
*
*************************************************************/

#define kexp exp

/************************************************************
*
*  Routine Name: klog - compute the natural logarithm of the argument.
*
*       Purpose: The klog function computes natural logarithm
*		 of the double precision input argument.
*
*		 This is provided in all POSIX and X/OPEN 
*		 compliant C math libraries.  Currently
*		 klog is simply a macro to log because
*		 Khoros does not run on any machine whose
*		 math library does not include this function.
*		 If Khoros is ported to a machine without this
*		 function then a replacement will be written.
*
*         Input: x - argument to the natural log function.
*
*        Output:
*
*	Returns: the double precision logarithm of the input argument
*               
*  Restrictions: 
*    Written By: Jeremy Worley 
*          Date: Mar 31, 1993 09:36
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: double klog(double x)
*
*************************************************************/

#define klog log

/************************************************************
*
*  Routine Name: klog10 - compute the base 10 logarithm of the argument.
*
*       Purpose: The klog10 function computes the base 10 logarithm
*		 of the double precision input argument.
*
*		 This is provided in all POSIX and X/OPEN 
*		 compliant C math libraries.  Currently
*		 klog10 is simply a macro to log10 because
*		 Khoros does not run on any machine whose
*		 math library does not include this function.
*		 If Khoros is ported to a machine without this
*		 function then a replacement will be written.
*
*         Input: x - argument to the base 10 log function.
*
*        Output:
*
*	Returns: the double precision base 10 logarithm of the input
*		 argument.
*               
*  Restrictions: 
*    Written By: Jeremy Worley 
*          Date: Mar 31, 1993 09:36
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: double klog10(double x)
*
*************************************************************/

#define klog10 log10


/************************************************************
*
*  Routine Name: kpow - compute x to the y power.
*
*       Purpose: The kpow function computes x**y, which is the value of
*		 one double precision input argument x raised to the
*		 power of another double precision input argument y.
*
*		 This is provided in all POSIX and X/OPEN 
*		 compliant C math libraries.  Currently
*		 kpow is simply a macro to pow because
*		 Khoros does not run on any machine whose
*		 math library does not include this function.
*		 If Khoros is ported to a machine without this
*		 function then a replacement will be written.
*
*         Input: x - base of the exponentiation operation.
*		 y - exponent to raise the base to.
*
*        Output:
*
*	Returns: the double precision result of the input argument x
*		 raised to the power of the input argument y.
*               
*  Restrictions: Neither argument can be 0.
*    Written By: Jeremy Worley 
*          Date: Mar 31, 1993 09:36
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: double kpow(
*		   double x,
*		   double y)
*
*************************************************************/

#define kpow pow

/************************************************************
*
*  Routine Name: ksqrt - compute the square root of the argument.
*
*       Purpose: The ksqrt function computes the square root
*		 of the double precision input argument.
*
*		 This is provided in all POSIX and X/OPEN 
*		 compliant C math libraries.  Currently
*		 ksqrt is simply a macro to sqrt because
*		 Khoros does not run on any machine whose
*		 math library does not include this function.
*		 If Khoros is ported to a machine without this
*		 function then a replacement will be written.
*
*         Input: x - number to take square root of.
*
*        Output:
*
*	Returns: the double precision square root of the input
*		 argument.
*               
*  Restrictions: 
*    Written By: Jeremy Worley 
*          Date: Mar 31, 1993 09:36
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: double ksqrt(double x)
*
*************************************************************/

#define ksqrt sqrt

/************************************************************
*
*  Routine Name: kfabs - compute the absolute value of the argument.
*
*       Purpose: The kfabs function computes the absolute
*		 value of the double precision input argument.
*
*		 This is provided in all POSIX and X/OPEN 
*		 compliant C math libraries.  Currently
*		 kfabs is simply a macro to fabs because
*		 Khoros does not run on any machine whose
*		 math library does not include this function.
*		 If Khoros is ported to a machine without this
*		 function then a replacement will be written.
*
*         Input: x - number to take the absolute value of.
*
*        Output:
*
*	Returns: the double precision absolute value of the
*		 input argument.
*               
*  Restrictions: 
*    Written By: Jeremy Worley 
*          Date: Mar 31, 1993 09:36
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: double kfabs(double x)
*
*************************************************************/

#define kfabs fabs

/************************************************************
*
*  Routine Name: kfmod - compute the floating point modulo of the arguments.
*
*       Purpose: The kfmod function computes the floating
*		 point remainder of x/y.
*
*		 This is provided in all POSIX and X/OPEN 
*		 compliant C math libraries.  On most machines,
*		 kfmod is a macro to fmod.  On certain 
*		 machines that are not fully POSIX compliant,
*		 this is a macro to an internal implementation
*		 of fmod.
*
*         Input: x - numerator of division operation.
*                y - denominator of division operation.
*
*        Output:
*
*	Returns: the double precision modulo remainder of x/y. 
*               
*  Restrictions: 
*    Written By: Jeremy Worley 
*          Date: Aug 01, 1991 15:30
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: double kfmod(
*		    double x,
*		    double y)
*
*************************************************************/

#if defined(KFMOD_DEF)
#  define kfmod __kfmod
#else
#  define kfmod fmod
   double  fmod  PROTO((double, double));
#endif

/************************************************************
*
*  Routine Name: kfrexp - compute significand and exponent of argument 
*
*       Purpose: The kfrexp function computes the significand  (or mantissa)
*		 of v as a double quantity between 0.5 and 1.0 and stores the 
*		 exponent in the output argument e.
*
*		 This is provided in all POSIX and X/OPEN 
*		 compliant C math libraries.  Currently
*		 kfrexp is simply a macro to frexp because
*		 Khoros does not run on any machine whose
*		 math library does not include this function.
*		 If Khoros is ported to a machine without this
*		 function then a replacement will be written.
*
*         Input: v - value to compute on.
*
*        Output: e - exponent of the input argument.
*
*	Returns: the significand of the input argument, or zero if the input
*		 argument is zero.
*               
*  Restrictions: 
*    Written By: Jeremy Worley 
*          Date: Mar 31, 1993 10:38
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: double kfrexp(
*		    double v,
*		    double *e)
*
*************************************************************/

#define kfrexp frexp

/************************************************************
*
*  Routine Name: kldexp - computes x * 2**n
*
*       Purpose: The kldexp function computes x * 2**n
*		 by performing exponent manipulation rather
*		 than multiplication.
*
*		 This is provided in all POSIX and X/OPEN 
*		 compliant C math libraries.  Currently
*		 kldexp is simply a macro to ldexp because
*		 Khoros does not run on any machine whose
*		 math library does not include this function.
*		 If Khoros is ported to a machine without this
*		 function then a replacement will be written.
*
*         Input: x - double argument to x * 2**n
*		 n - integer argument to x * 2**n
*
*        Output: 
*
*	Returns: the double precision value x * 2**n
*               
*  Restrictions:
*    Written By: Jeremy Worley
*          Date: Mar 31, 1993 10:38
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: double kldexp(
*		    double x,
*		    int n)
*
*************************************************************/

#define kldexp ldexp

/************************************************************
*
*  Routine Name: kmodf - compute the fractional component of the argument.
*
*       Purpose: The kmodf function computes the fractional
*		 and integral part of the double precision input argument x.
*		 After decomposing the input argument it returns the 
*		 fractional part and places
*		 the integral part in the argument i.
*
*		 This is provided in all POSIX and X/OPEN 
*		 compliant C math libraries.  Currently
*		 kmodf is simply a macro to modf because
*		 Khoros does not run on any machine whose
*		 math library does not include this function.
*		 If Khoros is ported to a machine without this
*		 function then a replacement will be written.
*
*         Input: x - input argument to separate into
*		     integral and fractional components.
*
*        Output: i - the integral part of the input argument 
*
*	Returns: the fractional component of the input argument.
*               
*  Restrictions:
*    Written By: Jeremy Worley
*          Date: Mar 31, 1993 10:38
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: double kmodf(
*		    double x,
*		    double *i)
*
*************************************************************/

#define kmodf  modf


/*
 * non posix functions
 */

/************************************************************
*
*  Routine Name: khypot - compute the euclidean distance from the origin of the arguments.
*
*       Purpose: The khypot function computes the Euclidean
*		 distance from (0,0) to (x,y).  This function 
*		 computes the square root of the sum of squares of the double
*		 precision input arguments x and y, giving the return value
*`		 sqrt(x*x + y*y)
*
*		 khypot is not a POSIX function, but is
*		 implemented in BSD math libraries.
*		 khypot is simply a macro to hypot because
*		 Khoros does not run on any machine whose
*		 math library does not include this function.
*		 If Khoros is ported to a machine without this
*		 function then a replacement will be written.
*
*         Input: x - value on x axis of vector to determine 
*		     length of.
*		 y - value on y axis of vector to determine 
*		     length of.
*
*        Output: 
*
*	Returns: returns the double precision euclidean distance from (0,0) to
*		 (x,y).
*               
*  Restrictions:
*    Written By: Jeremy Worley
*          Date: Mar 31, 1993 10:38
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: double khypot(
*		    double x,
*		    double y)
*
*************************************************************/

#define khypot hypot
double  hypot PROTO((double, double));

/************************************************************
*
*  Routine Name: ktrunc - truncate a number
*
*       Purpose: The ktrunc function truncates a double
*		 precision value by removing its fractional
*		 component.
*
*		 ktrunc is not a POSIX function, but is
*		 implemented in a few BSD math libraries.
*		 If it is implemented for the current architecture
*		 in libm.a, then this is simply a macro to trunc, 
*		 otherwise ktrunc is a macro to the Khoros
*		 portable version of trunc.
*
*         Input: x - value to truncate
*
*        Output: 
*
*	Returns: returns the integral component of the iput argument
*               
*  Restrictions:
*    Written By: Jeremy Worley
*          Date: Mar 31, 1993 10:38
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: double ktrunc(double x)
*
*************************************************************/

#if defined(KTRUNC_DEF)
#  define ktrunc  __ktrunc
#else
#  define ktrunc  trunc
   double  trunc PROTO((double));
#endif

/************************************************************
*
*  Routine Name: kgamma - compute the gamma function of the argument.
*
*       Purpose: The kgamma function computes the log gamma of the
*		 double precision input argument.
*
*         Input:  x - value to compute the log gamma of.
*        Output: 
*	Returns:  the double precision log gamma of the input argument.
*               
*  Restrictions: If gamma does not exist on you system in libm,
*		 then a Khoros version of the program will be 
*		 transparently installed in its place.  This
*		 version of the function is not very good.  The
*		 Khoros Group has plans to improve this function
*		 soon after the main release of Khoros 2.0.
*    Written By: Jeremy Worley
*          Date: Mar 31, 1993 10:38
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: double kgamma(double x)
*
*************************************************************/

#if defined(KGAMMA_DEF)
#  define kgamma  __kgamma
#else
#  define kgamma  gamma
   double  gamma PROTO((double));
#endif

/************************************************************
*
*  Routine Name: kcbrt - compute the cube root of the argument.
*
*       Purpose: The kcbrt function computes the cube root of the
*		 double precision input argument.
*
*         Input: x - value perform the cube root function on
*        Output: 
*	Returns: the double precision cube root of the input argument.
*               
*  Restrictions:
*    Written By: Jeremy Worley
*          Date: Mar 31, 1993 10:38
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: double kcbrt(double x)
*
*************************************************************/

#if defined(KCBRT_DEF)
#  define kcbrt  __kcbrt
#else
#  define kcbrt  cbrt
   double  cbrt PROTO((double));
#endif


/************************************************************
*
*  Routine Name: kj0 - compute the Bessel function j0 of the argument.
*
*       Purpose: The kj0 function computes an approximation
*		 of the j0 Bessel function.  This function evaluates
*		 the Bessel function of the first kind of integer
*		 order at an input argument x.
*
*		 kj0 is not a POSIX function, but is
*		 implemented in a few BSD math libraries.
*		 If it is implemented for the current architecture
*		 in libm.a, then this is simply a macro to j0, 
*		 otherwise kj0 is a macro to the Khoros
*		 portable version of j0.
*
*         Input: x - argument to j0
*
*        Output: approximation of the j0 Bessal function for x
*
*	Returns: the double precision value of the j0 Bessel function at x.
*               
*  Restrictions:
*    Written By: Jeremy Worley
*          Date: Mar 31, 1993 10:38
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: double kj0(double x)
*
*************************************************************/

#define kj0 j0
double  j0 PROTO((double));

/************************************************************
*
*  Routine Name: kj1 - compute the Bessel function j1 of the argument.
*
*       Purpose: The kj1 function computes an approximation
*		 of the j1 Bessel function.  This function evaluates
*		 the Bessel function of the first kind integer order
*		 of the input argument x.
*
*		 kj1 is not a POSIX function, but is
*		 implemented in a few BSD math libraries.
*		 If it is implemented for the current architecture
*		 in libm.a, then this is simply a macro to j1, 
*		 otherwise kj1 is a macro to the Khoros
*		 portable version of j1.
*
*         Input: x - argument to j1
*
*        Output: 
*
*	Returns: the double precision value of the j1 Bessel function at x.
*               
*  Restrictions:
*    Written By: Jeremy Worley
*          Date: Mar 31, 1993 10:38
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: double kj1(double x)
*
*************************************************************/

#define kj1 j1
double  j1 PROTO((double));

/************************************************************
*
*  Routine Name: kjn - compute the general Bessel function jn of the argument.
*
*       Purpose: The kjn function computes an approximation
*		 of the jn Bessel function.  This function evaluates
*		 the Bessel function of the first kind of integer n 
*		 order of the input argument x.
*
*		 kjn is not a POSIX function, but is
*		 implemented in a few BSD math libraries.
*		 If it is implemented for the current architecture
*		 in libm.a, then this is simply a macro to jn, 
*		 otherwise kjn is a macro to the Khoros
*		 portable version of jn.
*
*         Input: x - argument to jn
*		 n - integer order of the Bessel function.
*
*        Output: 
*
*	Returns: the double precision value of the n-order Bessel function
*		 at x.
*  Restrictions:
*    Written By: Jeremy Worley
*          Date: Mar 31, 1993 10:38
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: double kjn(int n, double x)
*
*************************************************************/

#define kjn jn
double  jn PROTO((int, double));

/************************************************************
*
*  Routine Name: ky0 - compute the Bessel function y0 of the argument.
*
*       Purpose: The ky0 function computes an approximation
*		 of the y0 Bessel function.  This function evaluates
*		 the Bessel function of the second kind of integer order
*		 at input argument x.  The input argument x must be positive.
*
*		 ky0 is not a POSIX function, but is
*		 implemented in a few BSD math libraries.
*		 If it is implemented for the current architecture
*		 in libm.a, then this is simply a macro to y0, 
*		 otherwise ky0 is a macro to the Khoros
*		 portable version of y0.
*
*         Input: x - argument to y0
*
*        Output: 
*
*	Returns: the double precision value of the y0 Bessel function at x.
*               
*  Restrictions:
*    Written By: Jeremy Worley
*          Date: Mar 31, 1993 10:38
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: double ky0(double x)
*
*************************************************************/

#define ky0 y0
double  y0 PROTO((double));

/************************************************************
*
*  Routine Name: ky1 - compute the Bessel function y1 of the argument.
*
*       Purpose: The ky1 function computes an approximation
*		 of the y1 Bessel function.  This function evaluates
*		 the Bessel function of the second kind of integer order
*	 	 at input argument x.  The input argument must be positive.
*
*		 ky1 is not a POSIX function, but is
*		 implemented in a few BSD math libraries.
*		 If it is implemented for the current architecture
*		 in libm.a, then this is simply a macro to y1, 
*		 otherwise ky1 is a macro to the Khoros
*		 portable version of y1.
*
*         Input: x - argument to y1
*
*        Output: 
*
*	Returns: the double precision value of the y1 Bessel function at x.
*               
*  Restrictions:
*    Written By: Jeremy Worley
*          Date: Mar 31, 1993 10:38
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: double ky1(double x)
*
*************************************************************/

#define ky1 y1
double  y1 PROTO((double));

/************************************************************
*
*  Routine Name: kyn - compute the general Bessel function yn of the argument.
*
*       Purpose: The kyn function computes an approximation
*		 of the yn Bessel function.  This function evaluates the
*		 Bessel function of the second kind of integer order n at the 
*		 input argument x.  The input argument x must be positive.
*
*		 kyn is not a POSIX function, but is
*		 implemented in a few BSD math libraries.
*		 If it is implemented for the current architecture
*		 in libm.a, then this is simply a macro to yn, 
*		 otherwise kyn is a macro to the Khoros
*		 portable version of yn.
*
*         Input: x - argument to yn
*		 n - integer order of the Bessel function
*
*        Output: 
*
*	Returns: the double precision value of the n-order Bessel function at
*		 x.
*               
*  Restrictions:
*    Written By: Jeremy Worley
*          Date: Mar 31, 1993 10:38
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: double kyn(int n, double x)
*
*************************************************************/

#define kyn yn
double  yn PROTO((int, double));

/************************************************************
*
*  Routine Name: kerf - compute the error function of the argument.
*
*       Purpose: The kerf function computes the error function,
*		 defined by:
*
*		! kerf(x) = 2.0/sqrt(pi)*{integral from 0 to x of exp(-t*t) dt}
*
*		 kerf is not a POSIX function, but is
*		 implemented in a few BSD math libraries.
*		 If it is implemented for the current architecture
*		 in libm.a, then this is simply a macro to erf, 
*		 otherwise kerf is a macro to the Khoros
*		 portable version of erf.
*
*         Input: x - argument to erf
*        Output: 
*	Returns: the error function of the input argument.
*               
*  Restrictions:
*    Written By: Jeremy Worley
*          Date: Mar 31, 1993 10:38
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: double kerf(double x)
*
*************************************************************/

#define kerf erf
double  erf PROTO((double));

/************************************************************
*
*  Routine Name: kerfc - compute the complement error function of the argument.
*
*       Purpose: The kerfc function computes the complement 
*		 error function, defined by:
*
*		! kerfc(x) = 1.0 - kerf(x)
*
*		 kerfc is not a POSIX function, but is
*		 implemented in a few BSD math libraries.
*		 If it is implemented for the current architecture
*		 in libm.a, then this is simply a macro to erfc, 
*		 otherwise kerfc is a macro to the Khoros
*		 portable version of erfc.
*
*         Input: x - argument to kerfc
*        Output: 
*	Returns: One minus the error function of the input argument.
*               
*  Restrictions:
*    Written By: Jeremy Worley
*          Date: Mar 31, 1993 10:38
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: double kerfc(double x)
*
*************************************************************/

#define kerfc erfc
double  erfc PROTO((double));

/************************************************************
*
*  Routine Name: kasinh - compute the arc hyperbolic sine of the argument.
*
*       Purpose: The kasinh function computes the arc hyperbolic
*		 sine function
*
*		 kasinh is not a POSIX function, but is
*		 implemented in a few BSD math libraries.
*		 If it is implemented for the current architecture
*		 in libm.a, then this is simply a macro to asinh, 
*		 otherwise kasinh is a macro to the Khoros
*		 portable version of asinh.
*
*         Input: x - argument to kasinh
*        Output: 
*	Returns: the evaluation of asinh(x)
*               
*  Restrictions:
*    Written By: Jeremy Worley
*          Date: Mar 31, 1993 10:38
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: double kasinh(double x)
*
*************************************************************/

#if defined(KASINH_DEF)
#  define kasinh __kasinh
#else
#  define kasinh asinh
   double  asinh PROTO((double));
#endif

/************************************************************
*
*  Routine Name: kacosh - compute the arc hyperbolic cosine of the argument.
*
*       Purpose: The kacosh function computes the arc hyperbolic
*		 cosine of the double precision argument.
*
*		 kacosh is not a POSIX function, but is
*		 implemented in a few BSD math libraries.
*		 If it is implemented for the current architecture
*		 in libm.a, then this is simply a macro to acosh, 
*		 otherwise kasinh is a macro to the Khoros
*		 portable version of acosh.
*
*         Input: x - argument to kacosh
*
*        Output: 
*
*	Returns: the double precision are hyperbolic cosine of the
*		 input argument or KMAXFLOAT if the input argument is
*		 less than one.
*               
*  Restrictions:
*    Written By: Jeremy Worley
*          Date: Mar 31, 1993 10:38
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: double kacosh(double x)
*
*************************************************************/

#if defined(KACOSH_DEF)
#  define kacosh __kacosh
#else
#  define kacosh acosh
   double  acosh PROTO((double));
#endif

/************************************************************
*
*  Routine Name: katanh - compute the arc hyperbolic tangent of the argument.
*
*       Purpose: The kasinh function computes the arc hyperbolic
*		 tangent of the double precision input argument whose
*		 value lies between -1 and 1.
*
*		 katanh is not a POSIX function, but is
*		 implemented in a few BSD math libraries.
*		 If it is implemented for the current architecture
*		 in libm.a, then this is simply a macro to atanh, 
*		 otherwise katanh is a macro to the Khoros
*		 portable version of atanh.
*
*         Input: x - argument to katanh
*
*        Output: 
*
*	Returns: the double precision arc hyperbolic tangent of the input
*		 argument or KMAXFLOAT if the input parameter is less 
*		 than -1 or greater than 1.
*               
*  Restrictions:
*    Written By: Jeremy Worley
*          Date: Mar 31, 1993 10:38
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: double katanh(double x)
*
*************************************************************/

#if defined(KATANH_DEF)
#  define katanh __katanh
#else
#  define katanh atanh
   double  atanh PROTO((double));
#endif


/************************************************************
*
*  Routine Name: kexpm1 - compute the exponential function minus 1 
*		 of the argument.
*
*       Purpose: The kexpm1 function computes the exponential
*		 function of the double precision input argument,
*		 then subtracts 1.
*
*		 kexpm1 is not a POSIX function, but is
*		 implemented in a few BSD math libraries.
*		 If it is implemented for the current architecture
*		 in libm.a, then this is simply a macro to expm1, 
*		 otherwise kexpm1 is a macro to the Khoros
*		 portable version of expm1.
*
*         Input: x - variable whose exponential minus 1 is to be computed.
*
*        Output: 
*
*	Returns: the double precision exponential minus 1 of the input
*		 argument.
*               
*  Restrictions:
*    Written By: Jeremy Worley
*          Date: Mar 31, 1993 10:38
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: double kexpm1(double x)
*
*************************************************************/

#if defined(KEXPM1_DEF)
#  define kexpm1 __kexpm1
#else
#  define kexpm1 expm1
   double  expm1 PROTO((double));
#endif


/************************************************************
*
*  Routine Name: kexp2 - compute the base 2 exponential function of the argument.
*
*       Purpose: The kexp2 function computes the base 2 exponential
*		 of the double precision input argument.
*
*		 kexp2 is not a POSIX function, but is
*		 implemented in a few BSD math libraries.
*		 If it is implemented for the current architecture
*		 in libm.a, then this is simply a macro to exp2, 
*		 otherwise kexp2 is a macro to the Khoros
*		 portable version of exp2.
*
*         Input: x - variable whose base 2 exponential is to be computed.
*
*        Output: 
*
*	Returns: the double precision base 2 exponential of the input argument.
*               
*  Restrictions:
*    Written By: Jeremy Worley
*          Date: Mar 31, 1993 10:38
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: double kexp2(double x)
*
*************************************************************/

#if defined(KEXP2_DEF)
#  define kexp2 __kexp2
#else
#  define kexp2 exp2
   double  exp2 PROTO((double));
#endif


/************************************************************
*
*  Routine Name: kexp10 - compute the base 10 exponential function of the argument.
*
*       Purpose: The kexp10 function computes the base 10 exponential
*		 function.
*
*		 kexp10 is not a POSIX function, but is
*		 implemented in a few BSD math libraries.
*		 If it is implemented for the current architecture
*		 in libm.a, then this is simply a macro to exp10, 
*		 otherwise kexp10 is a macro to the Khoros
*		 portable version of exp10.
*
*         Input: x - variable whose base 10 exponential is to be computed.
*
*        Output: 
*
*	Returns: the double precision base 10 exponential of the input 
*		 argument.
*               
*  Restrictions:
*    Written By: Jeremy Worley
*          Date: Mar 31, 1993 10:38
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: double kexp10(double x)
*
*************************************************************/

#if defined(KEXP10_DEF)
#  define kexp10 __kexp10
#else
#  define kexp10 exp10
   double  exp10 PROTO((double));
#endif


/************************************************************
*
*  Routine Name: klog1p - compute the logarithm of x+1 of the argument.
*
*       Purpose: The klog1p function computes the natural
*		 logarithm of the double precision argument plus 1.
*
*		 klog1p is not a POSIX function, but is
*		 implemented in a few BSD math libraries.
*		 If it is implemented for the current architecture
*		 in libm.a, then this is simply a macro to log1p, 
*		 otherwise klog1p is a macro to the Khoros
*		 portable version of log1p.
*
*         Input: x - argument to klog1p
*
*        Output: 
*
*	Returns: the double precision logarithm plus one of the input
*		 argument.
*               
*  Restrictions:
*    Written By: Jeremy Worley
*          Date: Mar 31, 1993 10:38
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: double klog1p(double x)
*
*************************************************************/

#if defined(KLOG1P_DEF)
#  define klog1p __klog1p
#else
#  define klog1p log1p
   double  log1p PROTO((double));
#endif

/************************************************************
*
*  Routine Name: klog2 - compute the base 2 logarithm of the argument.
*
*       Purpose: The klog2 function computes the base 2
*		 logarithm of the double precision input argument.
*
*		 klog2 is not a POSIX function, but is
*		 implemented in a few BSD math libraries.
*		 If it is implemented for the current architecture
*		 in libm.a, then this is simply a macro to log2, 
*		 otherwise klog2 is a macro to the Khoros
*		 portable version of log2.
*
*         Input: x - argument to klog2
*
*        Output: 
*
*	Returns: the double precision base 2 logarithm of the double
*		 precision input argument.
*               
*  Restrictions:
*    Written By: Jeremy Worley
*          Date: Mar 31, 1993 10:38
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: double klog2(double x)
*
*************************************************************/

#if defined(KLOG2_DEF)
#  define klog2 __klog2
#else
#  define klog2 log2
   double  log2 PROTO((double));
#endif

/************************************************************
*
*  Routine Name: krandom - generate random number in range [0,2**31-1]
*
*       Purpose: The krandom function generates a pseudo random
*		 number with a value in the range [0,2**31-1].
*		 The starting point of the pseudo random number sequence
*		 is set by calling ksrandom.
*
*		 krandom is not a POSIX function, but is
*		 implemented in most BSD libraries.
*		 If it is implemented for the current architecture
*		 in libc.a, then this is simply a macro to random, 
*		 otherwise klog2 is a macro to the Khoros
*		 portable version of random.
*
*         Input: 
*
*        Output: 
*
*       Returns: the pseudo random number generated.
*
*  Restrictions: 
*    Written By: Jeremy Worley
*          Date: Jul 11, 1993 13:49
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: long krandom(void)
*
*************************************************************/

#if defined(KRANDOM_DEF)
#  define krandom __krandom
#else
#  define krandom random
#  if KOPSYS_LOCAL != KOPSYS_OSF
     long random PROTO((void));
#  endif
#endif


/************************************************************
*
*  Routine Name: ksrandom - seed the krandom random number generator.
*
*       Purpose: The ksrandom function sets the seed (starting value)
*		 of the random number generation algorithm used by
*		 krandom.
*
*		 ksrandom is not a POSIX function, but is
*		 implemented in most BSD libraries.
*		 If it is implemented for the current architecture
*		 in libc.a, then this is simply a macro to srandom, 
*		 otherwise klog2 is a macro to the Khoros
*		 portable version of srandom.
*
*         Input: 
*
*        Output: 
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Jeremy Worley
*          Date: Jul 11, 1993 13:49
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: void ksrandom(int seed)
*
*************************************************************/

#if defined(KSRANDOM_DEF)
#  define ksrandom __ksrandom
#else
#  define ksrandom srandom
#  if KOPSYS_LOCAL != KOPSYS_OSF
     void srandom PROTO((int));
#  endif
#endif

	/*-------------------------------------*
	|       routine definitions
	---------------------------------------*/

/* libm.c */
double __kfmod   PROTO((double, double));
double __kasinh  PROTO((double));
double __kacosh  PROTO((double));
double __katanh  PROTO((double));
double __kexpm1  PROTO((double));
double __klog1p  PROTO((double));
double __kexp2   PROTO((double));
double __kexp10  PROTO((double));
double __klog2   PROTO((double));
double __kcbrt   PROTO((double));
double __kgamma  PROTO((double));
double __ktrunc  PROTO((double));

long __krandom   PROTO((void));
void __ksrandom  PROTO((int));

/* kcomplex.c */
kdcomplex kdcadd	PROTO((kdcomplex, kdcomplex));
kdcomplex kdcsub	PROTO((kdcomplex, kdcomplex));
kdcomplex kdcmult	PROTO((kdcomplex, kdcomplex));
kdcomplex kdcdiv	PROTO((kdcomplex, kdcomplex));
kdcomplex kdcsqrt	PROTO((kdcomplex));
kdcomplex kdcexp	PROTO((kdcomplex));
kdcomplex kdclog	PROTO((kdcomplex));
kdcomplex kdcpow	PROTO((kdcomplex, kdcomplex));
kdcomplex kdccomp	PROTO((double, double));
kdcomplex kdcconj	PROTO((kdcomplex));
kdcomplex kcomp2dcomp	PROTO((kcomplex));
kdcomplex kdcr2p	PROTO((kdcomplex));
kdcomplex kdcp2r	PROTO((kdcomplex));
kcomplex  kcadd		PROTO((kcomplex, kcomplex));
kcomplex  kcsub		PROTO((kcomplex, kcomplex));
kcomplex  kcmult	PROTO((kcomplex, kcomplex));
kcomplex  kcdiv		PROTO((kcomplex, kcomplex));
kcomplex  kcsqrt	PROTO((kcomplex));
kcomplex  kcexp		PROTO((kcomplex));
kcomplex  kclog		PROTO((kcomplex));
kcomplex  kcpow		PROTO((kcomplex, kcomplex));
kcomplex  kccomp	PROTO((double, double));
kcomplex  kcconj	PROTO((kcomplex));
kcomplex  kdcomp2comp	PROTO((kdcomplex));
kcomplex  kcr2p		PROTO((kcomplex));
kcomplex  kcp2r		PROTO((kcomplex));
double	  kdcmagsq	PROTO((kdcomplex));
double	  kdcmag	PROTO((kdcomplex));
double	  kdclogmagp1	PROTO((kdcomplex));
double	  kdclogmagsqp1	PROTO((kdcomplex));
double	  kdclogmagsq	PROTO((kdcomplex));
double	  kdclogmag	PROTO((kdcomplex));
double	  kdcang	PROTO((kdcomplex));
double	  kdcreal	PROTO((kdcomplex));
double	  kdcimag	PROTO((kdcomplex));
float	  kcmagsq	PROTO((kcomplex));
float	  kcmag		PROTO((kcomplex));
float	  kclogmagp1	PROTO((kcomplex));
float	  kclogmagsqp1	PROTO((kcomplex));
float	  kclogmagsq	PROTO((kcomplex));
float	  kclogmag	PROTO((kcomplex));
float	  kcang		PROTO((kcomplex));
float	  kcreal	PROTO((kcomplex));
float	  kcimag	PROTO((kcomplex));
int	  kdcomplex_to_arrays PROTO((kdcomplex *, int, double **, double **));

/* extensions.c */
int    kpowtwo	 PROTO((int));
double kfact	 PROTO((double));
double kimpulse  PROTO((double));
double kstep	 PROTO((double));
double ksign	 PROTO((double));
double ksinc	 PROTO((double));
double krecip	 PROTO((double));
double kneg	 PROTO((double));
double kfraction PROTO((double));
unsigned long   kset	 PROTO((unsigned long));
unsigned long   kclear	 PROTO((unsigned long));
unsigned long   knot	 PROTO((unsigned long));
long   klogical_not PROTO((long));
double klogn     PROTO((double, double));

/* flin.c */
void kblas_sscal PROTO((float *, float, int, int));
void kblas_saxpy PROTO((float *, float *, float, int, int, int));
void kblas_sswap PROTO((float *, float *, int, int, int));
int  klin_sgefa PROTO((float *, int, int, int *));
int  klin_sgedi PROTO((float *, int, int, int *, int, float *, float *));

/* dlin.c */
void kblas_dscal PROTO((double *, double, int, int));
void kblas_daxpy PROTO((double *, double *, double, int, int, int));
void kblas_dswap PROTO((double *, double *, int, int, int));
int  klin_dgefa PROTO((double *, int, int, int *));
int  klin_dgedi PROTO((double *, int, int, int *, int, double *, double *));
   
/* fmatrix.c */
float  kfmatrix_inner_prod  PROTO((float *, float *, int));
int    kfmatrix_vector_prod PROTO((float *, float *, int, int, float *));
int    kfmatrix_princ_axis  PROTO((float *, int, float *));
int    kfmatrix_multiply    PROTO((float *, int, int, 
				   float *, int, int, float *));
int    kfmatrix_clear	    PROTO((int, int, float *));
int    kfmatrix_identity    PROTO((int, int, float *));
int    kfmatrix_inverse	    PROTO((float *, int, float *));

/* dmatrix.c */
double kdmatrix_inner_prod  PROTO((double *, double *, int));
int    kdmatrix_vector_prod PROTO((double *, double *, int, int, double *));
int    kdmatrix_princ_axis  PROTO((double *, int, double *));
int    kdmatrix_multiply    PROTO((double *, int, int, 
				   double *, int, int, double *));
int    kdmatrix_clear	    PROTO((int, int, double *));
int    kdmatrix_identity    PROTO((int, int, double *));
int    kdmatrix_inverse	    PROTO((double *, int, double *));

/* noise.c */
double kurng PROTO((void));
int    kgen_gauss    PROTO((int, double, double, double *));
int    kgen_expon    PROTO((int, double, double *));
int    kgen_unif     PROTO((int, double, double, double *));
int    kgen_rayleigh PROTO((int, double, double *));
int    kgen_poisson  PROTO((int, double, double, double *));

/* signal.c */
int kgen_sine	PROTO((int, double, double, double, double, double *));
int kgen_linear PROTO((int, double, double, double, double, double, double, double, double *));
int kgen_sinec	PROTO((int, double, double, double, int, double *));

#endif /* _klibm_klibm_h_ */
/* Don't add after this point */
