/*
 * Khoros: $Id$
 */

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>       Purpose: Generalized interface to the
   >>>		       kexpr_evaluate_generic() function
   >>>>
   >>>>    Written By: Mark Young
   >>>>
   >>>>          Date: Jan 23, 1993
   >>>>
   >>>> Modifications:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#ifndef _kexpr_kevaluate_h_
#define _kexpr_kevaluate_h_


	/*-------------------------------------*
	|       #include
	---------------------------------------*/


	/*-------------------------------------*
	|       #defines
	---------------------------------------*/

/************************************************************
*
*  Routine Name: kexpr_evaluate_byte - evaluate char/byte expression
*
*       Purpose: This routine performs two functions: it sets
*		 character or byte variables and evaluates character or
*		 byte expressions.
*		 It will return TRUE (1) on success, FALSE (0)
*		 on failure.  If the routine fails for some reason,
*		 an error message will be passed back in the error
*		 string.  If the error string is NULL then the
*		 error is output using the kerror facility.
*
*		 The string passed in will indicate which function
*		 (variable or expression)
*		 kexpr_evaluate_byte() is to perform, and the byte
*		 value returned will reflect this.  Let us illustrate
*		 with an example.  Suppose a string of "i = 10" is passed
*		 to kexpr_evaluate_byte().  This indicates that the variable
*		 i is to be defined and assigned the value of 10; the
*		 value returned will be 10.  Later, suppose
*		 kexpr_evaluate_byte()
*		 is called again with the same id, with a string of
*		 "i*2+5".  Now, kexpr_evaluate_byte() will evaluate the
*		 expression, using the value of i defined by the previous
*		 call - [10*2+5].  The value returned in this case would be 25.
*
*         Input: id     - the variable identifier.
*                string - the string to be evaluated.
*
*        Output: value  - if no error occurred then the byte
*                         value of the expression is stored and
*			  TRUE returned.
*                error  - if an error occurred the error message is stored in
*                         the error string array and False returned.  The
*                         error string array must be at least a 1024 string
*                         array that is allocated by the calling routine.  If
*                         the error string array has not been allocated by the
*                         calling routine (error is passed in as NULL) then the
*                         error message is output with the kerror facility.
*
*	Returns: TRUE (1) on success, FALSE (0) otherwise
*  Restrictions:
*    Written By: Mark Young
*          Date: Jan 23, 1993
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kexpr_evaluate_byte(
*		 !   long id,
*		 !   char *string,
*		 !   char *value,
*		 !   char *error)
*
*************************************************************/

#define kexpr_evaluate_byte(id, string, value, error) \
		kexpr_evaluate_generic(id, KBYTE, string, (kaddr) value, error)


/************************************************************
*
*  Routine Name: kexpr_evaluate_ubyte - evaluate unsigned byte/char expression
*
*       Purpose: This routine performs two functions: it sets
*		 unsigned byte variables and evaluates unsigned byte
*		 expressions.
*		 It will return TRUE (1) on success, FALSE (0)
*		 on failure.  If the routine fails for some reason,
*		 an error message will be passed back in the error
*		 string.  If the error string is NULL then the
*		 error is output using the kerror facility.
*
*		 The string passed in will indicate which function
*		 (variable or expression)
*		 kexpr_evaluate_ubyte() is to perform, and the ubyte
*		 value returned will reflect this.  Let us illustrate
*		 with an example.  Suppose a string of "i = 10" is passed
*		 to kexpr_evaluate_ubyte().  This indicates that the variable
*		 i is to be defined and assigned the value of 10; the
*		 value returned will be 10.  Later, suppose
*		 kexpr_evaluate_ubyte()
*		 is called again with the same id, with a string of
*		 "i*2+5".  Now, kexpr_evaluate_ubyte() will evaluate the
*		 expression, using the value of i defined by the previous
*		 call - [10*2+5].  The value returned in this case would be 25.
*
*         Input: id     - the variable identifier.
*                string - the string to be evaluated.
*
*        Output: value  - if no error occurred then the unsigned byte/char
*                         value of the expression is stored and
*			  TRUE returned.
*                error  - if an error occurred the error message is stored in
*                         the error string array and False returned.  The
*                         error string array must be at least a 1024 string
*                         array that is allocated by the calling routine.  If
*                         the error string array has not been allocated by the
*                         calling routine (error is passed in as NULL) then the
*                         error message is output with the kerror facility.
*
*	Returns: TRUE (1) on success, FALSE (0) otherwise
*  Restrictions:
*    Written By: Mark Young
*          Date: Jan 23, 1993
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kexpr_evaluate_ubyte(
*		 !   long     id,
*		 !   char     *string,
*		 !   unsigned char *value,
*		 !   char     *error)
*
*************************************************************/

#define kexpr_evaluate_ubyte(id, string, value, error) \
		kexpr_evaluate_generic(id, string, KUBYTE, (kaddr) value, error)


/************************************************************
*
*  Routine Name: kexpr_evaluate_short - evaluate short expression
*
*       Purpose: This routine performs two functions: it sets
*		 short variables and evaluates short expressions.
*		 It will return TRUE (1) on success, FALSE (0)
*		 on failure.  If the routine fails for some reason,
*		 an error message will be passed back in the error
*		 string.  If the error string is NULL then the
*		 error is output using the kerror facility.
*
*		 The string passed in will indicate which function
*		 (variable or expression)
*		 kexpr_evaluate_short() is to perform, and the short
*		 value returned will reflect this.  Let us illustrate
*		 with an example.  Suppose a string of "i = 10" is passed
*		 to kexpr_evaluate_short().  This indicates that the variable
*		 i is to be defined and assigned the value of 10; the
*		 value returned will be 10.  Later, suppose
*		 kexpr_evaluate_short()
*		 is called again with the same id, with a string of
*		 "i*2+5".  Now, kexpr_evaluate_short() will evaluate the
*		 expression, using the value of i defined by the previous
*		 call - [10*2+5].  The value returned in this case would be 25.
*
*         Input: id     - the variable identifier.
*                string - the string to be evaluated.
*
*        Output: value  - if no error occurred then the short
*                         value of the expression is stored and
*			  TRUE returned.
*                error  - if an error occurred the error message is stored in
*                         the error string array and False returned.  The
*                         error string array must be at least a 1024 string
*                         array that is allocated by the calling routine.  If
*                         the error string array has not been allocated by the
*                         calling routine (error is passed in as NULL) then the
*                         error message is output with the kerror facility.
*
*	Returns: TRUE (1) on success, FALSE (0) otherwise
*    Written By: Mark Young
*          Date: Jan 23, 1993
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kexpr_evaluate_short(
*		 !   long  id,
*		 !   char  *string,
*		 !   short *value,
*		 !   char  *error)
*
*************************************************************/

#define kexpr_evaluate_short(id, string, value, error) \
		kexpr_evaluate_generic(id, string, KSHORT, (kaddr) value, error)


/************************************************************
*
*  Routine Name: kexpr_evaluate_ushort - evaluate unsigned short expression
*
*       Purpose: This routine performs two functions: it sets
*		 unsigned short variables and evaluates unsigned short
*		 expressions.
*		 It will return TRUE (1) on success, FALSE (0)
*		 on failure.  If the routine fails for some reason,
*		 an error message will be passed back in the error
*		 string.  If the error string is NULL then the
*		 error is output using the kerror facility.
*
*		 The string passed in will indicate which function
*		 (variable or expression)
*		 kexpr_evaluate_ushort() is to perform, and the unsigned short
*		 value returned will reflect this.  Let us illustrate
*		 with an example.  Suppose a string of "i = 10" is passed
*		 to kexpr_evaluate_ushort().  This indicates that the variable
*		 i is to be defined and assigned the value of 10; the
*		 value returned will be 10.  Later, suppose
*		 kexpr_evaluate_ushort()
*		 is called again with the same id, with a string of
*		 "i*2+5".  Now, kexpr_evaluate_ushort() will evaluate the
*		 expression, using the value of i defined by the previous
*		 call - [10*2+5].  The value returned in this case would be 25.
*
*         Input: id     - the variable identifier.
*                string - the string to be evaluated.
*
*        Output: value  - if no error occurred then the unsigned short
*                         value of the expression is stored and
*			  TRUE returned.
*                error  - if an error occurred the error message is stored in
*                         the error string array and False returned.  The
*                         error string array must be at least a 1024 string
*                         array that is allocated by the calling routine.  If
*                         the error string array has not been allocated by the
*                         calling routine (error is passed in as NULL) then the
*                         error message is output with the kerror facility.
*
*	Returns: TRUE (1) on success, FALSE (0) otherwise
*  Restrictions:
*    Written By: Mark Young
*          Date: Jan 23, 1993
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kexpr_evaluate_ushort(
*		 !   long     id,
*		 !   char     *string,
*		 !   unsigned short *value,
*		 !   char     *error)
*
*************************************************************/

#define kexpr_evaluate_ushort(id, string, value, error) \
	       kexpr_evaluate_generic(id, string, KUSHORT, (kaddr) value, error)


/************************************************************
*
*  Routine Name: kexpr_evaluate_int - evaluate integer expression
*
*       Purpose: This routine performs two functions: it sets
*		 integer variables and evaluates integer expressions.
*		 It will return TRUE (1) on success, FALSE (0)
*		 on failure.  If the routine fails for some reason,
*		 an error message will be passed back in the error
*		 string.  If the error string is NULL then the
*		 error is output using the kerror facility.
*
*		 The string passed in will indicate which function
*		 (variable or expression)
*		 kexpr_evaluate_int() is to perform, and the integer
*		 value returned will reflect this.  Let us illustrate
*		 with an example.  Suppose a string of "i = 10" is passed
*		 to kexpr_evaluate_int().  This indicates that the variable
*		 i is to be defined and assigned the value of 10; the
*		 value returned will be 10.  Later, suppose kexpr_evaluate_int()
*		 is called again with the same id, with a string of
*		 "i*2+5".  Now, kexpr_evaluate_int() will evaluate the
*		 expression, using the value of i defined by the previous
*		 call - [10*2+5].  The value returned in this case would be 25.
*
*         Input: id     - the variable identifier.
*                string - the string to be evaluated.
*
*        Output: value  - if no error occurred then the integer
*                         value of the expression is stored and
*			  TRUE returned.
*                error  - if an error occurred the error message is stored in
*                         the error string array and False returned.  The
*                         error string array must be at least a 1024 string
*                         array that is allocated by the calling routine.  If
*                         the error string array has not been allocated by the
*                         calling routine (error is passed in as NULL) then the
*                         error message is output with the kerror facility.
*
*	Returns: TRUE (1) on success, FALSE (0) otherwise
*  Restrictions:
*    Written By: Mark Young
*          Date: Jan 23, 1993
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kexpr_evaluate_int(
*		 !   long id,
*		 !   char *string,
*		 !   int  *value,
*		 !   char *error)
*
*************************************************************/

#define kexpr_evaluate_int(id, string, value, error) \
		kexpr_evaluate_generic(id, string, KINT, (kaddr) value, error)


/************************************************************
*
*  Routine Name: kexpr_evaluate_uint - evaluate unsigned integer expression
*
*       Purpose: This routine performs two functions: it sets
*		 unsigned integer variables and evaluates unsigned integer
*		 expressions.
*		 It will return TRUE (1) on success, FALSE (0)
*		 on failure.  If the routine fails for some reason,
*		 an error message will be passed back in the error
*		 string.  If the error string is NULL then the
*		 error is output using the kerror facility.
*
*		 The string passed in will indicate which function
*		 (variable or expression)
*		 kexpr_evaluate_uint() is to perform, and the unsigned integer
*		 value returned will reflect this.  Let us illustrate
*		 with an example.  Suppose a string of "i = 10" is passed
*		 to kexpr_evaluate_uint().  This indicates that the variable
*		 i is to be defined and assigned the value of 10; the
*		 value returned will be 10.  Later, suppose
*		 kexpr_evaluate_uint() is called again with the same id,
*		 with a string of "i*2+5".  Now, kexpr_evaluate_uint()
*		 will evaluate the expression, using the value of i
*		 defined by the previous call - [10*2+5].  The value
*		 returned in this case would be 25.
*
*         Input: id     - the variable identifier.
*                string - the string to be evaluated.
*
*        Output: value  - if no error occurred then the unsigned integer
*                         value of the expression is stored and
*			  TRUE returned.
*                error  - if an error occurred the error message is stored in
*                         the error string array and False returned.  The
*                         error string array must be at least a 1024 string
*                         array that is allocated by the calling routine.  If
*                         the error string array has not been allocated by the
*                         calling routine (error is passed in as NULL) then the
*                         error message is output with the kerror facility.
*
*	Returns: TRUE (1) on success, FALSE (0) otherwise
*  Restrictions:
*    Written By: Mark Young
*          Date: Jan 23, 1993
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kexpr_evaluate_uint(
*		 !   long     id,
*		 !   char     *string,
*		 !   unsigned int *value,
*		 !   char     *error)
*
*************************************************************/

#define kexpr_evaluate_uint(id, string, value, error) \
		kexpr_evaluate_generic(id, string, KUINT, (kaddr) value, error)


/************************************************************
*
*  Routine Name: kexpr_evaluate_long - evaluate long expression
*
*       Purpose: This routine performs two functions: it sets
*		 long integer variables and evaluates long integer
*		 expressions.  It will return TRUE (1) on success, FALSE (0)
*		 on failure.  If the routine fails for some reason,
*		 an error message will be passed back in the error
*		 string.  If the error string is NULL then the
*		 error is output using the kerror facility.
*
*		 The string passed in will indicate which function
*		 (variable or expression)
*		 kexpr_evaluate_long() is to perform, and the long integer
*		 value returned will reflect this.  Let us illustrate
*		 with an example.  Suppose a string of "i = 10" is passed
*		 to kexpr_evaluate_long().  This indicates that the variable
*		 i is to be defined and assigned the value of 10; the
*		 value returned will be 10.  Later, suppose
*		 kexpr_evaluate_long()
*		 is called again with the same id, with a string of
*		 "i*2+5".  Now, kexpr_evaluate_long() will evaluate the
*		 expression, using the value of i defined by the previous
*		 call - [10*2+5].
*		 The value returned in this case would be 25.
*
*         Input: id     - the variable identifier.
*                string - the string to be evaluated.
*
*        Output: value  - if no error occurred then the long
*                         value of the expression is stored and
*			  TRUE returned.
*                error  - if an error occurred the error message is stored in
*                         the error string array and False returned.  The
*                         error string array must be at least a 1024 string
*                         array that is allocated by the calling routine.  If
*                         the error string array has not been allocated by the
*                         calling routine (error is passed in as NULL) then the
*                         error message is output with the kerror facility.
*
*	Returns: TRUE (1) on success, FALSE (0) otherwise
*  Restrictions:
*    Written By: Mark Young
*          Date: Jan 23, 1993
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kexpr_evaluate_long(
*		 !   long id,
*		 !   char *string,
*		 !   long *value,
*		 !   char *error)
*
*************************************************************/

#define kexpr_evaluate_long(id, string, value, error) \
		kexpr_evaluate_generic(id, string, KLONG, (kaddr) value, error)


/************************************************************
*
*  Routine Name: kexpr_evaluate_ulong - evaluate unsigned long expression
*
*       Purpose: This routine performs two functions: it sets
*		 unsigned long variables and evaluates unsigned long
*		 expressions.
*		 It will return TRUE (1) on success, FALSE (0)
*		 on failure.  If the routine fails for some reason,
*		 an error message will be passed back in the error
*		 string.  If the error string is NULL then the
*		 error is output using the kerror facility.
*
*		 The string passed in will indicate which function
*		 (variable or expression)
*		 kexpr_evaluate_ulong() is to perform, and the unsigned long
*		 value returned will reflect this.  Let us illustrate
*		 with an example.  Suppose a string of "i = 10" is passed
*		 to kexpr_evaluate_ulong().  This indicates that the variable
*		 i is to be defined and assigned the value of 10; the
*		 value returned will be 10.  Later, suppose
*		 kexpr_evaluate_ulong()
*		 is called again with the same id, with a string of
*		 "i*2+5".  Now, kexpr_evaluate_ulong() will evaluate the
*		 expression, using the value of i defined by the previous
*		 call - [10*2+5].  The value returned in this case would be 25.
*
*         Input: id     - the variable identifier.
*                string - the string to be evaluated.
*
*        Output: value  - if no error occurred then the unsigned long
*                         value of the expression is stored and
*			  TRUE returned.
*                error  - if an error occurred the error message is stored in
*                         the error string array and False returned.  The
*                         error string array must be at least a 1024 string
*                         array that is allocated by the calling routine.  If
*                         the error string array has not been allocated by the
*                         calling routine (error is passed in as NULL) then the
*                         error message is output with the kerror facility.
*
*	Returns: TRUE (1) on success, FALSE (0) otherwise
*  Restrictions:
*    Written By: Mark Young
*          Date: Jan 23, 1993
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kexpr_evaluate_ulong(
*		 !   long     id,
*		 !   char     *string,
*		 !   unsigned long *value,
*		 !   char     *error)
*
*************************************************************/

#define kexpr_evaluate_ulong(id, string, value, error) \
		kexpr_evaluate_generic(id, string, KULONG, (kaddr) value, error)


/************************************************************
*
*  Routine Name: kexpr_evaluate_float - evaluate float expression
*
*       Purpose: This routine performs two functions: it sets floating
*		 point variables and evaluates floating point expressions.
*		 It will return TRUE (1) on success, FALSE (0) on failure.
*		 If the routine fails for some reason, an error message will
*		 be passed back in the error string. If the error string
*		 is NULL then the error is output using the kerror facility.
*
*		 The string passed in will indicate which function
*		 (variable or expression)
*		 kexpr_evaluate_float() is to perform, and the float value
*		 returned will reflect this.  Let us illustrate with
*		 an example.  Suppose a string of "x = 0.1" is passed
*		 to kexpr_evaluate_float().  This indicates that the variable
*		 x is to be defined and assigned the value of 0.1; the
*		 value returned will be 0.1.  Then, kexpr_evaluate_float()
*		 is called again with a \fIstring\fP of "y = 0.9"; now the
*		 value returned is 0.9. Finally, suppose kexpr_evaluate_float()
*		 is with the with a string of "(x+y)/2".  Now,
*		 kexpr_evaluate_float() will evaluate the expression, using the
*		 values of x and y defined by the previous call:
*		 [(0.1 + 0.9) / 2].  The value returned in this case would be
*		 0.5.
*
*         Input: id     - the variable identifier.
*                string - the string to be evaluated.
*
*        Output: value  - if no error occurred then the float
*                         value of the expression is stored and
*			  TRUE returned.
*                error  - if an error occurred the error message is stored in
*                         the error string array and False returned.  The
*                         error string array must be at least a 1024 string
*                         array that is allocated by the calling routine.  If
*                         the error string array has not been allocated by the
*                         calling routine (error is passed in as NULL) then the
*                         error message is output with the kerror facility.
*
*	Returns: TRUE (1) on success, FALSE (0) otherwise
*  Restrictions:
*    Written By: Mark Young  
*          Date: Jan 23, 1993
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kexpr_evaluate_float(
*		 !   long  id,
*		 !   char  *string,
*		 !   float *value,
*		 !   char  *error)
*
*************************************************************/

#define kexpr_evaluate_float(id, string, value, error) \
		kexpr_evaluate_generic(id, string, KFLOAT, (kaddr) value, error)


/************************************************************
*
*  Routine Name: kexpr_evaluate_complex - evaluate complex expression
*
*       Purpose: This routine performs two functions: it sets complex
*		 variables and evaluates complex expressions.
*		 It will return TRUE (1) on success, FALSE (0) on failure.
*		 If the routine fails for some reason, an error message will
*		 be passed back in the error string. If the error string
*		 is NULL then the error is output using the kerror facility.
*
*		 The string passed in will indicate which function
*		 (variable or expression)
*		 kexpr_evaluate_complex() is to perform, and the complex value
*		 returned will reflect this.  Let us illustrate with
*		 an example.  Suppose a string of "x = (0,1)" is passed
*		 to kexpr_evaluate_complex().  This indicates that the variable
*		 x is to be defined and assigned the complex value of (0i,1j)
*		 the value returned will be in kcomplex structure as floating
*		 point pairs of real = 0.0, and imaginary = 1.0.  Then,
*		 kexpr_evaluate_complex() is called again with a \fIstring\fP of
*		 "y = (0,9)"; now the value returned is (0i,9j). Finally,
*		 suppose kexpr_evaluate_complex() is with the with a string of
*		 "(x+y)/2".  Now, kexpr_evaluate_complex() will evaluate the
*		 expression, using the values of x and y defined by the
*		 previous call: [(0.1 + 0.9) / 2].  The value returned in
*		 this case would be (0.5i,0.0j).
*
*         Input: id     - the variable identifier.
*                string - the string to be evaluated.
*
*        Output: value  - if no error occurred then the complex
*                         value of the expression is stored and
*			  TRUE returned.
*                error  - if an error occurred the error message is stored in
*                         the error string array and False returned.  The
*                         error string array must be at least a 1024 string
*                         array that is allocated by the calling routine.  If
*                         the error string array has not been allocated by the
*                         calling routine (error is passed in as NULL) then the
*                         error message is output with the kerror facility.
*
*	Returns: TRUE (1) on success, FALSE (0) otherwise
*  Restrictions:
*    Written By: Mark Young  
*          Date: Jan 23, 1993
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kexpr_evaluate_complex(
*		 !   long     id,
*		 !   char     *string,
*		 !   kcomplex *value,
*		 !   char     *error)
*
*************************************************************/

#define kexpr_evaluate_complex(id, string, value, error) \
	      kexpr_evaluate_generic(id, string, KCOMPLEX, (kaddr) value, error)


/************************************************************
*
*  Routine Name: kexpr_evaluate_double - evaluate double expression
*
*       Purpose: This routine performs two functions: it sets double
*		 variables and evaluates double expressions.
*		 It will return TRUE (1) on success, FALSE (0) on failure.
*		 If the routine fails for some reason, an error message will
*		 be passed back in the error string. If the error string
*		 is NULL then the error is output using the kerror facility.
*
*		 The string passed in will indicate which function
*		 (variable or expression)
*		 kexpr_evaluate_double() is to perform, and the double value
*		 returned will reflect this.  Let us illustrate with
*		 an example.  Suppose a string of "x = 0.1" is passed
*		 to kexpr_evaluate_double().  This indicates that the variable
*		 x is to be defined and assigned the value of 0.1; the
*		 value returned will be 0.1.  Then, kexpr_evaluate_double()
*		 is called again with a \fIstring\fP of "y = 0.9"; now the
*		 value returned is 0.9. Finally, suppose kexpr_evaluate_double()
*		 is with the with a string of "(x+y)/2".  Now,
*		 kexpr_evaluate_double() will evaluate the expression, using the
*		 values of x and y defined by the previous call:
*		 [(0.1 + 0.9) / 2].  The value returned in this case would be
*		 0.5.
*
*         Input: id     - the variable identifier.
*                string - the string to be evaluated.
*
*        Output: value  - if no error occurred then the double
*                         value of the expression is stored and
*			  TRUE returned.
*                error  - if an error occurred the error message is stored in
*                         the error string array and False returned.  The
*                         error string array must be at least a 1024 string
*                         array that is allocated by the calling routine.  If
*                         the error string array has not been allocated by the
*                         calling routine (error is passed in as NULL) then the
*                         error message is output with the kerror facility.
*
*	Returns: TRUE (1) on success, FALSE (0) otherwise
*  Restrictions:
*    Written By: Mark Young  
*          Date: Jan 23, 1993
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kexpr_evaluate_double(
*		 !   long   id,
*		 !   char   *string,
*		 !   double *value,
*		 !   char   *error)
*
*************************************************************/

#define kexpr_evaluate_double(id, string, value, error) \
	       kexpr_evaluate_generic(id, string, KDOUBLE, (kaddr) value, error)


/************************************************************
*
*  Routine Name: kexpr_evaluate_dcomplex - evaluate double complex expression
*
*       Purpose: This routine performs two functions: it sets double complex
*		 variables and evaluates double complex expressions.
*		 It will return TRUE (1) on success, FALSE (0) on failure.
*		 If the routine fails for some reason, an error message will
*		 be passed back in the error string. If the error string
*		 is NULL then the error is output using the kerror facility.
*
*		 The string passed in will indicate which function
*		 (variable or expression)
*		 kexpr_evaluate_dcomplex() is to perform, and the double complex
*		 value returned will reflect this.  Let us illustrate with
*		 an example.  Suppose a string of "x = (0,1)" is passed
*		 to kexpr_evaluate_dcomplex().  This indicates that the variable
*		 x is to be defined and assigned the double complex value of
*		 (0i,1j) the value returned will be in kcomplex structure as
*		 double pair of real = 0.0, and imaginary = 1.0.  Then,
*		 kexpr_evaluate_dcomplex() is called again with a \fIstring\fP of
*		 "y = (0,9)"; now the value returned is (0i,9j). Finally,
*		 suppose kexpr_evaluate_dcomplex() is with the with a string of
*		 "(x+y)/2".  Now, kexpr_evaluate_complex() will evaluate the
*		 expression, using the values of x and y defined by the
*		 previous call: [(0.1 + 0.9) / 2].  The value returned in
*		 this case would be (0.5i,0.0j).
*
*         Input: id     - the variable identifier.
*                string - the string to be evaluated.
*
*        Output: value  - if no error occurred then the double complex
*                         value of the expression is stored and
*			  TRUE returned.
*                error  - if an error occurred the error message is stored in
*                         the error string array and False returned.  The
*                         error string array must be at least a 1024 string
*                         array that is allocated by the calling routine.  If
*                         the error string array has not been allocated by the
*                         calling routine (error is passed in as NULL) then the
*                         error message is output with the kerror facility.
*
*	Returns: TRUE (1) on success, FALSE (0) otherwise
*  Restrictions:
*    Written By: Mark Young  
*          Date: Jan 23, 1993
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kexpr_evaluate_dcomplex(
*		 !   long      id,
*		 !   char      *string,
*		 !   kdcomplex *value,
*		 !   char      *error)
*
*************************************************************/

#define kexpr_evaluate_dcomplex(id, string, value, error) \
	     kexpr_evaluate_generic(id, string, KDCOMPLEX, (kaddr) value, error)


/************************************************************
*
*  Routine Name: kexpr_evaluate_string - evaluate string expression
*
*       Purpose: This routine evaluates the string and returns a string
*		 value of the expression.
*		 It will return TRUE (1) on success, FALSE (0) on failure.
*		 If the routine fails for some reason, an error message will
*		 be passed back in the error string. If the error string
*		 is NULL then the error is output using the kerror facility.
*
*		 The string passed in will indicate which function
*		 (variable or expression)
*		 kexpr_evaluate_string() is to perform, and the string value
*		 returned will reflect this.  Let us illustrate with
*		 an example.  Suppose a string of "x = 0.1" is passed
*		 to kexpr_evaluate_string().  This indicates that the variable
*		 x is to be defined and assigned the value of 0.1; the
*		 value returned will be "0.1".  Then, kexpr_evaluate_string()
*		 is called again with a \fIstring\fP of "y = 0.9"; now the
*		 value returned is "0.9". Finally, suppose
*		 kexpr_evaluate_string()
*		 is with the with a string of "(x+y)/2".  Now,
*		 kexpr_evaluate_string() will evaluate the expression, using the
*		 values of x and y defined by the previous call:
*		 [(0.1 + 0.9) / 2].  The value returned in this case would be
*		 "0.5".
*
*         Input: id     - the variable identifier.
*                string - the string to be evaluated.
*
*        Output: value  - if no error occurred then the integer
*                         value of the expression is stored and
*			  TRUE returned.
*                error  - if an error occurred the error message is stored in
*                         the error string array and False returned.  The
*                         error string array must be at least a 1024 string
*                         array that is allocated by the calling routine.  If
*                         the error string array has not been allocated by the
*                         calling routine (error is passed in as NULL) then the
*                         error message is output with the kerror facility.
*
*	Returns: TRUE (1) on success, FALSE (0) otherwise
*  Restrictions:
*    Written By: Mark Young  
*          Date: Jan 23, 1993
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kexpr_evaluate_string(
*		 !   long id,
*		 !   char *string,
*		 !   char *value,
*		 !   char *error)
*
*************************************************************/

#define kexpr_evaluate_string(id, string, value, error) \
	       kexpr_evaluate_generic(id, string, KSTRING, (kaddr) value, error)

	/*-------------------------------------*
	|       typedefs
	---------------------------------------*/


	/*-------------------------------------*
	|       global variable declarations
	---------------------------------------*/


	/*-------------------------------------*
	|       macros
	---------------------------------------*/


	/*-------------------------------------*
	|       routine definitions
	---------------------------------------*/


#endif /* _kexpr_kevaluate_h_ */
/* Don't add after this point */
