#include "Hershey_hershey.h"

/*
 * hershey font data structure guide
 *
 * Data points to the store of all the information, structured as follows:
 *	[ left array ][ right array ][ draw array]
 * left points to the beginning of the left array; right to the beginning
 * of the right array; each pointer in draw points to the beginning of
 * the drawing information for the corresponding character.
 *
 * Only printing characters are stored; they are indexed starting at 0;
 * Thus, draw[0] points to the info for ' ', draw[1] for '!', etc.
 * In addition, all the values in all the arrays are stored as printing
 * characters. To convert them to the actual values, we subtract NORM_OFF
 * We perform this latter conversion using the macros xleft(), xright(),
 * xwidth() and xdelta() defined above.
 *
 * All character info is based on the character's origin. Thus, xleft
 * gives the leftmost x coordinate from the character's origin (thus,
 * <= 0) of the character's extent, including spacing. xright has a 
 * similar meaning to the right. xwidth yields the width of a character;
 * xdelta the distance from the center of one character to the center
 * of the next.
 *
 * The drawing info for a character consists of a CHAR_SEP terminated
 * record of characters, with STROKE_SEP fields. Each field describes
 * one stroke of the character. Each field consists of an even number
 * of characters, each pair specifying a point (x,y) in the 
 * character's coordinate system after being shifted, as usual, by NORM_OFF. 
 * A stroke is defined as the line segments connecting the first point 
 * to the second to the third, etc.
 */

typedef struct			/* single stroke font info		*/
{
	int	reference;	/* mount reference count		*/
	char*	left;		/* left offset info			*/
	char*	right;		/* right offset info			*/
	char*	draw[FONT_CHARS+1];/* pointer into data for each char	*/
	char	data[1];	/* actual stroke data			*/
} stroke;

typedef struct			/* loaded font info			*/
{
	char*	name;		/* font name				*/
	stroke*	info;		/* loaded from file			*/
} font;

static struct			/* global state for font routines	*/
{
	stroke*	current;	/* pointer to current font info		*/
	font	fonts[MAXFONT];	/* loaded font table			*/
} state;

extern char*	malloc();
extern int	strcmp();
extern char*	strcpy();
extern int	strlen();

/*
 * mount font name on font number n
 * name==0 unmounts font n
 */

int
h_mount(n, name)
int	n;
char*	name;
{
	register stroke*	ip;
	register char*		p;
	register char**		dp;
	int			fd;
	int			i;
	struct stat		st;
	char			buf[1024];

	if (n < 0 || n >= MAXFONT) return(-1);

	if (name)	 /* check if already mounted */
		for (i = 0; i < MAXFONT; i++)
			if (state.fonts[i].name && !strcmp(state.fonts[i].name, name)) {
				if (i != n) {
					state.fonts[i] = state.fonts[n];
					state.fonts[i].info->reference++;
					}
				state.current = state.fonts[i].info;
				return(0);
				}

	if (state.fonts[n].name) { 	/* unmount previous font */
		if (!--state.fonts[n].info->reference) {
			free(state.fonts[n].name);
			free(state.fonts[n].info);
			}
		state.fonts[n].name = 0;
		}
	if (!name) return(0);

	/*load the font */
	(void)sprintf(buf, "%s/%s.hf", HERSHEYDIR, name);
	if ((fd = open(buf, 0)) < 0) return(-1);
	ip = 0;
	if (fstat(fd, &st) || 
		!(ip = (stroke*)malloc(sizeof(stroke) + st.st_size)) || 
		!(p = malloc(strlen(name) + 1)) || 
		read(fd, ip->data, st.st_size) != st.st_size) {
		if (ip) { free(ip); if (p) free(p); }
		close(fd);
		return(-1);
		}
	close(fd);
	(void)strcpy(state.fonts[n].name = p, name);
	state.current = state.fonts[n].info = ip;
	ip->reference = 1;

	p = ip->data; /* compute the stroke font data pointers */
	ip->left = p;
	p += FONT_CHARS + 1;
	ip->right = p;
	p += FONT_CHARS + 1;
	dp = &(ip->draw[0]);
	for (i=0; i<FONT_CHARS; i++) {*dp++ =p; while (*p !=CHAR_SEP) p++; p++;}
	return(0);
}

/*switch to mounted font n */
int
h_font(n)
register int	n;
{
	if (n < 0 || n >= MAXFONT || !state.fonts[n].name) return(-1);
	state.current = state.fonts[n].info;
	return(0);
}

/* return pixel width of character c */
int
h_width(c)
register int	c;
{
	c = adjust(c);
	return(xwidth(c));
}

/*
 * draw character c using h_stroke(x, y, n)
 * the width of c in pixels is returned
 */

int
h_draw(c, h_stroke, x, y)
register int	c;
int		(*h_stroke)();
short		x[];
short		y[];
{
	register char*	p;
	register int	n;
	int		w;

	c = adjust(c);
	w = xwidth(c);
	p = state.current->draw[c];
	do 	{
		n = 0;
		while ((c = *p++) != CHAR_SEP && c != STROKE_SEP) {
			x[n] = c - NORM_OFF;
			y[n] = *p++ - NORM_OFF;
			n++;
			}
		if (n > 0) (*h_stroke)(x, y, n);
		} while (c != CHAR_SEP);
	return(w);
}
