/*
  This file is part of TALER
  Copyright (C) 2024 Taler Systems SA

  TALER is free software; you can redistribute it and/or modify it under the
  terms of the GNU Affero General Public License as published by the Free Software
  Foundation; either version 3, or (at your option) any later version.

  TALER is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
  A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

  You should have received a copy of the GNU General Public License along with
  TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
/**
 * @file taler-merchant-httpd_private-get-donau-instances.c
 * @brief implementation of GET /donau
 * @author Bohdan Potuzhnyi
 * @author Vlada Svirsh
 */
#include "platform.h"
#include <jansson.h>
#include <taler/taler_json_lib.h>
#include <taler/taler_dbevents.h>
#include "taler_merchant_donau.h"
#include "taler_merchant_service.h"
#include "taler-merchant-httpd_private-get-donau-instances.h"


/**
 * Add details about a Donau instance to the JSON array.
 *
 * @param cls json array to which the Donau instance details will be added
 * @param donau_instance_serial the serial number of the Donau instance
 * @param donau_url the URL of the Donau instance
 * @param charity_name the name of the charity
 * @param charity_pub_key the public key of the charity
 * @param charity_id the ID of the charity
 * @param charity_max_per_year the maximum donation amount per year
 * @param charity_receipts_to_date the total donations received so far this year
 * @param current_year the current year being tracked for donations
 * @param donau_keys_json JSON object with key information specific to the Donau instance, NULL if not (yet) available.
 */
static void
add_donau_instance (void *cls,
                    uint64_t donau_instance_serial,
                    const char *donau_url,
                    const char *charity_name,
                    const struct DONAU_CharityPublicKeyP *charity_pub_key,
                    uint64_t charity_id,
                    const struct TALER_Amount *charity_max_per_year,
                    const struct TALER_Amount *charity_receipts_to_date,
                    int64_t current_year,
                    const json_t *donau_keys_json)
{
  json_t *json_instances = cls;

  GNUNET_assert (
    0 == json_array_append_new (
      json_instances,
      GNUNET_JSON_PACK (
        GNUNET_JSON_pack_uint64 ("donau_instance_serial",
                                 donau_instance_serial),
        GNUNET_JSON_pack_string ("donau_url",
                                 donau_url),
        GNUNET_JSON_pack_string ("charity_name",
                                 charity_name),
        GNUNET_JSON_pack_data_auto ("charity_pub_key",
                                    charity_pub_key),
        GNUNET_JSON_pack_uint64 ("charity_id",
                                 charity_id),
        TALER_JSON_pack_amount ("charity_max_per_year",
                                charity_max_per_year),
        TALER_JSON_pack_amount ("charity_receipts_to_date",
                                charity_receipts_to_date),
        GNUNET_JSON_pack_int64 ("current_year",
                                current_year),
        GNUNET_JSON_pack_allow_null (
          GNUNET_JSON_pack_object_incref ("donau_keys_json",
                                          (json_t *) donau_keys_json))
        )));
}


/**
 * Handle a GET "/donau" request.
 *
 * @param rh context of the handler
 * @param connection the MHD connection to handle
 * @param[in,out] hc context with further information about the request
 * @return MHD result code
 */
MHD_RESULT
TMH_private_get_donau_instances (const struct TMH_RequestHandler *rh,
                                 struct MHD_Connection *connection,
                                 struct TMH_HandlerContext *hc)
{
  json_t *json_donau_instances = json_array ();
  enum GNUNET_DB_QueryStatus qs;

  TMH_db->preflight (TMH_db->cls);
  qs = TMH_db->select_donau_instances (TMH_db->cls,
                                       hc->instance->settings.id,
                                       &add_donau_instance,
                                       json_donau_instances);
  if (0 > qs)
  {
    GNUNET_break (0);
    json_decref (json_donau_instances);
    return TALER_MHD_reply_with_error (connection,
                                       MHD_HTTP_INTERNAL_SERVER_ERROR,
                                       TALER_EC_GENERIC_DB_FETCH_FAILED,
                                       NULL);
  }

  return TALER_MHD_REPLY_JSON_PACK (connection,
                                    MHD_HTTP_OK,
                                    GNUNET_JSON_pack_array_steal (
                                      "donau_instances",
                                      json_donau_instances));
}
