/* socks5.c - routines used to interact with socks5 proxy
 * Copyright (C) 2005, 2006, 2007 Jia Wang <skyroam@gmail.com>
 *
 *
 *
 * This file is part of GNU Proxyknife.
 * GNU Proxyknife is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License as published by the 
 * Free Software Foundation; either version 3 of the License, or (at your 
 * option) any later version.
 *
 * GNU Proxyknife is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 * General Public License for more details.
 *
 *
 * You should have received a copy of the GNU General Public License 
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. 
 */

#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <stdio.h>
#include <netdb.h>
#include <proxyknife.h>
#include <netdb.h>
extern int h_errno;

/* Add build_auth_s5 and nego_auth here 
 * for myproxy use build a global up
 * for testproxy build global up or dynatic create up from dict 
 */


/* Build username/password authentication data for socks5 */
void
s5_build_auth ()
{

  unsigned char ulen;
  unsigned char plen;
  char *p;
  int len;
  ulen = (unsigned char) strlen (my.myuser);
  if (ulen == 0)
    xexit (EXIT_FAILURE);	/* will not be used in thread */
  plen = (unsigned char) strlen (my.mypass);
  len = ulen + plen + 3;
  p = xmalloc (len + 1);
  snprintf (p, len + 1, "\x01%c%s%c%s", ulen, my.myuser, plen, my.mypass);
  my.socks5authreq = p;
}

/* Negotiation with socks5 proxy with username/password authentication.
 *
 * Return value:
 * 0: success!
 * -1: failed.
 *  */

int
s5_nego_auth (int sockfd)
{
  char ret[2];
  int len;

  if (write (sockfd, "\x05\x01\x02", 3) == 3)
    {
      if (read (sockfd, ret, 2) == 2)
	{
	  if (!strncmp (ret, "\x05\x02", 2))
	    {
	      len = strlen (my.socks5authreq);
	      if (write (sockfd, my.socks5authreq, len) == len)
		{
		  if (read (sockfd, ret, 2) == 2)
		    {
		      if (!strncmp (ret, "\x01\x00", 2))
			{
			  return 0;
			}
		    }
		}
	    }
	}
    }
  return -1;
}

/* Negotiation with free socks5 proxy. 
 *
 * Return value:
 * 0: success!
 * -1: failed.
 * */
int
s5_nego_free (int sockfd)
{
  char ret[2];
  if (write (sockfd, "\x05\x01\x00", 3) == 3)
    {
      if (read (sockfd, ret, 2) == 2)
	{
	  if (!strncmp (ret, "\x05\x00", 2))
	    return 0;
	}
    }
  return -1;
}

/* Send connect command (connect to serv) to socks5 tunnel. 
 *
 * Return value:
 * 0: success!
 * -1: failure.
 *
 *  This should be committed after successful negotiation with socks5 proxy.
 * */
int
s5con (int sockfd, struct sockaddr *serv)
{
  unsigned char p[10] = { "\x05\x01\x00\x01" };
  *(struct in_addr *) (p + 4) = ((struct sockaddr_in *) serv)->sin_addr;
  *(short int *) (p + 8) =
    (short int) (((struct sockaddr_in *) serv)->sin_port);

  if (write (sockfd, p, 10) == 10)
    {
      /* ATYP is expected as IPV4 . Need repaired here. */
      if (read (sockfd, p, 10) == 10)
	{			/* ATYP is not parsed  here. The chars unread
				   will not  the following GET and Verify. */
	  if (!strncmp ((const char *) p, "\x05\x00\x00", 3))
	    return 0;
	}
    }
  return -1;
}

/* Build connection to remote server through free or username/password-
 * authentication socks5 proxy. 
 *
 * Return value:
 * 0: success!
 * -1: failure.
 * */
int
CONVIASOCKS5_MAIN (int sockfd, struct sockaddr *addr)
{
  int ret, len, offset;
  char *p;

  ret =
    connect (sockfd, (struct sockaddr *) &(my.myaddr),
	     sizeof (struct sockaddr));
  if (ret == -1)
    {
      perror (__FILE__ ": CONVIASOCKS5_MAIN:connect:Connect to myaddr");
      return -1;
    }

  /* build request */
  if (my.mytype == SOCKS5_CONNECT_AUTH)

    {
      ret = s5_nego_auth (sockfd);
      if (ret == -1)
	return -1;
    }
  else if (my.mytype == SOCKS5_CONNECT)
    {
      ret = s5_nego_free (sockfd);
      if (ret == -1)
	return -1;
    }
  else
    {
      fprintf (stderr,
	       "%s %s: CONVIASOCKS5_MAIN:This message can't  be printed!\n",
	       progname, __FILE__);
      abort ();
      /* return -1; */
    }

  /* send */
  ret = s5con (sockfd, (struct sockaddr *) addr);
  if (ret == -1)
    return -1;
  return 0;
}

/* ***************************inside thread **********************/

/* Build connection to remote server through free or username/password-
 * authentication socks5 proxy. 
 *
 * The time sending connect command to socks5 proxy is stored in *start.
 * The time receiving success  reply from socks5 proxy is stored in * end.
 * 
 * Return value:
 * 0: success!
 * -1: failure.
 * */
int
CONVIASOCKS5 (int sockfd, struct timeval *start,
	      struct timeval *end, struct thread_mem *thread_mem)
{
  int ret, len, offset;
  char *p;

  ret =
    connect (sockfd, (struct sockaddr *) &(my.myaddr),
	     sizeof (struct sockaddr));
  if (ret == -1)
    {
      fprintf (stderr, "%s %s: CONVIASOCKS5:connect:Connect to myaddr",
	       progname, __FILE__);
      return -1;
    }

  /* build request */
  if (my.mytype == SOCKS5_CONNECT_AUTH)

    {
      ret = s5_nego_auth (sockfd);
      if (ret == -1)
	return -1;
    }
  else if (my.mytype == SOCKS5_CONNECT)
    {
      ret = s5_nego_free (sockfd);
      if (ret == -1)
	return -1;
    }
  else
    {
      fprintf (stderr, "%s: CONVIASOCKS5:This message can't  be printed!\n",
	       progname);
      abort ();
      /* return -1; */
    }

  /* send */
  gettimeofday (start, NULL);
  ret = s5con (sockfd, (struct sockaddr *) &(thread_mem->queue.proxyaddr));
  if (ret == -1)
    return -1;
  gettimeofday (end, NULL);
  return 0;
}


/* Negotiation with free socks5 proxy , build connection to target through 
 * free socks proxy. At last it call GET to send HTTP/1.0 GET request 
 * to target.
 *
 * The time sending connect command to socks5 proxy is stored in *start_CON.
 * The time receiving successfuly reply to this is stored in *end_CON.
 * 
 * Return value:
 *  >0: success!
 *  -1: failure.
 *  
 * Verify should be done immediately after this for correct delay.
 *  */
int
S5PCONGET (int sockfd, struct timeval *start_CON, struct timeval *end_CON,
	   struct timeval *start_GET, struct thread_mem *thread_mem)
{
  int ret;

  ret = s5_nego_free (sockfd);
  if (ret == -1)
    return -1;
  gettimeofday (start_CON, NULL);
  ret = s5con (sockfd, (struct sockaddr *) &(target.targetaddr));
  if (ret == -1)
    return -1;
  gettimeofday (end_CON, NULL);

  return GET (sockfd, start_GET, thread_mem);

}
