/* Functions for handling sentence data.

   Copyright (C) 2012 Ian Dunn.

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <stdlib.h>
#include <string.h>
#include "sen-data.h"
#include "process.h"
#include "list.h"
#include "vec.h"
#include "rules.h"
#include "var.h"
#include "aio.h"
#include "proof.h"

/* Initializes the sentence data.
 *  input:
 *    line_num - the line number to set to this sentence data.
 *    rule - the rule to set.
 *    text - the text to set.
 *    refs - the references for this sentence data.
 *    premise - whether or not this is a premise.
 *    file - the file name to set, in the event that rule == RULE_LM.
 *    subproof - whether or not this starts a subproof.
 *    depth - the depth of this sentence.
 *  output:
 *    the newly initialized sentence data, or NULL on error.
 */
sen_data *
sen_data_init (int line_num, int rule, unsigned char * text,
	       int * refs, int premise, unsigned char * file,
	       int subproof, int depth)
{
  sen_data * sd;
  sd = (sen_data *) calloc (1, sizeof (sen_data));
  if (!sd)
    {
      perror (NULL);
      return NULL;
    }

  sd->vars = NULL;

  sd->line_num = line_num;
  sd->rule = rule;

  if (text)
    {
      int text_len;
      text_len = strlen (text);
      sd->text = (unsigned char *) calloc (text_len + 1,
					   sizeof (char));
      if (!sd->text)
	{
	  perror (NULL);
	  return NULL;
	}

      strcpy (sd->text, text);
    }
  else
    {
      sd->text = (unsigned char *) calloc (512, sizeof (char));

      if (!sd->text)
	{
	  perror (NULL);
	  return NULL;
	}
    }

  if (file)
    {
      int file_len;
      file_len = strlen (file);

      sd->file = (unsigned char *) calloc (file_len + 1,
					   sizeof (char));
      if (!sd->file)
	{
	  perror (NULL);
	  return NULL;
	}
      strcpy (sd->file, file);
    }
  else
    {
      sd->file = NULL;
    }

  sd->refs = refs;

  sd->premise = premise;
  sd->depth = depth;
  sd->subproof = subproof;

  return sd;
}

/* Destroys sentence data.
 *  input:
 *    sd - the sentence data to destroy.
 *  output:
 *    none.
 */
void
sen_data_destroy (sen_data * sd)
{
  sd->line_num = sd->rule = -1;

  if (sd->text)
    free (sd->text);
  sd->text = NULL;

  if (sd->refs)
    free (sd->refs);

  if (sd->vars)
    {
      int i = 0;
      unsigned char * var = sd->vars[i];
      for (i = 0; sd->vars[i] != NULL; i++)
	free (sd->vars[i]);

      free (sd->vars);
    }
}

char *
sen_data_evaluate (sen_data * sd, int * ret_val, list_t * pf_vars, list_t * lines)
{
  if (sd->text[0] == '\0')
    {
      if (sd->line_num == 1)
	{
	  *ret_val = VALUE_TYPE_BLANK;
	  return CORRECT;
	}
      else
	{
	  *ret_val = VALUE_TYPE_ERROR;
	  return _("Only the first sentence can be blank.");
	}
    }

  int ret;

  ret = check_text (sd->text);
  if (ret == -2)
    return NULL;

  *ret_val = VALUE_TYPE_ERROR;

  switch (ret)
    {
    case 0:
      break;
    case -2:
      return _("The sentence has mismatched parenthesis.");
    case -3:
      return _("The sentence has invalid connectives.");
    case -4:
      return _("The sentence has invalid quantifiers.");
    }

  *ret_val = VALUE_TYPE_BLANK;

  if (sd->premise || sd->subproof)
    {
      return CORRECT;
    }

  if (sd->rule == -1)
    {
      *ret_val = VALUE_TYPE_RULE;
      return _("The sentence is missing a rule.");
    }

  const char * rule = rules_list[sd->rule];

  vec_t * refs;

  refs = init_vec (sizeof (char *));
  if (!refs)
    return NULL;

  int i;

  for (i = 0; sd->refs[i] != -1; i++)
    {
      item_t * cur_ref;
      sen_data * ref_data;
      int ret_chk;

      if (sd->refs[i] >= lines->num_stuff)
	return NULL;

      cur_ref = ls_nth (lines, sd->refs[i] - 1);
      ref_data = cur_ref->value;

      ret_chk = check_text (ref_data->text);
      if (ret_chk == -2)
	return NULL;

      if (ret_chk < 0)
	{
	  destroy_vec (refs);
	  *ret_val = VALUE_TYPE_REF;

	  return _("One of the sentence's references has a text error.");
	}

      unsigned char * ref_text;
      ref_text = ref_data->sexpr;

      ret_chk = vec_str_add_obj (refs, ref_text);
      if (ret_chk < 0)
	return NULL;
    }

  // Next, get the variables.

  vec_t * vars = init_vec (sizeof (variable));
  item_t * var_itr;
  proof_t * proof;

  for (var_itr = pf_vars->head; var_itr; var_itr = var_itr->next)
    {
      variable * var = var_itr->value;
      vec_add_obj (vars, var);
    }


  char * eval_text, * fin_text;
  if (!sd->sexpr)
    {
      eval_text = die_spaces_die (sd->text);
      if (!eval_text)
	return NULL;

      fin_text = convert_sexpr (eval_text);
      if (!fin_text)
	return NULL;
      free (eval_text);
      sd->sexpr = fin_text;
    }
  fin_text = sd->sexpr;

  // Check for a file.

  if (sd->file)
    {
      proof = aio_open (sd->file);
      if (!proof)
	return NULL;
    }

  char * proc_ret = process (fin_text, refs, rule, vars, proof);
  if (!proc_ret)
    return NULL;

  destroy_str_vec (refs);
  destroy_vec (vars);
  proof_destroy (proof);

  if (!strcmp (proc_ret, CORRECT))
    *ret_val = VALUE_TYPE_TRUE;
  else
    *ret_val = VALUE_TYPE_FALSE;

  return proc_ret;
}
