;;; -*- Mode: Lisp; Syntax: Common-Lisp; Package: XGINSENG; Base: 10 -*-

;; Copyright (C) 1984, 1988, 1989, 1993 Research Foundation of 
;;                                      State University of New York

;; Version: $Id: describe.lisp,v 1.4 1993/06/04 06:27:55 snwiz Exp $

;; This file is part of SNePS.

;; SNePS is free software; you may redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; SNePS is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with SNePS; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA, or to
;; Dr. Stuart C. Shapiro, Department of Computer Science, State University of
;; New York at Buffalo, 226 Bell Hall, Buffalo, NY 14260, USA

(in-package :xginseng)


;;;;   Contains the definitions for output functions


(defconstant *child-hsep* 100
  "Horizontal distance between child nodes of a `down-described' node")
(defconstant *child-vsep* 100
  "Vertical distance of children from parent node when down-described")
(defconstant *parent-hsep* 150
  "Horizontal distance between parent nodes of a `described' node")
(defconstant *parent-vsep* 100
  "Vertical distance of parents from parent node when described")

;;; Arg: A garnet node
;;; Returns the list of children nodes of <node>
(defun describe-down (a-node)
  (declare (special display-aggregate))
  (let ((down-fcs (sneps::down.fcs (g-value a-node :sneps-node)))
	(coord-list (child-coordinates a-node))
	child-list
	doubled-ns)
    (setf doubled-ns (do-double-nodes a-node down-fcs coord-list))
    (sneps::do.fcs (rel ns down-fcs)
		   (sneps::do.ns (newnode ns)
				 (let ((gi-node (intern (stringify.n newnode) 'xginseng)))
				   ;; if gi-node has not been built previously, then build it;
				   ;; Otherwise, make sure that it's not hidden
				   (if (not (on-display-p gi-node))
				       (new-node gi-node newnode (pop coord-list))
				     (s-value (eval gi-node) :visible t))
				   (push (eval gi-node) child-list)
				   (if (not (sneps::ismemb.ns newnode doubled-ns))
				       (opal:add-component display-aggregate
							   (make-arc a-node (eval gi-node) (stringify.r rel))
							   :back)))))
    (opal:update display-window)
    child-list))

;;; Arg: A garnet node
;;; Returns the list of parent nodes of <node>
(defun describe-up (a_node)
  (let* ((up-fcs (sneps::up.fcs (g-value a_node :sneps-node)))
	 (coord-list (parent-coordinates a_node))
	 parent-list
	 parent-node)
    (sneps::do.fcs (rel ns up-fcs parent-list)
		   (sneps::do.ns (newnode ns)
				 (let ((gi-node (intern (stringify.n newnode) 'xginseng)))
				   (if (not (on-display-p gi-node))
				       (setq parent-node (new-node gi-node newnode (pop coord-list)))
				     (setq parent-node (eval gi-node)))
				   (describe-down parent-node)
				   (push (eval gi-node) parent-list))))
    (opal:update display-window)
    parent-list))

(defun all-ancestors (a-node)
  (let (next-pars)
    (do* ((parents (describe-up a-node) (append (cdr parents) next-pars))
	  (par (first parents) (first parents)))
	 ((null parents))
      (setf next-pars (describe-up par)))))

(defun all-descendants (a-node)
  (let (next-children)
    (do* ((children (describe-down a-node) (append (cdr children) next-children))
	  (child (first children) (first children)))
	 ((null children))
      (setf next-children (describe-down child)))))

(defun child-coordinates (node)
"Returns a list of coordinate pairs for the children of the GARNET <node>
without moving outside the window"
(let ((num-child (count-children (g-value node :sneps-node)))
      lst)
  (mapcar #'(lambda (short-lst)
	      (rplacd short-lst 
		      (list (min (+ (g-value node :top) *child-vsep*)
				 (- *totaldisplayheight* 30))))) ;inserts :top coord
	  ; This DO returns a list of singleton lists, whose elements are the 
	  ; prospective values of the :left slots of the children nodes
	  (do ((leftmost
		(max 0 (- (g-value node :left) (floor (* (1- num-child) *child-hsep*) 2))))
	       (count 1 (+ count 1)))
	      ((> count  num-child) lst)
	    (push (list (min (+ (* *child-hsep* (1- count)) leftmost)
			     (- *totaldisplaywidth* 30)))
		  lst)))))

(defun parent-coordinates (node)
"Returns a list of <num-parent> coordinate pairs for the parents of GARNET <node>
without moving outside the window."
(let ((num-parent (count-parents (g-value node :sneps-node)))
      lst)
  (mapcar #'(lambda (short-lst)
	      (rplacd short-lst ;inserts :top coordinate
		      (list (max 0 (- (g-value node :top) *parent-vsep*)))))
	  ;; This DO returns a list of singleton lists, whose elements are the 
	  ;; prospective values of the :left slots of the parent nodes
	  (do ((leftmost
		(max 0 (- (g-value node :left) (floor (* (1- num-parent) *parent-hsep*) 2))))
	       (count 1 (+ count 1)))
	      ((> count  num-parent) lst)
	    (push (list (min (+ (* *parent-hsep* (1- count)) leftmost)
			     (- *totaldisplaywidth* 30)))
		  lst)))))

(defun do-double-nodes (node fcs coords)
  "Takes a GARNET <node>, the <fcs> flat-cable-set of the corresponding SNEPS
node, and the list of coordinates <coords>, returning a SNEPS node-set of
the dominated nodes which have two arcs between them and <node>"
  (let ((previous (sneps::new.ns))
	(doubled (sneps::new.ns))
	(intersect (sneps::new.ns)))
    (sneps::do.fcs (rel ns fcs)
      (if (setf intersect (sneps::intersect.ns ns previous))
	  (sneps::do.ns (dest-node intersect)
			(make-doubled-arc-node node dest-node (pop coords))
			(setf doubled (sneps::insert.ns dest-node doubled))))
      (setf previous (sneps::union.ns ns previous)))
    doubled))

(defun make-doubled-arc-node (node dom-node new-coords)
  "NB: <node> is a GARNET object, while <dom-node> is a SNEPS node which
has TWO arcs arriving at it from <node>"
  (let ((rels (downward-arcs-between (g-value node :sneps-node) dom-node))
	(gi-node (intern (stringify.n dom-node) 'xginseng)))
    (if (not (on-display-p gi-node))
	(progn 
	  (new-node gi-node dom-node new-coords)
	  (make-double-arc node (eval (intern (stringify.n dom-node) 'xginseng))
			   (stringify.r (first rels)) (stringify.r (second rels))))
      (make-double-arc node (eval (intern (stringify.n dom-node) 'xginseng))
		       (stringify.r (first rels)) (stringify.r (second rels))))))


(defun count-parents (node)
  "Returns number of parent nodes of sneps <node>"
  (let ((count 0))
    (sneps::do.fcs (rel ns (sneps::up.fcs node))
		   (incf count (sneps::cardinality.ns ns)))
    count))


(defun count-children (node)
  "Returns number of child nodes of sneps <node>"
  (let ((count 0))
    (sneps::do.fcs (rel ns (sneps::down.fcs node))
		   (incf count (sneps::cardinality.ns ns)))
    count))

;;; Thanks to Hans Chalupsky for this function
(defun downward-arcs-between (dominating-node dominated-node)
  (let ((down-arcs (sneps::new.rs)))
    (sneps::do.fcs (arc ns (sneps::down.fcs dominating-node))
	    (sneps::do.ns (node ns)
		   (if (sneps::iseq.n node dominated-node)
		       (setq down-arcs (sneps::insert.rs arc down-arcs)))))
    down-arcs))

