;;; -*- Mode: Lisp; Syntax: Common-Lisp; Package: USER; Base: 10 -*-

;; Copyright (C) 1984, 1988, 1989, 1993 Research Foundation of 
;;                                      State University of New York

;; Version: $Id: system-utils.lisp,v 1.25 1993/07/21 08:41:25 snwiz Exp $

;; This file is part of SNePS.

;; SNePS is free software; you may redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; SNePS is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with SNePS; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA, or to
;; Dr. Stuart C. Shapiro, Department of Computer Science, State University of
;; New York at Buffalo, 226 Bell Hall, Buffalo, NY 14260, USA

(in-package :user)


;;; Contains a set of utilities that allow to build a SNePS system
;;; on various operating systems from a simple system definition.
;;; The portability is achieved with the use of logical pathnames.
;;; The system definition is a list of entries that describe an 
;;; operation and a pathname. The entries have the following form:
;;;
;;;        (<operation> <logical pathname>)
;;;
;;;  e.g., (:COMPILE "sneps:ginseng;desc.LISP")
;;;
;;; where <operation> is either :COMPILE, :COMPILE-LOAD, :LOAD or
;;;       :SNEPSLOG-ATNIN.


(defstruct (DESCRIPTION (:type list))
  operation
  file)

(defmacro GET-SIMPLE-SYSTEM-INFO (system-name)
  "Retrieves information associated with SYSTEM-NAME"
  `(get (intern (string ,system-name) 'user) 'simple-system-info))

(defmacro SET-SIMPLE-SYSTEM-INFO (system-name info)
  "Stores INFOrmation associated with SYSTEM-NAME"
  `(setf (get-simple-system-info ,system-name) ,info))

(defmacro RECORD-SIMPLE-SYSTEM-ACTION (system-name action)
  "Records that a certain ACTION (a keyword) has been performed on
the system with SYSTEM-NAME."
  `(set-simple-system-info
    ,system-name
    (adjoin ,action (get-simple-system-info ,system-name))))

(defmacro FORGET-SIMPLE-SYSTEM-ACTION (system-name action)
  "Forgets that a certain ACTION (a keyword) has been performed on
the system with SYSTEM-NAME."
  `(set-simple-system-info
    ,system-name
    (remove ,action (get-simple-system-info ,system-name))))

(defmacro CHECK-SIMPLE-SYSTEM-ACTION (system-name action)
  "Checks whether a certain ACTION (a keyword) has been performed on
the system with SYSTEM-NAME."
  `(member ,action (get-simple-system-info ,system-name)))

(defun SIMPLE-SYSTEM-CREATED-P (system-name)
  "Returns non-NIL if SYSTEM-NAME has been created with a call to
make-simple-system."
  (or (check-simple-system-action system-name :load)
      (check-simple-system-action system-name :compile)
      (check-simple-system-action system-name :recompile)
      (check-simple-system-action system-name :load-uncompiled)))

(defun MAKE-SIMPLE-SYSTEM (system-name system-definition
			   &key (mode :load) (verbose *sneps-verbose*))
  "Makes the system SYSTEM-NAME from a simple SYSTEM-DEFINITION (a list of
descriptions). If MODE is :LOAD then :compile  descriptions will be ignored
and :compile-load descriptions will just load the (compiled) file. If MODE
is :RECOMPILE all :compile and :compile-load descriptions will execute an
unconditional compilation. If MODE is :COMPILE only :compile and :compile-load
descriptions with source files newer than the current binary file will
execute a compilation. If MODE is :LOAD-UNCOMPILED all :load and :compile-load
descriptions will load source files without any compilations taking place.
A :snepslog-atnin description will load the atn with the snepslog:atnin
function regardless of the value of MODE. 
   If VERBOSE is NIL all loading, compiling or warning messages will be
suppressed, i.e., sent to a null stream, and only the percentage of files
already loaded will be displayed in 10% steps.
   Once the system has been loaded successfully the SYSTEM-NAME will be
associated with the action specified in MODE."

  ;; First convert system-definition to description format if necessary
  (when (and (consp system-definition)
	     (consp (first system-definition)))
    (setq system-definition
	  (mapcar #'(lambda (entry)
		      (make-description
		       :operation (first entry)
		       :file (second entry)))
		  system-definition)))
  (let* ((*sneps-verbose* verbose)
	 ;; Stores output in non-VERBOSE mode. Use reusable buffer so
	 ;; not a lot of string space will be wasted.
	 (output-buffer (make-array 1024
				    :element-type 'string-char
				    :fill-pointer t))
	 ;; Percentage of already processed file descriptions (used in
	 ;; non-VERBOSE mode). Increases in 10% steps.
	 (percentage-processed 0)
	 ;; Temporary variable to to store the actual percentage of
	 ;; processed descriptions.
	 current-percentage
	 ;; Number of descriptions processed so far.
	 (descriptions-processed 0)
	 type-supplied-p pathname source-pathname binary-pathname)
    
    (format t "~&Loading system ~a..." system-name)
    (dolist (description system-definition)
      ;; If non-VERBOSE mode calculate percentage of descriptions
      ;; processed so far and print update if necessary.
      (unless verbose
	(incf descriptions-processed)
	;; This percentage changes every 10%
	(setq current-percentage
	      (* 10 (floor (* 10 descriptions-processed)
			   (length system-definition)))) 
	(unless (= percentage-processed current-percentage)
	  (setq percentage-processed current-percentage)
	  (format t "~d% " percentage-processed)
	  (force-output)
	  ))

      ;; Using all these internal `LP' functions is ugly, but the TI-Explorers
      ;; wouldn't let me do it the elegant way (this way I at least know what
      ;; the pathname functions are doing, relying on smooth cooperation of
      ;; the `LP' versions and the default versions is too dangerous - it works
      ;; quite often though not always):
      (setq pathname (description-file description))
      (setq pathname
	(lp::parse-generic-namestring
	  (if (eq (lp::host-type (lp::get-host-string pathname))
		  :logical)
	      (lp:translate-logical-pathname pathname)
	      pathname)
	  nil nil))
      ;; Now we have a physical-pathname:
      (setq type-supplied-p (lp::physical-pathname-type pathname))
      ;; Create pathnames with forced extensions (I used to do this with
      ;; `(make-pathname :type "lisp" :defaults pathname)' but the Explorers
      ;; insisted on upcasing it to "LISP" - sigh).
      (setq source-pathname
	(lp::physical-namestring
	  (if type-supplied-p
	      pathname
	      (lp::make-physical-pathname
		:host (lp::physical-pathname-host pathname)
		:device (lp::physical-pathname-device pathname)
		:directory (lp::physical-pathname-directory pathname)
		:name (lp::physical-pathname-name pathname)
		;; hardwire a canonical :LISP extension:
		:type :LISP))))
      (setq binary-pathname
	(lp::physical-namestring
	  (lp::make-physical-pathname
	    :host (lp::physical-pathname-host pathname)
	    :device (lp::physical-pathname-device pathname)
	    :directory (lp::physical-pathname-directory pathname)
	    :name (lp::physical-pathname-name pathname)
	    ;; hardwire a canonical :FASL extension:
	    :type :FASL)))

      ;; Reset output buffer for non-VERBOSE mode
      (setf (fill-pointer output-buffer) 0)
      (with-output-to-string (null-stream output-buffer)
	(let* ((*load-verbose* verbose)
	       (*terminal-io* (cond (verbose *terminal-io*)
				    (t null-stream)))
	       (*standard-output* *terminal-io*))

	  ;; First, check whether we have to compile
	  (case (description-operation description)
	    ((:compile :compile-load)
	     (when (and (not (eq mode :load-uncompiled))
			(or (eq mode :recompile)
			    (and (eq mode :compile)
				 (> (or (file-write-date source-pathname) 1)
				    (or (and (probe-file binary-pathname)
					     (file-write-date binary-pathname))
					0)))))
	       (format t "~&;; Compiling ~a" source-pathname)
	       (compile-file source-pathname))))

	  ;; Then, check whether and how we have to load (because we allow
	  ;; customization of binary file extensions we cannot rely on the
	  ;; standard source/binary mechanism provided by most `load's):
	  (case (description-operation description)
	    ((:load :compile-load)
	     (cond ((or (eq mode :load-uncompiled)
			(and type-supplied-p (eq mode :load)))
		    (sneps-load source-pathname))
		   ((probe-file binary-pathname)
		    (sneps-load binary-pathname))
		   (t (sneps-load source-pathname))))
	    (:snepslog-atnin
	     (format t "~&;; Loading ATN ~a" source-pathname)
	     ;; Don't have SNEPSLOG package when this function gets loaded:
	     (funcall (intern "ATNIN" 'snepslog) source-pathname))
	    ))))

    ;; Finally, record what we did for this particular system
    (record-simple-system-action system-name mode)
    ))


(defun LOAD-GARNET (&rest modules)
  "Loads all MODULES of the Garnet system. Returns T if Garnet is installed
and the loading completed successfully. The elements of MODULES can be
keywords, for example, :multifont, to specify that that module should be
loaded, or a string which is taken to be a particular Garnet file to be
loaded. First all modules and then all files will be loaded. A typical call
would be (load-garnet :kr :opal :inter :aggregadgets)."
  (when (and (probe-file "garnet:garnet-loader.DLISP")
	     (or (find-package 'opal)	; Garnet already loaded
		 (boundp '*default-x-display-name*) ; for compilation
		 #+(and lucid unix)(environment-variable "DISPLAY")
		 #+(and allegro unix)(sys:getenv "DISPLAY")
		 #+(and cmu unix)(cdr (assoc :DISPLAY *environment-list*))
		 #-(and unix (or allegro lucid cmu))
		 (y-or-n-p "Are you running X-Windows? ")))
    (let ((load-kr-p nil)
	  (load-opal-p nil)
	  (load-inter-p nil)
	  (load-multifont-p nil)
	  (load-gesture-p nil)
	  (load-ps-p nil)
	  (load-aggregadgets-p nil)
	  (load-aggregraphs-p nil)
	  (load-debug-p nil)
	  (load-gadgets-p nil)
	  (load-demos-p nil)
	  (load-lapidary-p nil)
	  (load-gilt-p nil)
	  ;; launch-process-p
	  ;; Lucid-4.0 bombs during Garnet load if *print-pretty* is t
	  (*print-pretty* nil)
	  load-variable)
      (declare (special load-kr-p load-opal-p load-inter-p load-multifont-p
			load-gesture-p load-ps-p load-aggregadgets-p
			load-aggregraphs-p load-debug-p load-gadgets-p
			load-demos-p load-lapidary-p load-gilt-p
			launch-process-p))
      #+(and lucid sparc)
        (setq launch-process-p nil)
      (dolist (module modules)
	(when (and (symbolp module)
		   (eq (symbol-package module)
		       (find-package 'keyword)))
	  (setq load-variable (intern (format nil "LOAD-~a-P" module)))
	  (when (and (boundp load-variable)
		     (null (eval load-variable)))
	    (set load-variable t))
	  ))
      (sneps-load "garnet:garnet-loader.DLISP")
      (dolist (module modules)
	(when (stringp module)
	  (sneps-load module)))
      t)))

#-explorer
;; A poor man's version of kill-package (just rename it):
(defun KILL-PACKAGE (package)
  (let ((package (find-package package)))
    (when package
      (unuse-package (package-use-list package) package)
      (rename-package package (gentemp)))))

(defun KILL-SNEPS ()
  "Kill all SNePS related packages and global variables. This allows
proper reloading of SNePS without having to restart or reboot the
lisp environment."
  (declare (special *sneps-packages*))
  (in-package :user)
  (let (;; Have to save these before they get unbound
	(sneps-packages (and (boundp '*sneps-packages*)
			       *sneps-packages*)))
    (setf (symbol-plist 'kill-sneps) nil)
    (dolist (fn '(sneps snepslog))
      (when (fboundp fn)
	(fmakunbound fn))
      (setf (symbol-plist fn) nil))
    (do-symbols (sym (find-package 'user))
      (when (search "*SNEPS-" (symbol-name sym) :test #'char-equal)
	;;(format t "~&Kill symbol: ~a" sym)
	(and (boundp sym)
	     (not (constantp sym))
	     (makunbound sym))
	(setf (symbol-plist sym) nil)))
    (dolist (p sneps-packages)
      (if (find-package p)
	  (kill-package p)))))

;; Clisp's version of `probe-file' has a problem with non-existing
;; intermediate directories, hence, I have to make it more tolerant:
#+(and clisp unix)
(defun clisp-new-probe-file (file)
  (let ((*error-handler* 
	 #'(lambda (&rest ignore)
	     (declare (ignore ignore))
	     (return-from clisp-new-probe-file nil))))
    (funcall #'clisp-old-probe-file file)))

#+(and clisp unix)
(unless (fboundp 'old-clisp-probe-file)
  (setf (symbol-function 'clisp-old-probe-file)
    (symbol-function 'probe-file))
  (setf (symbol-function 'probe-file)
    (symbol-function 'clisp-new-probe-file)))
