;;; -*- Mode: Lisp; Syntax: Common-Lisp; Package: SNIP; Base: 10 -*-

;; Copyright (C) 1984, 1988, 1989, 1993 Research Foundation of 
;;                                      State University of New York

;; Version: $Id: patch.lisp,v 1.4 1993/07/20 08:49:47 snwiz Exp $

;; This file is part of SNePS.

;; SNePS is free software; you may redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; SNePS is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with SNePS; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA, or to
;; Dr. Stuart C. Shapiro, Department of Computer Science, State University of
;; New York at Buffalo, 226 Bell Hall, Buffalo, NY 14260, USA

(in-package :snip)


;------------------------------------------------------------------------
;  send-node-to-node-requests  in "snip/fns/nrn-requests.lisp"
;------------------------------------------------------------------------
;
(defun send-node-to-node-requests (restriction context)
  "Modified to facilitate knowledge shadowing by using the most general 
   common instances (mgci) of two patterns.  It now sends requests only 
   to those nodes that are determined not to be redundant by checking  
   the mgci of the current node (source node) and the matched node 
   (target node).       modified by choi: 2/25/92"

  (let* ((rsbst (subst.restr restriction))
	 (rnode (match::applysubst *NODE* rsbst)))
    ; if the requesting node 'rnode' (*NODE* restricted by 'rsbst')
    ; is already asserted,
    ; send request directly to 'rnode'  without doing match.
    (if (sneps::isassert.n rnode)
	(send-request (make.ch nil rsbst context *NODE* 'OPEN) 
		      rnode restriction)
      ; otherwise, call match and check if the mgci of the current node
      ; (restricted by 'rsbst') and the target node is asserted.
      ; send request to such a node that mgci is not asserted.
      (do.supmatchingset (sup (match-in-context *NODE* context rsbst) t)
        (let ((tnode (tnode.supmatching sup))
	      (tsbst (target-sub.supmatching sup))
	      (ssbst (source-sub.supmatching sup)))
	  (unless (or (eq tnode *NODE*)
		      ; check if the mgci of the current and target node
		      ; is asserted so that it can block the request
		      (and (sneps::ispat.n tnode)
			   (is-mgci-asserted tnode tsbst)))
	    (send-request (make.ch tsbst ssbst context *NODE* 'OPEN)
			  tnode restriction)))))))


(defun is-mgci-asserted (tnode tsbst)
  "The most general common instance (mgci) of two patterns S and T 
   is Ss or Tt, where s is a source binding and t is a target binding.
   Obtaining mgci in SNIP is implemented by applying the target binding 
   to the target node.
   This procedure will return T if the mgci or the negation of the mgci
   is asserted in the current context."
  (let ((mgci (match::applysubst tnode tsbst)))
    (and (not (sneps::ispat.n mgci))
	 (or (sneps::isassert.n mgci)
	     (if (snebr::is-negated mgci)
		 (sneps::isassert.n (snebr::negator-of mgci)))))))



;------------------------------------------------------------------------
;   process-rule-use-request.rule  in file "snip/fns/rule-requests.lisp"
;------------------------------------------------------------------------

(defun process-rule-use-request.rule (request)
  "Modified to facilitate knowledge shadowing by using the instance set."
  (let* ((cqch (install-rule-use-channel request))
	 (sup (sneps:node-asupport *NODE*))
	 (filter (filter.ch request))
	 (dest (destination.ch request))
	 (crntct (context.ch request)))
    (declare (special crntct))
    (cond ((and (isassert.n *NODE*) (funcall *USABILITY-TEST* 'POS))
	   (if (member *TYPE* '(AND NOR))
	       (funcall *RULE-HANDLER*
			(make.rep filter sup 'POS dest nil crntct)
			cqch)
	       (send-ant-requests.rule cqch)))	       
	  (t (let ((rule-sbst 
		    (restrict-binding-to-pat (filter.ch request) *NODE*))
		   (restr nil))
	       ; the current rule (pattern) is activated only when 
	       ; there is no instance for this rule
	       (when (or (is-wh-question rule-sbst)
			 (not (apply-known-rule-instance cqch)))
		 (setq restr (make.restr rule-sbst))
		 (when (not-working-on restr crntct request)
		   (send-rule-use-requests restr crntct dest))))))))



(defun apply-known-rule-instance (cqch)
  "Check if there is any known rule instance to the given cqch."
  (let ((ch (channel.cqch cqch))
	(is-there-instance nil))
    (do.set (instance *KNOWN-INSTANCES*)
       (when (acceptable-rule-instances instance ch)
	 (setq is-there-instance t)
	 (if (eq *TYPE* 'num-quantifier)
	   (send-ant-requests.rule cqch))))
    is-there-instance))


;------------------------------------------------------------------------
;   install-rule-use-channel  in file "snip/fns/rule-requests.lisp"
;------------------------------------------------------------------------

(defun install-rule-use-channel (request)
  "Build a new cqch for a request, and insert it to the
   *RULE-USE-CHANNELS* (ruc) register of *NODE*.
       <ruc> = ( <cqch> <cqch> .... )
       <cqch> = ( <channel> <antecedents> <ruiset> )   
   Three kinds of data structures for <ruiset>:
       <ruiset> =  <ruis> | <ptree-ruiset> | <sindex-ruiset>     
   The data structure of <ruiset> for the linear RUI set method:
       <ruis> = ( <rui> <rui> .... )
       <rui> = ( <sbst> <poscount> <negcount> <fns> <remark-flag> ) 
   The data structure of <ruiset> for P-tree:
       <ptree-ruiset> = (<ptree> <adj-node-list> <pnode-ruiset>)
       <adj-node-list> = ((<pnode> <pnode>) (<pnode> <pnode>) ....)
             'adj-node-list' contains adjacent ptree node pairs
       <pnode-ruiset> = ((<pnode> <ruis>) (<pnode> <ruis>) ....)
            in 'pnode-ruiset', each ptree node is associated with its
            corresponding ruiset.  'pnode-ruiset' is initially empty.
   The data structure of <ruiset> for S-indexing:
      <sindex-ruiset> = ((<sindex> <rui>) (<sindex> <rui>) ...)
            'sindex-ruiset' is initially empty.                   "

  (let* ((dest (destination.ch request))
	 (destsub (restrict-binding-to-pat (filter.ch request) dest))
	 (ch (make.ch destsub
		      (switch.ch request)
		      (context.ch request)
		      dest
		      (valve.ch request)))
	 (ants (antecedents *NODE* dest))
	 cqch)
    (cond
     
      ; P-tree is built for numerical quantifier rules even if all
      ; antecedents have the same set of variables, mainly because 
      ; they have the characteristics of both conjunctiveness and 
      ; non-conjunctiveness.
     ((eq *TYPE* 'num-quantifier)
      (let* ((cqs (nodeset.n *NODE* 'sneps::cq))
	     (ptree (ptree-for-num-quant ants cqs))
	     (adj-node-list (ptree-to-adj-list ptree))
	     (ptree-ruiset (list ptree adj-node-list nil)))
	(setq cqch (make.cqch ch (append ants cqs) ptree-ruiset))))

      ; if a rule's antecedents have the same set of variables,
      ; apply the S-indexing method.
      ((or (null ants) (is-all-pat-same-vars ants))
       (setq cqch (make.cqch ch ants nil)))
      
      ; P-tree is built for and-entailment rules whose antecedents
      ; do not have the same set of variables.
      ((eq *TYPE* 'and-entailment)
       (let* ((ptree (ptree-for-and-ent ants))
	      (adj-node-list (ptree-to-adj-list ptree))
	      (ptree-ruiset (list ptree adj-node-list nil)))
	 (setq cqch (make.cqch ch ants ptree-ruiset))))

      ; a linear ruiset is built for non-conjunctive rules whose
      ; antecedents do not have the same set of variables.
      (t (let* ((fants (nodeset-to-fnodeset ants))
		(rui (make.rui destsub 0 0 fants nil)))
	   (setq cqch (make.cqch ch ants (makeone.ruis rui))))))
    
    (setq *RULE-USE-CHANNELS* (insert.cqchset cqch *RULE-USE-CHANNELS*))
    (activate.n dest)
    cqch))


(defun is-all-pat-same-vars (pat-list)
  "check if all patterns have the same set of variables"
  (if (null pat-list) t
  (let ((var-list (sneps::all-vars.n (first pat-list)))
	(result t))
    (do* ((pl pat-list (rest pl))
	  (pat (first pl) (first pl)))
	 ((or (null pl) (null result)))
      (if (not (equiv-set (sneps::all-vars.n pat) var-list))
	  (setq result nil)))
    result)))


(defun equiv-set (s1 s2)
  "test if two sets have the same elements"
  (and (subsetp s1 s2)
       (subsetp s2 s1)))


;------------------------------------------------------------------------
;   send-ant-requests.rule  in file "snip/fns/rule-reports.lisp"
;------------------------------------------------------------------------

(defun send-ant-requests.rule (cqch &optional subst)
  (let ((ch (channel.cqch cqch))
	(ants (ants.cqch cqch))
	(ruis (ruiset.cqch cqch)))
    (unless (or (member *TYPE* '(and-entailment num-quantifier))
		(is-all-pat-same-vars ants))
      (setq *RULE-USE-CHANNELS*
	    (update.cqchset
	     (make.cqch ch ants (request-all ruis))
	     *RULE-USE-CHANNELS*)))
    (broadcast-ant-request ch ants (ch-to-restr ch) subst)))


;------------------------------------------------------------------------
;   process-one-forward-inference.non-rule in file "snip/fns/nrn-finfers.lisp"
;------------------------------------------------------------------------

(defun process-one-forward-inference.non-rule (rep)
  (let* ((crntct (context.rep rep))
	 (sub (subst.rep rep))
	 (updated-rep (make.rep sub (support.rep rep) (sign.rep rep)
				*NODE* nil crntct)))
    (setq *INCOMING-CHANNELS*
          (insert.feedset
	    (make.feeder (make.restr sub) crntct (signature.rep rep) 'OPEN)
	    *INCOMING-CHANNELS*))
    (push-forward updated-rep (sneps:in-context.ns (nodeset.n *NODE* 'sneps:ant-) t))
    (push-forward updated-rep (sneps:in-context.ns (nodeset.n *NODE* 'sneps:&ant-) t))
    (push-forward updated-rep
		  (sneps:in-context.ns (sneps:remove-if-not.ns
					 #'(lambda (n) (not (or (is-and.n n) (is-nor.n n))))
					 (nodeset.n *NODE* 'sneps:arg-)) t))

    ; in a numerical quantifier rule, a report from consequent node
    ; is sent to the rule node
    (let ((numq-rule-nodes nil))
      (do.ns (rule-node (nodeset.n *NODE* 'sneps:cq-))
	  (if (sneps::is-num-quant.n rule-node)
	      (setq numq-rule-nodes
		    (append numq-rule-nodes (list rule-node)))))
      (push-forward updated-rep (sneps:in-context.ns numq-rule-nodes t)))

    (cond ((and (not (is-node-to-node.rep rep)) (enough-resources))
	   (decrease-resources)
	   (do.supmatchingset (m (forward-match-in-context *NODE* crntct sub))
	     (unless (eq (tnode.supmatching m) *NODE*)
	       (let (ch)
		 (setq ch (make.ch (target-sub.supmatching m)
				   (source-sub.supmatching m)
				   crntct
				   (tnode.supmatching m)
				   'OPEN))
		 (install-channel ch) 
		 (send-reports (makeone.repset updated-rep) ch))))))))

;------------------------------------------------------------------------
;   process-one-forward-inference.rule in file "snip/fns/rule-finfers.lisp"
;------------------------------------------------------------------------

(defun process-one-forward-inference.rule (report)
  (when (not (is-there-rule-instance-forward report))
    (set-up-rule-use-channels report)
    (cond ((or (is-ant-to-rule.rep report)
	      ; in a numerical quantifier rule, a report from
	      ; its consequent should also be broadcast
	      (and (eq *TYPE* 'num-quantifier)
		   (is-cq-to-rule.rep report)))
	   (setq *INCOMING-CHANNELS*
	       (insert.feedset
	        (make.feeder (make.restr (subst.rep report))
			     (context.rep report)
			     (signature.rep report)
			     'OPEN)
		*INCOMING-CHANNELS*))
	   (broadcast-ant-report.rule report))
	  (t (process-one-forward-inference.non-rule report)
	     (try-applying-one-report report)))))


(defun is-there-rule-instance-forward (report)
  "Check if there is any known rule instance in the *KNOWN-INSTANCES* 
   register with respect to the substitution of report."
  (let ((all-args (nodeset.n *NODE* 'sneps::arg))
	(any-instances-checked nil))
      (when (or (and all-args (is-all-pat-same-vars all-args))
		(eq *TYPE* 'num-quantifier))
	  (do.set (instance *KNOWN-INSTANCES*)
	       (if (acceptable-rule-instances-forward instance report)
		   (setq any-instances-checked t))))
    any-instances-checked))

(defun acceptable-rule-instances-forward (inst rep)
  (and (compatible (get-var-bindings (subst.inst inst))
		   (get-var-bindings (subst.rep rep)))
       (not (isnew.sup (filter.sup (support.inst inst)
				   (sneps:value.sv (context.rep rep)))))))

(defun get-var-bindings (sbst)
  ; removes those bindings that are not associated with variables.
  ; e.g. something like (P5 . M6!) is removed
  (let ((new-sbst nil))
    (do.set (binding sbst new-sbst)
	 (if (isvar.n (first binding))
	     (setq new-sbst (append new-sbst (list binding)))))))


;------------------------------------------------------------------------
;   set-up-rule-use-channels  in file "snip/fns/rule-finfers.lisp"
;------------------------------------------------------------------------

(defun set-up-rule-use-channels (report)
  "Install rule use channels for forward inference."
  (let ((subst (subst.rep report))
	(signature (signature.rep report))
	(ct (context.rep report)))
    (cond
      ((eq *TYPE* 'num-quantifier)
      ; P-tree for numerical quantifier rules
       (let* ((ants (nodeset.n *NODE* 'sneps::&ant))
	      (cq  (nodeset.n *NODE* 'sneps::cq))
	      (ptree (ptree-for-num-quant ants cq))
	      (adj-node-list (ptree-to-adj-list ptree))
;	      (chsub (restrict-binding-to-pat subst (first cq)))
	      (ch (make.ch (new.filter) (new.switch) ct (first cq) 'open))
	      (ptree-ruiset (list ptree adj-node-list nil))
	      (cqch (make.cqch ch (append ants cq) ptree-ruiset)))
	 (setq *RULE-USE-CHANNELS* (insert.cqchset cqch *RULE-USE-CHANNELS*))
          ; bi-directional inference :
	 (send-ant-requests.rule cqch)))

      ; if a rule's antecedents (or arguments) have the same set
      ; of variables, use the S-indexing method
      ((is-all-pat-same-vars
	(or (nodeset.n *NODE* 'sneps::ant)
	    (nodeset.n *NODE* 'sneps::&ant)
	    (nodeset.n *NODE* 'sneps::arg)))
       (do.ns (cq (if (is-ant-to-rule.rep report)
		      (consequents *NODE* signature)
		      (all-consequents *NODE*)))
	  (let* ((ants (antecedents *NODE* cq))
		 (chsub (restrict-binding-to-pat subst cq))
		 (ch (make.ch chsub (new.switch) ct cq 'open))
		 (cqch (make.cqch ch ants nil)))
	    (setq *RULE-USE-CHANNELS* (insert.cqchset cqch *RULE-USE-CHANNELS*))
	    ; bi-directional inference :
	    (send-ant-requests.rule cqch))))
;	    (broadcast-ant-request ch 
;				   (remove signature ants)
;				   (ch-to-restr ch)
;				   subst)

      ; a P-tree is built for an and-entailment rule whose antecedents
      ; do not have the same set of variables
      ((eq *TYPE* 'and-entailment)
       (let* ((ants (nodeset.n *NODE* 'sneps::&ant))
	      (ptree (ptree-for-and-ent ants))
	      (adj-node-list (ptree-to-adj-list ptree))
	      (ptree-ruiset (list ptree adj-node-list nil)))
	 (do.ns (cq (nodeset.n *NODE* 'sneps::cq))
	    (let* ((chsub (restrict-binding-to-pat subst cq))
		   (ch (make.ch chsub (new.switch) ct cq 'open))
		   (cqch (make.cqch ch ants ptree-ruiset)))
	      (setq *RULE-USE-CHANNELS* (insert.cqchset cqch *RULE-USE-CHANNELS*))
	      ; bi-directional inference :
	    (send-ant-requests.rule cqch)))))
;	      (broadcast-ant-request ch
;				     (remove signature ants)
;				     (ch-to-restr ch)
;				     subst)

      ; a linear RUI set is built for non-conjunctive rules whose
      ; antecedents do not have the same set of variables
      (t (do.ns (cq (if (is-ant-to-rule.rep report)
			(consequents *NODE* signature)
		      (all-consequents *NODE*)))
	  (let* ((ants (antecedents *NODE* cq))
		 (chsub (restrict-binding-to-pat subst cq))
		 (ch (make.ch chsub (new.switch) ct cq 'open))
		 (fants (nodeset-to-fnodeset ants))
		 (rui (make.rui chsub 0 0 fants nil)))
	    (setq *RULE-USE-CHANNELS*
		  (insert.cqchset 
		   (make.cqch ch ants (makeone.ruis rui))
		   *RULE-USE-CHANNELS*))
	    ; bi-directional inference :
	    (broadcast-ant-request ch
				   (remove signature ants)
				   (ch-to-restr ch)
				   subst)))))))


;------------------------------------------------------------------------
;   INTRODUCTION-END-HANDLER in file "snip/fns/rule-eintr.lisp"
;------------------------------------------------------------------------

(defun INTRODUCTION-END-HANDLER (intr-report ich)
  (do* ((ruis (get-introduction-info intr-report ich) (others.ruis ruis))
	(rui (choose.ruis ruis) (choose.ruis ruis))
	(ch (channel.ich ich))
	(cqs (consequents.ich ich))
	(updated-ruiset (ruiset.ich ich))
	(dest (destination.ch ch))
	(crntct (context.ch ch)))
       ((isnew.ruis ruis) t)

    ; the negation of an AND rule is derived when a negative instance 
    ; is reported from any argument of the rule.
    (if (and (equal *TYPE* 'AND)
	     (= (poscount.rui rui) 0)
	     (= (negcount.rui rui) 1))
	(let* ((restr (make.restr (subst.rui rui)))
	       (subst (if (is.n dest)
			  (restrict.sbst (subst.rui rui)
					 (sneps::all-vars.n dest))
			(subst.rui rui)))
	       (newsup (compute-new-support.intr (fns.rui rui)))
	       (report (make.rep subst newsup 'NEG  *NODE* nil crntct)))
	  (unless-remarkedp.rui
	   rui
	   (remark '"~%Since it is not the case that"
		   (makeone.ns (signature.rep intr-report)) restr)
	   (remark '"I infer it is not the case that"
		   (makeone.ns *NODE*) restr))
	  (send-reports (makeone.repset report) ch)))

   ; the negation of a NOR rule is derived when a positive instance 
   ; is reported from any argument of the rule.
    (if (and (equal *TYPE* 'NOR)
	     (= (poscount.rui rui) 1)
	     (= (negcount.rui rui) 0))
	(let* ((restr (make.restr (subst.rui rui)))
	       (subst (if (is.n dest)
			  (restrict.sbst (subst.rui rui)
					 (sneps::all-vars.n dest))
			  (subst.rui rui)))
	       (newsup (compute-new-support.intr (fns.rui rui)))
	       (report (make.rep subst newsup 'NEG  *NODE* nil crntct)))
	  (unless-remarkedp.rui
	   rui
	   (remark '"~%Since it is the case that"
		   (makeone.ns (signature.rep intr-report)) restr)
	   (remark '"I infer it is not the case that"
		   (makeone.ns *NODE*) restr))
	  (send-reports (makeone.repset report) ch)))

    ; an AND rule is derived when a positive report is received from 
    ; all arguments of the rule, and a NOR rule is derived when
    ; a negative report is received from all arguments of the rule.

    (when (eq (if (equal *TYPE* 'NOR) (negcount.rui rui) (poscount.rui rui))
	      (cardinality.ns cqs))
      (let* ((aants (context.ich ich))
	     (restr (make.restr (subst.rui rui)))
	     (subst (if (is.n dest)
			(restrict.sbst (subst.rui rui)
				       (sneps::all-vars.n dest))
		      (subst.rui rui)))
	     (newsup (compute-new-support.intr (fns.rui rui)))	     
	     (report (make.rep subst newsup 'POS *NODE* nil crntct)))
	(inform-introduction restr aants cqs)
	(send-reports (makeone.repset report) ch)
	(try-to-use-introduction-conclusion report)))
    (setq updated-ruiset (update.ruis rui updated-ruiset))
    (setq *INTRODUCTION-CHANNELS*
	  (update.ichset (make.ich ch (context.ich ich) cqs updated-ruiset)
			 *INTRODUCTION-CHANNELS*))))


;------------------------------------------------------------------------
;   get-rule-use-info  in file "snip/fns/rule-reports.lisp"
;------------------------------------------------------------------------

(defun get-rule-use-info (report cqch)
  "Returns the set of rule-use-infos which are compatible
   with the reported instance, with each rule-use-info
   updated to include any new bindings in the report. 
  
   modified by J. Choi 5/6/92.
   major modification:
       Add a routine for updating the *RULE-USE-CHANNELS* register
       with compatible rule-use-infos.
       In the old version, this update of the *RULE-USE-CHANNLES*
       register is done in each rule handler procedure.
       This modification is made to be consistent with
       'get-rule-use-info-ptree' and 'get-rule-use-info-sindexing'
       which also update the *RULE-USE-CHANNELS* register. "

  (let ((flag nil)
	(sbst (subst.rep report))
	(ant (signature.rep report))
	(supp (support.rep report))
	(ants (ants.cqch cqch))
	(updated-ruiset (ruiset.cqch cqch))
	(result (new.ruis)))
    (do.set (rui (ruiset.cqch cqch))
      (setq flag (flag.fns ant (fns.rui rui)))
      (when (match::is-compatible.sbst sbst (subst.rui rui))
	(cond
	  ((or (null (fns.rui rui)) (eq flag 'UNKNOWN) (eq flag 'REQUESTED))
	   (setq result
		 (update.ruis
		   (update.rui
		     (make.rui
		      (union.sbst (subst.rui rui) sbst) (poscount.rui rui)
		      (negcount.rui rui) (fns.rui rui) (remarkedp.rui rui))
		     ant supp (sign.rep report))
		   result)))
	  (t
	   (setq result
		 (update.ruis
		   (make.rui
		     (subst.rui rui) (poscount.rui rui) (negcount.rui rui)
		     (update.fns (fns.rui rui) ant supp flag)
		     (remarkedp.rui rui))
		   result))))))

    ; the following statements are added to update *RULE-USE-CHANNELS*
    ; with compatible rule-use-infos
    
    (do.set (rui result)
	(setq updated-ruiset (update.ruis rui updated-ruiset))
	(setq *RULE-USE-CHANNELS*
	      (update.cqchset
	        (make.cqch (channel.cqch cqch) ants updated-ruiset)
		*RULE-USE-CHANNELS*)))
    result))


(in-package :sneps)

(defsnepscom resetnet ((&optional (reset-relations? nil)) (top) t)
  (declare (special outunit crntct))

  ;; Do this to make processes garbage-collectible:
  (clear-infer-all)
  
  ;; Remove node/context access information:
  (do.ns (n (value.sv 'nodes))
	 (remprop (nodeaccess n) '=snode))
  (do.cts (ct (allct))
	  (remprop (contextaccess ct) '=scontext))

  ;; Initialize node-name counters:
  (dolist (prefix '(b m v p tm tv tp))
    (setf (get 'gennewnode prefix) 0))
  (setf (get 'gennewcontext 'c) 0)

  ;; Reset SNePSUL variables:
  (do.svs (v (value.sv 'variables))
	  (unless (member v '(relations contexts))
	    (remprop v :val)))
  (when reset-relations?
    ;; Reset relations AND paths!!
    (do.rs (r (value.sv 'relations))
	   (when (get r :pathdef)
	     (eval `(undefine-path ,r)))
	   (undefine.r r))
    (remprop 'relations :val)
    (mapcar #'(lambda (ident) (new.r ident))
	    *initial-relations*))

  ;; Initialize SNePSUL variables:
  (set.sv 'nodes nil)
  (set.sv 'varnodes nil)
  (set.sv 'variables '(variables relations nodes command 
		       lastcommand errorcommand contexts lastvalue 
		       assertions patterns defaultct))
  (if (hash-table-p (value.sv 'contexts))
      (clrhash (value.sv 'contexts))
    (set.sv 'contexts (make-hash-table :test #'equal)))
  (name.ct (buildcontext (new.ns)) 'default-defaultct)
  (set.sv 'defaultct 'default-defaultct)
  (setf crntct 'default-defaultct)
  (format outunit
	  "~&~%Net reset~:[ - Relations and paths are still defined~;~]"
	  reset-relations?)
  (values))

(defsnepscom clear-infer-all (())
  "Clears SNIP by removing all node activations."
  (do.ns (node (value.sv 'nodes))
    (deactivate.n node))
  ;; Unbind any stray multi processes (e.g., user processes)
  (do-symbols (multi-symbol (find-package 'multi))
    (when (multi:is-process-name multi-symbol)
      (makunbound multi-symbol)))
  "All node activations removed.")

(defsnepscom clear-infer (())
  "Modified not to purge some expertise information.
   For rule nodes, information in *KNOWN-INSTANCES* is retained."
  (let (node-process type known-instances)
    (do.ns (nde (value.sv 'nodes)
         "Node activation cleared. Some register information retained.")
	(setq node-process (activation.n nde))
	(when node-process
	  (setq type (multi::regfetch node-process 'snip::*NAME*))
	  (cond ((member type '(snip::rule snip::num-quant.rule))
		 (setq known-instances
		       (multi::regfetch node-process
					'snip::*KNOWN-INSTANCES*))
		 (setf (node-activation nde) nil)
		 (setq node-process (activate.n nde))
		 (multi::regstore node-process
				  'snip::*KNOWN-INSTANCES*
				  known-instances))
		(t (deactivate.n nde)))))))


(in-package :snebr)

(defun remove-new-hyp-1 (newhyp)
  (declare (special snip:crntctname))
  (change-context-name (compl.ns (context-hyps (value.sv snip:crntctname))
				 newhyp)))

; allow the user to type in SNePSLOG commands
(defun add-new-hyps ()
  (declare (special sneps::outunit))
  (let (option add-new? newhyps)
    (format sneps::outunit
	    "~%~T Do you want to add a new hypothesis?")
    (setq add-new? (user-says-yes))
    (if add-new?
	(setq option (snepsul-or-snepslog-option)))
    (do ()
	((not add-new?))
      (setq newhyps (insert.ns (request-new-hyp option) newhyps))
      (format sneps::outunit
	      "~%~T Do you want to add another hypothesis?")
      (setq add-new? (user-says-yes)))
    newhyps))


(defun snepsul-or-snepslog-option ()
  (declare (special sneps::outunit sneps::inunit))
  (format sneps::outunit
	  "~%~T Do you want to use SNePS[u]L or SNePSL[o]G?~
           ~%~T (please type u or o)")
  (let (ans)
    (loop (ct-prompt)
	  (setq ans (read sneps::inunit))
	  (if (or (eq ans 'snepsul::u)
		  (eq ans 'snepsul::o))
	      (RETURN ans))
	  (format sneps::outunit "Please type u or o"))))


; only one addtion of hypothesis at a time
(defun request-new-hyp (option)
  (declare (special sneps::outunit sneps::inunit))
  (if (eq option 'snepsul::u)
      (format sneps::outunit
	      "~%~T Enter a hypothesis using the SNePSUL command `assert': ")
      (format sneps::outunit
	      "~%~T Enter a hypothesis using SNePSLOG: "))
  (let (newhyp ans)
    (loop (ct-prompt)
	  (if (eq option 'snepsul::u)
	      (setq ans (read sneps::inunit))
	      (setq ans (snepslog::snepslog-read sneps::inunit)))
	  (setq newhyp (sneps::topsneval (insert-context ans)))
	  (cond (newhyp
		 (RETURN (sneps::choose.ns newhyp)))
		(t
		 (format sneps::outunit
			 "~%~T Oops... something went wrong~
                          ~%~T Would you try to enter again?")
		 (unless (user-says-yes)
		   (RETURN (new.ns))))))))


(defun ck-contradiction (newnode context flag)
  (let ((contr-nd (negation-or-negated-nd newnode)))
    (when (and contr-nd
	       (exists-contradiction contr-nd context))
      (ck-contradiction-1 newnode contr-nd context flag))
    newnode))

(defun ck-contradiction-1 (newnode contr-nd context flag)
  (let ((contrsup (sneps:ctcs-to-cts (sneps:node-asupport contr-nd))))
    (update-contexts (ctcs-to-cts (sneps:node-asupport newnode)) contrsup)
    (sneps:mark-inconsistent context)
    (sneps:updateall context)
    (if (eq flag 'sneps:assertion)
	(sneps-contr-handler newnode contr-nd)
	(snip-contr-handler newnode contr-nd context))))

