;;; -*- Mode: Lisp; Syntax: Common-Lisp; Package: SNIP; Base: 10 -*-

;; Copyright (C) 1984, 1988, 1989, 1993 Research Foundation of 
;;                                      State University of New York

;; Version: $Id: user.lisp,v 1.3 1993/06/04 06:27:49 snwiz Exp $

;; This file is part of SNePS.

;; SNePS is free software; you may redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; SNePS is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with SNePS; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA, or to
;; Dr. Stuart C. Shapiro, Department of Computer Science, State University of
;; New York at Buffalo, 226 Bell Hall, Buffalo, NY 14260, USA

(in-package :snip)


; =============================================================================
;
; tally
; -----
;
;       arguments     : report - <report>
;
;       nonlocal-vars : *POS-FOUND*, *NEG-FOUND* registers of the current process
;
;       description   : increments the appropriate count of reports received,
;                       based on the sign of the report
;
;                                        written :  rgh 11/18/85
;                                        modified:  rgh  3/08/86
;
;
(defmacro tally (report)
  `(cond ((eq (sign.rep ,report) 'POS)
             (setq *POS-FOUND* (1+ *POS-FOUND*)))
         (t
             (setq *NEG-FOUND* (1+ *NEG-FOUND*)))))
;
;
; =============================================================================
;
; enough-answers
; --------------
;
;       returns       : <boolean>
;
;       nonlocal-vars : *TOTAL-DESIRED*, *POS-DESIRED*, *NEG-DESIRED*, *POS-FOUND*,
;                       and *NEG-FOUND* registers of the current process
;
;       description   : returns "true" if the desired number of deduced
;                       results has been found, "false" otherwise
;
;                                        written :  rgh 11/18/85
;                                        modified:
;
;
(defmacro enough-answers ()
  `(cond ((and (null *TOTAL-DESIRED*) (null *POS-DESIRED*)) nil)
         ((numberp *TOTAL-DESIRED*)
             (not (< (+ *POS-FOUND* *NEG-FOUND*) *TOTAL-DESIRED*)))
         (t
             (not (or (< *POS-FOUND* *POS-DESIRED*)
                      (< *NEG-FOUND* *NEG-DESIRED*))))))
;
;
; =============================================================================
;
; user
; ----
;
;       description   : This is the process which represents the SNePS user
;                       who has asked the system to deduce something
;
;                                        written :  rgh 11/18/85
;                                        modified:  rgh 11/27/85
;                                                   rgh  3/09/86
;                                                   rgh  4/13/86
;                                                   njm  4/27/89
;
;
(defun user (*NAME* *REPORTS* *CONTEXT-NAME* *DEDUCED-NODES* *TOTAL-DESIRED*
	     *POS-DESIRED* *NEG-DESIRED* *POS-FOUND* *NEG-FOUND* *PRIORITY*)
  (declare (special multi::curnt%))
  (let ((reports *REPORTS*))
    (setq *REPORTS* (new.repset))
    (do.repset (report reports)
       (let ((n (node.rep report)))
	 (unless (ismemb.ns n *DEDUCED-NODES*)
		 (tally report)
		 (setq *DEDUCED-NODES* (insert.ns n *DEDUCED-NODES*))))))
  (setq *DEDUCTION-RESULTS* (union.ns *DEDUCED-NODES* *DEDUCTION-RESULTS*))
  (if (enough-answers) (suspend-inference))
  (set multi::curnt% (list *NAME* *REPORTS* *CONTEXT-NAME* *DEDUCED-NODES*
			   *TOTAL-DESIRED* *POS-DESIRED* *NEG-DESIRED*
			   *POS-FOUND* *NEG-FOUND* *PRIORITY*)))

(setf (get 'user 'multi::lregs%)
      '(*NAME* *REPORTS* *CONTEXT-NAME* *DEDUCED-NODES* *TOTAL-DESIRED*
	       *POS-DESIRED* *NEG-DESIRED* *POS-FOUND* *NEG-FOUND* *PRIORITY*))

;
;
; =============================================================================
;
; suspend-inference
; -----------------
;
;       returns       : nil
;
;       nonlocal-vars : evnts - the Multi events queue
;                       USER-PROCESS  - the user process for this deduction
;                       lastinfer
;
;       description   : saves the current state of a deduction on the
;                       property list of the atom 'lastinfer, when the
;                       deduction is halted due to enough answers having
;                       been found
;
;       side-effects  : modifies the property list of 'lastinfer
;
;                                        written :  rgh 11/18/85
;                                        modified:  rgh  3/08/86
;                                                   hc  04/26/89
;
;
;(defun suspend-inference ()
;   (setf (get 'lastinfer evnts) '%events-queue%)
;   (setf (get 'lastinfer *USER-PROCESS*) '%user-process%)
;   (setq evnts nil))

(defun suspend-inference ()
;  ;; this is how it should be if it was used (hc)
;  (setf (get 'lastinfer '%events-queue%) evnts)
;  (setf (get 'lastinfer '%user-process%) *USER-PROCESS*)
  (multi:clear-all-queues))
;
;
; =============================================================================
