;;; -*- Mode: Lisp; Syntax: Common-Lisp; Package: SNIP; Base: 10 -*-

;; Copyright (C) 1984, 1988, 1989, 1993 Research Foundation of 
;;                                      State University of New York

;; Version: $Id: rule-eintr.lisp,v 1.5 1993/06/04 06:27:30 snwiz Exp $

;; This file is part of SNePS.

;; SNePS is free software; you may redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; SNePS is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with SNePS; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA, or to
;; Dr. Stuart C. Shapiro, Department of Computer Science, State University of
;; New York at Buffalo, 226 Bell Hall, Buffalo, NY 14260, USA

(in-package :snip)


; =============================================================================
;
; process-one-introduction-report.rule 
; ------------------------------------
;
;       arguments     : report - <report>
;
;       returns       : <boolean>
;
;       nonlocal-vars : *INTRODUCTION-CHANNELS* register
;
;       description   : tries to send "report" to all channels in
;                       *INTRODUCTION-CHANNELS*.
;
;       side-effects  : asserts the rule node (or an instance rule node)
;                       if possible and sends a report.  
;
;                                        written : njm  11/06/88
;                                        modified:
;
;
(defun process-one-introduction-report.rule (report)
      (let ((anysent nil)) 
	(do.set (ich *INTRODUCTION-CHANNELS* anysent)
		(setq anysent (or (try-to-send-introduction-conclusion.rule report ich)
				  anysent)))))

;
;
; =============================================================================
;
; try-to-send-introduction-conclusion.rule 
; ----------------------------------------
;
;       arguments     : report - <report>
;                       ich - <i-channel>
;
;       returns       : <boolean>
;
;       nonlocal-vars : *NODE*, *KNOWN-INSTANCES*, *USABILITY-TEST* registers
;
;       description   : tests the consequent report "report" against the
;                       *INTRODUCTION-CHANNELS* "ich", and if it passes the
;                       channel filter and the support of the report has tag 'DER
;                       and includes the rose hypothesis, an attempt is made to
;                       draw conclusions using report.
;
;
;                                        written :  njm 11/06/88
;
; 
;
(defun try-to-send-introduction-conclusion.rule (report ich)
  (let ((ch (channel.cqch ich)))
    (when (compatible (subst.rep report) (filter.ch ch))
      (let ((filtered-support (if (member *TYPE* '(AND NOR))
				  (filter.sup (support.rep report) (context.ch ch))
				  (filter-introduction-support (support.rep report)
							       (context.ich ich)
							       (context.ch ch)))))
	(unless (isnew.sup filtered-support)
	  (apply #'INTRODUCTION-END-HANDLER
		 (list (make.rep (subst.rep report)
				 filtered-support
				 (sign.rep report)
				 (signature.rep report)
				 (node.rep report)
				 (context.rep report)) 
		       ich)))))))

;
;
; =============================================================================
;
; filter-introduction-support 
; ---------------------------
;
;       arguments     : support-rep - <support>
;                       extra-hyps  - <node set>
;                       ct-channel  - <context>
;
;       returns       : <support>
;
;       description   : filters "support-rep":
;                        a) only 'DER tags are accepted
;                        b) only contexts that:
;                             1) are a subset of the union of "extra-hyps" with
;                                "ct-channel", and
;                             2) include "extra-hyps"
;                           are returned.
;
;                                        written :  njm 11/06/88
;
;
;
(defun filter-introduction-support (support-rep extra-hyps ct-channel)
  (let ((new-support (new.sup))
	(ct-channel-hyps (sneps:context-hyps ct-channel)))
    (dolist (ct (getcontextset.sup 'sneps:der support-rep) new-support)
      (when (equal extra-hyps (sneps:compl.ns (sneps:context-hyps ct)
					      ct-channel-hyps))
	(setq new-support
	      (insert.sup 'der
			  (fullbuildcontext (sneps:compl.ns (sneps:context-hyps ct)
							    extra-hyps)
					    (new.cts))
			  new-support))))))

;
; =============================================================================
;
(defun INTRODUCTION-END-HANDLER (intr-report ich)
  (do* ((ruis (get-introduction-info intr-report ich) (others.ruis ruis))
	(rui (choose.ruis ruis) (choose.ruis ruis))
	(cqs (consequents.ich ich))
	(updated-ruiset (ruiset.ich ich))
	(count (if (equal *TYPE* 'NOR) (negcount.rui rui) (poscount.rui rui))
	       (if (equal *TYPE* 'NOR) (negcount.rui rui) (poscount.rui rui))))
       ((isnew.ruis ruis) t)
    (when (eql count (cardinality.ns cqs)) 
      (let* ((aants (context.ich ich))
	     (restr (make.restr (subst.rui rui)))
	     (ch (channel.ich ich))
	     (dest (destination.ch ch))
	     (subst (if (is.n dest)
		        (restrict.sbst
			 (subst.rui rui)
			 (union.ns (freevars.n dest)
				   (quantified-vars.n dest)))
		        (subst.rui rui)))
	     (report (make.rep subst 
			       (compute-new-support.intr (fns.rui rui))
			       'POS
			       *NODE*
			       nil
			       (context.ch ch)
			       )))
	(inform-introduction restr aants cqs)
	(send-reports (makeone.repset report) ch)
	(try-to-use-introduction-conclusion report)))
    (setq updated-ruiset (update.ruis rui updated-ruiset))
    (setq *INTRODUCTION-CHANNELS*
	  (update.ichset (make.ich (channel.ich ich) 
				   (context.ich ich)
				   cqs
				   updated-ruiset)
			 *INTRODUCTION-CHANNELS*))))

;
; =============================================================================
;
; try-to-use-introduction-conclusion
; ----------------------------------
;
;       arguments     : report - <report>
;
;       nonlocal-vars : the *KNOWN-INSTANCES* and *NODE* registers,
;                       plus *ADDED-NODES*
;
;       description   : handles reports of instances of the rule due
;                       to the aplication of a introduction deduction rule.
;                       The instance is broadcast to all apropriate outgoing
;                       channels.  Also, if this rule instance was
;                       requested for use in deducing a consequent, the new
;                       rule instance is applied.
;
;       side-effects  : *KNOWN-INSTANCES*, and *ADDED-NODES* are updated.
;
;                                        written :  njm 11/10/88
;                                        modified:  
;
;
(defun try-to-use-introduction-conclusion (report)
  (let ((instance (rep-to-instance report)))
    (declare (special *ADDED-NODES*))
    (cond ((isnew.sbst (subst.inst instance))
	   (broadcast-one-report report)
	   (try-applying-one-report report))
	  ((unknown.inst instance)
	   (let ((newnode (if (quantified-vars.n *NODE*)
			      (choose.ns
				(sneps:find-or-build
				  (sneps::clean-quantifiers.cs
				    (apply-subst.cs (subst.rep report)
						    (n-to-downcs *NODE*)))))
			      (match::applysubst *NODE* (subst.rep report)))))
	     (setq *ADDED-NODES* (insert.ns newnode *ADDED-NODES*)
		   *KNOWN-INSTANCES* (insert.iset instance *KNOWN-INSTANCES*)))
	   (broadcast-one-report report)
	   (try-applying-one-report report)))))
;
;
; =============================================================================
;
; get-introduction-info 
; ---------------------
;
;       arguments     : report - <report>
;                       ich - <i-channel>
;
;       returns       : <rule use info set>
;
;       description   : returns the set of rule-use-infos which are compatible
;                       with the reported instance, with each rule-use-info
;                       updated to include any new bindings in the report
;
;                                        written :  njm 11/07/88
;
(defun get-introduction-info (report ich)
  (let ((flag nil)
	(sbst (subst.rep report))
	(cq (signature.rep report))
	(supp (support.rep report))
	(result (new.ruis)))
    (do.set (rui (ruiset.ich ich) result)
      (setq flag (flag.fns cq (fns.rui rui)))
      (when (match::is-compatible.sbst sbst (subst.rui rui))
	(cond
	  ((or (null (fns.rui rui)) (eq flag 'unknown) (eq flag 'requested))
	   (setq result
		 (update.ruis
		   (update.rui
		     (make.rui (union.sbst (subst.rui rui) sbst) (poscount.rui rui)
			       (negcount.rui rui) (fns.rui rui) (remarkedp.rui rui))
		     cq supp (sign.rep report))
		   result)))
	  (t
	   (setq result
		 (update.ruis
		   (make.rui (subst.rui rui) (poscount.rui rui) (negcount.rui rui)
			     (update.fns (fns.rui rui) cq supp flag) (remarkedp.rui rui))
		   result))))
	))))

;
;
; =============================================================================
;
; inform-introduction 
; -------------------
;
;       arguments     : restr - <restriction>
;                       cqs   - <node set>
;                       aants - <node set>
;
;       returns       : <never mind>
;
;       description   : informs the user the inference of a rule, based on
;                       hypothectical reasoning.
;       
;
;                                        written :  njm  11/06/88
;                                        modified:  hc   10/08/92
;
(defun inform-introduction (restr aants cqs)
  (case *TYPE*
    (AND (remark "~%Since" (makeone.ns (choose.ns cqs)) restr)
	 (do.ns (cq (others.ns cqs))
	   (remark "and " (makeone.ns cq) restr))
	 (remark "I infer" (makeone.ns *NODE*) restr))
    (NOR (remark "~%Since it is not the case that"
		 (makeone.ns (choose.ns cqs))
		 restr)
	 (do.ns (cq (others.ns cqs))
	   (remark "and since not" (makeone.ns cq) restr))
	 (remark "I infer" (makeone.ns *NODE*) restr))
    (t (remark "~%Since" (makeone.ns (choose.ns cqs)) restr)
       (do.ns (cq (others.ns cqs))
	 (remark "and" (makeone.ns cq) restr))
       (remark (format nil "~:[were~;was~] derived assuming"
		       (= (cardinality.ns cqs) 1))
	       (makeone.ns (choose.ns aants))
	       restr)
       (do.ns (ant (others.ns aants))
	 (remark "and assuming" (makeone.ns ant) restr))
       (remark "I infer" (makeone.ns *NODE*) restr))))

;
;
; =============================================================================
;
; compute-new-support.intr 
; ------------------------
;
;       arguments     : fns - <fns>
;                       
;       returns       : <support>
;
;       description   : receives as arguments:
;                        'fns' -- a flag node set
;                       Computes a new support based on all supports
;                       present in 'fns'.
;
;       implementation: the structure of the variable allcombinations is as follows:
;                         ((ot ... ot) (ct ... ct) ... (ot ... ot) (ct ... ct))
;                       where each pair '(ot ... ot) (ct ... ct)' has an ot and a ct
;                       of each flag node present in 'fns'.
;       
;
;                                        written :  njm  11/06/88
;                                        modified: 
;
;
(defun compute-new-support.intr (fns)
  (let* ((newsupport (new.sup))
	 (suplist (fns-to-suplist fns))
	 (allcombinations (if (null (rest suplist))
			      (sup-to-combinations (first suplist))  
			      (combinations-of (first suplist) (rest suplist)))))
    (do ((s1 allcombinations (rest (rest s1))))
	((null s1) newsupport)
      (setq newsupport
	    (insert.sup (if (equal *TYPE* 'AND)
			    (combine-and-intr-ots* (first s1) (second s1))
			    (combine-ots* (first s1)))
			(fullbuildcontext (new.ns) (second s1))
			newsupport)))))









