;;; -*- Mode: Lisp; Syntax: Common-Lisp; Package: SNIP; Base: 10 -*-

;; Copyright (C) 1984, 1988, 1989, 1993 Research Foundation of 
;;                                      State University of New York

;; Version: $Id: iset.lisp,v 1.3 1993/06/04 06:26:28 snwiz Exp $

;; This file is part of SNePS.

;; SNePS is free software; you may redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; SNePS is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with SNePS; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA, or to
;; Dr. Stuart C. Shapiro, Department of Computer Science, State University of
;; New York at Buffalo, 226 Bell Hall, Buffalo, NY 14260, USA

(in-package :snip)


; =============================================================================
;
; <instance set> ::= { <instance> ... }
;
; -----------------------------------------------------------------------------
;
; PRIMITIVE      new.iset : --> <instance set>
;  ELEMENTS
;
; RECOGNIZERS    is.iset    : <universal> --> <boolean>
;                isnew.iset : <instance set> --> <boolean>
;
; TESTS          ismemb.iset : <instance> x <instance set> --> <boolean>
;
; SELECTORS      choose.iset : <instance set> --> <instance>
;                others.iset : <instance set> --> <instance set>
;
; CONSTRUCTORS   insert.iset     : <instance> x <instance set>
;                                         --> <instance set>
;                makeone.iset    : <instance> --> <instance set>
;                union.iset      : <instance set> x <instance set>
;                                         --> <instance set>
;                addbinding.iset : <mbind> x <instance set> --> <instance set>
;
; =============================================================================
;
; new.iset
; -----------
;
;       returns       : <instance set>
;
;       description   : returns a "new" <instance set>
;
;                                        written :  rgh 08/21/85
;                                        modified:
;
;
(defmacro new.iset ()
  `(new.Set))
;
;
; =============================================================================
;
; is.iset
; ----------
;
;       arguments     : u - <universal>
;
;       returns       : <boolean>
;
;       description   : returns "true" if "u" is a <instance set>,
;                               "false" otherwise
;
;                                        written :  rgh 08/21/85
;                                        modified:
;
;
(defmacro is.iset (u)
  `(and (is.Set ,u)
        (is.inst (choose.Set ,u))))
;
;
; =============================================================================
;
; isnew.iset
; -------------
;
;       arguments     : iset - <instance set>
;
;       returns       : <boolean>
;
;       description   : returns "true" if "iset" is a "new" <instance set>
;                               "false" otherwise
;
;                                        written :  rgh 08/21/85
;                                        modified:
;
;
(defmacro isnew.iset (iset)
  `(isnew.Set ,iset))
;
;
; =============================================================================
;
; choose.iset
; --------------
;
;       arguments     : iset - <instance set>
;
;       returns       : <instance>
;
;       description   : returns the first <instance> in "iset"
;
;                                        written :  rgh 08/21/85
;                                        modified:
;
;
(defmacro choose.iset (iset)
  `(choose.Set ,iset))
;
;
; =============================================================================
;
; others.iset
; --------------
;
;       arguments     : iset - <instance set>
;
;       returns       : <instance set>
;
;       description   : returns a <instance set> consisting of all of the
;                       <instance>s in "iset" except the first
;
;                                        written :  rgh 08/21/85
;                                        modified:
;
;
(defmacro others.iset (iset)
  `(others.Set ,iset))
;
;
; =============================================================================
;
; ismemb.iset
; -----------
;
;       arguments     : inst - <instance>
;                       iset - <instance set>
;
;       returns       : <boolean>
;
;       description   : returns "true" if "inst" is a member of "iset",
;                               "false" otherwise
;
;                                        written :  rgh 08/21/85
;                                        modified:  rgh 11/30/85
;                                        modified:  njm/cpf 10/21/88
;
(defmacro ismemb.iset (inst iset)
  `(do* ((is ,iset (others.iset is))
	 (i (choose.iset is) (choose.iset is))
	 (result nil))
	((or (isnew.iset is) result) result)
     (if (and (iseq.sbst (subst.inst ,inst) (subst.inst i))
	      (isincluded.sup (support.inst ,inst) (support.inst i))
	      (iseq.sign (sign.inst ,inst) (sign.inst i)))
	 (setq result t))))

;
;
; =============================================================================
;
; insert.iset
; -----------
;
;       arguments     : inst - <instance>
;                       iset - <instance set>
;
;       returns       : <instance set>
;
;       description   : returns "iset" with "inst" inserted if it was not
;                       already there
;
;                                        written :  rgh 11/24/85
;                                        modified:  rgh 11/30/85
;                                        modified:  njm/cpf 10/21/88
;
(defmacro insert.iset (inst iset)
  `(let ((result nil)
	 (finiset (new.iset)))
     (cond ((ismemb.iset ,inst ,iset) ,iset)
	   ((do.Set (i ,iset result)
	      (if (and (iseq.sbst (subst.inst i) (subst.inst ,inst))
		       (iseq.sign (sign.inst i) (sign.inst ,inst)))
		  (progn (setq finiset (putin.Set (merge.inst i ,inst)
						  finiset))
			 (setq result t))
		  (setq finiset (putin.Set i finiset))))
	    finiset)
	   (t (putin.Set ,inst ,iset)))))
  
;
;
; =============================================================================
;
; makeone.iset
; ------------
;
;       arguments     : inst - <instance>
;
;       returns       : <instance set>
;
;       description   : returns an <instance set> containing the one element
;                       "inst"
;
;                                        written :  rgh  3/30/86
;                                        modified:
;
;
(defmacro makeone.iset (inst)
  `(makeone.Set ,inst))
;
;
; =============================================================================
;
; union.iset
; ----------
;
;       arguments     : iset1, iset2 - <instance set>
;
;       returns       : <instance set>
;
;       description   : returns the set union of "iset1" and "iset2"
;
;                                        written :  rgh  3/30/86
;                                        modified:  njm/cpf 10/18/88
;
;

(defmacro union.iset (iset1 iset2)
  `(let ((result1 ,iset2))
    (do.Set (ii ,iset1 result1)
      (setq result1 (insert.iset ii result1)))))
      

;
; =============================================================================
;
; addbinding.iset
; ---------------
;
;       arguments     : mb - <mbind>
;                       inst-set - <instance set>
;
;       returns       : <instance set>
;
;       description   : returns an <instance set> with "mb" added to the
;                       substitutions of each of the instances in "inst-set"
;
;                                        written :  rgh  3/30/86
;                                        modified:  scs  3/15/88
;
;
(defun addbinding.iset (mb inst-set)
  (let ((result (new.iset)))
    (do.set (inst inst-set result)
       (setq result (insert.iset (addbinding.inst mb inst) result)))))
;
;
; =============================================================================
